#!/usr/bin/python

from picalo import *

DETECTLET_STANDARD = 1.0

wizard = '''
<wizard>
  <page>
    Select the table containing the invoices.  The table should
    contain a date column, an invoice number column, and (optionally)
    a vendor id column to stratify by.   
    <parameter type="Table" variable="table"/>
  </page>
  <page>
    Which column contains the invoice number?  This column should contain
    the actual invoice numbers assigned by the vendors, not the invoice amounts
    or your organization's internal invoice tracking number.
    <parameter type="Column" table="table" variable="invoicecol"/>
  </page>
  <page>
    Most invoice tables contain invoices from many vendors.  The table
    will need to be stratified into a single table containing each vendor's
    invoices.  If this table has multiple vendors, which column(s) uniquely
    identify the vendor?  This is usually a "Vendor ID" column.
    <parameter type="Column" multiple="true" table="table" variable="vendorcol"/>
  </page>
</wizard>
'''

RESULTS_TEXT = '''\
    The displayed results table shows the percentage of sequential invoices
    each vendor has.  While there may be legitimate reasons for vendors to
    have only one client, you should look into any vendors with high percentages.
'''


def run(table, invoicecol, vendorcol):
  '''A common fraud is for an employee within an organization to set 
     up a dummy company and send invoices or other bills to the organization.
     Since the company is not real, the organization is its only client,
     and since invoice booklets and computer software numbers invoices sequentially,
     invoices from phantom vendors are usually sequential. In other words, 
     invoices from a phantom vendor often start with id 1000,
     then 1001, then 1002, and so on.  
     
     This wizard takes an invoice file (or similar billing file) and 
     performs the following steps on it:
     - Sorts the input table by invoicecol.
     - Stratifies the file into individual tables for each vendor.
     - Scans each individual table and determines how many invoices
        are sequential.  
     - Calculates a percentage of the invoices that have sequential
        numbers.  This percentage is added to the results table.
     - When all tables are scanned, it sorts the results table by
        the percentage.
        
     In the resulting table, those vendors with a high percentage
     of sequential invoice numbers may be phantom vendors.  Of course,
     there may be legitimate reasons for vendors to have only one
     client, but it raises suspicion and prompts further analysis.
  '''
  # validate the input data
  assert invoicecol != vendorcol, 'The invoice number and vendor id columns must be different.  Please ensure you have selected different columns.'

  # sort the table by date
  Simple.sort(table, True, invoicecol)
  
  # stratify the table
  vendors = Grouping.stratify_by_value(table, *vendorcol)
  
  # check each vendor for sequential invoices
  vendorcolname = '_'.join(vendorcol)
  results = Table([
    ( vendorcolname,   unicode ),
    ( 'Sequential',    int),
    ( 'NonSequential', int),
    ( 'Percentage',    number),
  ])
  for vendorrecs in vendors:
    sequential_indices = Simple.compare_records(vendorrecs, "rec1['" + invoicecol + "'] + 1 == rec2['" + invoicecol + "']")
    rec = results.append()
    rec[vendorcolname] = vendorrecs.startvalue
    rec['Sequential'] = len(sequential_indices)
    rec['NonSequential'] = max(len(vendorrecs) - len(sequential_indices) - 1, 0)
    rec['Percentage'] = float(len(sequential_indices)) / float(len(vendorrecs) - 1)
  
  # sort and return
  Simple.sort(results, False, 'Percentage')
  return results, RESULTS_TEXT

  
  
def example_input():
  '''Returns the eaxmple input table so the user can
     see what their input should look like'''
  import StringIO  # to emulate a file for load_csv
  table = load_csv(StringIO.StringIO(csvdata))
  table.set_type('Invoice', int)
  table.set_type('Date',    DateTime)
  table.set_type('Amount',  number)
  return table  
  
  
# This next part is not required, but it's easier to put the example data
# directly in this file so I don't have to worry about directories.
csvdata = '''\
Vendor,Invoice,Date,Amount
Vendor01,1000,2006-01-01,7636.73
Vendor01,1001,2006-01-01,3401.31
Vendor01,1002,2006-01-07,2121.84
Vendor01,1003,2006-01-07,3592.86
Vendor01,1004,2006-01-08,5602.18
Vendor01,1005,2006-01-08,3610.11
Vendor01,1006,2006-01-08,4569.14
Vendor01,1007,2006-01-09,3643.24
Vendor01,1008,2006-01-19,2566.29
Vendor01,1009,2006-01-27,2456.09
Vendor01,1010,2006-02-02,1321.04
Vendor01,1011,2006-02-03,5490.54
Vendor01,1012,2006-02-04,5943.35
Vendor01,1013,2006-02-05,5940.22
Vendor01,1014,2006-02-13,9721.48
Vendor01,1015,2006-02-14,9668.81
Vendor01,1016,2006-02-24,4580.49
Vendor01,1017,2006-03-02,8346.86
Vendor01,1018,2006-03-02,2869.63
Vendor01,1019,2006-03-12,2751.64
Vendor01,1020,2006-03-20,9217.69
Vendor01,1021,2006-03-26,9442.18
Vendor01,1022,2006-04-05,6419.64
Vendor01,1023,2006-04-11,3530.55
Vendor01,1024,2006-04-12,1901.52
Vendor01,1025,2006-04-22,3707.74
Vendor01,1026,2006-04-28,9751.59
Vendor01,1027,2006-05-06,7870.72
Vendor01,1028,2006-05-07,2272.03
Vendor01,1029,2006-05-08,3839.99
Vendor01,1032,2006-05-16,7900.78
Vendor01,1034,2006-05-22,4476.24
Vendor01,1035,2006-05-28,6900.39
Vendor01,1036,2006-05-28,8819.22
Vendor01,1037,2006-05-28,7764.3
Vendor01,1038,2006-06-07,6127.03
Vendor01,1039,2006-06-13,5427.38
Vendor01,1040,2006-06-19,2499.69
Vendor01,1041,2006-06-19,9548.2
Vendor01,1042,2006-06-19,4183.61
Vendor01,1043,2006-06-19,8387.14
Vendor01,1045,2006-06-20,9110.9
Vendor01,1046,2006-06-20,4621.04
Vendor01,1047,2006-06-28,4982.35
Vendor01,1048,2006-07-04,7758.23
Vendor01,1049,2006-07-05,5513.6
Vendor01,1050,2006-07-11,8016.41
Vendor01,1051,2006-07-21,7629.48
Vendor01,1054,2006-07-29,3784.99
Vendor01,1055,2006-08-06,2339.43
Vendor01,1056,2006-08-12,1131.89
Vendor01,1057,2006-08-12,4484.97
Vendor01,1058,2006-08-22,8622.86
Vendor01,1061,2006-08-22,2645.9
Vendor01,1062,2006-08-30,5005.78
Vendor01,1063,2006-08-30,9211.57
Vendor01,1064,2006-09-09,4618.49
Vendor01,1065,2006-09-17,4623.71
Vendor01,1066,2006-09-25,3559.3
Vendor01,1067,2006-09-26,9672.57
Vendor01,1068,2006-09-26,8755.43
Vendor01,1069,2006-10-02,2860.2
Vendor01,1070,2006-10-12,5208.54
Vendor01,1071,2006-10-12,8705.76
Vendor01,1072,2006-10-18,3715.96
Vendor01,1073,2006-10-18,5730.77
Vendor01,1074,2006-10-28,2503.24
Vendor01,1075,2006-11-03,5888.42
Vendor01,1076,2006-11-09,1636.75
Vendor01,1077,2006-11-15,8343.28
Vendor01,1078,2006-11-16,7477.09
Vendor01,1079,2006-11-17,6879.27
Vendor01,1080,2006-11-18,9046.18
Vendor01,1081,2006-11-24,4656.82
Vendor01,1082,2006-12-02,6539.31
Vendor01,1083,2006-12-03,4694.23
Vendor01,1084,2006-12-04,7165.63
Vendor01,1085,2006-12-10,3447.87
Vendor01,1086,2006-12-18,6283.92
Vendor01,1087,2006-12-18,9220.86
Vendor01,1088,2006-12-26,2465.32
Vendor01,1089,2006-12-26,4796.84
Vendor01,1090,2007-01-01,5458.56
Vendor01,1091,2007-01-02,7708.66
Vendor01,1092,2007-01-03,4648.93
Vendor01,1093,2007-01-03,1441.17
Vendor01,1094,2007-01-04,7944.14
Vendor01,1095,2007-01-05,2488.96
Vendor01,1096,2007-01-06,2120.96
Vendor01,1097,2007-01-06,1152.81
Vendor01,1098,2007-01-06,4485.16
Vendor01,1099,2007-01-07,1235.22
Vendor01,1100,2007-01-13,5623.0
Vendor01,1101,2007-01-14,7375.46
Vendor01,1102,2007-01-20,5721.16
Vendor01,1103,2007-01-21,2129.51
Vendor01,1104,2007-01-21,8809.53
Vendor01,1105,2007-01-27,6549.25
Vendor01,1106,2007-02-02,3199.5
Vendor01,1107,2007-02-03,9623.6
Vendor01,1108,2007-02-04,6677.08
Vendor01,1109,2007-02-04,6514.19
Vendor01,1110,2007-02-10,7625.21
Vendor01,1111,2007-02-20,1951.44
Vendor01,1112,2007-02-20,3431.8
Vendor01,1113,2007-02-21,1022.47
Vendor01,1114,2007-02-22,9083.12
Vendor01,1115,2007-02-23,1073.89
Vendor01,1116,2007-02-24,8183.51
Vendor01,1117,2007-02-25,5061.08
Vendor01,1118,2007-02-26,2223.38
Vendor01,1119,2007-02-27,5958.96
Vendor01,1120,2007-03-05,4104.07
Vendor01,1121,2007-03-06,4041.25
Vendor01,1122,2007-03-06,6971.99
Vendor01,1123,2007-03-06,5594.44
Vendor01,1124,2007-03-07,1935.3
Vendor01,1125,2007-03-17,7452.12
Vendor01,1126,2007-03-17,1827.69
Vendor01,1127,2007-03-25,2151.04
Vendor01,1128,2007-03-25,6293.46
Vendor01,1129,2007-03-26,7435.71
Vendor01,1130,2007-03-26,5192.77
Vendor01,1131,2007-03-26,5501.99
Vendor01,1132,2007-04-05,1087.76
Vendor01,1133,2007-04-06,4176.53
Vendor01,1134,2007-04-06,3368.02
Vendor01,1135,2007-04-06,8056.19
Vendor01,1136,2007-04-16,1353.98
Vendor01,1137,2007-04-17,6569.06
Vendor01,1138,2007-04-18,3663.87
Vendor01,1139,2007-04-18,2198.48
Vendor01,1140,2007-04-18,9645.38
Vendor01,1141,2007-04-19,9431.16
Vendor01,1144,2007-04-25,7217.79
Vendor01,1145,2007-05-03,5089.6
Vendor01,1148,2007-05-09,2727.42
Vendor01,1149,2007-05-10,5638.87
Vendor01,1150,2007-05-10,2296.14
Vendor01,1151,2007-05-16,5410.36
Vendor01,1152,2007-05-22,3875.91
Vendor01,1153,2007-06-01,9103.99
Vendor01,1154,2007-06-07,1260.62
Vendor01,1155,2007-06-15,6479.23
Vendor01,1156,2007-06-16,9324.63
Vendor01,1158,2007-06-17,4230.66
Vendor01,1159,2007-06-25,8115.67
Vendor01,1160,2007-07-03,7263.01
Vendor01,1161,2007-07-11,6812.01
Vendor01,1162,2007-07-12,3081.92
Vendor01,1163,2007-07-13,8081.83
Vendor01,1164,2007-07-14,4145.46
Vendor01,1165,2007-07-22,7753.5
Vendor01,1166,2007-07-30,7029.05
Vendor01,1167,2007-08-07,6769.79
Vendor01,1168,2007-08-08,7279.88
Vendor01,1170,2007-08-08,7901.97
Vendor01,1171,2007-08-18,1323.05
Vendor01,1172,2007-08-18,2620.91
Vendor01,1173,2007-08-19,9807.14
Vendor01,1174,2007-08-20,6972.48
Vendor01,1175,2007-08-21,8233.55
Vendor01,1176,2007-08-22,1645.22
Vendor01,1177,2007-08-22,3604.14
Vendor01,1180,2007-09-01,8472.98
Vendor01,1182,2007-09-09,4359.33
Vendor01,1183,2007-09-10,9330.82
Vendor01,1184,2007-09-11,3146.27
Vendor01,1185,2007-09-11,1314.65
Vendor01,1186,2007-09-21,7219.29
Vendor01,1188,2007-09-21,4240.57
Vendor01,1189,2007-09-21,2138.09
Vendor01,1190,2007-10-01,7346.02
Vendor01,1191,2007-10-11,5549.49
Vendor01,1192,2007-10-12,7136.01
Vendor01,1193,2007-10-12,1371.36
Vendor01,1195,2007-10-13,6538.26
Vendor01,1196,2007-10-21,9444.62
Vendor01,1197,2007-10-21,2461.63
Vendor01,1198,2007-10-29,4619.71
Vendor01,1199,2007-11-06,4476.56
Vendor01,1200,2007-11-06,5826.27
Vendor01,1201,2007-11-07,2286.57
Vendor01,1202,2007-11-15,2515.76
Vendor01,1204,2007-11-16,1386.48
Vendor01,1206,2007-11-16,9107.16
Vendor01,1207,2007-11-24,1683.01
Vendor01,1208,2007-11-24,8764.57
Vendor01,1209,2007-11-24,7392.72
Vendor01,1210,2007-11-25,9795.69
Vendor01,1211,2007-12-05,2135.23
Vendor01,1212,2007-12-11,4134.71
Vendor01,1213,2007-12-21,1586.88
Vendor01,1214,2007-12-27,2235.48
Vendor01,1215,2008-01-02,5046.97
Vendor01,1216,2008-01-03,5722.6
Vendor01,1217,2008-01-03,4218.56
Vendor01,1219,2008-01-04,6626.11
Vendor01,1220,2008-01-12,8634.01
Vendor01,1221,2008-01-18,3793.39
Vendor01,1222,2008-01-18,5384.92
Vendor01,1223,2008-01-28,3527.81
Vendor01,1224,2008-01-28,4211.31
Vendor01,1225,2008-01-28,5304.53
Vendor01,1226,2008-02-07,6659.65
Vendor01,1227,2008-02-15,3103.98
Vendor01,1228,2008-02-21,5690.2
Vendor01,1229,2008-02-27,2641.78
Vendor01,1230,2008-02-27,9356.18
Vendor01,1232,2008-03-06,6338.06
Vendor01,1233,2008-03-06,8063.05
Vendor01,1234,2008-03-07,3686.17
Vendor01,1235,2008-03-08,9370.52
Vendor01,1236,2008-03-14,7297.99
Vendor01,1237,2008-03-20,4773.66
Vendor01,1238,2008-03-20,4982.97
Vendor01,1239,2008-03-20,3735.48
Vendor01,1240,2008-03-20,5072.85
Vendor01,1242,2008-03-30,2677.58
Vendor01,1245,2008-03-30,1654.07
Vendor01,1246,2008-03-31,6953.08
Vendor01,1247,2008-04-01,1855.56
Vendor01,1248,2008-04-01,6850.51
Vendor01,1249,2008-04-02,4076.97
Vendor01,1250,2008-04-02,3801.86
Vendor01,1251,2008-04-12,8603.99
Vendor01,1252,2008-04-13,5047.62
Vendor01,1253,2008-04-19,8282.79
Vendor01,1254,2008-04-20,1526.37
Vendor01,1257,2008-04-20,7676.95
Vendor01,1258,2008-04-20,9624.96
Vendor01,1259,2008-04-30,6054.74
Vendor01,1260,2008-04-30,9879.22
Vendor01,1261,2008-05-06,6764.22
Vendor01,1262,2008-05-06,4886.16
Vendor01,1263,2008-05-07,5824.12
Vendor01,1264,2008-05-07,3683.69
Vendor01,1265,2008-05-08,3086.13
Vendor01,1266,2008-05-18,4808.14
Vendor01,1267,2008-05-24,7739.02
Vendor01,1269,2008-05-24,8149.21
Vendor01,1270,2008-05-24,9978.23
Vendor01,1273,2008-05-25,2770.28
Vendor01,1274,2008-05-31,8816.17
Vendor01,1277,2008-05-31,4317.33
Vendor01,1278,2008-06-10,1990.19
Vendor01,1279,2008-06-11,3910.9
Vendor01,1280,2008-06-11,7324.63
Vendor01,1281,2008-06-12,8833.86
Vendor01,1282,2008-06-12,4859.52
Vendor01,1283,2008-06-13,5561.37
Vendor01,1284,2008-06-13,3467.45
Vendor01,1285,2008-06-13,7403.31
Vendor01,1286,2008-06-14,3567.1
Vendor01,1287,2008-06-15,8137.26
Vendor01,1288,2008-06-15,8119.31
Vendor01,1289,2008-06-21,6404.01
Vendor01,1290,2008-06-27,9318.03
Vendor01,1291,2008-06-27,1876.48
Vendor01,1292,2008-07-07,2011.37
Vendor01,1293,2008-07-17,6441.07
Vendor01,1294,2008-07-17,5315.62
Vendor01,1295,2008-07-27,4611.18
Vendor01,1296,2008-07-27,5521.38
Vendor01,1297,2008-07-28,5374.47
Vendor01,1298,2008-07-28,2259.17
Vendor01,1299,2008-07-29,9492.51
Vendor01,1300,2008-07-29,9646.14
Vendor01,1301,2008-08-08,7259.35
Vendor01,1302,2008-08-18,9609.05
Vendor01,1303,2008-08-19,6437.2
Vendor01,1304,2008-08-29,1839.17
Vendor01,1305,2008-09-06,5654.49
Vendor01,1306,2008-09-06,2761.28
Vendor01,1307,2008-09-06,6354.12
Vendor01,1308,2008-09-06,6558.72
Vendor01,1309,2008-09-06,7689.06
Vendor01,1310,2008-09-07,7593.34
Vendor01,1311,2008-09-08,1692.03
Vendor01,1314,2008-09-14,5612.99
Vendor01,1315,2008-09-14,7584.7
Vendor01,1316,2008-09-22,2852.58
Vendor01,1317,2008-09-22,1566.43
Vendor01,1320,2008-09-23,6261.69
Vendor01,1321,2008-09-24,3246.64
Vendor01,1322,2008-09-25,8533.9
Vendor01,1323,2008-10-01,9917.32
Vendor01,1325,2008-10-07,4162.84
Vendor01,1326,2008-10-17,8856.88
Vendor01,1327,2008-10-27,2963.68
Vendor01,1328,2008-10-27,1910.62
Vendor01,1329,2008-11-06,6732.13
Vendor01,1330,2008-11-07,5986.18
Vendor01,1331,2008-11-08,9981.03
Vendor01,1332,2008-11-09,7430.32
Vendor01,1333,2008-11-15,1321.33
Vendor01,1334,2008-11-25,7684.03
Vendor01,1335,2008-11-26,2085.44
Vendor01,1336,2008-12-06,3450.41
Vendor01,1337,2008-12-06,2407.98
Vendor01,1338,2008-12-16,8428.6
Vendor01,1339,2008-12-16,3293.75
Vendor01,1341,2008-12-17,5052.16
Vendor01,1342,2008-12-17,6044.26
Vendor01,1343,2008-12-25,7309.18
Vendor01,1344,2009-01-02,8686.81
Vendor01,1345,2009-01-02,7267.35
Vendor01,1346,2009-01-03,1557.9
Vendor01,1347,2009-01-13,8773.4
Vendor01,1348,2009-01-14,3932.73
Vendor01,1349,2009-01-15,8557.94
Vendor01,1350,2009-01-15,8819.02
Vendor01,1351,2009-01-15,1773.98
Vendor01,1354,2009-01-21,3983.24
Vendor01,1355,2009-01-29,3236.4
Vendor01,1356,2009-02-04,6106.91
Vendor01,1357,2009-02-14,6624.66
Vendor01,1358,2009-02-15,4183.74
Vendor01,1359,2009-02-21,7600.32
Vendor01,1360,2009-03-03,7949.5
Vendor01,1361,2009-03-04,1046.15
Vendor01,1362,2009-03-04,6591.83
Vendor01,1363,2009-03-04,2270.22
Vendor01,1364,2009-03-05,8989.92
Vendor01,1365,2009-03-06,7935.72
Vendor01,1366,2009-03-07,4239.56
Vendor01,1367,2009-03-08,1579.39
Vendor01,1369,2009-03-08,9184.35
Vendor01,1370,2009-03-08,5374.44
Vendor01,1371,2009-03-09,7706.33
Vendor01,1372,2009-03-09,1999.21
Vendor01,1373,2009-03-09,6915.09
Vendor01,1374,2009-03-09,6593.97
Vendor01,1375,2009-03-17,8283.64
Vendor01,1376,2009-03-23,1237.38
Vendor01,1377,2009-03-24,1640.82
Vendor01,1378,2009-04-03,2850.24
Vendor01,1379,2009-04-13,7530.64
Vendor01,1380,2009-04-14,6509.03
Vendor01,1381,2009-04-20,3587.43
Vendor01,1382,2009-04-20,7889.57
Vendor01,1383,2009-04-30,3146.76
Vendor01,1384,2009-05-01,8769.6
Vendor01,1385,2009-05-01,3343.17
Vendor01,1386,2009-05-07,7731.6
Vendor01,1387,2009-05-08,3459.16
Vendor01,1390,2009-05-14,6786.99
Vendor01,1391,2009-05-22,7754.68
Vendor01,1392,2009-05-28,7204.14
Vendor01,1393,2009-05-29,7162.73
Vendor01,1394,2009-06-06,8686.13
Vendor01,1395,2009-06-14,3399.21
Vendor01,1396,2009-06-14,7458.19
Vendor01,1397,2009-06-14,5662.08
Vendor01,1398,2009-06-15,6900.61
Vendor01,1399,2009-06-25,9255.53
Vendor01,1402,2009-07-03,1591.2
Vendor01,1403,2009-07-04,7049.87
Vendor01,1404,2009-07-12,9631.83
Vendor01,1405,2009-07-12,1745.63
Vendor01,1406,2009-07-20,6385.03
Vendor01,1407,2009-07-20,8068.03
Vendor01,1408,2009-07-28,3944.4
Vendor01,1409,2009-07-29,2143.66
Vendor01,1410,2009-07-30,5896.4
Vendor01,1411,2009-08-05,6366.38
Vendor01,1414,2009-08-15,1654.3
Vendor01,1415,2009-08-23,3138.35
Vendor01,1416,2009-08-31,3778.93
Vendor01,1417,2009-09-01,2333.91
Vendor01,1418,2009-09-02,7942.91
Vendor01,1419,2009-09-02,8141.58
Vendor01,1420,2009-09-03,8607.29
Vendor01,1421,2009-09-04,4747.79
Vendor01,1422,2009-09-14,1848.07
Vendor01,1423,2009-09-14,8479.79
Vendor01,1424,2009-09-14,6970.09
Vendor01,1425,2009-09-14,7904.14
Vendor01,1426,2009-09-14,9127.54
Vendor01,1427,2009-09-15,5458.76
Vendor01,1428,2009-09-16,8535.36
Vendor01,1429,2009-09-17,7429.61
Vendor01,1431,2009-09-17,7368.74
Vendor01,1432,2009-09-17,4810.08
Vendor01,1433,2009-09-17,9650.11
Vendor01,1434,2009-09-23,8557.97
Vendor01,1435,2009-09-24,5185.56
Vendor01,1436,2009-09-25,9445.13
Vendor01,1437,2009-10-05,8548.64
Vendor01,1438,2009-10-05,8315.6
Vendor01,1439,2009-10-05,1824.22
Vendor01,1440,2009-10-13,3822.7
Vendor01,1441,2009-10-19,5800.2
Vendor01,1442,2009-10-19,4233.91
Vendor01,1443,2009-10-20,7560.96
Vendor01,1446,2009-10-26,4060.61
Vendor01,1447,2009-11-05,9537.65
Vendor01,1448,2009-11-06,8468.2
Vendor01,1449,2009-11-12,1792.77
Vendor01,1450,2009-11-18,6786.1
Vendor01,1451,2009-11-28,3138.22
Vendor01,1452,2009-12-08,4142.4
Vendor01,1453,2009-12-08,8172.16
Vendor01,1454,2009-12-09,4985.61
Vendor01,1455,2009-12-10,8558.55
Vendor01,1456,2009-12-11,7976.31
Vendor01,1457,2009-12-12,8451.75
Vendor01,1458,2009-12-18,6165.72
Vendor01,1459,2009-12-24,4059.18
Vendor01,1460,2009-12-24,5921.35
Vendor01,1461,2009-12-24,6654.38
Vendor01,1464,2009-12-25,8913.62
Vendor01,1465,2009-12-25,8799.48
Vendor01,1466,2009-12-26,3306.47
Vendor01,1467,2010-01-01,1091.02
Vendor01,1468,2010-01-11,7367.79
Vendor01,1469,2010-01-12,8090.36
Vendor01,1472,2010-01-18,8829.61
Vendor01,1473,2010-01-18,6609.01
Vendor01,1474,2010-01-19,8901.67
Vendor01,1475,2010-01-25,2411.06
Vendor01,1476,2010-01-25,4619.43
Vendor01,1477,2010-02-02,6242.62
Vendor01,1478,2010-02-08,9054.66
Vendor01,1479,2010-02-08,9734.28
Vendor01,1480,2010-02-09,7581.18
Vendor01,1482,2010-02-10,2711.79
Vendor01,1483,2010-02-18,6700.35
Vendor01,1484,2010-02-19,4285.65
Vendor01,1485,2010-02-25,3538.4
Vendor01,1486,2010-03-03,5232.18
Vendor01,1487,2010-03-03,5603.95
Vendor01,1488,2010-03-03,5621.47
Vendor01,1489,2010-03-03,7882.33
Vendor01,1490,2010-03-11,7777.75
Vendor01,1491,2010-03-12,3628.91
Vendor01,1492,2010-03-12,1233.4
Vendor01,1493,2010-03-13,9372.94
Vendor01,1495,2010-03-13,8515.21
Vendor01,1496,2010-03-19,4295.66
Vendor01,1497,2010-03-27,3222.67
Vendor01,1498,2010-03-28,5436.72
Vendor01,1499,2010-04-05,3756.04
Vendor01,1500,2010-04-15,8135.51
Vendor01,1501,2010-04-25,6124.69
Vendor01,1502,2010-04-25,2810.56
Vendor01,1504,2010-05-01,9174.44
Vendor01,1505,2010-05-11,1519.21
Vendor01,1506,2010-05-21,4953.7
Vendor01,1507,2010-05-29,7264.47
Vendor01,1508,2010-05-29,5804.92
Vendor01,1509,2010-05-29,9221.24
Vendor01,1510,2010-05-30,9542.06
Vendor01,1511,2010-06-05,2828.65
Vendor01,1512,2010-06-06,7726.9
Vendor01,1513,2010-06-06,5277.13
Vendor01,1514,2010-06-16,8282.29
Vendor01,1515,2010-06-16,3589.21
Vendor01,1516,2010-06-24,7974.62
Vendor01,1517,2010-06-24,6273.67
Vendor01,1518,2010-07-04,8116.6
Vendor01,1519,2010-07-04,9970.64
Vendor01,1520,2010-07-04,3575.76
Vendor01,1521,2010-07-04,5159.58
Vendor01,1522,2010-07-12,9344.01
Vendor01,1523,2010-07-18,5182.09
Vendor01,1524,2010-07-19,9982.4
Vendor01,1526,2010-07-27,6098.55
Vendor01,1527,2010-07-27,8975.28
Vendor01,1528,2010-08-04,4670.78
Vendor01,1529,2010-08-10,7888.3
Vendor01,1530,2010-08-16,9467.7
Vendor01,1531,2010-08-17,9716.54
Vendor01,1532,2010-08-17,8787.26
Vendor01,1533,2010-08-25,5784.89
Vendor01,1534,2010-08-26,2774.57
Vendor01,1535,2010-09-01,9208.05
Vendor01,1537,2010-09-02,4527.88
Vendor01,1538,2010-09-02,4688.93
Vendor01,1539,2010-09-02,4058.99
Vendor01,1540,2010-09-12,4055.15
Vendor01,1541,2010-09-12,9701.12
Vendor01,1544,2010-09-20,4150.35
Vendor01,1545,2010-09-21,7071.96
Vendor01,1546,2010-09-27,2441.39
Vendor01,1549,2010-10-05,3096.6
Vendor01,1550,2010-10-13,1393.19
Vendor01,1553,2010-10-19,9232.83
Vendor01,1554,2010-10-25,8312.88
Vendor01,1555,2010-10-31,4367.24
Vendor01,1556,2010-11-06,4234.26
Vendor01,1557,2010-11-12,1914.82
Vendor01,1558,2010-11-12,9541.58
Vendor01,1559,2010-11-20,2983.46
Vendor01,1560,2010-11-21,3857.92
Vendor01,1561,2010-11-27,5785.04
Vendor01,1562,2010-11-28,5025.29
Vendor01,1563,2010-11-28,8538.88
Vendor01,1564,2010-11-29,8794.56
Vendor01,1565,2010-11-30,8005.18
Vendor01,1566,2010-12-08,3227.58
Vendor01,1567,2010-12-09,5569.47
Vendor01,1568,2010-12-15,6058.41
Vendor01,1569,2010-12-16,3489.36
Vendor01,1571,2010-12-22,4223.36
Vendor01,1572,2010-12-30,4891.46
Vendor01,1573,2010-12-31,6017.58
Vendor01,1574,2011-01-06,1957.33
Vendor01,1577,2011-01-06,6059.96
Vendor01,1578,2011-01-07,8083.89
Vendor01,1579,2011-01-13,4527.74
Vendor01,1580,2011-01-14,7747.46
Vendor01,1581,2011-01-24,7262.15
Vendor01,1582,2011-02-03,8273.91
Vendor01,1583,2011-02-03,2955.37
Vendor01,1584,2011-02-09,2898.86
Vendor01,1585,2011-02-10,4618.01
Vendor01,1586,2011-02-11,6197.22
Vendor01,1587,2011-02-12,6638.96
Vendor01,1588,2011-02-13,4440.5
Vendor01,1589,2011-02-19,8838.66
Vendor01,1590,2011-02-20,1651.94
Vendor01,1591,2011-02-20,5348.09
Vendor01,1592,2011-02-26,3586.11
Vendor01,1593,2011-02-27,8402.78
Vendor01,1594,2011-02-27,1041.94
Vendor01,1595,2011-02-27,1477.3
Vendor01,1596,2011-03-07,3500.24
Vendor01,1597,2011-03-08,6967.62
Vendor01,1598,2011-03-16,9230.7
Vendor01,1599,2011-03-26,6439.34
Vendor01,1600,2011-04-01,2028.94
Vendor01,1601,2011-04-01,6480.62
Vendor01,1604,2011-04-01,4657.12
Vendor01,1605,2011-04-01,6894.96
Vendor01,1606,2011-04-02,5418.18
Vendor01,1607,2011-04-02,8477.84
Vendor01,1609,2011-04-03,2795.1
Vendor01,1610,2011-04-04,2876.61
Vendor01,1611,2011-04-04,1415.21
Vendor01,1612,2011-04-04,2740.75
Vendor01,1613,2011-04-05,7133.19
Vendor01,1614,2011-04-05,7784.62
Vendor01,1615,2011-04-13,5494.47
Vendor01,1616,2011-04-14,6192.43
Vendor01,1619,2011-04-14,5525.37
Vendor01,1620,2011-04-14,1820.63
Vendor01,1621,2011-04-15,1728.72
Vendor01,1623,2011-04-16,9321.15
Vendor01,1624,2011-04-24,6394.35
Vendor01,1625,2011-05-02,2137.12
Vendor01,1626,2011-05-03,2097.67
Vendor01,1628,2011-05-03,7326.25
Vendor01,1629,2011-05-04,9491.17
Vendor01,1630,2011-05-14,9186.77
Vendor01,1631,2011-05-15,1414.12
Vendor01,1632,2011-05-16,9567.74
Vendor01,1633,2011-05-17,2312.13
Vendor01,1634,2011-05-17,7682.78
Vendor01,1635,2011-05-17,7202.9
Vendor01,1636,2011-05-17,3622.18
Vendor01,1637,2011-05-17,6844.7
Vendor01,1638,2011-05-18,3831.71
Vendor01,1639,2011-05-24,7662.15
Vendor01,1642,2011-06-01,4995.44
Vendor01,1643,2011-06-07,3395.79
Vendor01,1646,2011-06-07,6718.02
Vendor01,1647,2011-06-15,5958.79
Vendor01,1648,2011-06-25,8141.51
Vendor01,1650,2011-06-26,8164.83
Vendor01,1651,2011-06-27,1977.73
Vendor01,1652,2011-06-28,1954.48
Vendor01,1653,2011-06-28,6839.43
Vendor01,1654,2011-06-29,3507.02
Vendor01,1655,2011-07-07,9025.11
Vendor01,1656,2011-07-08,1133.81
Vendor01,1657,2011-07-08,1555.07
Vendor01,1658,2011-07-09,1983.07
Vendor01,1659,2011-07-10,2838.4
Vendor01,1660,2011-07-10,4612.77
Vendor01,1661,2011-07-11,4306.37
Vendor01,1662,2011-07-17,8017.3
Vendor01,1663,2011-07-18,2068.28
Vendor01,1665,2011-07-24,9778.34
Vendor01,1666,2011-07-24,4297.08
Vendor01,1667,2011-07-24,9083.65
Vendor01,1668,2011-07-30,9534.35
Vendor01,1669,2011-07-30,7406.88
Vendor01,1670,2011-07-30,4870.7
Vendor01,1671,2011-07-31,9587.3
Vendor01,1672,2011-08-06,1496.11
Vendor01,1673,2011-08-06,5250.62
Vendor01,1674,2011-08-06,8121.95
Vendor01,1677,2011-08-06,4524.88
Vendor01,1678,2011-08-07,1874.46
Vendor01,1680,2011-08-07,7192.5
Vendor01,1681,2011-08-07,5461.72
Vendor01,1682,2011-08-15,6109.21
Vendor01,1683,2011-08-16,5765.75
Vendor01,1684,2011-08-26,4419.89
Vendor01,1687,2011-09-03,2615.43
Vendor01,1688,2011-09-04,6834.3
Vendor01,1689,2011-09-04,2637.27
Vendor01,1690,2011-09-05,6519.86
Vendor01,1691,2011-09-06,9907.33
Vendor01,1692,2011-09-06,9373.27
Vendor01,1693,2011-09-06,8217.67
Vendor01,1694,2011-09-06,1994.31
Vendor01,1695,2011-09-12,5507.99
Vendor01,1696,2011-09-13,1407.34
Vendor01,1697,2011-09-13,1141.81
Vendor01,1698,2011-09-23,9983.76
Vendor01,1699,2011-09-29,7015.15
Vendor01,1700,2011-10-05,6302.04
Vendor01,1701,2011-10-05,6282.62
Vendor01,1704,2011-10-15,2909.23
Vendor01,1705,2011-10-15,8890.22
Vendor01,1706,2011-10-15,8915.16
Vendor01,1707,2011-10-16,7872.7
Vendor01,1708,2011-10-16,4608.22
Vendor01,1709,2011-10-17,4818.71
Vendor01,1710,2011-10-27,4076.53
Vendor01,1711,2011-11-02,6935.83
Vendor01,1712,2011-11-08,5540.54
Vendor01,1715,2011-11-09,8571.38
Vendor01,1716,2011-11-09,6276.22
Vendor01,1717,2011-11-17,2952.92
Vendor01,1718,2011-11-25,9044.05
Vendor01,1719,2011-11-25,4660.6
Vendor01,1720,2011-12-01,1043.88
Vendor01,1721,2011-12-09,9880.82
Vendor01,1722,2011-12-09,4764.14
Vendor01,1723,2011-12-17,2496.43
Vendor01,1724,2011-12-17,1384.3
Vendor01,1725,2011-12-17,9065.11
Vendor01,1726,2011-12-17,6585.39
Vendor01,1727,2011-12-18,6924.53
Vendor01,1728,2011-12-19,9953.27
Vendor01,1729,2011-12-20,1682.71
Vendor01,1731,2011-12-20,6816.5
Vendor01,1733,2011-12-20,7298.47
Vendor01,1734,2011-12-20,5595.28
Vendor01,1735,2011-12-28,2586.26
Vendor01,1736,2011-12-28,3586.2
Vendor01,1737,2012-01-07,9913.55
Vendor01,1738,2012-01-17,3263.82
Vendor01,1739,2012-01-17,2175.92
Vendor01,1740,2012-01-17,9064.27
Vendor01,1741,2012-01-17,2872.48
Vendor01,1742,2012-01-17,7305.57
Vendor01,1743,2012-01-27,1198.02
Vendor01,1744,2012-02-02,1669.85
Vendor01,1745,2012-02-10,4049.1
Vendor01,1746,2012-02-18,6768.83
Vendor01,1747,2012-02-18,7107.64
Vendor01,1748,2012-02-19,6572.61
Vendor01,1749,2012-02-25,4462.34
Vendor01,1750,2012-02-25,8044.07
Vendor01,1751,2012-02-25,6272.71
Vendor01,1752,2012-03-06,2225.9
Vendor01,1753,2012-03-07,9340.34
Vendor01,1754,2012-03-07,5748.04
Vendor01,1755,2012-03-07,2535.73
Vendor01,1756,2012-03-17,6792.29
Vendor01,1757,2012-03-25,7342.55
Vendor01,1758,2012-04-04,9677.89
Vendor01,1759,2012-04-10,7037.92
Vendor01,1760,2012-04-10,7402.19
Vendor01,1761,2012-04-16,8929.82
Vendor01,1762,2012-04-22,1029.71
Vendor01,1763,2012-04-23,9611.98
Vendor01,1764,2012-04-23,6459.25
Vendor01,1767,2012-05-03,8183.82
Vendor01,1768,2012-05-04,2791.84
Vendor01,1770,2012-05-05,7910.21
Vendor01,1771,2012-05-11,2180.47
Vendor01,1772,2012-05-11,5448.73
Vendor01,1773,2012-05-12,9893.35
Vendor01,1774,2012-05-13,5395.58
Vendor01,1775,2012-05-21,5184.75
Vendor01,1776,2012-05-27,4853.87
Vendor01,1777,2012-06-04,8124.19
Vendor01,1779,2012-06-05,4650.42
Vendor01,1780,2012-06-06,3331.45
Vendor01,1782,2012-06-06,5729.31
Vendor01,1783,2012-06-14,2868.95
Vendor01,1784,2012-06-20,7830.15
Vendor01,1785,2012-06-21,7006.5
Vendor01,1786,2012-06-22,7281.86
Vendor01,1787,2012-06-28,1456.39
Vendor01,1788,2012-06-29,9099.35
Vendor01,1789,2012-07-07,7689.09
Vendor01,1790,2012-07-07,8265.47
Vendor01,1792,2012-07-15,7220.38
Vendor01,1793,2012-07-16,8888.16
Vendor01,1794,2012-07-17,7928.51
Vendor01,1795,2012-07-23,4019.72
Vendor01,1796,2012-07-23,6647.89
Vendor01,1797,2012-07-29,3158.31
Vendor01,1798,2012-08-06,1217.47
Vendor01,1799,2012-08-06,7408.22
Vendor01,1800,2012-08-12,1901.55
Vendor01,1801,2012-08-12,7846.57
Vendor01,1802,2012-08-12,2556.16
Vendor01,1803,2012-08-12,1234.74
Vendor01,1804,2012-08-12,1493.82
Vendor01,1805,2012-08-12,4475.78
Vendor01,1806,2012-08-12,4726.43
Vendor01,1807,2012-08-13,8833.37
Vendor01,1808,2012-08-14,8840.53
Vendor01,1809,2012-08-15,4031.11
Vendor01,1812,2012-08-16,6595.94
Vendor01,1813,2012-08-16,1655.78
Vendor01,1814,2012-08-26,9461.71
Vendor01,1815,2012-08-27,3641.27
Vendor01,1816,2012-08-27,2652.18
Vendor01,1817,2012-09-06,6518.78
Vendor01,1818,2012-09-12,5371.49
Vendor01,1819,2012-09-13,8092.66
Vendor01,1820,2012-09-21,9605.35
Vendor01,1821,2012-09-27,9203.51
Vendor01,1824,2012-09-27,3310.93
Vendor01,1825,2012-10-03,2564.62
Vendor01,1826,2012-10-09,2335.27
Vendor01,1828,2012-10-15,4935.06
Vendor01,1829,2012-10-16,2644.12
Vendor01,1830,2012-10-16,7627.99
Vendor01,1831,2012-10-22,7332.92
Vendor01,1832,2012-10-22,3292.62
Vendor01,1834,2012-10-22,3634.27
Vendor01,1835,2012-10-22,3729.83
Vendor01,1836,2012-11-01,8833.18
Vendor01,1837,2012-11-02,1742.04
Vendor01,1838,2012-11-03,2997.61
Vendor01,1839,2012-11-03,1721.93
Vendor01,1840,2012-11-04,4361.96
Vendor01,1841,2012-11-10,4348.88
Vendor01,1842,2012-11-11,2471.49
Vendor01,1843,2012-11-12,7603.22
Vendor01,1844,2012-11-12,8615.41
Vendor01,1845,2012-11-22,8960.49
Vendor01,1846,2012-11-23,8602.24
Vendor01,1847,2012-12-03,1082.94
Vendor01,1848,2012-12-03,8185.45
Vendor01,1849,2012-12-09,7113.02
Vendor01,1850,2012-12-15,7222.67
Vendor01,1851,2012-12-16,5453.88
Vendor01,1852,2012-12-16,6570.74
Vendor01,1853,2012-12-24,4999.06
Vendor01,1854,2013-01-01,3948.07
Vendor01,1855,2013-01-11,6552.25
Vendor01,1856,2013-01-17,4261.36
Vendor01,1857,2013-01-17,7667.57
Vendor01,1858,2013-01-23,6187.62
Vendor01,1859,2013-01-24,3069.48
Vendor01,1860,2013-01-30,7826.99
Vendor01,1861,2013-02-09,5503.5
Vendor01,1862,2013-02-10,5902.95
Vendor01,1863,2013-02-18,7363.98
Vendor01,1864,2013-02-24,9653.66
Vendor01,1865,2013-03-04,8455.33
Vendor01,1866,2013-03-12,2957.29
Vendor01,1867,2013-03-12,4911.62
Vendor01,1868,2013-03-12,3650.28
Vendor01,1870,2013-03-13,7759.92
Vendor01,1871,2013-03-21,4302.2
Vendor01,1872,2013-03-21,8255.53
Vendor01,1873,2013-03-22,2703.35
Vendor01,1874,2013-03-30,1945.45
Vendor01,1875,2013-03-31,3062.22
Vendor01,1876,2013-04-06,4877.44
Vendor01,1877,2013-04-12,8745.88
Vendor01,1878,2013-04-13,3005.84
Vendor01,1879,2013-04-14,3271.88
Vendor01,1881,2013-04-15,2707.39
Vendor01,1883,2013-04-15,5176.54
Vendor01,1884,2013-04-23,1615.62
Vendor01,1885,2013-04-24,8476.26
Vendor01,1886,2013-04-25,4490.28
Vendor01,1887,2013-04-26,8931.74
Vendor01,1889,2013-05-02,2688.79
Vendor01,1890,2013-05-02,5296.06
Vendor01,1891,2013-05-12,3721.84
Vendor01,1893,2013-05-12,9768.52
Vendor01,1894,2013-05-13,1821.2
Vendor01,1895,2013-05-19,9552.32
Vendor01,1896,2013-05-25,1585.23
Vendor01,1899,2013-05-26,5967.53
Vendor01,1900,2013-06-05,2089.77
Vendor01,1901,2013-06-06,6342.83
Vendor01,1902,2013-06-12,8715.99
Vendor01,1903,2013-06-13,7459.8
Vendor01,1904,2013-06-19,5902.46
Vendor01,1905,2013-06-20,6674.55
Vendor01,1906,2013-06-20,1487.67
Vendor01,1908,2013-06-21,6838.96
Vendor01,1909,2013-06-21,2695.62
Vendor01,1910,2013-06-27,5249.09
Vendor01,1911,2013-07-03,5970.94
Vendor01,1912,2013-07-03,9817.91
Vendor01,1913,2013-07-03,9735.56
Vendor01,1914,2013-07-09,5709.81
Vendor01,1915,2013-07-09,4289.58
Vendor01,1916,2013-07-09,4885.53
Vendor01,1917,2013-07-19,8056.78
Vendor01,1918,2013-07-27,3771.91
Vendor01,1919,2013-07-28,9843.44
Vendor01,1920,2013-07-29,1312.74
Vendor01,1921,2013-07-29,5573.85
Vendor01,1922,2013-08-08,2544.46
Vendor01,1923,2013-08-08,1366.88
Vendor01,1924,2013-08-09,6026.48
Vendor01,1925,2013-08-15,9894.49
Vendor01,1926,2013-08-25,6916.52
Vendor01,1928,2013-08-26,7368.24
Vendor01,1930,2013-08-27,8093.77
Vendor01,1931,2013-08-27,7706.76
Vendor01,1932,2013-08-27,9198.35
Vendor01,1934,2013-08-27,9623.62
Vendor01,1935,2013-08-28,9871.8
Vendor01,1936,2013-09-03,2809.75
Vendor01,1937,2013-09-13,9860.04
Vendor01,1938,2013-09-13,5737.75
Vendor01,1939,2013-09-13,7413.71
Vendor01,1940,2013-09-13,8928.33
Vendor01,1941,2013-09-14,2941.72
Vendor01,1942,2013-09-20,3396.73
Vendor01,1943,2013-09-30,3723.43
Vendor01,1944,2013-10-06,2782.53
Vendor01,1945,2013-10-06,3989.82
Vendor01,1946,2013-10-06,3214.64
Vendor01,1947,2013-10-14,1395.88
Vendor01,1948,2013-10-20,8249.17
Vendor01,1949,2013-10-21,1428.68
Vendor01,1950,2013-10-21,7039.85
Vendor01,1951,2013-10-21,4878.68
Vendor01,1952,2013-10-27,6206.63
Vendor01,1953,2013-10-27,1879.92
Vendor01,1954,2013-10-27,8493.03
Vendor01,1955,2013-10-28,1645.17
Vendor01,1956,2013-10-29,8327.53
Vendor01,1957,2013-10-30,3529.15
Vendor01,1958,2013-10-30,9060.09
Vendor01,1959,2013-11-05,2013.25
Vendor01,1960,2013-11-11,9082.7
Vendor01,1961,2013-11-12,5897.74
Vendor01,1962,2013-11-12,5566.02
Vendor01,1963,2013-11-13,2455.67
Vendor01,1965,2013-11-19,2322.06
Vendor01,1966,2013-11-19,3426.64
Vendor01,1967,2013-11-19,3128.08
Vendor01,1968,2013-11-19,5311.4
Vendor01,1969,2013-11-20,4197.34
Vendor01,1970,2013-11-21,9919.92
Vendor01,1971,2013-11-21,4227.47
Vendor01,1972,2013-11-22,6197.89
Vendor01,1973,2013-11-28,6531.79
Vendor01,1975,2013-11-28,3055.59
Vendor01,1976,2013-12-04,6680.83
Vendor01,1977,2013-12-04,6145.59
Vendor01,1978,2013-12-12,6630.1
Vendor01,1979,2013-12-13,8677.1
Vendor01,1980,2013-12-14,7534.2
Vendor01,1981,2013-12-14,3527.55
Vendor01,1982,2013-12-14,2060.46
Vendor01,1983,2013-12-20,3912.99
Vendor01,1984,2013-12-28,4729.25
Vendor01,1985,2014-01-07,3106.77
Vendor01,1986,2014-01-07,9091.81
Vendor01,1987,2014-01-08,4901.46
Vendor01,1988,2014-01-09,8404.79
Vendor01,1989,2014-01-10,1961.19
Vendor01,1990,2014-01-16,8744.77
Vendor01,1991,2014-01-26,3486.13
Vendor01,1992,2014-02-05,7553.04
Vendor01,1993,2014-02-06,9402.99
Vendor01,1994,2014-02-06,4804.59
Vendor01,1995,2014-02-07,9204.86
Vendor01,1996,2014-02-15,5227.62
Vendor01,1997,2014-02-16,5077.8
Vendor01,1998,2014-02-17,2906.12
Vendor01,1999,2014-02-18,7535.15
Vendor01,2001,2014-02-24,2518.84
Vendor01,2002,2014-02-25,1030.98
Vendor01,2003,2014-02-26,1977.53
Vendor01,2004,2014-03-04,8153.3
Vendor01,2005,2014-03-04,5204.59
Vendor01,2007,2014-03-05,6237.36
Vendor01,2008,2014-03-06,5719.85
Vendor01,2009,2014-03-07,8955.68
Vendor01,2010,2014-03-07,9194.77
Vendor01,2011,2014-03-07,3699.87
Vendor01,2012,2014-03-17,4380.93
Vendor01,2013,2014-03-17,6930.55
Vendor01,2015,2014-03-23,8410.96
Vendor01,2016,2014-03-23,6772.65
Vendor01,2019,2014-03-24,1022.65
Vendor01,2020,2014-03-25,6255.43
Vendor01,2021,2014-04-04,3476.55
Vendor01,2022,2014-04-04,7695.47
Vendor01,2023,2014-04-04,5145.64
Vendor01,2024,2014-04-04,3381.09
Vendor01,2025,2014-04-04,9463.48
Vendor01,2026,2014-04-04,2337.55
Vendor01,2027,2014-04-10,2049.05
Vendor01,2028,2014-04-10,8119.4
Vendor01,2029,2014-04-11,4518.47
Vendor01,2030,2014-04-21,6192.6
Vendor01,2031,2014-04-21,6570.95
Vendor01,2033,2014-04-22,9826.87
Vendor01,2034,2014-05-02,5997.17
Vendor01,2035,2014-05-08,7965.93
Vendor01,2038,2014-05-09,1357.7
Vendor01,2039,2014-05-10,4135.4
Vendor01,2040,2014-05-18,4205.01
Vendor01,2041,2014-05-18,4852.48
Vendor01,2042,2014-05-28,4925.95
Vendor01,2043,2014-05-29,7281.19
Vendor01,2044,2014-05-29,2310.54
Vendor01,2045,2014-06-04,2390.86
Vendor01,2046,2014-06-04,9064.53
Vendor01,2047,2014-06-10,5946.07
Vendor07,8,2006-01-01,1014.14
Vendor07,14,2006-01-02,6029.25
Vendor07,18,2006-01-12,7528.88
Vendor07,23,2006-01-12,7368.73
Vendor07,29,2006-01-12,8580.51
Vendor07,32,2006-01-12,7442.71
Vendor07,36,2006-01-12,1369.86
Vendor07,42,2006-01-12,3323.41
Vendor07,47,2006-01-13,1219.99
Vendor07,51,2006-01-14,2211.84
Vendor07,60,2006-01-14,4577.15
Vendor07,65,2006-01-24,1776.87
Vendor07,69,2006-01-25,1061.12
Vendor07,73,2006-01-26,3472.97
Vendor07,77,2006-01-27,4159.8
Vendor07,81,2006-02-04,1888.98
Vendor07,88,2006-02-14,9112.54
Vendor07,93,2006-02-24,2973.85
Vendor07,97,2006-02-24,4908.43
Vendor07,101,2006-03-06,3333.82
Vendor07,110,2006-03-16,1165.33
Vendor07,112,2006-03-17,1125.47
Vendor07,116,2006-03-17,7631.98
Vendor07,125,2006-03-23,1040.82
Vendor07,126,2006-03-24,4537.73
Vendor07,134,2006-03-25,3201.07
Vendor07,138,2006-04-02,5723.02
Vendor07,143,2006-04-02,4503.87
Vendor07,148,2006-04-02,2975.02
Vendor07,151,2006-04-03,1791.63
Vendor07,159,2006-04-03,1704.57
Vendor07,161,2006-04-04,2138.12
Vendor07,167,2006-04-10,2364.91
Vendor07,171,2006-04-10,1447.87
Vendor07,180,2006-04-10,8072.82
Vendor07,184,2006-04-10,1443.26
Vendor07,189,2006-04-11,2170.75
Vendor07,191,2006-04-17,2701.28
Vendor07,199,2006-04-17,5187.26
Vendor07,201,2006-04-17,5081.3
Vendor07,208,2006-04-23,2410.47
Vendor07,214,2006-04-24,2777.1
Vendor07,220,2006-04-30,2607.94
Vendor07,223,2006-04-30,4638.31
Vendor07,228,2006-05-08,1309.61
Vendor07,235,2006-05-09,1719.18
Vendor07,239,2006-05-10,1292.03
Vendor07,244,2006-05-16,1542.87
Vendor07,248,2006-05-22,1821.42
Vendor07,255,2006-05-22,6839.84
Vendor07,259,2006-05-23,1027.42
Vendor07,261,2006-05-23,1130.15
Vendor07,267,2006-05-23,1705.56
Vendor07,271,2006-05-23,8450.78
Vendor07,277,2006-05-24,8843.1
Vendor07,281,2006-05-24,1171.96
Vendor07,289,2006-05-30,1300.67
Vendor07,291,2006-06-07,7340.29
Vendor07,297,2006-06-13,2612.96
Vendor07,304,2006-06-14,2735.54
Vendor07,306,2006-06-20,2413.67
Vendor07,314,2006-06-28,6446.77
Vendor07,320,2006-06-28,4840.07
Vendor07,322,2006-06-29,1285.19
Vendor07,326,2006-07-05,3026.8
Vendor07,334,2006-07-15,1784.8
Vendor07,340,2006-07-15,3941.41
Vendor07,341,2006-07-15,2240.56
Vendor07,349,2006-07-16,1938.29
Vendor07,355,2006-07-26,2734.38
Vendor07,358,2006-07-26,2184.42
Vendor07,363,2006-07-26,2426.13
Vendor07,367,2006-08-03,1956.67
Vendor07,374,2006-08-04,4522.71
Vendor07,377,2006-08-04,1083.6
Vendor07,385,2006-08-10,2849.48
Vendor07,388,2006-08-16,5784.31
Vendor07,392,2006-08-17,2070.18
Vendor07,397,2006-08-17,1088.38
Vendor07,405,2006-08-23,6960.39
Vendor07,408,2006-08-29,4148.65
Vendor07,411,2006-09-04,7382.67
Vendor07,417,2006-09-04,2248.69
Vendor07,421,2006-09-10,7410.17
Vendor07,427,2006-09-10,4343.0
Vendor07,432,2006-09-10,1732.92
Vendor07,440,2006-09-20,1257.27
Vendor07,441,2006-09-21,3082.03
Vendor07,446,2006-09-21,1852.51
Vendor07,455,2006-09-21,1255.65
Vendor07,457,2006-09-21,4435.7
Vendor07,461,2006-09-22,3800.42
Vendor07,469,2006-09-28,3088.66
Vendor07,473,2006-09-29,3652.28
Vendor07,480,2006-09-29,1916.65
Vendor07,484,2006-09-29,3198.96
Vendor07,487,2006-09-30,2563.18
Vendor07,493,2006-09-30,5136.67
Vendor07,496,2006-10-08,8275.35
Vendor07,502,2006-10-14,3917.69
Vendor07,510,2006-10-15,4830.95
Vendor07,511,2006-10-23,5111.71
Vendor07,520,2006-10-24,1692.39
Vendor07,523,2006-10-24,1748.04
Vendor07,529,2006-10-24,7767.26
Vendor07,533,2006-10-24,1055.65
Vendor07,539,2006-10-30,3829.5
Vendor07,543,2006-10-31,1483.04
Vendor07,549,2006-11-06,4099.11
Vendor07,554,2006-11-06,1713.32
Vendor07,556,2006-11-07,6969.51
Vendor07,561,2006-11-07,2075.45
Vendor07,568,2006-11-15,6307.01
Vendor07,574,2006-11-21,6601.49
Vendor07,579,2006-11-27,1053.95
Vendor07,583,2006-11-27,1132.2
Vendor07,590,2006-11-28,9753.19
Vendor07,593,2006-11-28,4882.15
Vendor07,599,2006-11-28,4509.7
Vendor07,603,2006-12-04,3629.5
Vendor07,609,2006-12-12,1033.56
Vendor07,611,2006-12-18,1024.74
Vendor07,617,2006-12-18,2204.59
Vendor07,621,2006-12-18,6632.31
Vendor07,629,2006-12-19,8740.64
Vendor07,632,2006-12-29,4991.74
Vendor07,637,2007-01-08,1134.67
Vendor07,644,2007-01-14,5665.69
Vendor07,647,2007-01-20,1654.61
Vendor07,655,2007-01-26,1631.25
Vendor07,658,2007-01-26,5412.13
Vendor07,665,2007-02-03,9677.21
Vendor07,670,2007-02-03,1721.97
Vendor07,674,2007-02-13,2965.87
Vendor07,677,2007-02-19,8525.91
Vendor07,682,2007-02-25,5060.95
Vendor07,688,2007-03-03,1453.46
Vendor07,693,2007-03-13,2851.82
Vendor07,696,2007-03-14,4004.88
Vendor07,704,2007-03-20,4619.11
Vendor07,710,2007-03-21,6955.1
Vendor07,711,2007-03-22,1553.37
Vendor07,720,2007-03-23,3926.4
Vendor07,723,2007-04-02,2465.18
Vendor07,727,2007-04-08,3359.13
Vendor07,732,2007-04-09,9822.55
Vendor07,737,2007-04-09,1522.15
Vendor07,744,2007-04-09,8165.08
Vendor07,747,2007-04-09,7006.5
Vendor07,755,2007-04-19,8098.67
Vendor07,757,2007-04-20,3443.25
Vendor07,762,2007-04-26,1930.74
Vendor07,766,2007-05-02,1462.57
Vendor07,774,2007-05-10,1703.44
Vendor07,780,2007-05-20,1446.53
Vendor07,781,2007-05-21,6897.05
Vendor07,789,2007-05-21,2125.0
Vendor07,793,2007-05-22,2203.03
Vendor07,796,2007-05-22,6485.69
Vendor07,801,2007-05-22,5909.47
Vendor07,810,2007-05-28,1860.46
Vendor07,814,2007-06-07,2944.72
Vendor07,817,2007-06-07,1734.79
Vendor07,822,2007-06-13,1812.17
Vendor07,828,2007-06-19,2003.37
Vendor07,831,2007-06-19,5585.4
Vendor07,837,2007-06-27,1644.97
Vendor07,845,2007-07-05,1152.7
Vendor07,846,2007-07-11,8855.52
Vendor07,855,2007-07-11,6218.47
Vendor07,860,2007-07-11,3160.42
Vendor07,862,2007-07-21,9857.81
Vendor07,870,2007-07-31,3237.37
Vendor07,872,2007-08-01,1834.24
Vendor07,879,2007-08-07,4296.95
Vendor07,885,2007-08-13,4966.54
Vendor07,886,2007-08-14,1145.51
Vendor07,893,2007-08-24,6425.17
Vendor07,897,2007-08-25,5628.83
Vendor07,905,2007-09-04,5312.01
Vendor07,907,2007-09-04,3614.37
Vendor07,914,2007-09-14,5225.01
Vendor07,916,2007-09-15,1676.5
Vendor07,923,2007-09-16,1963.53
Vendor07,927,2007-09-24,2777.07
Vendor07,933,2007-09-24,1150.85
Vendor07,938,2007-09-25,1260.87
Vendor07,942,2007-09-26,3544.46
Vendor07,948,2007-10-04,1455.94
Vendor07,951,2007-10-04,1371.39
Vendor07,959,2007-10-04,3312.72
Vendor07,963,2007-10-05,1349.32
Vendor07,968,2007-10-13,1075.88
Vendor07,975,2007-10-19,1333.43
Vendor07,977,2007-10-29,2033.44
Vendor07,983,2007-10-29,8722.44
Vendor07,988,2007-10-29,6193.13
Vendor07,994,2007-10-30,3477.46
Vendor07,996,2007-11-09,8968.58
Vendor07,1003,2007-11-10,5239.48
Vendor07,1006,2007-11-20,2533.7
Vendor07,1012,2007-11-26,1974.86
Vendor07,1018,2007-12-02,4775.85
Vendor07,1021,2007-12-08,5969.67
Vendor07,1030,2007-12-08,7732.94
Vendor07,1034,2007-12-16,3253.93
Vendor07,1038,2007-12-16,8180.8
Vendor07,1045,2007-12-22,1677.75
Vendor07,1050,2007-12-23,2789.22
Vendor07,1053,2007-12-24,1384.03
Vendor07,1058,2007-12-30,9769.87
Vendor07,1063,2007-12-30,1450.51
Vendor07,1069,2007-12-30,7313.4
Vendor07,1073,2007-12-31,1500.69
Vendor07,1076,2008-01-01,5266.6
Vendor07,1085,2008-01-01,1233.15
Vendor07,1087,2008-01-11,7119.78
Vendor07,1092,2008-01-11,9227.63
Vendor07,1098,2008-01-11,1278.29
Vendor07,1102,2008-01-12,1486.21
Vendor07,1110,2008-01-18,8362.73
Vendor07,1113,2008-01-24,2916.6
Vendor07,1117,2008-01-25,2184.51
Vendor07,1121,2008-01-25,1820.55
Vendor07,1129,2008-01-26,1516.18
Vendor07,1131,2008-02-05,1069.54
Vendor07,1140,2008-02-06,2670.08
Vendor07,1141,2008-02-07,7234.59
Vendor07,1149,2008-02-07,3017.05
Vendor07,1152,2008-02-17,4919.59
Vendor07,1160,2008-02-17,9810.75
Vendor07,1161,2008-02-27,1343.03
Vendor07,1169,2008-02-28,1299.73
Vendor07,1171,2008-02-28,1953.98
Vendor07,1179,2008-02-28,3364.01
Vendor07,1183,2008-03-07,4745.74
Vendor07,1189,2008-03-13,2225.78
Vendor07,1194,2008-03-23,1691.08
Vendor07,1199,2008-03-29,1185.89
Vendor07,1202,2008-04-04,3457.53
Vendor07,1209,2008-04-04,2367.47
Vendor07,1213,2008-04-04,4257.41
Vendor07,1216,2008-04-04,2482.33
Vendor07,1222,2008-04-10,2671.28
Vendor07,1230,2008-04-10,4265.18
Vendor07,1231,2008-04-10,3033.42
Vendor07,1236,2008-04-20,5225.96
Vendor07,1243,2008-04-21,3744.25
Vendor07,1249,2008-04-27,1017.93
Vendor07,1254,2008-04-27,9153.2
Vendor07,1256,2008-04-28,3436.46
Vendor07,1265,2008-05-08,8973.56
Vendor07,1268,2008-05-08,2788.07
Vendor07,1275,2008-05-14,5168.33
Vendor07,1280,2008-05-14,3797.31
Vendor07,1284,2008-05-15,6900.31
Vendor07,1290,2008-05-21,3086.31
Vendor07,1292,2008-05-27,1680.55
Vendor07,1298,2008-06-06,5914.91
Vendor07,1305,2008-06-07,8698.74
Vendor07,1306,2008-06-13,2319.71
Vendor07,1311,2008-06-13,9388.15
Vendor07,1318,2008-06-23,2368.93
Vendor07,1321,2008-07-01,3276.9
Vendor07,1326,2008-07-02,2403.4
Vendor07,1333,2008-07-03,2095.37
Vendor07,1337,2008-07-03,7052.55
Vendor07,1344,2008-07-04,1880.0
Vendor07,1350,2008-07-05,1756.73
Vendor07,1351,2008-07-06,2273.76
Vendor07,1358,2008-07-14,3010.76
Vendor07,1361,2008-07-15,1400.76
Vendor07,1370,2008-07-16,3032.34
Vendor07,1375,2008-07-24,1799.5
Vendor07,1376,2008-07-30,5123.52
Vendor07,1382,2008-07-31,5127.98
Vendor07,1386,2008-08-10,3295.2
Vendor07,1392,2008-08-10,1407.73
Vendor07,1396,2008-08-18,2064.6
Vendor07,1401,2008-08-18,6602.36
Vendor07,1409,2008-08-18,4368.02
Vendor07,1414,2008-08-28,2725.55
Vendor07,1419,2008-08-29,6091.34
Vendor07,1421,2008-09-04,1344.21
Vendor07,1427,2008-09-04,1995.23
Vendor07,1434,2008-09-14,7561.93
Vendor07,1436,2008-09-14,8156.66
Vendor07,1444,2008-09-20,1731.75
Vendor07,1447,2008-09-20,3016.41
Vendor07,1451,2008-09-21,3145.08
Vendor07,1458,2008-10-01,3670.45
Vendor07,1462,2008-10-01,1152.08
Vendor07,1467,2008-10-11,7664.1
Vendor07,1475,2008-10-19,3654.54
Vendor07,1476,2008-10-25,1302.38
Vendor07,1483,2008-10-26,2727.24
Vendor07,1490,2008-10-27,2386.72
Vendor07,1495,2008-10-28,4770.11
Vendor07,1500,2008-11-03,5480.38
Vendor07,1502,2008-11-03,5013.85
Vendor07,1507,2008-11-04,9762.07
Vendor07,1513,2008-11-05,1466.85
Vendor07,1519,2008-11-11,1036.1
Vendor07,1523,2008-11-11,1140.52
Vendor07,1527,2008-11-21,1277.07
Vendor07,1534,2008-11-22,1508.11
Vendor07,1538,2008-11-22,1212.3
Vendor07,1545,2008-11-22,2052.22
Vendor07,1546,2008-11-23,3340.13
Vendor07,1555,2008-12-03,1034.01
Vendor07,1559,2008-12-13,2213.65
Vendor07,1561,2008-12-13,2193.52
Vendor07,1569,2008-12-14,3297.57
Vendor07,1575,2008-12-15,1175.54
Vendor07,1579,2008-12-16,2182.12
Vendor07,1581,2008-12-17,5000.1
Vendor07,1590,2008-12-25,5370.91
Vendor07,1595,2008-12-25,2712.5
Vendor07,1600,2008-12-25,2806.44
Vendor07,1601,2008-12-25,2320.03
Vendor07,1609,2008-12-26,4726.16
Vendor07,1613,2009-01-03,1165.59
Vendor07,1620,2009-01-04,2717.92
Vendor07,1624,2009-01-05,6501.73
Vendor07,1626,2009-01-05,3987.69
Vendor07,1635,2009-01-05,3986.96
Vendor07,1639,2009-01-05,3625.38
Vendor07,1641,2009-01-06,5990.85
Vendor07,1650,2009-01-07,4483.16
Vendor07,1654,2009-01-13,8495.51
Vendor07,1660,2009-01-23,1459.03
Vendor07,1664,2009-01-24,1308.18
Vendor07,1669,2009-01-24,1631.09
Vendor07,1672,2009-02-03,1431.75
Vendor07,1680,2009-02-04,2892.15
Vendor07,1682,2009-02-12,6940.19
Vendor07,1688,2009-02-22,4855.53
Vendor07,1691,2009-02-22,3127.87
Vendor07,1697,2009-03-04,2343.79
Vendor07,1703,2009-03-14,3477.02
Vendor07,1708,2009-03-15,5994.61
Vendor07,1715,2009-03-21,4685.06
Vendor07,1720,2009-03-29,1066.71
Vendor07,1724,2009-03-30,5677.96
Vendor07,1727,2009-03-31,6620.21
Vendor07,1731,2009-04-10,4805.94
Vendor07,1739,2009-04-11,5002.5
Vendor07,1744,2009-04-17,5963.98
Vendor07,1750,2009-04-18,3245.36
Vendor07,1751,2009-04-19,1138.56
Vendor07,1756,2009-04-19,2343.6
Vendor07,1764,2009-04-27,4400.93
Vendor07,1769,2009-04-27,2473.08
Vendor07,1771,2009-05-03,5992.54
Vendor07,1780,2009-05-03,9161.13
Vendor07,1781,2009-05-03,1348.76
Vendor07,1787,2009-05-09,6676.96
Vendor07,1794,2009-05-19,1300.01
Vendor07,1800,2009-05-29,1943.69
Vendor07,1803,2009-05-30,1650.74
Vendor07,1806,2009-06-09,5511.8
Vendor07,1814,2009-06-17,1331.33
Vendor07,1816,2009-06-17,7097.95
Vendor07,1822,2009-06-18,2643.95
Vendor07,1828,2009-06-24,4083.02
Vendor07,1832,2009-06-25,5717.75
Vendor07,1836,2009-06-25,4440.95
Vendor07,1842,2009-06-26,1856.65
Vendor07,1850,2009-07-06,7395.81
Vendor07,1852,2009-07-07,8839.13
Vendor07,1860,2009-07-08,5418.04
Vendor07,1864,2009-07-14,1651.46
Vendor07,1867,2009-07-24,2493.08
Vendor07,1871,2009-08-03,9413.54
Vendor07,1876,2009-08-04,2022.51
Vendor07,1881,2009-08-10,3171.77
Vendor07,1888,2009-08-10,3408.46
Vendor07,1892,2009-08-16,6057.57
Vendor07,1897,2009-08-24,1811.52
Vendor07,1905,2009-08-25,1029.71
Vendor07,1909,2009-08-31,3672.35
Vendor07,1913,2009-09-06,2612.32
Vendor07,1919,2009-09-16,1486.55
Vendor07,1923,2009-09-24,1291.01
Vendor07,1929,2009-09-24,2806.06
Vendor07,1931,2009-09-24,1335.58
Vendor07,1940,2009-09-25,2030.99
Vendor07,1944,2009-10-01,2156.6
Vendor07,1949,2009-10-11,2860.33
Vendor07,1955,2009-10-11,8565.39
Vendor07,1960,2009-10-17,1575.42
Vendor07,1963,2009-10-18,5407.04
Vendor07,1970,2009-10-24,1670.83
Vendor07,1971,2009-10-30,2197.72
Vendor07,1979,2009-10-31,1766.7
Vendor07,1981,2009-11-01,1649.7
Vendor07,1988,2009-11-01,2062.36
Vendor07,1992,2009-11-11,1557.76
Vendor07,2000,2009-11-12,1629.08
Vendor07,2002,2009-11-18,5139.56
Vendor07,2010,2009-11-18,1811.26
Vendor07,2011,2009-11-19,1291.18
Vendor07,2019,2009-11-29,4313.85
Vendor07,2024,2009-12-07,2990.39
Vendor07,2030,2009-12-17,6834.98
Vendor07,2033,2009-12-18,4360.79
Vendor07,2037,2009-12-19,1882.53
Vendor07,2042,2009-12-25,3987.89
Vendor07,2049,2009-12-31,4237.05
Vendor07,2053,2009-12-31,4376.6
Vendor07,2059,2009-12-31,3647.12
Vendor07,2062,2010-01-08,4510.94
Vendor07,2067,2010-01-16,5514.29
Vendor07,2073,2010-01-16,5749.38
Vendor07,2078,2010-01-17,2328.95
Vendor07,2083,2010-01-17,7617.42
Vendor07,2089,2010-01-23,1124.26
Vendor07,2092,2010-02-02,3336.33
Vendor07,2098,2010-02-03,1032.44
Vendor07,2104,2010-02-03,4947.52
Vendor07,2107,2010-02-11,2267.17
Vendor07,2111,2010-02-21,1531.23
Vendor07,2120,2010-02-22,2466.19
Vendor07,2121,2010-02-23,9113.29
Vendor07,2129,2010-02-23,1920.29
Vendor07,2135,2010-03-01,1118.05
Vendor07,2139,2010-03-02,9763.1
Vendor07,2143,2010-03-02,9243.52
Vendor07,2147,2010-03-08,2135.23
Vendor07,2155,2010-03-08,3745.61
Vendor07,2160,2010-03-09,1911.47
Vendor07,2162,2010-03-19,3012.67
Vendor07,2169,2010-03-20,1331.69
Vendor07,2172,2010-03-30,7163.14
Vendor07,2178,2010-04-07,1686.8
Vendor07,2185,2010-04-17,2424.25
Vendor07,2186,2010-04-18,2383.59
Vendor07,2195,2010-04-18,9522.98
Vendor07,2197,2010-04-24,1712.48
Vendor07,2205,2010-04-25,6354.56
Vendor07,2207,2010-05-03,1165.25
Vendor07,2213,2010-05-03,4206.75
Vendor07,2219,2010-05-03,3116.57
Vendor07,2225,2010-05-04,1098.88
Vendor07,2227,2010-05-04,4197.05
Vendor07,2235,2010-05-05,6082.19
Vendor07,2240,2010-05-15,1799.67
Vendor07,2244,2010-05-16,2908.69
Vendor07,2246,2010-05-26,2693.34
Vendor07,2254,2010-06-03,7714.61
Vendor07,2256,2010-06-04,7112.45
Vendor07,2261,2010-06-10,2071.19
Vendor07,2268,2010-06-11,7889.89
Vendor07,2271,2010-06-11,9996.03
Vendor07,2276,2010-06-12,2025.1
Vendor07,2281,2010-06-18,1912.6
Vendor07,2288,2010-06-26,1739.87
Vendor07,2292,2010-06-27,6855.17
Vendor07,2296,2010-06-27,3867.15
Vendor07,2305,2010-06-27,1004.1
Vendor07,2306,2010-06-28,6703.36
Vendor07,2314,2010-07-04,4761.81
Vendor07,2318,2010-07-12,1767.54
Vendor07,2323,2010-07-13,1111.12
Vendor07,2330,2010-07-14,5990.27
Vendor07,2333,2010-07-14,1940.31
Vendor07,2337,2010-07-15,2874.48
Vendor07,2345,2010-07-16,1690.61
Vendor07,2348,2010-07-26,1694.07
Vendor07,2353,2010-07-27,1291.01
Vendor07,2359,2010-07-28,1734.82
Vendor07,2361,2010-08-03,9116.68
Vendor07,2369,2010-08-04,9714.72
Vendor07,2372,2010-08-05,4451.23
Vendor07,2377,2010-08-11,2360.51
Vendor07,2385,2010-08-12,4344.42
Vendor07,2386,2010-08-13,6943.39
Vendor07,2395,2010-08-14,2089.86
Vendor07,2400,2010-08-15,9687.74
Vendor07,2403,2010-08-15,2020.05
Vendor07,2409,2010-08-16,9254.82
Vendor07,2411,2010-08-22,6919.79
Vendor07,2418,2010-08-23,1425.22
Vendor07,2423,2010-08-31,7136.78
Vendor07,2429,2010-09-01,2145.59
Vendor07,2435,2010-09-01,1993.73
Vendor07,2437,2010-09-11,1646.75
Vendor07,2441,2010-09-17,2440.74
Vendor07,2450,2010-09-17,2853.75
Vendor07,2451,2010-09-23,3099.92
Vendor07,2460,2010-09-23,3056.92
Vendor07,2462,2010-10-03,4322.19
Vendor07,2468,2010-10-03,1422.62
Vendor07,2474,2010-10-03,2744.9
Vendor07,2480,2010-10-11,9438.85
Vendor07,2483,2010-10-21,2152.04
Vendor07,2487,2010-10-21,2712.97
Vendor07,2495,2010-10-22,2315.97
Vendor07,2497,2010-10-22,6911.2
Vendor07,2505,2010-11-01,6301.61
Vendor07,2506,2010-11-01,1656.88
Vendor07,2512,2010-11-01,5483.96
Vendor07,2518,2010-11-01,9553.02
Vendor07,2525,2010-11-02,1497.75
Vendor07,2530,2010-11-02,9643.92
Vendor07,2533,2010-11-08,5695.43
Vendor07,2539,2010-11-09,5188.32
Vendor07,2543,2010-11-15,4849.27
Vendor07,2549,2010-11-21,7364.02
Vendor07,2552,2010-11-27,5834.06
Vendor07,2556,2010-11-28,5728.97
Vendor07,2564,2010-11-29,7307.98
Vendor07,2568,2010-12-07,5441.24
Vendor07,2573,2010-12-17,5167.33
Vendor07,2576,2010-12-18,3836.1
Vendor07,2582,2010-12-28,8613.92
Vendor07,2587,2010-12-29,4333.68
Vendor07,2593,2010-12-30,1286.38
Vendor07,2596,2010-12-30,2165.04
Vendor07,2601,2011-01-05,6242.89
Vendor07,2608,2011-01-11,2928.88
Vendor07,2615,2011-01-19,1043.29
Vendor07,2617,2011-01-20,7032.98
Vendor07,2625,2011-01-20,5859.34
Vendor07,2629,2011-01-21,8564.53
Vendor07,2632,2011-01-27,3619.18
Vendor07,2638,2011-01-28,3261.09
Vendor07,2644,2011-01-29,1258.69
Vendor07,2649,2011-02-06,6745.76
Vendor07,2655,2011-02-07,2153.34
Vendor07,2659,2011-02-13,2773.21
Vendor07,2664,2011-02-19,5324.4
Vendor07,2666,2011-03-01,4211.69
Vendor07,2671,2011-03-02,7730.66
Vendor07,2676,2011-03-03,4389.7
Vendor07,2685,2011-03-04,2442.86
Vendor07,2689,2011-03-14,7341.47
Vendor07,2693,2011-03-15,1083.97
Vendor07,2697,2011-03-16,1113.58
Vendor07,2701,2011-03-24,8360.18
Vendor07,2707,2011-03-25,2755.24
Vendor07,2715,2011-03-26,8279.38
Vendor07,2717,2011-03-27,2079.26
Vendor07,2721,2011-04-02,7128.92
Vendor07,2727,2011-04-02,2824.37
Vendor07,2734,2011-04-02,1762.59
Vendor07,2739,2011-04-03,5380.42
Vendor07,2744,2011-04-04,2269.43
Vendor07,2748,2011-04-04,3573.81
Vendor07,2755,2011-04-10,5246.48
Vendor07,2760,2011-04-10,5507.06
Vendor07,2763,2011-04-16,9501.54
Vendor07,2769,2011-04-17,2671.94
Vendor07,2773,2011-04-27,1241.58
Vendor07,2780,2011-04-27,1202.62
Vendor07,2785,2011-04-27,4517.29
Vendor07,2788,2011-04-27,1636.0
Vendor07,2795,2011-04-28,1306.58
Vendor07,2800,2011-05-04,2466.39
Vendor07,2801,2011-05-14,3408.3
Vendor07,2810,2011-05-24,2544.06
Vendor07,2814,2011-05-24,3375.76
Vendor07,2819,2011-05-24,2431.09
Vendor07,2824,2011-05-30,2973.59
Vendor07,2827,2011-06-09,2185.89
Vendor07,2835,2011-06-09,5414.94
Vendor07,2837,2011-06-10,5912.82
Vendor07,2844,2011-06-20,2456.27
Vendor07,2848,2011-06-26,6937.77
Vendor07,2851,2011-06-27,1141.77
Vendor07,2856,2011-06-28,1836.51
Vendor07,2862,2011-06-29,4827.57
Vendor07,2867,2011-06-30,6007.41
Vendor07,2873,2011-06-30,1229.14
Vendor07,2876,2011-06-30,7431.45
Vendor07,2885,2011-06-30,2825.99
Vendor07,2890,2011-07-01,2214.41
Vendor07,2895,2011-07-01,7380.96
Vendor07,2896,2011-07-11,2370.9
Vendor07,2903,2011-07-12,1477.72
Vendor07,2908,2011-07-13,5357.51
Vendor07,2915,2011-07-14,1801.84
Vendor07,2917,2011-07-14,4611.49
Vendor07,2921,2011-07-15,6402.14
Vendor07,2930,2011-07-16,1625.36
Vendor07,2932,2011-07-24,1054.14
Vendor07,2939,2011-07-30,5380.08
Vendor07,2943,2011-07-30,2453.6
Vendor07,2950,2011-07-31,1894.82
Vendor07,2953,2011-08-08,1479.37
Vendor07,2957,2011-08-14,1418.4
Vendor07,2965,2011-08-24,6144.03
Vendor07,2967,2011-08-25,1144.63
Vendor07,2972,2011-09-02,4717.86
Vendor07,2976,2011-09-10,1345.59
Vendor07,2982,2011-09-11,5342.17
Vendor07,2987,2011-09-17,1531.86
Vendor07,2991,2011-09-18,2735.81
Vendor07,3000,2011-09-19,2688.75
Vendor07,3005,2011-09-29,3890.93
Vendor07,3009,2011-10-09,1591.53
Vendor07,3012,2011-10-15,2706.09
Vendor07,3020,2011-10-15,1582.63
Vendor07,3024,2011-10-16,8166.22
Vendor07,3026,2011-10-17,2388.68
Vendor07,3033,2011-10-17,9519.39
Vendor07,3038,2011-10-18,5913.48
Vendor07,3045,2011-10-18,3919.67
Vendor07,3050,2011-10-24,9251.18
Vendor07,3052,2011-10-25,1601.12
Vendor07,3056,2011-10-26,4196.66
Vendor07,3063,2011-10-27,7917.95
Vendor07,3067,2011-10-28,4767.91
Vendor07,3071,2011-10-28,2143.1
Vendor07,3078,2011-10-29,9276.2
Vendor07,3082,2011-10-29,1184.72
Vendor07,3086,2011-11-04,5933.01
Vendor07,3091,2011-11-05,3731.24
Vendor07,3096,2011-11-06,3607.25
Vendor07,3102,2011-11-07,1984.33
Vendor07,3108,2011-11-08,4115.44
Vendor07,3113,2011-11-14,1987.75
Vendor07,3120,2011-11-20,1233.2
Vendor07,3121,2011-11-26,1915.29
Vendor07,3126,2011-11-26,2350.39
Vendor07,3131,2011-12-06,1522.25
Vendor07,3137,2011-12-07,6929.06
Vendor07,3141,2011-12-07,9193.7
Vendor07,3146,2011-12-08,3429.0
Vendor07,3155,2011-12-14,1011.29
Vendor07,3160,2011-12-20,3569.74
Vendor07,3161,2011-12-28,3259.57
Vendor07,3168,2012-01-03,6258.35
Vendor07,3173,2012-01-11,4508.04
Vendor07,3179,2012-01-11,5049.88
Vendor07,3184,2012-01-17,4588.79
Vendor07,3189,2012-01-18,3744.35
Vendor07,3192,2012-01-24,6228.87
Vendor07,3200,2012-01-25,1590.07
Vendor07,3202,2012-02-04,1795.59
Vendor07,3207,2012-02-04,2643.11
Vendor07,3212,2012-02-04,8218.76
Vendor07,3219,2012-02-10,2633.08
Vendor07,3224,2012-02-16,1720.85
Vendor07,3227,2012-02-16,2014.11
Vendor07,3233,2012-02-22,4275.65
Vendor07,3236,2012-02-23,6134.59
Vendor07,3244,2012-03-04,2002.45
Vendor07,3246,2012-03-04,5865.59
Vendor07,3255,2012-03-04,8869.23
Vendor07,3256,2012-03-04,1743.0
Vendor07,3263,2012-03-14,1300.17
Vendor07,3268,2012-03-22,4499.82
Vendor07,3274,2012-03-22,2275.43
Vendor07,3277,2012-03-28,2640.56
Vendor07,3285,2012-03-29,4772.46
Vendor07,3289,2012-03-29,1496.33
Vendor07,3293,2012-03-30,1882.12
Vendor07,3296,2012-04-09,1494.68
Vendor07,3303,2012-04-19,2105.8
Vendor07,3306,2012-04-19,6088.88
Vendor07,3315,2012-04-29,6633.3
Vendor07,3319,2012-05-07,1014.75
Vendor07,3322,2012-05-13,7652.78
Vendor07,3330,2012-05-14,2081.7
Vendor07,3335,2012-05-15,4661.99
Vendor07,3337,2012-05-16,1588.9
Vendor07,3342,2012-05-24,1779.57
Vendor07,3348,2012-05-25,1622.4
Vendor07,3353,2012-06-04,1966.85
Vendor07,3359,2012-06-04,3068.83
Vendor07,3363,2012-06-04,1976.51
Vendor07,3369,2012-06-05,3428.63
Vendor07,3372,2012-06-11,1360.42
Vendor07,3376,2012-06-21,7556.79
Vendor07,3382,2012-06-27,3308.32
Vendor07,3387,2012-06-27,5191.22
Vendor07,3394,2012-07-03,1164.86
Vendor07,3397,2012-07-09,9596.18
Vendor07,3402,2012-07-19,5156.44
Vendor07,3410,2012-07-20,2308.47
Vendor07,3413,2012-07-30,1421.05
Vendor07,3417,2012-07-30,1839.34
Vendor07,3424,2012-07-30,4638.39
Vendor07,3428,2012-07-31,1314.65
Vendor07,3431,2012-07-31,1081.78
Vendor07,3438,2012-07-31,4826.44
Vendor07,3445,2012-08-08,2065.52
Vendor07,3447,2012-08-08,1628.05
Vendor07,3454,2012-08-16,4613.12
Vendor07,3457,2012-08-22,8431.31
Vendor07,3463,2012-08-22,1071.78
Vendor07,3466,2012-08-23,1619.89
Vendor07,3472,2012-09-02,1870.58
Vendor07,3477,2012-09-12,4460.21
Vendor07,3484,2012-09-12,1609.1
Vendor07,3490,2012-09-22,2039.22
Vendor07,3491,2012-09-30,1817.9
Vendor07,3498,2012-09-30,2932.28
Vendor07,3501,2012-09-30,4052.72
Vendor07,3509,2012-10-10,4885.52
Vendor07,3511,2012-10-18,3782.43
Vendor07,3520,2012-10-19,1914.58
Vendor07,3524,2012-10-19,1028.14
Vendor07,3527,2012-10-25,1353.31
Vendor07,3535,2012-10-26,8397.69
Vendor07,3537,2012-10-27,1801.09
Vendor07,3544,2012-10-27,8579.31
Vendor07,3547,2012-11-04,3217.24
Vendor07,3554,2012-11-10,5057.22
Vendor07,3556,2012-11-10,1803.15
Vendor07,3562,2012-11-20,7034.74
Vendor07,3567,2012-11-28,1644.34
Vendor07,3573,2012-11-28,2280.82
Vendor07,3579,2012-11-28,3848.57
Vendor07,3584,2012-11-28,4286.05
Vendor07,3590,2012-11-29,6268.88
Vendor07,3592,2012-11-29,5992.54
Vendor07,3598,2012-11-30,8489.24
Vendor07,3601,2012-12-01,4770.83
Vendor07,3610,2012-12-07,3796.01
Vendor07,3613,2012-12-08,4614.88
Vendor07,3616,2012-12-08,2293.2
Vendor07,3623,2012-12-09,1709.31
Vendor07,3627,2012-12-15,4292.01
Vendor07,3632,2012-12-16,1036.74
Vendor07,3637,2012-12-24,1450.39
Vendor07,3644,2012-12-24,3514.41
Vendor07,3647,2012-12-30,1372.59
Vendor07,3652,2013-01-05,5138.81
Vendor07,3660,2013-01-11,1053.42
Vendor07,3663,2013-01-12,7479.19
Vendor07,3668,2013-01-18,1426.61
Vendor07,3673,2013-01-28,1529.67
Vendor07,3678,2013-02-03,1698.1
Vendor07,3682,2013-02-03,1401.77
Vendor07,3686,2013-02-04,5928.25
Vendor07,3691,2013-02-04,9004.16
Vendor07,3696,2013-02-05,1885.82
Vendor07,3705,2013-02-05,8123.78
Vendor07,3708,2013-02-05,8692.98
Vendor07,3715,2013-02-06,2433.56
Vendor07,3718,2013-02-07,9501.53
Vendor07,3723,2013-02-07,1818.69
Vendor07,3726,2013-02-08,6086.58
Vendor07,3734,2013-02-18,4295.79
Vendor07,3738,2013-02-18,3772.01
Vendor07,3744,2013-02-18,3126.83
Vendor07,3749,2013-02-28,5912.92
Vendor07,3751,2013-02-28,6099.36
Vendor07,3757,2013-02-28,2001.74
Vendor07,3762,2013-03-01,1500.92
Vendor07,3769,2013-03-07,3862.91
Vendor07,3773,2013-03-13,2434.05
Vendor07,3776,2013-03-13,1020.29
Vendor07,3783,2013-03-13,5437.9
Vendor07,3790,2013-03-14,1805.5
Vendor07,3794,2013-03-15,3978.57
Vendor07,3796,2013-03-21,1924.58
Vendor07,3805,2013-03-27,2988.53
Vendor07,3806,2013-04-04,5496.54
Vendor07,3815,2013-04-05,8027.1
Vendor07,3818,2013-04-11,1398.83
Vendor07,3823,2013-04-17,6800.17
Vendor07,3827,2013-04-23,5113.23
Vendor07,3832,2013-04-24,6045.17
Vendor07,3837,2013-04-25,7571.59
Vendor07,3845,2013-05-01,6955.13
Vendor07,3850,2013-05-07,7166.94
Vendor07,3852,2013-05-07,6005.21
Vendor07,3856,2013-05-07,3765.47
Vendor07,3865,2013-05-08,2432.23
Vendor07,3869,2013-05-09,2928.99
Vendor07,3873,2013-05-10,5047.56
Vendor07,3879,2013-05-10,1228.49
Vendor07,3881,2013-05-16,8211.45
Vendor07,3888,2013-05-22,1441.53
Vendor07,3891,2013-05-28,2088.47
Vendor07,3897,2013-06-05,8484.12
Vendor07,3903,2013-06-13,7564.0
Vendor07,3907,2013-06-23,4828.47
Vendor07,3911,2013-06-23,3916.94
Vendor07,3918,2013-06-23,4259.14
Vendor07,3922,2013-07-01,1373.93
Vendor07,3927,2013-07-01,8098.87
Vendor07,3934,2013-07-07,1067.12
Vendor07,3939,2013-07-17,1108.37
Vendor07,3941,2013-07-27,5712.59
Vendor07,3947,2013-07-28,1586.37
Vendor07,3952,2013-07-28,1332.92
Vendor07,3956,2013-07-29,1210.33
Vendor07,3962,2013-07-29,2201.49
Vendor07,3968,2013-07-29,1291.74
Vendor07,3975,2013-08-04,2891.57
Vendor07,3977,2013-08-05,2958.31
Vendor07,3982,2013-08-05,2178.1
Vendor07,3986,2013-08-11,2347.51
Vendor07,3994,2013-08-12,1227.97
Vendor07,3996,2013-08-13,1882.13
Vendor07,4001,2013-08-19,1204.29
Vendor07,4006,2013-08-19,1370.65
Vendor07,4014,2013-08-19,2563.27
Vendor07,4017,2013-08-19,1695.53
Vendor07,4022,2013-08-29,9377.2
Vendor07,4030,2013-09-04,4863.78
Vendor07,4031,2013-09-10,5149.47
Vendor07,4039,2013-09-16,5534.66
Vendor07,4043,2013-09-16,1435.59
Vendor07,4048,2013-09-24,8178.13
Vendor07,4052,2013-10-02,7934.24
Vendor07,4056,2013-10-10,4228.29
Vendor07,4062,2013-10-18,2190.63
Vendor07,4067,2013-10-19,1770.96
Vendor07,4075,2013-10-19,6528.59
Vendor07,4078,2013-10-20,7287.15
Vendor07,4084,2013-10-21,1285.24
Vendor07,4089,2013-10-31,8049.16
Vendor07,4091,2013-11-08,2289.78
Vendor07,4098,2013-11-16,5551.18
Vendor07,4104,2013-11-26,5028.34
Vendor07,4109,2013-11-27,8256.17
Vendor07,4115,2013-12-07,8176.91
Vendor07,4119,2013-12-07,1443.66
Vendor07,4121,2013-12-07,3669.07
Vendor07,4130,2013-12-07,1070.98
Vendor07,4135,2013-12-07,8811.5
Vendor07,4140,2013-12-13,1851.69
Vendor07,4145,2013-12-13,5021.35
Vendor07,4149,2013-12-14,2234.76
Vendor07,4155,2013-12-14,6382.37
Vendor07,4157,2013-12-20,2669.95
Vendor07,4162,2013-12-28,3884.4
Vendor07,4169,2014-01-03,3105.11
Vendor07,4172,2014-01-09,1252.07
Vendor07,4179,2014-01-10,3573.39
Vendor07,4184,2014-01-11,2644.71
Vendor07,4188,2014-01-19,1345.64
Vendor07,4193,2014-01-19,5609.51
Vendor07,4197,2014-01-19,3366.07
Vendor07,4202,2014-01-20,4083.88
Vendor07,4208,2014-01-26,9960.85
Vendor07,4213,2014-01-26,1030.19
Vendor07,4217,2014-01-26,1368.02
Vendor07,4225,2014-02-01,1995.74
Vendor07,4228,2014-02-02,2744.92
Vendor07,4233,2014-02-08,1176.97
Vendor07,4237,2014-02-14,2676.88
Vendor07,4243,2014-02-14,3368.89
Vendor07,4249,2014-02-14,4990.17
Vendor07,4253,2014-02-15,8536.74
Vendor07,4260,2014-02-15,7309.06
Vendor07,4265,2014-02-15,9893.62
Vendor07,4269,2014-02-16,3078.7
Vendor07,4273,2014-02-22,1663.74
Vendor07,4278,2014-02-23,1985.43
Vendor07,4284,2014-03-01,8272.82
Vendor07,4290,2014-03-07,4779.55
Vendor07,4294,2014-03-13,1200.01
Vendor07,4300,2014-03-14,4965.92
Vendor07,4301,2014-03-20,2609.0
Vendor07,4306,2014-03-20,1971.27
Vendor07,4311,2014-03-21,9156.95
Vendor07,4317,2014-03-21,5008.35
Vendor07,4324,2014-03-27,4308.02
Vendor07,4330,2014-03-27,2736.64
Vendor07,4333,2014-03-27,7143.41
Vendor07,4336,2014-04-06,2709.38
Vendor07,4344,2014-04-07,4754.52
Vendor07,4346,2014-04-08,3052.51
Vendor07,4354,2014-04-14,4182.98
Vendor07,4358,2014-04-15,1672.4
Vendor07,4363,2014-04-15,1903.05
Vendor07,4367,2014-04-15,2094.26
Vendor07,4371,2014-04-21,6255.92
Vendor07,4377,2014-04-29,1396.53
Vendor07,4381,2014-05-05,1004.03
Vendor07,4386,2014-05-05,3960.87
Vendor12,9,2006-01-02,8763.91
Vendor12,13,2006-01-02,1292.94
Vendor12,17,2006-01-12,8194.42
Vendor12,22,2006-01-13,7025.67
Vendor12,30,2006-01-14,1200.95
Vendor12,33,2006-01-20,1514.54
Vendor12,39,2006-01-30,8909.76
Vendor12,43,2006-01-30,1853.27
Vendor12,46,2006-02-05,6682.52
Vendor12,55,2006-02-11,2130.02
Vendor12,57,2006-02-17,4623.93
Vendor12,65,2006-02-25,2780.18
Vendor12,68,2006-03-07,9131.65
Vendor12,74,2006-03-07,2080.44
Vendor12,78,2006-03-07,1404.37
Vendor12,85,2006-03-07,1159.31
Vendor12,87,2006-03-07,8593.58
Vendor12,92,2006-03-08,5863.19
Vendor12,97,2006-03-08,3260.08
Vendor12,105,2006-03-18,4374.97
Vendor12,107,2006-03-19,1897.08
Vendor12,112,2006-03-27,1942.15
Vendor12,116,2006-03-27,1156.0
Vendor12,122,2006-03-28,5964.45
Vendor12,127,2006-03-28,4489.19
Vendor12,134,2006-04-03,6817.48
Vendor12,136,2006-04-04,2537.05
Vendor12,145,2006-04-04,5642.34
Vendor12,148,2006-04-10,1374.95
Vendor12,153,2006-04-20,2244.71
Vendor12,158,2006-04-26,1331.11
Vendor12,164,2006-04-27,9228.17
Vendor12,166,2006-04-28,3603.03
Vendor12,175,2006-05-04,1110.06
Vendor12,176,2006-05-04,4131.77
Vendor12,182,2006-05-14,6964.39
Vendor12,188,2006-05-14,8251.2
Vendor12,195,2006-05-15,1327.47
Vendor12,200,2006-05-21,9257.09
Vendor12,205,2006-05-22,2947.04
Vendor12,208,2006-05-23,5950.91
Vendor12,215,2006-05-23,4186.44
Vendor12,217,2006-05-29,2009.22
Vendor12,224,2006-05-30,7717.8
Vendor12,226,2006-05-30,6930.34
Vendor12,234,2006-06-09,4056.07
Vendor12,236,2006-06-10,8843.59
Vendor12,245,2006-06-11,3883.45
Vendor12,247,2006-06-12,4543.83
Vendor12,252,2006-06-13,1241.68
Vendor12,256,2006-06-14,2457.06
Vendor12,261,2006-06-20,2895.02
Vendor12,268,2006-06-30,5317.13
Vendor12,273,2006-07-01,6859.35
Vendor12,276,2006-07-01,1767.58
Vendor12,282,2006-07-01,1560.47
Vendor12,286,2006-07-07,3462.79
Vendor12,293,2006-07-08,1510.43
Vendor12,300,2006-07-08,4968.72
Vendor12,305,2006-07-14,2629.3
Vendor12,307,2006-07-14,3171.46
Vendor12,315,2006-07-14,4428.51
Vendor12,320,2006-07-20,3492.43
Vendor12,325,2006-07-26,3541.7
Vendor12,329,2006-07-27,2336.25
Vendor12,332,2006-08-06,1595.23
Vendor12,340,2006-08-12,2228.2
Vendor12,343,2006-08-12,1854.91
Vendor12,347,2006-08-18,3945.21
Vendor12,355,2006-08-19,1400.06
Vendor12,359,2006-08-20,4827.51
Vendor12,361,2006-08-30,4048.21
Vendor12,366,2006-08-31,6975.89
Vendor12,372,2006-09-10,1041.03
Vendor12,380,2006-09-16,2587.06
Vendor12,385,2006-09-17,6096.26
Vendor12,387,2006-09-17,2177.01
Vendor12,395,2006-09-17,7291.52
Vendor12,399,2006-09-17,2725.07
Vendor12,404,2006-09-17,7813.31
Vendor12,406,2006-09-23,1615.94
Vendor12,412,2006-09-23,5224.76
Vendor12,420,2006-09-29,4812.18
Vendor12,423,2006-10-05,2913.13
Vendor12,428,2006-10-05,4353.77
Vendor12,431,2006-10-06,3892.18
Vendor12,438,2006-10-12,1363.12
Vendor12,441,2006-10-13,1242.51
Vendor12,450,2006-10-23,3760.97
Vendor12,455,2006-10-31,9996.87
Vendor12,458,2006-11-06,2482.62
Vendor12,461,2006-11-06,5613.64
Vendor12,467,2006-11-07,1096.05
Vendor12,474,2006-11-08,3348.95
Vendor12,480,2006-11-14,7296.3
Vendor12,485,2006-11-22,2183.04
Vendor12,490,2006-11-23,2237.65
Vendor12,491,2006-12-03,8860.86
Vendor12,499,2006-12-03,1653.77
Vendor12,504,2006-12-04,6665.95
Vendor12,506,2006-12-12,6051.07
Vendor12,513,2006-12-13,1532.83
Vendor12,518,2006-12-21,8493.01
Vendor12,524,2006-12-31,4151.62
Vendor12,528,2006-12-31,9708.81
Vendor12,535,2007-01-01,3743.99
Vendor12,538,2007-01-02,3308.5
Vendor12,542,2007-01-02,4072.87
Vendor12,549,2007-01-08,1603.61
Vendor12,551,2007-01-09,4700.08
Vendor12,557,2007-01-09,3852.17
Vendor12,565,2007-01-09,2045.88
Vendor12,567,2007-01-19,5237.08
Vendor12,574,2007-01-19,4355.33
Vendor12,580,2007-01-20,3101.79
Vendor12,585,2007-01-21,1685.37
Vendor12,589,2007-01-31,5911.54
Vendor12,593,2007-02-08,3465.96
Vendor12,596,2007-02-08,1086.13
Vendor12,602,2007-02-18,1943.44
Vendor12,610,2007-02-18,3234.11
Vendor12,612,2007-02-24,1691.12
Vendor12,620,2007-02-25,6638.22
Vendor12,625,2007-03-07,7138.08
Vendor12,626,2007-03-07,7990.06
Vendor12,631,2007-03-17,3467.91
Vendor12,639,2007-03-17,1717.49
Vendor12,644,2007-03-18,8417.31
Vendor12,649,2007-03-18,3194.88
Vendor12,655,2007-03-18,3942.86
Vendor12,660,2007-03-18,8430.18
Vendor12,661,2007-03-28,2715.22
Vendor12,666,2007-03-29,4497.58
Vendor12,675,2007-03-30,5314.93
Vendor12,676,2007-03-31,3004.68
Vendor12,682,2007-03-31,2883.31
Vendor12,687,2007-04-01,7431.5
Vendor12,694,2007-04-02,1320.55
Vendor12,696,2007-04-02,1602.85
Vendor12,701,2007-04-03,1142.81
Vendor12,706,2007-04-03,1996.22
Vendor12,711,2007-04-09,3534.68
Vendor12,720,2007-04-09,2660.87
Vendor12,721,2007-04-10,1686.16
Vendor12,727,2007-04-11,1871.68
Vendor12,734,2007-04-12,6934.82
Vendor12,737,2007-04-22,3343.34
Vendor12,742,2007-04-22,2611.96
Vendor12,750,2007-04-23,1181.79
Vendor12,751,2007-04-23,3704.99
Vendor12,759,2007-04-24,3718.74
Vendor12,765,2007-04-24,1099.97
Vendor12,768,2007-04-25,3982.61
Vendor12,774,2007-05-01,1054.53
Vendor12,777,2007-05-02,1847.6
Vendor12,784,2007-05-02,1670.08
Vendor12,787,2007-05-03,7152.2
Vendor12,791,2007-05-09,3342.77
Vendor12,798,2007-05-09,1400.17
Vendor12,804,2007-05-15,4595.4
Vendor12,806,2007-05-25,1331.64
Vendor12,812,2007-05-26,8088.04
Vendor12,818,2007-05-26,8749.75
Vendor12,821,2007-05-26,6794.29
Vendor12,826,2007-05-26,2150.85
Vendor12,833,2007-05-27,3266.98
Vendor12,837,2007-06-06,9351.48
Vendor12,842,2007-06-06,2348.44
Vendor12,847,2007-06-16,3293.95
Vendor12,854,2007-06-16,1436.87
Vendor12,860,2007-06-17,8993.57
Vendor12,863,2007-06-27,6831.34
Vendor12,868,2007-06-28,7246.48
Vendor12,875,2007-07-04,9351.39
Vendor12,879,2007-07-12,9392.57
Vendor12,883,2007-07-13,3803.18
Vendor12,886,2007-07-21,1146.16
Vendor12,894,2007-07-22,2127.23
Vendor12,900,2007-07-28,1571.55
Vendor12,905,2007-07-28,1829.75
Vendor12,906,2007-08-03,6652.46
Vendor12,911,2007-08-03,1217.8
Vendor12,917,2007-08-11,5069.45
Vendor12,924,2007-08-19,2396.88
Vendor12,930,2007-08-25,4515.35
Vendor12,933,2007-09-02,8948.7
Vendor12,936,2007-09-02,6522.04
Vendor12,943,2007-09-02,8145.01
Vendor12,949,2007-09-02,3138.19
Vendor12,952,2007-09-02,2735.85
Vendor12,959,2007-09-03,1401.22
Vendor12,962,2007-09-09,3245.65
Vendor12,968,2007-09-17,1713.18
Vendor12,974,2007-09-18,4897.82
Vendor12,977,2007-09-24,1006.03
Vendor12,983,2007-10-02,7630.71
Vendor12,990,2007-10-08,8389.28
Vendor12,995,2007-10-16,1713.93
Vendor12,1000,2007-10-24,1222.39
Vendor12,1004,2007-10-30,1211.22
Vendor12,1006,2007-11-09,1684.06
Vendor12,1014,2007-11-10,1781.95
Vendor12,1019,2007-11-11,1235.61
Vendor12,1023,2007-11-21,3302.06
Vendor12,1026,2007-11-21,6919.95
Vendor12,1035,2007-11-27,2112.47
Vendor12,1040,2007-12-03,1654.54
Vendor12,1041,2007-12-09,1558.84
Vendor12,1050,2007-12-15,1960.85
Vendor12,1052,2007-12-23,9259.68
Vendor12,1058,2007-12-29,1111.7
Vendor12,1065,2007-12-29,2841.16
Vendor12,1070,2007-12-30,1026.28
Vendor12,1074,2007-12-31,4284.6
Vendor12,1079,2008-01-08,2060.3
Vendor12,1083,2008-01-09,8729.12
Vendor12,1086,2008-01-10,1077.3
Vendor12,1093,2008-01-10,1239.72
Vendor12,1096,2008-01-11,1948.76
Vendor12,1104,2008-01-11,1963.57
Vendor12,1109,2008-01-19,3364.68
Vendor12,1113,2008-01-27,6025.43
Vendor12,1120,2008-02-06,8771.55
Vendor12,1121,2008-02-12,6217.46
Vendor12,1128,2008-02-20,1375.43
Vendor12,1134,2008-02-20,1063.05
Vendor12,1139,2008-02-20,6169.63
Vendor12,1142,2008-02-20,1250.55
Vendor12,1150,2008-02-20,8635.19
Vendor12,1152,2008-02-28,1503.27
Vendor12,1156,2008-02-28,1631.82
Vendor12,1165,2008-02-28,1034.9
Vendor12,1170,2008-03-05,3136.54
Vendor12,1173,2008-03-06,1118.76
Vendor12,1180,2008-03-07,8987.3
Vendor12,1181,2008-03-08,2000.12
Vendor12,1187,2008-03-14,3190.68
Vendor12,1192,2008-03-22,6958.42
Vendor12,1200,2008-03-28,8713.55
Vendor12,1202,2008-03-29,3748.41
Vendor12,1207,2008-04-04,6166.46
Vendor12,1213,2008-04-04,8096.58
Vendor12,1220,2008-04-10,6474.25
Vendor12,1223,2008-04-11,2883.88
Vendor12,1229,2008-04-19,1694.85
Vendor12,1233,2008-04-20,6725.6
Vendor12,1240,2008-04-20,8761.2
Vendor12,1242,2008-04-28,3345.56
Vendor12,1246,2008-05-06,1932.82
Vendor12,1252,2008-05-12,7013.03
Vendor12,1260,2008-05-12,5527.06
Vendor12,1263,2008-05-18,1624.08
Vendor12,1266,2008-05-18,5933.55
Vendor12,1275,2008-05-28,3707.76
Vendor12,1279,2008-05-28,2711.22
Vendor12,1283,2008-06-05,7919.39
Vendor12,1287,2008-06-11,8199.84
Vendor12,1291,2008-06-17,5935.64
Vendor12,1300,2008-06-18,2428.61
Vendor12,1303,2008-06-18,1944.13
Vendor12,1306,2008-06-24,9968.25
Vendor12,1315,2008-07-02,3097.67
Vendor12,1318,2008-07-02,2310.74
Vendor12,1325,2008-07-10,4486.19
Vendor12,1330,2008-07-11,3870.7
Vendor12,1333,2008-07-21,1343.99
Vendor12,1338,2008-07-22,3386.0
Vendor12,1345,2008-07-22,1642.48
Vendor12,1350,2008-07-22,3978.08
Vendor12,1355,2008-07-23,1563.3
Vendor12,1358,2008-07-23,2883.95
Vendor12,1363,2008-07-23,1937.35
Vendor12,1368,2008-07-29,1375.57
Vendor12,1375,2008-08-04,6486.35
Vendor12,1376,2008-08-04,3647.41
Vendor12,1384,2008-08-10,2063.77
Vendor12,1386,2008-08-11,1047.97
Vendor12,1394,2008-08-17,1592.54
Vendor12,1399,2008-08-18,6785.83
Vendor12,1404,2008-08-19,4826.04
Vendor12,1408,2008-08-25,2870.23
Vendor12,1413,2008-09-02,8435.69
Vendor12,1418,2008-09-10,1184.63
Vendor12,1423,2008-09-11,4862.89
Vendor12,1427,2008-09-12,2264.35
Vendor12,1433,2008-09-12,1726.41
Vendor12,1436,2008-09-20,7859.13
Vendor12,1445,2008-09-20,5534.45
Vendor12,1450,2008-09-21,3446.37
Vendor12,1455,2008-09-22,4542.01
Vendor12,1458,2008-09-28,5856.72
Vendor12,1464,2008-09-29,4024.53
Vendor12,1466,2008-09-29,2773.96
Vendor12,1474,2008-09-30,3696.74
Vendor12,1478,2008-09-30,3942.14
Vendor12,1485,2008-10-10,6324.18
Vendor12,1488,2008-10-11,1929.1
Vendor12,1493,2008-10-11,6284.63
Vendor12,1499,2008-10-12,7766.57
Vendor12,1504,2008-10-22,9410.98
Vendor12,1506,2008-10-30,1249.98
Vendor12,1512,2008-10-30,2636.32
Vendor12,1517,2008-10-31,8417.34
Vendor12,1525,2008-10-31,6427.24
Vendor12,1528,2008-11-06,7893.49
Vendor12,1535,2008-11-12,2912.74
Vendor12,1539,2008-11-13,1347.26
Vendor12,1542,2008-11-13,2709.4
Vendor12,1549,2008-11-13,3274.55
Vendor12,1554,2008-11-19,1330.47
Vendor12,1559,2008-11-20,7141.88
Vendor12,1562,2008-11-30,4703.5
Vendor12,1570,2008-11-30,2930.2
Vendor12,1572,2008-11-30,1405.75
Vendor12,1576,2008-12-01,4599.46
Vendor12,1585,2008-12-09,2303.65
Vendor12,1587,2008-12-19,5493.65
Vendor12,1592,2008-12-19,1133.85
Vendor12,1599,2008-12-20,1462.13
Vendor12,1601,2008-12-20,3508.49
Vendor12,1608,2008-12-20,4805.27
Vendor12,1611,2008-12-28,2684.58
Vendor12,1620,2008-12-28,7853.96
Vendor12,1623,2009-01-05,4420.07
Vendor12,1628,2009-01-05,2885.75
Vendor12,1635,2009-01-05,2624.45
Vendor12,1639,2009-01-13,8828.57
Vendor12,1645,2009-01-13,2062.74
Vendor12,1646,2009-01-13,2481.49
Vendor12,1655,2009-01-14,7262.0
Vendor12,1660,2009-01-24,5476.65
Vendor12,1664,2009-01-25,1530.83
Vendor12,1670,2009-01-25,4509.09
Vendor12,1673,2009-02-02,3876.98
Vendor12,1680,2009-02-10,6554.38
Vendor12,1685,2009-02-11,9125.15
Vendor12,1689,2009-02-12,1977.88
Vendor12,1695,2009-02-13,1222.73
Vendor12,1699,2009-02-19,7953.88
Vendor12,1703,2009-02-19,2647.42
Vendor12,1706,2009-02-25,7588.13
Vendor12,1711,2009-03-03,2444.51
Vendor12,1718,2009-03-04,2415.28
Vendor12,1721,2009-03-14,2236.02
Vendor12,1729,2009-03-15,2010.39
Vendor12,1733,2009-03-21,7835.71
Vendor12,1739,2009-03-29,1785.84
Vendor12,1741,2009-03-29,2080.18
Vendor12,1747,2009-03-29,5252.41
Vendor12,1751,2009-04-08,1120.69
Vendor12,1757,2009-04-14,7451.07
Vendor12,1761,2009-04-14,3893.47
Vendor12,1767,2009-04-24,1844.92
Vendor12,1774,2009-04-30,1722.46
Vendor12,1776,2009-05-10,4210.49
Vendor12,1783,2009-05-20,1732.37
Vendor12,1786,2009-05-20,1981.86
Vendor12,1794,2009-05-20,3055.0
Vendor12,1796,2009-05-21,2264.0
Vendor12,1805,2009-05-21,1800.9
Vendor12,1808,2009-05-22,1269.95
Vendor12,1812,2009-05-23,9805.38
Vendor12,1819,2009-05-24,1664.34
Vendor12,1821,2009-05-24,1945.09
Vendor12,1829,2009-05-25,2214.55
Vendor12,1835,2009-05-26,8098.51
Vendor12,1837,2009-05-27,2007.54
Vendor12,1842,2009-06-02,8719.46
Vendor12,1846,2009-06-03,1534.2
Vendor12,1854,2009-06-03,1755.24
Vendor12,1857,2009-06-09,5641.94
Vendor12,1861,2009-06-10,1391.39
Vendor12,1866,2009-06-11,1496.04
Vendor12,1872,2009-06-19,1254.91
Vendor12,1880,2009-06-25,4379.82
Vendor12,1884,2009-07-01,4616.61
Vendor12,1889,2009-07-02,2303.77
Vendor12,1892,2009-07-02,1029.7
Vendor12,1900,2009-07-08,1919.09
Vendor12,1904,2009-07-14,1611.0
Vendor12,1906,2009-07-24,5583.26
Vendor12,1912,2009-08-01,4120.95
Vendor12,1920,2009-08-09,1080.2
Vendor12,1925,2009-08-17,5110.12
Vendor12,1927,2009-08-17,9259.12
Vendor12,1932,2009-08-27,1010.28
Vendor12,1937,2009-09-06,1014.74
Vendor12,1941,2009-09-14,1547.76
Vendor12,1946,2009-09-24,1582.33
Vendor12,1953,2009-09-24,3038.0
Vendor12,1957,2009-09-30,3705.23
Vendor12,1961,2009-09-30,3563.1
Vendor12,1969,2009-10-06,1476.89
Vendor12,1971,2009-10-06,1404.82
Vendor12,1978,2009-10-12,1065.64
Vendor12,1984,2009-10-12,1330.57
Vendor12,1988,2009-10-22,1310.47
Vendor12,1995,2009-10-23,7312.52
Vendor12,1999,2009-10-24,1479.59
Vendor12,2003,2009-10-24,8040.15
Vendor12,2007,2009-10-24,2959.2
Vendor12,2015,2009-10-25,6939.57
Vendor12,2019,2009-10-26,4294.06
Vendor12,2025,2009-10-26,4624.48
Vendor12,2030,2009-10-27,9487.91
Vendor12,2033,2009-10-27,1745.08
Vendor12,2040,2009-11-04,7392.03
Vendor12,2045,2009-11-14,4105.12
Vendor12,2049,2009-11-20,7110.27
Vendor12,2054,2009-11-21,4387.66
Vendor12,2060,2009-11-29,5349.64
Vendor12,2063,2009-12-05,4135.87
Vendor12,2067,2009-12-05,1348.91
Vendor12,2075,2009-12-13,3245.15
Vendor12,2079,2009-12-13,2305.58
Vendor12,2085,2009-12-19,9191.8
Vendor12,2089,2009-12-19,1065.81
Vendor12,2091,2009-12-29,3473.46
Vendor12,2099,2010-01-04,4082.84
Vendor12,2103,2010-01-05,1783.76
Vendor12,2110,2010-01-05,6701.47
Vendor12,2113,2010-01-06,2996.81
Vendor12,2118,2010-01-12,3235.17
Vendor12,2123,2010-01-12,4577.32
Vendor12,2126,2010-01-12,8722.05
Vendor12,2133,2010-01-12,9338.67
Vendor12,2138,2010-01-13,1075.03
Vendor12,2145,2010-01-14,7608.24
Vendor12,2149,2010-01-20,1109.19
Vendor12,2152,2010-01-20,9227.26
Vendor12,2157,2010-01-20,6085.6
Vendor12,2161,2010-01-21,1594.83
Vendor12,2170,2010-01-21,1022.99
Vendor12,2175,2010-01-21,2599.01
Vendor12,2176,2010-01-22,1572.16
Vendor12,2181,2010-01-22,1059.45
Vendor12,2188,2010-01-23,3880.82
Vendor12,2195,2010-01-24,1651.52
Vendor12,2196,2010-01-24,8960.73
Vendor12,2201,2010-01-25,7613.87
Vendor12,2209,2010-01-31,1536.51
Vendor12,2212,2010-02-08,2706.03
Vendor12,2218,2010-02-14,2371.21
Vendor12,2225,2010-02-20,1341.26
Vendor12,2228,2010-02-21,1790.32
Vendor12,2234,2010-02-22,3985.5
Vendor12,2238,2010-02-28,5652.4
Vendor12,2244,2010-03-06,3154.57
Vendor12,2249,2010-03-12,6265.79
Vendor12,2251,2010-03-12,5604.73
Vendor12,2260,2010-03-12,1230.95
Vendor12,2261,2010-03-18,1524.97
Vendor12,2270,2010-03-28,3378.37
Vendor12,2274,2010-04-03,2392.7
Vendor12,2279,2010-04-03,4012.08
Vendor12,2284,2010-04-13,8344.57
Vendor12,2289,2010-04-14,2201.55
Vendor12,2294,2010-04-20,3760.13
Vendor12,2296,2010-04-21,7719.71
Vendor12,2305,2010-04-27,2911.96
Vendor12,2306,2010-04-27,1002.3
Vendor12,2313,2010-05-05,2135.97
Vendor12,2318,2010-05-06,1432.64
Vendor12,2322,2010-05-07,3118.38
Vendor12,2326,2010-05-13,3656.02
Vendor12,2335,2010-05-13,5317.36
Vendor12,2337,2010-05-14,2064.64
Vendor12,2345,2010-05-14,4520.89
Vendor12,2348,2010-05-15,9413.18
Vendor12,2352,2010-05-16,7608.16
Vendor12,2359,2010-05-22,1817.05
Vendor12,2361,2010-05-23,1475.15
Vendor12,2366,2010-05-24,1996.82
Vendor12,2371,2010-05-24,7462.18
Vendor12,2380,2010-06-01,6388.33
Vendor12,2381,2010-06-07,2329.41
Vendor12,2387,2010-06-07,3016.08
Vendor12,2393,2010-06-08,1400.76
Vendor12,2399,2010-06-09,4337.25
Vendor12,2401,2010-06-09,3934.98
Vendor12,2407,2010-06-19,7409.66
Vendor12,2414,2010-06-20,8005.79
Vendor12,2416,2010-06-20,3931.36
Vendor12,2425,2010-06-20,5326.64
Vendor12,2428,2010-06-21,2110.74
Vendor12,2433,2010-06-21,2253.28
Vendor12,2437,2010-06-27,2677.25
Vendor12,2441,2010-06-27,2630.89
Vendor12,2449,2010-07-07,2005.16
Vendor12,2451,2010-07-08,1984.35
Vendor12,2460,2010-07-09,6609.56
Vendor12,2461,2010-07-17,3106.92
Vendor12,2466,2010-07-23,1446.21
Vendor12,2474,2010-07-23,3568.42
Vendor12,2479,2010-07-23,4554.19
Vendor12,2482,2010-07-29,5364.74
Vendor12,2490,2010-07-29,1173.66
Vendor12,2494,2010-07-30,3659.34
Vendor12,2499,2010-07-30,1330.26
Vendor12,2503,2010-08-07,2223.24
Vendor12,2506,2010-08-13,1693.47
Vendor12,2511,2010-08-14,1013.55
Vendor12,2516,2010-08-24,8398.57
Vendor12,2522,2010-09-03,6658.49
Vendor12,2529,2010-09-04,7623.14
Vendor12,2531,2010-09-10,4704.62
Vendor12,2536,2010-09-20,4347.46
Vendor12,2542,2010-09-26,3060.87
Vendor12,2548,2010-09-27,5541.51
Vendor12,2552,2010-09-28,1116.54
Vendor12,2556,2010-09-29,1459.78
Vendor12,2565,2010-10-09,4995.85
Vendor12,2568,2010-10-15,1485.51
Vendor12,2575,2010-10-16,2657.18
Vendor12,2576,2010-10-17,4879.93
Vendor12,2585,2010-10-27,4104.99
Vendor12,2588,2010-10-28,3099.36
Vendor12,2593,2010-10-28,9781.48
Vendor12,2600,2010-10-29,2470.57
Vendor12,2602,2010-10-29,1774.27
Vendor12,2609,2010-11-06,1397.76
Vendor12,2612,2010-11-06,2580.87
Vendor12,2616,2010-11-06,1796.53
Vendor12,2621,2010-11-16,2490.86
Vendor12,2628,2010-11-17,8284.16
Vendor12,2631,2010-11-17,5562.44
Vendor12,2636,2010-11-18,3278.01
Vendor12,2641,2010-11-28,3713.39
Vendor12,2648,2010-11-29,1233.78
Vendor12,2653,2010-11-29,4459.7
Vendor12,2659,2010-11-29,2248.9
Vendor12,2663,2010-11-30,6624.19
Vendor12,2670,2010-11-30,2287.72
Vendor12,2675,2010-12-08,1737.39
Vendor12,2676,2010-12-09,1082.06
Vendor12,2681,2010-12-09,3243.52
Vendor12,2688,2010-12-10,2138.78
Vendor12,2693,2010-12-10,1903.5
Vendor12,2698,2010-12-10,1111.95
Vendor12,2703,2010-12-11,4867.76
Vendor12,2707,2010-12-11,9437.21
Vendor12,2713,2010-12-17,2150.05
Vendor12,2719,2010-12-17,3202.35
Vendor12,2722,2010-12-27,6220.9
Vendor12,2727,2010-12-28,3782.53
Vendor12,2735,2010-12-28,5887.88
Vendor12,2739,2011-01-07,7220.02
Vendor12,2743,2011-01-07,4883.3
Vendor12,2748,2011-01-15,4285.68
Vendor12,2755,2011-01-15,3912.41
Vendor12,2758,2011-01-16,1884.58
Vendor12,2761,2011-01-17,6331.87
Vendor12,2769,2011-01-25,2365.43
Vendor12,2773,2011-01-25,1513.07
Vendor12,2776,2011-01-25,1225.49
Vendor12,2782,2011-01-25,2159.0
Vendor12,2788,2011-01-31,1056.52
Vendor12,2794,2011-02-06,2282.91
Vendor12,2799,2011-02-12,6349.42
Vendor12,2803,2011-02-13,3682.44
Vendor12,2809,2011-02-14,2416.72
Vendor12,2814,2011-02-15,8381.28
Vendor12,2819,2011-02-16,8126.67
Vendor12,2821,2011-02-22,1600.36
Vendor12,2828,2011-02-22,4325.19
Vendor12,2832,2011-02-23,1035.03
Vendor12,2839,2011-03-03,3987.06
Vendor12,2842,2011-03-11,2312.63
Vendor12,2847,2011-03-17,1317.05
Vendor12,2852,2011-03-18,1664.14
Vendor12,2859,2011-03-18,8972.78
Vendor12,2863,2011-03-26,2724.18
Vendor12,2868,2011-04-05,4470.83
Vendor12,2872,2011-04-11,6287.83
Vendor12,2878,2011-04-12,5142.57
Vendor12,2884,2011-04-13,3741.87
Vendor12,2889,2011-04-13,1125.64
Vendor12,2892,2011-04-13,1115.56
Vendor12,2899,2011-04-23,9653.42
Vendor12,2903,2011-04-24,4801.5
Vendor12,2906,2011-04-25,3689.77
Vendor12,2912,2011-04-25,1788.93
Vendor12,2919,2011-05-01,2479.75
Vendor12,2922,2011-05-01,6095.08
Vendor12,2927,2011-05-01,3559.33
Vendor12,2931,2011-05-02,2001.54
Vendor12,2936,2011-05-03,3094.86
Vendor12,2941,2011-05-09,1133.77
Vendor12,2950,2011-05-19,2031.02
Vendor12,2953,2011-05-20,8428.03
Vendor12,2958,2011-05-21,3843.84
Vendor12,2964,2011-05-22,3330.9
Vendor12,2970,2011-05-28,1449.66
Vendor12,2973,2011-06-05,2812.3
Vendor12,2977,2011-06-05,2865.34
Vendor12,2983,2011-06-13,1846.03
Vendor12,2987,2011-06-13,2415.43
Vendor12,2993,2011-06-13,2934.78
Vendor12,2996,2011-06-14,4536.9
Vendor12,3004,2011-06-15,2784.57
Vendor12,3009,2011-06-21,1140.55
Vendor12,3011,2011-07-01,1488.9
Vendor12,3019,2011-07-07,9164.76
Vendor12,3024,2011-07-13,1086.61
Vendor12,3027,2011-07-14,5957.02
Vendor12,3033,2011-07-14,8102.63
Vendor12,3036,2011-07-15,5470.56
Vendor12,3045,2011-07-23,1785.86
Vendor12,3048,2011-07-23,5611.56
Vendor12,3053,2011-07-24,2095.25
Vendor12,3056,2011-08-01,8057.45
Vendor12,3063,2011-08-09,2283.37
Vendor12,3068,2011-08-09,1408.35
Vendor12,3073,2011-08-15,4325.61
Vendor12,3079,2011-08-16,8243.56
Vendor12,3085,2011-08-22,3183.43
Vendor12,3088,2011-08-22,3192.05
Vendor12,3095,2011-08-23,2850.02
Vendor12,3096,2011-08-23,1739.22
Vendor12,3101,2011-08-24,2914.85
Vendor12,3108,2011-09-01,4116.48
Vendor12,3111,2011-09-02,4584.29
Vendor12,3117,2011-09-02,4638.77
Vendor12,3123,2011-09-08,5456.15
Vendor12,3129,2011-09-08,6090.41
Vendor12,3135,2011-09-09,4521.46
Vendor12,3138,2011-09-09,2433.11
Vendor12,3144,2011-09-17,7867.61
Vendor12,3149,2011-09-23,3008.26
Vendor12,3154,2011-09-23,4621.65
Vendor12,3158,2011-09-23,1804.8
Vendor12,3163,2011-10-03,1565.2
Vendor12,3170,2011-10-09,2142.04
Vendor12,3171,2011-10-15,8006.27
Vendor12,3180,2011-10-16,2941.98
Vendor12,3181,2011-10-22,6157.81
Vendor12,3187,2011-10-30,5055.95
Vendor12,3191,2011-10-30,6516.88
Vendor12,3196,2011-10-31,3128.27
Vendor12,3201,2011-11-10,2755.39
Vendor12,3207,2011-11-10,9450.92
Vendor12,3212,2011-11-11,4293.5
Vendor12,3218,2011-11-11,5786.2
Vendor12,3224,2011-11-12,2905.15
Vendor12,3230,2011-11-12,1829.49
Vendor12,3232,2011-11-22,7539.4
Vendor12,3237,2011-11-23,6450.05
Vendor12,3245,2011-11-29,8234.58
Vendor12,3250,2011-12-09,7159.6
Vendor12,3253,2011-12-09,6226.55
Vendor12,3257,2011-12-09,3590.8
Vendor12,3261,2011-12-10,1203.36
Vendor12,3268,2011-12-18,4795.26
Vendor12,3272,2011-12-19,3944.04
Vendor12,3280,2011-12-19,1454.69
Vendor12,3283,2011-12-27,1878.61
Vendor12,3289,2012-01-04,1530.69
Vendor12,3291,2012-01-05,2431.19
Vendor12,3300,2012-01-15,5206.04
Vendor12,3304,2012-01-15,8848.47
Vendor12,3309,2012-01-23,1477.08
Vendor12,3312,2012-01-29,3658.18
Vendor12,3316,2012-01-29,5757.47
Vendor12,3323,2012-02-06,6874.07
Vendor12,3329,2012-02-14,3422.82
Vendor12,3331,2012-02-24,3420.38
Vendor12,3337,2012-02-24,9617.93
Vendor12,3345,2012-03-01,1417.24
Vendor12,3346,2012-03-01,2780.51
Vendor12,3352,2012-03-11,2416.18
Vendor12,3357,2012-03-19,6268.41
Vendor12,3363,2012-03-19,6350.6
Vendor12,3367,2012-03-20,4118.17
Vendor12,3372,2012-03-20,5205.28
Vendor12,3379,2012-03-21,1358.1
Vendor12,3385,2012-03-31,2604.41
Vendor12,3388,2012-04-10,2864.15
Vendor12,3393,2012-04-20,6501.84
Vendor12,3398,2012-04-28,4879.67
Vendor12,3401,2012-05-08,3091.34
Vendor12,3408,2012-05-09,5062.16
Vendor12,3412,2012-05-19,7210.46
Vendor12,3419,2012-05-20,1021.28
Vendor12,3424,2012-05-20,1036.24
Vendor12,3430,2012-05-20,4784.86
Vendor12,3431,2012-05-26,8224.64
Vendor12,3438,2012-06-01,4888.94
Vendor12,3443,2012-06-01,1789.22
Vendor12,3448,2012-06-09,8287.03
Vendor12,3452,2012-06-09,5499.07
Vendor12,3457,2012-06-15,2232.26
Vendor12,3465,2012-06-16,1059.4
Vendor12,3467,2012-06-16,2489.29
Vendor19,10,2006-01-01,2502.07
Vendor19,12,2006-01-07,5605.47
Vendor19,17,2006-01-15,1635.74
Vendor19,21,2006-01-15,7248.57
Vendor19,29,2006-01-16,2518.46
Vendor19,33,2006-01-16,9257.03
Vendor19,39,2006-01-16,1706.22
Vendor19,41,2006-01-16,2801.2
Vendor19,46,2006-01-26,1512.82
Vendor19,52,2006-01-26,1585.34
Vendor19,57,2006-02-01,8519.91
Vendor19,63,2006-02-02,1892.6
Vendor19,66,2006-02-08,8035.34
Vendor19,72,2006-02-14,2766.01
Vendor19,80,2006-02-15,5023.91
Vendor19,82,2006-02-15,5314.63
Vendor19,88,2006-02-16,3271.81
Vendor19,94,2006-02-16,3130.54
Vendor19,97,2006-02-17,4400.64
Vendor19,103,2006-02-23,3553.86
Vendor19,108,2006-03-05,6596.27
Vendor19,113,2006-03-05,3125.35
Vendor19,118,2006-03-06,5129.68
Vendor19,121,2006-03-14,1123.94
Vendor19,127,2006-03-15,9133.9
Vendor19,133,2006-03-15,3030.17
Vendor19,139,2006-03-16,1183.57
Vendor19,143,2006-03-16,3310.27
Vendor19,147,2006-03-16,1081.58
Vendor19,153,2006-03-17,1868.49
Vendor19,159,2006-03-23,3246.12
Vendor19,164,2006-03-23,1754.85
Vendor19,167,2006-03-31,1213.48
Vendor19,173,2006-03-31,1244.67
Vendor19,177,2006-03-31,1386.29
Vendor19,181,2006-04-01,7594.55
Vendor19,187,2006-04-01,2640.98
Vendor19,194,2006-04-02,4300.06
Vendor19,198,2006-04-02,8568.12
Vendor19,204,2006-04-12,9353.82
Vendor19,208,2006-04-12,1040.66
Vendor19,212,2006-04-13,8745.15
Vendor19,220,2006-04-19,1286.87
Vendor19,221,2006-04-20,1441.18
Vendor19,228,2006-04-21,2037.77
Vendor19,234,2006-04-27,4805.08
Vendor19,237,2006-05-03,4072.42
Vendor19,243,2006-05-03,1097.53
Vendor19,246,2006-05-13,3836.9
Vendor19,253,2006-05-13,1394.61
Vendor19,258,2006-05-13,2875.81
Vendor19,264,2006-05-14,2211.04
Vendor19,267,2006-05-15,6224.45
Vendor19,273,2006-05-21,5697.63
Vendor19,280,2006-05-21,4494.25
Vendor19,285,2006-05-22,8434.03
Vendor19,287,2006-05-23,6938.29
Vendor19,295,2006-05-23,2694.16
Vendor19,296,2006-05-24,5536.16
Vendor19,303,2006-05-24,3732.27
Vendor19,307,2006-06-01,1362.87
Vendor19,315,2006-06-02,9950.21
Vendor19,320,2006-06-03,1116.58
Vendor19,325,2006-06-04,4821.84
Vendor19,328,2006-06-04,3301.83
Vendor19,332,2006-06-10,3186.48
Vendor19,336,2006-06-10,1495.79
Vendor19,342,2006-06-11,2825.09
Vendor19,349,2006-06-11,2068.68
Vendor19,352,2006-06-17,1819.36
Vendor19,359,2006-06-18,2444.64
Vendor19,361,2006-06-19,2020.15
Vendor19,369,2006-06-20,1198.15
Vendor19,371,2006-06-20,4659.46
Vendor19,380,2006-06-26,4405.28
Vendor19,381,2006-06-27,6072.59
Vendor19,387,2006-06-28,1204.32
Vendor19,392,2006-06-29,2133.98
Vendor19,397,2006-06-29,6129.31
Vendor19,403,2006-07-07,1011.89
Vendor19,406,2006-07-13,7818.36
Vendor19,415,2006-07-19,2942.91
Vendor19,417,2006-07-19,2175.14
Vendor19,424,2006-07-19,1968.73
Vendor19,428,2006-07-27,1476.35
Vendor19,433,2006-08-04,1850.35
Vendor19,436,2006-08-05,3690.48
Vendor19,444,2006-08-15,5404.58
Vendor19,450,2006-08-15,1209.09
Vendor19,453,2006-08-15,1978.03
Vendor19,459,2006-08-15,4103.98
Vendor19,461,2006-08-16,1318.08
Vendor19,466,2006-08-22,1771.72
Vendor19,471,2006-08-23,2649.72
Vendor19,478,2006-08-23,2244.1
Vendor19,485,2006-08-24,2440.87
Vendor19,490,2006-08-25,6391.77
Vendor19,491,2006-09-04,1201.32
Vendor19,497,2006-09-05,9722.66
Vendor19,504,2006-09-05,7625.2
Vendor19,509,2006-09-11,9942.67
Vendor19,513,2006-09-12,9899.5
Vendor19,517,2006-09-22,1227.81
Vendor19,522,2006-09-23,2993.43
Vendor19,529,2006-09-29,6308.68
Vendor19,534,2006-09-29,7402.03
Vendor19,539,2006-09-29,2688.44
Vendor19,544,2006-09-30,9475.6
Vendor19,548,2006-09-30,8792.72
Vendor19,552,2006-10-08,1600.27
Vendor19,559,2006-10-08,1535.73
Vendor19,563,2006-10-14,1865.11
Vendor19,567,2006-10-14,2187.79
Vendor19,573,2006-10-20,9238.66
Vendor19,577,2006-10-20,1653.49
Vendor19,581,2006-10-20,1838.18
Vendor19,587,2006-10-28,8211.63
Vendor19,593,2006-10-29,7219.61
Vendor19,599,2006-10-29,3918.16
Vendor19,601,2006-10-30,5028.07
Vendor19,609,2006-10-31,1796.86
Vendor19,611,2006-11-06,6520.51
Vendor19,616,2006-11-06,1616.18
Vendor19,622,2006-11-07,1994.5
Vendor19,627,2006-11-07,3758.03
Vendor19,631,2006-11-15,5044.61
Vendor19,637,2006-11-21,8838.9
Vendor19,641,2006-11-22,1277.12
Vendor19,650,2006-11-28,5095.47
Vendor19,651,2006-12-06,9832.5
Vendor19,660,2006-12-12,2103.3
Vendor19,665,2006-12-22,7091.24
Vendor19,668,2006-12-22,7979.16
Vendor19,671,2006-12-22,9123.64
Vendor19,677,2006-12-28,5080.67
Vendor19,683,2007-01-03,2656.99
Vendor19,687,2007-01-04,8608.2
Vendor19,695,2007-01-04,1356.88
Vendor19,700,2007-01-05,9234.94
Vendor19,701,2007-01-11,2404.5
Vendor19,706,2007-01-17,1111.32
Vendor19,711,2007-01-18,1510.55
Vendor19,718,2007-01-18,1629.16
Vendor19,723,2007-01-19,7548.01
Vendor19,730,2007-01-19,4054.97
Vendor19,733,2007-01-25,1151.89
Vendor19,736,2007-01-25,7406.08
Vendor19,741,2007-01-26,3741.22
Vendor19,748,2007-01-26,3712.16
Vendor19,753,2007-01-27,1143.12
Vendor19,757,2007-01-28,4315.66
Vendor19,763,2007-01-29,3504.88
Vendor19,769,2007-02-04,3211.55
Vendor19,773,2007-02-12,1994.79
Vendor19,776,2007-02-18,1869.77
Vendor19,785,2007-02-28,2213.4
Vendor19,789,2007-02-28,1610.4
Vendor19,793,2007-03-01,2485.05
Vendor19,799,2007-03-02,2096.44
Vendor19,803,2007-03-08,2628.01
Vendor19,806,2007-03-14,1096.81
Vendor19,813,2007-03-14,5104.46
Vendor19,818,2007-03-20,2423.09
Vendor19,823,2007-03-26,2233.82
Vendor19,829,2007-03-26,6758.25
Vendor19,834,2007-04-03,1964.17
Vendor19,840,2007-04-04,1158.6
Vendor19,845,2007-04-04,3405.41
Vendor19,847,2007-04-04,8821.73
Vendor19,854,2007-04-05,2543.41
Vendor19,856,2007-04-06,3509.84
Vendor19,861,2007-04-06,3595.52
Vendor19,870,2007-04-12,4079.42
Vendor19,874,2007-04-13,3935.7
Vendor19,880,2007-04-13,7870.85
Vendor19,883,2007-04-19,1221.55
Vendor19,890,2007-04-19,1295.97
Vendor19,894,2007-04-20,8554.17
Vendor19,896,2007-04-20,2838.74
Vendor19,902,2007-04-28,1992.3
Vendor19,907,2007-04-28,2693.2
Vendor19,914,2007-04-29,8365.78
Vendor19,917,2007-04-30,6930.89
Vendor19,923,2007-05-08,1889.79
Vendor19,929,2007-05-14,5846.6
Vendor19,931,2007-05-15,4265.64
Vendor19,937,2007-05-16,3724.52
Vendor19,944,2007-05-17,7183.92
Vendor19,947,2007-05-23,6895.36
Vendor19,954,2007-05-24,5006.16
Vendor19,960,2007-05-24,4848.22
Vendor19,962,2007-05-25,2469.45
Vendor19,967,2007-05-25,6703.15
Vendor19,975,2007-06-02,8560.51
Vendor19,977,2007-06-02,3791.49
Vendor19,981,2007-06-02,1161.32
Vendor19,989,2007-06-03,2055.87
Vendor19,994,2007-06-11,7352.37
Vendor19,997,2007-06-12,4637.06
Vendor19,1004,2007-06-20,6701.64
Vendor19,1009,2007-06-28,2473.17
Vendor19,1015,2007-07-08,6804.03
Vendor19,1020,2007-07-14,1438.74
Vendor19,1025,2007-07-14,5918.98
Vendor19,1030,2007-07-14,8808.35
Vendor19,1035,2007-07-20,5717.8
Vendor19,1036,2007-07-20,1580.03
Vendor19,1045,2007-07-21,2969.09
Vendor19,1050,2007-07-21,4756.65
Vendor19,1055,2007-07-22,7260.11
Vendor19,1059,2007-08-01,1858.45
Vendor19,1062,2007-08-01,5791.92
Vendor19,1066,2007-08-01,4659.33
Vendor19,1073,2007-08-02,1131.44
Vendor19,1079,2007-08-02,2436.32
Vendor19,1083,2007-08-12,2767.51
Vendor19,1090,2007-08-13,5587.59
Vendor19,1092,2007-08-14,6580.0
Vendor19,1099,2007-08-24,1924.26
Vendor19,1104,2007-08-25,2509.9
Vendor19,1109,2007-08-25,1064.01
Vendor19,1113,2007-08-26,1244.71
Vendor19,1117,2007-09-01,1680.0
Vendor19,1124,2007-09-07,1712.87
Vendor19,1126,2007-09-08,1872.22
Vendor19,1132,2007-09-18,3504.98
Vendor19,1137,2007-09-19,6731.66
Vendor19,1142,2007-09-25,1792.45
Vendor19,1149,2007-10-05,9885.37
Vendor19,1153,2007-10-06,1693.61
Vendor19,1156,2007-10-07,4151.33
Vendor19,1164,2007-10-13,7121.18
Vendor19,1169,2007-10-14,1073.31
Vendor19,1174,2007-10-20,9362.85
Vendor19,1179,2007-10-26,8730.61
Vendor19,1184,2007-11-05,4459.89
Vendor19,1187,2007-11-13,4585.21
Vendor19,1193,2007-11-21,1423.04
Vendor19,1198,2007-11-21,7841.01
Vendor19,1201,2007-11-22,1053.54
Vendor19,1209,2007-11-22,1708.54
Vendor19,1214,2007-11-28,3759.61
Vendor19,1220,2007-12-06,8626.01
Vendor19,1221,2007-12-12,2520.46
Vendor19,1230,2007-12-12,1005.39
Vendor19,1233,2007-12-20,2167.89
Vendor19,1239,2007-12-20,1673.61
Vendor19,1244,2007-12-26,2991.2
Vendor19,1248,2007-12-26,1787.05
Vendor19,1255,2008-01-05,2142.12
Vendor19,1256,2008-01-05,5200.46
Vendor19,1265,2008-01-13,4491.46
Vendor19,1267,2008-01-13,2604.99
Vendor19,1273,2008-01-13,1727.0
Vendor19,1276,2008-01-14,3718.63
Vendor19,1284,2008-01-24,5028.28
Vendor19,1286,2008-02-03,7578.23
Vendor19,1293,2008-02-04,6524.38
Vendor19,1297,2008-02-14,1950.96
Vendor19,1302,2008-02-15,4342.92
Vendor19,1307,2008-02-25,2650.62
Vendor19,1311,2008-02-26,1417.55
Vendor19,1320,2008-03-03,1405.08
Vendor19,1323,2008-03-04,1673.81
Vendor19,1330,2008-03-14,3341.48
Vendor19,1334,2008-03-20,5725.37
Vendor19,1336,2008-03-20,1089.67
Vendor19,1344,2008-03-28,5612.2
Vendor19,1350,2008-04-07,2691.31
Vendor19,1354,2008-04-13,3595.03
Vendor19,1359,2008-04-21,7414.54
Vendor19,1363,2008-04-29,1359.42
Vendor19,1370,2008-04-30,2260.82
Vendor19,1372,2008-04-30,3007.17
Vendor19,1380,2008-05-01,4288.0
Vendor19,1383,2008-05-07,1217.92
Vendor19,1389,2008-05-13,7947.07
Vendor19,1395,2008-05-13,5405.13
Vendor19,1398,2008-05-13,2006.84
Vendor19,1403,2008-05-23,1759.03
Vendor19,1410,2008-05-24,1547.66
Vendor19,1414,2008-05-30,9661.47
Vendor19,1418,2008-05-30,4440.17
Vendor19,1421,2008-06-07,5732.81
Vendor19,1427,2008-06-07,1612.2
Vendor19,1434,2008-06-08,9342.36
Vendor19,1440,2008-06-09,1842.39
Vendor19,1443,2008-06-15,5385.2
Vendor19,1447,2008-06-15,9003.81
Vendor19,1453,2008-06-16,1456.26
Vendor19,1459,2008-06-16,5089.69
Vendor19,1464,2008-06-24,5771.24
Vendor19,1467,2008-06-25,4824.78
Vendor19,1473,2008-06-25,2528.62
Vendor19,1479,2008-07-01,1414.91
Vendor19,1481,2008-07-07,4320.65
Vendor19,1488,2008-07-17,8452.21
Vendor19,1494,2008-07-17,8794.18
Vendor19,1498,2008-07-25,2423.43
Vendor19,1501,2008-07-31,1033.65
Vendor19,1507,2008-08-01,1739.35
Vendor19,1513,2008-08-02,1199.64
Vendor19,1520,2008-08-03,1554.66
Vendor19,1522,2008-08-04,1315.43
Vendor19,1529,2008-08-10,1830.81
Vendor19,1531,2008-08-10,4980.84
Vendor19,1540,2008-08-11,3507.4
Vendor19,1542,2008-08-17,7516.14
Vendor19,1549,2008-08-23,2260.08
Vendor19,1552,2008-08-24,8343.3
Vendor19,1558,2008-08-25,8218.9
Vendor19,1564,2008-08-31,4612.7
Vendor19,1568,2008-08-31,9185.27
Vendor19,1575,2008-09-06,1869.84
Vendor19,1580,2008-09-07,3076.76
Vendor19,1582,2008-09-17,2294.3
Vendor19,1589,2008-09-25,1778.62
Vendor19,1593,2008-09-25,3954.62
Vendor19,1600,2008-09-26,5431.35
Vendor19,1601,2008-10-06,7932.58
Vendor19,1607,2008-10-14,3021.03
Vendor19,1613,2008-10-15,6043.27
Vendor19,1616,2008-10-15,1087.1
Vendor19,1621,2008-10-15,1405.09
Vendor19,1628,2008-10-16,2224.93
Vendor19,1634,2008-10-26,6616.03
Vendor19,1637,2008-11-01,4413.12
Vendor19,1641,2008-11-01,1644.8
Vendor19,1647,2008-11-01,9792.91
Vendor19,1651,2008-11-01,5240.2
Vendor19,1659,2008-11-02,6703.29
Vendor19,1664,2008-11-10,7801.86
Vendor19,1669,2008-11-20,3220.65
Vendor19,1672,2008-11-20,2753.33
Vendor19,1679,2008-11-20,2475.01
Vendor19,1684,2008-11-21,2620.31
Vendor19,1689,2008-11-22,4720.97
Vendor19,1695,2008-11-22,6829.27
Vendor19,1700,2008-11-28,1262.43
Vendor19,1705,2008-11-28,5128.88
Vendor19,1709,2008-11-29,6474.28
Vendor19,1712,2008-12-07,7560.14
Vendor19,1720,2008-12-13,1411.39
Vendor19,1724,2008-12-14,3594.15
Vendor19,1729,2008-12-20,2607.91
Vendor19,1732,2008-12-26,2638.46
Vendor19,1736,2009-01-05,1789.7
Vendor19,1743,2009-01-06,2129.45
Vendor19,1747,2009-01-16,1265.98
Vendor19,1752,2009-01-24,1757.64
Vendor19,1759,2009-02-03,1013.02
Vendor19,1761,2009-02-04,1663.01
Vendor19,1769,2009-02-04,7403.51
Vendor19,1771,2009-02-14,1181.26
Vendor19,1780,2009-02-22,3309.51
Vendor19,1782,2009-02-28,1339.28
Vendor19,1787,2009-03-06,1081.52
Vendor19,1792,2009-03-07,2194.54
Vendor19,1799,2009-03-08,2157.99
Vendor19,1802,2009-03-09,3003.2
Vendor19,1806,2009-03-09,1213.95
Vendor19,1814,2009-03-15,6443.71
Vendor19,1816,2009-03-23,1895.26
Vendor19,1823,2009-03-24,2323.62
Vendor19,1830,2009-04-01,6275.93
Vendor19,1831,2009-04-11,5358.81
Vendor19,1839,2009-04-21,5399.2
Vendor19,1845,2009-05-01,6803.52
Vendor19,1846,2009-05-09,6604.06
Vendor19,1854,2009-05-09,2148.61
Vendor19,1859,2009-05-10,9417.42
Vendor19,1864,2009-05-20,7575.3
Vendor19,1870,2009-05-30,6811.18
Vendor19,1872,2009-05-30,1998.22
Vendor19,1878,2009-06-05,1103.35
Vendor19,1885,2009-06-06,9882.52
Vendor19,1888,2009-06-12,1376.53
Vendor19,1891,2009-06-13,3180.92
Vendor19,1896,2009-06-13,5381.56
Vendor19,1902,2009-06-19,3929.52
Vendor19,1907,2009-06-19,2522.25
Vendor19,1913,2009-06-19,1258.13
Vendor19,1920,2009-06-20,9853.49
Vendor19,1925,2009-06-30,1812.4
Vendor19,1928,2009-07-01,2153.57
Vendor19,1933,2009-07-07,1220.89
Vendor19,1937,2009-07-07,4245.26
Vendor19,1943,2009-07-07,5275.31
Vendor19,1950,2009-07-08,7961.72
Vendor19,1951,2009-07-09,1939.07
Vendor19,1956,2009-07-19,1399.82
Vendor19,1961,2009-07-20,5245.52
Vendor19,1968,2009-07-21,3854.79
Vendor19,1972,2009-07-31,1964.6
Vendor19,1976,2009-08-08,1193.83
Vendor19,1985,2009-08-09,1773.04
Vendor19,1989,2009-08-10,2581.74
Vendor19,1993,2009-08-16,2833.75
Vendor19,1999,2009-08-16,2374.88
Vendor19,2005,2009-08-16,4746.82
Vendor19,2006,2009-08-22,1013.96
Vendor19,2011,2009-08-23,3597.13
Vendor19,2019,2009-08-23,1119.53
Vendor19,2024,2009-08-23,1606.29
Vendor19,2029,2009-08-24,6760.69
Vendor19,2033,2009-08-25,3992.88
Vendor19,2036,2009-08-25,2807.96
Vendor19,2041,2009-08-25,1973.19
Vendor19,2049,2009-08-31,3740.99
Vendor19,2051,2009-08-31,3045.45
Vendor19,2060,2009-09-10,1134.95
Vendor19,2063,2009-09-11,3430.39
Vendor19,2070,2009-09-19,3121.13
Vendor19,2072,2009-09-20,6194.19
Vendor19,2078,2009-09-26,6453.87
Vendor19,2085,2009-09-26,4624.71
Vendor19,2090,2009-09-26,3710.93
Vendor19,2095,2009-09-27,1341.2
Vendor19,2097,2009-09-27,1721.65
Vendor19,2104,2009-09-27,2617.98
Vendor19,2110,2009-09-27,3804.92
Vendor19,2112,2009-09-28,1052.89
Vendor19,2117,2009-10-04,7181.6
Vendor19,2124,2009-10-04,7279.69
Vendor19,2127,2009-10-10,9654.83
Vendor19,2133,2009-10-10,8475.44
Vendor19,2136,2009-10-16,5682.29
Vendor19,2141,2009-10-22,4270.13
Vendor19,2150,2009-10-30,8085.49
Vendor19,2154,2009-10-31,4900.32
Vendor19,2159,2009-11-10,5868.82
Vendor19,2165,2009-11-10,6154.58
Vendor19,2170,2009-11-20,1608.68
Vendor19,2174,2009-11-20,2823.42
Vendor19,2179,2009-11-20,1029.46
Vendor19,2181,2009-11-26,8105.88
Vendor19,2187,2009-12-02,1661.0
Vendor19,2193,2009-12-10,3323.16
Vendor19,2198,2009-12-11,4113.93
Vendor19,2203,2009-12-11,1288.55
Vendor19,2207,2009-12-19,9313.82
Vendor19,2211,2009-12-19,2958.03
Vendor19,2216,2009-12-20,2033.82
Vendor19,2223,2009-12-20,1909.7
Vendor19,2230,2009-12-26,7519.44
Vendor19,2233,2010-01-01,2779.93
Vendor19,2238,2010-01-01,3043.53
Vendor19,2242,2010-01-07,1232.06
Vendor19,2250,2010-01-08,1409.83
Vendor19,2254,2010-01-09,2525.1
Vendor19,2258,2010-01-19,1696.28
Vendor19,2261,2010-01-19,4742.92
Vendor19,2268,2010-01-20,1520.07
Vendor19,2271,2010-01-26,3397.37
Vendor19,2276,2010-02-03,1068.38
Vendor19,2284,2010-02-09,3081.04
Vendor19,2290,2010-02-19,2754.93
Vendor19,2292,2010-02-20,2267.48
Vendor19,2296,2010-03-02,4141.62
Vendor19,2305,2010-03-10,8713.78
Vendor19,2307,2010-03-11,3725.17
Vendor19,2311,2010-03-12,2708.72
Vendor19,2318,2010-03-13,5967.89
Vendor19,2321,2010-03-19,8590.8
Vendor19,2328,2010-03-19,2445.16
Vendor19,2331,2010-03-29,8402.75
Vendor19,2337,2010-03-30,7961.0
Vendor19,2342,2010-03-30,2565.09
Vendor19,2350,2010-04-05,2188.29
Vendor19,2354,2010-04-05,1461.0
Vendor19,2358,2010-04-06,1608.22
Vendor19,2365,2010-04-07,1174.51
Vendor19,2370,2010-04-13,6199.07
Vendor19,2371,2010-04-14,6512.21
Vendor19,2377,2010-04-14,2430.12
Vendor19,2385,2010-04-14,4594.66
Vendor19,2390,2010-04-24,7271.43
Vendor19,2394,2010-04-30,4224.32
Vendor19,2399,2010-04-30,9687.45
Vendor19,2403,2010-05-06,1111.66
Vendor19,2410,2010-05-06,1404.9
Vendor19,2415,2010-05-12,8047.65
Vendor19,2417,2010-05-12,9171.12
Vendor19,2424,2010-05-13,1158.24
Vendor19,2429,2010-05-13,6606.76
Vendor19,2433,2010-05-14,5992.85
Vendor19,2438,2010-05-14,8242.72
Vendor19,2442,2010-05-14,5690.6
Vendor19,2450,2010-05-14,2320.45
Vendor19,2454,2010-05-14,3746.53
Vendor19,2460,2010-05-14,1428.44
Vendor19,2461,2010-05-20,1726.39
Vendor19,2468,2010-05-21,3572.52
Vendor19,2474,2010-05-21,1171.84
Vendor19,2479,2010-05-27,8780.57
Vendor19,2481,2010-06-06,1204.61
Vendor19,2490,2010-06-12,1563.17
Vendor19,2492,2010-06-18,5720.09
Vendor19,2497,2010-06-19,1960.48
Vendor19,2501,2010-06-20,2404.41
Vendor19,2509,2010-06-21,2847.47
Vendor19,2515,2010-07-01,3593.06
Vendor19,2516,2010-07-01,6236.77
Vendor19,2523,2010-07-02,2316.31
Vendor19,2529,2010-07-02,2700.84
Vendor19,2533,2010-07-03,1420.92
Vendor19,2536,2010-07-09,3267.19
Vendor19,2541,2010-07-19,1207.82
Vendor19,2550,2010-07-25,2262.73
Vendor19,2555,2010-07-26,2017.27
Vendor19,2556,2010-07-26,4678.02
Vendor19,2565,2010-07-26,2695.56
Vendor19,2567,2010-07-26,1943.49
Vendor19,2572,2010-07-27,1904.38
Vendor19,2578,2010-07-28,1343.88
Vendor19,2583,2010-08-07,1866.21
Vendor19,2587,2010-08-08,6953.66
Vendor19,2595,2010-08-09,4428.72
Vendor19,2598,2010-08-09,3645.72
Vendor19,2601,2010-08-10,1822.58
Vendor19,2608,2010-08-10,1097.99
Vendor19,2614,2010-08-10,1206.79
Vendor19,2617,2010-08-11,2035.04
Vendor19,2621,2010-08-11,2790.12
Vendor19,2629,2010-08-17,1714.6
Vendor19,2634,2010-08-23,3667.84
Vendor19,2636,2010-08-24,4328.44
Vendor19,2642,2010-08-30,4290.46
Vendor19,2649,2010-09-05,4148.26
Vendor19,2651,2010-09-06,2547.33
Vendor19,2659,2010-09-07,1240.14
Vendor19,2663,2010-09-08,9053.37
Vendor19,2669,2010-09-14,5332.99
Vendor19,2673,2010-09-20,3675.68
Vendor19,2680,2010-09-21,3874.59
Vendor19,2683,2010-09-29,5489.16
Vendor19,2690,2010-10-07,2188.99
Vendor19,2693,2010-10-07,2932.56
Vendor19,2697,2010-10-08,1125.5
Vendor19,2701,2010-10-16,3352.79
Vendor19,2709,2010-10-26,3034.39
Vendor19,2714,2010-11-01,9811.04
Vendor19,2718,2010-11-01,1563.47
Vendor19,2723,2010-11-07,9702.2
Vendor19,2727,2010-11-17,2074.96
Vendor19,2733,2010-11-18,2001.97
Vendor19,2738,2010-11-19,1768.71
Vendor19,2741,2010-11-25,1301.21
Vendor19,2746,2010-12-03,3591.0
Vendor19,2752,2010-12-03,1638.55
Vendor19,2756,2010-12-11,7483.56
Vendor19,2765,2010-12-12,4405.43
Vendor19,2768,2010-12-20,1148.97
Vendor19,2772,2010-12-30,4421.65
Vendor19,2777,2011-01-05,2108.42
Vendor19,2784,2011-01-05,1275.79
Vendor19,2789,2011-01-06,4310.68
Vendor19,2792,2011-01-07,7038.27
Vendor19,2798,2011-01-13,4207.26
Vendor19,2801,2011-01-19,5843.26
Vendor19,2806,2011-01-27,6301.45
Vendor19,2814,2011-02-04,1749.8
Vendor19,2816,2011-02-10,6645.13
Vendor19,2825,2011-02-10,1417.6
Vendor19,2827,2011-02-11,3549.47
Vendor19,2835,2011-02-12,4846.27
Vendor19,2838,2011-02-13,2648.07
Vendor19,2844,2011-02-14,9133.3
Vendor19,2848,2011-02-24,2053.47
Vendor19,2851,2011-02-25,8634.27
Vendor19,2857,2011-02-26,4553.26
Vendor19,2862,2011-03-04,1280.96
Vendor19,2870,2011-03-12,3177.32
Vendor19,2875,2011-03-12,1630.97
Vendor19,2880,2011-03-18,4677.35
Vendor19,2885,2011-03-28,6414.02
Vendor19,2888,2011-03-29,1864.46
Vendor19,2891,2011-04-08,3560.26
Vendor19,2899,2011-04-16,1504.06
Vendor19,2903,2011-04-16,3993.65
Vendor19,2906,2011-04-24,1231.45
Vendor19,2912,2011-04-30,2715.25
Vendor19,2916,2011-05-01,2528.65
Vendor19,2922,2011-05-07,1406.61
Vendor19,2930,2011-05-08,1931.17
Vendor19,2935,2011-05-08,1266.32
Vendor19,2938,2011-05-18,1448.49
Vendor19,2944,2011-05-19,3245.62
Vendor19,2947,2011-05-29,8538.97
Vendor19,2955,2011-05-29,1999.19
Vendor19,2958,2011-05-29,1658.04
Vendor19,2963,2011-05-29,5796.18
Vendor19,2970,2011-06-06,2604.1
Vendor19,2971,2011-06-12,1586.32
Vendor19,2978,2011-06-12,1013.22
Vendor19,2982,2011-06-18,2967.24
Vendor19,2990,2011-06-24,3320.12
Vendor19,2992,2011-07-02,6223.06
Vendor19,3000,2011-07-08,4170.62
Vendor19,3002,2011-07-08,4737.88
Vendor19,3008,2011-07-18,1856.47
Vendor19,3012,2011-07-18,6206.96
Vendor19,3016,2011-07-28,8928.08
Vendor19,3021,2011-08-03,2539.88
Vendor19,3030,2011-08-03,2307.66
Vendor19,3033,2011-08-11,9827.41
Vendor19,3038,2011-08-11,7405.03
Vendor19,3041,2011-08-11,5919.99
Vendor19,3046,2011-08-17,4196.64
Vendor19,3053,2011-08-23,1118.73
Vendor19,3057,2011-08-23,5103.86
Vendor19,3061,2011-08-23,2094.13
Vendor19,3070,2011-08-24,1489.64
Vendor19,3071,2011-08-24,2266.84
Vendor19,3076,2011-08-24,3701.59
Vendor19,3082,2011-08-30,1344.59
Vendor19,3087,2011-09-05,8514.04
Vendor19,3093,2011-09-13,8643.22
Vendor19,3100,2011-09-13,1376.98
Vendor19,3102,2011-09-19,3827.45
Vendor19,3106,2011-09-29,9862.28
Vendor19,3112,2011-09-30,1398.55
Vendor19,3116,2011-09-30,4730.92
Vendor19,3122,2011-10-01,2899.02
Vendor19,3130,2011-10-01,5661.32
Vendor19,3133,2011-10-01,1093.32
Vendor19,3136,2011-10-07,1784.89
Vendor19,3142,2011-10-07,2096.05
Vendor19,3147,2011-10-13,7614.31
Vendor19,3152,2011-10-19,2230.94
Vendor19,3160,2011-10-29,6194.5
Vendor19,3161,2011-10-29,1340.42
Vendor19,3169,2011-10-30,3768.84
Vendor19,3175,2011-10-31,1966.28
Vendor19,3178,2011-11-01,1955.35
Vendor19,3181,2011-11-01,2300.84
Vendor19,3188,2011-11-02,4080.02
Vendor19,3191,2011-11-08,2688.23
Vendor19,3197,2011-11-09,1114.1
Vendor19,3202,2011-11-10,6069.73
Vendor19,3209,2011-11-16,2589.71
Vendor19,3214,2011-11-16,1305.18
Vendor19,3217,2011-11-17,2667.39
Vendor19,3222,2011-11-17,4341.91
Vendor19,3227,2011-11-17,3709.6
Vendor19,3232,2011-11-25,1421.15
Vendor19,3240,2011-11-26,4685.59
Vendor19,3242,2011-12-06,1731.76
Vendor19,3247,2011-12-16,3315.89
Vendor19,3255,2011-12-26,6242.34
Vendor19,3257,2012-01-05,4124.3
Vendor19,3261,2012-01-11,4471.54
Vendor19,3270,2012-01-12,1086.13
Vendor19,3272,2012-01-18,5431.93
Vendor19,3276,2012-01-18,4795.74
Vendor19,3285,2012-01-18,8948.48
Vendor19,3287,2012-01-26,4974.77
Vendor19,3292,2012-01-26,1089.44
Vendor19,3296,2012-01-26,1873.59
Vendor19,3304,2012-01-26,1014.27
Vendor19,3310,2012-01-26,5111.72
Vendor19,3313,2012-01-26,1681.37
Vendor19,3319,2012-01-27,3000.24
Vendor19,3323,2012-01-28,4941.77
Vendor19,3326,2012-01-28,1111.21
Vendor19,3335,2012-02-03,5592.12
Vendor19,3337,2012-02-13,4735.55
Vendor19,3344,2012-02-19,6094.78
Vendor19,3350,2012-02-20,3888.99
Vendor19,3355,2012-02-21,4655.44
Vendor19,3357,2012-02-22,6333.88
Vendor19,3364,2012-03-03,4236.56
Vendor19,3366,2012-03-04,1316.21
Vendor19,3372,2012-03-04,2249.78
Vendor19,3378,2012-03-10,4674.43
Vendor19,3382,2012-03-11,4489.18
Vendor19,3390,2012-03-17,1461.1
Vendor19,3392,2012-03-18,2792.62
Vendor19,3399,2012-03-28,3879.2
Vendor19,3405,2012-03-28,5460.69
Vendor19,3410,2012-03-28,2514.29
Vendor19,3414,2012-03-29,4744.41
Vendor19,3416,2012-03-29,3283.3
Vendor19,3423,2012-03-29,4873.06
Vendor19,3426,2012-03-30,3508.81
Vendor19,3433,2012-03-31,4951.88
Vendor19,3440,2012-04-01,7250.22
Vendor19,3445,2012-04-02,6822.47
Vendor19,3448,2012-04-12,5077.25
Vendor19,3453,2012-04-18,7778.8
Vendor19,3458,2012-04-28,5976.1
Vendor19,3461,2012-04-28,1256.99
Vendor19,3468,2012-05-08,1314.63
Vendor19,3472,2012-05-08,2221.09
Vendor19,3478,2012-05-14,1701.31
Vendor19,3483,2012-05-14,1844.62
Vendor19,3486,2012-05-14,2579.94
Vendor19,3494,2012-05-20,1645.27
Vendor19,3497,2012-05-21,1295.56
Vendor19,3503,2012-05-22,3277.94
Vendor19,3506,2012-05-22,1123.33
Vendor19,3512,2012-05-22,2941.92
Vendor19,3518,2012-05-23,1221.44
Vendor19,3523,2012-05-24,2175.86
Vendor19,3527,2012-05-25,9147.65
Vendor19,3535,2012-06-02,7796.95
Vendor19,3540,2012-06-02,2321.55
Vendor19,3544,2012-06-12,1960.35
Vendor19,3549,2012-06-22,4890.03
Vendor19,3553,2012-07-02,2215.3
Vendor19,3557,2012-07-02,1191.03
Vendor19,3562,2012-07-02,3375.71
Vendor19,3567,2012-07-02,4818.97
Vendor19,3575,2012-07-12,5956.8
Vendor19,3580,2012-07-13,7327.11
Vendor19,3583,2012-07-14,3305.47
Vendor19,3589,2012-07-24,1795.7
Vendor19,3592,2012-08-03,9458.1
Vendor19,3597,2012-08-09,1472.26
Vendor19,3603,2012-08-09,1368.85
Vendor19,3607,2012-08-09,4547.12
Vendor19,3611,2012-08-19,6881.06
Vendor19,3617,2012-08-25,3504.03
Vendor19,3621,2012-09-02,3222.93
Vendor19,3626,2012-09-08,1161.87
Vendor19,3635,2012-09-08,3199.12
Vendor19,3639,2012-09-09,2262.85
Vendor19,3645,2012-09-19,6525.29
Vendor19,3646,2012-09-27,2937.67
Vendor19,3651,2012-09-27,8478.77
Vendor19,3656,2012-10-07,8982.37
Vendor19,3664,2012-10-13,3122.8
Vendor19,3669,2012-10-14,1592.34
Vendor19,3672,2012-10-20,7006.04
Vendor19,3677,2012-10-20,1137.45
Vendor19,3682,2012-10-26,1441.66
Vendor19,3687,2012-10-27,1379.32
Vendor19,3695,2012-11-02,1872.33
Vendor19,3696,2012-11-02,3581.35
Vendor19,3704,2012-11-10,2200.34
Vendor19,3709,2012-11-10,9328.06
Vendor19,3712,2012-11-20,3796.86
Vendor19,3717,2012-11-20,1106.92
Vendor19,3723,2012-11-20,1991.06
Vendor19,3726,2012-11-20,1962.12
Vendor19,3733,2012-11-30,6374.01
Vendor19,3740,2012-12-10,6987.5
Vendor19,3744,2012-12-10,3285.97
Vendor19,3748,2012-12-10,1001.07
Vendor19,3752,2012-12-11,5205.09
Vendor19,3758,2012-12-12,1534.71
Vendor19,3764,2012-12-12,5134.64
Vendor19,3767,2012-12-12,2814.5
Vendor19,3772,2012-12-13,6858.35
Vendor19,3778,2012-12-19,3624.51
Vendor19,3782,2012-12-19,2333.37
Vendor19,3786,2012-12-19,2979.66
Vendor19,3794,2012-12-19,7327.47
Vendor19,3796,2012-12-20,1121.23
Vendor19,3803,2012-12-26,1062.79
Vendor19,3809,2013-01-01,2969.75
Vendor19,3813,2013-01-07,5217.68
Vendor19,3816,2013-01-07,8623.23
Vendor19,3825,2013-01-13,3816.07
Vendor19,3827,2013-01-21,3186.12
Vendor19,3834,2013-01-21,1633.2
Vendor19,3840,2013-01-22,3242.83
Vendor19,3842,2013-01-22,6043.84
Vendor19,3850,2013-01-22,3433.99
Vendor19,3855,2013-02-01,2277.78
Vendor19,3860,2013-02-02,3974.25
Vendor19,3862,2013-02-12,1658.46
Vendor19,3869,2013-02-20,8380.52
Vendor19,3873,2013-02-21,7745.86
Vendor19,3879,2013-03-01,8703.39
Vendor19,3883,2013-03-01,8248.16
Vendor19,3887,2013-03-02,4067.42
Vendor19,3891,2013-03-12,5807.79
Vendor19,3900,2013-03-18,4557.4
Vendor19,3901,2013-03-19,5249.79
Vendor19,3907,2013-03-29,7429.4
Vendor19,3914,2013-04-08,1137.52
Vendor19,3918,2013-04-18,9442.89
Vendor19,3921,2013-04-19,6080.87
Vendor19,3929,2013-04-20,2636.83
Vendor19,3931,2013-04-20,3098.1
Vendor19,3938,2013-04-28,2356.46
Vendor19,3943,2013-04-28,2731.06
Vendor19,3949,2013-05-04,4484.42
Vendor19,3954,2013-05-10,7465.71
Vendor19,3957,2013-05-16,1885.25
Vendor19,3961,2013-05-17,1092.99
Vendor19,3970,2013-05-17,6094.83
Vendor19,3974,2013-05-25,1785.13
Vendor19,3979,2013-05-31,1302.33
Vendor19,3981,2013-06-10,5430.04
Vendor19,3987,2013-06-16,3710.82
Vendor19,3995,2013-06-16,4534.9
Vendor19,3999,2013-06-17,2380.22
Vendor19,4003,2013-06-23,3509.51
Vendor19,4010,2013-06-29,7316.32
Vendor19,4011,2013-06-29,3647.49
Vendor19,4020,2013-07-05,2729.47
Vendor19,4022,2013-07-05,6837.63
Vendor19,4028,2013-07-13,4381.66
Vendor19,4031,2013-07-13,2060.94
Vendor19,4036,2013-07-13,1194.82
Vendor19,4041,2013-07-19,8107.42
Vendor19,4050,2013-07-20,6852.74
Vendor19,4052,2013-07-21,2580.79
Vendor19,4058,2013-07-27,2387.69
Vendor19,4062,2013-08-06,1481.97
Vendor19,4069,2013-08-14,2019.9
Vendor19,4075,2013-08-20,4418.51
Vendor19,4078,2013-08-26,4195.1
Vendor19,4082,2013-09-01,9554.77
Vendor19,4088,2013-09-02,9791.31
Vendor19,4091,2013-09-02,1582.15
Vendor19,4096,2013-09-10,1469.22
Vendor19,4104,2013-09-10,9297.36
Vendor21,9,2006-01-07,9077.87
Vendor21,15,2006-01-15,1214.69
Vendor21,20,2006-01-15,2632.45
Vendor21,22,2006-01-15,5267.88
Vendor21,30,2006-01-21,2642.71
Vendor21,31,2006-01-27,3095.29
Vendor21,36,2006-01-28,6167.58
Vendor21,44,2006-01-29,1313.27
Vendor21,47,2006-02-06,3022.74
Vendor21,52,2006-02-07,2635.32
Vendor21,60,2006-02-08,4934.72
Vendor21,64,2006-02-08,1857.91
Vendor21,66,2006-02-14,6310.71
Vendor21,72,2006-02-24,1407.75
Vendor21,76,2006-03-02,2063.11
Vendor21,83,2006-03-08,6332.03
Vendor21,89,2006-03-09,5678.84
Vendor21,91,2006-03-17,7225.91
Vendor21,98,2006-03-18,1610.7
Vendor21,105,2006-03-18,4255.44
Vendor21,107,2006-03-18,1427.91
Vendor21,115,2006-03-28,3529.1
Vendor21,120,2006-04-03,2315.33
Vendor21,122,2006-04-04,9704.82
Vendor21,126,2006-04-05,5033.84
Vendor21,133,2006-04-05,7092.76
Vendor21,136,2006-04-06,2698.55
Vendor21,141,2006-04-06,8088.75
Vendor21,147,2006-04-14,1754.36
Vendor21,151,2006-04-15,6542.91
Vendor21,157,2006-04-16,1859.37
Vendor21,161,2006-04-22,2451.92
Vendor21,166,2006-04-23,7113.95
Vendor21,173,2006-04-23,1353.87
Vendor21,179,2006-05-01,5847.69
Vendor21,183,2006-05-09,1604.96
Vendor21,188,2006-05-19,1025.92
Vendor21,195,2006-05-19,1625.42
Vendor21,199,2006-05-20,1635.27
Vendor21,201,2006-05-21,2993.17
Vendor21,210,2006-05-27,8839.97
Vendor21,213,2006-05-27,7196.81
Vendor21,217,2006-05-27,2803.39
Vendor21,221,2006-06-02,6408.05
Vendor21,228,2006-06-03,1105.07
Vendor21,235,2006-06-04,1990.77
Vendor21,237,2006-06-14,5841.41
Vendor21,245,2006-06-22,5158.6
Vendor21,246,2006-06-30,8237.25
Vendor21,254,2006-06-30,2641.02
Vendor21,258,2006-07-06,6201.49
Vendor21,265,2006-07-12,5223.55
Vendor21,267,2006-07-22,1040.23
Vendor21,271,2006-07-22,1340.74
Vendor21,279,2006-07-22,5140.06
Vendor21,282,2006-07-22,9042.25
Vendor21,289,2006-07-23,1553.09
Vendor21,295,2006-08-02,1021.65
Vendor21,300,2006-08-02,1004.94
Vendor21,301,2006-08-12,7832.71
Vendor21,308,2006-08-20,5143.18
Vendor21,312,2006-08-20,9325.66
Vendor21,317,2006-08-20,3685.2
Vendor21,323,2006-08-26,2790.05
Vendor21,330,2006-09-01,2676.34
Vendor21,331,2006-09-01,3491.97
Vendor21,337,2006-09-01,2573.92
Vendor21,343,2006-09-02,7860.71
Vendor21,347,2006-09-03,7431.39
Vendor21,351,2006-09-04,2827.41
Vendor21,357,2006-09-05,4152.66
Vendor21,364,2006-09-13,4728.82
Vendor21,367,2006-09-19,1883.26
Vendor21,371,2006-09-25,4318.92
Vendor21,379,2006-10-03,1020.7
Vendor21,382,2006-10-04,1828.87
Vendor21,388,2006-10-04,1363.22
Vendor21,393,2006-10-10,6101.79
Vendor21,398,2006-10-11,6333.38
Vendor21,404,2006-10-17,2130.0
Vendor21,408,2006-10-27,6612.28
Vendor21,412,2006-10-28,1927.37
Vendor21,419,2006-10-29,4974.4
Vendor21,425,2006-11-04,2699.14
Vendor21,429,2006-11-10,8699.84
Vendor21,435,2006-11-16,6736.26
Vendor21,436,2006-11-16,6299.63
Vendor21,445,2006-11-17,8500.32
Vendor21,449,2006-11-25,4502.83
Vendor21,453,2006-11-25,5654.77
Vendor21,458,2006-11-26,4058.72
Vendor21,464,2006-11-26,2694.48
Vendor21,467,2006-12-06,1075.36
Vendor21,472,2006-12-12,7182.45
Vendor21,476,2006-12-12,2693.22
Vendor21,481,2006-12-12,8803.55
Vendor21,486,2006-12-20,1934.97
Vendor21,491,2006-12-26,2690.97
Vendor21,496,2006-12-26,2761.59
Vendor21,501,2006-12-26,4172.62
Vendor21,507,2007-01-05,1373.05
Vendor21,514,2007-01-06,4853.63
Vendor21,518,2007-01-12,3779.37
Vendor21,523,2007-01-18,1600.19
Vendor21,530,2007-01-19,4646.82
Vendor21,531,2007-01-29,5536.24
Vendor21,536,2007-01-29,7543.44
Vendor21,541,2007-02-04,1864.42
Vendor21,549,2007-02-04,2738.24
Vendor21,555,2007-02-12,3104.34
Vendor21,557,2007-02-13,6787.27
Vendor21,562,2007-02-13,6410.85
Vendor21,570,2007-02-23,3628.9
Vendor21,575,2007-02-24,7829.95
Vendor21,580,2007-03-02,6162.73
Vendor21,581,2007-03-08,2586.3
Vendor21,586,2007-03-08,1075.37
Vendor21,591,2007-03-09,1287.07
Vendor21,596,2007-03-10,9802.1
Vendor21,605,2007-03-11,1370.91
Vendor21,606,2007-03-11,1147.34
Vendor21,613,2007-03-21,1229.76
Vendor21,617,2007-03-21,7823.06
Vendor21,624,2007-03-27,1025.76
Vendor21,630,2007-04-02,5512.2
Vendor21,634,2007-04-02,3841.88
Vendor21,636,2007-04-03,5365.45
Vendor21,641,2007-04-04,3810.27
Vendor21,649,2007-04-05,1034.46
Vendor21,655,2007-04-05,2547.31
Vendor21,658,2007-04-11,2252.22
Vendor21,662,2007-04-12,2679.61
Vendor21,666,2007-04-13,5151.55
Vendor21,674,2007-04-14,2966.05
Vendor21,680,2007-04-20,1963.48
Vendor21,682,2007-04-30,1949.1
Vendor21,689,2007-04-30,1736.91
Vendor21,695,2007-04-30,2628.19
Vendor21,697,2007-05-06,2305.97
Vendor21,704,2007-05-06,1278.77
Vendor21,710,2007-05-12,4148.3
Vendor21,714,2007-05-13,9847.02
Vendor21,718,2007-05-13,1034.34
Vendor21,722,2007-05-14,2835.76
Vendor21,730,2007-05-20,6358.07
Vendor21,734,2007-05-21,2686.79
Vendor21,740,2007-05-27,1415.82
Vendor21,745,2007-06-04,2773.65
Vendor21,750,2007-06-05,7071.7
Vendor21,753,2007-06-06,1849.41
Vendor21,759,2007-06-06,1972.92
Vendor21,763,2007-06-07,1165.99
Vendor21,767,2007-06-13,1276.06
Vendor21,771,2007-06-14,2962.21
Vendor21,777,2007-06-15,6234.67
Vendor21,781,2007-06-16,7375.15
Vendor21,787,2007-06-22,3448.13
Vendor21,791,2007-06-22,1281.75
Vendor21,799,2007-06-22,2475.27
Vendor21,801,2007-06-23,7332.02
Vendor21,808,2007-06-24,7099.74
Vendor21,814,2007-06-25,1794.23
Vendor21,818,2007-06-26,7229.57
Vendor21,821,2007-06-27,9974.12
Vendor21,827,2007-07-07,4198.28
Vendor21,834,2007-07-08,2733.17
Vendor21,838,2007-07-09,3635.35
Vendor21,843,2007-07-10,4728.65
Vendor21,847,2007-07-16,2675.04
Vendor21,854,2007-07-17,3033.13
Vendor21,857,2007-07-25,1731.16
Vendor21,863,2007-07-25,6718.84
Vendor21,867,2007-07-25,1946.19
Vendor21,871,2007-07-26,1303.48
Vendor21,879,2007-07-26,4176.63
Vendor21,883,2007-08-05,2149.99
Vendor21,888,2007-08-05,2374.93
Vendor21,894,2007-08-06,4477.26
Vendor21,897,2007-08-12,8701.81
Vendor21,903,2007-08-13,9449.74
Vendor21,906,2007-08-14,5624.99
Vendor21,912,2007-08-24,8246.61
Vendor21,918,2007-08-25,3347.06
Vendor21,921,2007-08-25,1982.5
Vendor21,926,2007-08-26,1283.79
Vendor21,932,2007-08-26,6328.04
Vendor21,937,2007-08-26,1321.18
Vendor21,941,2007-09-05,5764.65
Vendor21,946,2007-09-15,1234.0
Vendor21,953,2007-09-25,1723.05
Vendor21,958,2007-09-25,2916.14
Vendor21,965,2007-10-01,2427.25
Vendor21,968,2007-10-02,2077.56
Vendor21,971,2007-10-03,2594.04
Vendor21,978,2007-10-03,6581.15
Vendor21,983,2007-10-13,2734.52
Vendor21,986,2007-10-14,8078.4
Vendor21,995,2007-10-14,9255.37
Vendor21,1000,2007-10-15,4690.98
Vendor21,1001,2007-10-16,1316.17
Vendor21,1010,2007-10-17,3010.87
Vendor21,1011,2007-10-18,1993.16
Vendor21,1019,2007-10-18,9838.34
Vendor21,1021,2007-10-19,4951.55
Vendor21,1029,2007-10-25,1214.7
Vendor21,1035,2007-10-25,1263.58
Vendor21,1039,2007-10-31,1651.64
Vendor21,1044,2007-10-31,7527.82
Vendor21,1046,2007-11-06,8236.81
Vendor21,1054,2007-11-07,1038.86
Vendor21,1060,2007-11-07,2524.71
Vendor21,1064,2007-11-13,2099.24
Vendor21,1068,2007-11-13,2445.91
Vendor21,1075,2007-11-13,2375.47
Vendor21,1076,2007-11-19,8704.32
Vendor21,1084,2007-11-27,1274.83
Vendor21,1086,2007-12-03,5918.14
Vendor21,1094,2007-12-03,1089.51
Vendor21,1099,2007-12-04,1853.25
Vendor21,1101,2007-12-04,2245.21
Vendor21,1107,2007-12-05,1602.02
Vendor21,1113,2007-12-05,4861.41
Vendor21,1116,2007-12-05,2516.97
Vendor21,1123,2007-12-05,2943.55
Vendor21,1129,2007-12-11,6601.31
Vendor21,1133,2007-12-11,1262.75
Vendor21,1139,2007-12-11,7407.65
Vendor21,1142,2007-12-17,1710.96
Vendor21,1149,2007-12-17,7139.11
Vendor21,1152,2007-12-18,3334.2
Vendor21,1160,2007-12-19,1660.78
Vendor21,1165,2007-12-20,3989.53
Vendor21,1168,2007-12-21,1344.35
Vendor21,1171,2007-12-31,1617.16
Vendor21,1177,2008-01-01,1849.17
Vendor21,1184,2008-01-01,4016.1
Vendor21,1187,2008-01-07,5135.13
Vendor21,1193,2008-01-08,6134.12
Vendor21,1199,2008-01-08,4103.48
Vendor21,1204,2008-01-16,5624.48
Vendor21,1209,2008-01-24,2612.09
Vendor21,1213,2008-01-30,2458.11
Vendor21,1218,2008-02-05,1857.96
Vendor21,1223,2008-02-06,1180.41
Vendor21,1228,2008-02-06,6018.31
Vendor21,1231,2008-02-07,1135.17
Vendor21,1238,2008-02-07,1532.98
Vendor21,1241,2008-02-08,2475.73
Vendor21,1249,2008-02-18,3532.34
Vendor21,1253,2008-02-26,9490.82
Vendor21,1259,2008-03-05,2619.5
Vendor21,1261,2008-03-05,7379.49
Vendor21,1270,2008-03-13,7042.23
Vendor21,1274,2008-03-13,6033.21
Vendor21,1279,2008-03-19,4485.91
Vendor21,1282,2008-03-25,4965.04
Vendor21,1286,2008-03-25,1840.09
Vendor21,1293,2008-03-31,1272.88
Vendor21,1297,2008-04-06,1344.86
Vendor21,1305,2008-04-16,3832.13
Vendor21,1309,2008-04-16,3126.81
Vendor21,1311,2008-04-16,1745.2
Vendor21,1318,2008-04-24,1326.21
Vendor21,1324,2008-04-30,1681.27
Vendor21,1328,2008-05-10,5448.18
Vendor21,1334,2008-05-11,1862.19
Vendor21,1340,2008-05-11,3461.6
Vendor21,1341,2008-05-11,2305.93
Vendor21,1348,2008-05-19,3909.29
Vendor21,1355,2008-05-20,4248.01
Vendor21,1358,2008-05-20,3247.32
Vendor21,1364,2008-05-26,2668.98
Vendor21,1368,2008-06-01,2582.95
Vendor21,1373,2008-06-11,6515.86
Vendor21,1378,2008-06-11,4529.46
Vendor21,1382,2008-06-12,2440.26
Vendor21,1389,2008-06-18,1371.27
Vendor21,1394,2008-06-18,4120.24
Vendor21,1400,2008-06-18,5664.62
Vendor21,1401,2008-06-24,3734.42
Vendor21,1408,2008-07-04,2898.59
Vendor21,1411,2008-07-04,2114.44
Vendor21,1416,2008-07-12,9082.59
Vendor21,1424,2008-07-12,2917.68
Vendor21,1428,2008-07-13,2920.09
Vendor21,1432,2008-07-13,9872.63
Vendor21,1437,2008-07-13,3900.88
Vendor21,1443,2008-07-21,1126.1
Vendor21,1447,2008-07-22,1986.13
Vendor21,1455,2008-08-01,1738.38
Vendor21,1456,2008-08-01,1073.8
Vendor21,1463,2008-08-01,8937.58
Vendor21,1470,2008-08-09,3684.07
Vendor21,1471,2008-08-15,1761.13
Vendor21,1479,2008-08-16,1936.08
Vendor21,1483,2008-08-16,7923.51
Vendor21,1487,2008-08-16,1012.43
Vendor21,1491,2008-08-16,1953.57
Vendor21,1496,2008-08-22,5680.5
Vendor21,1505,2008-08-28,8073.45
Vendor21,1506,2008-08-28,6092.91
Vendor21,1515,2008-08-28,1845.14
Vendor21,1517,2008-09-05,5266.61
Vendor21,1523,2008-09-06,7953.19
Vendor21,1527,2008-09-06,8565.44
Vendor21,1532,2008-09-12,4419.47
Vendor21,1537,2008-09-13,3407.36
Vendor21,1544,2008-09-13,1149.0
Vendor21,1548,2008-09-23,1495.3
Vendor21,1551,2008-09-24,4861.39
Vendor21,1559,2008-09-24,2202.64
Vendor21,1563,2008-09-25,5404.91
Vendor21,1566,2008-09-25,1225.93
Vendor21,1571,2008-09-26,2617.37
Vendor21,1580,2008-09-26,5329.48
Vendor21,1583,2008-09-26,3227.03
Vendor21,1586,2008-09-26,1095.15
Vendor21,1593,2008-09-27,1941.46
Vendor21,1598,2008-10-05,2085.57
Vendor21,1602,2008-10-06,6642.38
Vendor21,1607,2008-10-16,2136.76
Vendor21,1612,2008-10-26,1454.85
Vendor21,1618,2008-10-26,5153.09
Vendor21,1624,2008-10-26,8086.56
Vendor21,1628,2008-11-01,2000.31
Vendor21,1633,2008-11-01,2802.32
Vendor21,1638,2008-11-07,2427.06
Vendor21,1642,2008-11-13,5430.35
Vendor21,1647,2008-11-19,9401.81
Vendor21,1655,2008-11-19,7445.62
Vendor21,1656,2008-11-19,4602.66
Vendor21,1665,2008-11-19,1203.01
Vendor21,1667,2008-11-20,4315.3
Vendor21,1672,2008-11-21,2772.63
Vendor21,1676,2008-11-21,5180.28
Vendor21,1685,2008-11-22,7508.4
Vendor21,1686,2008-11-23,1844.26
Vendor21,1693,2008-12-01,1604.73
Vendor21,1698,2008-12-09,6603.75
Vendor21,1703,2008-12-15,1913.53
Vendor21,1707,2008-12-16,1032.39
Vendor21,1714,2008-12-22,3230.79
Vendor21,1719,2008-12-30,8306.05
Vendor21,1725,2008-12-30,4652.15
Vendor21,1728,2008-12-31,1118.42
Vendor21,1735,2008-12-31,1272.92
Vendor21,1738,2009-01-01,1408.96
Vendor21,1741,2009-01-07,5551.95
Vendor21,1749,2009-01-07,5309.43
Vendor21,1755,2009-01-08,4494.84
Vendor21,1757,2009-01-09,5852.17
Vendor21,1763,2009-01-19,1320.16
Vendor21,1766,2009-01-20,3622.45
Vendor21,1772,2009-01-20,1312.56
Vendor21,1779,2009-01-21,2841.1
Vendor21,1781,2009-01-29,6494.04
Vendor21,1787,2009-01-30,2520.83
Vendor21,1795,2009-02-05,1924.23
Vendor21,1798,2009-02-11,1644.77
Vendor21,1802,2009-02-19,1750.44
Vendor21,1806,2009-02-20,5294.29
Vendor21,1815,2009-02-26,2254.88
Vendor21,1817,2009-02-27,5837.84
Vendor21,1822,2009-03-07,1789.9
Vendor21,1826,2009-03-07,1190.21
Vendor21,1832,2009-03-07,3464.48
Vendor21,1840,2009-03-13,2195.1
Vendor21,1842,2009-03-19,8472.81
Vendor21,1848,2009-03-19,1151.5
Vendor21,1851,2009-03-19,1801.22
Vendor21,1858,2009-03-25,1886.0
Vendor21,1865,2009-03-25,9244.59
Vendor21,1868,2009-03-31,3520.98
Vendor21,1875,2009-04-01,8054.64
Vendor21,1877,2009-04-07,1260.71
Vendor21,1885,2009-04-15,3882.75
Vendor21,1889,2009-04-16,7641.39
Vendor21,1893,2009-04-17,8555.59
Vendor21,1898,2009-04-27,5645.96
Vendor21,1902,2009-04-28,3964.27
Vendor21,1906,2009-04-28,2644.16
Vendor21,1912,2009-04-29,1083.81
Vendor21,1916,2009-04-29,3181.98
Vendor21,1925,2009-04-29,2643.45
Vendor21,1927,2009-04-29,1838.59
Vendor21,1931,2009-04-29,6789.22
Vendor21,1938,2009-04-29,1346.0
Vendor21,1944,2009-04-29,2311.39
Vendor21,1947,2009-04-30,6887.17
Vendor21,1952,2009-04-30,6749.43
Vendor21,1959,2009-05-06,1904.73
Vendor21,1964,2009-05-07,1509.33
Vendor21,1970,2009-05-15,8898.86
Vendor21,1975,2009-05-15,5244.72
Vendor21,1979,2009-05-15,2989.94
Vendor21,1981,2009-05-15,7165.96
Vendor21,1989,2009-05-25,9168.32
Vendor21,1995,2009-05-25,5435.69
Vendor21,1997,2009-05-31,1194.74
Vendor21,2001,2009-06-01,9799.49
Vendor21,2006,2009-06-09,2739.64
Vendor21,2014,2009-06-17,5730.45
Vendor21,2019,2009-06-18,1039.41
Vendor21,2024,2009-06-19,1678.31
Vendor21,2026,2009-06-25,2252.54
Vendor21,2032,2009-06-25,4995.63
Vendor21,2038,2009-06-25,1363.94
Vendor21,2043,2009-06-25,8902.42
Vendor21,2050,2009-07-01,2885.32
Vendor21,2051,2009-07-01,8672.07
Vendor21,2056,2009-07-02,1165.5
Vendor21,2061,2009-07-08,2596.62
Vendor21,2069,2009-07-14,1708.29
Vendor21,2074,2009-07-20,1448.6
Vendor21,2077,2009-07-20,1109.65
Vendor21,2083,2009-07-26,2128.64
Vendor21,2088,2009-07-27,1308.5
Vendor21,2091,2009-07-28,1079.33
Vendor21,2097,2009-07-29,3974.07
Vendor21,2101,2009-07-29,6736.33
Vendor21,2107,2009-07-29,1275.07
Vendor21,2111,2009-08-08,7338.03
Vendor21,2120,2009-08-09,1721.72
Vendor21,2122,2009-08-17,3007.31
Vendor21,2127,2009-08-27,1167.15
Vendor21,2132,2009-08-28,1049.31
Vendor21,2137,2009-08-28,6077.87
Vendor21,2143,2009-08-28,8222.97
Vendor21,2146,2009-09-03,2683.01
Vendor21,2152,2009-09-03,1184.32
Vendor21,2158,2009-09-03,1441.42
Vendor21,2164,2009-09-09,2484.63
Vendor21,2168,2009-09-09,7009.42
Vendor21,2175,2009-09-15,2545.01
Vendor21,2180,2009-09-16,1501.95
Vendor21,2181,2009-09-24,5553.84
Vendor21,2188,2009-09-24,2473.19
Vendor21,2192,2009-09-25,4620.83
Vendor21,2199,2009-09-25,2825.31
Vendor21,2201,2009-09-26,3364.24
Vendor21,2210,2009-09-27,4312.61
Vendor21,2212,2009-09-28,1216.5
Vendor21,2216,2009-09-29,2106.3
Vendor21,2224,2009-10-07,5088.29
Vendor21,2230,2009-10-13,8753.63
Vendor21,2234,2009-10-13,1371.1
Vendor21,2238,2009-10-13,7457.67
Vendor21,2242,2009-10-14,3701.84
Vendor21,2250,2009-10-20,9372.42
Vendor21,2251,2009-10-20,8320.43
Vendor21,2258,2009-10-30,5430.12
Vendor21,2261,2009-10-31,4520.52
Vendor21,2266,2009-11-06,2786.45
Vendor21,2272,2009-11-14,4346.44
Vendor21,2280,2009-11-15,8995.86
Vendor21,2284,2009-11-16,5138.36
Vendor21,2289,2009-11-17,1931.41
Vendor21,2293,2009-11-18,3183.76
Vendor21,2300,2009-11-24,1930.28
Vendor21,2302,2009-12-02,4240.05
Vendor21,2307,2009-12-10,2584.2
Vendor21,2312,2009-12-11,7974.65
Vendor21,2316,2009-12-11,5173.26
Vendor21,2322,2009-12-12,4174.47
Vendor21,2327,2009-12-13,1761.28
Vendor21,2331,2009-12-14,7069.84
Vendor21,2338,2009-12-15,7056.92
Vendor21,2343,2009-12-15,9616.49
Vendor21,2346,2009-12-21,1859.74
Vendor21,2354,2009-12-22,8169.68
Vendor21,2359,2009-12-22,5143.24
Vendor21,2364,2009-12-28,1156.05
Vendor21,2367,2010-01-03,8943.86
Vendor21,2374,2010-01-09,4042.98
Vendor21,2380,2010-01-19,1383.37
Vendor21,2385,2010-01-20,2674.85
Vendor21,2389,2010-01-20,1893.44
Vendor21,2394,2010-01-20,2507.82
Vendor21,2399,2010-01-30,3420.4
Vendor21,2404,2010-01-30,9627.89
Vendor21,2410,2010-01-30,1564.84
Vendor21,2412,2010-02-09,1122.35
Vendor21,2416,2010-02-10,1335.25
Vendor21,2424,2010-02-16,1004.43
Vendor21,2430,2010-02-17,1598.31
Vendor21,2432,2010-02-17,2225.61
Vendor21,2439,2010-02-17,5793.73
Vendor21,2442,2010-02-18,3351.24
Vendor21,2449,2010-02-26,2329.48
Vendor21,2452,2010-03-08,5985.5
Vendor21,2456,2010-03-18,1115.61
Vendor21,2463,2010-03-24,7834.13
Vendor21,2468,2010-04-03,1292.85
Vendor21,2473,2010-04-09,1094.45
Vendor21,2480,2010-04-09,2300.12
Vendor21,2482,2010-04-15,4314.95
Vendor21,2486,2010-04-23,1901.69
Vendor21,2493,2010-04-24,2020.12
Vendor21,2500,2010-04-25,1792.67
Vendor21,2505,2010-05-01,1131.19
Vendor21,2508,2010-05-07,3829.65
Vendor21,2514,2010-05-07,4845.8
Vendor21,2519,2010-05-13,5146.26
Vendor21,2522,2010-05-14,2062.24
Vendor21,2527,2010-05-20,3433.14
Vendor21,2535,2010-05-26,1955.67
Vendor21,2537,2010-06-03,5096.37
Vendor21,2543,2010-06-03,8419.89
Vendor21,2546,2010-06-09,5614.48
Vendor21,2554,2010-06-19,7881.75
Vendor21,2560,2010-06-25,7378.36
Vendor21,2561,2010-07-03,2193.67
Vendor21,2569,2010-07-03,2455.85
Vendor21,2572,2010-07-03,2181.69
Vendor21,2579,2010-07-03,1509.11
Vendor21,2582,2010-07-04,1239.84
Vendor21,2589,2010-07-05,2503.83
Vendor21,2592,2010-07-05,1259.58
Vendor21,2599,2010-07-06,5316.19
Vendor21,2601,2010-07-16,3045.59
Vendor21,2606,2010-07-26,1528.67
Vendor21,2612,2010-07-26,1172.84
Vendor21,2617,2010-08-01,1851.44
Vendor21,2621,2010-08-02,6760.02
Vendor21,2626,2010-08-08,1007.48
Vendor21,2634,2010-08-08,2900.91
Vendor21,2639,2010-08-16,1555.28
Vendor21,2641,2010-08-16,1056.58
Vendor21,2648,2010-08-17,2055.95
Vendor21,2651,2010-08-18,5423.81
Vendor21,2656,2010-08-24,3237.85
Vendor21,2661,2010-08-24,2252.16
Vendor21,2668,2010-08-30,3242.34
Vendor21,2674,2010-08-31,3700.75
Vendor21,2680,2010-09-08,4327.97
Vendor21,2683,2010-09-08,4341.33
Vendor21,2687,2010-09-09,9468.67
Vendor21,2692,2010-09-09,1660.92
Vendor21,2698,2010-09-10,1133.47
Vendor21,2705,2010-09-16,4952.13
Vendor21,2710,2010-09-17,7787.71
Vendor21,2712,2010-09-17,5491.35
Vendor21,2718,2010-09-18,2967.72
Vendor21,2725,2010-09-24,8378.9
Vendor21,2726,2010-09-30,5710.97
Vendor21,2735,2010-10-08,6545.03
Vendor21,2736,2010-10-09,9450.3
Vendor21,2744,2010-10-19,7151.96
Vendor21,2747,2010-10-19,1274.63
Vendor21,2751,2010-10-20,2198.86
Vendor21,2757,2010-10-26,2313.73
Vendor21,2762,2010-11-05,4859.39
Vendor21,2767,2010-11-06,4930.59
Vendor21,2773,2010-11-07,1665.23
Vendor21,2780,2010-11-17,1807.98
Vendor21,2782,2010-11-18,3704.1
Vendor21,2787,2010-11-19,1076.61
Vendor21,2791,2010-11-19,1280.78
Vendor21,2796,2010-11-25,2344.76
Vendor21,2804,2010-12-01,9045.03
Vendor21,2807,2010-12-07,8282.69
Vendor21,2812,2010-12-17,4324.9
Vendor21,2820,2010-12-17,6177.37
Vendor21,2824,2010-12-17,4824.7
Vendor21,2830,2010-12-17,8207.23
Vendor21,2832,2010-12-17,1621.6
Vendor21,2839,2010-12-18,1683.22
Vendor21,2844,2010-12-18,1440.78
Vendor21,2846,2010-12-19,8165.97
Vendor21,2854,2010-12-25,4876.85
Vendor21,2858,2011-01-02,6417.17
Vendor21,2862,2011-01-08,1785.44
Vendor21,2868,2011-01-14,7884.17
Vendor21,2871,2011-01-15,1525.97
Vendor21,2880,2011-01-21,9863.08
Vendor21,2882,2011-01-22,2124.44
Vendor21,2886,2011-01-28,2107.6
Vendor21,2895,2011-02-07,4132.88
Vendor21,2896,2011-02-07,4835.36
Vendor21,2905,2011-02-17,1020.55
Vendor21,2906,2011-02-27,4431.05
Vendor21,2912,2011-03-07,4829.53
Vendor21,2918,2011-03-07,5552.16
Vendor21,2921,2011-03-13,6385.69
Vendor21,2928,2011-03-13,4568.93
Vendor21,2931,2011-03-13,4754.25
Vendor21,2937,2011-03-21,1907.65
Vendor21,2942,2011-03-22,8822.45
Vendor21,2947,2011-03-28,1729.12
Vendor21,2952,2011-04-07,2431.43
Vendor21,2958,2011-04-07,9339.68
Vendor21,2962,2011-04-13,1123.57
Vendor21,2966,2011-04-19,5546.63
Vendor21,2974,2011-04-25,2362.04
Vendor21,2978,2011-05-01,1671.14
Vendor21,2982,2011-05-02,1927.14
Vendor21,2988,2011-05-03,1933.91
Vendor21,2992,2011-05-04,4532.95
Vendor21,2999,2011-05-12,2173.32
Vendor21,3001,2011-05-20,3621.01
Vendor21,3008,2011-05-21,5890.06
Vendor21,3013,2011-05-21,2002.92
Vendor21,3020,2011-05-22,1473.16
Vendor21,3024,2011-05-30,1070.97
Vendor21,3027,2011-05-30,3631.87
Vendor21,3033,2011-06-05,1606.9
Vendor21,3038,2011-06-06,1362.43
Vendor21,3045,2011-06-06,1087.24
Vendor21,3047,2011-06-12,1980.67
Vendor21,3052,2011-06-12,6814.57
Vendor21,3056,2011-06-13,2768.17
Vendor21,3065,2011-06-14,2618.84
Vendor21,3066,2011-06-20,2403.5
Vendor21,3071,2011-06-20,1364.97
Vendor21,3080,2011-06-20,4535.26
Vendor21,3082,2011-06-21,8084.62
Vendor21,3090,2011-06-22,5730.69
Vendor21,3092,2011-06-22,4417.39
Vendor21,3098,2011-06-23,2814.3
Vendor21,3103,2011-07-01,1220.63
Vendor21,3107,2011-07-02,3098.56
Vendor21,3114,2011-07-02,4064.8
Vendor21,3118,2011-07-08,6915.87
Vendor21,3125,2011-07-08,5616.84
Vendor21,3130,2011-07-08,4929.88
Vendor21,3132,2011-07-09,1551.99
Vendor21,3139,2011-07-19,1018.1
Vendor21,3141,2011-07-20,6371.71
Vendor21,3147,2011-07-20,1618.94
Vendor21,3153,2011-07-30,1915.15
Vendor21,3156,2011-08-05,1767.01
Vendor21,3162,2011-08-05,2977.73
Vendor21,3168,2011-08-05,5054.14
Vendor21,3172,2011-08-05,1981.55
Vendor21,3177,2011-08-06,2019.57
Vendor21,3182,2011-08-06,3245.57
Vendor21,3189,2011-08-12,2880.05
Vendor21,3193,2011-08-13,9678.82
Vendor21,3198,2011-08-14,2439.26
Vendor21,3201,2011-08-20,1216.11
Vendor21,3207,2011-08-26,1732.15
Vendor21,3213,2011-08-27,2211.22
Vendor21,3220,2011-09-02,8696.05
Vendor21,3225,2011-09-02,6310.15
Vendor21,3229,2011-09-03,1344.45
Vendor21,3232,2011-09-04,5721.3
Vendor21,3236,2011-09-12,6567.07
Vendor21,3241,2011-09-18,4329.61
Vendor21,3250,2011-09-19,1839.95
Vendor21,3252,2011-09-19,2443.8
Vendor21,3259,2011-09-19,5603.86
Vendor21,3262,2011-09-19,2940.6
Vendor21,3268,2011-09-20,3829.32
Vendor21,3272,2011-09-21,7599.7
Vendor21,3280,2011-09-21,1194.95
Vendor21,3284,2011-09-21,5951.49
Vendor21,3288,2011-09-29,7359.34
Vendor21,3294,2011-09-30,1357.52
Vendor21,3300,2011-10-01,1439.28
Vendor21,3304,2011-10-01,6089.26
Vendor21,3306,2011-10-11,7416.84
Vendor21,3314,2011-10-21,2814.33
Vendor21,3317,2011-10-21,8685.19
Vendor21,3324,2011-10-21,2632.85
Vendor21,3327,2011-10-22,2391.61
Vendor21,3331,2011-10-30,9472.25
Vendor21,3340,2011-10-30,1157.33
Vendor21,3344,2011-10-30,1783.86
Vendor21,3349,2011-10-30,1550.29
Vendor21,3355,2011-11-05,3192.77
Vendor21,3359,2011-11-05,1792.56
Vendor21,3361,2011-11-06,1363.54
Vendor21,3370,2011-11-06,8792.22
Vendor21,3374,2011-11-12,3239.21
Vendor21,3380,2011-11-13,2560.97
Vendor21,3382,2011-11-13,5077.62
Vendor21,3390,2011-11-23,2508.21
Vendor21,3391,2011-11-23,1830.55
Vendor21,3399,2011-11-24,1971.8
Vendor21,3404,2011-11-24,1772.5
Vendor21,3406,2011-11-25,8514.5
Vendor21,3415,2011-11-25,5803.93
Vendor21,3420,2011-12-05,7214.52
Vendor21,3423,2011-12-11,2079.75
Vendor21,3429,2011-12-17,3785.46
Vendor21,3433,2011-12-17,3444.0
Vendor21,3438,2011-12-18,6675.1
Vendor21,3442,2011-12-18,1115.3
Vendor21,3446,2011-12-18,7043.82
Vendor21,3455,2011-12-26,1998.7
Vendor21,3460,2011-12-26,5095.69
Vendor21,3464,2012-01-03,1974.55
Vendor21,3469,2012-01-03,5378.18
Vendor21,3471,2012-01-13,4421.13
Vendor21,3480,2012-01-13,8956.64
Vendor21,3482,2012-01-14,4879.0
Vendor21,3488,2012-01-15,5004.62
Vendor21,3491,2012-01-15,3524.44
Vendor21,3497,2012-01-23,2307.68
Vendor21,3503,2012-01-24,1931.41
Vendor21,3506,2012-01-24,8243.64
Vendor21,3514,2012-01-25,7500.41
Vendor21,3519,2012-02-04,1460.48
Vendor21,3525,2012-02-04,1603.55
Vendor21,3527,2012-02-04,1566.02
Vendor21,3532,2012-02-14,1563.1
Vendor21,3539,2012-02-20,6279.29
Vendor21,3541,2012-02-26,2834.48
Vendor21,3549,2012-03-03,1224.58
Vendor21,3553,2012-03-09,2486.41
Vendor21,3558,2012-03-10,3389.53
Vendor21,3565,2012-03-16,8218.47
Vendor21,3570,2012-03-24,4339.53
Vendor21,3573,2012-03-24,9809.66
Vendor21,3578,2012-03-25,3647.98
Vendor21,3583,2012-04-02,6946.01
Vendor21,3586,2012-04-08,3026.25
Vendor21,3595,2012-04-09,1557.4
Vendor21,3600,2012-04-09,5411.93
Vendor21,3602,2012-04-09,7899.71
Vendor21,3608,2012-04-09,1062.98
Vendor21,3614,2012-04-09,3611.26
Vendor21,3618,2012-04-10,4054.44
Vendor21,3623,2012-04-11,1467.34
Vendor21,3629,2012-04-12,4000.25
Vendor21,3633,2012-04-12,5678.68
Vendor21,3640,2012-04-12,6342.47
Vendor21,3644,2012-04-22,3707.9
Vendor21,3650,2012-04-22,4075.46
Vendor21,3653,2012-04-28,5373.08
Vendor21,3658,2012-04-29,7286.13
Vendor21,3661,2012-04-30,3395.06
Vendor21,3668,2012-05-06,4869.15
Vendor21,3671,2012-05-06,7891.09
Vendor21,3680,2012-05-07,1972.03
Vendor21,3681,2012-05-13,2802.26
Vendor21,3689,2012-05-14,8432.47
Vendor21,3695,2012-05-20,2366.95
Vendor21,3696,2012-05-26,2722.79
Vendor21,3701,2012-05-26,9469.83
Vendor21,3710,2012-05-27,1103.61
Vendor21,3714,2012-05-27,3649.97
Vendor21,3719,2012-05-28,1452.31
Vendor21,3724,2012-05-29,3620.85
Vendor21,3727,2012-05-30,5747.46
Vendor21,3731,2012-06-05,3929.14
Vendor21,3739,2012-06-05,4632.38
Vendor21,3743,2012-06-05,1380.6
Vendor21,3750,2012-06-05,1753.29
Vendor21,3751,2012-06-05,3226.94
Vendor21,3759,2012-06-06,1157.11
Vendor21,3763,2012-06-16,8055.32
Vendor21,3766,2012-06-22,3108.62
Vendor21,3774,2012-06-23,1186.64
Vendor21,3778,2012-06-24,3551.02
Vendor21,3783,2012-06-24,1703.13
Vendor21,3789,2012-06-25,3591.01
Vendor21,3791,2012-07-05,8922.27
Vendor21,3798,2012-07-05,7858.66
Vendor21,3802,2012-07-05,1158.6
Vendor21,3808,2012-07-06,1602.12
Vendor21,3814,2012-07-07,3304.14
Vendor21,3819,2012-07-07,9283.1
Vendor21,3822,2012-07-07,3147.92
Vendor21,3827,2012-07-13,5552.0
Vendor21,3832,2012-07-19,1737.38
Vendor21,3837,2012-07-20,1549.84
Vendor21,3843,2012-07-28,4644.27
Vendor21,3847,2012-08-03,1540.83
Vendor21,3853,2012-08-03,1750.71
Vendor21,3856,2012-08-11,2360.4
Vendor21,3862,2012-08-17,2556.8
Vendor21,3866,2012-08-23,7639.99
Vendor21,3872,2012-08-24,3448.74
Vendor21,3878,2012-08-25,6784.29
Vendor21,3884,2012-09-04,3178.0
Vendor21,3889,2012-09-10,2126.58
Vendor21,3892,2012-09-11,1909.02
Vendor21,3898,2012-09-17,5226.6
Vendor21,3902,2012-09-23,6283.64
Vendor21,3907,2012-09-23,1158.83
Vendor21,3914,2012-09-23,2735.21
Vendor21,3919,2012-09-23,1630.11
Vendor21,3923,2012-10-01,6900.24
Vendor21,3927,2012-10-09,1830.51
Vendor21,3935,2012-10-10,8385.42
Vendor21,3937,2012-10-11,4040.43
Vendor21,3942,2012-10-12,1676.09
Vendor21,3946,2012-10-12,2905.8
Vendor21,3953,2012-10-12,9065.73
Vendor21,3959,2012-10-12,2889.26
Vendor21,3962,2012-10-12,3346.18
Vendor21,3968,2012-10-12,6751.57
Vendor21,3971,2012-10-18,4805.97
Vendor21,3976,2012-10-24,5892.59
Vendor21,3983,2012-10-30,3523.51
Vendor21,3989,2012-10-31,1024.22
Vendor21,3994,2012-11-06,1114.93
Vendor21,3997,2012-11-16,1503.39
Vendor21,4005,2012-11-16,7242.72
Vendor21,4008,2012-11-26,1433.78
Vendor21,4012,2012-11-26,3777.59
Vendor21,4016,2012-12-02,2430.08
Vendor21,4024,2012-12-12,5371.1
Vendor21,4027,2012-12-12,1466.13
Vendor21,4035,2012-12-18,1843.31
Vendor21,4039,2012-12-19,2245.34
Vendor21,4042,2012-12-20,1633.98
Vendor21,4047,2012-12-21,5961.77
Vendor21,4051,2012-12-21,5859.18
Vendor21,4059,2012-12-31,5642.63
Vendor21,4062,2013-01-06,8830.24
Vendor21,4066,2013-01-16,4860.2
Vendor21,4074,2013-01-24,1605.67
Vendor21,4076,2013-02-03,1369.68
Vendor21,4083,2013-02-03,8103.98
Vendor21,4088,2013-02-04,3744.84
Vendor21,4091,2013-02-04,2283.07
Vendor21,4099,2013-02-05,4937.61
Vendor21,4105,2013-02-05,1880.52
Vendor21,4110,2013-02-11,6103.15
Vendor21,4112,2013-02-12,1854.41
Vendor21,4118,2013-02-13,1277.52
Vendor21,4124,2013-02-14,3439.29
Vendor21,4126,2013-02-20,3242.89
Vendor21,4133,2013-02-20,6024.78
Vendor21,4139,2013-03-02,3939.51
Vendor21,4141,2013-03-10,1679.0
Vendor21,4150,2013-03-20,6948.12
Vendor21,4154,2013-03-20,4590.41
Vendor21,4158,2013-03-21,5095.31
Vendor21,4161,2013-03-29,9372.92
Vendor21,4166,2013-04-06,1013.43
Vendor21,4175,2013-04-06,1169.16
Vendor21,4176,2013-04-14,1876.48
Vendor21,4181,2013-04-15,3401.46
Vendor21,4187,2013-04-25,8739.95
Vendor21,4193,2013-05-05,1593.38
Vendor21,4200,2013-05-06,1993.71
Vendor21,4202,2013-05-14,1956.22
Vendor21,4208,2013-05-20,4457.17
Vendor21,4214,2013-05-26,6685.28
Vendor21,4219,2013-05-26,1046.78
Vendor21,4223,2013-05-27,4521.59
Vendor21,4229,2013-05-28,5074.32
Vendor21,4234,2013-06-05,2113.71
Vendor21,4240,2013-06-11,1544.76
Vendor21,4245,2013-06-12,9678.26
Vendor21,4249,2013-06-12,5432.01
Vendor21,4251,2013-06-18,7889.01
Vendor21,4258,2013-06-24,5035.25
Vendor21,4263,2013-06-24,7101.8
Vendor21,4270,2013-06-25,7093.1
Vendor21,4274,2013-07-05,1306.51
Vendor21,4276,2013-07-06,1721.87
Vendor21,4284,2013-07-06,3876.27
Vendor21,4287,2013-07-07,1581.42
Vendor21,4291,2013-07-13,2348.39
Vendor21,4298,2013-07-19,3143.18
Vendor21,4305,2013-07-20,4537.22
Vendor21,4308,2013-07-20,6915.15
Vendor21,4312,2013-07-30,2009.81
Vendor21,4317,2013-08-07,7279.7
Vendor21,4323,2013-08-07,9121.7
Vendor21,4329,2013-08-07,3128.46
Vendor21,4335,2013-08-13,2158.32
Vendor21,4338,2013-08-14,3945.55
Vendor21,4343,2013-08-14,2696.59
Vendor21,4350,2013-08-14,6506.32
Vendor21,4355,2013-08-15,2494.63
Vendor21,4356,2013-08-21,4192.77
Vendor30,7,2006-01-01,1753.22
Vendor30,15,2006-01-07,1760.53
Vendor30,16,2006-01-07,4136.16
Vendor30,25,2006-01-13,1002.11
Vendor30,27,2006-01-14,3458.96
Vendor30,33,2006-01-20,4130.27
Vendor30,40,2006-01-20,8191.66
Vendor30,45,2006-01-26,5012.68
Vendor30,46,2006-01-27,5401.3
Vendor30,52,2006-01-28,2007.29
Vendor30,59,2006-01-29,2071.4
Vendor30,61,2006-01-30,3759.4
Vendor30,67,2006-01-30,7832.38
Vendor30,72,2006-01-30,3061.93
Vendor30,79,2006-01-31,9754.95
Vendor30,81,2006-01-31,6649.24
Vendor30,89,2006-02-08,8150.9
Vendor30,92,2006-02-09,2506.92
Vendor30,97,2006-02-10,1675.83
Vendor30,103,2006-02-16,5521.75
Vendor30,109,2006-02-22,4782.98
Vendor30,113,2006-03-02,1677.41
Vendor30,117,2006-03-02,2067.33
Vendor30,123,2006-03-10,4710.15
Vendor30,128,2006-03-16,1131.99
Vendor30,133,2006-03-16,1733.1
Vendor30,137,2006-03-22,1665.42
Vendor30,141,2006-03-28,1083.12
Vendor30,147,2006-04-03,1683.59
Vendor30,154,2006-04-13,1284.58
Vendor30,159,2006-04-19,7098.46
Vendor30,164,2006-04-20,1356.93
Vendor30,168,2006-04-28,1771.32
Vendor30,174,2006-05-04,1458.95
Vendor30,180,2006-05-04,2310.12
Vendor30,184,2006-05-05,3624.29
Vendor30,187,2006-05-06,2043.02
Vendor30,194,2006-05-07,3464.01
Vendor30,200,2006-05-07,1081.46
Vendor30,205,2006-05-07,4940.58
Vendor30,210,2006-05-07,2569.9
Vendor30,211,2006-05-15,1402.39
Vendor30,217,2006-05-15,9833.31
Vendor30,222,2006-05-15,4939.76
Vendor30,228,2006-05-21,4675.75
Vendor30,235,2006-05-31,4131.35
Vendor30,238,2006-06-06,1847.89
Vendor30,241,2006-06-06,3105.0
Vendor30,250,2006-06-06,2242.68
Vendor30,254,2006-06-07,4004.66
Vendor30,257,2006-06-07,2840.13
Vendor30,264,2006-06-08,4640.05
Vendor30,268,2006-06-14,5376.07
Vendor30,271,2006-06-14,1613.04
Vendor30,279,2006-06-22,3733.61
Vendor30,282,2006-06-23,5322.04
Vendor30,289,2006-06-23,4890.96
Vendor30,292,2006-06-24,3871.6
Vendor30,296,2006-06-30,1132.05
Vendor30,305,2006-07-08,3755.36
Vendor30,310,2006-07-08,7694.16
Vendor30,314,2006-07-08,5599.13
Vendor30,320,2006-07-09,1575.46
Vendor30,323,2006-07-19,2021.06
Vendor30,327,2006-07-29,1581.17
Vendor30,333,2006-08-04,6051.3
Vendor30,336,2006-08-12,1517.48
Vendor30,345,2006-08-12,4424.45
Vendor30,349,2006-08-13,7984.57
Vendor30,351,2006-08-13,9175.09
Vendor30,357,2006-08-14,1418.26
Vendor30,365,2006-08-14,3197.85
Vendor30,369,2006-08-15,1920.41
Vendor30,375,2006-08-16,3731.42
Vendor30,378,2006-08-22,6789.64
Vendor30,384,2006-08-28,1117.91
Vendor30,387,2006-08-28,1790.56
Vendor30,391,2006-08-28,3514.7
Vendor30,396,2006-09-05,1085.06
Vendor30,404,2006-09-05,5676.06
Vendor30,410,2006-09-06,9523.56
Vendor30,414,2006-09-12,4501.68
Vendor30,417,2006-09-20,1185.91
Vendor30,423,2006-09-20,4950.72
Vendor30,429,2006-09-26,1321.49
Vendor30,434,2006-10-04,4187.5
Vendor30,440,2006-10-05,2658.03
Vendor30,444,2006-10-06,3151.94
Vendor30,450,2006-10-07,1174.7
Vendor30,453,2006-10-17,3655.76
Vendor30,456,2006-10-23,1377.81
Vendor30,462,2006-11-02,1501.3
Vendor30,469,2006-11-08,9170.73
Vendor30,471,2006-11-09,2662.81
Vendor30,480,2006-11-19,1448.47
Vendor30,482,2006-11-20,1367.0
Vendor30,487,2006-11-20,2697.43
Vendor30,495,2006-11-26,2959.63
Vendor30,498,2006-11-26,3322.4
Vendor30,502,2006-12-02,7571.13
Vendor30,510,2006-12-03,1256.17
Vendor30,513,2006-12-04,3148.63
Vendor30,516,2006-12-14,9460.17
Vendor30,524,2006-12-15,3428.22
Vendor30,526,2006-12-21,1440.13
Vendor30,531,2006-12-27,4365.01
Vendor30,540,2006-12-28,5099.66
Vendor30,545,2007-01-03,7665.12
Vendor30,546,2007-01-03,3608.65
Vendor30,553,2007-01-04,3673.09
Vendor30,558,2007-01-04,3913.75
Vendor30,564,2007-01-10,1193.48
Vendor30,569,2007-01-11,5219.73
Vendor30,575,2007-01-11,9329.65
Vendor30,579,2007-01-19,3974.63
Vendor30,585,2007-01-27,3693.31
Vendor30,590,2007-01-27,1350.26
Vendor30,595,2007-02-06,5828.17
Vendor30,598,2007-02-12,7120.3
Vendor30,604,2007-02-18,8281.59
Vendor30,607,2007-02-24,6877.77
Vendor30,612,2007-03-06,2318.53
Vendor30,619,2007-03-07,9546.75
Vendor30,625,2007-03-07,1672.6
Vendor30,626,2007-03-13,8455.93
Vendor30,635,2007-03-23,2188.11
Vendor30,638,2007-03-29,1422.96
Vendor30,642,2007-04-04,2638.4
Vendor30,647,2007-04-04,9684.66
Vendor30,653,2007-04-14,1524.44
Vendor30,659,2007-04-14,8294.8
Vendor30,661,2007-04-14,3990.09
Vendor30,669,2007-04-14,7129.56
Vendor30,675,2007-04-20,1181.25
Vendor30,678,2007-04-21,1742.89
Vendor30,685,2007-04-22,1515.67
Vendor30,690,2007-04-23,7490.66
Vendor30,693,2007-04-23,1361.05
Vendor30,697,2007-04-29,6064.62
Vendor30,701,2007-05-05,2105.19
Vendor30,709,2007-05-11,4158.65
Vendor30,713,2007-05-17,4243.75
Vendor30,717,2007-05-18,1401.82
Vendor30,721,2007-05-18,5750.31
Vendor30,728,2007-05-28,1129.39
Vendor30,732,2007-05-29,3090.27
Vendor30,737,2007-06-04,6049.3
Vendor30,744,2007-06-10,4516.29
Vendor30,748,2007-06-20,4554.97
Vendor30,752,2007-06-26,1818.73
Vendor30,757,2007-06-26,9448.23
Vendor30,761,2007-06-26,1300.64
Vendor30,768,2007-06-27,8907.66
Vendor30,775,2007-07-05,6517.05
Vendor30,779,2007-07-05,3232.47
Vendor30,783,2007-07-05,4439.84
Vendor30,790,2007-07-11,4224.04
Vendor30,795,2007-07-11,5265.73
Vendor30,796,2007-07-21,7400.02
Vendor30,801,2007-07-27,2855.94
Vendor30,810,2007-07-28,2107.0
Vendor30,812,2007-07-28,6443.65
Vendor30,818,2007-07-29,5565.13
Vendor30,825,2007-07-29,1227.08
Vendor30,829,2007-07-30,4592.35
Vendor30,835,2007-07-30,1820.36
Vendor30,836,2007-07-30,1344.48
Vendor30,844,2007-08-07,3380.94
Vendor30,846,2007-08-08,8294.25
Vendor30,854,2007-08-18,1366.78
Vendor30,856,2007-08-18,1518.31
Vendor30,861,2007-08-24,6747.66
Vendor30,870,2007-08-25,4441.65
Vendor30,875,2007-09-04,3089.23
Vendor30,878,2007-09-10,1180.92
Vendor30,881,2007-09-18,1469.83
Vendor30,890,2007-09-18,3328.37
Vendor30,891,2007-09-18,9097.12
Vendor30,900,2007-09-24,1390.72
Vendor30,901,2007-09-25,4384.82
Vendor30,910,2007-09-25,2928.09
Vendor30,913,2007-09-26,1162.63
Vendor30,918,2007-09-27,9497.37
Vendor30,922,2007-09-28,1992.2
Vendor30,928,2007-10-04,1828.74
Vendor30,932,2007-10-04,1076.12
Vendor30,938,2007-10-04,3103.46
Vendor30,945,2007-10-10,1284.9
Vendor30,946,2007-10-20,5128.96
Vendor30,953,2007-10-20,1445.33
Vendor30,957,2007-10-26,1059.01
Vendor30,964,2007-10-26,6897.11
Vendor30,968,2007-10-26,1593.01
Vendor30,974,2007-11-05,8514.64
Vendor30,980,2007-11-11,9283.73
Vendor30,982,2007-11-17,3780.21
Vendor30,990,2007-11-17,3355.02
Vendor30,992,2007-11-17,1544.56
Vendor30,999,2007-11-25,1836.1
Vendor30,1005,2007-12-03,1914.31
Vendor30,1010,2007-12-13,2173.69
Vendor30,1015,2007-12-14,2957.77
Vendor30,1016,2007-12-14,9267.07
Vendor30,1021,2007-12-24,2385.43
Vendor30,1026,2007-12-25,6219.3
Vendor30,1034,2008-01-04,8336.88
Vendor30,1040,2008-01-12,6482.28
Vendor30,1043,2008-01-12,2463.63
Vendor30,1048,2008-01-20,3455.01
Vendor30,1053,2008-01-20,4470.85
Vendor30,1060,2008-01-26,3267.78
Vendor30,1061,2008-01-27,5196.47
Vendor30,1066,2008-01-28,6920.68
Vendor30,1073,2008-01-29,1180.34
Vendor30,1077,2008-02-04,9965.21
Vendor30,1082,2008-02-05,7605.88
Vendor30,1087,2008-02-13,3289.12
Vendor30,1092,2008-02-21,5688.96
Vendor30,1099,2008-02-27,4299.13
Vendor30,1101,2008-02-27,6322.12
Vendor30,1109,2008-03-06,4907.3
Vendor30,1114,2008-03-06,3067.36
Vendor30,1118,2008-03-07,2556.99
Vendor30,1121,2008-03-15,5249.99
Vendor30,1126,2008-03-16,3715.83
Vendor30,1131,2008-03-22,1890.7
Vendor30,1138,2008-03-23,3552.03
Vendor30,1143,2008-03-23,2215.32
Vendor30,1147,2008-03-23,2047.93
Vendor30,1155,2008-03-23,2230.11
Vendor30,1159,2008-03-29,1105.1
Vendor30,1163,2008-04-08,1097.82
Vendor30,1166,2008-04-09,2157.06
Vendor30,1173,2008-04-09,4450.59
Vendor30,1179,2008-04-15,1218.58
Vendor30,1181,2008-04-16,1269.85
Vendor30,1188,2008-04-17,9204.02
Vendor30,1194,2008-04-17,3595.14
Vendor30,1198,2008-04-17,2690.91
Vendor30,1202,2008-04-17,1023.49
Vendor30,1208,2008-04-25,1093.02
Vendor30,1215,2008-04-25,1612.23
Vendor30,1216,2008-04-26,3165.95
Vendor30,1221,2008-04-26,1623.97
Vendor30,1229,2008-05-06,1123.94
Vendor30,1231,2008-05-06,8795.71
Vendor30,1240,2008-05-07,9105.45
Vendor30,1245,2008-05-15,1372.11
Vendor30,1248,2008-05-25,2493.82
Vendor30,1252,2008-05-26,3608.54
Vendor30,1256,2008-05-27,1538.19
Vendor30,1261,2008-05-27,4212.45
Vendor30,1269,2008-06-02,3210.7
Vendor30,1273,2008-06-02,6182.63
Vendor30,1278,2008-06-03,5249.37
Vendor30,1281,2008-06-13,5433.43
Vendor30,1290,2008-06-14,4875.35
Vendor30,1291,2008-06-14,4661.57
Vendor30,1298,2008-06-14,6981.5
Vendor30,1302,2008-06-14,5647.42
Vendor30,1306,2008-06-14,2844.91
Vendor30,1314,2008-06-14,3528.95
Vendor30,1317,2008-06-15,1726.52
Vendor30,1321,2008-06-15,8740.13
Vendor30,1330,2008-06-15,2525.72
Vendor30,1335,2008-06-21,4800.75
Vendor30,1336,2008-06-29,5281.26
Vendor30,1342,2008-06-29,1034.24
Vendor30,1346,2008-06-29,4761.15
Vendor30,1351,2008-06-30,3414.52
Vendor30,1356,2008-07-01,1217.78
Vendor30,1363,2008-07-02,3759.07
Vendor30,1366,2008-07-08,1608.63
Vendor30,1371,2008-07-09,6026.23
Vendor30,1380,2008-07-17,8923.4
Vendor30,1385,2008-07-23,2032.48
Vendor30,1389,2008-07-29,5222.46
Vendor30,1394,2008-08-06,2150.09
Vendor30,1400,2008-08-07,2244.68
Vendor30,1402,2008-08-17,7305.09
Vendor30,1407,2008-08-18,2736.8
Vendor30,1414,2008-08-24,1176.28
Vendor30,1417,2008-08-25,5630.48
Vendor30,1425,2008-08-31,3003.93
Vendor30,1429,2008-09-01,2038.97
Vendor30,1435,2008-09-07,1648.57
Vendor30,1440,2008-09-07,2029.07
Vendor30,1443,2008-09-07,1732.09
Vendor30,1450,2008-09-08,6337.53
Vendor30,1452,2008-09-08,6296.74
Vendor30,1457,2008-09-16,6243.42
Vendor30,1463,2008-09-17,5443.44
Vendor30,1467,2008-09-17,1860.2
Vendor30,1475,2008-09-25,1027.52
Vendor30,1478,2008-09-26,1617.48
Vendor30,1481,2008-10-02,3384.54
Vendor30,1489,2008-10-10,4249.94
Vendor30,1491,2008-10-11,4705.09
Vendor30,1500,2008-10-11,1841.74
Vendor30,1502,2008-10-17,1044.47
Vendor30,1506,2008-10-23,3452.8
Vendor30,1512,2008-10-24,1282.87
Vendor30,1520,2008-10-24,5814.82
Vendor30,1524,2008-10-25,3546.19
Vendor30,1527,2008-10-26,3724.31
Vendor30,1535,2008-10-27,3660.48
Vendor30,1537,2008-11-04,1009.94
Vendor30,1541,2008-11-05,1047.99
Vendor30,1547,2008-11-11,1337.36
Vendor30,1551,2008-11-11,2799.42
Vendor30,1556,2008-11-12,1456.93
Vendor30,1561,2008-11-13,6251.31
Vendor30,1568,2008-11-14,1772.62
Vendor30,1573,2008-11-20,6710.49
Vendor30,1576,2008-11-20,3910.99
Vendor30,1583,2008-11-28,1178.0
Vendor30,1586,2008-11-29,4663.26
Vendor30,1592,2008-11-29,8675.88
Vendor30,1598,2008-11-29,6607.69
Vendor30,1602,2008-12-07,5280.0
Vendor30,1607,2008-12-13,1619.32
Vendor30,1612,2008-12-21,4127.24
Vendor30,1619,2008-12-31,4828.41
Vendor30,1624,2009-01-01,2635.01
Vendor30,1629,2009-01-07,1930.68
Vendor30,1632,2009-01-17,4570.23
Vendor30,1640,2009-01-23,1479.13
Vendor30,1644,2009-01-23,1576.94
Vendor30,1650,2009-01-24,8139.4
Vendor30,1655,2009-02-01,1401.78
Vendor30,1658,2009-02-01,5634.63
Vendor30,1665,2009-02-07,1092.85
Vendor30,1669,2009-02-13,2959.51
Vendor30,1674,2009-02-14,1890.51
Vendor30,1676,2009-02-14,3527.2
Vendor30,1684,2009-02-14,1620.22
Vendor30,1687,2009-02-15,2556.27
Vendor30,1692,2009-02-21,8971.69
Vendor30,1696,2009-02-21,1088.44
Vendor30,1701,2009-02-27,1412.03
Vendor30,1710,2009-02-28,1375.72
Vendor30,1712,2009-02-28,1763.92
Vendor30,1717,2009-03-01,4461.24
Vendor30,1724,2009-03-07,1393.16
Vendor30,1726,2009-03-08,8943.84
Vendor30,1732,2009-03-16,2884.95
Vendor30,1740,2009-03-16,1155.59
Vendor30,1744,2009-03-22,1421.73
Vendor30,1748,2009-03-23,7617.85
Vendor30,1753,2009-03-24,3264.21
Vendor30,1759,2009-03-24,4101.41
Vendor30,1765,2009-03-24,1589.17
Vendor30,1768,2009-03-24,4112.76
Vendor30,1771,2009-03-25,3899.26
Vendor30,1780,2009-03-31,1955.3
Vendor30,1785,2009-04-10,4197.52
Vendor30,1786,2009-04-18,3795.42
Vendor30,1794,2009-04-18,1293.95
Vendor30,1796,2009-04-24,9092.19
Vendor30,1801,2009-04-25,2946.64
Vendor30,1810,2009-04-26,9612.63
Vendor30,1813,2009-05-02,1287.49
Vendor30,1818,2009-05-12,7891.94
Vendor30,1825,2009-05-22,2630.71
Vendor30,1829,2009-05-22,2995.35
Vendor30,1833,2009-05-28,2671.7
Vendor30,1840,2009-06-03,2885.81
Vendor30,1842,2009-06-03,6727.71
Vendor30,1849,2009-06-03,2212.57
Vendor30,1855,2009-06-04,2793.66
Vendor30,1856,2009-06-05,4134.5
Vendor30,1865,2009-06-15,3087.47
Vendor30,1867,2009-06-15,1512.31
Vendor30,1873,2009-06-21,4667.94
Vendor30,1879,2009-07-01,2444.35
Vendor30,1884,2009-07-02,1104.9
Vendor30,1888,2009-07-03,1433.68
Vendor30,1892,2009-07-09,1384.07
Vendor30,1900,2009-07-09,6770.1
Vendor30,1903,2009-07-09,5419.77
Vendor30,1910,2009-07-09,4514.58
Vendor30,1914,2009-07-17,2101.55
Vendor30,1919,2009-07-18,9144.07
Vendor30,1921,2009-07-26,3350.14
Vendor30,1929,2009-08-01,1627.11
Vendor30,1934,2009-08-11,4440.56
Vendor30,1940,2009-08-11,2609.93
Vendor30,1944,2009-08-17,1613.37
Vendor30,1950,2009-08-17,1377.07
Vendor30,1955,2009-08-23,2569.25
Vendor30,1956,2009-08-23,2913.29
Vendor30,1965,2009-08-29,5643.34
Vendor30,1968,2009-08-30,5482.1
Vendor30,1974,2009-09-05,4320.64
Vendor30,1976,2009-09-06,1430.86
Vendor30,1981,2009-09-07,2112.63
Vendor30,1987,2009-09-13,9743.46
Vendor30,1994,2009-09-14,7387.8
Vendor30,2000,2009-09-24,1231.05
Vendor30,2004,2009-10-02,3196.23
Vendor30,2010,2009-10-02,2057.38
Vendor30,2011,2009-10-08,4798.81
Vendor30,2017,2009-10-08,1563.37
Vendor30,2022,2009-10-18,9941.24
Vendor30,2028,2009-10-18,1475.14
Vendor30,2035,2009-10-26,2459.1
Vendor30,2038,2009-10-26,2740.67
Vendor30,2042,2009-11-01,2637.16
Vendor30,2050,2009-11-02,8649.66
Vendor30,2055,2009-11-03,2074.83
Vendor30,2059,2009-11-04,4846.94
Vendor30,2064,2009-11-10,6443.69
Vendor30,2069,2009-11-16,2583.14
Vendor30,2075,2009-11-16,1701.27
Vendor30,2076,2009-11-16,1591.32
Vendor30,2084,2009-11-16,9346.58
Vendor30,2090,2009-11-17,1133.1
Vendor30,2092,2009-11-17,4617.47
Vendor30,2099,2009-11-17,1230.84
Vendor30,2104,2009-11-17,6612.13
Vendor30,2106,2009-11-17,4696.51
Vendor30,2114,2009-11-18,1870.31
Vendor30,2117,2009-11-18,4247.04
Vendor30,2124,2009-11-28,9911.74
Vendor30,2128,2009-11-29,1112.62
Vendor30,2135,2009-11-30,3778.31
Vendor30,2138,2009-12-01,4067.97
Vendor30,2144,2009-12-02,1318.44
Vendor30,2147,2009-12-02,6979.75
Vendor30,2154,2009-12-10,3565.75
Vendor30,2158,2009-12-10,9933.92
Vendor30,2163,2009-12-16,2317.49
Vendor30,2168,2009-12-22,2127.89
Vendor30,2172,2009-12-23,6413.99
Vendor30,2176,2009-12-23,9420.87
Vendor30,2181,2010-01-02,1882.21
Vendor30,2189,2010-01-02,1755.94
Vendor30,2195,2010-01-03,1347.65
Vendor30,2197,2010-01-11,2779.29
Vendor30,2204,2010-01-17,1044.83
Vendor30,2209,2010-01-18,2706.77
Vendor30,2212,2010-01-18,3651.3
Vendor30,2217,2010-01-18,5091.91
Vendor30,2224,2010-01-19,8357.93
Vendor30,2226,2010-01-25,3317.84
Vendor30,2233,2010-01-31,2241.0
Vendor30,2238,2010-02-01,5205.0
Vendor30,2243,2010-02-09,4031.89
Vendor30,2250,2010-02-10,7704.49
Vendor30,2253,2010-02-10,7856.97
Vendor30,2256,2010-02-11,4540.66
Vendor30,2263,2010-02-21,2375.62
Vendor30,2270,2010-02-27,5402.08
Vendor30,2273,2010-03-09,5459.11
Vendor30,2276,2010-03-10,4047.68
Vendor30,2285,2010-03-18,1854.53
Vendor30,2289,2010-03-19,8368.13
Vendor30,2294,2010-03-20,2339.74
Vendor30,2298,2010-03-20,9117.85
Vendor30,2304,2010-03-21,1846.86
Vendor30,2310,2010-03-27,1122.95
Vendor30,2315,2010-03-28,3043.55
Vendor30,2316,2010-03-28,1187.34
Vendor30,2325,2010-03-29,1027.51
Vendor30,2329,2010-03-30,4768.16
Vendor30,2332,2010-03-30,2438.12
Vendor30,2339,2010-04-05,2321.97
Vendor30,2345,2010-04-05,4470.78
Vendor30,2347,2010-04-11,2760.2
Vendor30,2353,2010-04-21,1275.14
Vendor30,2356,2010-04-22,9825.97
Vendor30,2362,2010-04-22,6704.79
Vendor30,2366,2010-04-22,2338.41
Vendor30,2375,2010-05-02,1372.47
Vendor30,2377,2010-05-03,1607.17
Vendor30,2383,2010-05-04,1296.47
Vendor30,2386,2010-05-10,3262.88
Vendor30,2393,2010-05-11,2156.51
Vendor30,2396,2010-05-11,3446.59
Vendor30,2404,2010-05-12,4104.44
Vendor30,2410,2010-05-18,1434.72
Vendor30,2412,2010-05-18,1994.01
Vendor30,2416,2010-05-19,9548.58
Vendor30,2422,2010-05-19,6063.07
Vendor30,2429,2010-05-20,1591.67
Vendor30,2432,2010-05-26,3721.66
Vendor30,2439,2010-05-27,1010.07
Vendor30,2443,2010-05-28,2786.65
Vendor30,2450,2010-05-29,2100.02
Vendor30,2452,2010-06-08,8919.14
Vendor30,2458,2010-06-18,7568.54
Vendor30,2462,2010-06-24,2043.38
Vendor30,2467,2010-06-25,6394.12
Vendor30,2472,2010-07-01,9118.69
Vendor30,2480,2010-07-02,8501.44
Vendor30,2481,2010-07-12,1795.17
Vendor30,2486,2010-07-20,4177.17
Vendor30,2495,2010-07-20,3400.76
Vendor30,2497,2010-07-20,4979.81
Vendor30,2504,2010-07-21,9332.28
Vendor30,2507,2010-07-29,1551.5
Vendor30,2514,2010-08-04,9351.16
Vendor30,2517,2010-08-10,5782.28
Vendor30,2525,2010-08-10,5082.06
Vendor30,2530,2010-08-11,2688.38
Vendor30,2535,2010-08-12,2481.4
Vendor30,2540,2010-08-13,3039.48
Vendor30,2543,2010-08-14,7095.79
Vendor30,2549,2010-08-15,6603.98
Vendor30,2552,2010-08-16,1347.27
Vendor30,2558,2010-08-26,2108.27
Vendor30,2564,2010-09-03,2339.66
Vendor30,2567,2010-09-11,3340.98
Vendor30,2574,2010-09-11,9971.86
Vendor30,2578,2010-09-19,3077.98
Vendor30,2585,2010-09-25,2301.21
Vendor30,2587,2010-09-25,4488.87
Vendor30,2592,2010-09-26,2498.36
Vendor30,2597,2010-09-27,7472.32
Vendor30,2603,2010-09-28,2934.1
Vendor30,2609,2010-10-06,4345.75
Vendor30,2612,2010-10-06,5522.15
Vendor30,2618,2010-10-07,1069.74
Vendor30,2623,2010-10-13,6538.99
Vendor30,2627,2010-10-21,4510.9
Vendor30,2634,2010-10-27,1853.74
Vendor30,2638,2010-11-04,1180.53
Vendor30,2642,2010-11-10,1043.79
Vendor30,2646,2010-11-10,2668.57
Vendor30,2653,2010-11-16,9896.08
Vendor30,2656,2010-11-16,2496.74
Vendor30,2663,2010-11-22,1007.18
Vendor30,2666,2010-12-02,1276.91
Vendor30,2672,2010-12-03,2048.2
Vendor30,2680,2010-12-13,6882.32
Vendor30,2681,2010-12-13,1537.48
Vendor30,2690,2010-12-14,1410.35
Vendor30,2693,2010-12-22,6599.82
Vendor30,2698,2010-12-28,2443.08
Vendor30,2701,2011-01-05,1028.25
Vendor30,2709,2011-01-05,1872.31
Vendor30,2715,2011-01-11,1205.44
Vendor30,2719,2011-01-11,9587.6
Vendor30,2725,2011-01-19,8854.51
Vendor30,2727,2011-01-19,3381.85
Vendor30,2733,2011-01-20,1335.86
Vendor30,2739,2011-01-20,2229.97
Vendor30,2741,2011-01-20,5703.6
Vendor30,2749,2011-01-21,6766.55
Vendor30,2755,2011-01-21,3861.39
Vendor30,2757,2011-01-21,1943.96
Vendor30,2761,2011-01-22,2429.67
Vendor30,2767,2011-01-28,3742.52
Vendor30,2774,2011-02-05,2029.68
Vendor30,2776,2011-02-06,6791.0
Vendor30,2781,2011-02-16,1967.61
Vendor30,2788,2011-02-17,1125.01
Vendor30,2792,2011-02-18,1798.43
Vendor30,2796,2011-02-18,5227.38
Vendor30,2804,2011-02-18,3427.61
Vendor30,2808,2011-02-26,9253.59
Vendor30,2814,2011-02-26,1229.52
Vendor30,2819,2011-03-04,3206.06
Vendor30,2825,2011-03-10,3249.22
Vendor30,2827,2011-03-10,1092.13
Vendor30,2835,2011-03-20,5035.98
Vendor30,2836,2011-03-21,2763.73
Vendor30,2844,2011-03-27,7781.65
Vendor30,2846,2011-03-28,8016.07
Vendor30,2852,2011-03-29,3935.96
Vendor30,2859,2011-03-29,3373.11
Vendor30,2862,2011-03-30,1933.52
Vendor30,2869,2011-03-31,3220.72
Vendor30,2873,2011-04-10,1160.03
Vendor30,2877,2011-04-18,8937.51
Vendor30,2883,2011-04-18,2538.69
Vendor30,2888,2011-04-26,1993.2
Vendor30,2893,2011-04-27,2445.14
Vendor30,2896,2011-04-27,7080.71
Vendor30,2905,2011-04-28,1095.04
Vendor30,2910,2011-04-28,1846.16
Vendor30,2913,2011-05-04,2718.15
Vendor30,2918,2011-05-05,1238.0
Vendor30,2924,2011-05-15,1357.41
Vendor30,2930,2011-05-25,5705.5
Vendor30,2932,2011-06-02,2144.91
Vendor30,2940,2011-06-02,4498.37
Vendor30,2945,2011-06-02,1216.43
Vendor30,2949,2011-06-03,2672.18
Vendor30,2951,2011-06-09,2489.96
Vendor30,2960,2011-06-10,8384.85
Vendor30,2964,2011-06-16,3259.83
Vendor30,2968,2011-06-17,8414.29
Vendor30,2971,2011-06-27,5733.47
Vendor30,2977,2011-06-27,1464.84
Vendor30,2981,2011-07-03,6401.71
Vendor30,2987,2011-07-04,1838.73
Vendor30,2995,2011-07-12,4935.13
Vendor30,2996,2011-07-13,9250.18
Vendor30,3004,2011-07-19,2256.32
Vendor30,3006,2011-07-25,1654.88
Vendor30,3014,2011-07-25,1884.39
Vendor30,3016,2011-07-26,5617.64
Vendor30,3024,2011-07-27,3473.42
Vendor30,3029,2011-08-04,2945.37
Vendor30,3033,2011-08-10,8731.74
Vendor30,3038,2011-08-11,8314.02
Vendor30,3043,2011-08-12,1029.76
Vendor30,3049,2011-08-18,3001.09
Vendor30,3052,2011-08-19,1438.81
Vendor30,3058,2011-08-25,3368.02
Vendor30,3061,2011-08-25,6096.2
Vendor30,3066,2011-08-25,2713.86
Vendor30,3075,2011-08-25,1499.02
Vendor30,3080,2011-08-31,6361.42
Vendor30,3084,2011-08-31,3513.7
Vendor30,3090,2011-08-31,1528.93
Vendor30,3093,2011-08-31,3589.17
Vendor30,3097,2011-09-08,1073.65
Vendor30,3104,2011-09-09,2513.54
Vendor30,3107,2011-09-10,2819.8
Vendor30,3114,2011-09-18,1454.62
Vendor30,3117,2011-09-19,3241.54
Vendor30,3122,2011-09-25,1406.27
Vendor30,3129,2011-10-03,1943.7
Vendor30,3133,2011-10-03,8224.01
Vendor30,3137,2011-10-09,1160.64
Vendor30,3141,2011-10-10,1672.59
Vendor30,3147,2011-10-11,2902.21
Vendor30,3155,2011-10-11,6593.32
Vendor30,3156,2011-10-12,7466.44
Vendor30,3162,2011-10-18,2172.76
Vendor30,3167,2011-10-19,3713.22
Vendor30,3172,2011-10-25,1548.19
Vendor30,3177,2011-10-25,2224.71
Vendor30,3181,2011-11-02,6283.59
Vendor30,3189,2011-11-02,7211.12
Vendor30,3193,2011-11-10,3779.1
Vendor30,3197,2011-11-10,6891.55
Vendor30,3203,2011-11-11,1417.66
Vendor30,3206,2011-11-11,1362.73
Vendor30,3212,2011-11-12,5523.31
Vendor30,3220,2011-11-22,1268.44
Vendor30,3221,2011-12-02,1494.95
Vendor30,3228,2011-12-02,5234.69
Vendor30,3233,2011-12-12,2471.55
Vendor30,3236,2011-12-18,5292.75
Vendor30,3244,2011-12-19,5737.81
Vendor30,3250,2011-12-29,1401.91
Vendor30,3253,2011-12-29,2056.69
Vendor30,3260,2012-01-06,4734.25
Vendor30,3265,2012-01-16,1203.43
Vendor30,3269,2012-01-22,4985.06
Vendor30,3271,2012-02-01,4617.8
Vendor30,3277,2012-02-07,6193.7
Vendor30,3284,2012-02-17,1529.5
Vendor30,3289,2012-02-27,2044.02
Vendor30,3292,2012-03-08,2387.43
Vendor30,3296,2012-03-09,3310.7
Vendor30,3301,2012-03-19,2207.94
Vendor30,3307,2012-03-19,1753.51
Vendor30,3314,2012-03-25,1155.55
Vendor30,3319,2012-03-26,8709.93
Vendor30,3321,2012-04-03,1881.73
Vendor30,3327,2012-04-04,1581.51
Vendor30,3334,2012-04-10,5863.27
Vendor30,3336,2012-04-16,3983.14
Vendor30,3345,2012-04-17,6843.21
Vendor30,3346,2012-04-17,2633.02
Vendor30,3354,2012-04-27,7193.38
Vendor30,3358,2012-05-03,2223.54
Vendor30,3364,2012-05-03,4232.7
Vendor30,3369,2012-05-04,3370.14
Vendor30,3371,2012-05-10,1001.51
Vendor30,3379,2012-05-20,1224.92
Vendor30,3385,2012-05-28,1605.87
Vendor30,3389,2012-05-29,9425.93
Vendor30,3395,2012-06-08,5833.3
Vendor30,3398,2012-06-14,1139.01
Vendor30,3405,2012-06-14,8382.44
Vendor30,3409,2012-06-14,3222.03
Vendor30,3414,2012-06-14,3106.16
Vendor30,3416,2012-06-14,9539.89
Vendor30,3422,2012-06-15,2339.69
Vendor30,3426,2012-06-15,4343.97
Vendor30,3433,2012-06-21,2108.34
Vendor30,3437,2012-06-22,1866.65
Vendor30,3445,2012-06-23,6859.46
Vendor30,3450,2012-07-03,1213.16
Vendor30,3455,2012-07-11,1677.79
Vendor30,3459,2012-07-12,1753.72
Vendor30,3465,2012-07-13,2656.73
Vendor30,3469,2012-07-13,4055.43
Vendor30,3472,2012-07-13,2112.56
Vendor30,3479,2012-07-19,3319.66
Vendor30,3481,2012-07-25,7520.68
Vendor30,3490,2012-08-02,1442.1
Vendor30,3494,2012-08-02,4723.18
Vendor30,3500,2012-08-02,5099.4
Vendor30,3505,2012-08-08,2727.47
Vendor30,3510,2012-08-09,1561.59
Vendor30,3513,2012-08-17,3083.49
Vendor34,10,2006-01-02,6180.26
Vendor34,13,2006-01-03,1070.31
Vendor34,17,2006-01-09,4621.69
Vendor34,23,2006-01-10,1763.76
Vendor34,29,2006-01-18,7637.12
Vendor34,34,2006-01-18,3089.71
Vendor34,38,2006-01-28,3600.46
Vendor34,45,2006-01-28,2633.42
Vendor34,48,2006-02-05,4145.15
Vendor34,53,2006-02-05,7401.4
Vendor34,58,2006-02-05,3344.65
Vendor34,64,2006-02-11,4342.06
Vendor34,66,2006-02-11,1257.33
Vendor34,72,2006-02-11,3112.58
Vendor34,79,2006-02-17,2806.49
Vendor34,85,2006-02-25,2727.75
Vendor34,88,2006-02-25,5793.65
Vendor34,93,2006-02-25,7645.88
Vendor34,98,2006-02-26,1165.67
Vendor34,104,2006-03-04,7782.2
Vendor34,106,2006-03-10,2578.81
Vendor34,112,2006-03-11,9748.69
Vendor34,120,2006-03-17,2851.22
Vendor34,122,2006-03-17,2694.83
Vendor34,128,2006-03-17,2852.42
Vendor34,131,2006-03-25,2568.52
Vendor34,136,2006-03-31,4200.36
Vendor34,144,2006-04-08,2486.03
Vendor34,150,2006-04-08,2081.16
Vendor34,152,2006-04-09,7303.37
Vendor34,160,2006-04-09,1044.87
Vendor34,164,2006-04-10,1538.29
Vendor34,167,2006-04-10,4142.71
Vendor34,173,2006-04-11,3355.83
Vendor34,179,2006-04-12,4609.13
Vendor34,184,2006-04-12,3205.42
Vendor34,189,2006-04-12,2619.51
Vendor34,191,2006-04-13,8693.35
Vendor34,200,2006-04-13,3204.12
Vendor34,201,2006-04-14,8576.92
Vendor34,210,2006-04-15,1739.07
Vendor34,213,2006-04-21,1504.14
Vendor34,216,2006-04-27,1914.17
Vendor34,221,2006-05-05,1333.68
Vendor34,229,2006-05-11,2472.92
Vendor34,235,2006-05-11,1740.59
Vendor34,236,2006-05-11,2340.63
Vendor34,243,2006-05-11,1122.94
Vendor34,247,2006-05-12,1347.08
Vendor34,252,2006-05-18,2163.64
Vendor34,260,2006-05-24,2656.06
Vendor34,261,2006-05-25,5393.05
Vendor34,266,2006-05-26,5143.75
Vendor34,275,2006-05-26,3170.15
Vendor34,276,2006-06-05,7173.76
Vendor34,285,2006-06-05,4178.64
Vendor34,290,2006-06-05,1589.88
Vendor34,291,2006-06-11,2059.98
Vendor34,297,2006-06-19,1661.74
Vendor34,302,2006-06-19,1016.64
Vendor34,310,2006-06-29,1502.87
Vendor34,315,2006-07-07,3635.19
Vendor34,320,2006-07-07,3732.9
Vendor34,325,2006-07-08,8412.76
Vendor34,329,2006-07-16,2611.37
Vendor34,335,2006-07-16,1675.64
Vendor34,340,2006-07-16,9646.41
Vendor34,342,2006-07-24,3117.87
Vendor34,347,2006-07-25,1834.7
Vendor34,352,2006-07-25,7136.51
Vendor34,356,2006-08-04,3648.03
Vendor34,364,2006-08-05,1438.34
Vendor34,368,2006-08-06,2404.35
Vendor34,374,2006-08-12,1188.57
Vendor34,378,2006-08-20,2005.75
Vendor34,381,2006-08-26,1757.4
Vendor34,388,2006-08-27,8342.28
Vendor34,392,2006-08-27,1425.57
Vendor34,400,2006-08-27,1252.78
Vendor34,403,2006-08-27,3552.12
Vendor34,409,2006-08-27,4032.86
Vendor34,415,2006-09-02,1170.85
Vendor34,420,2006-09-02,2312.91
Vendor34,424,2006-09-02,3123.47
Vendor34,428,2006-09-02,3117.07
Vendor34,435,2006-09-03,1594.78
Vendor34,439,2006-09-11,4044.05
Vendor34,443,2006-09-11,3137.97
Vendor34,449,2006-09-17,6515.8
Vendor34,455,2006-09-23,4944.54
Vendor34,457,2006-09-29,2789.98
Vendor34,462,2006-09-30,1518.61
Vendor34,468,2006-10-10,3402.66
Vendor34,471,2006-10-10,3356.5
Vendor34,477,2006-10-16,2635.81
Vendor34,482,2006-10-17,4986.4
Vendor34,488,2006-10-17,2712.23
Vendor34,492,2006-10-23,1335.51
Vendor34,500,2006-10-29,5089.02
Vendor34,504,2006-11-06,3298.03
Vendor34,507,2006-11-14,1145.37
Vendor34,514,2006-11-22,2728.89
Vendor34,516,2006-11-28,2039.44
Vendor34,525,2006-12-04,2798.96
Vendor34,528,2006-12-04,7469.9
Vendor34,532,2006-12-10,1214.39
Vendor34,538,2006-12-16,1047.29
Vendor34,542,2006-12-16,7190.95
Vendor34,549,2006-12-16,1178.85
Vendor34,555,2006-12-17,1480.41
Vendor34,560,2006-12-18,2177.4
Vendor34,562,2006-12-28,1092.53
Vendor34,567,2007-01-03,5292.84
Vendor34,572,2007-01-09,8664.41
Vendor34,579,2007-01-15,4401.87
Vendor34,583,2007-01-15,1777.78
Vendor34,589,2007-01-23,4599.37
Vendor34,593,2007-01-29,1094.42
Vendor34,599,2007-01-29,4495.07
Vendor34,604,2007-01-29,1754.78
Vendor34,610,2007-01-30,3419.03
Vendor34,614,2007-01-31,2106.02
Vendor34,620,2007-01-31,4903.8
Vendor34,623,2007-01-31,1307.67
Vendor34,629,2007-02-01,4668.78
Vendor34,633,2007-02-01,5731.28
Vendor34,637,2007-02-07,1308.42
Vendor34,645,2007-02-13,5037.64
Vendor34,650,2007-02-14,1708.27
Vendor34,655,2007-02-14,1827.25
Vendor34,656,2007-02-15,9706.64
Vendor34,662,2007-02-23,1315.6
Vendor34,669,2007-03-01,3141.82
Vendor34,672,2007-03-01,3385.62
Vendor34,676,2007-03-09,1048.39
Vendor34,684,2007-03-15,1917.3
Vendor34,690,2007-03-16,1735.99
Vendor34,692,2007-03-16,7142.99
Vendor34,698,2007-03-17,1342.57
Vendor34,701,2007-03-18,5942.42
Vendor34,710,2007-03-24,7527.75
Vendor34,715,2007-03-24,2654.76
Vendor34,719,2007-03-25,6943.81
Vendor34,724,2007-04-04,3720.36
Vendor34,729,2007-04-12,3085.09
Vendor34,734,2007-04-22,6991.97
Vendor34,740,2007-04-23,5831.34
Vendor34,744,2007-04-23,3855.3
Vendor34,747,2007-05-01,2000.07
Vendor34,751,2007-05-01,5143.36
Vendor34,756,2007-05-07,7630.44
Vendor34,764,2007-05-07,2438.97
Vendor34,768,2007-05-07,9085.99
Vendor34,773,2007-05-07,7122.76
Vendor34,780,2007-05-08,2461.12
Vendor34,784,2007-05-16,3433.95
Vendor34,790,2007-05-24,1342.71
Vendor34,795,2007-05-30,5735.91
Vendor34,796,2007-06-05,2063.7
Vendor34,804,2007-06-06,2716.7
Vendor34,809,2007-06-12,3582.03
Vendor34,815,2007-06-13,3151.21
Vendor34,816,2007-06-13,2558.26
Vendor34,821,2007-06-14,3598.16
Vendor34,829,2007-06-14,8018.7
Vendor34,835,2007-06-15,3703.72
Vendor34,837,2007-06-15,8122.82
Vendor34,844,2007-06-16,4934.72
Vendor34,849,2007-06-22,6040.77
Vendor34,851,2007-06-22,1073.87
Vendor34,860,2007-06-22,1769.52
Vendor34,862,2007-06-23,5637.42
Vendor34,866,2007-06-23,5513.85
Vendor34,872,2007-06-24,8639.48
Vendor34,877,2007-06-30,1161.09
Vendor34,884,2007-07-06,1483.25
Vendor34,889,2007-07-12,3725.42
Vendor34,893,2007-07-13,2252.76
Vendor34,900,2007-07-14,2217.14
Vendor34,901,2007-07-14,2619.25
Vendor34,908,2007-07-24,6229.48
Vendor34,912,2007-07-30,1507.07
Vendor34,917,2007-08-09,1844.83
Vendor34,925,2007-08-19,1328.91
Vendor34,928,2007-08-20,2298.12
Vendor34,933,2007-08-21,9731.75
Vendor34,939,2007-08-27,1550.9
Vendor34,943,2007-09-06,8491.28
Vendor34,948,2007-09-06,3899.55
Vendor34,953,2007-09-06,1463.21
Vendor34,960,2007-09-06,3811.42
Vendor34,964,2007-09-07,3932.24
Vendor34,966,2007-09-07,9498.3
Vendor34,975,2007-09-17,3769.6
Vendor34,978,2007-09-17,1354.79
Vendor34,982,2007-09-25,2821.23
Vendor34,989,2007-10-01,3340.46
Vendor34,992,2007-10-07,6601.52
Vendor34,997,2007-10-13,8737.4
Vendor34,1002,2007-10-23,1459.85
Vendor34,1006,2007-10-23,4445.81
Vendor34,1014,2007-10-23,8279.43
Vendor34,1016,2007-10-29,5721.45
Vendor34,1025,2007-11-04,7306.35
Vendor34,1030,2007-11-05,5042.71
Vendor34,1034,2007-11-11,1917.0
Vendor34,1039,2007-11-12,1566.55
Vendor34,1045,2007-11-12,1354.48
Vendor34,1050,2007-11-22,6148.91
Vendor34,1055,2007-11-22,3722.95
Vendor34,1059,2007-11-23,8084.25
Vendor34,1062,2007-11-24,4661.91
Vendor34,1067,2007-11-30,7051.9
Vendor34,1072,2007-11-30,1711.84
Vendor34,1080,2007-12-06,1811.77
Vendor34,1084,2007-12-07,1669.66
Vendor34,1089,2007-12-15,1755.15
Vendor34,1094,2007-12-16,1209.8
Vendor34,1100,2007-12-16,1780.71
Vendor34,1102,2007-12-17,1133.33
Vendor34,1110,2007-12-18,1396.83
Vendor34,1111,2007-12-18,1895.38
Vendor34,1116,2007-12-18,1791.16
Vendor34,1124,2007-12-24,7857.6
Vendor34,1127,2008-01-03,3471.32
Vendor34,1132,2008-01-13,2782.22
Vendor34,1138,2008-01-13,2296.75
Vendor34,1144,2008-01-19,1064.05
Vendor34,1147,2008-01-29,3632.33
Vendor34,1151,2008-02-04,1043.25
Vendor34,1158,2008-02-04,1618.44
Vendor34,1163,2008-02-10,6080.01
Vendor34,1167,2008-02-18,6445.62
Vendor34,1175,2008-02-18,1039.86
Vendor34,1176,2008-02-28,6389.11
Vendor34,1185,2008-03-05,1273.7
Vendor34,1189,2008-03-05,2054.0
Vendor34,1192,2008-03-06,2752.57
Vendor34,1199,2008-03-14,5319.78
Vendor34,1202,2008-03-20,1154.78
Vendor34,1206,2008-03-20,1285.5
Vendor34,1214,2008-03-20,3211.01
Vendor34,1218,2008-03-21,1302.06
Vendor34,1224,2008-03-27,5779.81
Vendor34,1230,2008-03-27,1025.35
Vendor34,1235,2008-03-28,6698.29
Vendor34,1236,2008-04-05,3029.68
Vendor34,1242,2008-04-11,2145.12
Vendor34,1246,2008-04-17,1561.04
Vendor34,1251,2008-04-18,2829.2
Vendor34,1256,2008-04-18,4115.58
Vendor34,1261,2008-04-24,4618.26
Vendor34,1267,2008-04-24,2366.29
Vendor34,1271,2008-04-24,5510.07
Vendor34,1276,2008-04-30,3273.86
Vendor34,1285,2008-05-10,5478.23
Vendor34,1290,2008-05-20,4445.57
Vendor34,1295,2008-05-21,1101.28
Vendor34,1299,2008-05-22,3979.81
Vendor34,1302,2008-05-22,5420.3
Vendor34,1310,2008-05-22,4155.53
Vendor34,1314,2008-05-28,3358.3
Vendor34,1320,2008-05-29,1672.32
Vendor34,1324,2008-05-30,4112.75
Vendor34,1329,2008-06-05,3846.36
Vendor34,1333,2008-06-05,6619.21
Vendor34,1337,2008-06-06,9846.9
Vendor34,1345,2008-06-12,4404.16
Vendor34,1347,2008-06-12,9586.62
Vendor34,1352,2008-06-18,2473.0
Vendor34,1357,2008-06-18,1528.39
Vendor34,1364,2008-06-28,2712.44
Vendor34,1367,2008-06-29,3896.18
Vendor34,1374,2008-06-29,1974.23
Vendor34,1380,2008-06-29,5857.41
Vendor34,1381,2008-06-30,2661.65
Vendor34,1386,2008-06-30,4193.56
Vendor34,1392,2008-07-10,4827.43
Vendor34,1400,2008-07-20,7917.79
Vendor34,1401,2008-07-30,2207.43
Vendor34,1410,2008-08-07,1063.04
Vendor34,1413,2008-08-13,6501.25
Vendor34,1419,2008-08-19,1637.07
Vendor34,1423,2008-08-19,9445.73
Vendor34,1427,2008-08-20,3472.68
Vendor34,1432,2008-08-20,5199.46
Vendor34,1436,2008-08-20,4652.45
Vendor34,1443,2008-08-26,1032.06
Vendor34,1448,2008-08-27,1774.19
Vendor34,1454,2008-09-02,3813.92
Vendor34,1459,2008-09-12,2435.19
Vendor34,1464,2008-09-20,1924.1
Vendor34,1470,2008-09-26,3478.76
Vendor34,1472,2008-09-27,1094.97
Vendor34,1480,2008-09-28,2955.24
Vendor34,1481,2008-10-08,2176.86
Vendor34,1489,2008-10-16,2236.84
Vendor34,1491,2008-10-22,7119.55
Vendor34,1498,2008-10-22,1695.56
Vendor34,1503,2008-10-23,1033.2
Vendor34,1506,2008-10-29,1671.45
Vendor34,1511,2008-11-04,1568.36
Vendor34,1516,2008-11-04,1683.53
Vendor34,1525,2008-11-04,7328.33
Vendor34,1529,2008-11-10,2309.53
Vendor34,1535,2008-11-16,2851.59
Vendor34,1537,2008-11-22,1595.87
Vendor34,1544,2008-11-23,1202.67
Vendor34,1547,2008-12-01,1783.11
Vendor34,1553,2008-12-01,2276.36
Vendor34,1560,2008-12-11,2709.28
Vendor34,1565,2008-12-11,3505.3
Vendor34,1569,2008-12-11,4160.54
Vendor34,1573,2008-12-12,2361.74
Vendor34,1580,2008-12-12,1974.88
Vendor34,1585,2008-12-18,7351.98
Vendor34,1587,2008-12-24,4609.48
Vendor34,1593,2008-12-30,7717.35
Vendor34,1596,2009-01-09,6099.75
Vendor34,1602,2009-01-15,6070.91
Vendor34,1606,2009-01-16,1534.55
Vendor34,1612,2009-01-16,3636.16
Vendor34,1619,2009-01-17,3148.39
Vendor34,1622,2009-01-27,2851.15
Vendor34,1628,2009-02-04,1460.19
Vendor34,1635,2009-02-04,6003.94
Vendor34,1640,2009-02-04,1869.02
Vendor34,1641,2009-02-05,2771.52
Vendor34,1648,2009-02-06,8660.33
Vendor34,1653,2009-02-16,8248.84
Vendor34,1656,2009-02-22,2406.14
Vendor34,1662,2009-02-22,3492.82
Vendor34,1669,2009-03-04,1374.94
Vendor34,1675,2009-03-14,1105.86
Vendor34,1679,2009-03-15,1432.84
Vendor34,1681,2009-03-23,3946.36
Vendor34,1689,2009-03-23,4990.02
Vendor34,1692,2009-03-29,1104.57
Vendor34,1698,2009-04-08,3458.32
Vendor34,1705,2009-04-08,7337.9
Vendor34,1709,2009-04-08,8810.56
Vendor34,1713,2009-04-09,3937.37
Vendor34,1716,2009-04-19,4163.95
Vendor34,1721,2009-04-29,2102.08
Vendor34,1729,2009-05-05,1415.48
Vendor34,1734,2009-05-11,4097.17
Vendor34,1739,2009-05-21,1447.31
Vendor34,1741,2009-05-22,3863.03
Vendor34,1746,2009-05-28,1078.06
Vendor34,1751,2009-06-03,5877.56
Vendor34,1758,2009-06-03,2680.51
Vendor34,1763,2009-06-03,6723.6
Vendor34,1770,2009-06-09,8882.73
Vendor34,1774,2009-06-09,2334.37
Vendor34,1776,2009-06-15,1334.99
Vendor34,1783,2009-06-16,1809.18
Vendor34,1787,2009-06-17,7017.29
Vendor34,1794,2009-06-17,2136.35
Vendor34,1800,2009-06-17,4324.94
Vendor34,1803,2009-06-18,1151.83
Vendor34,1806,2009-06-19,5330.79
Vendor34,1813,2009-06-20,3341.18
Vendor34,1816,2009-06-26,1430.07
Vendor34,1825,2009-06-27,7742.95
Vendor34,1826,2009-07-07,3835.32
Vendor34,1834,2009-07-08,4256.51
Vendor34,1836,2009-07-18,2410.01
Vendor34,1841,2009-07-26,1310.4
Vendor34,1850,2009-07-26,1554.35
Vendor34,1854,2009-08-01,1090.83
Vendor34,1857,2009-08-02,7715.21
Vendor34,1865,2009-08-02,8259.34
Vendor34,1869,2009-08-10,2125.62
Vendor34,1873,2009-08-10,6005.37
Vendor34,1878,2009-08-11,3287.02
Vendor34,1881,2009-08-21,3578.33
Vendor34,1889,2009-08-22,2504.75
Vendor34,1892,2009-08-30,1072.95
Vendor34,1897,2009-08-30,8674.11
Vendor34,1902,2009-09-05,3046.89
Vendor34,1906,2009-09-06,5403.92
Vendor34,1914,2009-09-06,7020.35
Vendor34,1918,2009-09-12,2171.34
Vendor34,1924,2009-09-20,8544.08
Vendor34,1929,2009-09-21,5124.69
Vendor34,1934,2009-09-21,1494.28
Vendor34,1940,2009-09-21,3100.83
Vendor34,1941,2009-09-29,6953.57
Vendor34,1950,2009-10-05,2791.51
Vendor34,1952,2009-10-06,6425.65
Vendor34,1960,2009-10-12,1767.84
Vendor34,1963,2009-10-13,2644.03
Vendor34,1967,2009-10-14,1272.88
Vendor34,1975,2009-10-20,1858.08
Vendor34,1976,2009-10-20,4106.83
Vendor34,1982,2009-10-21,1339.09
Vendor34,1986,2009-10-22,3327.94
Vendor34,1993,2009-10-22,1041.71
Vendor34,1997,2009-10-22,4067.99
Vendor34,2003,2009-10-22,5816.46
Vendor34,2009,2009-10-28,8035.11
Vendor34,2015,2009-11-03,1726.48
Vendor34,2020,2009-11-03,6691.58
Vendor34,2024,2009-11-13,1740.29
Vendor34,2030,2009-11-13,5938.0
Vendor34,2031,2009-11-14,1977.12
Vendor34,2037,2009-11-22,1013.05
Vendor34,2043,2009-11-22,4456.49
Vendor34,2050,2009-11-23,5938.83
Vendor34,2051,2009-11-24,1542.49
Vendor34,2057,2009-11-24,5939.79
Vendor34,2061,2009-11-24,1734.97
Vendor34,2067,2009-11-25,6295.99
Vendor34,2071,2009-11-25,1272.75
Vendor34,2080,2009-11-26,1051.44
Vendor34,2082,2009-12-02,9783.31
Vendor34,2090,2009-12-12,6743.82
Vendor34,2092,2009-12-22,2176.28
Vendor34,2100,2009-12-22,8678.79
Vendor34,2102,2009-12-23,1548.99
Vendor34,2109,2009-12-31,1120.69
Vendor34,2113,2010-01-01,1996.21
Vendor34,2117,2010-01-01,1732.36
Vendor34,2124,2010-01-07,2580.13
Vendor34,2126,2010-01-13,1060.08
Vendor34,2133,2010-01-13,3408.3
Vendor34,2140,2010-01-21,2385.03
Vendor34,2145,2010-01-27,6142.16
Vendor34,2150,2010-02-06,1044.09
Vendor34,2155,2010-02-14,1961.4
Vendor34,2157,2010-02-15,1386.59
Vendor34,2165,2010-02-15,2402.51
Vendor34,2168,2010-02-16,1614.55
Vendor34,2175,2010-02-17,3342.76
Vendor34,2180,2010-02-18,2760.06
Vendor34,2185,2010-02-24,4932.16
Vendor34,2188,2010-02-25,3170.44
Vendor34,2194,2010-02-26,3657.47
Vendor34,2197,2010-02-26,7532.74
Vendor34,2204,2010-02-26,3391.95
Vendor34,2210,2010-02-27,7688.59
Vendor34,2215,2010-02-27,1706.03
Vendor34,2219,2010-03-05,7707.27
Vendor34,2221,2010-03-06,1062.27
Vendor34,2226,2010-03-07,2165.38
Vendor34,2232,2010-03-15,4903.44
Vendor34,2239,2010-03-15,2023.96
Vendor34,2243,2010-03-21,3920.06
Vendor34,2246,2010-03-21,2553.27
Vendor34,2254,2010-03-27,3838.05
Vendor34,2259,2010-03-27,2833.33
Vendor34,2264,2010-03-27,3043.36
Vendor34,2266,2010-04-04,2665.41
Vendor34,2273,2010-04-10,1767.09
Vendor34,2279,2010-04-16,7063.85
Vendor34,2282,2010-04-22,8620.65
Vendor34,2290,2010-04-30,2917.2
Vendor34,2295,2010-05-01,1193.5
Vendor34,2299,2010-05-11,1609.93
Vendor34,2305,2010-05-12,7698.96
Vendor34,2309,2010-05-18,5718.9
Vendor34,2314,2010-05-18,1751.51
Vendor34,2319,2010-05-19,2134.58
Vendor34,2325,2010-05-19,5050.44
Vendor34,2326,2010-05-29,1039.59
Vendor34,2332,2010-06-08,8221.61
Vendor34,2338,2010-06-08,4209.22
Vendor34,2342,2010-06-14,1182.94
Vendor34,2346,2010-06-22,5734.28
Vendor34,2354,2010-06-28,3530.47
Vendor34,2357,2010-07-08,2512.23
Vendor34,2365,2010-07-08,2737.77
Vendor34,2368,2010-07-08,2745.39
Vendor34,2373,2010-07-14,6913.86
Vendor34,2378,2010-07-20,1606.4
Vendor34,2385,2010-07-26,4607.61
Vendor34,2388,2010-07-26,9773.25
Vendor34,2394,2010-08-01,8305.82
Vendor34,2398,2010-08-02,2786.34
Vendor34,2402,2010-08-03,7077.85
Vendor34,2406,2010-08-09,4547.28
Vendor34,2413,2010-08-09,1892.07
Vendor34,2416,2010-08-17,7249.03
Vendor34,2423,2010-08-25,1198.21
Vendor34,2430,2010-08-31,7465.42
Vendor34,2435,2010-09-10,1195.37
Vendor34,2439,2010-09-10,5249.34
Vendor34,2441,2010-09-20,1201.93
Vendor34,2447,2010-09-20,3617.08
Vendor34,2453,2010-09-21,1479.46
Vendor34,2460,2010-09-22,8300.4
Vendor34,2465,2010-09-23,3551.12
Vendor34,2467,2010-09-23,4042.1
Vendor34,2472,2010-09-23,3816.44
Vendor34,2477,2010-09-24,5754.03
Vendor34,2485,2010-09-25,2156.83
Vendor34,2490,2010-10-01,9675.4
Vendor34,2491,2010-10-11,2452.58
Vendor34,2498,2010-10-17,4554.99
Vendor34,2501,2010-10-27,4755.92
Vendor34,2507,2010-10-28,8807.02
Vendor34,2513,2010-10-28,1684.4
Vendor34,2518,2010-10-28,2668.4
Vendor34,2524,2010-10-28,9043.92
Vendor34,2528,2010-10-28,1525.4
Vendor34,2535,2010-11-07,8693.88
Vendor34,2539,2010-11-07,4391.27
Vendor34,2542,2010-11-08,1320.95
Vendor34,2547,2010-11-14,4357.89
Vendor34,2555,2010-11-14,4212.33
Vendor34,2556,2010-11-14,1380.03
Vendor34,2563,2010-11-24,4106.09
Vendor34,2566,2010-11-24,6554.74
Vendor34,2572,2010-12-02,2659.69
Vendor34,2577,2010-12-08,7889.07
Vendor34,2584,2010-12-14,1660.07
Vendor34,2587,2010-12-15,3013.77
Vendor34,2594,2010-12-16,2310.55
Vendor34,2599,2010-12-17,6746.49
Vendor34,2604,2010-12-27,3377.25
Vendor34,2610,2010-12-27,6721.08
Vendor34,2611,2010-12-27,4157.97
Vendor34,2618,2011-01-02,2670.07
Vendor34,2622,2011-01-03,1515.45
Vendor34,2626,2011-01-03,4543.5
Vendor34,2635,2011-01-04,7607.26
Vendor34,2640,2011-01-05,1137.12
Vendor34,2643,2011-01-13,5131.92
Vendor34,2649,2011-01-23,9175.74
Vendor34,2653,2011-01-23,2470.49
Vendor34,2659,2011-01-24,3363.67
Vendor34,2661,2011-01-30,1654.58
Vendor34,2670,2011-02-07,1139.21
Vendor34,2672,2011-02-07,4542.07
Vendor34,2678,2011-02-08,2878.52
Vendor34,2685,2011-02-08,5983.14
Vendor34,2688,2011-02-08,1011.83
Vendor34,2695,2011-02-18,2512.34
Vendor34,2700,2011-02-19,1405.6
Vendor34,2705,2011-02-19,3240.1
Vendor34,2707,2011-02-25,1921.47
Vendor34,2715,2011-02-26,8140.82
Vendor34,2720,2011-02-27,9045.82
Vendor34,2725,2011-02-27,2911.14
Vendor34,2728,2011-03-07,7770.01
Vendor34,2735,2011-03-17,2055.52
Vendor34,2737,2011-03-25,6672.71
Vendor34,2741,2011-04-02,3060.89
Vendor34,2746,2011-04-10,1142.21
Vendor34,2752,2011-04-18,1211.66
Vendor34,2758,2011-04-24,6197.84
Vendor34,2764,2011-04-24,7317.67
Vendor34,2767,2011-05-02,5530.12
Vendor34,2773,2011-05-03,2291.36
Vendor34,2777,2011-05-03,1348.52
Vendor34,2782,2011-05-03,5397.1
Vendor34,2789,2011-05-09,5213.71
Vendor34,2792,2011-05-09,1279.74
Vendor34,2800,2011-05-15,2948.66
Vendor34,2805,2011-05-15,1761.29
Vendor34,2808,2011-05-16,7806.81
Vendor34,2812,2011-05-17,1627.37
Vendor34,2818,2011-05-25,2518.54
Vendor34,2823,2011-05-31,3783.48
Vendor34,2830,2011-06-06,2036.59
Vendor34,2831,2011-06-14,5443.18
Vendor34,2836,2011-06-20,1452.23
Vendor34,2843,2011-06-20,1888.65
Vendor34,2850,2011-06-30,1438.48
Vendor34,2851,2011-07-01,1480.0
Vendor34,2858,2011-07-09,6313.36
Vendor34,2862,2011-07-09,5186.61
Vendor34,2867,2011-07-15,5421.25
Vendor34,2871,2011-07-16,4246.77
Vendor34,2876,2011-07-24,6802.23
Vendor34,2884,2011-07-25,1690.2
Vendor34,2886,2011-07-25,6065.99
Vendor34,2891,2011-07-26,1522.94
Vendor34,2898,2011-07-27,9807.24
Vendor34,2901,2011-08-04,2362.14
Vendor34,2909,2011-08-04,1896.32
Vendor34,2913,2011-08-05,1326.95
Vendor34,2917,2011-08-05,2445.66
Vendor34,2925,2011-08-06,1426.56
Vendor34,2929,2011-08-07,2603.82
Vendor34,2931,2011-08-07,3469.28
Vendor34,2937,2011-08-13,2497.29
Vendor34,2942,2011-08-14,8403.77
Vendor34,2947,2011-08-24,8937.6
Vendor34,2951,2011-08-25,7385.39
Vendor34,2956,2011-08-31,1639.61
Vendor34,2965,2011-09-10,1470.26
Vendor34,2969,2011-09-16,1072.14
Vendor34,2971,2011-09-26,3749.4
Vendor34,2979,2011-09-26,2180.26
Vendor34,2984,2011-10-02,4108.94
Vendor34,2989,2011-10-03,2267.52
Vendor34,2993,2011-10-03,6025.17
Vendor34,2996,2011-10-13,1377.41
Vendor34,3004,2011-10-21,3960.75
Vendor34,3006,2011-10-27,3842.16
Vendor34,3014,2011-10-28,1507.76
Vendor34,3016,2011-11-05,8385.21
Vendor34,3021,2011-11-11,2565.31
Vendor34,3030,2011-11-11,3348.32
Vendor34,3031,2011-11-19,3292.07
Vendor34,3038,2011-11-29,1146.08
Vendor34,3041,2011-11-29,5786.06
Vendor34,3050,2011-11-29,1236.26
Vendor34,3055,2011-12-09,3810.85
Vendor34,3059,2011-12-10,1573.41
Vendor34,3064,2011-12-10,5420.38
Vendor34,3070,2011-12-10,1830.81
Vendor34,3072,2011-12-18,5222.09
Vendor34,3077,2011-12-24,1577.35
Vendor34,3083,2011-12-30,4964.75
Vendor34,3090,2012-01-07,6130.72
Vendor34,3093,2012-01-17,7908.73
Vendor34,3097,2012-01-18,3653.55
Vendor34,3103,2012-01-19,1518.01
Vendor34,3108,2012-01-27,2634.48
Vendor34,3114,2012-01-27,1766.2
Vendor34,3117,2012-01-28,1593.11
Vendor34,3123,2012-01-29,2244.86
Vendor34,3127,2012-01-29,1623.44
Vendor34,3134,2012-02-06,1066.24
Vendor34,3140,2012-02-07,6220.44
Vendor34,3144,2012-02-08,3404.49
Vendor34,3146,2012-02-16,3198.95
Vendor34,3151,2012-02-22,7418.7
Vendor34,3159,2012-02-28,8929.58
Vendor34,3165,2012-03-09,2220.82
Vendor34,3170,2012-03-10,1259.21
Vendor34,3172,2012-03-16,1225.64
Vendor34,3177,2012-03-16,4265.55
Vendor34,3185,2012-03-22,9519.51
Vendor34,3187,2012-03-23,5137.21
Vendor34,3191,2012-03-31,4355.99
Vendor34,3200,2012-04-08,6823.52
Vendor34,3205,2012-04-09,3223.78
Vendor34,3207,2012-04-10,1163.92
Vendor34,3214,2012-04-11,6521.49
Vendor34,3220,2012-04-21,9675.94
Vendor34,3222,2012-04-22,7129.38
Vendor34,3227,2012-04-30,4023.9
Vendor34,3233,2012-05-10,6521.57
Vendor34,3237,2012-05-11,1260.5
Vendor34,3242,2012-05-19,7851.96
Vendor34,3250,2012-05-20,1994.49
Vendor39,7,2006-01-09,9714.45
Vendor39,13,2006-01-09,3499.58
Vendor39,20,2006-01-15,8433.86
Vendor39,25,2006-01-23,3463.4
Vendor39,29,2006-01-23,1189.17
Vendor39,32,2006-01-24,9134.26
Vendor39,38,2006-01-30,5302.12
Vendor39,44,2006-01-31,5487.22
Vendor39,50,2006-01-31,1346.87
Vendor39,51,2006-02-01,1288.38
Vendor39,57,2006-02-02,1181.19
Vendor39,65,2006-02-08,3632.68
Vendor39,68,2006-02-08,1513.69
Vendor39,74,2006-02-09,8967.13
Vendor39,76,2006-02-09,1006.58
Vendor39,81,2006-02-17,1251.07
Vendor39,87,2006-02-25,9745.27
Vendor39,92,2006-02-26,2882.37
Vendor39,99,2006-02-26,1190.78
Vendor39,104,2006-02-26,1281.47
Vendor39,107,2006-03-06,4438.73
Vendor39,114,2006-03-06,6513.46
Vendor39,118,2006-03-07,5106.08
Vendor39,121,2006-03-07,8458.59
Vendor39,127,2006-03-08,6548.95
Vendor39,135,2006-03-14,4009.21
Vendor39,139,2006-03-15,3825.57
Vendor39,144,2006-03-25,1567.6
Vendor39,150,2006-03-25,8156.96
Vendor39,155,2006-03-31,1845.52
Vendor39,156,2006-04-01,4028.73
Vendor39,165,2006-04-01,1128.79
Vendor39,167,2006-04-07,5548.2
Vendor39,173,2006-04-07,2167.0
Vendor39,177,2006-04-17,3401.81
Vendor39,184,2006-04-18,3611.32
Vendor39,187,2006-04-28,1763.54
Vendor39,195,2006-04-28,1344.16
Vendor39,196,2006-04-29,1938.57
Vendor39,205,2006-05-09,1531.09
Vendor39,210,2006-05-10,5473.64
Vendor39,214,2006-05-10,3791.5
Vendor39,216,2006-05-10,1142.35
Vendor39,223,2006-05-11,2232.8
Vendor39,226,2006-05-11,1772.07
Vendor39,231,2006-05-21,6232.81
Vendor39,240,2006-05-21,2587.12
Vendor39,242,2006-05-22,2779.35
Vendor39,247,2006-05-22,1584.18
Vendor39,254,2006-05-28,4718.37
Vendor39,257,2006-05-29,4353.69
Vendor39,263,2006-05-29,7788.21
Vendor39,268,2006-06-08,5177.39
Vendor39,273,2006-06-14,1943.23
Vendor39,279,2006-06-15,1029.53
Vendor39,281,2006-06-25,1412.55
Vendor39,289,2006-06-26,2410.37
Vendor39,292,2006-07-02,7139.85
Vendor39,296,2006-07-12,3982.8
Vendor39,302,2006-07-12,6916.76
Vendor39,308,2006-07-20,6473.88
Vendor39,314,2006-07-26,1099.85
Vendor39,316,2006-08-05,1340.22
Vendor39,325,2006-08-13,6134.04
Vendor39,330,2006-08-14,9971.44
Vendor39,331,2006-08-15,9381.48
Vendor39,339,2006-08-15,8684.99
Vendor39,341,2006-08-16,1486.62
Vendor39,347,2006-08-16,1844.73
Vendor39,351,2006-08-16,1784.4
Vendor39,357,2006-08-22,1147.07
Vendor39,362,2006-08-23,2939.96
Vendor39,366,2006-08-29,6957.6
Vendor39,373,2006-09-06,1694.94
Vendor39,376,2006-09-07,3266.28
Vendor39,382,2006-09-15,2989.37
Vendor39,390,2006-09-25,9589.21
Vendor39,392,2006-09-25,7793.38
Vendor39,396,2006-10-05,8133.83
Vendor39,404,2006-10-06,5857.45
Vendor39,408,2006-10-06,1815.62
Vendor39,412,2006-10-07,1520.56
Vendor39,417,2006-10-13,4712.06
Vendor39,425,2006-10-14,1277.77
Vendor39,428,2006-10-24,9872.85
Vendor39,431,2006-10-30,6523.18
Vendor39,437,2006-10-30,1455.2
Vendor39,442,2006-11-09,4033.66
Vendor39,450,2006-11-10,4743.18
Vendor39,452,2006-11-10,3649.16
Vendor39,459,2006-11-10,1623.68
Vendor39,465,2006-11-10,1296.43
Vendor39,466,2006-11-16,3285.9
Vendor39,474,2006-11-16,7410.84
Vendor39,480,2006-11-17,7654.64
Vendor39,483,2006-11-23,1514.12
Vendor39,486,2006-11-23,2272.68
Vendor39,491,2006-12-01,1192.82
Vendor39,499,2006-12-07,2552.99
Vendor39,501,2006-12-08,5274.15
Vendor39,508,2006-12-16,2085.88
Vendor39,514,2006-12-26,1187.43
Vendor39,518,2007-01-03,5111.67
Vendor39,524,2007-01-11,6703.56
Vendor39,526,2007-01-11,3244.88
Vendor39,533,2007-01-21,2701.35
Vendor39,540,2007-01-27,8002.63
Vendor39,544,2007-02-06,9892.95
Vendor39,546,2007-02-12,5111.09
Vendor39,551,2007-02-12,1069.11
Vendor39,557,2007-02-13,7895.29
Vendor39,561,2007-02-19,3473.81
Vendor39,566,2007-02-19,8539.8
Vendor39,575,2007-02-27,4225.36
Vendor39,579,2007-03-07,1856.3
Vendor39,582,2007-03-17,3125.41
Vendor39,588,2007-03-23,2204.88
Vendor39,594,2007-03-29,9000.36
Vendor39,599,2007-03-30,6443.54
Vendor39,601,2007-03-31,1031.88
Vendor39,606,2007-04-01,2710.21
Vendor39,612,2007-04-07,2759.46
Vendor39,616,2007-04-13,8645.0
Vendor39,623,2007-04-23,5612.43
Vendor39,628,2007-04-24,1858.49
Vendor39,633,2007-05-04,2814.72
Vendor39,640,2007-05-10,2179.46
Vendor39,643,2007-05-16,2092.43
Vendor39,646,2007-05-17,2739.84
Vendor39,653,2007-05-25,4325.99
Vendor39,658,2007-05-31,8421.95
Vendor39,664,2007-06-01,1235.0
Vendor39,670,2007-06-02,9476.91
Vendor39,673,2007-06-03,3330.95
Vendor39,677,2007-06-03,9647.0
Vendor39,683,2007-06-04,3720.66
Vendor39,686,2007-06-05,3158.68
Vendor39,692,2007-06-05,9330.27
Vendor39,696,2007-06-05,1962.58
Vendor39,704,2007-06-11,4212.3
Vendor39,710,2007-06-11,6599.02
Vendor39,715,2007-06-17,5204.9
Vendor39,718,2007-06-18,5537.28
Vendor39,725,2007-06-18,7477.3
Vendor39,728,2007-06-19,1143.38
Vendor39,731,2007-06-20,5670.05
Vendor39,738,2007-06-21,1062.58
Vendor39,745,2007-06-22,3671.98
Vendor39,746,2007-06-23,2865.98
Vendor39,752,2007-06-23,8871.02
Vendor39,759,2007-06-24,1552.6
Vendor39,763,2007-06-30,9723.12
Vendor39,768,2007-06-30,9147.56
Vendor39,771,2007-07-01,6307.97
Vendor39,777,2007-07-07,1648.78
Vendor39,783,2007-07-07,4850.23
Vendor39,789,2007-07-13,2012.85
Vendor39,795,2007-07-13,1314.34
Vendor39,797,2007-07-14,6036.05
Vendor39,802,2007-07-15,3045.27
Vendor39,809,2007-07-16,3592.2
Vendor39,811,2007-07-22,5291.14
Vendor39,820,2007-07-28,8724.39
Vendor39,822,2007-08-03,9324.27
Vendor39,827,2007-08-03,4745.89
Vendor39,833,2007-08-03,2550.73
Vendor39,836,2007-08-03,4041.17
Vendor39,841,2007-08-04,2051.33
Vendor39,846,2007-08-10,2402.38
Vendor39,854,2007-08-10,7704.21
Vendor39,859,2007-08-10,8174.77
Vendor39,864,2007-08-10,1034.38
Vendor39,866,2007-08-18,9053.44
Vendor39,875,2007-08-19,1511.79
Vendor39,876,2007-08-19,3206.56
Vendor39,882,2007-08-20,9981.17
Vendor39,890,2007-08-28,1152.39
Vendor39,891,2007-09-03,2078.74
Vendor39,898,2007-09-09,1344.32
Vendor39,905,2007-09-10,7236.27
Vendor39,910,2007-09-10,3140.37
Vendor39,911,2007-09-16,9449.02
Vendor39,917,2007-09-22,4249.48
Vendor39,924,2007-09-23,2648.15
Vendor39,927,2007-09-23,5807.44
Vendor39,935,2007-09-24,2204.33
Vendor39,940,2007-09-25,8419.21
Vendor39,941,2007-09-26,3029.1
Vendor39,946,2007-10-04,2381.06
Vendor39,952,2007-10-05,9163.13
Vendor39,958,2007-10-06,4107.22
Vendor39,964,2007-10-07,3545.31
Vendor39,966,2007-10-07,3532.69
Vendor39,973,2007-10-07,1162.67
Vendor39,978,2007-10-08,2988.79
Vendor39,982,2007-10-08,2737.52
Vendor39,989,2007-10-14,3977.47
Vendor39,993,2007-10-22,1138.51
Vendor39,997,2007-10-30,2317.77
Vendor39,1001,2007-10-30,1352.61
Vendor39,1008,2007-10-30,1888.62
Vendor39,1011,2007-10-30,1196.83
Vendor39,1018,2007-11-05,3341.02
Vendor39,1024,2007-11-13,1667.75
Vendor39,1027,2007-11-14,2611.04
Vendor39,1034,2007-11-14,5105.36
Vendor39,1039,2007-11-14,7272.38
Vendor39,1042,2007-11-14,1382.61
Vendor39,1049,2007-11-14,1214.13
Vendor39,1055,2007-11-20,7962.25
Vendor39,1058,2007-11-21,1559.27
Vendor39,1065,2007-11-27,5062.78
Vendor39,1068,2007-12-05,7344.39
Vendor39,1074,2007-12-05,7543.75
Vendor39,1076,2007-12-06,4368.75
Vendor39,1081,2007-12-07,5049.1
Vendor39,1086,2007-12-07,3845.28
Vendor39,1095,2007-12-07,8255.74
Vendor39,1100,2007-12-17,1933.9
Vendor39,1104,2007-12-17,6075.66
Vendor39,1106,2007-12-23,1999.91
Vendor39,1112,2008-01-02,4258.34
Vendor39,1117,2008-01-02,2470.95
Vendor39,1122,2008-01-03,1414.3
Vendor39,1127,2008-01-13,2039.21
Vendor39,1133,2008-01-13,6725.3
Vendor39,1138,2008-01-14,1021.66
Vendor39,1141,2008-01-15,2442.07
Vendor39,1147,2008-01-15,1689.58
Vendor39,1154,2008-01-25,1645.14
Vendor39,1156,2008-01-31,5145.65
Vendor39,1162,2008-02-10,1924.11
Vendor39,1170,2008-02-20,2175.35
Vendor39,1171,2008-02-28,5065.31
Vendor39,1176,2008-02-29,1235.14
Vendor39,1184,2008-03-01,2194.51
Vendor39,1187,2008-03-02,1216.79
Vendor39,1192,2008-03-10,3400.47
Vendor39,1200,2008-03-10,2790.11
Vendor39,1201,2008-03-16,5200.73
Vendor39,1210,2008-03-24,3113.47
Vendor39,1215,2008-03-25,2881.77
Vendor39,1219,2008-03-26,6052.07
Vendor39,1221,2008-03-26,4698.46
Vendor39,1227,2008-04-05,2047.06
Vendor39,1231,2008-04-05,4284.75
Vendor39,1239,2008-04-11,1076.0
Vendor39,1243,2008-04-19,5260.84
Vendor39,1250,2008-04-27,1589.74
Vendor39,1254,2008-05-05,1249.88
Vendor39,1260,2008-05-11,5382.7
Vendor39,1265,2008-05-12,7424.4
Vendor39,1269,2008-05-18,2555.67
Vendor39,1274,2008-05-28,2288.87
Vendor39,1279,2008-05-29,6322.38
Vendor39,1282,2008-05-29,4026.41
Vendor39,1286,2008-05-29,2509.99
Vendor39,1291,2008-05-29,1200.88
Vendor39,1297,2008-06-04,6310.08
Vendor39,1303,2008-06-04,6841.13
Vendor39,1310,2008-06-04,3860.15
Vendor39,1314,2008-06-04,2697.86
Vendor39,1316,2008-06-04,2462.86
Vendor39,1324,2008-06-10,7702.15
Vendor39,1330,2008-06-11,2184.51
Vendor39,1333,2008-06-12,1055.24
Vendor39,1337,2008-06-18,4514.91
Vendor39,1343,2008-06-24,6774.91
Vendor39,1349,2008-06-30,1731.5
Vendor39,1353,2008-06-30,1504.82
Vendor39,1357,2008-07-01,1800.78
Vendor39,1364,2008-07-02,2126.13
Vendor39,1366,2008-07-02,9272.97
Vendor39,1374,2008-07-03,7114.63
Vendor39,1377,2008-07-04,5777.03
Vendor39,1383,2008-07-12,4674.85
Vendor39,1387,2008-07-12,2616.45
Vendor39,1393,2008-07-18,1781.95
Vendor39,1400,2008-07-28,1463.22
Vendor39,1404,2008-07-28,1845.79
Vendor39,1410,2008-08-05,1816.62
Vendor39,1414,2008-08-06,1451.32
Vendor39,1416,2008-08-06,1691.82
Vendor39,1421,2008-08-12,5529.26
Vendor39,1429,2008-08-12,3986.14
Vendor39,1435,2008-08-12,4871.27
Vendor39,1439,2008-08-12,1668.29
Vendor39,1442,2008-08-22,9538.67
Vendor39,1449,2008-08-22,1932.8
Vendor39,1451,2008-09-01,6734.6
Vendor39,1459,2008-09-01,5593.22
Vendor39,1465,2008-09-01,8072.27
Vendor39,1469,2008-09-01,1512.71
Vendor39,1471,2008-09-01,5307.89
Vendor39,1479,2008-09-01,8100.65
Vendor39,1481,2008-09-11,1342.56
Vendor39,1489,2008-09-17,2125.93
Vendor39,1494,2008-09-23,8075.94
Vendor39,1498,2008-09-23,1179.53
Vendor39,1504,2008-09-23,2748.54
Vendor39,1506,2008-09-23,5730.81
Vendor39,1512,2008-09-24,8745.55
Vendor39,1518,2008-09-24,4936.15
Vendor39,1525,2008-09-25,3342.33
Vendor39,1530,2008-09-26,1517.14
Vendor39,1534,2008-10-02,2159.37
Vendor39,1538,2008-10-02,6284.74
Vendor39,1544,2008-10-02,5461.45
Vendor39,1547,2008-10-02,3231.11
Vendor39,1551,2008-10-02,5215.6
Vendor39,1558,2008-10-08,6847.58
Vendor39,1561,2008-10-08,6163.57
Vendor39,1569,2008-10-08,8089.76
Vendor39,1571,2008-10-09,8046.4
Vendor39,1577,2008-10-15,6018.32
Vendor39,1581,2008-10-21,1812.91
Vendor39,1588,2008-10-31,3538.87
Vendor39,1594,2008-11-06,5966.13
Vendor39,1597,2008-11-07,1225.46
Vendor39,1603,2008-11-08,8762.99
Vendor39,1606,2008-11-18,2825.09
Vendor39,1615,2008-11-19,1456.54
Vendor39,1618,2008-11-29,1851.18
Vendor39,1621,2008-12-07,1486.57
Vendor39,1627,2008-12-13,2532.05
Vendor39,1633,2008-12-19,3726.45
Vendor39,1640,2008-12-20,5152.66
Vendor39,1641,2008-12-21,1351.18
Vendor39,1647,2008-12-22,2524.05
Vendor39,1655,2009-01-01,8787.68
Vendor39,1656,2009-01-01,6815.33
Vendor39,1663,2009-01-07,1169.16
Vendor39,1670,2009-01-07,1152.95
Vendor39,1671,2009-01-13,1931.08
Vendor39,1677,2009-01-14,9840.1
Vendor39,1682,2009-01-14,7136.75
Vendor39,1690,2009-01-15,6296.45
Vendor39,1695,2009-01-15,5160.64
Vendor39,1700,2009-01-15,1312.31
Vendor39,1701,2009-01-25,3031.97
Vendor39,1710,2009-01-25,3396.85
Vendor39,1711,2009-01-26,9878.24
Vendor39,1716,2009-01-27,5615.63
Vendor39,1721,2009-02-06,4454.32
Vendor39,1726,2009-02-06,7482.43
Vendor39,1734,2009-02-06,2991.08
Vendor39,1736,2009-02-06,2534.04
Vendor39,1743,2009-02-07,1102.07
Vendor39,1748,2009-02-07,8325.97
Vendor39,1753,2009-02-13,4061.29
Vendor39,1757,2009-02-19,8663.69
Vendor39,1763,2009-02-20,2751.28
Vendor39,1767,2009-02-21,1338.93
Vendor39,1775,2009-03-01,1102.98
Vendor39,1779,2009-03-01,1025.22
Vendor39,1784,2009-03-09,2331.23
Vendor39,1786,2009-03-19,3924.59
Vendor39,1791,2009-03-25,6204.29
Vendor39,1799,2009-03-26,9645.44
Vendor39,1803,2009-03-26,4585.86
Vendor39,1806,2009-04-05,7709.22
Vendor39,1812,2009-04-05,1072.29
Vendor39,1816,2009-04-15,4982.07
Vendor39,1821,2009-04-16,3464.36
Vendor39,1829,2009-04-17,1560.99
Vendor39,1834,2009-04-17,2391.0
Vendor39,1839,2009-04-17,4031.51
Vendor39,1845,2009-04-18,1565.91
Vendor39,1847,2009-04-24,4420.95
Vendor39,1851,2009-04-25,8167.83
Vendor39,1857,2009-04-26,6605.25
Vendor39,1862,2009-05-06,6709.38
Vendor39,1866,2009-05-16,1815.95
Vendor39,1872,2009-05-22,1659.27
Vendor39,1878,2009-05-28,4383.16
Vendor39,1881,2009-05-28,1383.53
Vendor39,1887,2009-05-29,6264.3
Vendor39,1893,2009-05-30,1599.69
Vendor39,1896,2009-06-05,1549.24
Vendor39,1901,2009-06-05,1295.97
Vendor39,1906,2009-06-06,1360.84
Vendor39,1911,2009-06-12,2424.62
Vendor39,1918,2009-06-22,1212.17
Vendor39,1921,2009-06-30,7053.55
Vendor39,1928,2009-07-06,6268.75
Vendor39,1935,2009-07-06,1363.04
Vendor39,1937,2009-07-12,1560.95
Vendor39,1943,2009-07-18,2747.47
Vendor39,1950,2009-07-24,3142.98
Vendor39,1951,2009-07-25,1108.16
Vendor39,1956,2009-07-25,2897.48
Vendor39,1964,2009-07-25,1407.64
Vendor39,1967,2009-07-31,7462.78
Vendor39,1975,2009-08-08,6250.56
Vendor39,1980,2009-08-14,6005.24
Vendor39,1983,2009-08-15,1329.29
Vendor39,1986,2009-08-23,2853.13
Vendor39,1995,2009-08-31,3153.62
Vendor39,1996,2009-09-06,5738.39
Vendor39,2004,2009-09-12,4762.64
Vendor39,2008,2009-09-12,1303.01
Vendor39,2012,2009-09-22,1881.1
Vendor39,2017,2009-09-28,1514.25
Vendor39,2022,2009-09-28,3424.17
Vendor39,2028,2009-09-28,1899.76
Vendor39,2033,2009-10-06,7984.25
Vendor39,2040,2009-10-06,1383.21
Vendor39,2043,2009-10-16,4962.14
Vendor39,2049,2009-10-22,6241.95
Vendor39,2055,2009-10-28,1786.21
Vendor39,2060,2009-10-28,1882.72
Vendor39,2062,2009-10-28,2591.22
Vendor39,2067,2009-10-28,3735.93
Vendor39,2073,2009-11-03,1665.21
Vendor39,2077,2009-11-09,3499.36
Vendor39,2083,2009-11-19,6503.54
Vendor39,2088,2009-11-20,1427.17
Vendor39,2094,2009-11-21,1384.92
Vendor39,2097,2009-12-01,7449.22
Vendor39,2104,2009-12-09,7654.48
Vendor39,2106,2009-12-10,5521.5
Vendor39,2113,2009-12-10,1542.39
Vendor39,2117,2009-12-16,8009.13
Vendor39,2123,2009-12-17,2049.54
Vendor39,2130,2009-12-18,1712.5
Vendor39,2135,2009-12-28,1806.6
Vendor39,2137,2009-12-28,3458.35
Vendor39,2141,2009-12-28,3155.49
Vendor39,2148,2009-12-29,1977.27
Vendor39,2154,2010-01-04,2398.68
Vendor39,2160,2010-01-10,2504.34
Vendor39,2165,2010-01-10,1875.97
Vendor39,2169,2010-01-11,8307.26
Vendor39,2174,2010-01-11,1647.07
Vendor39,2176,2010-01-12,1138.21
Vendor39,2183,2010-01-12,5751.43
Vendor39,2189,2010-01-20,3850.97
Vendor39,2193,2010-01-26,1361.31
Vendor39,2199,2010-01-27,2054.98
Vendor39,2201,2010-02-04,2417.98
Vendor39,2206,2010-02-14,4014.25
Vendor39,2214,2010-02-15,2699.25
Vendor39,2219,2010-02-15,1178.88
Vendor39,2221,2010-02-15,3313.8
Vendor39,2228,2010-02-16,1749.11
Vendor39,2231,2010-02-22,2396.24
Vendor39,2238,2010-02-22,4581.23
Vendor39,2243,2010-02-23,1687.8
Vendor39,2247,2010-02-24,2156.75
Vendor39,2254,2010-02-25,3704.84
Vendor39,2256,2010-02-26,1423.34
Vendor39,2261,2010-02-27,9594.89
Vendor39,2269,2010-02-28,4752.53
Vendor39,2275,2010-03-01,3185.86
Vendor39,2276,2010-03-07,2095.77
Vendor39,2284,2010-03-13,6015.03
Vendor39,2288,2010-03-19,8767.29
Vendor39,2295,2010-03-20,8315.45
Vendor39,2297,2010-03-26,1750.58
Vendor39,2304,2010-03-26,5090.45
Vendor39,2309,2010-04-01,4129.85
Vendor39,2311,2010-04-09,4182.53
Vendor39,2320,2010-04-15,1049.3
Vendor39,2324,2010-04-23,4455.74
Vendor39,2328,2010-04-29,2105.4
Vendor39,2335,2010-04-29,1711.98
Vendor39,2340,2010-04-30,3408.37
Vendor39,2344,2010-05-01,3734.95
Vendor39,2348,2010-05-02,1285.89
Vendor39,2353,2010-05-03,8655.51
Vendor39,2358,2010-05-03,1416.41
Vendor39,2364,2010-05-09,2446.56
Vendor39,2367,2010-05-10,2311.41
Vendor39,2374,2010-05-16,1321.88
Vendor39,2378,2010-05-22,4874.86
Vendor39,2381,2010-06-01,1349.38
Vendor39,2388,2010-06-11,6099.34
Vendor39,2395,2010-06-11,5809.12
Vendor39,2396,2010-06-11,1278.05
Vendor39,2403,2010-06-12,8015.51
Vendor39,2409,2010-06-18,6702.64
Vendor39,2414,2010-06-18,2603.71
Vendor39,2420,2010-06-24,7685.91
Vendor39,2423,2010-06-24,2822.54
Vendor39,2427,2010-06-24,9295.81
Vendor39,2435,2010-06-24,5506.93
Vendor39,2438,2010-06-24,7510.38
Vendor39,2443,2010-06-24,2975.93
Vendor39,2446,2010-06-25,1900.76
Vendor39,2455,2010-06-25,1787.55
Vendor39,2460,2010-06-26,1781.16
Vendor39,2463,2010-07-02,3875.76
Vendor39,2466,2010-07-08,9134.01
Vendor39,2474,2010-07-09,6665.53
Vendor39,2478,2010-07-09,1068.58
Vendor39,2482,2010-07-19,1992.86
Vendor39,2487,2010-07-19,6245.86
Vendor39,2493,2010-07-19,2972.44
Vendor39,2497,2010-07-29,1737.77
Vendor39,2501,2010-07-29,4127.96
Vendor39,2506,2010-07-29,1332.92
Vendor39,2515,2010-07-30,1642.86
Vendor39,2517,2010-07-31,6670.67
Vendor39,2524,2010-08-08,1077.98
Vendor39,2530,2010-08-09,2157.18
Vendor39,2532,2010-08-15,1144.64
Vendor39,2539,2010-08-16,3607.94
Vendor39,2543,2010-08-17,2132.39
Vendor39,2550,2010-08-25,1400.91
Vendor39,2555,2010-08-25,4517.89
Vendor39,2559,2010-09-02,3540.78
Vendor39,2562,2010-09-12,1057.65
Vendor39,2568,2010-09-12,7155.76
Vendor39,2574,2010-09-18,3535.7
Vendor39,2579,2010-09-26,4269.58
Vendor39,2584,2010-09-26,4329.7
Vendor39,2590,2010-09-26,1600.84
Vendor39,2591,2010-09-27,5678.49
Vendor39,2598,2010-10-05,5514.66
Vendor39,2604,2010-10-05,1933.98
Vendor39,2609,2010-10-11,8804.75
Vendor39,2612,2010-10-12,1270.4
Vendor39,2618,2010-10-22,3062.02
Vendor39,2624,2010-10-22,2735.84
Vendor39,2626,2010-10-22,4101.99
Vendor39,2632,2010-10-28,3431.25
Vendor39,2636,2010-10-29,4570.27
Vendor39,2642,2010-10-29,9220.62
Vendor39,2647,2010-10-30,2017.03
Vendor39,2654,2010-10-31,4853.15
Vendor39,2658,2010-11-10,7544.38
Vendor39,2661,2010-11-20,1205.31
Vendor39,2668,2010-11-21,8159.32
Vendor39,2673,2010-11-22,4447.34
Vendor39,2676,2010-11-23,1350.84
Vendor39,2682,2010-11-24,1047.64
Vendor39,2688,2010-11-25,6087.99
Vendor39,2691,2010-12-01,6583.89
Vendor39,2700,2010-12-01,3577.96
Vendor39,2701,2010-12-01,5625.87
Vendor39,2708,2010-12-02,3037.57
Vendor39,2713,2010-12-03,2316.79
Vendor39,2718,2010-12-11,4554.54
Vendor39,2722,2010-12-17,1253.55
Vendor39,2727,2010-12-27,1630.67
Vendor39,2731,2010-12-27,3964.95
Vendor39,2737,2010-12-27,2818.75
Vendor39,2743,2011-01-02,5084.01
Vendor39,2747,2011-01-03,1707.11
Vendor39,2752,2011-01-09,5248.02
Vendor39,2759,2011-01-10,8164.05
Vendor39,2763,2011-01-10,2682.43
Vendor39,2768,2011-01-18,1060.92
Vendor39,2771,2011-01-19,2857.09
Vendor39,2780,2011-01-19,6600.9
Vendor39,2785,2011-01-27,1512.22
Vendor39,2787,2011-01-27,1063.78
Vendor39,2792,2011-01-27,6462.78
Vendor39,2797,2011-01-27,1127.51
Vendor39,2801,2011-01-28,2147.4
Vendor39,2807,2011-01-29,9022.94
Vendor39,2813,2011-02-04,2428.14
Vendor39,2816,2011-02-12,1367.5
Vendor39,2824,2011-02-20,5138.46
Vendor39,2828,2011-02-20,4590.3
Vendor39,2831,2011-02-20,8698.93
Vendor39,2837,2011-03-02,7403.71
Vendor39,2845,2011-03-08,4475.89
Vendor39,2850,2011-03-08,9182.18
Vendor39,2854,2011-03-16,6649.05
Vendor39,2856,2011-03-22,1194.15
Vendor39,2865,2011-03-28,1424.06
Vendor39,2868,2011-03-29,9222.77
Vendor39,2871,2011-03-29,1298.05
Vendor39,2878,2011-04-04,3749.61
Vendor39,2882,2011-04-05,1949.17
Vendor39,2886,2011-04-05,1689.76
Vendor39,2892,2011-04-05,6494.3
Vendor39,2899,2011-04-11,7284.44
Vendor39,2901,2011-04-21,2651.17
Vendor39,2908,2011-04-27,1032.24
Vendor39,2914,2011-04-28,4776.92
Vendor39,2917,2011-05-04,5880.74
Vendor39,2924,2011-05-04,2223.56
Vendor39,2926,2011-05-14,1784.7
Vendor39,2931,2011-05-14,1823.44
Vendor39,2937,2011-05-14,2227.31
Vendor39,2944,2011-05-14,4513.5
Vendor39,2948,2011-05-15,2131.94
Vendor39,2955,2011-05-21,9302.66
Vendor39,2956,2011-05-27,9136.91
Vendor39,2962,2011-05-27,2724.65
Vendor39,2966,2011-05-27,3643.32
Vendor39,2972,2011-06-02,1087.77
Vendor39,2978,2011-06-12,1510.55
Vendor39,2981,2011-06-12,3332.69
Vendor39,2986,2011-06-12,3593.89
Vendor39,2994,2011-06-22,3083.11
Vendor39,2999,2011-06-28,6334.13
Vendor39,3001,2011-06-29,1123.84
Vendor39,3008,2011-06-30,1925.05
Vendor39,3012,2011-07-06,4403.95
Vendor39,3018,2011-07-06,8305.12
Vendor39,3022,2011-07-14,9965.84
Vendor39,3026,2011-07-22,4568.05
Vendor39,3031,2011-07-23,6500.21
Vendor39,3039,2011-07-24,1330.11
Vendor39,3043,2011-07-24,9585.88
Vendor39,3050,2011-07-25,2448.91
Vendor39,3053,2011-07-25,3522.1
Vendor39,3059,2011-07-31,3561.29
Vendor39,3064,2011-07-31,1325.99
Vendor39,3066,2011-08-10,4001.6
Vendor39,3071,2011-08-11,2129.2
Vendor39,3078,2011-08-11,1815.47
Vendor39,3081,2011-08-11,1624.49
Vendor39,3088,2011-08-19,3632.45
Vendor39,3092,2011-08-29,1616.37
Vendor39,3098,2011-08-30,8011.36
Vendor39,3104,2011-09-05,4698.49
Vendor39,3106,2011-09-06,1285.6
Vendor39,3111,2011-09-14,4361.76
Vendor39,3118,2011-09-14,9239.56
Vendor39,3121,2011-09-20,2517.19
Vendor39,3129,2011-09-28,1833.48
Vendor39,3135,2011-09-29,1022.78
Vendor39,3140,2011-09-30,2447.78
Vendor39,3145,2011-10-06,3326.41
Vendor39,3147,2011-10-07,7869.27
Vendor39,3152,2011-10-17,1370.59
Vendor39,3156,2011-10-18,1181.78
Vendor39,3162,2011-10-26,4496.94
Vendor39,3168,2011-10-26,2253.93
Vendor39,3171,2011-10-27,1690.58
Vendor39,3177,2011-11-02,2857.41
Vendor39,3182,2011-11-03,1430.49
Vendor39,3187,2011-11-09,2159.75
Vendor39,3193,2011-11-19,6722.01
Vendor39,3200,2011-11-20,9849.4
Vendor39,3205,2011-11-20,1379.3
Vendor39,3207,2011-11-21,7541.48
Vendor39,3215,2011-11-22,1032.3
Vendor39,3217,2011-12-02,2820.37
Vendor39,3222,2011-12-02,3185.15
Vendor39,3229,2011-12-03,5637.41
Vendor39,3232,2011-12-04,1910.36
Vendor39,3237,2011-12-14,3086.16
Vendor39,3241,2011-12-22,2072.26
Vendor39,3246,2011-12-23,3176.19
Vendor39,3251,2011-12-23,1316.89
Vendor39,3257,2011-12-23,2988.56
Vendor39,3262,2011-12-23,1154.51
Vendor39,3270,2011-12-29,1749.41
Vendor39,3273,2011-12-29,3353.79
Vendor39,3277,2011-12-29,3029.88
Vendor39,3284,2011-12-29,2695.8
Vendor39,3289,2012-01-04,2675.79
Vendor39,3295,2012-01-05,7493.25
Vendor39,3299,2012-01-06,2863.71
Vendor39,3304,2012-01-06,3965.61
Vendor39,3309,2012-01-12,2537.59
Vendor39,3315,2012-01-22,2281.44
Vendor39,3317,2012-01-28,1614.76
Vendor39,3323,2012-01-28,8099.14
Vendor39,3329,2012-01-29,1682.04
Vendor39,3332,2012-01-29,3420.67
Vendor39,3338,2012-02-04,8088.94
Vendor39,3345,2012-02-05,6929.42
Vendor39,3350,2012-02-05,2490.86
Vendor39,3354,2012-02-05,1193.3
Vendor39,3356,2012-02-05,9076.35
Vendor39,3362,2012-02-13,3497.89
Vendor39,3369,2012-02-19,1958.1
Vendor39,3373,2012-02-19,3229.34
Vendor39,3377,2012-02-20,3998.93
Vendor39,3385,2012-02-26,3882.06
Vendor39,3386,2012-02-26,2529.95
Vendor39,3395,2012-02-26,3877.14
Vendor39,3399,2012-02-27,9827.43
Vendor39,3401,2012-03-08,6550.75
Vendor39,3409,2012-03-14,9219.36
Vendor39,3411,2012-03-15,1557.31
Vendor39,3417,2012-03-16,7051.37
Vendor39,3425,2012-03-26,4301.31
Vendor39,3430,2012-03-26,2394.02
Vendor39,3431,2012-04-03,1492.56
Vendor39,3440,2012-04-03,4803.33
Vendor39,3441,2012-04-03,5066.84
Vendor39,3447,2012-04-09,2692.85
Vendor39,3452,2012-04-10,7753.7
Vendor39,3458,2012-04-16,3784.16
Vendor39,3462,2012-04-17,1877.76
Vendor39,3468,2012-04-17,2416.83
Vendor39,3473,2012-04-25,3824.94
Vendor39,3479,2012-04-26,3710.82
Vendor39,3483,2012-04-26,4609.89
Vendor39,3488,2012-04-26,3311.85
Vendor39,3491,2012-04-27,1749.03
Vendor39,3498,2012-04-28,1443.93
Vendor39,3504,2012-04-28,1470.1
Vendor39,3506,2012-04-29,8275.35
Vendor39,3512,2012-05-05,7042.56
Vendor39,3517,2012-05-11,1840.82
Vendor39,3524,2012-05-11,1769.25
Vendor39,3528,2012-05-12,1054.18
Vendor39,3533,2012-05-22,2194.3
Vendor39,3538,2012-05-23,2643.23
Vendor39,3544,2012-05-24,2259.21
Vendor39,3550,2012-05-30,1247.16
Vendor39,3553,2012-05-30,6870.08
Vendor39,3559,2012-06-05,2713.6
Vendor39,3565,2012-06-05,1451.54
Vendor39,3569,2012-06-05,8165.27
Vendor39,3571,2012-06-05,2560.4
Vendor39,3579,2012-06-11,2220.03
Vendor39,3583,2012-06-12,2150.72
Vendor39,3590,2012-06-12,2985.36
Vendor39,3593,2012-06-12,5687.44
Vendor39,3597,2012-06-18,5290.97
Vendor39,3605,2012-06-19,1223.28
Vendor39,3607,2012-06-25,4380.85
Vendor39,3613,2012-06-26,3539.92
Vendor39,3617,2012-07-06,8477.65
Vendor39,3622,2012-07-06,1648.4
Vendor39,3630,2012-07-07,3352.36
Vendor39,3631,2012-07-15,1485.06
Vendor39,3636,2012-07-21,2662.24
Vendor39,3641,2012-07-22,1943.78
Vendor42,7,2006-01-01,7160.76
Vendor42,11,2006-01-11,1652.91
Vendor42,19,2006-01-17,3552.17
Vendor42,24,2006-01-27,1641.08
Vendor42,28,2006-01-27,4767.16
Vendor42,32,2006-01-28,3231.91
Vendor42,38,2006-02-03,9512.44
Vendor42,42,2006-02-13,2826.12
Vendor42,50,2006-02-19,1718.99
Vendor42,53,2006-02-20,2884.79
Vendor42,58,2006-02-20,8558.68
Vendor42,61,2006-02-21,4285.22
Vendor42,68,2006-02-21,4792.16
Vendor42,71,2006-02-27,6116.07
Vendor42,76,2006-02-28,2797.57
Vendor42,82,2006-03-10,2716.59
Vendor42,90,2006-03-11,2318.36
Vendor42,95,2006-03-21,2912.11
Vendor42,100,2006-03-27,1400.32
Vendor42,103,2006-03-27,8025.69
Vendor42,106,2006-04-06,2201.0
Vendor42,111,2006-04-07,1548.43
Vendor42,116,2006-04-07,3338.08
Vendor42,124,2006-04-07,1128.55
Vendor42,127,2006-04-08,3100.14
Vendor42,134,2006-04-08,1475.81
Vendor42,136,2006-04-14,1096.2
Vendor42,142,2006-04-15,2005.01
Vendor42,147,2006-04-21,2604.92
Vendor42,152,2006-04-22,8325.02
Vendor42,157,2006-04-22,4414.6
Vendor42,164,2006-04-30,5293.9
Vendor42,166,2006-05-01,2007.34
Vendor42,171,2006-05-09,1758.12
Vendor42,176,2006-05-09,7602.62
Vendor42,183,2006-05-15,4055.69
Vendor42,188,2006-05-23,2949.6
Vendor42,192,2006-05-29,1070.02
Vendor42,200,2006-06-06,2502.47
Vendor42,205,2006-06-07,1284.3
Vendor42,207,2006-06-08,4290.99
Vendor42,211,2006-06-08,6632.85
Vendor42,219,2006-06-09,8695.98
Vendor42,221,2006-06-09,2923.07
Vendor42,230,2006-06-09,1542.68
Vendor42,235,2006-06-10,9923.25
Vendor42,236,2006-06-10,4184.4
Vendor42,245,2006-06-11,1804.72
Vendor42,250,2006-06-17,2411.62
Vendor42,255,2006-06-23,7206.02
Vendor42,260,2006-06-23,5318.74
Vendor42,264,2006-06-24,6074.94
Vendor42,268,2006-06-30,3266.94
Vendor42,275,2006-06-30,4953.92
Vendor42,277,2006-06-30,2712.58
Vendor42,284,2006-07-01,2809.68
Vendor42,288,2006-07-01,3761.27
Vendor42,292,2006-07-07,4395.65
Vendor42,297,2006-07-17,4192.96
Vendor42,305,2006-07-23,5085.99
Vendor42,307,2006-07-23,8203.6
Vendor42,313,2006-07-29,1474.25
Vendor42,318,2006-07-30,2281.29
Vendor42,322,2006-08-07,3308.89
Vendor42,330,2006-08-15,1462.93
Vendor42,331,2006-08-15,3010.84
Vendor42,339,2006-08-21,5050.2
Vendor42,341,2006-08-27,5971.08
Vendor42,347,2006-08-28,3219.78
Vendor42,352,2006-08-28,2452.82
Vendor42,358,2006-09-03,7724.6
Vendor42,361,2006-09-13,5458.9
Vendor42,370,2006-09-13,1106.54
Vendor42,374,2006-09-14,1436.89
Vendor42,378,2006-09-15,2776.99
Vendor42,385,2006-09-23,3283.18
Vendor42,388,2006-09-24,1530.64
Vendor42,395,2006-09-24,1542.11
Vendor42,396,2006-09-25,7391.78
Vendor42,405,2006-10-03,2003.58
Vendor42,407,2006-10-03,1073.55
Vendor42,413,2006-10-04,5350.4
Vendor42,418,2006-10-10,1572.57
Vendor42,424,2006-10-10,3658.36
Vendor42,428,2006-10-11,1223.52
Vendor42,435,2006-10-11,8028.07
Vendor42,440,2006-10-11,4341.32
Vendor42,445,2006-10-11,1799.35
Vendor42,447,2006-10-12,4556.27
Vendor42,452,2006-10-13,6552.19
Vendor42,459,2006-10-14,4566.14
Vendor42,463,2006-10-22,2010.08
Vendor42,467,2006-10-23,4292.1
Vendor42,475,2006-10-29,6279.54
Vendor42,478,2006-10-30,1519.83
Vendor42,482,2006-11-09,1476.71
Vendor42,489,2006-11-10,9278.41
Vendor42,492,2006-11-18,9306.61
Vendor42,498,2006-11-28,5449.62
Vendor42,502,2006-12-08,3881.46
Vendor42,509,2006-12-16,3833.07
Vendor42,515,2006-12-17,8515.11
Vendor42,517,2006-12-17,1332.8
Vendor42,523,2006-12-23,1817.51
Vendor42,529,2006-12-29,4779.53
Vendor42,531,2006-12-30,8857.22
Vendor42,540,2006-12-30,2456.09
Vendor42,543,2007-01-05,4423.0
Vendor42,548,2007-01-06,2732.7
Vendor42,554,2007-01-07,3864.61
Vendor42,557,2007-01-15,8751.9
Vendor42,564,2007-01-15,1978.86
Vendor42,566,2007-01-16,5280.93
Vendor42,572,2007-01-16,1207.1
Vendor42,578,2007-01-22,3490.71
Vendor42,583,2007-01-22,2879.96
Vendor42,589,2007-01-23,2741.86
Vendor42,591,2007-01-24,1430.14
Vendor42,600,2007-01-25,2174.95
Vendor42,602,2007-01-25,6236.34
Vendor42,606,2007-01-26,5893.33
Vendor42,613,2007-01-27,2981.54
Vendor42,616,2007-02-06,1485.32
Vendor42,621,2007-02-06,3120.7
Vendor42,628,2007-02-07,4929.93
Vendor42,632,2007-02-07,8471.69
Vendor42,640,2007-02-15,1599.81
Vendor42,644,2007-02-21,4028.36
Vendor42,646,2007-02-27,1106.37
Vendor42,653,2007-03-07,5009.85
Vendor42,659,2007-03-08,5143.05
Vendor42,663,2007-03-09,1585.39
Vendor42,669,2007-03-09,5296.67
Vendor42,672,2007-03-09,3437.3
Vendor42,678,2007-03-15,3325.46
Vendor42,683,2007-03-16,1469.48
Vendor42,687,2007-03-22,4000.2
Vendor42,693,2007-03-23,9822.71
Vendor42,697,2007-03-23,3565.83
Vendor42,705,2007-03-29,3422.83
Vendor42,708,2007-03-29,2169.03
Vendor42,713,2007-03-30,3207.51
Vendor42,720,2007-03-31,3337.57
Vendor42,725,2007-04-01,2026.0
Vendor42,726,2007-04-07,1301.89
Vendor42,733,2007-04-13,2329.08
Vendor42,737,2007-04-19,2494.56
Vendor42,743,2007-04-20,9957.34
Vendor42,750,2007-04-26,3173.22
Vendor42,751,2007-04-26,8403.08
Vendor42,760,2007-04-26,1195.9
Vendor42,762,2007-05-06,3956.2
Vendor42,768,2007-05-07,5522.69
Vendor42,772,2007-05-07,2963.58
Vendor42,777,2007-05-07,1599.09
Vendor42,781,2007-05-07,3356.47
Vendor42,787,2007-05-08,2199.04
Vendor42,793,2007-05-14,1876.05
Vendor42,796,2007-05-15,8913.19
Vendor42,801,2007-05-15,1198.85
Vendor42,806,2007-05-16,6150.96
Vendor42,813,2007-05-16,7561.69
Vendor42,819,2007-05-26,4070.07
Vendor42,825,2007-05-27,8794.13
Vendor42,827,2007-06-04,6974.31
Vendor42,833,2007-06-05,1656.12
Vendor42,836,2007-06-05,3183.42
Vendor42,842,2007-06-06,4832.84
Vendor42,848,2007-06-07,2158.74
Vendor42,852,2007-06-13,6467.33
Vendor42,858,2007-06-21,4029.17
Vendor42,865,2007-06-27,6603.68
Vendor42,867,2007-06-27,4080.33
Vendor42,875,2007-06-28,3504.94
Vendor42,879,2007-06-28,7822.89
Vendor42,885,2007-07-04,2878.17
Vendor42,886,2007-07-12,3819.83
Vendor42,891,2007-07-18,4208.36
Vendor42,896,2007-07-26,1406.13
Vendor42,902,2007-08-05,1585.75
Vendor42,908,2007-08-11,9107.58
Vendor42,911,2007-08-12,1844.37
Vendor42,917,2007-08-12,7967.44
Vendor42,921,2007-08-18,1655.42
Vendor42,927,2007-08-24,5161.1
Vendor42,933,2007-08-24,3869.76
Vendor42,936,2007-09-03,6409.85
Vendor42,945,2007-09-11,4045.23
Vendor42,946,2007-09-17,2111.1
Vendor42,951,2007-09-17,4035.69
Vendor42,959,2007-09-17,3913.91
Vendor42,964,2007-09-18,3305.42
Vendor42,970,2007-09-18,1980.69
Vendor42,972,2007-09-18,3312.32
Vendor42,976,2007-09-18,4989.39
Vendor42,981,2007-09-19,2013.92
Vendor42,989,2007-09-19,5709.36
Vendor42,995,2007-09-19,3840.0
Vendor42,996,2007-09-20,8844.4
Vendor42,1004,2007-09-30,6864.53
Vendor42,1010,2007-09-30,2876.59
Vendor42,1015,2007-10-06,5065.19
Vendor42,1020,2007-10-07,2015.07
Vendor42,1023,2007-10-13,1740.32
Vendor42,1029,2007-10-19,6471.91
Vendor42,1034,2007-10-25,3795.92
Vendor42,1038,2007-10-26,1532.02
Vendor42,1045,2007-10-27,2138.13
Vendor42,1046,2007-11-02,1091.54
Vendor42,1052,2007-11-10,4611.64
Vendor42,1059,2007-11-10,2275.89
Vendor42,1062,2007-11-16,4058.62
Vendor42,1069,2007-11-16,9143.91
Vendor42,1074,2007-11-26,5303.6
Vendor42,1076,2007-11-26,3196.69
Vendor42,1083,2007-11-27,3362.5
Vendor42,1089,2007-11-27,1440.51
Vendor42,1092,2007-11-28,2907.06
Vendor42,1099,2007-11-29,2741.48
Vendor42,1101,2007-11-30,2182.8
Vendor42,1108,2007-11-30,5811.11
Vendor42,1114,2007-12-01,2343.88
Vendor42,1118,2007-12-02,1763.14
Vendor42,1124,2007-12-03,4000.88
Vendor42,1126,2007-12-11,1235.69
Vendor42,1135,2007-12-11,8084.08
Vendor42,1139,2007-12-11,3074.62
Vendor42,1145,2007-12-12,1560.55
Vendor42,1148,2007-12-20,9216.53
Vendor42,1152,2007-12-21,1026.55
Vendor42,1160,2007-12-27,1841.47
Vendor42,1164,2007-12-28,7959.38
Vendor42,1166,2007-12-28,5934.72
Vendor42,1172,2007-12-29,5417.44
Vendor42,1177,2007-12-30,2725.47
Vendor42,1184,2007-12-31,9503.82
Vendor42,1186,2008-01-01,3430.89
Vendor42,1195,2008-01-02,4483.87
Vendor42,1196,2008-01-03,1244.12
Vendor42,1201,2008-01-04,2110.45
Vendor42,1209,2008-01-10,8507.8
Vendor42,1213,2008-01-16,1337.64
Vendor42,1218,2008-01-24,5114.76
Vendor42,1223,2008-02-01,5239.37
Vendor42,1229,2008-02-07,4389.28
Vendor42,1233,2008-02-07,3507.41
Vendor42,1240,2008-02-07,1960.02
Vendor42,1245,2008-02-17,4455.7
Vendor42,1248,2008-02-18,1746.14
Vendor42,1251,2008-02-19,1446.04
Vendor42,1258,2008-02-19,1941.25
Vendor42,1264,2008-02-19,1117.88
Vendor42,1267,2008-02-29,5817.07
Vendor42,1274,2008-02-29,2696.77
Vendor42,1278,2008-03-06,1501.62
Vendor42,1282,2008-03-16,1023.4
Vendor42,1288,2008-03-22,5182.79
Vendor42,1295,2008-03-23,3500.62
Vendor42,1297,2008-04-02,2803.31
Vendor42,1304,2008-04-12,1145.59
Vendor42,1310,2008-04-18,1310.17
Vendor42,1312,2008-04-19,6451.51
Vendor42,1319,2008-04-19,8780.94
Vendor42,1324,2008-04-20,7653.8
Vendor42,1326,2008-04-21,6801.1
Vendor42,1332,2008-04-27,1361.94
Vendor42,1339,2008-04-28,5728.08
Vendor42,1342,2008-04-29,3537.0
Vendor42,1348,2008-05-07,3823.39
Vendor42,1355,2008-05-15,2391.89
Vendor42,1358,2008-05-15,7513.95
Vendor42,1361,2008-05-16,8050.79
Vendor42,1369,2008-05-22,4217.97
Vendor42,1372,2008-05-28,5257.4
Vendor42,1376,2008-05-29,5934.62
Vendor42,1382,2008-06-08,2631.91
Vendor42,1389,2008-06-08,4278.04
Vendor42,1395,2008-06-14,2786.99
Vendor42,1398,2008-06-14,9155.63
Vendor42,1404,2008-06-15,1084.35
Vendor42,1406,2008-06-16,2021.01
Vendor42,1413,2008-06-17,1608.61
Vendor42,1418,2008-06-23,9048.19
Vendor42,1425,2008-07-03,5600.73
Vendor42,1430,2008-07-04,6640.55
Vendor42,1435,2008-07-05,4684.65
Vendor42,1439,2008-07-05,4136.42
Vendor42,1441,2008-07-15,6150.48
Vendor42,1450,2008-07-21,1190.83
Vendor42,1451,2008-07-31,4219.19
Vendor42,1457,2008-08-08,4089.59
Vendor42,1461,2008-08-09,1933.15
Vendor42,1467,2008-08-09,2008.96
Vendor42,1473,2008-08-10,8651.89
Vendor42,1476,2008-08-11,9614.32
Vendor42,1482,2008-08-17,3140.31
Vendor42,1489,2008-08-23,3527.75
Vendor42,1491,2008-08-23,2359.85
Vendor42,1496,2008-09-02,6412.32
Vendor42,1504,2008-09-02,1126.26
Vendor42,1508,2008-09-02,1407.52
Vendor42,1513,2008-09-08,1773.58
Vendor42,1517,2008-09-08,5257.18
Vendor42,1521,2008-09-08,4028.47
Vendor42,1529,2008-09-08,3446.38
Vendor42,1534,2008-09-18,3072.51
Vendor42,1540,2008-09-18,1532.1
Vendor42,1544,2008-09-26,4546.36
Vendor42,1548,2008-09-26,6502.55
Vendor42,1553,2008-10-06,5469.71
Vendor42,1560,2008-10-12,5949.78
Vendor42,1561,2008-10-13,5043.45
Vendor42,1569,2008-10-14,9309.14
Vendor42,1572,2008-10-14,1748.25
Vendor42,1580,2008-10-20,3571.79
Vendor42,1582,2008-10-20,1176.17
Vendor42,1587,2008-10-20,4337.26
Vendor42,1591,2008-10-20,1475.37
Vendor42,1599,2008-10-21,1197.0
Vendor42,1602,2008-10-22,4530.88
Vendor42,1607,2008-10-23,1652.24
Vendor42,1612,2008-10-23,5500.03
Vendor42,1620,2008-10-23,4864.83
Vendor42,1624,2008-10-23,1394.0
Vendor42,1629,2008-10-24,2585.68
Vendor42,1634,2008-10-24,2464.78
Vendor42,1639,2008-10-24,1714.68
Vendor42,1643,2008-10-30,1500.06
Vendor42,1649,2008-10-30,3237.87
Vendor42,1651,2008-10-30,1502.79
Vendor42,1656,2008-10-30,4456.94
Vendor42,1662,2008-10-31,2612.17
Vendor42,1666,2008-11-01,1402.06
Vendor42,1673,2008-11-01,4206.78
Vendor42,1679,2008-11-07,4441.92
Vendor42,1684,2008-11-08,1813.11
Vendor42,1688,2008-11-09,2970.7
Vendor42,1693,2008-11-09,5535.27
Vendor42,1699,2008-11-10,1803.67
Vendor42,1704,2008-11-11,6380.17
Vendor42,1708,2008-11-21,8008.77
Vendor42,1712,2008-11-22,3025.9
Vendor42,1718,2008-12-02,1068.11
Vendor42,1724,2008-12-08,4163.61
Vendor42,1727,2008-12-09,2202.24
Vendor42,1733,2008-12-15,1539.4
Vendor42,1736,2008-12-25,4111.98
Vendor42,1744,2008-12-31,2054.42
Vendor42,1748,2009-01-06,7916.13
Vendor42,1753,2009-01-14,5234.89
Vendor42,1756,2009-01-22,2812.22
Vendor42,1765,2009-01-22,3655.27
Vendor42,1770,2009-01-22,2866.64
Vendor42,1774,2009-01-28,2639.36
Vendor42,1779,2009-01-29,1638.63
Vendor42,1782,2009-02-08,5323.21
Vendor42,1788,2009-02-08,6176.92
Vendor42,1792,2009-02-09,1189.37
Vendor42,1798,2009-02-15,2459.69
Vendor42,1805,2009-02-21,5434.18
Vendor42,1806,2009-03-01,3225.51
Vendor42,1814,2009-03-07,3415.8
Vendor42,1817,2009-03-08,6256.7
Vendor42,1824,2009-03-08,1127.24
Vendor42,1826,2009-03-08,3381.68
Vendor42,1832,2009-03-08,1267.61
Vendor42,1840,2009-03-09,6298.61
Vendor42,1845,2009-03-09,1005.48
Vendor42,1850,2009-03-10,1439.4
Vendor42,1851,2009-03-11,4207.9
Vendor42,1859,2009-03-17,9623.14
Vendor42,1863,2009-03-18,1044.08
Vendor42,1869,2009-03-28,9312.89
Vendor42,1872,2009-03-28,6714.87
Vendor42,1880,2009-03-29,4619.08
Vendor42,1884,2009-04-04,5815.56
Vendor42,1890,2009-04-05,1097.9
Vendor42,1894,2009-04-11,2036.56
Vendor42,1900,2009-04-12,3074.04
Vendor42,1901,2009-04-13,1385.57
Vendor42,1907,2009-04-13,7084.69
Vendor42,1911,2009-04-14,1724.69
Vendor42,1917,2009-04-20,4798.59
Vendor42,1921,2009-04-26,8343.76
Vendor42,1927,2009-05-06,7253.31
Vendor42,1931,2009-05-07,1958.01
Vendor42,1936,2009-05-15,3690.6
Vendor42,1941,2009-05-15,4440.69
Vendor42,1947,2009-05-21,4590.37
Vendor42,1955,2009-05-29,2230.5
Vendor42,1956,2009-05-29,8195.79
Vendor42,1964,2009-06-08,1788.1
Vendor42,1968,2009-06-09,1724.18
Vendor42,1974,2009-06-10,1360.66
Vendor42,1978,2009-06-10,4826.9
Vendor42,1982,2009-06-11,5874.25
Vendor42,1986,2009-06-11,1158.17
Vendor42,1995,2009-06-12,1042.35
Vendor42,2000,2009-06-12,6219.93
Vendor42,2003,2009-06-20,3698.75
Vendor42,2009,2009-06-26,8846.62
Vendor42,2011,2009-06-26,4107.2
Vendor42,2018,2009-07-06,2137.01
Vendor42,2024,2009-07-06,2691.33
Vendor42,2027,2009-07-06,1393.33
Vendor42,2034,2009-07-07,1655.71
Vendor42,2038,2009-07-07,1023.0
Vendor42,2042,2009-07-07,2475.36
Vendor42,2049,2009-07-07,2268.8
Vendor42,2052,2009-07-07,1728.01
Vendor42,2059,2009-07-13,8470.22
Vendor42,2065,2009-07-19,1536.79
Vendor42,2068,2009-07-25,5063.93
Vendor42,2073,2009-07-26,6368.43
Vendor42,2076,2009-07-27,2240.05
Vendor42,2083,2009-08-06,2800.02
Vendor42,2087,2009-08-14,8692.99
Vendor42,2095,2009-08-14,1398.2
Vendor42,2096,2009-08-20,3057.96
Vendor42,2103,2009-08-21,8342.58
Vendor42,2109,2009-08-21,3360.43
Vendor42,2114,2009-08-22,1645.25
Vendor42,2116,2009-08-22,3257.73
Vendor42,2125,2009-08-28,3809.35
Vendor42,2130,2009-09-05,3952.17
Vendor42,2135,2009-09-11,2210.47
Vendor42,2137,2009-09-11,4446.97
Vendor42,2142,2009-09-11,4789.64
Vendor42,2147,2009-09-12,5915.12
Vendor42,2152,2009-09-18,4922.06
Vendor42,2158,2009-09-19,1045.37
Vendor42,2162,2009-09-19,2932.17
Vendor42,2170,2009-09-25,6842.03
Vendor42,2172,2009-10-03,2013.22
Vendor42,2179,2009-10-03,1710.76
Vendor42,2183,2009-10-03,9987.11
Vendor42,2189,2009-10-11,1868.68
Vendor42,2191,2009-10-21,7720.8
Vendor42,2197,2009-10-31,5669.32
Vendor42,2202,2009-11-10,1673.46
Vendor42,2210,2009-11-11,7034.4
Vendor42,2214,2009-11-12,1216.73
Vendor42,2218,2009-11-18,1107.88
Vendor42,2225,2009-11-18,3640.65
Vendor42,2230,2009-11-19,1144.7
Vendor42,2231,2009-11-20,9014.63
Vendor42,2240,2009-11-26,1835.86
Vendor42,2241,2009-12-02,5079.75
Vendor42,2247,2009-12-02,2627.89
Vendor42,2255,2009-12-03,6414.64
Vendor42,2259,2009-12-04,1125.5
Vendor42,2262,2009-12-04,3345.86
Vendor42,2266,2009-12-04,3070.36
Vendor42,2271,2009-12-05,2513.18
Vendor42,2277,2009-12-11,4288.43
Vendor42,2284,2009-12-11,2061.2
Vendor42,2288,2009-12-21,5554.67
Vendor42,2292,2009-12-21,2978.98
Vendor42,2299,2009-12-22,8622.98
Vendor42,2302,2010-01-01,4480.59
Vendor42,2308,2010-01-02,2741.4
Vendor42,2314,2010-01-03,3001.41
Vendor42,2318,2010-01-09,3222.9
Vendor42,2321,2010-01-10,2775.21
Vendor42,2329,2010-01-10,2426.05
Vendor42,2335,2010-01-16,5702.9
Vendor42,2340,2010-01-22,2129.43
Vendor42,2345,2010-01-22,1661.32
Vendor42,2347,2010-01-23,2795.09
Vendor42,2355,2010-01-23,3796.65
Vendor42,2356,2010-01-31,1698.38
Vendor42,2362,2010-02-01,1335.74
Vendor42,2368,2010-02-02,2340.52
Vendor42,2374,2010-02-10,7142.04
Vendor42,2379,2010-02-10,3019.68
Vendor42,2385,2010-02-16,6288.87
Vendor42,2387,2010-02-17,1807.16
Vendor42,2394,2010-02-17,1409.47
Vendor42,2400,2010-02-18,3490.75
Vendor42,2405,2010-02-28,5327.54
Vendor42,2409,2010-03-10,2508.13
Vendor42,2415,2010-03-18,4030.28
Vendor42,2416,2010-03-26,5155.79
Vendor42,2424,2010-04-01,4249.26
Vendor42,2426,2010-04-01,1316.91
Vendor42,2434,2010-04-09,5612.75
Vendor42,2436,2010-04-10,4123.39
Vendor42,2442,2010-04-10,6042.88
Vendor42,2448,2010-04-18,5664.22
Vendor42,2455,2010-04-24,1153.66
Vendor42,2457,2010-04-25,4813.4
Vendor42,2464,2010-04-25,3288.77
Vendor42,2468,2010-05-01,6653.2
Vendor42,2475,2010-05-01,1898.21
Vendor42,2476,2010-05-02,6652.98
Vendor42,2484,2010-05-08,3326.45
Vendor42,2488,2010-05-18,8239.04
Vendor42,2495,2010-05-19,2508.86
Vendor42,2497,2010-05-19,1879.64
Vendor42,2503,2010-05-20,9978.88
Vendor42,2510,2010-05-21,6359.43
Vendor42,2511,2010-05-27,6485.32
Vendor42,2516,2010-05-27,1202.33
Vendor42,2524,2010-05-27,1952.57
Vendor42,2527,2010-05-27,1676.36
Vendor42,2532,2010-06-02,2664.26
Vendor42,2538,2010-06-02,1414.29
Vendor42,2544,2010-06-03,1349.32
Vendor42,2550,2010-06-03,8363.51
Vendor42,2554,2010-06-04,1047.96
Vendor42,2558,2010-06-10,4725.74
Vendor42,2564,2010-06-11,9462.88
Vendor42,2567,2010-06-11,2012.56
Vendor42,2575,2010-06-11,2995.65
Vendor42,2577,2010-06-17,1039.17
Vendor42,2581,2010-06-27,1815.72
Vendor42,2586,2010-06-27,1474.38
Vendor42,2591,2010-06-27,7461.7
Vendor42,2598,2010-06-28,1414.02
Vendor42,2601,2010-06-29,2422.64
Vendor42,2608,2010-06-29,4688.37
Vendor42,2612,2010-06-29,8606.27
Vendor42,2617,2010-06-29,6631.74
Vendor42,2623,2010-07-07,7422.61
Vendor42,2628,2010-07-07,4122.97
Vendor42,2635,2010-07-07,2347.41
Vendor42,2639,2010-07-13,1807.29
Vendor42,2645,2010-07-14,5495.37
Vendor42,2650,2010-07-20,6745.06
Vendor42,2651,2010-07-21,7078.45
Vendor42,2656,2010-07-21,5097.79
Vendor42,2665,2010-07-22,4028.02
Vendor42,2668,2010-07-23,3339.07
Vendor42,2672,2010-07-24,2629.76
Vendor42,2679,2010-07-24,4672.19
Vendor42,2683,2010-07-25,1250.77
Vendor42,2690,2010-07-25,2499.17
Vendor42,2695,2010-07-25,2145.93
Vendor42,2700,2010-08-04,2137.55
Vendor42,2701,2010-08-12,1946.67
Vendor42,2710,2010-08-12,3705.51
Vendor42,2714,2010-08-12,6836.58
Vendor42,2717,2010-08-20,8973.63
Vendor42,2723,2010-08-20,8895.78
Vendor42,2730,2010-08-28,1492.03
Vendor42,2733,2010-08-28,3918.54
Vendor42,2736,2010-08-28,1209.24
Vendor42,2742,2010-08-28,4632.56
Vendor42,2749,2010-08-29,8800.69
Vendor42,2753,2010-08-29,4682.06
Vendor42,2758,2010-08-30,2222.86
Vendor42,2762,2010-08-30,4602.0
Vendor42,2766,2010-09-05,1158.42
Vendor42,2773,2010-09-11,5859.5
Vendor42,2777,2010-09-12,2325.9
Vendor42,2784,2010-09-13,2558.95
Vendor42,2786,2010-09-21,9841.3
Vendor42,2795,2010-09-27,3065.36
Vendor42,2800,2010-10-07,1048.91
Vendor42,2801,2010-10-07,6579.72
Vendor42,2806,2010-10-17,4031.44
Vendor42,2813,2010-10-18,8202.91
Vendor42,2818,2010-10-18,3754.26
Vendor42,2824,2010-10-19,6648.54
Vendor42,2827,2010-10-27,1200.29
Vendor42,2833,2010-10-28,4493.12
Vendor42,2840,2010-10-28,4315.87
Vendor42,2844,2010-10-28,3650.59
Vendor42,2850,2010-11-07,9388.23
Vendor42,2852,2010-11-07,1893.11
Vendor42,2859,2010-11-13,1804.17
Vendor42,2864,2010-11-19,2556.61
Vendor42,2870,2010-11-29,1714.85
Vendor42,2871,2010-11-30,9504.33
Vendor42,2877,2010-11-30,1990.89
Vendor42,2881,2010-12-01,1281.88
Vendor42,2889,2010-12-02,1648.87
Vendor42,2892,2010-12-08,8260.5
Vendor42,2900,2010-12-09,1329.06
Vendor42,2905,2010-12-15,2237.72
Vendor42,2910,2010-12-15,1732.71
Vendor42,2914,2010-12-16,1262.22
Vendor42,2916,2010-12-16,2077.35
Vendor42,2921,2010-12-16,2034.05
Vendor42,2926,2010-12-17,2631.94
Vendor42,2934,2010-12-17,2177.12
Vendor42,2938,2010-12-18,1819.01
Vendor42,2941,2010-12-19,6829.38
Vendor42,2947,2010-12-19,5933.57
Vendor42,2954,2010-12-25,8175.99
Vendor42,2956,2010-12-25,2548.79
Vendor42,2961,2010-12-26,9148.93
Vendor42,2968,2011-01-03,4447.04
Vendor42,2971,2011-01-04,3449.43
Vendor42,2976,2011-01-14,9264.77
Vendor42,2985,2011-01-15,2604.45
Vendor42,2988,2011-01-21,2847.1
Vendor42,2993,2011-01-21,1431.73
Vendor42,2997,2011-01-22,3513.46
Vendor42,3003,2011-01-30,2863.55
Vendor42,3007,2011-02-05,1829.48
Vendor42,3011,2011-02-05,1002.21
Vendor42,3019,2011-02-05,3616.19
Vendor42,3025,2011-02-13,4476.08
Vendor42,3029,2011-02-21,6609.18
Vendor42,3032,2011-03-01,2820.48
Vendor42,3037,2011-03-01,3406.73
Vendor42,3043,2011-03-02,4491.51
Vendor42,3050,2011-03-03,1144.3
Vendor42,3052,2011-03-13,5995.44
Vendor42,3058,2011-03-13,2859.41
Vendor42,3064,2011-03-14,7303.46
Vendor42,3070,2011-03-20,1581.19
Vendor42,3071,2011-03-30,3063.05
Vendor42,3080,2011-03-31,2440.24
Vendor42,3085,2011-04-10,2914.21
Vendor42,3087,2011-04-10,1156.12
Vendor42,3092,2011-04-11,8871.91
Vendor42,3098,2011-04-11,1391.73
Vendor42,3103,2011-04-11,3043.04
Vendor42,3106,2011-04-11,6165.69
Vendor42,3111,2011-04-21,6138.7
Vendor42,3117,2011-04-21,4422.84
Vendor42,3123,2011-04-22,3272.12
Vendor42,3126,2011-04-28,1982.83
Vendor42,3135,2011-05-06,1091.48
Vendor42,3136,2011-05-07,3294.62
Vendor42,3141,2011-05-15,5262.69
Vendor42,3150,2011-05-15,1803.11
Vendor42,3151,2011-05-23,2979.98
Vendor42,3159,2011-05-24,5520.05
Vendor42,3163,2011-05-30,3565.61
Vendor42,3166,2011-06-05,1063.95
Vendor42,3171,2011-06-06,7877.44
Vendor42,3176,2011-06-06,3583.38
Vendor42,3183,2011-06-16,2628.88
Vendor42,3187,2011-06-16,7385.33
Vendor42,3192,2011-06-17,1250.2
Vendor42,3197,2011-06-23,1930.13
Vendor42,3204,2011-06-24,2962.77
Vendor42,3207,2011-06-30,5653.17
Vendor42,3211,2011-06-30,2285.59
Vendor42,3216,2011-07-01,3865.21
Vendor42,3222,2011-07-01,1836.1
Vendor42,3229,2011-07-02,2114.82
Vendor42,3232,2011-07-10,2558.47
Vendor42,3238,2011-07-16,1563.54
Vendor42,3245,2011-07-17,4501.67
Vendor42,3247,2011-07-27,3516.98
Vendor42,3255,2011-07-27,1762.13
Vendor42,3257,2011-08-02,2106.98
Vendor42,3264,2011-08-03,3534.33
Vendor42,3268,2011-08-04,3697.87
Vendor42,3274,2011-08-10,3667.09
Vendor42,3278,2011-08-10,3042.46
Vendor42,3282,2011-08-18,2004.24
Vendor42,3286,2011-08-18,3630.67
Vendor42,3295,2011-08-18,1223.88
Vendor42,3299,2011-08-28,1794.5
Vendor42,3302,2011-08-28,1702.1
Vendor42,3310,2011-09-05,2553.14
Vendor42,3311,2011-09-06,2558.74
Vendor42,3317,2011-09-07,2442.69
Vendor42,3321,2011-09-15,1065.54
Vendor42,3330,2011-09-16,1306.05
Vendor42,3334,2011-09-17,3360.63
Vendor42,3337,2011-09-18,1815.37
Vendor42,3341,2011-09-18,1125.87
Vendor42,3347,2011-09-18,4280.69
Vendor42,3354,2011-09-19,6268.03
Vendor42,3360,2011-09-25,4830.17
Vendor42,3361,2011-10-01,1845.24
Vendor42,3368,2011-10-11,9130.0
Vendor42,3373,2011-10-17,2000.76
Vendor42,3377,2011-10-18,1390.84
Vendor42,3381,2011-10-19,5715.58
Vendor42,3388,2011-10-27,7186.22
Vendor42,3392,2011-10-28,1198.4
Vendor42,3398,2011-10-28,5100.48
Vendor42,3401,2011-10-28,7447.37
Vendor42,3410,2011-11-07,3277.65
Vendor42,3411,2011-11-17,1312.64
Vendor42,3418,2011-11-27,1191.31
Vendor42,3422,2011-11-28,1381.04
Vendor42,3426,2011-12-04,2258.56
Vendor42,3431,2011-12-04,4491.37
Vendor42,3437,2011-12-12,4589.87
Vendor42,3445,2011-12-12,8827.81
Vendor42,3448,2011-12-13,6378.19
Vendor42,3455,2011-12-13,2455.21
Vendor42,3457,2011-12-21,3039.0
Vendor42,3461,2011-12-21,2065.23
Vendor42,3470,2011-12-21,1285.06
Vendor42,3472,2011-12-31,4028.72
Vendor42,3477,2012-01-01,3125.3
Vendor42,3483,2012-01-02,6429.6
Vendor42,3488,2012-01-03,2571.94
Vendor42,3494,2012-01-09,2168.36
Vendor42,3499,2012-01-09,1997.51
Vendor42,3502,2012-01-15,1317.99
Vendor42,3510,2012-01-15,1009.53
Vendor42,3511,2012-01-15,2540.28
Vendor42,3519,2012-01-25,6057.33
Vendor42,3522,2012-01-26,2136.85
Vendor42,3527,2012-02-03,1387.55
Vendor42,3535,2012-02-03,1472.12
Vendor42,3537,2012-02-03,3305.07
Vendor42,3542,2012-02-03,2681.53
Vendor42,3549,2012-02-04,2050.23
Vendor42,3552,2012-02-12,1710.41
Vendor42,3558,2012-02-12,2264.47
Vendor42,3564,2012-02-18,6632.12
Vendor42,3569,2012-02-19,2246.7
Vendor42,3575,2012-02-29,6540.63
Vendor42,3576,2012-03-01,2982.98
Vendor42,3581,2012-03-02,2831.92
Vendor42,3590,2012-03-10,2431.23
Vendor42,3595,2012-03-16,2223.69
Vendor42,3599,2012-03-17,2299.64
Vendor42,3602,2012-03-18,3543.87
Vendor42,3606,2012-03-24,1456.11
Vendor42,3612,2012-03-25,1161.82
Vendor42,3617,2012-03-31,2576.12
Vendor42,3625,2012-04-01,2028.6
Vendor42,3627,2012-04-02,5977.17
Vendor42,3635,2012-04-03,3779.44
Vendor42,3636,2012-04-03,1624.67
Vendor42,3644,2012-04-04,1057.56
Vendor42,3646,2012-04-14,3321.31
Vendor42,3654,2012-04-14,4678.47
Vendor42,3658,2012-04-22,2562.96
Vendor42,3662,2012-04-22,2579.38
Vendor42,3668,2012-04-28,5832.05
Vendor42,3672,2012-04-28,5798.83
Vendor42,3677,2012-04-29,1312.05
Vendor42,3685,2012-05-05,2479.98
Vendor42,3688,2012-05-06,8616.21
Vendor42,3692,2012-05-16,5968.17
Vendor42,3698,2012-05-26,1658.85
Vendor42,3702,2012-06-01,1555.97
Vendor42,3707,2012-06-01,8327.59
Vendor42,3713,2012-06-07,8568.42
Vendor42,3717,2012-06-07,8384.41
Vendor42,3724,2012-06-13,1987.23
Vendor42,3729,2012-06-13,4178.42
Vendor42,3733,2012-06-14,1978.86
Vendor42,3737,2012-06-22,7081.21
Vendor42,3743,2012-06-28,4054.29
Vendor42,3747,2012-07-06,3256.3
Vendor42,3753,2012-07-12,9128.73
Vendor42,3757,2012-07-18,2964.48
Vendor42,3761,2012-07-18,2018.86
Vendor42,3767,2012-07-26,1056.61
Vendor42,3773,2012-08-03,3972.67
Vendor42,3776,2012-08-09,8800.56
Vendor42,3785,2012-08-10,1165.95
Vendor42,3787,2012-08-11,4658.13
Vendor42,3792,2012-08-12,9857.81
Vendor42,3800,2012-08-12,4983.54
Vendor42,3805,2012-08-12,1911.81
Vendor42,3809,2012-08-12,5181.1
Vendor42,3815,2012-08-12,1301.91
Vendor42,3816,2012-08-13,3427.72
Vendor42,3822,2012-08-21,7065.44
Vendor42,3828,2012-08-21,1484.46
Vendor42,3834,2012-08-21,1646.05
Vendor42,3839,2012-08-21,4676.32
Vendor42,3843,2012-08-31,1835.46
Vendor42,3848,2012-09-01,1607.82
Vendor42,3853,2012-09-11,4363.83
Vendor42,3860,2012-09-12,5070.38
Vendor42,3863,2012-09-12,1617.33
Vendor42,3869,2012-09-22,3332.65
Vendor42,3871,2012-09-23,8318.86
Vendor42,3876,2012-10-03,2023.21
Vendor42,3882,2012-10-03,9047.39
Vendor42,3890,2012-10-03,2669.89
Vendor42,3894,2012-10-04,2732.14
Vendor42,3900,2012-10-04,4219.97
Vendor42,3904,2012-10-12,2504.31
Vendor42,3907,2012-10-18,9002.85
Vendor42,3913,2012-10-18,1687.43
Vendor42,3920,2012-10-24,2323.27
Vendor42,3924,2012-10-25,5413.97
Vendor42,3926,2012-10-26,1448.35
Vendor42,3932,2012-11-01,2485.51
Vendor42,3937,2012-11-09,1216.29
Vendor42,3941,2012-11-10,4421.36
Vendor42,3947,2012-11-16,9189.86
Vendor42,3955,2012-11-16,1859.27
Vendor42,3959,2012-11-17,6816.57
Vendor42,3961,2012-11-25,6428.83
Vendor42,3967,2012-11-26,8008.08
Vendor42,3973,2012-11-27,5670.85
Vendor42,3977,2012-12-03,3324.04
Vendor42,3984,2012-12-09,1922.14
Vendor42,3987,2012-12-15,3846.17
Vendor42,3991,2012-12-15,4545.53
Vendor42,3997,2012-12-16,1104.92
Vendor42,4003,2012-12-17,7053.59
Vendor42,4006,2012-12-17,5108.56
Vendor42,4011,2012-12-23,7780.56
Vendor42,4020,2012-12-23,3917.29
Vendor42,4021,2013-01-02,1063.08
Vendor42,4029,2013-01-08,2521.0
Vendor42,4035,2013-01-08,8864.66
Vendor42,4039,2013-01-16,1756.12
Vendor42,4043,2013-01-24,2852.91
Vendor42,4049,2013-02-03,5275.51
Vendor42,4055,2013-02-03,1163.21
Vendor42,4059,2013-02-11,2253.35
Vendor42,4064,2013-02-11,1553.84
Vendor42,4067,2013-02-12,2646.4
Vendor42,4074,2013-02-18,1533.55
Vendor42,4077,2013-02-19,9843.14
Vendor42,4083,2013-02-19,2076.76
Vendor42,4089,2013-02-19,2198.99
Vendor42,4091,2013-02-20,1083.02
Vendor42,4099,2013-02-21,2404.1
Vendor42,4104,2013-03-01,2694.78
Vendor42,4106,2013-03-01,5216.91
Vendor42,4113,2013-03-11,2070.32
Vendor42,4117,2013-03-17,2687.72
Vendor42,4121,2013-03-17,1656.12
Vendor42,4130,2013-03-17,9511.51
Vendor42,4131,2013-03-17,8513.47
Vendor42,4136,2013-03-25,7014.93
Vendor42,4141,2013-03-26,1034.0
Vendor42,4149,2013-03-26,3814.28
Vendor42,4152,2013-04-03,6808.94
Vendor42,4157,2013-04-03,6005.37
Vendor42,4164,2013-04-11,6621.69
Vendor42,4170,2013-04-11,5308.31
Vendor42,4173,2013-04-11,2248.78
Vendor42,4179,2013-04-21,6148.67
Vendor42,4183,2013-04-27,1677.48
Vendor42,4189,2013-04-28,4509.8
Vendor42,4195,2013-04-28,2881.71
Vendor42,4199,2013-04-29,1029.32
Vendor42,4202,2013-04-30,3006.97
Vendor42,4210,2013-05-01,3922.61
Vendor42,4213,2013-05-11,2834.21
Vendor42,4219,2013-05-11,6222.02
Vendor48,7,2006-01-11,2588.42
Vendor48,15,2006-01-17,3096.99
Vendor48,18,2006-01-23,1647.36
Vendor48,24,2006-01-29,3085.89
Vendor48,26,2006-02-06,6596.71
Vendor48,32,2006-02-06,2652.36
Vendor48,39,2006-02-12,3345.19
Vendor48,41,2006-02-20,7708.55
Vendor48,46,2006-02-21,1153.39
Vendor48,51,2006-02-22,1515.57
Vendor48,57,2006-02-23,1707.58
Vendor48,64,2006-02-23,9387.45
Vendor48,66,2006-03-03,1653.4
Vendor48,73,2006-03-03,8166.87
Vendor48,78,2006-03-04,1585.52
Vendor48,85,2006-03-04,6026.87
Vendor48,87,2006-03-05,4245.93
Vendor48,93,2006-03-06,4890.25
Vendor48,99,2006-03-07,1542.62
Vendor48,105,2006-03-08,6125.28
Vendor48,107,2006-03-08,1488.85
Vendor48,114,2006-03-16,2207.51
Vendor48,119,2006-03-22,7713.73
Vendor48,122,2006-04-01,1435.94
Vendor48,128,2006-04-07,8195.4
Vendor48,133,2006-04-07,3381.25
Vendor48,136,2006-04-07,9102.36
Vendor48,144,2006-04-08,9873.33
Vendor48,148,2006-04-18,1601.69
Vendor48,151,2006-04-18,6384.75
Vendor48,156,2006-04-28,5093.85
Vendor48,161,2006-04-29,1008.51
Vendor48,168,2006-04-30,1988.34
Vendor48,173,2006-05-06,8902.82
Vendor48,180,2006-05-12,9849.95
Vendor48,181,2006-05-13,5495.1
Vendor48,187,2006-05-13,3499.35
Vendor48,192,2006-05-13,2970.18
Vendor48,200,2006-05-19,6036.71
Vendor48,204,2006-05-29,4368.61
Vendor48,208,2006-05-30,5230.77
Vendor48,213,2006-06-07,1190.98
Vendor48,217,2006-06-08,1780.21
Vendor48,223,2006-06-14,1795.61
Vendor48,226,2006-06-22,3163.02
Vendor48,232,2006-06-23,4885.57
Vendor48,239,2006-06-24,5623.42
Vendor48,242,2006-06-25,2188.3
Vendor48,249,2006-06-26,1930.1
Vendor48,255,2006-06-26,4452.98
Vendor48,257,2006-06-27,9830.16
Vendor48,261,2006-06-27,2276.97
Vendor48,269,2006-06-28,1295.08
Vendor48,271,2006-06-29,3309.39
Vendor48,277,2006-07-05,8223.71
Vendor48,284,2006-07-13,6418.36
Vendor48,288,2006-07-13,2400.27
Vendor48,291,2006-07-19,2724.67
Vendor48,297,2006-07-19,5601.73
Vendor48,301,2006-07-20,7011.69
Vendor48,307,2006-07-30,2453.1
Vendor48,312,2006-07-31,9739.9
Vendor48,316,2006-07-31,1597.37
Vendor48,324,2006-08-10,2801.12
Vendor48,328,2006-08-10,4485.06
Vendor48,331,2006-08-16,4215.3
Vendor48,339,2006-08-22,3695.75
Vendor48,341,2006-09-01,2595.62
Vendor48,346,2006-09-02,4009.38
Vendor48,353,2006-09-12,6656.8
Vendor48,358,2006-09-13,3300.98
Vendor48,363,2006-09-14,2162.93
Vendor48,366,2006-09-14,4058.06
Vendor48,375,2006-09-22,1751.82
Vendor48,379,2006-09-22,5443.72
Vendor48,385,2006-09-22,1645.42
Vendor48,388,2006-09-23,1125.57
Vendor48,391,2006-09-29,4280.33
Vendor48,399,2006-10-07,7790.61
Vendor48,402,2006-10-07,7001.27
Vendor48,410,2006-10-07,2522.0
Vendor48,413,2006-10-13,5159.14
Vendor48,417,2006-10-19,4500.39
Vendor48,422,2006-10-25,2859.81
Vendor48,429,2006-10-25,1212.39
Vendor48,432,2006-10-26,2585.43
Vendor48,438,2006-10-26,5142.87
Vendor48,442,2006-10-26,4737.57
Vendor48,447,2006-11-03,1683.02
Vendor48,454,2006-11-09,3267.23
Vendor48,459,2006-11-09,1657.63
Vendor48,464,2006-11-09,2518.79
Vendor48,468,2006-11-15,5797.62
Vendor48,471,2006-11-15,1138.75
Vendor48,477,2006-11-15,3499.85
Vendor48,483,2006-11-16,2911.99
Vendor48,486,2006-11-24,1264.1
Vendor48,492,2006-11-30,3521.58
Vendor48,500,2006-12-06,1411.47
Vendor48,502,2006-12-16,1541.62
Vendor48,509,2006-12-26,2835.63
Vendor48,514,2007-01-01,4365.91
Vendor48,518,2007-01-02,4293.34
Vendor48,524,2007-01-02,5164.23
Vendor48,526,2007-01-12,3031.58
Vendor48,535,2007-01-13,9853.03
Vendor48,536,2007-01-13,1950.95
Vendor48,545,2007-01-14,5217.77
Vendor48,546,2007-01-14,5190.24
Vendor48,552,2007-01-22,4567.23
Vendor48,560,2007-01-30,9746.81
Vendor48,562,2007-01-30,2003.41
Vendor48,570,2007-02-05,3086.66
Vendor48,573,2007-02-11,9173.65
Vendor48,577,2007-02-11,1224.69
Vendor48,583,2007-02-12,3014.9
Vendor48,590,2007-02-13,1229.65
Vendor48,591,2007-02-13,6163.37
Vendor48,597,2007-02-19,3255.31
Vendor48,602,2007-02-19,1662.72
Vendor48,607,2007-02-20,8433.79
Vendor48,615,2007-02-21,1862.38
Vendor48,620,2007-02-21,3825.93
Vendor48,623,2007-02-22,1121.21
Vendor48,627,2007-02-23,2494.06
Vendor48,633,2007-02-23,2073.45
Vendor48,636,2007-02-23,2216.11
Vendor48,645,2007-02-24,4492.57
Vendor48,647,2007-03-04,1356.81
Vendor48,651,2007-03-10,3448.86
Vendor48,656,2007-03-16,2682.91
Vendor48,662,2007-03-24,6168.01
Vendor48,670,2007-04-01,1534.85
Vendor48,672,2007-04-01,2201.46
Vendor48,680,2007-04-02,2221.57
Vendor48,681,2007-04-10,1598.33
Vendor48,686,2007-04-16,8030.74
Vendor48,694,2007-04-17,2478.71
Vendor48,698,2007-04-17,4880.63
Vendor48,705,2007-04-18,7871.62
Vendor48,710,2007-04-28,5901.51
Vendor48,714,2007-05-06,2404.98
Vendor48,716,2007-05-12,1817.43
Vendor48,721,2007-05-12,2154.96
Vendor48,729,2007-05-13,2962.5
Vendor48,732,2007-05-14,5285.64
Vendor48,739,2007-05-14,1127.06
Vendor48,743,2007-05-20,5762.16
Vendor48,748,2007-05-28,2681.47
Vendor48,751,2007-05-29,1831.74
Vendor48,757,2007-05-29,2153.53
Vendor48,762,2007-05-29,8677.86
Vendor48,770,2007-05-29,4639.35
Vendor48,772,2007-05-30,9276.75
Vendor48,777,2007-06-05,4816.55
Vendor48,784,2007-06-11,9573.02
Vendor48,788,2007-06-12,3745.93
Vendor48,791,2007-06-13,8715.89
Vendor48,800,2007-06-14,3831.79
Vendor48,804,2007-06-15,2536.63
Vendor48,808,2007-06-15,3110.57
Vendor48,814,2007-06-15,8952.17
Vendor48,819,2007-06-21,1117.12
Vendor48,825,2007-06-29,1411.43
Vendor48,826,2007-06-29,1328.57
Vendor48,833,2007-07-05,8432.94
Vendor48,839,2007-07-15,9222.4
Vendor48,845,2007-07-25,2501.66
Vendor48,846,2007-07-31,3997.12
Vendor48,854,2007-08-01,3149.23
Vendor48,856,2007-08-01,1717.0
Vendor48,864,2007-08-02,1232.91
Vendor48,868,2007-08-02,1849.8
Vendor48,872,2007-08-08,1254.66
Vendor48,877,2007-08-14,7726.73
Vendor48,883,2007-08-15,1333.84
Vendor48,887,2007-08-16,2881.45
Vendor48,893,2007-08-17,7537.34
Vendor48,899,2007-08-17,1675.54
Vendor48,902,2007-08-17,3600.66
Vendor48,907,2007-08-17,2118.61
Vendor48,912,2007-08-23,5605.11
Vendor48,918,2007-08-31,1426.67
Vendor48,924,2007-09-01,1963.56
Vendor48,930,2007-09-09,2051.04
Vendor48,932,2007-09-09,7257.83
Vendor48,940,2007-09-10,2950.84
Vendor48,942,2007-09-11,2210.06
Vendor48,949,2007-09-11,5244.01
Vendor48,955,2007-09-11,4501.02
Vendor48,957,2007-09-12,1414.51
Vendor48,965,2007-09-22,2581.8
Vendor48,970,2007-09-28,4034.07
Vendor48,972,2007-09-28,2363.38
Vendor48,980,2007-09-28,2216.95
Vendor48,983,2007-09-29,1699.64
Vendor48,989,2007-09-30,3953.63
Vendor48,992,2007-09-30,9638.06
Vendor48,999,2007-10-10,7692.35
Vendor48,1001,2007-10-18,1763.57
Vendor48,1007,2007-10-19,3642.39
Vendor48,1011,2007-10-19,2566.05
Vendor48,1020,2007-10-20,7213.06
Vendor48,1021,2007-10-20,2762.2
Vendor48,1028,2007-10-21,7012.35
Vendor48,1033,2007-10-27,3462.26
Vendor48,1039,2007-10-28,2488.29
Vendor48,1044,2007-10-29,7563.25
Vendor48,1047,2007-11-08,3489.73
Vendor48,1053,2007-11-08,1513.23
Vendor48,1058,2007-11-16,7115.38
Vendor48,1065,2007-11-17,9929.44
Vendor48,1069,2007-11-18,3999.19
Vendor48,1073,2007-11-28,1853.01
Vendor48,1078,2007-12-06,8826.54
Vendor48,1083,2007-12-12,1489.38
Vendor48,1090,2007-12-18,4292.85
Vendor48,1094,2007-12-24,6147.66
Vendor48,1099,2007-12-24,1646.49
Vendor48,1101,2007-12-24,9083.78
Vendor48,1107,2007-12-24,2251.77
Vendor48,1111,2008-01-01,3761.85
Vendor48,1118,2008-01-01,1439.85
Vendor48,1123,2008-01-07,7135.75
Vendor48,1128,2008-01-08,2459.29
Vendor48,1131,2008-01-18,1795.08
Vendor48,1136,2008-01-28,2530.57
Vendor48,1141,2008-01-28,3072.15
Vendor48,1146,2008-01-29,1495.75
Vendor48,1154,2008-01-30,2728.85
Vendor48,1160,2008-02-09,7301.11
Vendor48,1164,2008-02-15,2205.41
Vendor48,1170,2008-02-23,1127.47
Vendor48,1173,2008-02-23,4810.71
Vendor48,1178,2008-02-23,3262.2
Vendor48,1182,2008-02-24,1662.41
Vendor48,1190,2008-02-24,3752.24
Vendor48,1192,2008-03-01,5490.08
Vendor48,1198,2008-03-07,2926.97
Vendor48,1201,2008-03-07,7213.73
Vendor48,1208,2008-03-08,3314.9
Vendor48,1215,2008-03-08,6587.78
Vendor48,1218,2008-03-16,2322.34
Vendor48,1222,2008-03-17,6962.82
Vendor48,1227,2008-03-17,2259.06
Vendor48,1235,2008-03-18,4645.62
Vendor48,1239,2008-03-24,5782.52
Vendor48,1243,2008-03-24,8864.07
Vendor48,1250,2008-04-03,1522.27
Vendor48,1251,2008-04-04,2306.1
Vendor48,1260,2008-04-04,1624.36
Vendor48,1262,2008-04-10,8760.52
Vendor48,1267,2008-04-10,1055.31
Vendor48,1271,2008-04-18,4083.98
Vendor48,1278,2008-04-19,1115.5
Vendor48,1285,2008-04-25,2307.91
Vendor48,1286,2008-05-01,3582.0
Vendor48,1294,2008-05-07,2346.13
Vendor48,1296,2008-05-13,1993.94
Vendor48,1302,2008-05-14,3938.55
Vendor48,1310,2008-05-15,6476.77
Vendor48,1311,2008-05-16,2491.38
Vendor48,1317,2008-05-22,9719.7
Vendor48,1325,2008-05-28,1895.55
Vendor48,1328,2008-05-29,1140.73
Vendor48,1333,2008-06-04,1115.17
Vendor48,1337,2008-06-04,1161.91
Vendor48,1342,2008-06-04,9107.75
Vendor48,1346,2008-06-05,3437.98
Vendor48,1355,2008-06-13,4572.08
Vendor48,1360,2008-06-23,6144.73
Vendor48,1365,2008-06-24,2015.83
Vendor48,1368,2008-06-24,8859.33
Vendor48,1375,2008-06-25,1046.53
Vendor48,1378,2008-07-01,9787.64
Vendor48,1385,2008-07-07,3409.27
Vendor48,1386,2008-07-07,2931.13
Vendor48,1393,2008-07-08,3557.71
Vendor48,1399,2008-07-09,9897.55
Vendor48,1402,2008-07-19,1624.14
Vendor48,1410,2008-07-27,1333.75
Vendor48,1411,2008-08-06,5204.06
Vendor48,1416,2008-08-06,1337.95
Vendor48,1424,2008-08-12,1236.81
Vendor48,1430,2008-08-13,1206.68
Vendor48,1433,2008-08-23,1094.19
Vendor48,1439,2008-08-29,9488.51
Vendor48,1444,2008-09-06,7288.86
Vendor48,1450,2008-09-06,9089.08
Vendor48,1452,2008-09-12,3560.95
Vendor48,1456,2008-09-13,2801.17
Vendor48,1462,2008-09-13,2367.56
Vendor48,1467,2008-09-13,2385.86
Vendor48,1472,2008-09-13,8032.17
Vendor48,1476,2008-09-13,7065.29
Vendor48,1483,2008-09-13,5830.68
Vendor48,1490,2008-09-19,2528.49
Vendor48,1493,2008-09-20,1790.56
Vendor48,1500,2008-09-26,4763.08
Vendor48,1503,2008-09-27,2613.77
Vendor48,1507,2008-10-03,2648.46
Vendor48,1515,2008-10-04,1114.69
Vendor48,1519,2008-10-05,6603.49
Vendor48,1524,2008-10-11,5187.71
Vendor48,1526,2008-10-17,2516.45
Vendor48,1535,2008-10-18,2288.24
Vendor48,1536,2008-10-24,6651.73
Vendor48,1541,2008-11-01,7852.31
Vendor48,1547,2008-11-07,2776.37
Vendor48,1551,2008-11-07,1598.53
Vendor48,1560,2008-11-08,2743.76
Vendor48,1561,2008-11-14,4256.15
Vendor48,1568,2008-11-15,3220.32
Vendor48,1575,2008-11-15,1934.18
Vendor48,1579,2008-11-16,5939.49
Vendor48,1583,2008-11-16,1283.76
Vendor48,1587,2008-11-26,2794.14
Vendor48,1591,2008-11-26,2012.21
Vendor48,1599,2008-11-27,2881.32
Vendor48,1603,2008-11-27,1065.39
Vendor48,1608,2008-12-03,1910.12
Vendor48,1613,2008-12-11,2972.78
Vendor48,1616,2008-12-12,1955.84
Vendor48,1624,2008-12-18,1312.9
Vendor48,1630,2008-12-18,1900.33
Vendor48,1633,2008-12-19,3145.39
Vendor48,1638,2008-12-20,9185.03
Vendor48,1641,2008-12-30,2941.86
Vendor48,1649,2008-12-31,1245.16
Vendor48,1655,2009-01-01,1033.42
Vendor48,1656,2009-01-09,4078.28
Vendor48,1665,2009-01-10,8345.92
Vendor48,1668,2009-01-11,3079.59
Vendor48,1674,2009-01-11,1750.04
Vendor48,1679,2009-01-12,1749.3
Vendor48,1684,2009-01-12,9840.41
Vendor48,1687,2009-01-13,1201.8
Vendor48,1695,2009-01-14,8376.85
Vendor48,1698,2009-01-20,1946.21
Vendor48,1704,2009-01-20,2144.04
Vendor48,1706,2009-01-20,2995.05
Vendor48,1711,2009-01-21,1347.53
Vendor48,1719,2009-01-21,2890.36
Vendor48,1723,2009-01-22,5053.63
Vendor48,1729,2009-01-22,4585.81
Vendor48,1731,2009-01-28,8475.06
Vendor48,1738,2009-01-28,2315.69
Vendor48,1744,2009-02-03,5582.88
Vendor48,1746,2009-02-13,1313.79
Vendor48,1754,2009-02-23,2415.59
Vendor48,1757,2009-03-01,3921.93
Vendor48,1763,2009-03-01,7485.62
Vendor48,1768,2009-03-09,3364.87
Vendor48,1771,2009-03-15,7849.51
Vendor48,1779,2009-03-15,7734.35
Vendor48,1784,2009-03-15,1347.13
Vendor48,1787,2009-03-16,7305.51
Vendor48,1792,2009-03-24,1175.93
Vendor48,1797,2009-03-25,1474.76
Vendor48,1805,2009-03-26,1257.97
Vendor48,1809,2009-03-27,2563.89
Vendor48,1814,2009-03-28,3130.69
Vendor48,1819,2009-04-05,1791.58
Vendor48,1823,2009-04-11,3839.56
Vendor48,1826,2009-04-12,1639.24
Vendor48,1833,2009-04-13,2602.9
Vendor48,1839,2009-04-14,1975.48
Vendor48,1843,2009-04-15,9671.17
Vendor48,1849,2009-04-21,9088.29
Vendor48,1853,2009-04-21,2874.3
Vendor48,1859,2009-05-01,6802.93
Vendor48,1863,2009-05-09,3961.64
Vendor48,1866,2009-05-09,2068.55
Vendor48,1873,2009-05-09,8098.39
Vendor48,1878,2009-05-09,1237.96
Vendor48,1883,2009-05-15,4167.51
Vendor48,1890,2009-05-21,5734.98
Vendor48,1894,2009-05-21,1079.09
Vendor48,1896,2009-05-22,6697.02
Vendor48,1905,2009-05-22,1214.95
Vendor48,1908,2009-05-23,7360.62
Vendor48,1912,2009-06-02,4536.98
Vendor48,1917,2009-06-10,1758.42
Vendor48,1922,2009-06-16,2316.1
Vendor48,1930,2009-06-22,2229.32
Vendor48,1933,2009-07-02,7314.3
Vendor48,1939,2009-07-08,3385.99
Vendor48,1945,2009-07-09,5836.91
Vendor48,1947,2009-07-19,1208.21
Vendor48,1953,2009-07-19,2608.61
Vendor48,1960,2009-07-19,5547.91
Vendor48,1961,2009-07-20,1113.84
Vendor48,1967,2009-07-26,6224.41
Vendor48,1974,2009-08-01,1618.86
Vendor48,1977,2009-08-02,3579.83
Vendor48,1982,2009-08-03,5796.77
Vendor48,1988,2009-08-09,2048.55
Vendor48,1993,2009-08-10,2144.82
Vendor48,1998,2009-08-18,1031.79
Vendor48,2001,2009-08-18,1456.37
Vendor48,2006,2009-08-19,5898.2
Vendor48,2011,2009-08-25,5659.76
Vendor48,2017,2009-08-26,5068.39
Vendor48,2025,2009-08-26,6548.55
Vendor48,2028,2009-08-26,3693.45
Vendor48,2034,2009-09-03,1152.95
Vendor48,2036,2009-09-03,2267.06
Vendor48,2043,2009-09-09,7366.89
Vendor48,2048,2009-09-15,4871.2
Vendor48,2054,2009-09-21,2319.14
Vendor48,2059,2009-09-21,1155.26
Vendor48,2062,2009-09-22,9108.74
Vendor48,2069,2009-10-02,2450.69
Vendor48,2074,2009-10-03,1390.01
Vendor48,2076,2009-10-03,1700.2
Vendor48,2084,2009-10-04,6926.97
Vendor48,2089,2009-10-04,1533.9
Vendor48,2095,2009-10-10,5344.37
Vendor48,2096,2009-10-16,6426.65
Vendor48,2101,2009-10-24,4342.3
Vendor48,2107,2009-11-01,6882.97
Vendor48,2111,2009-11-02,3766.29
Vendor48,2118,2009-11-10,7653.28
Vendor48,2124,2009-11-16,2015.95
Vendor48,2126,2009-11-26,2174.62
Vendor48,2132,2009-12-02,1505.44
Vendor48,2139,2009-12-10,3678.08
Vendor48,2145,2009-12-11,6005.65
Vendor48,2148,2009-12-12,3604.57
Vendor48,2152,2009-12-18,1085.92
Vendor48,2159,2009-12-19,9449.54
Vendor48,2163,2009-12-25,2197.83
Vendor48,2170,2010-01-04,1237.94
Vendor48,2171,2010-01-04,2558.97
Vendor48,2179,2010-01-04,3962.76
Vendor48,2181,2010-01-05,1791.32
Vendor48,2188,2010-01-15,2754.71
Vendor48,2191,2010-01-16,2515.5
Vendor48,2198,2010-01-17,2618.12
Vendor48,2205,2010-01-18,4553.56
Vendor48,2208,2010-01-18,3303.83
Vendor48,2215,2010-01-18,2889.84
Vendor48,2218,2010-01-18,7151.46
Vendor48,2221,2010-01-19,7439.9
Vendor48,2227,2010-01-25,4775.08
Vendor48,2233,2010-02-02,2398.75
Vendor48,2240,2010-02-02,5291.28
Vendor48,2244,2010-02-03,1117.14
Vendor48,2246,2010-02-03,3518.26
Vendor48,2252,2010-02-09,6003.41
Vendor48,2256,2010-02-17,1479.77
Vendor48,2263,2010-02-18,2144.87
Vendor48,2270,2010-02-19,2348.11
Vendor48,2273,2010-02-20,8441.97
Vendor48,2278,2010-02-21,2351.15
Vendor48,2283,2010-03-03,3513.23
Vendor48,2289,2010-03-04,1424.44
Vendor48,2293,2010-03-10,1069.21
Vendor48,2296,2010-03-11,1144.25
Vendor48,2304,2010-03-11,1350.83
Vendor48,2307,2010-03-12,4613.78
Vendor48,2314,2010-03-20,2275.85
Vendor48,2319,2010-03-20,1401.84
Vendor48,2325,2010-03-28,1332.04
Vendor48,2330,2010-04-05,9396.42
Vendor48,2334,2010-04-11,4422.84
Vendor48,2340,2010-04-19,1770.41
Vendor48,2341,2010-04-20,1112.11
Vendor48,2347,2010-04-20,4885.76
Vendor48,2352,2010-04-28,4814.19
Vendor48,2360,2010-04-28,8165.66
Vendor48,2364,2010-05-04,1662.36
Vendor48,2366,2010-05-14,7168.44
Vendor48,2373,2010-05-15,4411.81
Vendor48,2377,2010-05-21,4557.87
Vendor48,2382,2010-05-22,1989.68
Vendor48,2389,2010-05-23,4980.14
Vendor48,2391,2010-05-24,1473.13
Vendor48,2400,2010-05-24,1146.9
Vendor48,2405,2010-05-25,4229.67
Vendor48,2410,2010-05-31,1028.45
Vendor48,2411,2010-06-01,1596.63
Vendor48,2416,2010-06-02,2021.07
Vendor48,2423,2010-06-03,1153.42
Vendor48,2427,2010-06-04,5024.55
Vendor48,2435,2010-06-04,2167.54
Vendor48,2439,2010-06-10,1022.9
Vendor48,2445,2010-06-10,8315.99
Vendor48,2449,2010-06-11,1143.08
Vendor48,2455,2010-06-21,3217.46
Vendor48,2457,2010-06-22,1063.96
Vendor48,2463,2010-06-30,1065.13
Vendor48,2468,2010-06-30,9095.92
Vendor48,2473,2010-06-30,1374.15
Vendor48,2477,2010-06-30,5004.69
Vendor48,2484,2010-07-06,4179.81
Vendor48,2489,2010-07-07,1596.42
Vendor48,2491,2010-07-07,1747.97
Vendor48,2497,2010-07-08,1517.09
Vendor48,2501,2010-07-08,5904.37
Vendor48,2507,2010-07-16,4157.43
Vendor48,2515,2010-07-17,1496.3
Vendor48,2518,2010-07-17,2640.01
Vendor48,2521,2010-07-17,7059.95
Vendor48,2526,2010-07-23,2677.8
Vendor48,2534,2010-07-23,6524.0
Vendor48,2540,2010-07-23,1276.63
Vendor48,2542,2010-07-23,8764.24
Vendor48,2547,2010-07-23,4666.98
Vendor48,2551,2010-07-29,3519.2
Vendor48,2560,2010-07-30,1481.25
Vendor48,2565,2010-07-30,1834.24
Vendor48,2567,2010-07-31,3422.5
Vendor48,2572,2010-07-31,4965.77
Vendor48,2579,2010-08-10,2273.83
Vendor48,2583,2010-08-18,3616.29
Vendor48,2588,2010-08-28,1683.26
Vendor48,2595,2010-09-03,7718.47
Vendor48,2598,2010-09-09,5598.93
Vendor48,2602,2010-09-10,4196.94
Vendor48,2606,2010-09-11,1887.64
Vendor48,2613,2010-09-21,2180.39
Vendor48,2617,2010-09-27,4054.59
Vendor48,2625,2010-09-27,1245.45
Vendor48,2629,2010-10-03,1352.97
Vendor48,2632,2010-10-09,2559.71
Vendor48,2636,2010-10-17,1118.54
Vendor48,2641,2010-10-18,2522.69
Vendor48,2646,2010-10-28,5122.48
Vendor48,2652,2010-11-03,4057.73
Vendor48,2657,2010-11-11,4793.25
Vendor48,2665,2010-11-17,1485.03
Vendor48,2670,2010-11-18,1437.51
Vendor48,2675,2010-11-18,2127.98
Vendor48,2678,2010-11-19,1660.4
Vendor48,2685,2010-11-20,6056.67
Vendor48,2687,2010-11-21,1978.25
Vendor48,2695,2010-11-22,6500.77
Vendor48,2696,2010-11-23,7581.32
Vendor48,2702,2010-11-24,4263.62
Vendor48,2707,2010-11-25,1464.92
Vendor48,2714,2010-11-26,1490.63
Vendor48,2720,2010-12-06,9136.89
Vendor48,2724,2010-12-12,8763.71
Vendor48,2730,2010-12-13,5895.18
Vendor48,2732,2010-12-13,6477.1
Vendor48,2739,2010-12-21,6072.55
Vendor48,2744,2010-12-27,3065.01
Vendor48,2747,2010-12-27,1649.27
Vendor48,2752,2010-12-27,3123.7
Vendor48,2759,2011-01-04,1582.77
Vendor48,2763,2011-01-05,1274.25
Vendor48,2766,2011-01-06,3525.64
Vendor48,2774,2011-01-12,2326.74
Vendor48,2778,2011-01-13,8126.65
Vendor48,2782,2011-01-23,7916.77
Vendor48,2790,2011-01-31,1163.72
Vendor48,2791,2011-01-31,3227.25
Vendor48,2800,2011-02-08,3797.86
Vendor48,2803,2011-02-09,3495.38
Vendor48,2810,2011-02-15,5047.13
Vendor48,2811,2011-02-25,2412.45
Vendor48,2818,2011-02-26,1367.11
Vendor48,2825,2011-03-04,2361.96
Vendor48,2826,2011-03-05,1523.3
Vendor48,2833,2011-03-06,4391.02
Vendor48,2837,2011-03-06,1281.12
Vendor48,2842,2011-03-12,4470.78
Vendor48,2848,2011-03-12,6080.13
Vendor48,2855,2011-03-13,7840.18
Vendor48,2858,2011-03-19,1975.59
Vendor48,2865,2011-03-25,1755.18
Vendor48,2867,2011-03-31,1366.69
Vendor48,2875,2011-04-10,7580.39
Vendor48,2880,2011-04-11,3077.65
Vendor48,2883,2011-04-12,4790.2
Vendor48,2887,2011-04-18,1888.22
Vendor48,2893,2011-04-26,2067.99
Vendor48,2898,2011-05-02,3797.34
Vendor48,2903,2011-05-02,3001.12
Vendor48,2910,2011-05-08,1069.67
Vendor48,2913,2011-05-08,4699.96
Vendor48,2916,2011-05-08,3195.52
Vendor48,2925,2011-05-09,8747.27
Vendor48,2927,2011-05-10,3737.5
Vendor48,2931,2011-05-11,2507.84
Vendor48,2938,2011-05-11,5372.18
Vendor48,2941,2011-05-17,2504.75
Vendor48,2948,2011-05-25,1803.39
Vendor48,2953,2011-05-25,2310.73
Vendor48,2956,2011-05-31,9797.0
Vendor48,2964,2011-06-01,4093.52
Vendor48,2969,2011-06-01,6428.51
Vendor48,2975,2011-06-07,1371.82
Vendor48,2978,2011-06-07,7191.49
Vendor48,2983,2011-06-17,1130.39
Vendor48,2989,2011-06-25,4249.75
Vendor48,2992,2011-06-26,2480.66
Vendor48,2998,2011-06-26,3094.48
Vendor48,3002,2011-06-26,1449.96
Vendor48,3009,2011-07-02,1708.87
Vendor48,3011,2011-07-02,2061.79
Vendor48,3016,2011-07-12,9398.44
Vendor48,3024,2011-07-18,2012.53
Vendor48,3029,2011-07-18,7864.0
Vendor48,3032,2011-07-24,1639.8
Vendor48,3039,2011-07-24,1964.82
Vendor48,3042,2011-07-25,1737.02
Vendor48,3049,2011-08-02,7688.1
Vendor48,3053,2011-08-03,2081.54
Vendor48,3060,2011-08-03,1637.5
Vendor48,3065,2011-08-03,3283.98
Vendor48,3068,2011-08-09,6641.92
Vendor48,3075,2011-08-15,4449.0
Vendor48,3079,2011-08-21,1314.27
Vendor48,3084,2011-08-22,3804.16
Vendor48,3086,2011-08-23,7166.88
Vendor48,3093,2011-08-23,5104.29
Vendor48,3098,2011-08-23,2138.5
Vendor48,3101,2011-08-29,6744.2
Vendor48,3108,2011-08-29,2714.96
Vendor48,3114,2011-08-30,1563.81
Vendor48,3116,2011-08-30,2741.1
Vendor48,3123,2011-09-09,5380.59
Vendor48,3130,2011-09-15,1712.37
Vendor48,3135,2011-09-16,4890.14
Vendor48,3136,2011-09-26,2110.95
Vendor48,3144,2011-09-26,1168.98
Vendor48,3150,2011-10-06,2169.73
Vendor48,3152,2011-10-12,3703.46
Vendor48,3159,2011-10-18,6170.0
Vendor48,3165,2011-10-24,3464.4
Vendor48,3168,2011-10-25,1949.88
Vendor48,3171,2011-10-25,2127.25
Vendor48,3178,2011-10-25,1500.38
Vendor48,3183,2011-10-25,8729.13
Vendor48,3190,2011-10-25,1906.05
Vendor48,3193,2011-10-26,1113.32
Vendor48,3198,2011-10-26,3328.13
Vendor48,3201,2011-11-05,6927.53
Vendor48,3207,2011-11-15,3404.17
Vendor48,3211,2011-11-15,2011.21
Vendor48,3219,2011-11-16,2415.68
Vendor48,3224,2011-11-16,1159.84
Vendor48,3227,2011-11-16,1088.92
Vendor48,3234,2011-11-17,1216.24
Vendor48,3238,2011-11-18,6070.49
Vendor48,3242,2011-11-26,7679.35
Vendor48,3250,2011-12-06,3908.82
Vendor48,3252,2011-12-16,9272.72
Vendor48,3259,2011-12-17,1269.07
Vendor48,3264,2011-12-23,2418.43
Vendor48,3269,2011-12-23,2994.54
Vendor48,3273,2011-12-31,5350.9
Vendor48,3279,2011-12-31,5347.25
Vendor48,3281,2012-01-01,1286.69
Vendor48,3289,2012-01-01,3087.42
Vendor48,3294,2012-01-07,2709.48
Vendor48,3298,2012-01-13,2674.49
Vendor48,3301,2012-01-14,4325.75
Vendor48,3307,2012-01-24,7056.82
Vendor48,3314,2012-01-25,1460.65
Vendor48,3317,2012-01-26,2402.06
Vendor48,3322,2012-01-26,1014.04
Vendor48,3327,2012-01-26,5283.01
Vendor48,3335,2012-02-05,1959.06
Vendor48,3337,2012-02-06,1746.47
Vendor48,3344,2012-02-16,5020.07
Vendor48,3346,2012-02-17,6224.05
Vendor48,3352,2012-02-17,2433.52
Vendor48,3360,2012-02-17,2651.72
Vendor48,3364,2012-02-18,1202.7
Vendor48,3368,2012-02-28,1052.04
Vendor48,3372,2012-03-05,4421.36
Vendor48,3376,2012-03-05,1008.52
Vendor48,3384,2012-03-06,9080.43
Vendor48,3387,2012-03-12,5201.08
Vendor48,3392,2012-03-12,4070.31
Vendor48,3397,2012-03-12,5550.09
Vendor48,3404,2012-03-13,1270.7
Vendor48,3407,2012-03-21,1767.18
Vendor48,3412,2012-03-22,3286.33
Vendor48,3416,2012-04-01,7349.98
Vendor48,3423,2012-04-02,1881.6
Vendor48,3426,2012-04-08,7408.25
Vendor48,3433,2012-04-14,9471.58
Vendor48,3436,2012-04-20,1327.64
Vendor48,3441,2012-04-30,2113.96
Vendor48,3447,2012-05-01,3022.5
Vendor48,3455,2012-05-02,6111.26
Vendor48,3459,2012-05-03,7927.43
Vendor48,3463,2012-05-03,1543.1
Vendor48,3467,2012-05-03,3975.09
Vendor48,3474,2012-05-04,1662.54
Vendor48,3480,2012-05-12,4842.87
Vendor48,3485,2012-05-12,9872.96
Vendor48,3488,2012-05-20,2002.62
Vendor48,3494,2012-05-26,5134.67
Vendor48,3499,2012-05-27,1413.86
Vendor48,3502,2012-05-27,1129.89
Vendor48,3508,2012-05-27,3470.49
Vendor48,3514,2012-06-06,1145.07
Vendor48,3518,2012-06-07,4735.61
Vendor48,3522,2012-06-07,4303.35
Vendor48,3529,2012-06-15,1879.27
Vendor48,3531,2012-06-15,4444.97
Vendor48,3538,2012-06-16,7863.46
Vendor48,3544,2012-06-24,4912.79
Vendor48,3550,2012-06-25,2729.7
Vendor48,3553,2012-07-01,8943.89
Vendor48,3559,2012-07-07,1475.17
Vendor48,3562,2012-07-08,3642.16
Vendor48,3568,2012-07-14,2060.58
Vendor48,3574,2012-07-20,5799.92
Vendor48,3579,2012-07-21,6064.13
Vendor48,3582,2012-07-31,4382.2
Vendor48,3587,2012-07-31,1046.97
Vendor48,3591,2012-08-08,1337.16
Vendor48,3599,2012-08-08,3052.75
Vendor48,3605,2012-08-14,2740.49
Vendor48,3610,2012-08-14,5801.24
Vendor48,3615,2012-08-14,3386.53
Vendor48,3620,2012-08-15,1510.77
Vendor48,3625,2012-08-23,1319.54
Vendor48,3627,2012-08-29,6450.09
Vendor48,3631,2012-09-04,3414.74
Vendor48,3639,2012-09-04,2320.41
Vendor48,3641,2012-09-14,1408.88
Vendor48,3647,2012-09-24,8554.33
Vendor48,3654,2012-09-24,1722.83
Vendor48,3658,2012-10-04,2355.06
Vendor48,3664,2012-10-04,5828.2
Vendor48,3670,2012-10-10,1832.77
Vendor48,3672,2012-10-10,2824.89
Vendor48,3677,2012-10-16,1330.89
Vendor48,3684,2012-10-22,9450.33
Vendor48,3686,2012-10-30,1620.4
Vendor48,3694,2012-10-31,1843.91
Vendor48,3700,2012-11-06,1985.12
Vendor48,3704,2012-11-06,3671.03
Vendor48,3709,2012-11-16,6258.78
Vendor48,3714,2012-11-16,1498.05
Vendor48,3718,2012-11-17,1770.07
Vendor48,3723,2012-11-17,3762.7
Vendor48,3729,2012-11-23,1396.16
Vendor48,3734,2012-11-24,1069.85
Vendor48,3738,2012-11-30,4871.89
Vendor48,3745,2012-12-06,1587.45
Vendor48,3750,2012-12-07,1624.57
Vendor48,3752,2012-12-13,5046.03
Vendor48,3760,2012-12-13,1747.18
Vendor48,3763,2012-12-19,1736.49
Vendor48,3769,2012-12-29,2817.7
Vendor48,3772,2012-12-30,8517.36
Vendor48,3780,2012-12-31,1938.34
Vendor48,3783,2012-12-31,1449.03
Vendor48,3787,2013-01-01,6556.89
Vendor48,3795,2013-01-01,2048.12
Vendor48,3799,2013-01-02,5791.27
Vendor48,3804,2013-01-02,8618.73
Vendor48,3807,2013-01-08,9382.85
Vendor48,3812,2013-01-14,4986.0
Vendor48,3816,2013-01-20,6878.74
Vendor48,3823,2013-01-30,2365.81
Vendor48,3830,2013-01-30,2736.18
Vendor48,3832,2013-01-31,2737.35
Vendor48,3839,2013-02-10,1455.65
Vendor48,3843,2013-02-11,5412.57
Vendor48,3848,2013-02-21,8306.27
Vendor48,3853,2013-02-22,8705.18
Vendor48,3858,2013-02-22,1033.67
Vendor48,3861,2013-02-23,2638.51
Vendor48,3868,2013-03-03,8245.96
Vendor48,3872,2013-03-03,6562.13
Vendor48,3880,2013-03-11,2779.34
Vendor48,3881,2013-03-12,4526.06
Vendor48,3889,2013-03-12,4747.37
Vendor48,3892,2013-03-12,1400.11
Vendor48,3900,2013-03-13,2533.57
Vendor48,3902,2013-03-19,2635.9
Vendor48,3907,2013-03-20,2341.18
Vendor48,3915,2013-03-26,4143.16
Vendor48,3919,2013-04-05,6893.39
Vendor48,3921,2013-04-15,1353.52
Vendor48,3929,2013-04-25,2350.78
Vendor48,3933,2013-04-26,3578.01
Vendor48,3938,2013-04-26,2192.51
Vendor48,3942,2013-05-02,1120.31
Vendor48,3948,2013-05-02,8134.43
Vendor48,3955,2013-05-02,8209.99
Vendor48,3960,2013-05-02,3415.27
Vendor48,3962,2013-05-02,2509.6
Vendor48,3968,2013-05-12,3021.47
Vendor48,3971,2013-05-13,2065.38
Vendor48,3976,2013-05-13,1504.79
Vendor48,3981,2013-05-19,8641.45
Vendor48,3990,2013-05-19,5593.12
Vendor48,3995,2013-05-25,2805.72
Vendor48,3998,2013-05-25,1614.55
Vendor48,4003,2013-05-25,1470.48
Vendor48,4010,2013-06-04,1576.17
Vendor48,4014,2013-06-10,5265.62
Vendor48,4018,2013-06-16,6473.28
Vendor48,4024,2013-06-16,5470.54
Vendor48,4026,2013-06-24,5979.36
Vendor48,4031,2013-06-25,7577.39
Vendor48,4036,2013-06-25,4287.2
Vendor48,4042,2013-06-25,2674.12
Vendor48,4046,2013-07-01,4344.67
Vendor48,4051,2013-07-01,8734.57
Vendor48,4059,2013-07-02,2439.71
Vendor48,4062,2013-07-03,1391.73
Vendor48,4070,2013-07-03,8859.02
Vendor48,4075,2013-07-09,7954.0
Vendor48,4080,2013-07-17,1780.53
Vendor48,4083,2013-07-23,3170.96
Vendor48,4090,2013-07-23,1441.39
Vendor48,4092,2013-07-23,3077.64
Vendor48,4096,2013-07-29,1715.87
Vendor48,4104,2013-07-30,1759.47
Vendor48,4109,2013-07-31,5940.21
Vendor48,4112,2013-07-31,6064.02
Vendor48,4116,2013-07-31,1358.43
Vendor48,4121,2013-07-31,1162.49
Vendor48,4129,2013-08-10,1493.85
Vendor48,4132,2013-08-16,1106.03
Vendor48,4139,2013-08-16,1858.92
Vendor48,4142,2013-08-24,1689.76
Vendor48,4149,2013-08-25,2669.94
Vendor48,4153,2013-08-25,2740.88
Vendor48,4160,2013-08-31,1618.7
Vendor48,4161,2013-09-10,1825.0
Vendor48,4166,2013-09-16,6298.58
Vendor48,4171,2013-09-16,6570.31
Vendor48,4177,2013-09-17,7014.26
Vendor48,4181,2013-09-17,1835.81
Vendor48,4186,2013-09-25,1152.41
Vendor48,4191,2013-09-25,2134.35
Vendor48,4197,2013-09-25,2407.67
Vendor48,4201,2013-10-01,2634.26
Vendor48,4208,2013-10-11,3816.63
Vendor48,4215,2013-10-17,2081.64
Vendor48,4220,2013-10-17,1259.95
Vendor48,4224,2013-10-17,7515.4
Vendor48,4227,2013-10-18,1178.39
Vendor48,4234,2013-10-26,1876.87
Vendor48,4236,2013-11-03,1405.12
Vendor48,4242,2013-11-03,9904.41
Vendor48,4248,2013-11-13,9866.15
Vendor48,4253,2013-11-19,4868.68
Vendor48,4257,2013-11-25,1814.73
Vendor48,4265,2013-12-01,3753.98
Vendor48,4270,2013-12-01,3811.7
Vendor48,4274,2013-12-07,9577.75
Vendor48,4277,2013-12-07,2784.23
Vendor48,4285,2013-12-15,8869.9
Vendor48,4289,2013-12-21,3354.63
Vendor48,4293,2013-12-27,1560.81
Vendor48,4299,2013-12-27,1231.5
Vendor48,4302,2014-01-04,8345.1
Vendor48,4308,2014-01-10,6528.57
Vendor48,4312,2014-01-11,1797.61
Vendor48,4319,2014-01-21,3591.46
Vendor48,4322,2014-01-27,1992.02
Vendor48,4328,2014-01-28,1033.1
Vendor48,4334,2014-01-29,1213.45
Vendor48,4338,2014-01-30,1578.56
Vendor48,4344,2014-01-30,4785.26
Vendor48,4347,2014-01-30,3156.63
Vendor48,4354,2014-01-30,1074.39
Vendor48,4358,2014-01-31,1621.44
Vendor48,4365,2014-01-31,4296.63
Vendor48,4367,2014-02-06,9052.15
Vendor48,4373,2014-02-07,5725.12
Vendor48,4380,2014-02-13,1113.07
Vendor48,4383,2014-02-21,1390.95
Vendor48,4387,2014-02-21,1242.19
Vendor48,4392,2014-02-22,6359.63
Vendor48,4397,2014-02-22,1614.89
Vendor48,4401,2014-02-22,7612.67
Vendor48,4410,2014-02-22,1485.14
Vendor48,4413,2014-03-04,2472.24
Vendor48,4419,2014-03-05,6375.38
Vendor48,4421,2014-03-11,8485.31
Vendor48,4426,2014-03-17,2392.95
Vendor48,4431,2014-03-18,3555.53
Vendor48,4440,2014-03-24,8900.08
Vendor48,4444,2014-04-01,1097.2
Vendor48,4447,2014-04-01,1439.0
Vendor48,4454,2014-04-02,1252.5
Vendor48,4457,2014-04-12,3356.04
Vendor48,4462,2014-04-20,8344.33
Vendor48,4469,2014-04-20,4755.56
Vendor48,4474,2014-04-21,4238.5
Vendor48,4478,2014-04-21,3686.44
Vendor48,4485,2014-04-27,4341.55
Vendor48,4489,2014-04-28,3330.47
Vendor48,4493,2014-04-28,7676.79
Vendor48,4499,2014-05-08,3080.92
Vendor48,4501,2014-05-08,4175.09
Vendor48,4507,2014-05-16,2943.09
Vendor48,4513,2014-05-22,3058.84
Vendor48,4517,2014-05-23,2659.15
Vendor48,4522,2014-05-23,1871.52
Vendor48,4528,2014-05-23,1402.09
Vendor48,4531,2014-05-23,4177.28
Vendor48,4539,2014-05-24,3058.92
Vendor48,4544,2014-05-24,3379.83
Vendor48,4547,2014-05-24,2304.22
Vendor48,4553,2014-05-25,2744.47
Vendor48,4556,2014-06-04,1115.07
Vendor48,4562,2014-06-05,1691.32
Vendor48,4567,2014-06-15,3659.76
Vendor48,4571,2014-06-15,7173.06
Vendor48,4576,2014-06-21,4003.71
Vendor48,4581,2014-07-01,3243.27
Vendor48,4589,2014-07-09,8902.69
Vendor48,4592,2014-07-10,1888.65
Vendor48,4596,2014-07-11,1077.64
Vendor48,4603,2014-07-12,7229.15
Vendor48,4606,2014-07-18,5082.97
Vendor48,4611,2014-07-28,1764.6
Vendor48,4619,2014-07-28,1660.82
Vendor48,4624,2014-08-03,9439.62
Vendor48,4627,2014-08-04,5100.04
Vendor48,4634,2014-08-04,8195.86
Vendor48,4638,2014-08-04,3740.32
Vendor48,4642,2014-08-05,1118.98
Vendor48,4649,2014-08-05,1478.11
Vendor48,4653,2014-08-05,1441.59
Vendor48,4656,2014-08-05,4249.16
Vendor48,4662,2014-08-06,2252.36
Vendor48,4669,2014-08-14,1382.05
Vendor48,4673,2014-08-20,5011.04
Vendor48,4678,2014-08-28,4797.86
Vendor48,4684,2014-09-07,4114.44
Vendor48,4688,2014-09-17,2755.55
Vendor48,4693,2014-09-25,3770.89
Vendor48,4700,2014-10-03,4422.47
Vendor48,4702,2014-10-11,9792.22
Vendor48,4708,2014-10-12,1433.72
Vendor48,4715,2014-10-13,1560.79
Vendor48,4716,2014-10-14,3131.24
Vendor48,4721,2014-10-24,2947.15
Vendor48,4726,2014-10-24,5284.78
Vendor48,4735,2014-10-25,1585.3
Vendor48,4740,2014-10-31,7934.93
Vendor48,4743,2014-11-06,4430.0
Vendor48,4750,2014-11-14,2826.43
Vendor48,4754,2014-11-14,1286.55
Vendor48,4760,2014-11-20,5061.44
Vendor48,4764,2014-11-20,1871.06
Vendor48,4768,2014-11-21,1757.42
Vendor48,4771,2014-11-21,2234.15
Vendor48,4779,2014-11-21,4841.83
Vendor48,4781,2014-11-27,2924.05
Vendor48,4786,2014-12-03,2328.41
Vendor48,4795,2014-12-03,5856.85
Vendor48,4797,2014-12-03,3066.86
Vendor48,4804,2014-12-09,6293.76
Vendor48,4809,2014-12-10,2362.25
Vendor48,4812,2014-12-10,1661.07
Vendor48,4819,2014-12-18,3348.81
Vendor48,4824,2014-12-24,1128.33
Vendor48,4828,2014-12-30,1744.94
Vendor48,4833,2014-12-30,1235.45
Vendor48,4839,2014-12-31,9165.25
Vendor48,4843,2015-01-01,1814.0
Vendor48,4849,2015-01-09,8938.96
Vendor48,4851,2015-01-09,3288.76
Vendor48,4858,2015-01-09,5673.73
Vendor48,4864,2015-01-19,6230.24
Vendor48,4867,2015-01-20,4589.23
'''  