# Grant Gordon <grantjgordon@gmail.com>
# 10/10
#
# Version 1.0
#
####################################################################################
#                                                                                  
# Copyright (c) 2008 by Grant Gordon                        
#                                                                                  
# Running this Detectlet with the example data yields the following results:
#
# +--------+--------+--------------------+
# | BillID | TestID |     BillAmount     |
# +--------+--------+--------------------+
# |    301 |   1065 | 7177.3400000000001 |
# |    378 |   1142 | 331.79000000000002 |
# |    491 |   1255 | 9885.4099999999999 |
# |    509 |   1273 | 634.33000000000004 |
# |    561 |   1325 | 6645.3699999999999 |
# |    608 |   1372 | 2492.2800000000002 |
# +--------+--------+--------------------+
####################################################################################

DETECTLET_STANDARD = 1.0

from picalo import *

wizard = '''
<wizard>
  <page>
    Select the TABLE containing Test records:
    <parameter type="Table" variable="tbl_tests"/>
    Select the TABLE containing Bill records:
    <parameter type="Table" variable="tbl_bills"/>
  </page>
  <page>
    Select the COLUMN containing Test IDs:
    <parameter type="Column" table="tbl_tests" variable="clm_test_id"/>
    Select the COLUMN containing the dates treatments were performed:
    <parameter type="Column" table="tbl_tests" variable="clm_date_performed"/>
  </page>
  <page>
    Select the COLUMN containing Bill IDs:
    <parameter type="Column" table="tbl_bills" variable="clm_bill_id"/>
    Select the COLUMN containing the Bill Test IDs:
    <parameter type="Column" table="tbl_bills" variable="clm_bill_testid"/>
  </page>
  <page>
    Select the COLUMN containing Bill Amounts:
    <parameter type="Column" table="tbl_bills" variable="clm_bill_amt"/>
  </page>
</wizard>
'''


RESULTS_COUNT = 0


def run(tbl_tests, 
        tbl_bills,
        clm_test_id,
        clm_date_performed, 
        clm_bill_id,
        clm_bill_testid,
        clm_bill_amt):
    '''Sometimes doctors try to extract money from insurance companies or the goverment
        by billing for tests they prescribed but never performed.\n
        This detectlet searches for such incidences.
    '''
    
    RESULTS_TEXT = '''\
        No tests were billed but not performed.
    '''
    
    tbl_results = Table([
        ('BillID', int),
        ('TestID', int),
        ('BillAmount', float),
    ])
    
    #make copies of the tables so as to not modify the original data
    wrk_tbl_tests = tbl_tests[:]
    wrk_tbl_bills = tbl_bills[:]
    
    # Get list of medical tests prescribed but never performed
    tbl_not_performed = Table([
        (clm_test_id, int),
        ('RecNum', int),
    ])
    for n, rec in enumerate(wrk_tbl_tests):
        if type(rec[clm_date_performed]) != type(DateTime()):
            tbl_not_performed.append(rec[clm_test_id], n)

    # Search for bills that include those tests
    for rec in wrk_tbl_bills:
        for test in tbl_not_performed:
            if rec[clm_bill_testid] == test[clm_test_id]:
                tbl_results.append(rec[clm_bill_id],test[clm_test_id],rec[clm_bill_amt])
                RESULTS_TEXT = '''\
                    <p>It appears that tests that have not been performed have been billed.</p>
                    <p>The table below shows the ID of the Bill in question, the ID of the Test
                    in question, and the Amount of the Bill in question.</p>
                    <p>This warrants further investigation.</p>
                '''
        
    return tbl_results, RESULTS_TEXT

csv_bills = '''\
BillID,TestID,DateBilled,Amount,
237,1001,5/10/2006,3221.93,
238,1002,5/13/2006,1923.22,
239,1003,5/14/2006,1608.47,
240,1004,5/13/2006,2171.78,
241,1005,5/14/2006,2881.98,
242,1006,5/14/2006,4968.36,
243,1007,5/16/2006,9606.53,
244,1008,5/18/2006,2468.24,
245,1009,5/18/2006,3400.20,
246,1010,5/17/2006,1924.00,
247,1011,5/21/2006,7301.07,
248,1012,5/18/2006,3839.61,
249,1013,5/29/2006,1602.89,
250,1014,5/30/2006,1641.34,
251,1015,5/31/2006,4713.46,
252,1016,6/2/2006,9024.90,
253,1017,6/1/2006,9131.77,
254,1018,6/2/2006,9414.35,
255,1019,6/6/2006,7146.94,
256,1020,6/7/2006,8323.56,
257,1021,6/9/2006,6264.28,
258,1022,6/13/2006,5252.86,
259,1023,6/14/2006,2277.38,
260,1024,6/12/2006,3675.96,
261,1025,6/15/2006,887.93,
262,1026,6/16/2006,9789.82,
263,1027,6/19/2006,9837.35,
264,1028,6/21/2006,9015.54,
265,1029,6/18/2006,372.98,
266,1030,6/21/2006,6852.03,
267,1031,6/24/2006,6183.90,
268,1032,6/25/2006,7401.74,
269,1033,6/25/2006,9647.15,
270,1034,6/24/2006,3733.78,
271,1035,6/25/2006,3718.55,
272,1036,6/28/2006,5883.53,
273,1037,6/28/2006,5755.19,
274,1038,6/29/2006,762.28,
275,1039,6/30/2006,2108.32,
276,1040,6/30/2006,9298.61,
277,1041,6/27/2006,6181.43,
278,1042,6/29/2006,8355.56,
279,1043,7/1/2006,1512.93,
280,1044,6/29/2006,5859.96,
281,1045,6/30/2006,6814.93,
282,1046,7/3/2006,272.90,
283,1047,7/5/2006,3733.78,
284,1048,7/7/2006,2673.06,
285,1049,7/10/2006,4328.42,
286,1050,7/7/2006,7557.22,
287,1051,7/10/2006,6580.19,
288,1052,7/7/2006,8133.70,
289,1053,7/9/2006,688.92,
290,1054,7/9/2006,660.32,
291,1055,7/12/2006,9830.00,
292,1056,7/12/2006,4979.83,
293,1057,7/11/2006,2809.92,
294,1058,7/11/2006,3963.13,
295,1059,7/11/2006,2759.63,
296,1060,7/14/2006,7571.21,
297,1061,7/14/2006,8840.07,
298,1062,7/16/2006,3353.91,
299,1063,7/14/2006,3736.36,
300,1064,7/13/2006,1133.13,
301,1065,1/15/1900,7177.34,
302,1066,7/18/2006,1625.16,
303,1067,7/17/2006,2347.07,
304,1068,7/20/2006,8314.03,
305,1069,7/18/2006,1818.05,
306,1070,7/24/2006,7751.99,
307,1071,7/21/2006,3096.55,
308,1072,7/23/2006,5254.43,
309,1073,7/26/2006,8869.83,
310,1074,7/25/2006,9677.07,
311,1075,7/29/2006,6783.98,
312,1076,7/25/2006,1448.13,
313,1077,8/1/2006,9789.82,
314,1078,7/31/2006,4898.90,
315,1079,7/31/2006,4657.86,
316,1080,8/4/2006,8713.66,
317,1081,8/2/2006,5474.97,
318,1082,8/5/2006,3823.27,
319,1083,8/7/2006,658.34,
320,1084,8/8/2006,331.79,
321,1085,8/8/2006,4988.96,
322,1086,8/11/2006,6580.19,
323,1087,8/12/2006,8891.58,
324,1088,8/15/2006,6001.58,
325,1089,8/14/2006,2456.17,
326,1090,8/17/2006,8869.56,
327,1091,8/21/2006,2218.87,
328,1092,8/20/2006,2079.87,
329,1093,8/19/2006,7649.79,
330,1094,8/23/2006,3534.35,
331,1095,8/25/2006,3679.93,
332,1096,8/25/2006,2925.98,
333,1097,8/26/2006,7403.28,
334,1098,8/27/2006,9298.61,
335,1099,8/31/2006,3685.32,
336,1100,9/1/2006,4686.77,
337,1101,9/1/2006,2767.20,
338,1102,9/1/2006,5694.09,
339,1103,8/31/2006,7076.73,
340,1104,8/30/2006,836.38,
341,1105,9/3/2006,3269.90,
342,1106,9/4/2006,7848.82,
343,1107,9/2/2006,3215.47,
344,1108,9/4/2006,6597.26,
345,1109,9/6/2006,9182.57,
346,1110,9/9/2006,9885.41,
347,1111,9/10/2006,8434.06,
348,1112,9/11/2006,9237.29,
349,1113,9/9/2006,1351.97,
350,1114,9/14/2006,5758.11,
351,1115,9/15/2006,7753.10,
352,1116,9/11/2006,2516.67,
353,1117,9/17/2006,2397.38,
354,1118,9/15/2006,660.32,
355,1119,9/16/2006,6844.46,
356,1120,9/15/2006,5781.35,
357,1121,9/20/2006,7066.07,
358,1122,9/21/2006,892.29,
359,1123,9/23/2006,2702.69,
360,1124,9/20/2006,7418.01,
361,1125,9/21/2006,488.82,
362,1126,9/19/2006,9525.48,
363,1127,9/23/2006,4345.79,
364,1128,9/24/2006,1831.50,
365,1129,9/22/2006,3749.74,
366,1130,9/24/2006,6333.25,
367,1131,9/26/2006,6777.48,
368,1132,9/28/2006,5707.45,
369,1133,10/1/2006,4687.35,
370,1134,9/30/2006,5694.09,
371,1135,10/3/2006,7851.21,
372,1136,10/2/2006,2205.23,
373,1137,10/3/2006,6001.58,
374,1138,9/30/2006,2355.64,
375,1139,10/3/2006,4345.79,
376,1140,10/5/2006,653.70,
377,1141,10/5/2006,2695.96,
378,1142,1/15/1900,331.79,
379,1143,10/13/2006,5453.67,
380,1144,10/16/2006,4065.55,
381,1145,10/16/2006,3620.74,
382,1146,10/19/2006,3314.21,
383,1147,10/18/2006,8434.06,
384,1148,10/20/2006,4657.86,
385,1149,10/18/2006,9361.88,
386,1150,10/22/2006,3826.34,
387,1151,10/23/2006,5028.37,
388,1152,10/24/2006,3823.27,
389,1153,10/22/2006,8067.69,
390,1154,10/24/2006,2397.38,
391,1155,10/23/2006,7092.62,
392,1156,10/24/2006,3716.65,
393,1157,10/27/2006,7418.01,
394,1158,10/25/2006,6638.96,
395,1159,10/25/2006,6090.37,
396,1160,10/27/2006,9529.34,
397,1161,10/30/2006,4164.26,
398,1162,11/1/2006,5883.53,
399,1163,10/28/2006,704.63,
400,1164,11/1/2006,3277.02,
401,1165,10/29/2006,8741.19,
402,1166,11/3/2006,2205.30,
403,1167,11/5/2006,1919.11,
404,1168,11/2/2006,8105.88,
405,1169,11/5/2006,1011.63,
406,1170,11/3/2006,306.08,
407,1171,11/6/2006,5423.02,
408,1172,11/7/2006,679.74,
409,1173,11/9/2006,3364.61,
410,1174,11/6/2006,4579.06,
411,1175,11/7/2006,9126.34,
412,1176,11/9/2006,5109.66,
413,1177,11/12/2006,5737.95,
414,1178,11/10/2006,3323.25,
415,1179,11/10/2006,5907.58,
416,1180,11/12/2006,3450.24,
417,1181,11/12/2006,8067.69,
418,1182,11/13/2006,6852.03,
419,1183,11/16/2006,5758.11,
420,1184,11/14/2006,892.29,
421,1185,11/17/2006,3017.75,
422,1186,11/14/2006,3037.86,
423,1187,11/18/2006,3826.34,
424,1188,11/19/2006,7571.21,
425,1189,11/21/2006,1188.64,
426,1190,11/21/2006,1718.22,
427,1191,11/18/2006,1448.13,
428,1192,11/22/2006,2394.20,
429,1193,11/23/2006,4995.40,
430,1194,11/24/2006,5189.84,
431,1195,11/25/2006,3233.41,
432,1196,11/25/2006,1080.78,
433,1197,11/25/2006,678.71,
434,1198,11/23/2006,887.24,
435,1199,11/27/2006,8741.19,
436,1200,11/29/2006,1034.75,
437,1201,11/28/2006,5254.43,
438,1202,12/1/2006,2936.90,
439,1203,12/1/2006,8293.90,
440,1204,12/1/2006,9501.34,
441,1205,12/3/2006,2195.26,
442,1206,12/1/2006,5781.02,
443,1207,12/6/2006,460.85,
444,1208,12/11/2006,4815.80,
445,1209,12/10/2006,775.03,
446,1210,12/9/2006,7146.94,
447,1211,12/9/2006,9183.43,
448,1212,12/13/2006,1719.19,
449,1213,12/13/2006,7066.07,
450,1214,12/15/2006,7142.03,
451,1215,12/16/2006,5515.79,
452,1216,12/18/2006,922.29,
453,1217,12/16/2006,4920.77,
454,1218,12/16/2006,3052.11,
455,1219,12/17/2006,3107.57,
456,1220,12/20/2006,3679.93,
457,1221,12/19/2006,8361.62,
458,1222,12/23/2006,3122.78,
459,1223,12/23/2006,2706.66,
460,1224,12/24/2006,2694.30,
461,1225,12/24/2006,2759.63,
462,1226,12/25/2006,9249.63,
463,1227,12/26/2006,9081.56,
464,1228,12/28/2006,4638.39,
465,1229,12/26/2006,1516.40,
466,1230,12/28/2006,4968.36,
467,1231,12/30/2006,6491.86,
468,1232,12/29/2006,4568.64,
469,1233,1/2/2007,9732.11,
470,1234,1/5/2007,6608.12,
471,1235,1/6/2007,8972.48,
472,1236,1/3/2007,716.39,
473,1237,1/7/2007,5165.58,
474,1238,1/8/2007,9229.81,
475,1239,1/11/2007,8936.66,
476,1240,1/9/2007,5357.07,
477,1241,1/11/2007,9860.51,
478,1242,1/8/2007,815.96,
479,1243,1/9/2007,7652.73,
480,1244,1/13/2007,6580.19,
481,1245,1/11/2007,2094.29,
482,1246,1/10/2007,7583.12,
483,1247,1/13/2007,1512.93,
484,1248,1/14/2007,8945.45,
485,1249,1/17/2007,9244.00,
486,1250,1/19/2007,4781.91,
487,1251,1/19/2007,3482.49,
488,1252,1/17/2007,9391.74,
489,1253,1/18/2007,5357.74,
490,1254,1/20/2007,2771.15,
491,1255,1/15/1900,9885.41,
492,1256,1/21/2007,5236.31,
493,1257,1/23/2007,4568.64,
494,1258,1/24/2007,1607.19,
495,1259,1/22/2007,4581.52,
496,1260,1/23/2007,7602.51,
497,1261,1/26/2007,2706.66,
498,1262,1/27/2007,8851.63,
499,1263,1/24/2007,5828.67,
500,1264,1/25/2007,1387.87,
501,1265,1/26/2007,7795.57,
502,1266,1/26/2007,8372.47,
503,1267,1/26/2007,6185.16,
504,1268,1/25/2007,9298.61,
505,1269,1/27/2007,6814.93,
506,1270,1/26/2007,7546.85,
507,1271,1/30/2007,8425.18,
508,1272,1/28/2007,9015.54,
509,1273,1/15/1900,634.33,
510,1274,1/31/2007,6036.09,
511,1275,2/3/2007,3093.81,
512,1276,2/5/2007,2707.72,
513,1277,2/3/2007,1178.07,
514,1278,2/6/2007,7401.74,
515,1279,2/5/2007,6682.86,
516,1280,2/8/2007,4670.74,
517,1281,2/11/2007,7598.87,
518,1282,2/9/2007,483.25,
519,1283,2/10/2007,1210.06,
520,1284,2/14/2007,5565.54,
521,1285,2/14/2007,5636.19,
522,1286,2/13/2007,8521.59,
523,1287,2/10/2007,1192.09,
524,1288,2/13/2007,3610.27,
525,1289,2/13/2007,887.24,
526,1290,2/14/2007,4687.35,
527,1291,2/15/2007,6638.96,
528,1292,2/14/2007,5621.44,
529,1293,2/18/2007,5737.95,
530,1294,2/15/2007,8361.62,
531,1295,2/18/2007,552.71,
532,1296,2/15/2007,1355.15,
533,1297,2/16/2007,3122.78,
534,1298,2/19/2007,8533.82,
535,1299,2/19/2007,6101.88,
536,1300,2/20/2007,8128.62,
537,1301,2/23/2007,9830.00,
538,1302,2/24/2007,9675.65,
539,1303,2/24/2007,1608.47,
540,1304,2/26/2007,6813.22,
541,1305,3/1/2007,3685.32,
542,1306,2/28/2007,8860.01,
543,1307,3/1/2007,3096.55,
544,1308,3/2/2007,2091.24,
545,1309,2/28/2007,1100.89,
546,1310,3/3/2007,3134.26,
547,1311,3/5/2007,4661.55,
548,1312,3/3/2007,4483.59,
549,1313,3/2/2007,2492.28,
550,1314,3/4/2007,9539.10,
551,1315,3/5/2007,9183.43,
552,1316,3/4/2007,2047.04,
553,1317,3/5/2007,7143.91,
554,1318,3/9/2007,522.68,
555,1319,3/7/2007,3679.93,
556,1320,3/9/2007,1469.07,
557,1321,3/8/2007,850.91,
558,1322,3/9/2007,1900.38,
559,1323,3/10/2007,7512.79,
560,1324,3/12/2007,8605.06,
561,1325,1/15/1900,6645.37,
562,1326,3/13/2007,8024.61,
563,1327,3/11/2007,369.06,
564,1328,3/15/2007,1081.91,
565,1329,3/17/2007,678.71,
566,1330,3/16/2007,1223.37,
567,1331,3/17/2007,7744.03,
568,1332,3/17/2007,3438.14,
569,1333,3/17/2007,3093.81,
570,1334,3/19/2007,856.04,
571,1335,3/19/2007,4638.39,
572,1336,3/20/2007,3480.19,
573,1337,3/21/2007,3269.95,
574,1338,3/20/2007,8721.01,
575,1339,3/25/2007,6379.51,
576,1340,3/25/2007,2879.84,
577,1341,3/24/2007,2277.38,
578,1342,3/27/2007,3450.24,
579,1343,3/28/2007,7082.96,
580,1344,3/26/2007,766.23,
581,1345,3/26/2007,836.04,
582,1346,3/26/2007,8357.42,
583,1347,3/29/2007,3934.06,
584,1348,3/29/2007,9643.88,
585,1349,4/2/2007,9303.42,
586,1350,4/1/2007,5815.29,
587,1351,3/31/2007,497.76,
588,1352,3/31/2007,7583.12,
589,1353,4/2/2007,9231.67,
590,1354,4/3/2007,5423.02,
591,1355,4/1/2007,1206.75,
592,1356,4/3/2007,9852.55,
593,1357,4/5/2007,6379.51,
594,1358,4/4/2007,7751.99,
595,1359,4/9/2007,9767.79,
596,1360,4/5/2007,9182.57,
597,1361,4/7/2007,4518.78,
598,1362,4/9/2007,2361.56,
599,1363,4/11/2007,5423.02,
600,1364,4/11/2007,8681.62,
601,1365,4/12/2007,2915.99,
602,1366,4/12/2007,2695.52,
603,1367,4/11/2007,4981.60,
604,1368,4/12/2007,4283.96,
605,1369,4/16/2007,5474.97,
606,1370,4/17/2007,8512.20,
607,1371,4/19/2007,1802.77,
608,1372,1/15/1900,2492.28,
609,1373,4/21/2007,5907.58,
610,1374,4/22/2007,9160.66,
611,1375,4/24/2007,2399.22,
612,1376,4/25/2007,2266.99,
613,1377,4/27/2007,5612.54,
614,1378,4/25/2007,9219.13,
615,1379,4/25/2007,449.56,
616,1380,4/29/2007,684.91,
617,1381,4/30/2007,306.08,
618,1382,4/28/2007,1816.47,
619,1383,4/27/2007,5515.79,
620,1384,5/2/2007,7111.20,
621,1385,4/29/2007,8419.72,
622,1386,5/6/2007,4961.23,
623,1387,5/4/2007,8428.83,
624,1388,5/7/2007,1684.37,
625,1389,5/9/2007,1318.40,
626,1390,5/6/2007,4483.62,
627,1391,5/5/2007,7429.60,
628,1392,5/7/2007,6326.34,
629,1393,5/6/2007,2227.24,
630,1394,5/6/2007,9529.34,
631,1395,5/8/2007,1044.57,
632,1396,5/9/2007,3450.55,
633,1397,5/7/2007,8936.66,
634,1398,5/9/2007,6645.37,
'''

csv_tests = '''\
TestID,DatePrescribed,DatePerformed,DrID,Description
1001,4/25/2006,4/25/2006,337,A Medical Test of Sorts
1002,4/25/2006,4/28/2006,342,A Medical Test of Sorts
1003,4/25/2006,4/29/2006,339,A Medical Test of Sorts
1004,4/25/2006,4/28/2006,334,A Medical Test of Sorts
1005,4/26/2006,4/29/2006,334,A Medical Test of Sorts
1006,4/27/2006,4/29/2006,337,A Medical Test of Sorts
1007,4/29/2006,5/1/2006,342,A Medical Test of Sorts
1008,4/29/2006,5/3/2006,342,A Medical Test of Sorts
1009,5/2/2006,5/3/2006,332,A Medical Test of Sorts
1010,5/2/2006,5/2/2006,333,A Medical Test of Sorts
1011,5/2/2006,5/6/2006,341,A Medical Test of Sorts
1012,5/3/2006,5/3/2006,341,A Medical Test of Sorts
1013,5/11/2006,5/14/2006,341,A Medical Test of Sorts
1014,5/13/2006,5/15/2006,332,A Medical Test of Sorts
1015,5/14/2006,5/16/2006,332,A Medical Test of Sorts
1016,5/15/2006,5/18/2006,334,A Medical Test of Sorts
1017,5/17/2006,5/17/2006,332,A Medical Test of Sorts
1018,5/17/2006,5/18/2006,342,A Medical Test of Sorts
1019,5/20/2006,5/22/2006,332,A Medical Test of Sorts
1020,5/23/2006,5/23/2006,339,A Medical Test of Sorts
1021,5/23/2006,5/25/2006,335,A Medical Test of Sorts
1022,5/27/2006,5/29/2006,339,A Medical Test of Sorts
1023,5/27/2006,5/30/2006,340,A Medical Test of Sorts
1024,5/28/2006,5/28/2006,334,A Medical Test of Sorts
1025,5/29/2006,5/31/2006,334,A Medical Test of Sorts
1026,6/1/2006,6/1/2006,334,A Medical Test of Sorts
1027,6/2/2006,6/4/2006,335,A Medical Test of Sorts
1028,6/2/2006,6/6/2006,339,A Medical Test of Sorts
1029,6/3/2006,6/3/2006,338,A Medical Test of Sorts
1030,6/4/2006,6/6/2006,332,A Medical Test of Sorts
1031,6/5/2006,6/9/2006,338,A Medical Test of Sorts
1032,6/6/2006,6/10/2006,337,A Medical Test of Sorts
1033,6/6/2006,6/10/2006,337,A Medical Test of Sorts
1034,6/7/2006,6/9/2006,336,A Medical Test of Sorts
1035,6/9/2006,6/10/2006,334,A Medical Test of Sorts
1036,6/9/2006,6/13/2006,332,A Medical Test of Sorts
1037,6/10/2006,6/13/2006,337,A Medical Test of Sorts
1038,6/11/2006,6/14/2006,333,A Medical Test of Sorts
1039,6/11/2006,6/15/2006,341,A Medical Test of Sorts
1040,6/12/2006,6/15/2006,338,A Medical Test of Sorts
1041,6/12/2006,6/12/2006,341,A Medical Test of Sorts
1042,6/13/2006,6/14/2006,333,A Medical Test of Sorts
1043,6/14/2006,6/16/2006,335,A Medical Test of Sorts
1044,6/14/2006,6/14/2006,338,A Medical Test of Sorts
1045,6/14/2006,6/15/2006,333,A Medical Test of Sorts
1046,6/15/2006,6/18/2006,339,A Medical Test of Sorts
1047,6/18/2006,6/20/2006,336,A Medical Test of Sorts
1048,6/20/2006,6/22/2006,340,A Medical Test of Sorts
1049,6/21/2006,6/25/2006,340,A Medical Test of Sorts
1050,6/21/2006,6/22/2006,341,A Medical Test of Sorts
1051,6/21/2006,6/25/2006,333,A Medical Test of Sorts
1052,6/21/2006,6/22/2006,335,A Medical Test of Sorts
1053,6/21/2006,6/24/2006,340,A Medical Test of Sorts
1054,6/22/2006,6/24/2006,339,A Medical Test of Sorts
1055,6/23/2006,6/27/2006,337,A Medical Test of Sorts
1056,6/24/2006,6/27/2006,332,A Medical Test of Sorts
1057,6/24/2006,6/26/2006,337,A Medical Test of Sorts
1058,6/26/2006,6/26/2006,337,A Medical Test of Sorts
1059,6/26/2006,6/26/2006,336,A Medical Test of Sorts
1060,6/26/2006,6/29/2006,336,A Medical Test of Sorts
1061,6/27/2006,6/29/2006,342,A Medical Test of Sorts
1062,6/27/2006,7/1/2006,340,A Medical Test of Sorts
1063,6/27/2006,6/29/2006,340,A Medical Test of Sorts
1064,6/27/2006,6/28/2006,342,A Medical Test of Sorts
1065,6/29/2006,,336,A Medical Test of Sorts
1066,6/29/2006,7/3/2006,336,A Medical Test of Sorts
1067,6/30/2006,7/2/2006,338,A Medical Test of Sorts
1068,7/1/2006,7/5/2006,337,A Medical Test of Sorts
1069,7/2/2006,7/3/2006,337,A Medical Test of Sorts
1070,7/5/2006,7/9/2006,341,A Medical Test of Sorts
1071,7/5/2006,7/6/2006,338,A Medical Test of Sorts
1072,7/7/2006,7/8/2006,341,A Medical Test of Sorts
1073,7/8/2006,7/11/2006,340,A Medical Test of Sorts
1074,7/9/2006,7/10/2006,336,A Medical Test of Sorts
1075,7/10/2006,7/14/2006,341,A Medical Test of Sorts
1076,7/10/2006,7/10/2006,335,A Medical Test of Sorts
1077,7/14/2006,7/17/2006,332,A Medical Test of Sorts
1078,7/16/2006,7/16/2006,332,A Medical Test of Sorts
1079,7/16/2006,7/16/2006,334,A Medical Test of Sorts
1080,7/18/2006,7/20/2006,332,A Medical Test of Sorts
1081,7/18/2006,7/18/2006,340,A Medical Test of Sorts
1082,7/20/2006,7/21/2006,333,A Medical Test of Sorts
1083,7/20/2006,7/23/2006,333,A Medical Test of Sorts
1084,7/21/2006,7/24/2006,332,A Medical Test of Sorts
1085,7/22/2006,7/24/2006,335,A Medical Test of Sorts
1086,7/26/2006,7/27/2006,341,A Medical Test of Sorts
1087,7/27/2006,7/28/2006,338,A Medical Test of Sorts
1088,7/28/2006,7/31/2006,338,A Medical Test of Sorts
1089,7/29/2006,7/30/2006,339,A Medical Test of Sorts
1090,8/2/2006,8/2/2006,340,A Medical Test of Sorts
1091,8/3/2006,8/6/2006,339,A Medical Test of Sorts
1092,8/3/2006,8/5/2006,340,A Medical Test of Sorts
1093,8/4/2006,8/4/2006,338,A Medical Test of Sorts
1094,8/5/2006,8/8/2006,337,A Medical Test of Sorts
1095,8/6/2006,8/10/2006,341,A Medical Test of Sorts
1096,8/7/2006,8/10/2006,340,A Medical Test of Sorts
1097,8/8/2006,8/11/2006,342,A Medical Test of Sorts
1098,8/8/2006,8/12/2006,338,A Medical Test of Sorts
1099,8/12/2006,8/16/2006,340,A Medical Test of Sorts
1100,8/13/2006,8/17/2006,334,A Medical Test of Sorts
1101,8/13/2006,8/17/2006,339,A Medical Test of Sorts
1102,8/13/2006,8/17/2006,335,A Medical Test of Sorts
1103,8/13/2006,8/16/2006,342,A Medical Test of Sorts
1104,8/13/2006,8/15/2006,337,A Medical Test of Sorts
1105,8/16/2006,8/19/2006,339,A Medical Test of Sorts
1106,8/18/2006,8/20/2006,335,A Medical Test of Sorts
1107,8/18/2006,8/18/2006,342,A Medical Test of Sorts
1108,8/19/2006,8/20/2006,338,A Medical Test of Sorts
1109,8/19/2006,8/22/2006,341,A Medical Test of Sorts
1110,8/21/2006,8/25/2006,342,A Medical Test of Sorts
1111,8/23/2006,8/26/2006,340,A Medical Test of Sorts
1112,8/24/2006,8/27/2006,339,A Medical Test of Sorts
1113,8/25/2006,8/25/2006,335,A Medical Test of Sorts
1114,8/26/2006,8/30/2006,332,A Medical Test of Sorts
1115,8/27/2006,8/31/2006,336,A Medical Test of Sorts
1116,8/27/2006,8/27/2006,338,A Medical Test of Sorts
1117,8/29/2006,9/2/2006,337,A Medical Test of Sorts
1118,8/30/2006,8/31/2006,335,A Medical Test of Sorts
1119,8/31/2006,9/1/2006,336,A Medical Test of Sorts
1120,8/31/2006,8/31/2006,333,A Medical Test of Sorts
1121,9/3/2006,9/5/2006,342,A Medical Test of Sorts
1122,9/3/2006,9/6/2006,332,A Medical Test of Sorts
1123,9/4/2006,9/8/2006,339,A Medical Test of Sorts
1124,9/4/2006,9/5/2006,342,A Medical Test of Sorts
1125,9/4/2006,9/6/2006,338,A Medical Test of Sorts
1126,9/4/2006,9/4/2006,336,A Medical Test of Sorts
1127,9/5/2006,9/8/2006,339,A Medical Test of Sorts
1128,9/5/2006,9/9/2006,341,A Medical Test of Sorts
1129,9/6/2006,9/7/2006,338,A Medical Test of Sorts
1130,9/7/2006,9/9/2006,335,A Medical Test of Sorts
1131,9/9/2006,9/11/2006,339,A Medical Test of Sorts
1132,9/10/2006,9/13/2006,334,A Medical Test of Sorts
1133,9/13/2006,9/16/2006,332,A Medical Test of Sorts
1134,9/13/2006,9/15/2006,337,A Medical Test of Sorts
1135,9/14/2006,9/18/2006,337,A Medical Test of Sorts
1136,9/14/2006,9/17/2006,342,A Medical Test of Sorts
1137,9/14/2006,9/18/2006,340,A Medical Test of Sorts
1138,9/15/2006,9/15/2006,339,A Medical Test of Sorts
1139,9/18/2006,9/18/2006,336,A Medical Test of Sorts
1140,9/18/2006,9/20/2006,342,A Medical Test of Sorts
1141,9/19/2006,9/20/2006,332,A Medical Test of Sorts
1142,9/22/2006,,336,A Medical Test of Sorts
1143,9/24/2006,9/28/2006,339,A Medical Test of Sorts
1144,9/28/2006,10/1/2006,336,A Medical Test of Sorts
1145,9/28/2006,10/1/2006,341,A Medical Test of Sorts
1146,10/1/2006,10/4/2006,339,A Medical Test of Sorts
1147,10/1/2006,10/3/2006,341,A Medical Test of Sorts
1148,10/2/2006,10/5/2006,337,A Medical Test of Sorts
1149,10/2/2006,10/3/2006,337,A Medical Test of Sorts
1150,10/3/2006,10/7/2006,339,A Medical Test of Sorts
1151,10/4/2006,10/8/2006,340,A Medical Test of Sorts
1152,10/6/2006,10/9/2006,332,A Medical Test of Sorts
1153,10/7/2006,10/7/2006,334,A Medical Test of Sorts
1154,10/7/2006,10/9/2006,336,A Medical Test of Sorts
1155,10/7/2006,10/8/2006,342,A Medical Test of Sorts
1156,10/7/2006,10/9/2006,342,A Medical Test of Sorts
1157,10/8/2006,10/12/2006,334,A Medical Test of Sorts
1158,10/8/2006,10/10/2006,339,A Medical Test of Sorts
1159,10/9/2006,10/10/2006,342,A Medical Test of Sorts
1160,10/9/2006,10/12/2006,339,A Medical Test of Sorts
1161,10/11/2006,10/15/2006,342,A Medical Test of Sorts
1162,10/13/2006,10/17/2006,340,A Medical Test of Sorts
1163,10/13/2006,10/13/2006,333,A Medical Test of Sorts
1164,10/14/2006,10/17/2006,336,A Medical Test of Sorts
1165,10/14/2006,10/14/2006,338,A Medical Test of Sorts
1166,10/17/2006,10/19/2006,333,A Medical Test of Sorts
1167,10/18/2006,10/21/2006,333,A Medical Test of Sorts
1168,10/18/2006,10/18/2006,337,A Medical Test of Sorts
1169,10/19/2006,10/21/2006,341,A Medical Test of Sorts
1170,10/19/2006,10/19/2006,338,A Medical Test of Sorts
1171,10/19/2006,10/22/2006,334,A Medical Test of Sorts
1172,10/20/2006,10/23/2006,334,A Medical Test of Sorts
1173,10/21/2006,10/25/2006,335,A Medical Test of Sorts
1174,10/22/2006,10/22/2006,335,A Medical Test of Sorts
1175,10/23/2006,10/23/2006,337,A Medical Test of Sorts
1176,10/24/2006,10/25/2006,339,A Medical Test of Sorts
1177,10/24/2006,10/28/2006,338,A Medical Test of Sorts
1178,10/25/2006,10/26/2006,340,A Medical Test of Sorts
1179,10/25/2006,10/26/2006,337,A Medical Test of Sorts
1180,10/25/2006,10/28/2006,340,A Medical Test of Sorts
1181,10/26/2006,10/28/2006,334,A Medical Test of Sorts
1182,10/28/2006,10/29/2006,333,A Medical Test of Sorts
1183,10/28/2006,11/1/2006,336,A Medical Test of Sorts
1184,10/28/2006,10/30/2006,333,A Medical Test of Sorts
1185,10/29/2006,11/2/2006,338,A Medical Test of Sorts
1186,10/30/2006,10/30/2006,335,A Medical Test of Sorts
1187,11/1/2006,11/3/2006,339,A Medical Test of Sorts
1188,11/1/2006,11/4/2006,334,A Medical Test of Sorts
1189,11/3/2006,11/6/2006,334,A Medical Test of Sorts
1190,11/3/2006,11/6/2006,337,A Medical Test of Sorts
1191,11/3/2006,11/3/2006,338,A Medical Test of Sorts
1192,11/4/2006,11/7/2006,341,A Medical Test of Sorts
1193,11/6/2006,11/8/2006,337,A Medical Test of Sorts
1194,11/6/2006,11/9/2006,342,A Medical Test of Sorts
1195,11/6/2006,11/10/2006,338,A Medical Test of Sorts
1196,11/6/2006,11/10/2006,341,A Medical Test of Sorts
1197,11/7/2006,11/10/2006,340,A Medical Test of Sorts
1198,11/8/2006,11/8/2006,334,A Medical Test of Sorts
1199,11/9/2006,11/12/2006,336,A Medical Test of Sorts
1200,11/11/2006,11/14/2006,337,A Medical Test of Sorts
1201,11/11/2006,11/13/2006,332,A Medical Test of Sorts
1202,11/14/2006,11/16/2006,336,A Medical Test of Sorts
1203,11/15/2006,11/16/2006,332,A Medical Test of Sorts
1204,11/15/2006,11/16/2006,332,A Medical Test of Sorts
1205,11/15/2006,11/18/2006,338,A Medical Test of Sorts
1206,11/16/2006,11/16/2006,339,A Medical Test of Sorts
1207,11/18/2006,11/21/2006,341,A Medical Test of Sorts
1208,11/22/2006,11/26/2006,332,A Medical Test of Sorts
1209,11/23/2006,11/25/2006,340,A Medical Test of Sorts
1210,11/23/2006,11/24/2006,341,A Medical Test of Sorts
1211,11/24/2006,11/24/2006,332,A Medical Test of Sorts
1212,11/26/2006,11/28/2006,333,A Medical Test of Sorts
1213,11/27/2006,11/28/2006,337,A Medical Test of Sorts
1214,11/27/2006,11/30/2006,333,A Medical Test of Sorts
1215,11/29/2006,12/1/2006,335,A Medical Test of Sorts
1216,11/30/2006,12/3/2006,338,A Medical Test of Sorts
1217,11/30/2006,12/1/2006,339,A Medical Test of Sorts
1218,12/1/2006,12/1/2006,336,A Medical Test of Sorts
1219,12/2/2006,12/2/2006,333,A Medical Test of Sorts
1220,12/4/2006,12/5/2006,342,A Medical Test of Sorts
1221,12/4/2006,12/4/2006,335,A Medical Test of Sorts
1222,12/5/2006,12/8/2006,333,A Medical Test of Sorts
1223,12/5/2006,12/8/2006,337,A Medical Test of Sorts
1224,12/5/2006,12/9/2006,337,A Medical Test of Sorts
1225,12/6/2006,12/9/2006,341,A Medical Test of Sorts
1226,12/9/2006,12/10/2006,337,A Medical Test of Sorts
1227,12/11/2006,12/11/2006,336,A Medical Test of Sorts
1228,12/11/2006,12/13/2006,342,A Medical Test of Sorts
1229,12/11/2006,12/11/2006,341,A Medical Test of Sorts
1230,12/12/2006,12/13/2006,338,A Medical Test of Sorts
1231,12/12/2006,12/15/2006,338,A Medical Test of Sorts
1232,12/13/2006,12/14/2006,333,A Medical Test of Sorts
1233,12/17/2006,12/18/2006,332,A Medical Test of Sorts
1234,12/18/2006,12/21/2006,339,A Medical Test of Sorts
1235,12/18/2006,12/22/2006,337,A Medical Test of Sorts
1236,12/19/2006,12/19/2006,339,A Medical Test of Sorts
1237,12/20/2006,12/23/2006,332,A Medical Test of Sorts
1238,12/21/2006,12/24/2006,336,A Medical Test of Sorts
1239,12/23/2006,12/27/2006,337,A Medical Test of Sorts
1240,12/23/2006,12/25/2006,339,A Medical Test of Sorts
1241,12/23/2006,12/27/2006,339,A Medical Test of Sorts
1242,12/24/2006,12/24/2006,337,A Medical Test of Sorts
1243,12/25/2006,12/25/2006,342,A Medical Test of Sorts
1244,12/25/2006,12/29/2006,340,A Medical Test of Sorts
1245,12/25/2006,12/27/2006,338,A Medical Test of Sorts
1246,12/26/2006,12/26/2006,332,A Medical Test of Sorts
1247,12/28/2006,12/29/2006,335,A Medical Test of Sorts
1248,12/28/2006,12/30/2006,333,A Medical Test of Sorts
1249,12/31/2006,1/2/2007,340,A Medical Test of Sorts
1250,12/31/2006,1/4/2007,340,A Medical Test of Sorts
1251,12/31/2006,1/4/2007,332,A Medical Test of Sorts
1252,1/1/2007,1/2/2007,342,A Medical Test of Sorts
1253,1/2/2007,1/3/2007,336,A Medical Test of Sorts
1254,1/3/2007,1/5/2007,340,A Medical Test of Sorts
1255,1/3/2007,,340,A Medical Test of Sorts
1256,1/4/2007,1/6/2007,334,A Medical Test of Sorts
1257,1/4/2007,1/8/2007,334,A Medical Test of Sorts
1258,1/6/2007,1/9/2007,341,A Medical Test of Sorts
1259,1/6/2007,1/7/2007,341,A Medical Test of Sorts
1260,1/6/2007,1/8/2007,333,A Medical Test of Sorts
1261,1/7/2007,1/11/2007,340,A Medical Test of Sorts
1262,1/8/2007,1/12/2007,334,A Medical Test of Sorts
1263,1/8/2007,1/9/2007,335,A Medical Test of Sorts
1264,1/9/2007,1/10/2007,335,A Medical Test of Sorts
1265,1/9/2007,1/11/2007,342,A Medical Test of Sorts
1266,1/9/2007,1/11/2007,334,A Medical Test of Sorts
1267,1/9/2007,1/11/2007,340,A Medical Test of Sorts
1268,1/9/2007,1/10/2007,332,A Medical Test of Sorts
1269,1/10/2007,1/12/2007,342,A Medical Test of Sorts
1270,1/10/2007,1/11/2007,337,A Medical Test of Sorts
1271,1/13/2007,1/15/2007,334,A Medical Test of Sorts
1272,1/13/2007,1/13/2007,342,A Medical Test of Sorts
1273,1/14/2007,,335,A Medical Test of Sorts
1274,1/14/2007,1/16/2007,337,A Medical Test of Sorts
1275,1/16/2007,1/19/2007,338,A Medical Test of Sorts
1276,1/18/2007,1/21/2007,332,A Medical Test of Sorts
1277,1/18/2007,1/19/2007,338,A Medical Test of Sorts
1278,1/19/2007,1/22/2007,337,A Medical Test of Sorts
1279,1/19/2007,1/21/2007,338,A Medical Test of Sorts
1280,1/23/2007,1/24/2007,337,A Medical Test of Sorts
1281,1/23/2007,1/27/2007,334,A Medical Test of Sorts
1282,1/25/2007,1/25/2007,337,A Medical Test of Sorts
1283,1/25/2007,1/26/2007,336,A Medical Test of Sorts
1284,1/26/2007,1/30/2007,340,A Medical Test of Sorts
1285,1/26/2007,1/30/2007,339,A Medical Test of Sorts
1286,1/26/2007,1/29/2007,338,A Medical Test of Sorts
1287,1/26/2007,1/26/2007,336,A Medical Test of Sorts
1288,1/28/2007,1/29/2007,340,A Medical Test of Sorts
1289,1/28/2007,1/29/2007,333,A Medical Test of Sorts
1290,1/29/2007,1/30/2007,335,A Medical Test of Sorts
1291,1/29/2007,1/31/2007,332,A Medical Test of Sorts
1292,1/29/2007,1/30/2007,340,A Medical Test of Sorts
1293,1/30/2007,2/3/2007,335,A Medical Test of Sorts
1294,1/30/2007,1/31/2007,342,A Medical Test of Sorts
1295,1/30/2007,2/3/2007,342,A Medical Test of Sorts
1296,1/31/2007,1/31/2007,332,A Medical Test of Sorts
1297,1/31/2007,2/1/2007,334,A Medical Test of Sorts
1298,2/1/2007,2/4/2007,333,A Medical Test of Sorts
1299,2/3/2007,2/4/2007,333,A Medical Test of Sorts
1300,2/5/2007,2/5/2007,338,A Medical Test of Sorts
1301,2/6/2007,2/8/2007,341,A Medical Test of Sorts
1302,2/7/2007,2/9/2007,332,A Medical Test of Sorts
1303,2/9/2007,2/9/2007,332,A Medical Test of Sorts
1304,2/10/2007,2/11/2007,340,A Medical Test of Sorts
1305,2/10/2007,2/14/2007,342,A Medical Test of Sorts
1306,2/10/2007,2/13/2007,341,A Medical Test of Sorts
1307,2/11/2007,2/14/2007,334,A Medical Test of Sorts
1308,2/11/2007,2/15/2007,333,A Medical Test of Sorts
1309,2/11/2007,2/13/2007,333,A Medical Test of Sorts
1310,2/12/2007,2/16/2007,338,A Medical Test of Sorts
1311,2/14/2007,2/18/2007,342,A Medical Test of Sorts
1312,2/14/2007,2/16/2007,336,A Medical Test of Sorts
1313,2/15/2007,2/15/2007,335,A Medical Test of Sorts
1314,2/15/2007,2/17/2007,334,A Medical Test of Sorts
1315,2/16/2007,2/18/2007,337,A Medical Test of Sorts
1316,2/17/2007,2/17/2007,336,A Medical Test of Sorts
1317,2/18/2007,2/18/2007,338,A Medical Test of Sorts
1318,2/18/2007,2/22/2007,342,A Medical Test of Sorts
1319,2/19/2007,2/20/2007,340,A Medical Test of Sorts
1320,2/19/2007,2/22/2007,338,A Medical Test of Sorts
1321,2/20/2007,2/21/2007,340,A Medical Test of Sorts
1322,2/21/2007,2/22/2007,340,A Medical Test of Sorts
1323,2/21/2007,2/23/2007,338,A Medical Test of Sorts
1324,2/21/2007,2/25/2007,340,A Medical Test of Sorts
1325,2/22/2007,,340,A Medical Test of Sorts
1326,2/23/2007,2/26/2007,342,A Medical Test of Sorts
1327,2/23/2007,2/24/2007,335,A Medical Test of Sorts
1328,2/25/2007,2/28/2007,334,A Medical Test of Sorts
1329,2/26/2007,3/2/2007,342,A Medical Test of Sorts
1330,2/26/2007,3/1/2007,340,A Medical Test of Sorts
1331,2/27/2007,3/2/2007,338,A Medical Test of Sorts
1332,2/28/2007,3/2/2007,337,A Medical Test of Sorts
1333,2/28/2007,3/2/2007,337,A Medical Test of Sorts
1334,3/2/2007,3/4/2007,341,A Medical Test of Sorts
1335,3/3/2007,3/4/2007,340,A Medical Test of Sorts
1336,3/3/2007,3/5/2007,334,A Medical Test of Sorts
1337,3/4/2007,3/6/2007,337,A Medical Test of Sorts
1338,3/5/2007,3/5/2007,334,A Medical Test of Sorts
1339,3/7/2007,3/10/2007,342,A Medical Test of Sorts
1340,3/7/2007,3/10/2007,339,A Medical Test of Sorts
1341,3/8/2007,3/9/2007,341,A Medical Test of Sorts
1342,3/8/2007,3/12/2007,333,A Medical Test of Sorts
1343,3/9/2007,3/13/2007,341,A Medical Test of Sorts
1344,3/10/2007,3/11/2007,334,A Medical Test of Sorts
1345,3/10/2007,3/11/2007,335,A Medical Test of Sorts
1346,3/11/2007,3/11/2007,338,A Medical Test of Sorts
1347,3/13/2007,3/14/2007,333,A Medical Test of Sorts
1348,3/14/2007,3/14/2007,338,A Medical Test of Sorts
1349,3/14/2007,3/18/2007,338,A Medical Test of Sorts
1350,3/14/2007,3/17/2007,333,A Medical Test of Sorts
1351,3/15/2007,3/16/2007,341,A Medical Test of Sorts
1352,3/15/2007,3/16/2007,334,A Medical Test of Sorts
1353,3/15/2007,3/18/2007,338,A Medical Test of Sorts
1354,3/16/2007,3/19/2007,333,A Medical Test of Sorts
1355,3/16/2007,3/17/2007,336,A Medical Test of Sorts
1356,3/16/2007,3/19/2007,341,A Medical Test of Sorts
1357,3/17/2007,3/21/2007,334,A Medical Test of Sorts
1358,3/19/2007,3/20/2007,342,A Medical Test of Sorts
1359,3/21/2007,3/25/2007,342,A Medical Test of Sorts
1360,3/21/2007,3/21/2007,337,A Medical Test of Sorts
1361,3/22/2007,3/23/2007,338,A Medical Test of Sorts
1362,3/24/2007,3/25/2007,342,A Medical Test of Sorts
1363,3/24/2007,3/27/2007,332,A Medical Test of Sorts
1364,3/25/2007,3/27/2007,338,A Medical Test of Sorts
1365,3/25/2007,3/28/2007,340,A Medical Test of Sorts
1366,3/26/2007,3/28/2007,334,A Medical Test of Sorts
1367,3/27/2007,3/27/2007,338,A Medical Test of Sorts
1368,3/28/2007,3/28/2007,333,A Medical Test of Sorts
1369,3/30/2007,4/1/2007,342,A Medical Test of Sorts
1370,3/31/2007,4/2/2007,332,A Medical Test of Sorts
1371,3/31/2007,4/4/2007,337,A Medical Test of Sorts
1372,4/2/2007,,335,A Medical Test of Sorts
1373,4/2/2007,4/6/2007,341,A Medical Test of Sorts
1374,4/3/2007,4/7/2007,336,A Medical Test of Sorts
1375,4/7/2007,4/9/2007,335,A Medical Test of Sorts
1376,4/7/2007,4/10/2007,337,A Medical Test of Sorts
1377,4/8/2007,4/12/2007,333,A Medical Test of Sorts
1378,4/9/2007,4/10/2007,339,A Medical Test of Sorts
1379,4/10/2007,4/10/2007,334,A Medical Test of Sorts
1380,4/10/2007,4/14/2007,340,A Medical Test of Sorts
1381,4/12/2007,4/15/2007,338,A Medical Test of Sorts
1382,4/12/2007,4/13/2007,336,A Medical Test of Sorts
1383,4/12/2007,4/12/2007,334,A Medical Test of Sorts
1384,4/13/2007,4/17/2007,341,A Medical Test of Sorts
1385,4/14/2007,4/14/2007,339,A Medical Test of Sorts
1386,4/17/2007,4/21/2007,341,A Medical Test of Sorts
1387,4/17/2007,4/19/2007,332,A Medical Test of Sorts
1388,4/18/2007,4/22/2007,342,A Medical Test of Sorts
1389,4/20/2007,4/24/2007,335,A Medical Test of Sorts
1390,4/20/2007,4/21/2007,332,A Medical Test of Sorts
1391,4/20/2007,4/20/2007,341,A Medical Test of Sorts
1392,4/20/2007,4/22/2007,336,A Medical Test of Sorts
1393,4/20/2007,4/21/2007,332,A Medical Test of Sorts
1394,4/21/2007,4/21/2007,335,A Medical Test of Sorts
1395,4/21/2007,4/23/2007,342,A Medical Test of Sorts
1396,4/21/2007,4/24/2007,336,A Medical Test of Sorts
1397,4/22/2007,4/22/2007,341,A Medical Test of Sorts
1398,4/22/2007,4/24/2007,332,A Medical Test of Sorts
1399,4/22/2007,4/26/2007,337,A Medical Test of Sorts
1400,4/23/2007,4/27/2007,336,A Medical Test of Sorts
1401,4/24/2007,4/27/2007,336,A Medical Test of Sorts
'''

def example_input():
  '''Returns the example input table so the user can
     see what their input should look like'''
     
  import StringIO  # to emulate a file for load_csv
  bill_table = load_csv(StringIO.StringIO(csv_bills))
  bill_table.set_type('BillID', int)
  bill_table.set_type('TestID', int)
  bill_table.set_type('DateBilled', DateTime)
  bill_table.set_type('Amount',  float)
  
  test_table = load_csv(StringIO.StringIO(csv_tests))
  test_table.set_type('TestID', int)
  test_table.set_type('DatePrescribed', DateTime)
  test_table.set_type('DatePerformed',  DateTime)
  test_table.set_type('DrID', int)
  test_table.set_type('Description', unicode)
  
  return bill_table, test_table

#-----------------------------------------------------------------------------------#
#           Testing
#-----------------------------------------------------------------------------------#
if __name__=="__main__":
    tbl = example_input()
    results = run(tbl[1], 
                tbl[0],
                'TestID', 
                'DatePerformed', 
                'BillID',
                'TestID',
                'Amount')
    results[0].view()