#!/usr/bin/python

####################################################################################
#                                                                                  #
# Copyright (c) 2006 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This file is part of Picalo.                                                     #
#                                                                                  #
# Picalo is free software; you can redistribute it and/or modify                   #
# it under the terms of the GNU General Public License as published by             #
# the Free Software Foundation; either version 2 of the License, or                # 
# (at your option) any later version.                                              #
#                                                                                  #
# Picalo is distributed in the hope that it will be useful,                        #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                   #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    #
# GNU General Public License for more details.                                     #
#                                                                                  #
# You should have received a copy of the GNU General Public License                #
# along with Foobar; if not, write to the Free Software                            #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA        #
#                                                                                  #
####################################################################################


import wx
import Dialogs, Spreadsheet, TableProperties, Preferences, Editor
import os, os.path, StringIO
from Languages import lang
from picalo.base.Global import make_valid_variable
from picalo import *
import picalo.lib.pyExcelerator


class TextImportWizard(Dialogs.Wizard):
  def __init__(self, mainframe):
    '''Constructor'''
    Dialogs.Wizard.__init__(self, mainframe, 'TextImportWizard')
    self.filelines = None
    self.updatefieldinfo = True

  
  def init(self):
    '''Initializes the dialog'''
    # set up the start page
    self.getControl('fileButton').Bind(wx.EVT_BUTTON, self.onFileButton)
    self.getControl('filename').Bind(wx.EVT_TEXT, self.onFileText)
    self.getControl('encoding').Clear()
    for codec, description in CODECS:
      self.getControl('encoding').Append(description)
    self.getControl('encoding').SetSelection(0)
    
    # set up the file type page
    self.getControl('filetype').Bind(wx.EVT_RADIOBOX, self.onFileType)
    
    # set up the delimiters page
    self.previewtable = Table(1)
    self.stringtable = Table(1)
    self.preview = Spreadsheet.Spreadsheet(self.getControl('previewpanel'), self.mainframe, '', self.previewtable)
    self.preview.showToolbar(False)
    self.preview.setAutoNewRows(False)
    self.preview.setEventsEnabled(False)
    self.getControl('previewpanel').GetSizer().Add(self.preview, proportion=1, flag=wx.EXPAND|wx.ALL)
    self.getControl('previewpanel').Layout()
    self.getControl('headerRow').Bind(wx.EVT_CHECKBOX, self.updatePreview)
    self.getControl('qualifier').Bind(wx.EVT_TEXT, self.updatePreview)
    self.getControl('delimiterComma').Bind(wx.EVT_RADIOBUTTON, self.updatePreview)
    self.getControl('delimiterTab').Bind(wx.EVT_RADIOBUTTON, self.updatePreview)
    self.getControl('delimiterSpace').Bind(wx.EVT_RADIOBUTTON, self.updatePreview)
    self.getControl('delimiterOther').Bind(wx.EVT_RADIOBUTTON, self.updatePreview)
    self.getControl('delimiterOtherChar').Bind(wx.EVT_TEXT, self.updatePreview)
    
    # set up the fixed width page
    self.selectfixedcontrol = SelectFixedControl(self.getControl('fixedpreview'), self)
    self.getControl('fixedpreview').GetSizer().Add(self.selectfixedcontrol, proportion=1, flag=wx.EXPAND|wx.ALL)
    
    # set up the excel page
    self.preview_excel = Spreadsheet.Spreadsheet(self.getControl('previewpanel_excel'), self.mainframe, '', self.previewtable)
    self.preview_excel.showToolbar(False)
    self.preview_excel.setAutoNewRows(False)
    self.preview_excel.setEventsEnabled(False)
    self.getControl('previewpanel_excel').GetSizer().Add(self.preview_excel, proportion=1, flag=wx.EXPAND|wx.ALL)
    self.getControl('previewpanel_excel').Layout()
    self.getControl('excel_updatepreview').Bind(wx.EVT_BUTTON, self.updatePreview)
    
    # set up the field type page
    self.udpatefieldinfo = True
    self.preview_delimited = Spreadsheet.Spreadsheet(self.getControl('previewpanel_fieldtype'), self.mainframe, '', self.previewtable)
    self.preview_delimited.showToolbar(False)
    self.preview_delimited.setAutoNewRows(False)
    self.preview_delimited.setEventsEnabled(False)
    self.getControl('previewpanel_fieldtype').GetSizer().Add(self.preview_delimited, proportion=1, flag=wx.EXPAND|wx.ALL)
    self.getControl('previewpanel_fieldtype').Layout()
    self.preview_delimited.Bind(wx.grid.EVT_GRID_SELECT_CELL, self.onpreview_delimitedSelected)
    self.preview_delimited.Bind(wx.grid.EVT_GRID_LABEL_LEFT_CLICK, self.onpreview_delimitedSelected)
    self.preview_delimited.Bind(wx.grid.EVT_GRID_LABEL_RIGHT_CLICK, self.onpreview_delimitedSelected)
    self.getControl('fieldname').Bind(wx.EVT_TEXT, self.onFieldName)
    self.getControl('fieldtype').Bind(wx.EVT_CHOICE, self.onFieldType)
    self.tablepropertiesdlg = TableProperties.TableProperties(self.mainframe, self.preview_delimited)  # this is so we can access the list of field types, etc.
    self.tablepropertiesdlg.createDialog()
    self.tablepropertiesdlg.init()
    self.getControl('fieldtype').Clear()
    for item in [ typ.name for typ in TableProperties.FIELD_TYPES ]:
      self.getControl('fieldtype').Append(item)
    self.getControl('fieldformat').Bind(wx.EVT_CHOICE, self.onFieldFormat)
    
    # set the size of the wizard
    size = self.dlg.GetPageAreaSizer().GetMinSize()
    size.height += 150
    self.dlg.GetPageAreaSizer().SetMinSize(size)
    
    
  def onFileButton(self, event):
    '''Responds to the file browser button'''
    dlg = wx.FileDialog(self.dlg, message="Select Text File", defaultFile="", style=wx.OPEN | wx.CHANGE_DIR | wx.FILE_MUST_EXIST)
    if dlg.ShowModal() != wx.ID_OK:
      return
    self.getControl('filename').SetValue(dlg.GetPath())


  def onFileText(self, event=None):    
    '''Responds to a change in the filename'''
    filepath = self.getControl('filename').GetValue()
    self.getControl('tablename').SetValue(make_valid_variable(os.path.splitext(os.path.split(filepath)[1])[0]))
    if os.path.splitext(filepath)[1].lower() == '.xls':
      self.getControl('filetype').SetSelection(0)
    else:
      self.getControl('filetype').SetSelection(1)
    self.onFileType()


  def onFileType(self, event=None):
    '''Responds to a change in the file type'''
    filetype = self.getControl('filetype').GetSelection()
    if filetype == 0:
      self.setPageOrder([ 'startpage', 'page_filetype', 'page_excel', 'page_fieldtypes' ])
      filename = self.getControl('filename').GetValue()
      if os.path.exists(filename):
        workbook = picalo.lib.pyExcelerator.parse_xls(filename) 
        assert len(workbook) > 0, 'No worksheets were found in this Excel file.'
        self.getControl('excel_worksheet').Clear()
        for name, sht in workbook:
          self.getControl('excel_worksheet').Append(name)
        self.getControl('excel_worksheet').SetSelection(0)
    
    elif filetype == 1:
      self.setPageOrder([ 'startpage', 'page_filetype', 'page_delimiters', 'page_fieldtypes' ])
    
    elif filetype == 2:
      self.setPageOrder([ 'startpage', 'page_filetype', 'page_fixedcolumns', 'page_fieldtypes' ])
    
    
  def onpreview_delimitedSelected(self, event):
    '''Responds to a cell selection on the table'''
    col = event.GetCol()
    if col >= 0:
      self.preview_delimited.grid.SelectCol(col)
      self.updateFieldInfo()
    else:
      pass
      # do nothing -- don't process the event
      
      
  def onFieldName(self, event=None):
    '''Responds to a change in the field name'''
    selectedcols = self.preview_delimited.grid.GetSelectedCols()
    if self.updatefieldinfo and len(selectedcols) > 0:
      col = selectedcols[0]
      self.previewtable.set_readonly(False)
      self.previewtable.column(col).set_name(self.getControl('fieldname').GetValue())
      self.previewtable.set_readonly(True)
      self.updatePreviewTables()
      
      
  def onFieldType(self, event=None):
    '''Responds to a change in the field type'''
    selectedcols = self.preview_delimited.grid.GetSelectedCols()
    if self.updatefieldinfo and len(selectedcols) > 0:
      col = selectedcols[0]
      typeinfo = TableProperties.FIELD_TYPES[self.getControl('fieldtype').GetSelection()]
      self.previewtable.set_readonly(False)
      self.previewtable.column(col).set_type(unicode)  # reset the values in case last time caused an error
      for i in range(len(self.stringtable)):
        self.previewtable[i][col] = self.stringtable[i][col]
      self.previewtable.column(col).set_type(typeinfo.realtype)
      self.populateFieldFormatChoice()
      self.onFieldFormat()
      self.previewtable.set_readonly(True)
      self.updatePreviewTables()
      
      
  def onFieldFormat(self, event=None):
    '''Responds to a change in the field format'''
    selectedcols = self.preview_delimited.grid.GetSelectedCols()
    if self.updatefieldinfo and len(selectedcols) > 0:
      col = selectedcols[0]
      selection = self.getControl('fieldformat').GetStringSelection()
      typeinfo = TableProperties.FIELD_TYPES[self.getControl('fieldtype').GetSelection()]
      self.previewtable.set_readonly(False)
      fmt = self.getFormat(typeinfo, selection)
      self.previewtable.column(col).set_format(fmt)
      self.previewtable.set_readonly(True)
      self.updatePreviewTables()
      
      
  def getFormat(self, typeinfo, selection):
    '''Returns the format for the given date selection'''
    return ''
      
  
  def updateFieldInfo(self):
    '''Called when the column changes in the preview_delimited table'''
    selectedcols = self.preview_delimited.grid.GetSelectedCols()
    if len(selectedcols) > 0:
      col = self.previewtable.columns[selectedcols[0]]
      self.updatefieldinfo = False
      # update the field name
      self.getControl('fieldname').SetValue(col.name)
      # update the field type choice box
      typeinfo = TableProperties.FIELD_TYPES[0]
      for tinfo in TableProperties.FIELD_TYPES:
        if tinfo.realtype == col.column_type:
          self.getControl('fieldtype').SetStringSelection(tinfo.name)
          typeinfo = tinfo
          break
      # update the field format choice box
#      self.populateFieldFormatChoice()
#      if typeinfo and typeinfo.editor:
#        for item in typeinfo.editor.options:  # figure out what the current format is
#          if col.format == self.getFormat(typeinfo, item):
#            self.getControl('fieldformat').SetStringSelection(item)
#            break
      self.updatefieldinfo = True
      
      
  def getEncoding(self):
    return CODECS[self.getControl('encoding').GetSelection()][0]
    
    
  def populateFieldFormatChoice(self):
    '''Called when the column type changes or a new column is selected'''
    # update the format choice box
    self.getControl('fieldformat').Clear()
    typeinfo = TableProperties.FIELD_TYPES[self.getControl('fieldtype').GetSelection()]
#    if typeinfo and typeinfo.editor:
#      self.getControl('fieldformatlabel').Enable()
#      self.getControl('fieldformat').Enable()
#      for item in typeinfo.editor.options:
#        self.getControl('fieldformat').Append(item)
#      self.getControl('fieldformat').SetSelection(0)
#    else:
    self.getControl('fieldformatlabel').Disable()
    self.getControl('fieldformat').Disable()
  
    
  def updatePreview(self, event=None):
    '''Updates the preview windows'''
    filetype = self.getControl('filetype').GetSelection()
    filename = self.getControl('filename').GetValue()
    if filetype == 0: # excel file
      worksheet, topleft, bottomright, headerrow = self.getExcelSettings()
      self.previewtable = load_excel(filename, worksheet_name=worksheet, topleft_cell=topleft, bottomright_cell=bottomright, header_row=headerrow)
      self.previewtable.set_readonly(True)
      self.preview_delimited.grid.SelectCol(0)
      self.updatePreviewTables()
    
    elif filetype == 1: # delimited file
      delimiter, qualifier, headerrow = self.getDelimiterSettings()
      if delimiter != '':
        self.previewtable = load_delimited(StringIO.StringIO(''.join(self.filelines)), header_row=headerrow, delimiter=delimiter, qualifier=qualifier, encoding=self.getEncoding())
        self.stringtable = self.previewtable[:]
        self.previewtable.set_readonly(True)
        self.updatePreviewTables()
        self.preview_delimited.grid.SelectCol(0)
        self.updateFieldInfo()
    
    elif filetype == 2: # fixed width
      column_positions = self.selectfixedcontrol.textpreview.column_positions
      self.previewtable = load_fixed(StringIO.StringIO(''.join(self.filelines)), column_positions=column_positions, header_row=self.getControl('fixedHeaderRow').IsChecked(), encoding=self.getEncoding())
      self.previewtable.set_readonly(True)
      self.updatePreviewTables()
    
    
  def updatePreviewTables(self, event=None):
    '''Allows the preview tables to update their UI'''
    self.preview.onIdle(self.previewtable)
    self.preview_delimited.onIdle(self.previewtable)
    self.preview_excel.onIdle(self.previewtable)
    
    
  def getExcelSettings(self):
    '''Returns the (worksheet, topleft, bottomright, headerrow)'''
    worksheet = None
    if self.getControl("excel_worksheet").GetSelection() >= 0:
      worksheet = self.getControl("excel_worksheet").GetStringSelection()
    topleft = self.getControl('excel_topleft').GetValue()
    bottomright = self.getControl('excel_bottomright').GetValue()
    headerrow = self.getControl('excel_headerrow').IsChecked()
    return worksheet, topleft, bottomright, headerrow
    
  
  def getDelimiterSettings(self):
    '''Returns the (delimiter, qualifier, headerrow)'''
    # qualifier
    qualifier = self.getControl('qualifier').GetValue()
    if qualifier == '{None}' or qualifier == '':
      qualifier = None
    else:
      qualifier = str(qualifier[:1])  # the csv reader requires a single character
      self.getControl('qualifier').SetValue(qualifier)
    # delimiter
    if self.getControl('delimiterComma').GetValue():
      delimiter = ','
      self.getControl('delimiterOtherChar').SetValue('')
    elif self.getControl('delimiterTab').GetValue():
      delimiter = '\t'
      self.getControl('delimiterOtherChar').SetValue('')
    elif self.getControl('delimiterSpace').GetValue():
      delimiter = ' '
      self.getControl('delimiterOtherChar').SetValue('')
    else:
      self.getControl('delimiterOtherChar').SetValue(self.getControl('delimiterOtherChar').GetValue()[:1])
      delimiter = str(self.getControl('delimiterOtherChar').GetValue())
    headerrow = self.getControl('headerRow').IsChecked()
    return (delimiter, qualifier, headerrow)
    

  def validate_startpage(self):
    '''Validates page 1'''
    self.checkValidVariable('the new table name', self.getControl('tablename').GetValue())
    filename = self.getControl('filename').GetValue()
    assert os.path.exists(filename), 'Please enter a valid filename or use the selection button.'
    if os.path.splitext(filename)[1].lower() == '.xls':
      self.getControl('filetype').SetSelection(0)
    else:
      self.getControl('filetype').SetSelection(1)
    
  
  def validate_page_filetype(self):
    '''Validates the file type page'''
    filetype = self.getControl('filetype').GetSelection()
    filename = self.getControl('filename').GetValue()
    self.filelines = []
    if filetype == 0:
      pass  # The excel importer library only supports the filename, not file bytes, so no need to preload

    elif filetype == 1:
      # get the first 10 lines of the file
      fp = open(filename, 'rb')
      for i in range(10):
        line = fp.readline()
        if not line:
          break
        self.filelines.append(line)
      fp.close()
      assert self.filelines, 'The file appears to be empty.'

    elif filetype == 2:
      # get the first 10 lines of the file
      fp = open(filename, 'rb')
      for i in range(10):
        line = fp.readline()
        if not line:
          break
        self.filelines.append(line)
      fp.close()
      assert self.filelines, 'The file appears to be empty.'

    self.updatePreview()
    
    
  def validate_page_delimiters(self):
    assert not self.getControl('delimiterOther').GetValue() or len(self.getControl('delimiterOtherChar').GetValue()) > 0, 'Please enter a delimiter to split fields on.'
    

  def validate_page_fixedcolumns(self):
    self.updatePreview()

  def finish(self):
    '''Runs the finisher code'''
    cmds = []
    filetype = self.getControl('filetype').GetSelection()
    tablename = self.getControl('tablename').GetValue()
    filename = self.getControl('filename').GetValue().replace('\\', '/').replace('"', '\\"')
    delimiter, qualifier, headerrow = self.getDelimiterSettings()
    if filetype == 0:
      worksheet, topleft, bottomright, headerrow = self.getExcelSettings()
      params = []
      params.append('"%s"' % (filename,))
      params.append('worksheet_name="%s"' % (worksheet.replace('"', '\\"'), ))
      params.append('header_row=%s' % (headerrow,))
      if topleft:
        params.append('topleft_cell="%s"' % (topleft.replace('"', '\\"')))
      if bottomright:
        params.append('bottomright_cell="%s"' % (bottomright.replace('"', '\\"')))
      cmds.append('%s = load_excel(%s)' % (tablename, ', '.join(params)))
      
    elif filetype == 1:
      cmds.append('%s = load_delimited("%s", header_row=%s, delimiter="%s", qualifier="%s", encoding=%s)' % (tablename, filename, headerrow, delimiter.replace('"', '\\"'), qualifier.replace('"', '\\"'), self.getEncoding() != None and '"' + self.getEncoding() + '"' or None))
      
    elif filetype == 2:
      column_positions = [ str(i) for i in self.selectfixedcontrol.textpreview.column_positions ]
      cmds.append('%s = load_fixed("%s", [%s], header_row=%s, encoding=%s)' % (tablename, filename, ', '.join(column_positions), self.getControl('fixedHeaderRow').IsChecked()), self.getEncoding() != None and '"' + self.getEncoding() + '"' or None)

    # set column names and types
    for i, col in enumerate(self.previewtable.get_columns()):
      cmds.append('%s.set_name(%i, "%s")' % (tablename, i, col.name))
      if col.column_type != unicode:
        if col.format:
          cmds.append('%s.set_type("%s", %s, "%s")' % (tablename, col.name.replace('"', '\\"'), col.column_type.__name__, col.format.expression.replace('"', '\\"')))
        else:
          cmds.append('%s.set_type("%s", %s)' % (tablename, col.name.replace('"', '\\"'), col.column_type.__name__, ))
    # run the commands
    cmds.append('%s.view()' % (tablename, ))
    self.mainframe.execute(cmds)
    
   
##############################################################
###   Load fixed control to set the widths   
   
class SelectFixedControl(wx.PyScrolledWindow):
  '''Allows the user to select the fixed column locations'''
  def __init__(self, parent, textimportwizard):
    '''Constructor'''
    wx.PyScrolledWindow.__init__(self, parent)
    self.textpreview = FixedTextPreview(self, textimportwizard)
    self.textpreview.SetDimensions(0, 0, 150, 150)
    self.SetScrollRate(20, 20)
    self.SetBackgroundColour(wx.WHITE)
    
    
class FixedTextPreview(wx.PyWindow):
  '''The display for a fixed text preview'''
  def __init__(self, parent, textimportwizard):
    '''Constructor'''
    wx.PyWindow.__init__(self, parent)
    self.parent = parent
    self.textimportwizard = textimportwizard
    self.numbers = []
    self.numberpixels = []
    self.ypadding = 2
    self.column_positions = [ 0 ]
    self.Bind(wx.EVT_PAINT, self.onPaint)
    self.Bind(wx.EVT_LEFT_DOWN, self.onMouseDown)

    
  def get_col_indices(self):
    '''Returns the column indices'''
    return range(max([len(line) for line in self.textimportwizard.filelines]))
    
    
  def onPaint(self, event=None):
    '''Paints the joins on the window'''
    dc = wx.PaintDC(self)
    dc.SetPen(wx.Pen(wx.BLACK, width=1))
    dc.SetTextForeground(wx.BLACK)
    dc.SetFont(wx.Font(Preferences.get('editorfontsize', Editor.DEFAULT_FONT_SIZE), wx.FONTFAMILY_MODERN, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL))
    y = 0
    # create the column index line
    self.numbers = []
    for num in self.get_col_indices():
      if num % 10 == 0:
        self.numbers.append('+')
      else:
        self.numbers.append(str(num % 10))
      self.numberpixels.append(dc.GetTextExtent(''.join(self.numbers)))
    numline = ''.join(self.numbers)
    # print the data
    for line in [ numline ] + self.textimportwizard.filelines:
      width, height = dc.GetTextExtent(line)
      dc.DrawText(line, 0, y)
      y += height + self.ypadding
    # set the sizes
    numlinewidth, numlineheight = dc.GetTextExtent(numline)
    parentwidth, parentheight = self.parent.GetSize()
    myheight = max(y, parentheight)
    self.parent.SetVirtualSize([numlinewidth, myheight])
    self.SetSize([numlinewidth, myheight])
    # draw the lines
    for pos in self.column_positions[1:]: # don't draw the first one (position 0)
      numline = ''.join(self.numbers[:pos])
      width, height = dc.GetTextExtent(numline)
      width -= 1  # sorry for the magic number, but GetTextExtent goes just a bit too wide
      dc.DrawLine(width, 0, width, myheight)
    event.Skip()
    
    
  def onMouseDown(self, event=None):
    '''Captures a mouse down event'''
    x, y = event.m_x, event.m_y
    # go through the numbers one by one and figure out where the mouse clicked
    for num, size in enumerate(self.numberpixels):
      if size[0] >= x:
        if num+1 in self.column_positions:
          self.column_positions.remove(num+1)
        else:
          self.column_positions.append(num+1)
        break
    self.column_positions.sort()
    # force a repaint
    self.Refresh()
    self.Update()
    event.Skip()



CODECS = [
  [ None, "Automatic (let Picalo choose)" ],
  [ "ascii", "ASCII (ascii, 646, us-ascii)" ],
  [ "utf_7", "UTF-7 (utf_7, U7, unicode-1-1-utf-7)" ],
  [ "utf_8", "UTF-8 (utf_8, U8, UTF, utf8)" ],
  [ "utf_16", "UTF-16 (utf_16, U16, utf16)" ],
  [ "utf_16_be", "UTF-16-Big Endian (utf_16_be, UTF-16BE)" ],
  [ "utf_16_le", "UTF-16-Little Endian (utf_16_le, UTF-16LE)" ],
  [ "utf_32", "UTF-32 (utf_32, U32, utf32)" ],
  [ "utf_32_be", "UTF-32-Big Endian (utf_32_be, UTF-32BE)" ],
  [ "utf_32_le", "UTF-32-Little Endian (utf_32_le, UTF-32LE)" ],
  [ "cp1256", "Arabic (cp1256, windows1256)" ],
  [ "cp864", "Arabic (cp864, IBM864)" ],
  [ "iso8859_6", "Arabic (iso8859_6, iso-8859-6, arabic)" ],
  [ "cp1257", "Baltic languages (cp1257, windows-1257)" ],
  [ "cp775", "Baltic languages (cp775, IBM775)" ],
  [ "iso8859_13", "Baltic languages (iso8859_13, iso-8859-13)" ],
  [ "iso8859_4", "Baltic languages (iso8859_4, iso-8859-4, latin4, L4)" ],
  [ "iso8859_5", "Bulgarian, Byelorussian, (iso8859_5, iso-8859-5, cyrillic)" ],
  [ "mac_cyrillic", "Bulgarian, Byelorussian, (mac_cyrillic, maccyrillic)" ],
  [ "cp1251", "Bulgarian, Byelorussian, Macedonian, Russian, Serbian (cp1251, windows-1251)" ],
  [ "cp855", "Bulgarian, Byelorussian, Macedonian, Russian, Serbian (cp855, 855, IBM855)" ],
  [ "cp863", "Canadian (cp863, 863, IBM863)" ],
  [ "iso8859_14", "Celtic languages (iso8859_14, iso-8859-14, latin8, L8)" ],
  [ "cp1250", "Central and Eastern Europe (cp1250, windows-1250)" ],
  [ "cp852", "Central and Eastern Europe (cp852, 852, IBM852)" ],
  [ "iso8859_2", "Central and Eastern Europe (iso8859_2, iso-8859-2, latin2, L2)" ],
  [ "mac_latin2", "Central and Eastern Europe (mac_latin2, maclatin2, maccentraleurope)" ],
  [ "cp865", "Danish, Norwegian (cp865, 865, IBM865)" ],
  [ "cp037", "English (cp037, IBM037, IBM039)" ],
  [ "cp437", "English (cp437, 437, IBM437)" ],
  [ "iso8859_3", "Esperanto, Maltese (iso8859_3, iso-8859-3, latin3, L3)" ],
  [ "cp1253", "Greek (cp1253, windows-1253)" ],
  [ "cp737", "Greek (cp737)" ],
  [ "cp869", "Greek (cp869, 869, CP-GR, IBM869)" ],
  [ "cp875", "Greek (cp875)" ],
  [ "iso8859_7", "Greek (iso8859_7, iso-8859-7, greek, greek8)" ],
  [ "mac_greek", "Greek (mac_greek, macgreek)" ],
  [ "cp1255", "Hebrew (cp1255, windows-1255)" ],
  [ "cp424", "Hebrew (cp424, EBCDIC-CP-HE, IBM424)" ],
  [ "cp856", "Hebrew (cp856)" ],
  [ "cp862", "Hebrew (cp862, 862, IBM862)" ],
  [ "iso8859_8", "Hebrew (iso8859_8, iso-8859-8, hebrew)" ],
  [ "cp861", "Icelandic (cp861, 861, CP-IS, IBM861)" ],
  [ "mac_iceland", "Icelandic (mac_iceland, maciceland)" ],
  [ "cp932", "Japanese (cp932, 932, ms932, mskanji, ms-kanji)" ],
  [ "euc_jisx0213", "Japanese (euc_jisx0213, eucjisx0213)" ],
  [ "euc_jis_2004", "Japanese (euc_jis_2004, jisx0213, eucjis2004)" ],
  [ "euc_jp", "Japanese (euc_jp, eucjp, ujis, u-jis)" ],
  [ "iso2022_jp", "Japanese (iso2022_jp, csiso2022jp, iso2022jp, iso-2022-jp)" ],
  [ "iso2022_jp_1", "Japanese (iso2022_jp_1, iso2022jp-1, iso-2022-jp-1)" ],
  [ "iso2022_jp_2004", "Japanese (iso2022_jp_2004, iso2022jp-2004, iso-2022-jp-2004)" ],
  [ "iso2022_jp_3", "Japanese (iso2022_jp_3, iso2022jp-3, iso-2022-jp-3)" ],
  [ "iso2022_jp_ext", "Japanese (iso2022_jp_ext, iso2022jp-ext, iso-2022-jp-ext)" ],
  [ "shift_jis", "Japanese (shift_jis, csshiftjis, shiftjis, sjis, s_jis)" ],
  [ "shift_jisx0213", "Japanese (shift_jisx0213, shiftjisx0213, sjisx0213, s_jisx0213)" ],
  [ "shift_jis_2004", "Japanese (shift_jis_2004, shiftjis2004, sjis_2004, sjis2004)" ],
  [ "iso2022_jp_2", "Japanese, Korean, Simplified (iso2022_jp_2, iso2022jp-2, iso-2022-jp-2)" ],
  [ "ptcp154", "Kazakh (ptcp154, csptcp154, pt154, cp154, cyrillic-asian)" ],
  [ "cp949", "Korean (cp949, 949, ms949, uhc)" ],
  [ "euc_kr", "Korean (euc_kr, euckr, korean, ksc5601, ks_c-5601, ks_c-5601-1987, ksx1001, ks_x-1001)" ],
  [ "iso2022_kr", "Korean (iso2022_kr, csiso2022kr, iso2022kr, iso-2022-kr)" ],
  [ "johab", "Korean (johab, cp1361, ms1361)" ],
  [ "iso8859_10", "Nordic languages (iso8859_10, iso-8859-10, latin6, L6)" ],
  [ "cp860", "Portuguese (cp860, 860, IBM860)" ],
  [ "cp866", "Russian (cp866, 866, IBM866)" ],
  [ "koi8_r", "Russian (koi8_r)" ],
  [ "gb2312", "Simplified Chinese (gb2312, chinese, csiso58gb231280, euc-cn, euccn, eucgb2312-cn, gb2312-1980, gb2312-80, iso-ir-58)" ],
  [ "hz", "Simplified Chinese (hz, hzgb, hz-gb, hz-gb-2312)" ],
  [ "cp874", "Thai (cp874)" ],
  [ "big5", "Traditional Chinese (big5, big5-tw, csbig5)" ],
  [ "big5hkscs", "Traditional Chinese (big5hkscs, big5-hkscs, hkscs)" ],
  [ "cp950", "Traditional Chinese (cp950, 950, ms950)" ],
  [ "cp1026", "Turkish (cp1026, ibm1026)" ],
  [ "cp1254", "Turkish (cp1254, windows-1254)" ],
  [ "cp857", "Turkish (cp857, 857, IBM857)" ],
  [ "iso8859_9", "Turkish (iso8859_9, iso-8859-9, latin5, L5)" ],
  [ "mac_turkish", "Turkish (mac_turkish, macturkish)" ],
  [ "koi8_u", "Ukrainian (koi8_u)" ],
  [ "gb18030", "Unified Chinese (gb18030, gb18030-2000)" ],
  [ "gbk", "Unified Chinese (gbk, 936, cp936, ms936)" ],
  [ "cp1006", "Urdu (cp1006)" ],
  [ "cp1258", "Vietnamese (cp1258, windows-1258)" ],
  [ "latin_1", "West Europe (latin_1, iso-8859-1, iso8859-1, 8859, cp819, latin, latin1, L1)" ],
  [ "cp1140", "Western Europe (cp1140, ibm1140)" ],
  [ "cp1252", "Western Europe (cp1252, windows-1252)" ],
  [ "cp500", "Western Europe (cp500, EBCDIC-CP-BE, EBCDIC-CP-CH)" ],
  [ "cp850", "Western Europe (cp850, 850, IBM850)" ],
  [ "iso8859_15", "Western Europe (iso8859_15, iso-8859-15)" ],
  [ "mac_roman", "Western Europe (mac_roman, macroman)" ],
]