#!/usr/bin/python

####################################################################################
#                                                                                  #
# Copyright (c) 2005 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This file is part of Picalo.                                                     #
#                                                                                  #
# Picalo is free software; you can redistribute it and/or modify                   #
# it under the terms of the GNU General Public License as published by             #
# the Free Software Foundation; either version 2 of the License, or                # 
# (at your option) any later version.                                              #
#                                                                                  #
# Picalo is distributed in the hope that it will be useful,                        #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                   #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    #
# GNU General Public License for more details.                                     #
#                                                                                  #
# You should have received a copy of the GNU General Public License                #
# along with Foobar; if not, write to the Free Software                            #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA        #
#                                                                                  #
####################################################################################
#
# 12 Sep 2005 First version
#
####################################################################################

import re, wx, wx.grid, Spreadsheet, Dialogs, Utils, sys
from Languages import lang
from picalo import *
from picalo.base.Calendar import RE_DATE_FORMATS, RE_DATETIME_FORMATS, DateFormatInfo
from picalo.base.Global import is_valid_variable, ensure_unique_list_value, make_valid_variable
from picalo.gui.Dialogs import ExpressionBuilder, FormatEditor
from picalo.base.Expression import PicaloExpression
from ControlValidator import ControlValidator
import MiniHelp


###########################################################
###   Table properties dialog
###
###   This class has a lot of messy code because of string
###   literals having to match.  However, I let this be in this
###   class because it allows the regular Picalo Table class to
###   use simple input_exp and output_exp.  The mess here keeps
###   the rest of the code simple.  Note that I do not want to
###   tie the Picalo code to the GUI -- the GUI is an add-on that
###   is not required.

class FieldType:
  '''Simple class to hold the types of fields Picalo officially supports'''
  def __init__(self, name, typename, realtype, format):
    self.name = name
    self.typename = typename
    self.realtype = realtype
    self.format = format
# IMPORTANT: these strings are referenced throughout this file, if you change any, you must change
# the file methods below to match the new names
FIELD_TYPES = [
  FieldType( 'String',           'unicode',   unicode,  None                ),
  FieldType( 'Integer',          'int',       int,      '#'                 ),
  FieldType( 'Long Integer',     'long',      long,     '#'                 ),
  FieldType( 'Decimal Number',   'number',    number,   '#.00'              ),
  FieldType( 'Date',             'Date',      Date,     '%Y-%m-%d'          ),
  FieldType( 'DateTime',         'DateTime',  DateTime, '%Y-%m-%d %H:%M:%S' ),
  FieldType( 'True/False',       'boolean',   boolean,  None                ),
]
 


class FieldInfo:
  '''Small class to hold the field information for a given field in the table.  See __init__ below.'''
  def __init__(self, name=''):
    self.name = name
    self.type = FIELD_TYPES[0]  # string
    self.format = ''
    self.expression = ''
    self.active = False
    self.index = None  # saves the original position of the column, so we can rearrange at the end
EMPTY_FIELD_INFO = FieldInfo()  # used when no fields are present in the table


class TableProperties(Dialogs.Dialog):
  '''Table setup dialog box'''
  def __init__(self, mainframe, spreadsheet):
    '''Constructor'''
    Dialogs.Dialog.__init__(self, mainframe, 'TableProperties') 
    self.spreadsheet = spreadsheet
    self.new_field_index = 1
    self.table = None
    if spreadsheet:
      self.table = spreadsheet.table
  

  def init(self):
    # set up the name
    if self.spreadsheet != None:
      self.dlg.SetTitle('Table Properties: ' + self.spreadsheet.name)
    self.now = DateTime()  # to show date formats
    self.neworder = False  # set to True if the user changes the order of the columns
    
    # add the types to the drop down
    for fieldtype in FIELD_TYPES:
      self.getControl('type').Append(fieldtype.name, fieldtype)
  
    # set up the dates list for date formatting options (so it has today's date and time on it)
    self.now = DateTime()
    self.datetime_format_options = [ DateTimeFormat(self.now, item.format) for item in RE_DATETIME_FORMATS ]
    self.date_format_options = [ DateFormat(self.now, item.format) for item in RE_DATE_FORMATS]
      
    # validators
    self.getControl('name').SetValidator(ControlValidator('0123456789_ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz'))
      
    # bind events
    self.getControl("insertabove").Bind(wx.EVT_BUTTON, self.insertRowAbove)
    self.getControl("insertbelow").Bind(wx.EVT_BUTTON, self.insertRowBelow)
    self.getControl("deletefield").Bind(wx.EVT_BUTTON, self.deleteRow)
    self.getControl("fieldup").Bind(wx.EVT_BUTTON, self.fieldUp)
    self.getControl("fielddown").Bind(wx.EVT_BUTTON, self.fieldDown)
    self.getControl('formatbutton').Bind(wx.EVT_BUTTON, self.onFormatButton)
    self.getControl('expressionbutton').Bind(wx.EVT_BUTTON, self.onExpressionBuilderButton)
    self.getControl('expressionhelp').Bind(wx.EVT_HYPERLINK, self.onExpressionHelpButton)
    self.getControl('fields').Bind(wx.EVT_LISTBOX, self.onFieldSelect)
    self.getControl('name').Bind(wx.EVT_TEXT, self.onNameChange)
    self.getControl('format').Bind(wx.EVT_TEXT, self.onFormatChange)
    self.getControl('type').Bind(wx.EVT_CHOICE, self.onTypeChange)
    self.getControl('expression').Bind(wx.EVT_TEXT, self.onExpressionChange)
    self.getControl('expressiontype').Bind(wx.EVT_CHECKBOX, self.onExpressionTypeChange)
    
    # finally, if we have an initial table, set up the grid to match it's structure
    if self.table != None:
      for i, col in enumerate(self.table.get_columns()):
        info = FieldInfo()
        info.index = i
        info.name = col.name
        info.type = FIELD_TYPES[0]
        info.format = col.format or ''
        for ft in FIELD_TYPES:
          if (col.column_type == None and ft.typename == 'None') or (col.column_type != None and col.column_type.__name__ == ft.typename):
            info.type = ft
            break
        if isinstance(col.expression, PicaloExpression): 
          info.expression = col.expression.expression
          info.active = True
        elif col.expression:
          info.expression = col.expression
          info.active = True
        else:
          info.expression = col.static_expression or ''
        self.getControl('fields').Append('', info)
        self.updateListBoxItem(self.getControl('fields').GetCount() - 1)
      self.getControl('fields').SetSelection(0)
      self.initPropertiesFields()
      if self.table.is_readonly():
        wx.MessageBox(lang('Please note that this table is read only.  You can inspect it, but any changes you make cannot be saved.'), lang('Table Properties'))
    else:
      self.initPropertiesFields()
      
  def onExpressionHelpButton(self, event=None):
    '''Shows the active expression help text'''
    MiniHelp.show(self.dlg, "Active Calculations", '''
    <html>
    <body>
    Short Answer: Leave this box unchecked.
    <p>
    Long Answer: Picalo features two types of calculated columns: static and active.  Static calculated columns are calculated only
    once, and the results of the calculation is stored in the new field.  The calculation itself is discarded once the column
    values are set.  Because most data do not change when you are analyzing them, static calculated columns are more efficient and stable.  
    <p>
    Active calculated columns are like formulas in a spreadsheet: they update their value when fields in their calculation change.
    Because Picalo needs to recalculate the value each time the field is used, active calculated columns are less efficient.  Use them
    only when you have changing data and need the calculation to update.
    </body>
    </html>
    ''')
    
      
  def onExpressionBuilderButton(self, event=None):
    '''Runs the expression builder'''
    expression = self.getControl('expression').GetValue()
    local_tables = []
    if self.spreadsheet:
      local_tables.append(self.spreadsheet.name)
    builder = ExpressionBuilder(self.mainframe, expression=expression, local_tables=local_tables)
    if builder.runModal():
      self.getControl('expression').SetValue(builder.expression)
      self.onExpressionChange()
      

  def onFormatButton(self, event=None):
    '''Runs the format builder'''
    index = self.getControl('fields').GetSelection()
    if index >= 0:
      info = self.getControl('fields').GetClientData(index)
      if info.type.typename in ('Date', 'DateTime', 'int', 'long', 'float', 'number'):
        editor = FormatEditor(self.mainframe, info.type.typename, info.format)
        if editor.runModal():
          self.getControl('format').SetValue(editor.format)
      else:
        wx.MessageBox(lang('Only number and date fields can be formatted.'), lang('Format Builder'))


  def onFieldSelect(self, event=None):
    '''Responds to the selection of a field in the list box'''
    self.initPropertiesFields()
    
    
  def onNameChange(self, event=None):
    '''Responds to a change in the name'''
    name = self.getControl('name').GetValue()
    index = self.getControl('fields').GetSelection()
    if index >= 0:
      self.getControl('fields').GetClientData(index).name = name
      self.updateListBoxItem()
    
    
  def onTypeChange(self, event=None):
    '''Responds to a change in the type of a box'''
    fieldtype = FIELD_TYPES[self.getControl('type').GetSelection()]
    index = self.getControl('fields').GetSelection()
    if index >= 0:
      self.getControl('fields').GetClientData(index).type = fieldtype
      self.getControl('fields').GetClientData(index).format = fieldtype.format
      self.updateListBoxItem()
      self.initPropertiesFields()
      
      
  def onFormatChange(self, event=None):
    '''Responds to a change in the format of a field'''
    format = self.getControl('format').GetValue()
    index = self.getControl('fields').GetSelection()
    if index >= 0:
      self.getControl('fields').GetClientData(index).format = format != '' and format or None
      

  def onExpressionChange(self, event=None):
    '''Responds to a change in the expression of the field'''
    expression = self.getControl('expression').GetValue()
    index = self.getControl('fields').GetSelection()
    if index >= 0:
      self.getControl('fields').GetClientData(index).expression = expression


  def onExpressionTypeChange(self, event=None):
    '''Responds to a click on the expression type box field'''
    expressionactive = self.getControl('expressiontype').IsChecked()
    index = self.getControl('fields').GetSelection()
    if index >= 0:
      self.getControl('fields').GetClientData(index).active = expressionactive


  def updateListBoxItem(self, index=-1):
    '''Updates the current list box item based on it's FieldInfo object'''
    if index < 0:
      index = self.getControl('fields').GetSelection()
    if index >= 0:
      name = self.getControl('fields').GetClientData(index).name
      typ = self.getControl('fields').GetClientData(index).type.name
      self.getControl('fields').SetString(index, '%s (%s)' % (name, typ))
    
    
  def initPropertiesFields(self):
    '''Initializes all the properties fields based on the selected item in the list'''
    self.dlg.Freeze()
    try:
      # get the currently selected item
      index = self.getControl('fields').GetSelection()
      if index < 0:  # nothing selected
        info = EMPTY_FIELD_INFO
        self.getControl('name').Enable(False)
        self.getControl('namelabel').Enable(False)
        self.getControl('type').Enable(False)
        self.getControl('typelabel').Enable(False)
      else:
        info = self.getControl('fields').GetClientData(index)
        self.getControl('name').Enable(True)
        self.getControl('namelabel').Enable(True)
        self.getControl('type').Enable(True)
        self.getControl('typelabel').Enable(True)

      # name information
      self.getControl('name').SetValue(info.name)
    
      # type information
      self.getControl('type').SetSelection(FIELD_TYPES.index(info.type))
    
      # format information
      self.getControl('format').SetValue(info.format != None and info.format or '')
    
      # expression information
      if info.expression != None:
        self.getControl('expression').SetValue(info.expression)
      else:
        self.getControl('expression').SetValue('')
      self.getControl('expressiontype').SetValue(info.active)
        
    finally:
      self.dlg.Thaw()      


  def insertRowAbove(self, event=None):
    '''Called when the user clicks the insert row above button'''
    fieldbox = self.getControl('fields')
    index = self.getControl('fields').GetSelection()
    if index < 0:
      index = 0
    name = ensure_unique_list_value([ fieldbox.GetClientData(i).name for i in range(fieldbox.GetCount()) ], 'Unnamed')
    self.getControl('fields').Insert(name, index, FieldInfo(name))
    self.new_field_index += 1
    fieldbox.SetSelection(index)
    self.updateListBoxItem(index)
    self.initPropertiesFields()
    

  def insertRowBelow(self, event=None):
    '''Called when the user clicks the insert row below button'''  
    fieldbox = self.getControl('fields')
    index = fieldbox.GetSelection() + 1
    if index <= 0:
      index = 0
    name = ensure_unique_list_value([ fieldbox.GetClientData(i).name for i in range(fieldbox.GetCount()) ], 'Unnamed')
    fieldbox.Insert(name, index, FieldInfo(name))
    self.new_field_index += 1
    fieldbox.SetSelection(index)
    self.updateListBoxItem(index)
    self.initPropertiesFields()
    
  
  def deleteRow(self, event=None):
    '''Called when the user clicks the delete row button'''
    fieldbox = self.getControl('fields')
    if fieldbox.GetCount() > 0:
      index = fieldbox.GetSelection()
      if wx.MessageBox(lang('Remove this field?'), lang('Remove field'), style=wx.YES_NO) == wx.YES:
        fieldbox.Delete(fieldbox.GetSelection())
        fieldbox.SetSelection(min(index, fieldbox.GetCount() - 1))
        self.initPropertiesFields()
    
  
  def fieldUp(self, event=None):
    '''Called when the user clicks the Move Up button'''
    fieldbox = self.getControl('fields')
    index = fieldbox.GetSelection()
    if index > 0:
      info = fieldbox.GetClientData(index)
      fieldbox.SetClientData(index, fieldbox.GetClientData(index-1))
      fieldbox.SetClientData(index-1, info)
      self.updateListBoxItem(index)
      self.updateListBoxItem(index-1)
      fieldbox.SetSelection(index-1)
      self.neworder = True
      
    
  def fieldDown(self, event=None):
    '''Called when the user clicks the Move Down button'''
    fieldbox = self.getControl('fields')
    index = fieldbox.GetSelection()
    if index < fieldbox.GetCount() - 1:
      info = fieldbox.GetClientData(index)
      fieldbox.SetClientData(index, fieldbox.GetClientData(index+1))
      fieldbox.SetClientData(index+1, info)
      self.updateListBoxItem(index)
      self.updateListBoxItem(index+1)
      fieldbox.SetSelection(index+1)
      self.neworder = True
  
    
  def ok(self, event=None):
    '''Handles the save button.  It compares what is in the properties view with
       what was in the table and makes the necessary changes.  This is a little more
       complex to code, but it allows the user to change whatever they want in the 
       view (back and forth) and not really change anything until they hit the
       save button.'''
    # ensure that everything is in order
    fieldbox = self.getControl('fields')
    assert fieldbox.GetCount() > 0, lang('Tables have to have at least one field.')
    colnames = {}
    for i in range(fieldbox.GetCount()):
      info = fieldbox.GetClientData(i)
      assert info.name != '', lang('Please enter a name for all fields before submitting.')
      assert info.type != None, lang('Please enter a field type for all fields.')
      assert not colnames.has_key(info.name), lang('Please ensure that all field names are unique.')
      assert is_valid_variable(info.name), lang('Invalid field name: ') + info.name
      colnames[info.name] = ''
      
    cmds = []
    if self.table == None:   # a new table
      # get a name for the new table
      tablename = ''
      while True:
        tablename = wx.GetTextFromUser(lang('Please enter a name for this new table:'), lang('New Table'), tablename)
        if not tablename:
          return
        if not is_valid_variable(tablename):
          wx.MessageBox(lang('Some changes were made to the new table name to make it a valid Picalo name.'), lang('New Table'))
          tablename = make_valid_variable(tablename)
          continue
        if self.mainframe.getVariable(tablename) != None:
          if wx.MessageBox(lang('An object with this name already exists.  Overwrite it?'), lang('New Table'), style=wx.YES_NO) != wx.YES:
            continue
        break  # if we get here, we have a good table name
      
      # create the new table
      fields = []
      for i in range(fieldbox.GetCount()):
        info = fieldbox.GetClientData(i)
        if info.expression and info.active:   # an expression column
          fields.append('("%s", %s, "%s", None, "%s")' % (info.name, info.type.typename, info.format.replace('"', '\\"'), info.expression))
        elif info.expression:  # a static calculated column
          fields.append('("%s", %s, "%s", "%s")' % (info.name, info.type.typename, info.format.replace('"', '\\"'), info.expression))
        else:  # regular data type column
          fields.append('("%s", %s, "%s")' % (info.name, info.type.typename, info.format.replace('"', '\\"')))
      allfields = ', '.join(fields)
      cmds.append('%s = Table([%s])' % (tablename, allfields))
      
      # set format for the fields
      for i in range(fieldbox.GetCount()):
        info = fieldbox.GetClientData(i)
        if info.format:
          cmds.append('%s.set_format("%s", "%s")' % (tablename, info.name, info.format.replace('"', '\\"')))
          
      # show the new table
      cmds.append(tablename + '.view()')
      
    else:  # modifying an existing table       
      tablename = self.spreadsheet.name
      # change any names or types that were changed
      for i in range(fieldbox.GetCount()):
        info = fieldbox.GetClientData(i)
        if info.index != None:  # only pre-existing columns can have changed names
          # has the name changed?
          if self.table.columns[info.index].name != info.name:
            cmds.append('%s.set_name(%i, "%s")' % (tablename, info.index, info.name.replace('"', '\\"')))
          # has the type changed?
          if self.table.columns[info.index].column_type.__name__ != info.type.typename:
            cmds.append('%s.set_type(%i, %s)' % (tablename, info.index, info.type.typename))
          # has the format changed?
          if not info.format and self.table.columns[info.index].format:  # if format was removed
            cmds.append('%s.set_format(%i, None)' % (tablename, info.index))
          elif info.format and self.table.columns[info.index].format != info.format:  # if format was changed
            cmds.append('%s.set_format(%i, "%s")' % (tablename, info.index, info.format.replace('"', '\\"')))
          # expressions
          oldexp = self.table.columns[info.index].expression and self.table.columns[info.index].expression.expression or None
          if not info.expression and (self.table.columns[info.index].static_expression or self.table.columns[info.index].expression):
            self.table.columns[info.index].static_expression = None
            cmds.append('%s.set_type(%i, %s)' % (tablename, info.index, info.type.typename))
          elif info.expression and not info.active and self.table.columns[info.index].static_expression != info.expression:
            cmds.append('%s.set_type(%i, %s)' % (tablename, info.index, info.type.typename))
            cmds.append('%s.replace_column_values(%s, "%s")' % (tablename, info.index, info.expression.replace('"', '\\"')))
          elif info.expression and info.active and oldexp != info.expression:
            cmds.append('%s.set_type(%i, %s, expression="%s")' % (tablename, info.index, info.type.typename, info.expression.replace('"', '\\"')))
      
      # delete any columns that have been removed
      undeleted_indices = [ fieldbox.GetClientData(i).index for i in range(fieldbox.GetCount()) ]
      for i, colname in enumerate(self.table.get_column_names()):
        if not i in undeleted_indices:  # it's been removed
          cmds.append('%s.delete_column("%s")' % (tablename, colname))
  
      # add any new columns, with their types
      for i in range(fieldbox.GetCount()):
        info = fieldbox.GetClientData(i)
        if info.index == None: # new columns have None
          if info.expression and info.active:
            cmds.append('%s.insert_calculated(%i, "%s", %s, "%s")' % (tablename, i, info.name, info.type.typename, info.expression.replace('"', '\\"')))
          elif info.expression: 
            cmds.append('%s.insert_calculated_static(%i, "%s", %s, "%s")' % (tablename, i, info.name, info.type.typename, info.expression.replace('"', '\\"')))
          else:
            cmds.append('%s.insert_column(%i, "%s", %s)' % (tablename, i, info.name, info.type.typename))
          if info.format:
            cmds.append('%s.set_format(%i, "%s")' % (tablename, i, info.format.replace('"', '\\"')))

      # reorder the columns if needed
      if self.neworder:
        neworder = [ ('"' + fieldbox.GetClientData(i).name + '"') for i in range(fieldbox.GetCount()) ]
        cmds.append("%s.reorder_columns([ %s ])" % (tablename, ', '.join(neworder) ))
            
    # run the commands
    if self.mainframe.execute(cmds):
      self.close()
    
     