#!/usr/bin/env python

####################################################################################
#                                                                                  #
# Copyright (c) 2005 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This file is part of Picalo.                                                     #
#                                                                                  #
# Picalo is free software; you can redistribute it and/or modify                   #
# it under the terms of the GNU General Public License as published by             #
# the Free Software Foundation; either version 2 of the License, or                # 
# (at your option) any later version.                                              #
#                                                                                  #
# Picalo is distributed in the hope that it will be useful,                        #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                   #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    #
# GNU General Public License for more details.                                     #
#                                                                                  #
# You should have received a copy of the GNU General Public License                #
# along with Foobar; if not, write to the Free Software                            #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA        #
#                                                                                  #
####################################################################################


from picalo import *
from picalo.lib import GUID
import os.path, sys, threading, time, types, wx, wx.grid, wx.lib.buttons, decimal, re
from NotebookComponent import NotebookComponent
from TableProperties import TableProperties
import MainFrame, Utils
import Dialogs, Preferences, QueryBuilder
from Languages import lang
from picalo.base.Global import make_valid_variable, is_valid_variable
from picalo.base.Error import error
from picalo.base.Expression import PicaloExpression

  
####################################################
###   Main spreadsheet frame

class Spreadsheet(wx.Panel, NotebookComponent):
  '''A spreadsheet window showing data from a table.
     
     The table variable can be either a Table, a TableList, or a TableArray.
  '''
  def __init__(self, parent, mainframe, name, table):
    self.name = name
    self.objtype = None    # set in onIdle (called below)
    self.objid = None      # set in onIdle (called below)
    self.table = None      # set in onIdle (called below)
    self.tablelist = None  # set in onIdle (called below)
    self.grid_dirty = 0
    self.eventsEnabled = True
    wx.Panel.__init__(self, parent)
    NotebookComponent.__init__(self, parent, mainframe, name)
    
    # some defaults caller can change
    self.autoNewRows = False  # whether to automatically append new rows when the user cursors beyond the end of the table
    self.font = None # set in updatePreferences below
    self.updatePreferences()
    
    # main sizer
    self.SetSizer(wx.BoxSizer(wx.VERTICAL))
    
    # the toolbar
    self.toolsizer = wx.FlexGridSizer(1,3)
    self.toolsizer.AddGrowableCol(1, 1)
    self.GetSizer().Add(self.toolsizer, flag=wx.EXPAND|wx.ALL, border=5)

    # read only button
    self.readonlybutton = wx.lib.buttons.GenBitmapToggleButton(self, -1, None)
    self.readonlybutton.SetBitmapLabel(Utils.getBitmap('editable.png'))
    self.readonlybutton.SetBitmapSelected(Utils.getBitmap('locked.png'))
    self.readonlybutton.Bind(wx.EVT_BUTTON, self.onReadOnlyButton)
    self.toolsizer.Add(self.readonlybutton, border=10, flag=wx.RIGHT|wx.EXPAND|wx.ALIGN_LEFT)      
    
    # filter options
    self.filtertools = wx.Panel(self) 
    self.toolsizer.Add(self.filtertools, flag=wx.ALL|wx.EXPAND|wx.ALIGN_LEFT)
    filtersizer = wx.BoxSizer(wx.HORIZONTAL)
    self.filtertools.SetSizer(filtersizer)
    filtersizer.Add(wx.StaticText(self.filtertools, label=lang("Filter Expression: ")), flag=wx.ALIGN_CENTRE_VERTICAL)
    self.filter = wx.ComboBox(self.filtertools, size=(300, -1))
    filtersizer.Add(self.filter, flag=wx.ALIGN_CENTRE_VERTICAL)
    self.filtertablebutton = wx.BitmapButton(self.filtertools, bitmap=Utils.getBitmap('wizard.png'))
    filtersizer.Add(self.filtertablebutton, flag=wx.ALIGN_CENTRE_VERTICAL | wx.LEFT, border=4)
    self.expressionbuilderbutton = wx.BitmapButton(self.filtertools, bitmap=Utils.getBitmap('expression_builder.png'))
    filtersizer.Add(self.expressionbuilderbutton, flag=wx.ALIGN_CENTRE_VERTICAL | wx.LEFT, border=4)
    self.filterbutton = wx.ToggleButton(self.filtertools, label="Set")
    self.filterbutton.SetValue(False)
    self.filterbutton.SetInitialSize()
    filtersizer.Add(self.filterbutton, flag=wx.ALIGN_CENTRE_VERTICAL | wx.LEFT, border=7)
    
    # right side (table list options)
    self.tablelistindex = 0
    self.tablelistlen = 0
    self.tablelisttools = wx.Panel(self)  # put all in a panel so I can make it invisible with one call
    toolsizer = wx.BoxSizer(wx.HORIZONTAL)
    self.tablelisttools.SetSizer(toolsizer)
    boxsizer = wx.BoxSizer(wx.HORIZONTAL)
    toolsizer.Add(boxsizer, flag=wx.ALL|wx.EXPAND|wx.ALIGN_RIGHT)
    self.slider = wx.SpinCtrl(self.tablelisttools, size=(75, -1), style=wx.SP_ARROW_KEYS|wx.SP_WRAP)
    boxsizer.Add(self.slider)
    self.maxlabel = wx.StaticText(self.tablelisttools)
    boxsizer.Add(self.maxlabel, flag=wx.ALIGN_CENTRE_VERTICAL)
    self.tablelisttools.Show(False)
    
    # right side (query options)
    self.querytools = wx.Panel(self)
    toolsizer = wx.BoxSizer(wx.HORIZONTAL)
    self.querytools.SetSizer(toolsizer)
    boxsizer = wx.BoxSizer(wx.HORIZONTAL)
    toolsizer.Add(boxsizer, flag=wx.ALL|wx.EXPAND|wx.ALIGN_RIGHT|wx.ALIGN_CENTER_VERTICAL)
    self.editquerybutton = wx.Button(self.querytools, label=lang("Edit Query"))
    boxsizer.Add(self.editquerybutton, flag=wx.ALIGN_CENTER_VERTICAL)
    self.querytools.Show(False)
    
    # add the grid components
    self.grid = None
    self.onIdle(table)
    
    # events
    self.Bind(wx.EVT_KEY_DOWN, self.onKeyDown)
    self.filter.Bind(wx.EVT_KEY_DOWN, self.onFilterKeyDown)
    self.expressionbuilderbutton.Bind(wx.EVT_BUTTON, self.onExpressionBuilderButton)
    self.filterbutton.Bind(wx.EVT_TOGGLEBUTTON, self.onFilterButton)
    self.filtertablebutton.Bind(wx.EVT_BUTTON, self.onFilterTableButton)
    self.editquerybutton.Bind(wx.EVT_BUTTON, self.onEditQueryButton)
    def bindFocus(parent):  # bind to all controls since FocusEvent does not propogate (not a CommandEvent)
      for control in parent.GetChildren():
        control.Bind(wx.EVT_SET_FOCUS, self.onFocus)
        bindFocus(control)
    bindFocus(self)
    
    # menu and toolbar
    offset = 9
    self.menuitems = [
      Utils.MenuItem(lang('&File') + '/' + str(offset+0) + '|SEPARATOR'),
      Utils.MenuItem(lang('&File') + '/' + str(offset+1) + '|' + lang('Filter...'), lang('Filter the current table'),  self.onFilterTableButton),
      Utils.MenuItem(lang('&File') + '/' + str(offset+2) + '|' + lang('Sort...'), lang('Sort the current table'),  Dialogs.SortTable(self, self.mainframe)),
      Utils.MenuItem(lang('&File') + '/'+ str(offset+3) + '|' + lang('Transpose...'), lang('Transpose the current table'),  self.menuFileTranspose),
      Utils.MenuItem(lang('&File') + '/' + str(offset+4) + '|' + lang('Row') + '/' + lang('Insert Row'), lang('Insert a new row before the current row'), self.insertRow),
      Utils.MenuItem(lang('&File') + '/' + str(offset+4) + '|' + lang('Row') + '/' + lang('Append Row'), lang('Append a new row at the end of the table'), self.appendRow),
      Utils.MenuItem(lang('&File') + '/' + str(offset+4) + '|' + lang('Row') + '/SEPARATOR'),
      Utils.MenuItem(lang('&File') + '/' + str(offset+4) + '|' + lang('Row') + '/' + lang('Delete Row...'), lang('Delete the current row'), self.deleteRow),
      Utils.MenuItem(lang('&File') + '/' + str(offset+5) + '|SEPARATOR'),
      Utils.MenuItem(lang('&File') + '/' + str(offset+6) + '|' + lang('Save...') + '\t' + lang('Ctrl+S'), lang('Save the current table'), self.menuFileSave),
      Utils.MenuItem(lang('&File') + '/' + str(offset+7) + '|' + lang('Save As...'), lang('Save the current table'), self.menuFileSaveAs),
      Utils.MenuItem(lang('&File') + '/' + str(offset+8) + '|' + lang('Export...'), lang('Export this table to CSV, TSV, or Excel format'), Utils.MenuCaller(self.runFunction, self.name, 'menuExportTable')),
      Utils.MenuItem(lang('&File') + '/' + str(offset+9) + '|SEPARATOR'),
      Utils.MenuItem(lang('&File') + '/' + str(offset+10) + '|' + lang('Copy Table...'), lang('Copy this table to another name'), Utils.MenuCaller(self.runFunction, self.name, 'menuCopyTable')),
      Utils.MenuItem(lang('&File') + '/' + str(offset+11) + '|' + lang('Rename Table...'), lang('Rename this table to another name'), Utils.MenuCaller(self.runFunction, self.name, 'menuRenameTable')),
      Utils.MenuItem(lang('&File') + '/' + str(offset+12) + '|SEPARATOR'),
      Utils.MenuItem(lang('&File') + '/' + str(offset+13) + '|' + lang('Close Table'), lang('Close the current table.'), Utils.MenuCaller(self.runFunction, self.name, 'menuCloseTable')),
      Utils.MenuItem(lang('&File') + '/' + str(offset+14) + '|' + lang('Delete Table...'), lang('Delete this table permanently from disk.'), Utils.MenuCaller(self.runFunction, self.name, 'menuDeleteTable')),
      Utils.MenuItem(lang('&File') + '/' + str(offset+15) + '|SEPARATOR'),
      Utils.MenuItem(lang('&File') + '/' + str(offset+16) + '|' + lang('Table Properties...'), lang('Modify the table properties, such as column names or types'), self.menuTableProperties),
      Utils.MenuItem(lang('&File') + '/' + str(offset+17) + '|SEPARATOR'),
      Utils.MenuItem(lang('&File') + '/' + str(offset+18) + '|' + lang('Page Setup...'), lang('Open the page setup dialog'), self.menuFilePageSetup),
      Utils.MenuItem(lang('&File') + '/' + str(offset+19) + '|' + lang('Print Preview...'), lang('Preview this table as it would print'), self.menuFilePrintPreview),
      Utils.MenuItem(lang('&File') + '/' + str(offset+20) + '|' + lang('Print...'), lang('Print this table'), self.menuFilePrint),
    ]
    self.tabmenuitems = [
      Utils.MenuItem(lang('Close'), lang('Close this tab'), mainframe.menuWindowClose),
      Utils.MenuItem(lang('Close All'), lang('Close all tabs'), mainframe.menuWindowCloseAll),
    ]
    self.tabmenu = Utils.MenuManager(mainframe, wx.Menu(), self.tabmenuitems)
    self.toolbar = Utils.create_toolbar(mainframe, [ 
      Utils.ToolbarItem('SEPARATOR'),
      Utils.ToolbarItem(lang('Save'), 'filesave.png', longhelp=lang('Save the current table'), callback=self.menuFileSave),
      Utils.ToolbarItem(lang('Save As...'), 'filesaveas.png', longhelp=lang('Save the current table'), callback=self.menuFileSaveAs),
      Utils.ToolbarItem('SEPARATOR'),
      Utils.ToolbarItem(lang('Insert Row'), 'view_bottom.png', longhelp=lang('Insert a new row before the current row'), callback=self.insertRow),
      Utils.ToolbarItem(lang('Append Row'), 'view_top_bottom.png', longhelp=lang('Append a new row at the end of the table'), callback=self.appendRow),
      Utils.ToolbarItem('SEPARATOR'),
      Utils.ToolbarItem(lang('Table Properties'), 'configure.png', longhelp=lang('Modify the table properties, such as column types'), callback=self.menuTableProperties),
    ])
    
    
  def get_filename(self):
    '''Returns the filanme of the table in this window'''
    return self.table.filename


  def updatePreferences(self):
    '''Signals that the preferences have been updated and this window should update itself'''
    self.font = wx.Font(10, wx.DEFAULT, wx.NORMAL, wx.NORMAL)
    if Preferences.get('tablefont', ''):
      self.font.SetNativeFontInfoFromString(Preferences.get('tablefont', ''))
    
    
  def showToolbar(self, show=True):
    '''Sets whether to show the table toolbar or not'''
    if show:
      if not self.GetSizer().GetItem(self.toolsizer):
        self.GetSizer().Insert(0, self.toolsizer)      
    else:
      self.GetSizer().Detach(self.toolsizer)
    

  def onFocusSub(self, event=None):
    '''Called from NotebookComponent when this page is focused on'''
    if self.table != None:  # it is None when first loading
      self.mainframe.setStatus(str(self.table.record_count(True)) + " / " + str(self.table.record_count(False)), 1)

    
  def onTableChange(self, table, level):
    '''Called from the Dataset table when data has changed'''
    try:
      self.grid_dirty = max(level, self.grid_dirty)  # if level is already higher than the one being set, don't downgrade it
    except wx.PyDeadObjectError:  # happens when the spreadsheet for a given table has been removed but not garbage collected
      pass
  
  
  def onFilterButton(self, event=None):
    '''Called when the filter button is pressed'''
    if not self.eventsEnabled and event:
      event.Skip()
      return
    # if no text, the button can't be down
    if not self.filter.GetValue() and self.filterbutton.GetValue():
      self.filterbutton.SetValue(False)
    if self.filterbutton.GetValue():  # filter on
      self.filterbutton.SetLabel(lang("Release"))
      filterexpression = self.filter.GetValue().replace("'", "\\'")
      newfilterexpression = re.sub('(?<=[^\=\<\>\!])\=(?=[^\=])', '==', filterexpression)
      if newfilterexpression != filterexpression:
        if wx.MessageBox(lang('Your expression contains a single equals (=) sign which is used for assignment in Picalo and is not normally used in filter expressions.  You probably meant to use a double equals (==), indicating a comparison.  Would you like to replace the expression with the following?') + '\n\n' + newfilterexpression, lang('Warning'), style=wx.YES_NO) == wx.YES:
          filterexpression = newfilterexpression
      self.mainframe.execute("%s.filter('%s')" % (self.name, filterexpression))
    else:  # filter off
      self.filterbutton.SetLabel(lang("Set"))
      if self.table.is_filtered():
        self.mainframe.execute('%s.clear_filter()' % (self.name,))
  
  
  def onFilterKeyDown(self, event=None):
    '''Called when the user hits the enter key in the filter control'''
    if not self.eventsEnabled and event:
      event.Skip()
      return
    if event.KeyCode == wx.WXK_RETURN:
      self.filterbutton.SetValue(True)
      self.onFilterButton()
    else:
      event.Skip() # tells the event system to continue to process the event
      
      
  def onFilterMenu(self, formula):
    '''Called when the user right-clicks a cell in the table and selects a filter'''
    self.filter.SetValue(formula)
    self.filterbutton.SetValue(True)
    self.onFilterButton()


  def onFilterTableButton(self, event=None):
    '''Called hwen the user clicks the filter table button'''
    dlg = Dialogs.FilterTable(self, self.mainframe)
    if dlg.runModal():
      self.filter.SetValue(dlg.expression)
      self.filterbutton.SetValue(True)
      self.onFilterButton()
    

  def onExpressionBuilderButton(self, event=None):
    '''Called when the user clicks on the expression builder button'''
    expression = self.filter.GetValue()
    builder = Dialogs.ExpressionBuilder(self.mainframe, expression=expression, local_tables=[self.name])
    if builder.runModal():
      self.filter.SetValue(builder.expression)


  def onReadOnlyButton(self, event):
    '''Called when the user clicks on the read only button'''
    if not self.eventsEnabled and event:
      event.Skip()
      return
    self.mainframe.execute('%s.set_readonly(%s)' % (self.name, event.GetIsDown()))
    self.readonlybutton.SetToggle(self.table.is_readonly())  # allows button to reset if the set_readonly call failed for some reason
    
    
  def update_readonly(self):
    '''Called from onIdle when a table changes occurs'''
    self.readonlybutton.SetToggle(self.table.is_readonly())
    
      
  def onClosePage(self):
    '''Called when the page is closed (see MainFrame.closePage)'''
    self.table._remove_listener(self.onTableChange)
    if '.' not in self.name and self.mainframe.shell.varExists(self.name):
      self.mainframe.execute('del %s' % (self.name))

    
  def onEditQueryButton(self, event=None):
    '''Called when the user clicks the edit query button'''
    sql = wx.GetTextFromUser(lang('SQL Text:'), lang('Modify Query'), self.table.get_sql())
    if sql and sql != self.table.get_sql():
      connname = self.mainframe.determineQueryConnection(self.table)
      if connname:
        self.mainframe.execute('%s.set_sql("%s", conn=%s)' % (self.name, sql.replace('"', '\\"'), connname))
        
    
  def setAutoNewRows(self, autoNewRows):
    '''Whether to automatically add new rows to the table when the user cursors beyond the end of table.'''
    self.autoNewRows = autoNewRows


  def setEventsEnabled(self, enabled):
    '''Whether or not to process events on this window, such as a right-click on a cell. 
       This allows other dialogs that own a spreadsheet to attach their own events and override
       the events normally attached to a Spreadsheet.
    '''
    self.eventsEnabled = enabled

    
  def onKeyDown(self, event=None):
    if self.autoNewRows and event.KeyCode == wx.WXK_DOWN and self.grid.GetGridCursorRow() == len(self.table)-1:
      self.appendRow()
    else:
      event.Skip() # tells the event system to continue to process the event


  def setModel(self):
    '''Recreates the grid component with the given table.  The grid doesn't like
       changes in its model once it is created, so we create the grid every
       time the model changes (i.e. it is a new table).
    '''
    self.Freeze()
    # remove from our sizer
    if self.grid:
      if self.table:
        self.table._remove_listener(self.onTableChange)
      self.GetSizer().Detach(self.grid)
      self.grid.Unbind(wx.EVT_SET_FOCUS)
      self.grid.Destroy()
      
    # create the grid component and add to our sizer
    self.grid = DatasetGrid(self)
    self.table._add_listener(self.onTableChange)
    self.GetSizer().Add(self.grid, proportion=1, flag=wx.ALL|wx.EXPAND)
    self.grid_dirty = 2  # forces update of the GUI for this new table

    # we have to bind the focus event to everything in the window because focus is not a command event
    self.grid.Bind(wx.EVT_SET_FOCUS, self.onFocus)

    # initialize the slider if a tablelist
    self.toolsizer.Detach(self.querytools)
    self.querytools.Show(False)
    self.toolsizer.Detach(self.tablelisttools)
    self.tablelisttools.Show(False)
    if self.objtype in (TableList, TableArray):
      self.slider.SetRange(0, len(self.tablelist)-1)
      self.slider.SetValue(self.tablelistindex)
      self.maxlabel.SetLabel(" of " + str(len(self.tablelist)))
      self.toolsizer.Add(self.tablelisttools, flag=wx.ALL|wx.EXPAND|wx.ALIGN_RIGHT)
      self.tablelisttools.Show(True)
    elif self.objtype == Database.Query:
      self.toolsizer.Add(self.querytools, flag=wx.ALL|wx.EXPAND|wx.ALIGN_RIGHT)
      self.querytools.Show(True)
      
    # set the readonly toggle
    self.update_readonly()
    
    # thaw back out
    self.Layout()
    self.Thaw()
  

  def onIdle(self, myvar=None):
    '''Called when the system is idle.  We use it here to refresh the GUI.
       We can't always know when changes occur because the user might change
       the table in a script, a wizard, the shell, or a million other ways.
       So we check in the idle time of the system.
       This method is actually called by the mainframe.
    '''
    if myvar == None:  # Dialogs.TextImportWizard sends its variable in to trigger this
      myvar = self.mainframe.shell.getVariable(self.name)
    if myvar == None:  # if still None
      return
    
    # value has changed to a TableList
    if self.objid != id(myvar) and isinstance(myvar, (TableList, TableArray)):  
      if len(myvar) > 0:
        self.objtype = myvar.__class__
        self.objid = id(myvar)
        self.tablelist = myvar
        self.tablelistindex = 0
        self.tablelistlen = len(self.tablelist)
        self.table = self.tablelist[self.tablelistindex]
        self.setModel()
        self.onFocusSub()
      else:
        self.mainframe.closePage(self)
  
    # value has changed to a Table
    elif self.objid != id(myvar) and isinstance(myvar, Table):
      self.objtype = myvar.__class__
      self.objid = id(myvar)
      self.table = myvar
      self.tablelist = None
      self.setModel()
      self.onFocusSub()
      
    # value has changed to a Query
    elif (self.objid != id(myvar) or self.grid_dirty == 3) and isinstance(myvar, Database.Query):  
      assert myvar.table != None, 'You must first execute the query before viewing it.'
      self.objtype = myvar.__class__
      self.objid = id(myvar)
      self.table = myvar
      self.tablelist = None
      self.setModel()
      self.onFocusSub()
    
    # change in tablelist, such as new tables added or removed
    elif self.objid == id(myvar) and self.objtype in (TableList, TableArray) and (self.tablelistindex != self.slider.GetValue() or self.tablelistlen != len(self.tablelist) or self.tablelist[self.tablelistindex] is not self.table):  
      self.tablelistindex = min(self.slider.GetValue(), len(self.tablelist) - 1)
      if self.tablelistindex >= 0:
        self.table = myvar[self.tablelistindex]
        self.setModel()
        self.onFocusSub()
      else:
        self.mainframe.closePage(self)
        
    # just a level 1 data change (cell value changed)
    elif self and self.grid_dirty == 1:  
      self.update_readonly()
      self.grid.ForceRefresh()
      self.onFocusSub()
      self.grid_dirty = 0
      
    # level 2 data change (rows added, deleted, etc.)
    elif self and self.grid_dirty >= 2:  
      self.update_readonly()
      self.grid.BeginBatch()
      
      # update the filter display
      if self.table.get_filter_expression():
        self.filter.SetValue(self.table.get_filter_expression().expression)
        self.filter.Insert(self.filter.GetValue(), 0)
        self.filter.Enable(False)
        self.expressionbuilderbutton.Enable(False)
        self.filtertablebutton.Enable(False)
        self.filterbutton.SetValue(True)
        self.filterbutton.SetLabel("Release")
        for i in range(self.filter.GetCount() - 1, 0, -1):
          if self.filter.GetValue() == self.filter.GetString(i):
            self.filter.Delete(i)
        while self.filter.GetCount() > 7:  # keep the last 7 filter expressions
          self.filter.Delete(self.filter.GetCount() - 1)
      else:
        self.filterbutton.SetValue(False)
        self.filter.Enable(True)
        self.filterbutton.SetLabel("Set")
        self.expressionbuilderbutton.Enable(True)
        self.filtertablebutton.Enable(True)
        
      # update for added or removed rows or cols
      for current, new, delmsg, addmsg in [
          (self.grid.model._rows, len(self.table), wx.grid.GRIDTABLE_NOTIFY_ROWS_DELETED, wx.grid.GRIDTABLE_NOTIFY_ROWS_APPENDED),
          (self.grid.model._cols, len(self.table.columns), wx.grid.GRIDTABLE_NOTIFY_COLS_DELETED, wx.grid.GRIDTABLE_NOTIFY_COLS_APPENDED),
      ]:
        if new < current:
          msg = wx.grid.GridTableMessage(self.grid.model,delmsg,new,current-new)
          self.grid.ProcessTableMessage(msg)
        elif new > current:
          msg = wx.grid.GridTableMessage(self.grid.model,addmsg,new-current)
          self.grid.ProcessTableMessage(msg)
      self.grid.EndBatch()
      self.grid.model._rows = self.grid.model.GetNumberRows()
      self.grid.model._cols = self.grid.model.GetNumberCols()

      # update the scrollbars and the displayed part of the grid
      self.grid.AdjustScrollbars()
      self.grid.ForceRefresh()
      self.onFocusSub()
      self.grid_dirty = 0
      
    else:
      self.grid_dirty = 0
      
        
  def is_changed(self):
    '''Returns whether the window has changes'''
    return self.table.is_changed()        
        
  
  def refresh(self, refresh_level=2):
    '''Forces a refresh of the table''' 
    self.grid_dirty = refresh_level

    
  def runFunction(self, tablename, funcname):
    '''Runs the given function.  These functions are shared with right-clicking from MainFrame.'''
    func = globals()[funcname]
    func(self.mainframe, tablename)
    

  def menuFileSave(self, event=None):
    '''Saves a table if it has a filename, otherwise calls save as'''
    if self.objtype in (TableList, TableArray):
      return self.mainframe.objecttree.menuSave(value=self.tablelist)
    else:
      return self.mainframe.objecttree.menuSave(value=self.table)
  

  def menuFileSaveAs(self, event=None):
    '''Saves a table to a new filename'''
    if self.objtype in (TableList, TableArray):
      return self.mainframe.objecttree.menuSaveAs(value=self.tablelist)
    else:
      return self.mainframe.objecttree.menuSaveAs(value=self.table)


  def menuFileRemove(self, event=None):
    '''Removes the table from active memory'''
    if not self.canClose():
      return False
    if wx.MessageBox(lang('Close this table and remove from the list of tables?\n\n(any unsaved changes will be lost)'), lang('Close Table'), style=wx.YES_NO) == wx.YES:
      self.mainframe.shell.run('del ' + self.name)


  def menuFilePageSetup(self, event=None):
    '''Opens the page setup dialog'''
    self.mainframe.printer.PageSetup()
    
    
  def print_html_data(self):
    '''Creates the html document for printing'''
    html = []
    html.append('<html>')
    html.append('<body>')
    html.append('<table border="2" cellspacing="0" cellpadding="2">')
    if self.table:
      # column headings
      colnames = self.table.get_column_names()
      html.append('<tr>')
      html.append('<th>&nbsp;</th>')
      for col in colnames:
        html.append('<th><b>' + col + '</b></th>')
      html.append('</tr>')
      
      # rows of the table
      for i, row in enumerate(self.table):
        html.append('<tr>')
        html.append('<td>' + unicode(i+1) + '</td>')
        for col in colnames:
          html.append('<td>' + unicode(row[col]) + '</td>')
        html.append('</tr>')
    html.append('</table>')
    html.append('</body>')
    html.append('</html>')
    return ''.join(html)
    
    
  def menuFilePrintPreview(self, event=None):
    '''Previews the spreadsheet'''
    self.mainframe.printer.SetHeader('<center><font size=+1><b>' + self.name + '</b></font></center>')
    self.mainframe.printer.SetFooter('<center>' + lang('Page') + ' @PAGENUM@</center>')
    self.mainframe.printer.PreviewText(self.print_html_data())


  def menuFilePrint(self, event=None):
    '''Prints the spreadsheet'''
    self.mainframe.printer.SetHeader('<center><font size=+1><b>' + self.name + '</b></font></center>')
    self.mainframe.printer.SetFooter('<center>' + lang('Page') + '@PAGENUM@</center>')
    self.mainframe.printer.PrintText(self.print_html_data())
    

  #####  ADDING NEW ROWS AND DELETING EXISTING ROWS  #####
  def insertRow(self, event=None):
    if len(self.table) == 0:
      self.appendRow(event)
    else:
      self.table.insert(self.grid.GetGridCursorRow())

  def insertRowBelow(self, event=None):
    if self.grid.GetGridCursorRow() == len(self.table) - 1:
      self.appendRow(event)
    else:
      self.table.insert(self.grid.GetGridCursorRow()+1)
      if len(self.table) > 1:
        self.grid.SetGridCursor(self.grid.GetGridCursorRow()+1, self.grid.GetGridCursorCol())
    
  def appendRow(self, event=None):
    self.table.append()
    if len(self.table) > 1:
      self.grid.SetGridCursor(len(self.table)-1, self.grid.GetGridCursorCol())
    
  def deleteRow(self, event=None):
    if wx.MessageBox(lang('Delete the current row?\n\n(this action is permanent)'), lang('Delete Row'), wx.YES_NO | wx.ICON_QUESTION) == wx.YES:
      row = self.grid.GetGridCursorRow()
      del self.table[row]
      
    

  ###  TABLE OPERATIONS   ###
  
  def menuTableProperties(self, event=None):
    dlg = TableProperties(self.mainframe, self)
    dlg.runModal()
    self.grid.ForceRefresh()


  def findDialog(self):
    '''Shows the find dialog'''
    if not self.finddlg:
      self.finddlg = Dialogs.FindInTable(self.mainframe, self)
    self.finddlg()
    
    
  def findNextDialog(self):
    '''Shows the find dialog'''
    if not self.finddlg:
      return self.findDialog()
    else:
      self.finddlg.doFind()
      

  def menuFileTranspose(self, event=None):
    '''Tranposes the currently-selected table'''
    assert isinstance(self.table, (Table, Database.Query)), lang('Only regular tables or queries can be transposed.')

    # get the new table name
    newtablename = self.name + '_transposed'
    while True:
      newtablename = wx.GetTextFromUser(lang("Please enter the new table name"), lang("Transpose"), newtablename)
      if not newtablename:
        return
      if not self.mainframe.getVariable(newtablename) or wx.MessageBox(lang('A variable with this name already exists.  Overwrite it?'), lang('Copy To Picalo Table'), style=wx.YES_NO) == wx.YES:
        break
    self.mainframe.execute([
      self.name + '_transposed = Simple.transpose(' + self.name + ')',
      self.name + '_transposed.view()',
    ])
      


  ########  COPY AND PASTE   ###########

  def onCopy(self, event=None):
    '''Copies data to the clipboard'''
    if (wx.TheClipboard.Open()):
      try:
        # get the start and end of the range to copy
        if len(self.grid.GetSelectedCols()) > 0:  # entire columns selected
          cols = self.grid.GetSelectedCols()
          rows = range(len(self.table))
          
        elif len(self.grid.GetSelectedRows()) > 0:  # entire rows selected
          cols = range(len(self.table.get_column_names()))
          rows = self.grid.GetSelectedRows()
          
        elif len(self.grid.GetSelectionBlockTopLeft()) > 0 and len(self.grid.GetSelectionBlockBottomRight()) > 0: # multiple cells selected
          row1, col1 = self.grid.GetSelectionBlockTopLeft()[0]
          row2, col2 = self.grid.GetSelectionBlockBottomRight()[0]
          cols = range(col1, col2+1)
          rows = range(row1, row2+1)
          
        else:  # single cell selected
          cols = [ self.grid.GetGridCursorCol() ]
          rows = [ self.grid.GetGridCursorRow() ]

        # get the data
        datarows = []
        for row in rows:
          datacols = []
          for col in cols:
            datacols.append(unicode(self.table[row][col]))
          datarows.append('\t'.join(datacols))
        data = '\n'.join(datarows)

        # set the data in the clipboard  
        wx.TheClipboard.SetData(wx.TextDataObject(data))
        
      finally: # make sure we release our hold on the clipboard
        wx.TheClipboard.Close()
      
    
  def onPaste(self, event=None):
    '''Pastes clipboard data the current Picalo table'''
    if (wx.TheClipboard.Open()):
      try:
        # get the clipboard data
        clip = wx.TextDataObject()
        wx.TheClipboard.GetData(clip)
        data = clip.GetText()
        
        # set the current cell to the start of any selected portion
        if len(self.grid.GetSelectedCols()) > 0:
          self.grid.SetGridCursor(0, self.grid.GetSelectedCols()[0])
        elif len(self.grid.GetSelectedRows()) > 0:
          self.grid.SetGridCursor(self.grid.GetSelectedRows()[0], 0)
        elif len(self.grid.GetSelectionBlockTopLeft()) > 0:
          self.grid.SetGridCursor(self.grid.GetSelectionBlockTopLeft()[0][0], self.grid.GetSelectionBlockTopLeft()[0][1])
        
        # go through from the current cell and insert the data
        row = self.grid.GetGridCursorRow()
        col = self.grid.GetGridCursorCol()
        for line in data.splitlines():
          # ensure we have enough rows to add to
          if row >= len(self.table): # need to add a row
            rowobj = self.table.append()
          else:
            rowobj = self.table[row]
          row += 1
          
          # add the values in the columns
          for c, val in enumerate(line.split('\t')):
            colidx = col + c
            if colidx < len(rowobj):
              rowobj[colidx] = val
              
        # refresh in case we added rows
        self.refresh()
      
      finally: # make sure we release our hold on the clipboard
        wx.TheClipboard.Close()
    
    
  def onSetNone(self, event=None):
    row = self.grid.GetGridCursorRow()
    col = self.grid.GetGridCursorCol()
    if col >= 0 and row >= 0:
      self.table[row][col] = None


  def onEditInDialog(self, event=None):
    row = self.grid.GetGridCursorRow()
    col = self.grid.GetGridCursorCol()
    if col >= 0 and row >= 0:
      dlg = Dialogs.ValueEditor(self.mainframe, self.table[row][col])
      if dlg.runModal():
        self.table[row][col] = dlg.value


  def onExpressionBuilder(self, event=None):
    row = self.grid.GetGridCursorRow()
    col = self.grid.GetGridCursorCol()
    if col >= 0 and row >= 0:
      builder = Dialogs.ExpressionBuilder(self.mainframe, local_tables=[self.name])
      while builder.runModal():
        expression = PicaloExpression(builder.expression)
        value = expression.evaluate([{'record': self.table[row], 'recordindex':row}, self.table[row]])
        if isinstance(value, error):
          wx.MessageBox(str(value.get_message()), lang("Expression Error"))
          builder.initial_expression = builder.expression
        else:
          self.table[row][col] = value
          break
        


###################################################################################
###   Menu options shared between the main menu and right-clicking in the tree


def menuCopyTable(mainframe, tablename):
  '''Copies a table to another name'''
  # get the new name and ensure we have a valid variable name
  newtablename = ''
  while True:
    newtablename = wx.GetTextFromUser(lang("Please enter the new table name"), lang("Copy Table"), default_value=newtablename)
    if not newtablename:  # if cancelled
      return
    # ensure it's a valid variable name
    if not is_valid_variable(newtablename):
      newtablename = make_valid_variable(newtablename)
      if wx.MessageBox(lang('In order to make a valid variable name, your entry has been changed to ') + newtablename + lang('.  Continue?'), lang('Copy Table'), style=wx.YES_NO) != wx.YES:
        continue
    # see if we already have a variable by this name
    if mainframe.getVariable(newtablename) != None:
      if wx.MessageBox(lang('A variable by this name exists and will be overwritten.  Continue?'), lang('Copy Table'), style=wx.YES_NO) != wx.YES:
        continue
    # if we get here, break out and do it
    break
  
  # run the copy statement
  mainframe.execute([
    "%s = %s[:]" % (newtablename, tablename),
    "%s.view()" % (newtablename, ),
  ])
    

def menuRenameTable(mainframe, tablename):
  '''Renames a table to another name'''
  if not tablename or not mainframe.getVariable(tablename):
    wx.MessageBox(lang('You must open the table before it can be renamed.'), lang('Rename Table'))
    return
  # get the new name and ensure we have a valid variable name
  newtablename = ''
  while True:
    newtablename = wx.GetTextFromUser(lang("Please enter the new table name"), lang("Rename Table"), default_value=newtablename)
    if not newtablename:  # if cancelled
      return
    # ensure it's a valid variable name
    if not is_valid_variable(newtablename):
      newtablename = make_valid_variable(newtablename)
      if wx.MessageBox(lang('In order to make a valid variable name, your entry has been changed to ') + newtablename + lang('.  Continue?'), lang('Rename Table'), style=wx.YES_NO) != wx.YES:
        continue
    # see if we already have a variable by this name
    if mainframe.getVariable(newtablename) != None:
      if wx.MessageBox(lang('A variable by this name exists and will be overwritten.  Continue?'), lang('Rename Table'), style=wx.YES_NO) != wx.YES:
        continue
    # if we get here, break out and do it
    break
  
  # run the copy statement
  mainframe.execute([
    "%s = %s" % (newtablename, tablename),
    "del %s" % (tablename, ),
    "%s.view()" % (newtablename, ),
  ])

    
def menuExportTable(mainframe, tablename):
  '''Exports a table to CSV or TSV'''
  table = mainframe.getVariable(tablename)
  # open the file dialog
  while True:
    dlg = wx.FileDialog(mainframe, message=lang('Export Table...'), defaultDir=mainframe.projectdir, defaultFile="", style=wx.SAVE | wx.CHANGE_DIR, wildcard='|'.join([a+'|*'+b for a,b in MainFrame.FILE_TYPES[1:]]))
    if dlg.ShowModal() != wx.ID_OK:
      return
    filename = dlg.GetPath()
    if os.path.splitext(filename)[1] != MainFrame.FILE_TYPES[dlg.GetFilterIndex()+1][1]:
      filename += MainFrame.FILE_TYPES[dlg.GetFilterIndex()+1][1]
    if os.path.exists(filename):
      if wx.MessageBox(lang('A file with this name already exists.  Overwrite it?'), lang('Export'), style=wx.YES_NO) != wx.YES:
        continue
    # save the table
    filetype = MainFrame.FILE_TYPES[dlg.GetFilterIndex()+1][1]
    cmd = []
    if filetype == '.csv':
      cmd = tablename + '.save_csv("' + filename.replace('\\', '/').replace('"', '\\"') + '")'
    elif filetype == '.tsv':
      cmd = tablename + '.save_tsv("' + filename.replace('\\', '/').replace('"', '\\"') + '")'
    elif filetype == '.xls':
      cmd = tablename + '.save_excel("' + filename.replace('\\', '/').replace('"', '\\"') + '")'
    mainframe.execute(cmd)
    return
          
          
def menuCloseTable(mainframe, tablename):
  '''Closes the table with the given name'''
  page = mainframe.getPage(tablename, Spreadsheet)
  if page != None and page.canClose():
    mainframe.closePage(page)
    
    
def menuDeleteTable(mainframe, tablename):
  if wx.MessageBox(lang('This table will be permanently removed from disk.  Continue?'), lang('Delete Table'), style=wx.YES_NO) != wx.YES:
    return
  cmds = []
  table = mainframe.getVariable(tablename)
  if table:
    cmds.append('del ' + tablename)
  if table.filename:
    cmds.append('os.remove("' + table.filename + '")')
  if len(cmds) > 0:
    mainframe.execute(cmds)
    
    
def menuCombineTableList(mainframe, tablename):
  '''Recombines a table list back into a single table.'''
  table = mainframe.getVariable(tablename)
  assert isinstance(table, TableArray), lang('This type of table list cannot be combined into a regular table.')
  newtablename = tablename + '_combined'
  while True:
    newtablename = wx.GetTextFromUser(lang("Please enter the new table name"), lang("Combine Table List"), default_value=newtablename)
    if not newtablename:  # if cancelled
      return
    # ensure it's a valid variable name
    if not is_valid_variable(newtablename):
      newtablename = make_valid_variable(newtablename)
      if wx.MessageBox(lang('In order to make a valid variable name, your entry has been changed to ') + newtablename + lang('.  Continue?'), lang('Combine Table List'), style=wx.YES_NO) != wx.YES:
        continue
    # see if we already have a variable by this name
    if mainframe.getVariable(newtablename) != None:
      if wx.MessageBox(lang('A variable by this name exists and will be overwritten.  Continue?'), lang('Combine Table List'), style=wx.YES_NO) != wx.YES:
        continue
    # if we get here, break out and do it
    break

  # do it!
  mainframe.execute([
    '%s = %s.combine()' % (newtablename, tablename),
    '%s.view()' % (newtablename, ),
  ])
  
  
  
  

####################################################
###   The visual Grid component

class DatasetGrid(wx.grid.Grid):
  def __init__(self, parent):
    '''The visual grid component for the table'''
    wx.grid.Grid.__init__(self, parent)
    self.parent = parent # link back to parent
    self.model = DatasetModel(self, self.parent.table)
    self.SetTable(self.model, 1)
    self.col_types = []
    self.cell_renderer = PicaloCellRenderer(self)
    self.SetDefaultRenderer(self.cell_renderer)
    self.needs_sizing = False
    self.editor_start_value = None
    # note: do not call AutoSizeColumns as it kills memory on large tables
      
    # events
    self.Bind(wx.grid.EVT_GRID_LABEL_RIGHT_CLICK, self.onLabelRight)
    self.Bind(wx.grid.EVT_GRID_CELL_RIGHT_CLICK, self.onCellRight)
    self.Bind(wx.EVT_SIZE, self.onSizer)
    
    # popup menu variables
    self.selected_col = -1
    self.selected_row = -1
    

  def onSizer(self, event):
    '''Processes a size change on the grid'''
    width = self.GetClientSize()[0]
    if width > 0:
      self.Freeze()
      try:
        # size the columns as best as possible
        xpadding = 10
        dc = wx.ClientDC(self)
        columns = self.parent.table.get_columns()
        col_widths = [ 0 for col in columns ]
        # set the row label width
        rowlabelwidth = dc.GetTextExtent(str(len(self.parent.table)))[0] + (xpadding*2)
        self.SetRowLabelSize(rowlabelwidth)
        # inspect up to the first 100 rows for their width
        dc.SetFont(wx.NORMAL_FONT) 
        for r in range(min(len(self.parent.table), 100)):
          for c in range(len(columns)):
            col_widths[c] = max(col_widths[c], self.cell_renderer.GetBestSize(self, None, dc, r, c)[0] + xpadding)
        # set the column widths
        dc.SetFont(self.GetLabelFont()) 
        for c in range(len(columns)):
          # account for the width of the column header
          col_widths[c] = max(col_widths[c], dc.GetTextExtent(columns[c].name)[0] + (xpadding*2))
          # max the width out at some level
          col_widths[c] = min(col_widths[c], width)
        # if we're smaller than the window, fill it out (up to 1.5 times the width of any column)
        percentmore = float(width - rowlabelwidth) / float(sum(col_widths))
        if percentmore > 1.0:
          for c in range(len(columns)):
            col_widths[c] = int(min(width, col_widths[c] * 1.5, col_widths[c] * percentmore))
        # finally, set the column width
        for c in range(len(columns)):
          self.SetColSize(c, col_widths[c])
        del dc
      finally:
        self.Thaw()
    event.Skip()
  

  def onLabelRight(self, event=None):
    '''Processes a right click on a column'''
    if not self.parent.eventsEnabled and event:
      event.Skip()
      return
    self.selected_col = event.GetCol()
    self.selected_row = event.GetRow()
    if self.selected_col >= 0:  # column selected
      colmenu = Utils.MenuManager(self.parent.mainframe, wx.Menu(), [
        Utils.MenuItem(lang('Sort ascending'), lang('Sort the table by this column in ascending order'), self.sortAscending),
        Utils.MenuItem(lang('Sort descending'), lang('Sort the table by this column in descending order'), self.sortDescending),
        Utils.MenuItem('SEPARATOR'),
        Utils.MenuItem(lang('Table properties...'), lang('Change the table properties'), self.parent.menuTableProperties),
      ])
      self.SetGridCursor(self.GetGridCursorRow(), self.selected_col)
      self.SelectCol(self.selected_col)
      self.parent.mainframe.PopupMenu(colmenu.topmenu)
      colmenu.Destroy()
      
    elif self.selected_row >= 0:  # row selected
      self.SetGridCursor(self.selected_row, self.GetGridCursorCol())
      self.SelectRow(self.selected_row)
      rowmenu = Utils.MenuManager(self.parent.mainframe, wx.Menu(), [
        Utils.MenuItem(lang('Insert row above'), lang('Insert a row above this row'), self.parent.insertRow),
        Utils.MenuItem(lang('Insert row below'), lang('Insert a row below this row'), self.parent.insertRowBelow),
        Utils.MenuItem('SEPARATOR'),
        Utils.MenuItem(lang('Delete row...'), lang('Delete this row'), self.parent.deleteRow),
      ])
      self.parent.mainframe.PopupMenu(rowmenu.topmenu)
      rowmenu.Destroy()
      
    else:  # table selected
      tablemenu = Utils.MenuManager(self.parent.mainframe, wx.Menu(), [
        Utils.MenuItem(lang('Table properties...'), lang('Change the table properties'), self.parent.menuTableProperties),
      ])
      self.parent.mainframe.PopupMenu(tablemenu.topmenu)
      tablemenu.Destroy()
      
      
  def onCellRight(self, event=None):
    '''Processes a right click of a cell'''
    if not self.parent.eventsEnabled and event:
      event.Skip()
      return
    self.selected_col = event.GetCol()
    self.selected_row = event.GetRow()
    if self.selected_col >= 0 and self.selected_row >= 0:
      self.SetGridCursor(self.selected_row, self.selected_col)
      colname = self.parent.table.column(self.selected_col).name
      menuitems = [
        Utils.MenuItem(lang('Copy'), lang('Copy the contents of this cell to the clipboard.'), self.parent.onCopy),
        Utils.MenuItem(lang('Paste'), lang('Paste the contents of the clipboard starting at this cell.'), self.parent.onPaste),
        Utils.MenuItem('SEPARATOR'),
        Utils.MenuItem(lang('Edit in Separate Dialog'), lang('View/Edit the value of this cell in a separate dialog.'), self.parent.onEditInDialog),
        Utils.MenuItem(lang('Calculate in the Expression Builder'), lang('View/Edit the value of this cell in the expression builder.'), self.parent.onExpressionBuilder),
        Utils.MenuItem('SEPARATOR'),
        Utils.MenuItem(lang('Set to None (<N>)'), lang('Set this cell equal to the None value'), self.parent.onSetNone),      
      ]
      if isinstance(self.parent.table[self.selected_row][self.selected_col], (unicode, str, int, long, bool, decimal.Decimal, DateTime, Date, currency)):
        cellvalue = unicode(self.parent.table[self.selected_row][self.selected_col])
        if not isinstance(self.parent.table[self.selected_row][self.selected_col], (int, long, bool, decimal.Decimal, DateTime, Date, currency)):
          cellvalue = '"' + cellvalue.replace('"', '\\"') + '"'
        if len(cellvalue) < 50:
          menuitems.extend([
            Utils.MenuItem('SEPARATOR'),
            Utils.MenuItem(lang('Filter: ') + colname + ' == ' + cellvalue, lang('Filter to rows that equal ') + cellvalue, Utils.MenuCaller(self.parent.onFilterMenu, "%s == %s" % (colname, cellvalue))),
            Utils.MenuItem(lang('Filter: ') + colname + ' > ' + cellvalue, lang('Filter to rows that are greater than ') + cellvalue, Utils.MenuCaller(self.parent.onFilterMenu, "%s > %s" % (colname, cellvalue))),
            Utils.MenuItem(lang('Filter: ') + colname + ' >= ' + cellvalue, lang('Filter to rows that are greater than or equal to ') + cellvalue, Utils.MenuCaller(self.parent.onFilterMenu, "%s >= %s" % (colname, cellvalue))),
            Utils.MenuItem(lang('Filter: ') + colname + ' < ' + cellvalue, lang('Filter to rows that are less ') + cellvalue, Utils.MenuCaller(self.parent.onFilterMenu, "%s < %s" % (colname, cellvalue))),
            Utils.MenuItem(lang('Filter: ') + colname + ' <= ' + cellvalue, lang('Filter to rows that are less than or equal to ') + cellvalue, Utils.MenuCaller(self.parent.onFilterMenu, "%s <= %s" % (colname, cellvalue))),
            Utils.MenuItem(lang('Clear All Filters'), lang('Clear all filters on this table.'), Utils.MenuCaller(self.parent.onFilterMenu, "")),
          ])
      menu = Utils.MenuManager(self.parent.mainframe, wx.Menu(), menuitems)
      self.parent.mainframe.PopupMenu(menu.topmenu)
      menu.Destroy()
   
  
  def sortAscending(self, event=None):
    '''Sorts the table by the currently-selected column'''
    cmd = 'Simple.sort(' + self.parent.name + ', True, "' + self.parent.table.column(self.selected_col).name + '")'
    self.parent.mainframe.execute(cmd)
    
    
  def sortDescending(self, event=None):
    '''Sorts the table by the currently-selected column'''
    cmd = 'Simple.sort(' + self.parent.name + ', False, "' + self.parent.table.column(self.selected_col).name + '")'
    self.parent.mainframe.execute(cmd)
    
    
  # note that you can't really override much in Grid because most of the calls
  # stay in the wxWidgets C++ code.  When you override you only override the 
  # python proxy.  Try using the DatasetModel object below as it gets called
  # by the C++ code.
  

###################################################  
###   The back-end model for the visual grid
###   This is what interfaces with the actual table

class DatasetModel(wx.grid.PyGridTableBase):
  def __init__(self, parent, table):
    wx.grid.PyGridTableBase.__init__(self)
    self.parent = parent
    self.table = table
    self._rows = self.GetNumberRows()
    self._cols = self.GetNumberCols()
    
    
  def GetNumberRows(self):
    return len(self.table)
    
    
  def GetNumberCols(self):
    return len(self.table.get_columns())
    
    
  def IsEmptyCell(self, row, col):
    if row < len(self.table) and col < len(self.table.columns): # can get out of sync during refreshing
      return self.table[row][col] == None
    return False
    
    
  def GetRowLabelValue(self, rownum):
    return unicode(rownum)  # the standard grid table base returns rownum+1, we want regular zero based
    

  def GetColLabelValue(self, col):
    if col < len(self.table.columns): # can get out of sync during refreshing
      return self.table.get_columns()[col].name
    return ''
    
    
  def GetValue(self, row, col):
    if row < len(self.table) and col < len(self.table.columns): # can get out of sync during refreshing
      return self.table[row][col]
    return None
    
    
  def SetValue(self, row, col, value):
    if row < len(self.table) and col < len(self.table.columns): # can get out of sync during refreshing
      if value == '' and self.table[row][col] == None: # leave as none
        return
      self.table[row][col] = value
  


##############################################
###  The custom Picalo cell renderer

class PicaloCellRenderer(wx.grid.PyGridCellRenderer):
  '''A custom Picalo cell renderer that uses the output expression for display'''

  def __init__(self, parent):
    wx.grid.PyGridCellRenderer.__init__(self)
    self.parent = parent
    self.BACKGROUND_SELECTED = wx.SystemSettings_GetColour(wx.SYS_COLOUR_HIGHLIGHT) 
    self.TEXT_SELECTED = wx.SystemSettings_GetColour(wx.SYS_COLOUR_HIGHLIGHTTEXT) 
    self.BACKGROUND = wx.WHITE #wx.SystemSettings_GetColour(wx.SYS_COLOUR_WINDOW) 
    self.TEXT = wx.SystemSettings_GetColour(wx.SYS_COLOUR_WINDOWTEXT) 


  def Draw(self, grid, attr, dc, rect, row, col, isSelected): 
    """Customisation Point: Draw the data from grid in the rectangle with attributes using the dc""" 
    # when columns/rows are removed, the table temporarily asks for values that no longer exists
    if row >= len(self.parent.model.table) or col >= len(self.parent.model.table.columns):
      return
    
    # start the clipping region
    dc.SetClippingRegion(rect.x, rect.y, rect.width, rect.height) 
    try: 

      # color in the background
      if isSelected: 
        dc.SetBrush(wx.Brush(self.BACKGROUND_SELECTED, wx.SOLID)) 
      else: 
        dc.SetBrush(wx.Brush(self.BACKGROUND, wx.SOLID)) 
      try: 
        dc.SetPen(wx.TRANSPARENT_PEN) 
        dc.DrawRectangle(rect.x, rect.y, rect.width, rect.height) 
      finally: 
        dc.SetPen(wx.NullPen) 
        dc.SetBrush(wx.NullBrush) 
  
      # draw the text    
      value = self.parent.model.table[row][col]
      text = self.GetValueAsText(row, col) 
      dc.SetFont(grid.parent.font)
      textwidth, textheight = dc.GetTextExtent(text)
      previousText = dc.GetTextForeground() 
      if isSelected: 
        dc.SetTextForeground(self.TEXT_SELECTED) 
      else: 
        dc.SetTextForeground(self.TEXT) 
      try: 
        ypos = rect.y + rect.height - textheight - 2
        if value is None:  # center this one
          xpos = rect.x + 2 + ((rect.width - textwidth) / 2)
        elif isinstance(value, (types.IntType, types.LongType, types.FloatType, number, decimal.Decimal)):  # right aligned
          xpos = rect.x + rect.width - textwidth - 2
        else:  # left aligned
          xpos = rect.x + 2
        dc.DrawText(text, xpos, ypos) 
      finally: 
        dc.SetTextForeground(previousText) 

    finally: 
      # end the clipping region
      dc.DestroyClippingRegion() 


  def GetBestSize(self, grid, attr, dc, row, col): 
    '''Customisation Point: Determine the appropriate (best) size for the control, return as wxSize''' 
    dc.SetFont(grid.parent.font)
    x, y = dc.GetTextExtent(self.GetValueAsText(row, col)) 
    return wx.Size(x+2, y+2) 
    
    
  def GetValueAsText(self, row, col): 
    '''Retrieve the current value for row,col as a text string'''
    try:
      val = self.parent.model.table[row][col]
      return self.parent.model.table.columns[col].format_value(val)
    except Exception, e:
      return unicode(e)



