#!/usr/bin/python

####################################################################################
#                                                                                  #
# Copyright (c) 2005 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This file is part of Picalo.                                                     #
#                                                                                  #
# Picalo is free software; you can redistribute it and/or modify                   #
# it under the terms of the GNU General Public License as published by             #
# the Free Software Foundation; either version 2 of the License, or                # 
# (at your option) any later version.                                              #
#                                                                                  #
# Picalo is distributed in the hope that it will be useful,                        #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                   #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    #
# GNU General Public License for more details.                                     #
#                                                                                  #
# You should have received a copy of the GNU General Public License                #
# along with Foobar; if not, write to the Free Software                            #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA        #
#                                                                                  #
####################################################################################
#
# 12 Sep 2005 First version
#
####################################################################################


import wx.py, sys, StringIO, time, os, traceback
from code import InteractiveInterpreter
from NotebookComponent import NotebookComponent
import Editor, Utils, Preferences, Version
from Languages import lang
from picalo.base.Global import show_progress, clear_progress


#################################
###   PyShell extension
class Shell(wx.py.shell.Shell, NotebookComponent):
  '''Customized version of PyShell'''
  def __init__(self, parent, mainframe):
    self.mainframe = mainframe
    self.redirectErrorsToGUI = False
    wx.py.shell.Shell.__init__(self, parent, InterpClass=Interpreter)
    NotebookComponent.__init__(self, parent, mainframe)
  
    # add the initial commands
    for cmd in Editor.INITIAL_COMMANDS:
      self.push(cmd)
    
    # update my styles
    Editor.updateStyles(self)
    

    
  def onCut(self, event=None):
    self.Cut()
  def onCopy(self, event=None):
    self.Copy()
  def onPaste(self, event=None):
    self.Paste()
  def onSelectAll(self, event=None):
    self.SelectAll()


  def setRedirectErrors(self, redirectErrors):
    '''Redirects shell errors to the GUI'''
    self.interp.setRedirectErrors(redirectErrors)
    
    
  def updatePreferences(self):
    '''Signals that the preferences have been updated and this window should update itself'''
    Editor.updateStyles(self)
    
    
  def showIntro(self, text=''):
    '''Overrides the regular shell introductory text to the Picalo intro text'''
    text = os.linesep.join([
      lang('Welcome to Picalo') + ' ' + Version.VERSION + lang('.  This shell allows you to interact'),
      lang('directly with Picalo, test commands, and learn Picalo.'),
      lang('Watch this window as you select options from the Picalo'),
      lang('menu above.')
    ])
    self.write(text)


  def addHistory(self, command):
    '''Override the add history command so we can keep a log of commands that are entered'''
    # do the default behavior
    wx.py.shell.Shell.addHistory(self, command)
    
    # update the command log window
    if command:
      now = time.strftime("%d %b %Y %H:%M:%S", time.localtime())
      self.mainframe.commandlog.writeln("%s >>> %s" % (now, command))
          
        
  def varExists(self, name):
    '''Returns whether the given name exists in the interpreter's local dictionary'''
    return self.interp.locals.has_key(name)
    
  
  def getVariable(self, name, default=None):
    '''Returns the given variable (given as a string) from the interpreter's local dictionary'''
    parts = name.split('.')
    if len(parts) == 1:
      if self.varExists(name):
        return self.interp.locals[name]
      return default
    else:
      var = self.getVariable(parts[0])
      try:
        for part in parts[1:]:
          var = getattr(var, part)
        return var
      except AttributeError:
        return default
    
    
  def getLocals(self):
    '''Returns the locals dictionary from the interpreter.  DO NOT modify its values directly.
       I would return a copy but it would be inefficient to copy when large tables are in the locals dict.
    '''
    return self.interp.locals
    

  def push(self, command, silent = False):
      """Send command to the interpreter for execution.
         This is a near exact copy of wx.py.shell.push.  I modified the wx.BusyCursor
         lines beacuse they don't free the cursor for some reason.  If new versions of wx
         fix this problem, then this method can be taken out.
      """
      if not silent:
          self.write(os.linesep)
      wx.BeginBusyCursor()
      try:
        self.waiting = True
        self.more = self.interp.push(command)
        self.waiting = False
      finally:
        wx.EndBusyCursor()
        clear_progress(force=True)
      if not self.more:
          self.addHistory(command.rstrip())
      if not silent:
          self.prompt()


  def runScript(self, code, syspath_dirs=None):
    '''Runs a script in the shell.
    
       @param code          The actual code object (not the filename).
       @param syspath_dirs  A list of directories to add to sys.path during the run.
    '''
    # save sys.stdout
    oldouts = sys.stdout, sys.stderr
    try:
      # redirect standard streams
      sys.stdout = self.mainframe.output
      sys.stderr = self.mainframe.output

      # save the current sys.path, then add any directories to it
      syspath = sys.path
      if syspath_dirs:
        sys.path = syspath_dirs + sys.path
      
      # update the ui
      self.mainframe.output.setTitle(lang('Script Output (running)'))
      self.mainframe.output.clear()
      self.mainframe.output.select()
  
      # record the start time
      scriptstart = time.time()
      self.mainframe.output.writeln(lang('===   SCRIPT STARTED AT %s  ===') % time.strftime(lang("%d %b %Y %H:%M:%S"), time.localtime(scriptstart)))
  
      # run the script
      self.interp.rundirect(code)
      
      # record the stop time
      finish = time.time()
      self.mainframe.output.writeln(lang('===   SCRIPT ENDED AT %s (Running time %0.2f minutes)  ===') % (time.strftime(lang("%d %b %Y %H:%M:%S"), time.localtime()), (finish - scriptstart) / 60.0))
  
    finally:
      # set the system path back to what it was before the script
      sys.path = syspath
      # set the title back to normal
      self.mainframe.output.setTitle('Script Output (finished)')
      # clear any progress windows that were left open
      clear_progress(force=True)
      # redirect the standard streams back
      sys.stdout, sys.stderr = oldouts
    


  def OnChar(self, event):
    '''Overrides the OnChar to circumvent autocompletion if it's been disabled'''
    key = event.KeyCode
    if key in self.autoCompleteKeys and not self.automethodlist:
      event.Skip()
      return
    elif key == ord('(') and not self.automethodhelp:
      event.Skip()
      return
    # call the super
    wx.py.shell.Shell.OnChar(self, event)
  
##############################
###   Customized interpreter

class Interpreter(wx.py.interpreter.Interpreter):
  def __init__(self, locals=None, rawin=None, stdin=sys.stdin, stdout=sys.stdout, stderr=sys.stderr):
    self.shell = None
    wx.py.interpreter.Interpreter.__init__(self, locals, rawin, stdin, stdout, stderr)
    self.redirectErrors = False
    self.exc_info = None
    

  def rundirect(self, code):
    '''Runs code directly, without going to the shell'''
    InteractiveInterpreter.runcode(self, code)
    
    
  def setRedirectErrors(self, redirect):
    '''Sets whether to redirect errors or not'''
    self.redirectErrors = redirect


  def showsyntaxerror(self, filename=None):
    '''Shows an exception to the shell.  We override this from code.InteractiveInterpreter
       so we can show dialog boxes for errors that occur based on GUI events (see mainframe.execute)
       but still show the traceback for things typed into the shell.
    '''
    if self.redirectErrors:
      self.exc_info = sys.exc_info()
    else:
      wx.py.interpreter.Interpreter.showsyntaxerror(self, filename)

  
  def showtraceback(self):
    '''Shows an exception to the shell.  We override this from code.InteractiveInterpreter
       so we can show dialog boxes for errors that occur based on GUI events (see mainframe.execute)
       but still show the traceback for things typed into the shell.
    '''
    if self.redirectErrors:
      self.exc_info = sys.exc_info()
    else:
      wx.py.interpreter.Interpreter.showtraceback(self)
    
    
