#!/usr/bin/python

####################################################################################
#                                                                                  #
# Copyright (c) 2005 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This file is part of Picalo.                                                     #
#                                                                                  #
# Picalo is free software; you can redistribute it and/or modify                   #
# it under the terms of the GNU General Public License as published by             #
# the Free Software Foundation; either version 2 of the License, or                # 
# (at your option) any later version.                                              #
#                                                                                  #
# Picalo is distributed in the hope that it will be useful,                        #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                   #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    #
# GNU General Public License for more details.                                     #
#                                                                                  #
# You should have received a copy of the GNU General Public License                #
# along with Foobar; if not, write to the Free Software                            #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA        #
#                                                                                  #
####################################################################################
#
# 14 Jul 2006 First version
#
####################################################################################

from picalo import Table, TableList, TableArray, load
from picalo import Database
from Languages import lang
import os, os.path, gzip, pickle, time
import wx

PICKLE_PROTOCOL = 2

class Project:
  '''A Picalo project file, which contains the state of a given project'''  
  def __init__(self, mainframe, filename=None):
    self.filename = filename
    self.mainframe = mainframe
    if filename:
      self.name = os.path.splitext(os.path.split(filename)[1])[0]
      self.load(mainframe)
    else:
      self.name = 'Untitled Project'
      self.tables = {}
      self.databases = {}
      self.queries = {}
      self.scripts = {}
    
  
  def update(self, shell):
    '''Updates the list of tables and databases in this project.
       This is called automatically when commands are run from the
       program, a script, or the shell.
       Returns True or False (whether changes were found)
    '''
    found_changes = False
    
    # add any new ones to the project
    for key, val in shell.interp.locals.items():
      if isinstance(val, Table) and isinstance(val._datasource, Database.DatabaseDataSource):
        self.queries[key] = val
        found_changes = True
      
      elif isinstance(val, (Table, TableList, TableArray)) and not key in self.tables:
        self.tables[key] = val
        found_changes = True
        
      elif isinstance(val, Database._Connection) and not key in self.databases:
        self.databases[key] = val
        found_changes = True

    # remove any deleted ones from the project
    for objlist in [ self.tables, self.databases ]:
      for key in objlist.keys():
        if not key in shell.interp.locals:
          del objlist[key]
          found_changes = True

    # return whether we found changes          
    return found_changes
    
    
  def load(self, mainframe):
    '''Loads this project from the current filename'''
    assert os.path.exists(self.filename), 'The given file does not exist.'
    shell = mainframe.shell
    
    # open the file and gunzip it on the fly
    fp = open(self.filename, 'rb')
    gin = gzip.GzipFile(fileobj=fp, mode='rb')

    # get the version number and timestamp (we don't do anything with it right now)
    version = pickle.load(gin)
    timestamp = pickle.load(gin)
    
    # get the dictionary of items
    items = pickle.load(gin)
    self.tables = {}
    self.databases = {}
    self.queries = {}
    self.scripts = {}
    
    # load the tables
    for key, filename in items['tables'].items():
      shell.interp.locals[key] = load(filename)
    
    # load the databases
    for key, dbinfo in items['databases'].items():
      try:
        connect_func, connect_args = dbinfo
        func = getattr(Database, connect_func)
        shell.interp.locals[key] = func(**connect_args)
      except Exception, e:
        wx.MessageBox(str(e), lang('Database Could Not Be Loaded: %s' % (key),))
    
    # load the queries
    for key, queryinfo in items['queries'].items():
      try:
        connectionname, sql, cols = queryinfo
        conn = shell.interp.locals[connectionname]
        shell.interp.locals[key] = conn._delayed_table(sql, cols)
      except Exception, e:
        wx.MessageBox(str(e), lang('Query could not be loaded: %s' + key))
    
    # load the scripts
    # TO DO
    
    # load the command log history
    self.mainframe.commandlog.clear()
    if items.has_key('commandlog'):
      self.mainframe.commandlog.SetText(items['commandlog'])
    
    # close up shop
    gin.close()
    fp.close()
    
        
    
    
  def save(self, mainframe):
    '''Saves this project to the current filename'''
    assert self.filename, 'Programming error: the project did not have a filename'
    
    # create the dictionary of items
    items = {}
    
    # add the tables to the items
    items['tables'] = {}
    shown_save_message = False
    for key, table in self.tables.items():
      # give the user a chance to save if necessary
      filename = table.filename
      if not filename:
        if not shown_save_message:
          wx.MessageBox(lang('Some of your tables have not been saved before.  Please save these tables using the presented save dialog(s).'), lang('Save Project'))
          shown_save_message = True
        while 1:
          dlg = wx.FileDialog(mainframe, message=lang('Save Table: ') + key + '...', defaultDir=os.getcwd(), defaultFile=key, style=wx.SAVE | wx.CHANGE_DIR, wildcard='Picalo Tables|*.pco')
          if dlg.ShowModal() == wx.ID_OK:
            filename = dlg.GetPath()
            if os.path.splitext(filename)[1] != '.pco':
              filename += '.pco'
            if os.path.exists(filename):
              if wx.MessageBox(lang('A file with this name already exists.  Overwrite it?'), lang('Save Table'), style=wx.YES_NO) != wx.YES:
                continue
            break
          else:
            wx.MessageBox(lang('Table "') + key + lang('" was skipped and will not be saved with the project.'), lang('Save Tables'))
            break
      if filename:
        table.save(filename)
        items['tables'][key] = table.filename

    # add the databases to the items dict  
    items['databases'] = {}
    for name, db in self.databases.items():
      items['databases'][name] = (db._connect_func, db._connect_args)

    # add the scripts to the items dict      
    items['queries'] = self.queries
    for name, table in self.queries.items():
      for connectioname, db in self.databases.items():
        if db == table._datasource._connection:
          cols = [ col._get_columnloader() for col in table.get_columns() ]
          items['queries'][name] = (connectioname, table._datasource._sql, cols)
          break
      else:
        wx.MessageBox(lang("The query named %s was not saved with the project because it's connection object no longer exists." % (name, )), lang('Save Project'))
        
    # add the scripts to the items dict      
    items['scripts'] = self.scripts
    
    # add the command log history
    items['commandlog'] = self.mainframe.commandlog.GetText()

    # open the file and gzip it on the fly
    fp = open(self.filename, 'wb')
    gout = gzip.GzipFile(fileobj=fp, mode='wb')

    # write the version number and timestamp
    pickle.dump(1, gout, PICKLE_PROTOCOL)
    pickle.dump(time.time(), gout, PICKLE_PROTOCOL)
    
    # write the dictionary of items
    pickle.dump(items, gout, PICKLE_PROTOCOL)
    
    # close up shop
    gout.close()
    fp.close()
    
    # update the user interface    
    self.name = os.path.splitext(os.path.split(self.filename)[1])[0]
    
