#!/usr/bin/python
'''
The main Picalo application class.
'''

####################################################################################
#                                                                                  #
# Copyright (c) 2005 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This file is part of Picalo.                                                     #
#                                                                                  #
# Picalo is free software; you can redistribute it and/or modify                   #
# it under the terms of the GNU General Public License as published by             #
# the Free Software Foundation; either version 2 of the License, or                # 
# (at your option) any later version.                                              #
#                                                                                  #
# Picalo is distributed in the hope that it will be useful,                        #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                   #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    #
# GNU General Public License for more details.                                     #
#                                                                                  #
# You should have received a copy of the GNU General Public License                #
# along with Foobar; if not, write to the Free Software                            #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA        #
#                                                                                  #
####################################################################################
#
# 12 Sep 2005 First version
#
####################################################################################


import wx, MainFrame, sys, Utils, os.path, Preferences, Languages, Dialogs, Version
from picalo.base import Global
from Languages import lang
import os.path


class PicaloApp(wx.App):
  '''Main Picalo application class'''
  
  def __init__(self, options):
    '''Constructor.  Options are the command line options as returned by the optparse module.'''
    self.options = options
    wx.App.__init__(self, redirect=False)  # we'll redirect ourselves
    

  def OnInit(self):
    '''Sets everything up'''
    # initialize the preferences
    Preferences.init()
    
    # load the dialog code - must be done here in OnInit or we get an error message to the console
    Dialogs.init_xrc()

    # set the language
    Languages.set_language(Preferences.get('language', 'English'))

    # show the splash screen (allow bypass for debugging where we start the app over and over and over)
    if self.options.splash and Preferences.get('showsplash', True):
      splashimg = Utils.getBitmap('splash.png')
      splash = wx.SplashScreen(splashimg, wx.SPLASH_CENTRE_ON_SCREEN | wx.SPLASH_TIMEOUT, 5000, None)
      splash.Show()

    # set up the mainframe
    self.SetAppName('Picalo')
    self.mainframe = MainFrame.MainFrame(self)

    # Record a pointer to mainframe in PicaloHelpers.  Some functions, like Table.view(), need it.
    Global.mainframe = self.mainframe

    # show it!
    self.mainframe.Show(True)
    self.SetTopWindow(self.mainframe)

    # windows won't let me do GetSize() until the window is shown    
    self.mainframe.splitter.SetSashPosition(200)  # default
    size = Preferences.get('mainframesize')
    if size != None and size[0] > 50 and size[1] > 50:
      self.mainframe.SetSize(Preferences.get('mainframesize'))
      if Preferences.get('mainframeposition'): self.mainframe.Move(Preferences.get('mainframeposition'))
      if Preferences.get('mainframerightpanel'): self.mainframe.rightPanel.SetSashPosition(Preferences.get('mainframerightpanel'))
      if Preferences.get('mainframesplitter'): self.mainframe.splitter.SetSashPosition(Preferences.get('mainframesplitter'))

    else:
      self.mainframe.Maximize(True)
      self.mainframe.rightPanel.SetSashPosition(self.mainframe.rightPanel.GetSize()[1]-250)  # position a few pixels from bottom
    
    # windows doesn't lay out the disk browser right, so adjust the slider to force and layout (calling Layout doesn't work either)
    self.mainframe.splitter.SetSashPosition(self.mainframe.splitter.GetSashPosition() + 1)
    self.mainframe.splitter.SetSashPosition(self.mainframe.splitter.GetSashPosition() - 1)
      
    # if we had a filename to run, run it!
    if self.options.script:
      if not os.path.exists(self.options.script):
        print lang('The given script,') + ' ' + self.options.script + lang(', is not a valid file.  Exiting.')
        self.mainframe.Close()
      else:
        try:
          f = open(self.options.script)
          text = f.read().replace('\r\n', '\n').replace('\r', '\n') + '\n'  # required by compile
          f.close()
          code = compile(text, self.options.script, 'exec')      
          self.mainframe.shell.runScript(code)
        except:
          t, v, tb = sys.exc_info()
          if self.mainframe:
            self.mainframe.errorhandler(t, v, tb)
            self.mainframe.Close()  # close this app

    # check for updates
    if Preferences.get('checkforupdates', True):
      wx.CallAfter(self.mainframe.menuHelpCheckForUpdates, auto=True)
      
    # show the changelog if the user is running this version for the first time
    if Preferences.get('lastrunversion') != Version.VERSION:
      Preferences.put('lastrunversion', Version.VERSION)
      self.mainframe.openFilename(Utils.getResourcePath('CHANGELOG.TXT'))

    # show the startup hints
    if Preferences.get('showhintsatstartup') != False:
      Dialogs.Hints(self.mainframe)()
      
    # return true to tell wx we started successfully
    return True
    
    
