#!/usr/bin/python

'''
Supports multiple languages by replacing text with text from
the given language.  Instead of using keywords to map to text
in languages (like most programs do), I simply use the English
version throughout the code.  This allows several things:

 1. It makes the code much more readable (assuming you speak English :)
 2. It enables autodiscovery of all text that needs translating. See
    the update_language_files.py file.
 3. It provides a default text if the language hasn't translated
    a given term yet.

The update_language_files.py script can be run from the command line
to discover and autopopulate all .language files.  To make this work,
ALL text in the program should call Languages.lang().  Look around the
source code and you'll see lang('my text') all over the place.  That's 
all that is needed to get the text translated.

In addition, dialogs that are created in the XRCed graphical editor
(and that use the Dialogs.py infrastructure) have all text, labels,
and titles translated automatically.  Simply put the English text
in XRCed and Picalo will replace it for you.
   
If a translation for the given text is not found, the text is returned.  This 
provides a translation where possible with the English text being the backup
if a translation has not been made. 

To add a language, simply create an empty .language file in the 
picalo/resources directory.  Name the language file with the name
you want to show up in Picalo.  The file will be empty at this point.
Run update_language_files.py to automatically populate the new language
file with all the text that needs translating.  Finally, go through
the .language file with a text editor and add translation as described below.

Translation files have the following format:
  - English text to be tranlated is on one or more lines
  - Blank line(s)
  - Translated text is on the next line
  - 2 (or more) Blank lines
  - (repeat as needed for additional translations)
     
'''

import Utils, re, os.path, codecs

#################################################################
###   Multiple language support

# the current LanguageFile object
current_language = None

class LanguageFile(dict):
  """Representation of a language file."""
  def __init__(self, language):
    langfile = Utils.getResourcePath(language + '.language')
    if os.path.exists(langfile):
      f = codecs.open(langfile, 'r', 'ISO-8859-1')
      alltext = f.read().replace('\n\r', '\n').replace('\r\n', '\n').replace('\r', '\n')
      for pair in [ pair.strip() for pair in alltext.split('\n\n\n') ]:
        parts = [ part.strip() for part in pair.split('\n\n') ]
        if len(parts) == 1 and parts[0]:
          self[parts[0]] = ''
        elif len(parts) >= 2 and parts[0] and parts[1]:
          self[parts[0]] = parts[1]
      f.close()
    
    
  def __getitem__(self, text):
    '''Returns the translation for the given text, or the text itself if it isn't found in the translation file'''
    ret = ''
    if self.has_key(text):
      ret = dict.__getitem__(self, text)
    return ret or text
    
    
  def __call__(self, text):
    '''Returns the translation for the given text, or the text itself if it isn't found in the translation file'''
    return self[text]
    
    
  def has_explicit_key(self, text):
    '''Returns true only if the language file has a translation for the given text.
       The normal dict method, has_key, always returns true because it returns the text
       if a transation is not found.  Use this method to determine if a translation
       truely exists'''
    return self.has_key(text) and dict.__getitem__(self, text) not in (None, '')


########################################
###   Module-level functions

def set_language(language):
  '''Sets the language of the program to the given language'''
  global current_language
  current_language = LanguageFile(language)


def lang(text):
  '''If a language is set, returns the given text in the given language;
     otherwise, returns the text itself'''
  if current_language:
    return current_language[text]
  return text
  
  
def has_key(text):
  '''Returns whether the text has a translation in the current language'''
  if current_language:
    return current_language.has_key(text) and dict.__getitem__(current_language, text)
  return False
  