#!/usr/bin/python

####################################################################################
#                                                                                  #
# Copyright (c) 2005 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This file is part of Picalo.                                                     #
#                                                                                  #
# Picalo is free software; you can redistribute it and/or modify                   #
# it under the terms of the GNU General Public License as published by             #
# the Free Software Foundation; either version 2 of the License, or                # 
# (at your option) any later version.                                              #
#                                                                                  #
# Picalo is distributed in the hope that it will be useful,                        #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                   #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    #
# GNU General Public License for more details.                                     #
#                                                                                  #
# You should have received a copy of the GNU General Public License                #
# along with Foobar; if not, write to the Free Software                            #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA        #
#                                                                                  #
####################################################################################

import wx.py, wx.stc, os, os.path, StringIO, code, time, sys
from NotebookComponent import NotebookComponent
from wx.py import introspect
import Utils, Preferences, Dialogs, MainFrame
from Languages import lang
from picalo import python_modules


# set up the static locals used in the class
interpreter = code.InteractiveInterpreter()
interpreter.runsource('from picalo import *')

# script counter (global)
script_counter = 0

# things to import into scripts
INITIAL_COMMANDS = [ 
  '# import the PIcalo libraries',
  'from picalo import *',
  '',
  '# import commonly-needed built-in libraries',
  'import ' + ', '.join(python_modules),
]

# fonts - dialog choice name, windows name, Unix name, Mac name
FONTS = (
  ( lang('Monospaced (Courier)'),         lang('Courier New'),     lang('Courier'),   lang('Monaco')   ),
  ( lang('Sans Serif (Arial/Helvetica)'), lang('Arial)'),           lang('Helvetica'), lang('Helvetica')),
  ( lang('Serif (Times Roman)'),          lang('Times New Roman'), lang('Times'),     lang('Times')    ),
)
if 'wxMSW' in wx.PlatformInfo:
  DEFAULT_FONT_SIZE = 10
  FONT_INDEX = 1
elif 'wxMac' in wx.PlatformInfo:
  DEFAULT_FONT_SIZE = 10
  FONT_INDEX = 3
else: # default to Linux
  DEFAULT_FONT_SIZE = 10
  FONT_INDEX = 2

# this method is separated from the class so we can call it on
# the shell as well
def updateStyles(editor):
  '''Sets the styles of this editor from the Preferences'''
  editor.SetUseAntiAliasing(Preferences.get('useantialiasedfonts', True))
  editor.SetUseTabs(not Preferences.get('usespacesfortabs', True))
  editor.SetIndent(Preferences.get('tabwidth', 4))
  editor.automethodlist = Preferences.get('automethodlist', True)
  editor.automethodhelp = Preferences.get('automethodhelp', True)
  faces = { 'font' : FONTS[Preferences.get('editorfontname', 0)][FONT_INDEX],
            'size' : Preferences.get('editorfontsize', DEFAULT_FONT_SIZE),
            'size2': Preferences.get('editorfontsize', DEFAULT_FONT_SIZE) - 1,
           }
  # Global default styles for all languages
  editor.StyleSetSpec(wx.stc.STC_STYLE_DEFAULT,     "face:%(font)s,size:%(size)d" % faces)
  editor.StyleClearAll() # sets all styles to the default style, now we make color changes
  editor.StyleSetSpec(wx.stc.STC_STYLE_LINENUMBER,  "back:#C0C0C0,size:%(size2)d" % faces)
  editor.StyleSetSpec(wx.stc.STC_STYLE_CONTROLCHAR, "")
  editor.StyleSetSpec(wx.stc.STC_STYLE_INDENTGUIDE, "")
  editor.StyleSetSpec(wx.stc.STC_STYLE_BRACELIGHT,  "fore:#FFFFFF,back:#0000FF")
  editor.StyleSetSpec(wx.stc.STC_STYLE_BRACEBAD,    "fore:#000000,back:#FF0000")
  # Python styles
  editor.StyleSetSpec(wx.stc.STC_P_DEFAULT,         "")              # white space
  editor.StyleSetSpec(wx.stc.STC_P_COMMENTLINE,     "fore:#555555")  # comment
  editor.StyleSetSpec(wx.stc.STC_P_COMMENTBLOCK,    "fore:#555555")  # comment-blocks
  editor.StyleSetSpec(wx.stc.STC_P_NUMBER,          "fore:#007F7F")  # number
  editor.StyleSetSpec(wx.stc.STC_P_STRING,          "fore:#FF3399")  # string
  editor.StyleSetSpec(wx.stc.STC_P_CHARACTER,       "fore:#FF3399")  # single quoted string
  editor.StyleSetSpec(wx.stc.STC_P_TRIPLE,          "fore:#FF3399")  # triple quotes
  editor.StyleSetSpec(wx.stc.STC_P_TRIPLEDOUBLE,    "fore:#FF3399")  # triple double quotes
  editor.StyleSetSpec(wx.stc.STC_P_WORD,            "fore:#0000CC")  # keyword
  editor.StyleSetSpec(wx.stc.STC_P_CLASSNAME,       "fore:#0000CC")  # class name definition
  editor.StyleSetSpec(wx.stc.STC_P_DEFNAME,         "fore:#0000CC")  # function or method name
  editor.StyleSetSpec(wx.stc.STC_P_OPERATOR,        "")              # operators
  editor.StyleSetSpec(wx.stc.STC_P_IDENTIFIER,      "")  # identifiers
  editor.StyleSetSpec(wx.stc.STC_P_STRINGEOL,       "back:#E0C0E0,eol")  # end of line where string is not closed   


#################################
###   PyEditor extension

class Editor(wx.py.editwindow.EditWindow, NotebookComponent):
  '''Customized version of py's Editor'''
  def __init__(self, parent, mainframe, filename=None):
    global script_counter
    wx.py.editwindow.EditWindow.__init__(self, parent)
    NotebookComponent.__init__(self, parent, mainframe)
    self.SetUseTabs(False) # most people don't understand tab/space difference (and python is picky), so default to spaces
    self.SetEOLMode(wx.stc.STC_EOL_LF)  # always use Unix line endings, even on Windows (most Windows editors support \n anyway)
    self.filename = filename
    self.filetimestamp = None
    self.simplefilename = ''
    self.finddlg = None
    self.lastcurpos = None
    self.modified = False
    if self.filename:
      path, self.simplefilename = os.path.split(self.filename)
      self.LoadFile(self.filename)
      self.title = self.simplefilename
      self.filetimestamp = os.stat(self.filename).st_mtime
    else:
      self.SetText('\n'.join(INITIAL_COMMANDS) + '\n\n')
      self.SetCurrentPos(self.GetLength())
      self.SetSelection(self.GetCurrentPos(),self.GetCurrentPos())
      script_counter += 1
      self.title = 'Untitled %s' % script_counter
      
    # set the styles of this editor
    updateStyles(self)
    self.setDisplayLineNumbers(True)
    
    # events
    self.Bind(wx.stc.EVT_STC_CHANGE, self.onChange)
    self.Bind(wx.EVT_CHAR, self.onChar)     # for all chars
    self.Bind(wx.EVT_KEY_UP, self.onKeyUp)  # for enter key

    # menu and toolbar
    offset = 9
    self.menuitems = [
      Utils.MenuItem(lang('&File') + '/' + str(offset+0) + '|' + 'SEPARATOR'),
      Utils.MenuItem(lang('&File') + '/' + str(offset+1) + '|' + lang('Save') + '\t' + lang('Ctrl+S'), lang('Save the current script'), self.menuFileSave, id=wx.ID_SAVE),
      Utils.MenuItem(lang('&File') + '/' + str(offset+2) + '|' + lang('Save As...'), lang('Save the current script'), self.menuFileSaveAs, id=wx.ID_SAVEAS),
      Utils.MenuItem(lang('&File') + '/' + str(offset+3) + '|' + 'SEPARATOR'),
      Utils.MenuItem(lang('&File') + '/' + str(offset+4) + '|' + lang('Check Syntax'), lang('Check the syntax of the current script'), self.menuScriptCheckSyntax),
      Utils.MenuItem(lang('&File') + '/' + str(offset+5) + '|' + 'SEPARATOR'),
      Utils.MenuItem(lang('&File') + '/' + str(offset+6) + '|' + lang('Run Script') + '\t' + lang('Ctrl+R'), lang('Run the current script in the shell'), self.menuScriptRun),
      Utils.MenuItem(lang('&File') + '/' + str(offset+7) + '|' + lang('Run Script In New Picalo'), lang('Run the current script in a new Picalo instance'), self.menuScriptRunSeparate),
    ]
    self.tabmenuitems = [
      Utils.MenuItem(lang('Close'), lang('Close this tab'), mainframe.menuWindowClose),
      Utils.MenuItem(lang('Close All'), lang('Close all tabs'), mainframe.menuWindowCloseAll),
    ]
    self.tabmenu = Utils.MenuManager(mainframe, wx.Menu(), self.tabmenuitems)
    self.toolbar = Utils.create_toolbar(mainframe, [
      Utils.ToolbarItem('SEPARATOR'),
      Utils.ToolbarItem(lang('Save'), 'filesave.png', longhelp='Save this script', callback=self.menuFileSave),
      Utils.ToolbarItem(lang('Save As'), 'filesaveas.png', longhelp='Save this script with a new filename', callback=self.menuFileSaveAs),
      Utils.ToolbarItem('SEPARATOR'),
      Utils.ToolbarItem(lang('Undo'), 'undo.png', longhelp='Undo the last action', callback=self.menuEditUndo),
      Utils.ToolbarItem(lang('Redo'), 'redo.png', longhelp='Redo the last action', callback=self.menuEditRedo),
      Utils.ToolbarItem('SEPARATOR'),
      Utils.ToolbarItem(lang('Check Syntax'), 'apply.png', longhelp='Check the syntax of the current script', callback=self.menuScriptCheckSyntax),
      Utils.ToolbarItem('SEPARATOR'),
      Utils.ToolbarItem(lang('Run Script'), 'player_play.png', longhelp='Run the current script', callback=self.menuScriptRun),
      Utils.ToolbarItem(lang('Run Script In New Picalo'), 'player_fwd.png', longhelp='Run the current script', callback=self.menuScriptRunSeparate),
    ])
   
  
  def updatePreferences(self):
    '''Signals that the preferences have been updated and this window should update itself'''
    updateStyles(self)
    
    
  def get_filename(self):
    '''Returns the filanme of the table in this window'''
    return self.filename


  def onFocusSub(self, event=None):
    '''Handles an on focus event.  The MainFrame also calls this 
       when the application is focused.
       In particular, this one checks the script file (assuming it is saved)
       to make sure it hasn't changed.
    '''
    # check for data changes
    if self.filename and self.filetimestamp != None:
      if self.filetimestamp != os.stat(self.filename).st_mtime:
        self.filetimestamp = os.stat(self.filename).st_mtime
        if wx.MessageBox(lang('The content of this file has changed on disk.  You might be changing it in another editor.  Would you like to synchronize the text here with the file on disk?'), lang('File Changed'), style=wx.YES_NO) == wx.YES:
          self.LoadFile(self.filename)
          self.title = self.simplefilename
          self.filetimestamp = os.stat(self.filename).st_mtime
    # set the last cursor position to be None so the onIdle will update hte GUI
    self.lastcurpos = None
    event.Skip()
    

  def onIdle(self):
    '''Called from the mainframe when the system is idle.
       Subclasses can perform work in this method like updating
       their GUIs.  This method is called *a lot*, so keep it simple.
    '''
    curpos = self.GetCurrentPos()
    if curpos != self.lastcurpos:
      self.lastcurpos = curpos
      self.mainframe.setStatus(str(self.GetCurrentLine()+1) + ', ' + str(self.GetColumn(curpos)+1), -1)
      
  
  def is_changed(self):
    '''Returns whether the window has changed.'''
    return self.modified
    
  
  def onChange(self, event=None):
    '''Respond to the onchange method'''
    if not self.modified:
      self.modified = True
      self.setTitle(self.getTitle() + '*', False)

    
  def onKeyUp(self, event=None):
    '''Catches the enter key (it doesn't produce an onChar event)'''
    key = event.KeyCode
    if key == wx.WXK_RETURN or key == wx.WXK_NUMPAD_ENTER:  # the return/enter key
      currentline = self.GetCurrentLine()
      if currentline > 0:
        # get the indentation on the line before this one
        indent = self.GetLineIndentation(currentline - 1)
        if indent > 0:
          # set the new indent on this line
          self.Freeze()
          newpos = self.GetCurrentPos() + indent
          self.SetLineIndentation(currentline, indent)
          self.SetCurrentPos(newpos)
          self.SetSelectionStart(newpos)
          self.SetSelectionEnd(newpos)
          self.Thaw()
    event.Skip()
    
    
  def onChar(self, event=None):
    '''Called when a character is pressed'''
    key = event.KeyCode
    text, pos = self.GetCurLine()
    if (self.automethodlist and key == ord('.')) or (key == ord(' ') and event.m_controlDown and text[pos-1:pos] == '.'):
        if self.AutoCompActive(): 
            self.AutoCompCancel()
        self.ReplaceSelection('')
        if key == ord('.'):
          self.AddText(chr(key))
        text = text[:pos]
        if self.autoComplete: 
          self.autoCompleteShow(text)
    elif (self.automethodhelp and key == ord('(')) or (key == ord(' ') and event.m_controlDown and text[pos-1:pos] == '('):
      # The left paren activates a call tip and cancels an
      # active auto completion.
      if self.AutoCompActive(): 
          self.AutoCompCancel()
      self.ReplaceSelection('')
      if key == ord('('):
        self.AddText('(')
      text = text[:pos]
      self.autoCallTipShow(text)
    else:
      # Allow the normal event handling to take place.
      event.Skip()    
      
    
  def autoCompleteShow(self, command):
    """Display auto-completion popup list."""
    list = introspect.getAutoCompleteList(command, interpreter.locals, includeMagic=self.autoCompleteIncludeMagic, includeSingle=self.autoCompleteIncludeSingle, includeDouble=self.autoCompleteIncludeDouble)
    if list:
      options = ' '.join(list)
      offset = 0
      self.AutoCompShow(offset, options)  

  
  def autoCallTipShow(self, command):
    """Display argument spec and docstring in a popup window."""
    if self.CallTipActive():
      self.CallTipCancel()
    (name, argspec, tip) = introspect.getCallTip(command, interpreter.locals)
    if not self.autoCallTip:
      return
    if argspec:
      startpos = self.GetCurrentPos()
      self.AddText(argspec + ')')
      endpos = self.GetCurrentPos()
      self.SetSelection(endpos, startpos)
    if tip:
      curpos = self.GetCurrentPos()
      size = len(name)
      tippos = curpos - (size + 1)
      fallback = curpos - self.GetColumn(curpos)
      # In case there isn't enough room, only go back to the
      # fallback.
      tippos = max(tippos, fallback)
      self.CallTipShow(tippos, tip)
      self.CallTipSetHighlight(0, size)


  def menuFileSave(self, event=None):
    '''Saves the script'''
    if self.filename:
      self.SaveFile(self.filename)
      self.modified = False
      self.filetimestamp = os.stat(self.filename).st_mtime
      self.setTitle(self.simplefilename)

      # add this script to the recent files list
      recent = Preferences.get('recentscripts', [])
      f = self.filename
      while f in recent:
        recent.remove(f)
      recent.insert(0, f)
      Preferences.put('recentscripts', recent[:MainFrame.NUM_RECENT_FILES])  # save only eight recent files
      self.mainframe.updateMenu(force_refresh=True)

      return True
    else:
      return self.menuFileSaveAs(event)
    
    
  def menuFileSaveAs(self, event=None):
    '''Saves As'''
    # open the file dialog
    dlg = wx.FileDialog(self, message='Save Script As...', defaultDir=self.mainframe.projectdir, defaultFile="", style=wx.SAVE | wx.CHANGE_DIR | wx.OVERWRITE_PROMPT)
    if dlg.ShowModal() == wx.ID_OK:
      self.filename = dlg.GetPath()
      filepath, ext = os.path.splitext(self.filename)
      if ext.lower() != '.py':
        self.filename += '.py'
      self.SaveFile(self.filename.replace('\\', '/'))
      path, self.simplefilename = os.path.split(self.filename)
      self.setTitle(self.simplefilename)
      self.filetimestamp = os.stat(self.filename).st_mtime
      self.modified = False

      # add this script to the recent files list
      recent = Preferences.get('recentscripts', [])
      f = self.filename
      while f in recent:
        recent.remove(f)
      recent.insert(0, f)
      Preferences.put('recentscripts', recent[:MainFrame.NUM_RECENT_FILES])  # save only eight recent files
      self.mainframe.updateMenu(force_refresh=True)

      return True
    return False
    
      
  def menuEditUndo(self, event=None):
    self.Undo()
  def menuEditRedo(self, event=None):
    self.Redo()
  def onCut(self, event=None):
    self.Cut()
  def onCopy(self, event=None):
    self.Copy()
  def onPaste(self, event=None):
    self.Paste()
  def onSelectAll(self, event=None):
    self.SelectAll()
  
  
  def menuScriptCheckSyntax(self, event=None):
    '''Checks the syntax of the current script and returns the code object created from it'''
    if self.getCodeObject():
      wx.MessageBox(lang('Your script syntax is correct!'), lang('Syntax Check'))
    
    
  def menuScriptRunSeparate(self, event=None):
    '''Runs the current script'''
    code = self.getCodeObject()  # make sure it can compile
    if code: 
      if not self.filename:
        wx.MessageBox(lang('Please save this script to a file so it can be run.'), lang('Run Script'))
      if self.menuFileSave() and self.filename:  # saves if modified or saves as if needed
        os.popen2(sys.executable + ' ' + sys.argv[0] + ' --nosplash --script="' + self.filename + '"')
        
        
  def menuScriptRun(self, event=None):
    '''Runs the current script in the shell'''
    # save the script
    if not self.menuFileSave():
      return
    syspath = [ os.path.split(self.filename)[0] ]  # add the path of this script so we can import things
     
    # create a code object and run it in the main thread
    code = self.getCodeObject()
    if code:
      self.mainframe.shell.runScript(code, syspath)
      

  def getCodeObject(self):
    '''Retrieves the code object created from this script'''
    text = self.GetText().replace('\r\n', '\n').replace('\r', '\n') + '\n'  # required by compile
    try:
      return compile(text, '<script>', 'exec')
    except Exception, e:
      offset = e.offset or 1  # sometimes these are None
      lineno = e.lineno or 1
      self.GotoPos(self.PositionFromLine(lineno-1) + offset - 1)
      wx.MessageBox(lang('You have a syntax error on line') + ' ' + str(lineno) + ', ' + lang('column') + ' ' + str(offset) + '.', lang('Syntax Error'))
      return None      


  def findDialog(self):
    '''Shows the find dialog'''
    if not self.finddlg:
      self.finddlg = Dialogs.FindInScript(self.mainframe, self)
    self.finddlg()
    

    
  def findNextDialog(self):
    '''Shows the find dialog'''
    if not self.finddlg:
      return self.findDialog()
    else:
      self.finddlg.doFind()

