#!/usr/bin/python


import Utils, Preferences, Editor, Spreadsheet, TableProperties
import MainFrame, Languages
import wx, re, os, os.path, imp, sys, traceback, StringIO
import wx.wizard, wx.lib.masked
from wx import xrc
from picalo import *
import wx.py.editwindow
from Languages import lang
from picalo.base.Global import ensure_unique_colname, make_valid_variable, format_value_from_type
from picalo.base.Table import Table
from picalo.base.Number import number

# the dialog xrc file (gets set up in init_xrc below, called from PicaloApp.OnInit)
dialogxrc = None

VARIABLE_RE = re.compile('^[A-Za-z_][A-Za-z0-9_]*$')
  
###########################################################
###   Base object for dialogs, frames, wizards

def init_xrc():
  '''Initializes the xrc file, called from PicaloApp.OnInit (where it has to be called)'''
  global dialogxrc
  dialogxrc = xrc.XmlResource(Utils.getResourcePath('dialogs.xrc'))


class XRCObject:
  '''A simple mix in to provide convenience methods for classes that wrap XRC dialogs'''
  def __init__(self, mainframe, dialog_xmlid, objtype='wxDialog'):
    '''Constructor'''
    self.dialog_xmlid = dialog_xmlid
    self.objtype = objtype
    self.mainframe = mainframe
    self.dlg = None
  
  
  def createDialog(self):
    '''Creates the dialog'''
    # create the dialog
    self.dlg = dialogxrc.LoadObject(self.mainframe, self.dialog_xmlid, self.objtype)
    assert self.dlg != None, lang('Programming error -- could not instantiate dialog') + ' ' + str(self.dialog_xmlid)
    
    # replace any English text with corresponding text from the language file
    # this takes care of all language changes for all dialogs in the program.
    # Cool, huh?
    def localize(control):
      for name in [ 'Title', 'Label', 'Value' ]:
        if hasattr(control, 'Get' + name) and hasattr(control, 'Set' + name):
          currenttext = getattr(control, 'Get' + name)()
          if Languages.has_key(currenttext):
            getattr(control, 'Set' + name)(lang(currenttext))
      for child in control.GetChildren():
        localize(child)
    localize(self.dlg)
  
  
  def destroyDialog(self):
    '''Destroys the dialog'''
    if self.dlg:
      self.dlg.Destroy()
      self.dlg = None
  

  def getControl(self, xmlid):
    '''Retrieves the given control (within a dialog) by its xmlid'''
    control = self.dlg.FindWindowById(xrc.XRCID(xmlid))
    assert control != None, 'Programming error: a control with xml id ' + xmlid + ' was not found.'
    return control

  
  def fmt(self, st):
    '''Formats a string for insertion into the shell'''
    return st.replace('\\', '\\\\')
  
    
  def checkValidVariable(self, controltitle, varname):
    '''Checks that a variable name is valid'''
    # ensure the variable is of the correct type
    assert VARIABLE_RE.match(varname), 'Please enter a valid name for ' + controltitle + ' (letters and numbers only)'
    # check to see if the variable already exists
    existing = self.mainframe.getVariable(varname)
    if existing != None:
      question = lang('A variable with the name "') + varname + lang('" already exists.\n\nDo you want to overwrite it?')
      if isinstance(existing, Table):
        question = lang('A table with the name "') + varname + lang('" already exists.\n\nDo you want to overwrite it?')
      elif isinstance(existing, TableArray):
        question = lang('A table array with the name "') + varname + lang('" already exists.\n\nDo you want to overwrite it?')
      elif isinstance(existing, TableList):
        question = lang('A table list with the name "') + varname + lang('" already exists.\n\nDo you want to overwrite it?')
      if wx.MessageBox(question, lang('Overwrite?'), style=wx.YES_NO) != wx.YES:
        assert False, ''
        
    
  def initTables(self, tablechoice, tables=True, tablearrays=True, queries=True, relations=True):
    '''Initializes the control with the list of tables, selecting the currently-viewed table/
       Returns the sorted list of tables.'''
    tables = self.mainframe.getTables(tables=tables, tablearrays=tablearrays, queries=queries, relations=relations)
    tables.sort()
    assert len(tables) > 0, lang('Please create at least one table first.')
    tablechoice.Clear()
    tablechoice.AppendItems(tables)
    tablechoice.SetSelection(0)
    currenttable = self.mainframe.getCurrentPageName()
    for i, name in enumerate(tables):
      if currenttable == tables[i]:
        tablechoice.SetSelection(i)
        break
    return tables



  
############################################################
###   Main dialog superclass

class Dialog(XRCObject):
  '''Superclass for the dialogs of the program'''
  
  def __init__(self, mainframe, dialog_xmlid, objtype='wxDialog'):
    '''Constructor'''
    XRCObject.__init__(self, mainframe, dialog_xmlid, objtype=objtype)
    

  def close(self, okid=wx.OK):
    '''Closes the dialog with the given id (defaults to wx.OK)'''
    self.dlg.EndModal(okid)
    self.dlg.Destroy()
    self.dlg = None
    
    
  def hide(self, id=wx.OK):
    '''Hides the dialog (but doesn't destroy it) with the given id (defaults to wx.OK)'''
    self.dlg.EndModal(wx.OK)
    
    
  def __call__(self, event=None):
    '''Called when the menu option is selected'''
    return self.runModal()
    
    
  def runModal(self):
    '''Shows the dialog in modal mode'''
    # create the dialog
    if self.dlg == None:
      self.createDialog()

      # bind
      try:
        self.getControl('ok').Bind(wx.EVT_BUTTON, self.onOKButton)
      except AssertionError, e:  # in case ok buttond doesn't exist
        pass
      try:
        self.getControl('cancel').Bind(wx.EVT_BUTTON, self.onCancelButton)
      except AssertionError, e:  # in case cancel buttond doesn't exist
        pass

    # let the subclass do initialization
    try:
      self.init()
    except AssertionError, e:
      wx.MessageDialog(self.dlg, str(e), lang('Picalo'), wx.OK | wx.ICON_WARNING).ShowModal()
      self.dlg.Destroy()
      self.dlg = None
      return False

    # run the thing  
    self.dlg.Fit()  # sizes the dialog to the right size
    self.dlg.Center()
    if self.dlg.ShowModal() == wx.OK:
      return True
    else:
      return False
    

  def init(self):
    '''Initializes the dialog before ShowModal is called'''
    pass


  def onOKButton(self, event=None):
    '''Responds to an OK event.  Subclasses need to override this method to provide custom OK behavior.
       Subclasses can throw AssertionErrors to show messages to the user and to cancel the OK operation.
    '''
    try:
      self.ok()
      event.Skip()
    except AssertionError, e:
      if str(e):
        wx.MessageDialog(self.dlg, str(e), lang('Picalo'), wx.OK | wx.ICON_WARNING).ShowModal()
      return False
      

  def ok(self):
    '''Default OK method.  Subclasses should override this to provide behavior when the OK button is pressed'''
    self.close(wx.OK)

  
  def onCancelButton(self, event=None):
    '''Responds to a cancel event.'''
    self.cancel()
    event.Skip()
    
    
  def cancel(self):
    '''Default cancel method'''
    self.close(wx.CANCEL)

    


############################################################
###   Main Frame superclass

class Frame(Dialog):
  '''Superclass for the XRC-based frames of the program'''
  def __init__(self, mainframe, dialog_xmlid):
    Dialog.__init__(self, mainframe, dialog_xmlid, objtype='wxFrame')

  def __call__(self, event=None):
    '''Called when the menu option is selected'''
    self.runFrame()
    
    
  def runFrame(self):
    '''Shows the dialog in nonmodal mode'''
    # create the dialog
    if self.dlg == None:
      self.createDialog()

      # bind
      try:
        self.getControl('ok').Bind(wx.EVT_BUTTON, self.onOKButton)
      except AssertionError, e:  # in case ok buttond doesn't exist
        pass
      try:
        self.getControl('cancel').Bind(wx.EVT_BUTTON, self.onCancelButton)
      except AssertionError, e:  # in case cancel buttond doesn't exist
        pass

    # let the subclass do initialization
    try:
      self.init()
    except AssertionError, e:
      if str(e):
        wx.MessageDialog(self.dlg, str(e), lang('Picalo'), wx.OK | wx.ICON_WARNING).ShowModal()
      self.dlg.Destroy()
      self.dlg = None
      return

    # run the dialog
    self.dlg.Fit()  # sizes the dialog to the right size
    self.dlg.Center()
    result = self.dlg.Show()
      
    

  def close(self, id=wx.OK):
    '''Closes the dialog with the given id (defaults to wx.OK)'''
    self.dlg.Hide()
    self.dlg.Destroy()
    self.dlg = None
    


############################################################
###   Main wizard superclass

class Wizard(XRCObject):
  '''Superclass for the wizards of the program.  Follow these rules:
     * The first page of the wizard must be have the XRC ID of 'page1'.
     * Subsequent pages can be named anything you want.
     * When a Next is clicked on a page, the wizard automatically calls
       validate_pagexrcid(), if it exists.
  '''
  
  def __init__(self, mainframe, dialog_xmlid):
    '''Constructor'''
    XRCObject.__init__(self, mainframe, dialog_xmlid, objtype='wxWizard')


  def __call__(self, event=None):
    '''Called when the menu option is selected'''
    # create the dialog
    self.createDialog()
    
    # set up the events
    self.dlg.Bind(wx.wizard.EVT_WIZARD_PAGE_CHANGING, self.onPageChanging)
    
    # let the subclass do initialization
    try:
      self.init()

    except AssertionError, e:
      if str(e):
        wx.MessageDialog(self.dlg, str(e), lang('Picalo'), wx.OK | wx.ICON_WARNING).ShowModal()
      self.dlg.Destroy()
      self.dlg = None
      return

    # show the wizard   
    self.dlg.FitToPage(self.getControl('startpage'))  # sizes the wizard to the right size
    self.dlg.Center()
    self.dlg.RunWizard(self.getControl('startpage'))
      
    
    
  def init(self):
    '''Initializes the dialog before ShowModal is called'''
    pass
    
    
  def setPageOrder(self, pages):
    '''Reorders the pages of the wizard to the given list.  Use this method 
       to enable dynamic wizards.  The pages list should be a list of page XRC ID
       names (strings).'''
    for i in range(len(pages)-1):
      page1 = self.getControl(pages[i])
      page2 = self.getControl(pages[i+1])
      page1.SetNext(page2)
      page2.SetPrev(page1)
    
      
  def onPageChanging(self, event):
    '''Responds to a page change event'''
    if event.GetDirection():  # only validate when going forward
      page = event.GetPage()

      # validate the current page
      pagename = page.GetName()
      if hasattr(self, 'validate_' + pagename):
        try:
          getattr(self, 'validate_' + pagename)()
        except AssertionError, e:
          if str(e):
            wx.MessageBox(str(e), lang('Warning'))
          event.Veto()
          return
      
      # if we're on the last page, call the finisher
      if not page.GetNext():
        try:
          self.finish()
        except Exception, e:
          if str(e):
            wx.MessageBox(str(e), lang('Error'))
          event.Veto()
          return
    event.Skip()
    
          
  def finish(self):
    '''Runs the finish code for this wizard.  This is called when the finish button is pressed.
       Subclasses should override this method'''
    pass
  
  
############################################
###   Filter table dialog

class FilterTable(Dialog):
  def __init__(self, spreadsheet, mainframe):
    Dialog.__init__(self, mainframe, 'FilterTable')
    self.spreadsheet = spreadsheet
    self.expression = None
    

  def init(self):
    '''Initializes the dialog'''
    choice = self.getControl('columns')
    choice.AppendItems(self.spreadsheet.table.get_column_names())
    choice.SetSelection(0)
    for name in ( 'typewildcard', 'typeregex', 'matchentire', 'matchpartial', 'casespecific', 'caseignore', 'includematching', 'excludematching' ):
      self.getControl(name).Bind(wx.EVT_RADIOBUTTON, self.updateResult)
    for name in ( 'pattern', 'testtext' ):
      self.getControl(name).Bind(wx.EVT_TEXT, self.updateResult)


  def updateResult(self, event=None):
    '''Updates the results field'''
    # try to compile the expression
    try:
      fullmatch = self.getControl('matchentire').GetValue()
      ignorecase = self.getControl('caseignore').GetValue()
      pattern = self.getControl('pattern').GetValue()
      direction = self.getControl('includematching').GetValue()
      assert pattern, lang('Start by entering a pattern to match')
      testtext = self.getControl('testtext').GetValue()
      assert testtext, lang('To test your pattern, enter some match text')
      if self.getControl('typewildcard').GetValue():
        result = Simple.wildcard_match(pattern, testtext, ignorecase, fullmatch)
      else:
        if fullmatch and pattern[:1] != '^':
          pattern = '^' + pattern
        if fullmatch and pattern[-1:] != '$':
          pattern += '$'
        result = Simple.regex_match(pattern, testtext, ignorecase)
      if not direction:
        result = not result
      if result:
        self.getControl('testresult').SetValue(lang('Successful match for this pattern'))
      else:
        self.getControl('testresult').SetValue(lang('No match for this pattern'))
    except Exception, e:
      self.getControl('testresult').SetValue(lang('Pattern error: ') + str(e))


  def ok(self, event=None):
    '''Responds to the ok button'''
    try:
      fullmatch = self.getControl('matchentire').GetValue()
      ignorecase = self.getControl('caseignore').GetValue()
      pattern = self.getControl('pattern').GetValue()
      direction = self.getControl('excludematching').GetValue() and 'not ' or ''
      assert pattern, lang('Start by entering a pattern to match')
      colname = self.getControl('columns').GetStringSelection()
      if self.getControl('typewildcard').GetValue():
        Simple.wildcard_match(pattern, '', ignorecase, fullmatch)  # run to check for errors
        self.expression = direction + "Simple.wildcard_match('%s', %s, %s, %s)" % (pattern.replace("'", "\\'"), colname, ignorecase, fullmatch)
      else:
        if fullmatch and pattern[:1] != '^':
          pattern = '^' + pattern
        if fullmatch and pattern[-1:] != '$':
          pattern += '$'
        Simple.regex_match(pattern, '', ignorecase)  # run to check for errors
        self.expression = direction + "Simple.regex_match('%s', %s, %s)" % (pattern.replace("'", "\\'"), colname, ignorecase)
      self.close()
    except Exception, e:
      raise AssertionError(lang('Pattern error: ') + unicode(e))


    
 
#######################################################
###   Database connection wizard

    
class DatabaseConnection(Wizard):
  '''The Database Connection wizard'''
  def __init__(self, mainframe):
    Wizard.__init__(self, mainframe, 'DatabaseConnectionWizard')
    
  def init(self):  
    self.getControl('fileButton').Bind(wx.EVT_BUTTON, self.onFileButton)
    self.getControl('sqliteFileButton').Bind(wx.EVT_BUTTON, self.onSqliteFileButton)
    self.getControl('dbtype').Bind(wx.EVT_RADIOBOX, self.changeDatabaseType)
    self.changeDatabaseType()
    

  def changeDatabaseType(self, event=None):
    '''Called when user selects a different database'''
    dbtype = self.getControl('dbtype').GetSelection()
    if dbtype == 0 or dbtype == 5:   # ODBC or Oracle
      self.setPageOrder([ 'startpage', 'page_odbc_parameters', 'page_filename' ])
    elif dbtype == 1:  # SQLite
      self.setPageOrder([ 'startpage', 'page_sqlite_parameters', 'page_filename' ])
    else:    # MySQL or PostgreSQL
      self.setPageOrder([ 'startpage', 'page_database_parameters', 'page_filename' ])


  def onFileButton(self, event):
    '''Responds to the file browser button'''
    dlg = wx.FileDialog(self.dlg, message="Select Filename", defaultDir=self.mainframe.projectdir, defaultFile="", style=wx.SAVE)
    if dlg.ShowModal() != wx.ID_OK:
      return
    filename = dlg.GetPath()
    if os.path.splitext(filename)[1] != '.pcd':
      filename += '.pcd'
    self.getControl('filename').SetValue(filename)


  def onSqliteFileButton(self, event):
    '''Responds to the directory name browser button'''
    dlg = wx.FileDialog(self.dlg, message="Select filename to store database in", defaultDir=self.mainframe.projectdir, defaultFile="", style=wx.SAVE)
    if dlg.ShowModal() != wx.ID_OK:
      return
    filename = dlg.GetPath()
    if os.path.splitext(filename)[1] != '.sqlite3':
      filename += '.sqlite3'
    self.getControl('sqlitefilename').SetValue(filename)


  def validate_page_odbc_parameters(self):
    '''Validates the ODBC page'''
    pass
    
    
  def validate_page_database_parameters(self):
    '''Validates the database page'''
    assert self.getControl("database").GetValue(), lang('Please enter the database name.')


  def validate_page_sqlite_parameters(self):
    '''Validates the sqlite page'''
    assert self.getControl("sqlitefilename").GetValue(), lang('Please select a filename to save the database in.')


  def validate_page_filename(self):
    '''Validates the filename page'''
    varname = self.getControl('varname').GetValue()
    assert varname, lang('Please enter a name for the connection.')
    newvarname = make_valid_variable(varname)
    if varname != newvarname:
      self.getControl('varname').SetValue(newvarname)
      raise AssertionError, lang('The connection name has been changed to %s to comply with Picalo naming conventions.') % (newvarname, )


  def finish(self):
    '''Finishes the wizard'''
    dbtype = self.getControl('dbtype').GetSelection()
    varname = self.getControl('varname').GetValue()
    if dbtype == 0: # ODBC
      func = 'OdbcConnection'
      dbname = 'dsn_name'
      controlnames = ( 'odbcusername', 'odbcpassword' )
      otherargs = ( 'username', 'password' )
    elif dbtype == 1: # SQLite
      func = 'SqliteConnection'
      dbname = 'sqlitefilename'
      controlnames = ()
      otherargs = ()
    elif dbtype == 2: # MySQL
      func = 'MySQLConnection'
      dbname = 'database'
      controlnames = ( 'username', 'password', 'host', 'port' )
      otherargs = ('username', 'password', 'host', 'port')
    elif dbtype == 3: # PostgreSQL pygresql
      func = 'PyGreSQLConnection'
      dbname = 'database'
      controlnames = ( 'username', 'password', 'host', 'port' )
      otherargs = ('username', 'password', 'host', 'port')
    elif dbtype == 4: # PostgreSQL psycopg2
      func = 'PostgreSQLConnection'
      dbname = 'database'
      controlnames = ( 'username', 'password', 'host', 'port' )
      otherargs = ('username', 'password', 'host', 'port')
    elif dbtype == 5: # Oracle cx_Oracle
      func = 'OracleConnection'
      dbname = 'dsn_name'
      controlnames = ( 'odbcusername', 'odbcpassword' )
      otherargs = ( 'username', 'password' )
      
    args = []
    args.append('"%s"' % (self.getControl(dbname).GetValue().replace('"', '\\"'),))
    for term, controlname in zip(otherargs, controlnames):
      if self.getControl(controlname).GetValue():
        args.append('%s="%s"' % (term, self.getControl(controlname).GetValue().replace('"', '\\"')))
    cmd = []
    cmd.append('%s = Database.%s(%s)' % (varname, func, ', '.join(args)))
    if self.getControl('filename').GetValue():
      filename = self.getControl('filename').GetValue().replace('"', '\\"')
      if os.path.splitext(filename)[1] != '.pcd':
        filename += '.pcd'
      cmd.append('%s.save("%s")' % (varname, filename))
    self.mainframe.execute(cmd)

    
    
############################################
###   Upload Table Dialog

class DatabasePostTable(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'DatabasePostTable')
    
  
  def init(self):
    '''Initializes the dialog'''
    dbchoice = self.getControl('databasename')
    dbchoice.Clear()
    tablechoice = self.getControl('tablename')
    tablechoice.Clear()
    dbnames = self.mainframe.getDatabases()
    assert len(dbnames) > 0, lang('You must connect to a database before uploading a table.')
    tablenames = self.mainframe.getTables()
    assert len(tablenames) > 0, lang('You must load or create a table to upload it.')
    dbnames.sort()
    tablenames.sort()
    dbchoice.AppendItems(dbnames)
    dbchoice.SetSelection(0)
    tablechoice.AppendItems(tablenames)
    tablechoice.SetSelection(0)
    
  
  def ok(self, event=None):
    '''Responds to the ok button'''
    dbchoice = self.getControl('databasename')
    dbname = dbchoice.GetStringSelection()
    tablechoice = self.getControl('tablename')
    tablename = tablechoice.GetStringSelection()
    replacetable = self.getControl('replacetable').IsChecked()
    relationname = self.fmt(self.getControl('relationname').GetValue().strip())
    assert re.match('.+', relationname), lang('Please enter a name for the new database table.')
    if self.mainframe.execute('%s.post_table(%s, "%s", replace=%s)' % (dbname, tablename, relationname, replacetable)):
      self.close()
    
    
    
############################################
###   Sort Table Dialog

class SortTable(Dialog):
  def __init__(self, spreadsheet, mainframe):
    Dialog.__init__(self, mainframe, 'SortTable')
    self.spreadsheet = spreadsheet
    
  
  def init(self):
    '''Initializes the dialog'''
    cols = ['(none)'] + self.spreadsheet.table.get_column_names()
    for choicename in [ 'sort1', 'sort2', 'sort3' ]:
      choice = self.getControl(choicename)
      choice.AppendItems(cols)
      choice.SetSelection(0)
    
  
  def ok(self, event=None):
    '''Responds to the ok button'''
    sortcols = []
    for choicename in [ 'sort1', 'sort2', 'sort3' ]:
      choice = self.getControl(choicename)
      idx = choice.GetSelection() - 1  # first item is (none)
      if idx >= 0:
        sortcols.append('"' + self.spreadsheet.table.column(idx).name + '"')
    ascending = self.getControl('sortascending').GetValue()
    if len(sortcols) > 0:
      if self.mainframe.execute('Simple.sort(' + self.spreadsheet.name + ', ' + str(ascending) + ', ' + (', '.join(sortcols)) + ')'):
        self.close()
    
    
    
############################################
###   Find Unordered Dialog

class FindUnordered(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'FindUnordered')
    
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'), tablearrays=True)
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.refreshColumns)

    # tell the columns to refresh
    self.refreshColumns()
    
  
  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = self.getControl('columns')
    cols.Clear()
    cols.AppendItems(table.get_column_names())
    self.getControl('results').SetValue(tablename.replace('.', '_') + '_unordered')
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    resultstable = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable('Results Name', resultstable)
    if not resultstable:
      wx.MessageDialog(self.dlg, lang('Please enter a table for the results.'), lang(lang('Warning')), wx.OK | wx.ICON_WARNING).ShowModal()
      return False
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = [ '"' + table.column(i).name + '"' for i in self.getControl('columns').GetSelections() ]
    if len(cols) == 0:
      wx.MessageDialog(self.dlg, lang('Please select at least on column.'), lang(lang('Warning')), wx.OK | wx.ICON_WARNING).ShowModal()
      return False
    ascending = self.getControl('sortascending').GetValue()
    if self.mainframe.execute([
      resultstable + ' = Simple.get_unordered(' + tablename + ', ' + str(ascending) + ', ' + (', '.join(cols)) + ')',
      resultstable + '.view()',
    ]):
      self.close()
    
    
    
############################################
###   Find Duplicates Dialog

class FindDuplicates(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'FindDuplicates')
    
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'), tablearrays=True)
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.refreshColumns)

    # tell the columns to refresh
    self.refreshColumns()
    
  
  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = self.getControl('columns')
    cols.Clear()
    cols.AppendItems(table.get_column_names())
    self.getControl('results').SetValue(tablename.replace('.', '_') + '_duplicates')
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    resultstable = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable('Results Name', resultstable)
    if not resultstable:
      wx.MessageDialog(self.dlg, lang('Please enter a table for the results.'), lang(lang('Warning')), wx.OK | wx.ICON_WARNING).ShowModal()
      return False
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = [ '"' + table.column(i).name + '"' for i in self.getControl('columns').GetSelections() ]
    if len(cols) == 0:
      wx.MessageDialog(self.dlg, lang('Please select at least on column.'), lang('Warning'), wx.OK | wx.ICON_WARNING).ShowModal()
      return False
    if self.mainframe.execute([
      resultstable + ' = Simple.find_duplicates(' + tablename + ', ' + (', '.join(cols)) + ')',
      resultstable + '.view()',
    ]):
      self.close()
    
    
    
############################################
###   Find Gaps Dialog

class FindGaps(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'FindGaps') 
    
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'), tablearrays=True)
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.refreshColumns)

    # tell the columns to refresh
    self.refreshColumns()
    
  
  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = self.getControl('columns')
    cols.Clear()
    cols.AppendItems(table.get_column_names())
    self.getControl('results').SetValue(tablename.replace('.', '_') + '_gaps')


  def ok(self, event=None):
    '''Responds to the ok button'''
    resultstable = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable('Results Name', resultstable)
    if not resultstable:
      wx.MessageDialog(self.dlg, lang('Please enter a table for the results.'), lang('Warning'), wx.OK | wx.ICON_WARNING).ShowModal()
      return False
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = [ '"' + table.column(i).name + '"' for i in self.getControl('columns').GetSelections() ]
    if len(cols) == 0:
      wx.MessageDialog(self.dlg, lang('Please select at least on column.'), lang('Warning'), wx.OK | wx.ICON_WARNING).ShowModal()
      return False
    ascending = self.getControl('sortascending').GetValue()
    if self.mainframe.execute([
      resultstable + ' = Simple.find_gaps(' + tablename + ', ' + str(ascending) + ', ' + (', '.join(cols)) + ')',
      resultstable + '.view()',
    ]):
      self.close()
    
    
    
############################################
###   Select By Key Dialog

class SelectByKey(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'SelectByKey')
    self.operations = [
      [ '==', '"', '"', lang('is equal to the string:') ], 
      [ '== ', '', '',  lang('is equal to the number:') ],
      [ '!=', '', '', lang('is not equal to the number:') ], 
      [ '>', '', '',  lang('is greater than to the number:') ],  
      [ '<', '', '',  lang('is less than the number:') ], 
      [ '>=', '', '',  lang('is greater than or equal to the number:') ],  
      [ '<=', '', '',  lang('is less than or equal to the number:') ], 
    ]
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'), tablearrays=True)
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.refreshColumns)
    self.getControl('criteria').Clear()
    self.getControl('results').SetValue('')
    self.getControl('addcriteriabutton').Bind(wx.EVT_BUTTON, self.onAddCriteria)
    self.getControl('deletecriteriabutton').Bind(wx.EVT_BUTTON, self.onDeleteCriteria)
    self.getControl('selectallbutton').Bind(wx.EVT_BUTTON, self.onSelectAll)
    self.getControl('selectnonebutton').Bind(wx.EVT_BUTTON, self.onSelectNone)
    operation = self.getControl('operation')
    operation.Clear()
    operation.AppendItems([ op[-1] for op in self.operations ])
    operation.SetSelection(0)
    # tell the columns to refresh
    self.refreshColumns()
    
  
  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = self.getControl('columns')
    cols.Clear()
    cols.AppendItems(table.get_column_names())
    cols.SetSelection(0)
    incl = self.getControl('includefields')
    incl.Clear()
    incl.AppendItems(table.get_column_names())
    for i in range(incl.GetCount()):
      incl.Check(i)
    self.getControl('results').SetValue(tablename.replace('.', '_') + '_selected')
    
    
  def onSelectAll(self, event=None):
    '''Selects all the fields in the include fields box'''
    incl = self.getControl('includefields')
    for i in range(incl.GetCount()):
      incl.Check(i)


  def onSelectNone(self, event=None):
    '''Deselects all the fields in the include fields box'''
    incl = self.getControl('includefields')
    for i in range(incl.GetCount()):
      incl.Check(i, False)
    
    
  def onAddCriteria(self, event=None):
    '''Adds criteria to the criteria list box'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    criteria = self.getControl('criteria')
    col = table.get_columns()[self.getControl('columns').GetSelection()].name
    op = self.operations[self.getControl('operation').GetSelection()]
    value = self.getControl('value').GetValue()
    criteria.Append('record.' + col + ' ' + op[0] + ' ' + op[1] + value + op[2])
    self.getControl('value').SetValue('')
    
    
  def onDeleteCriteria(self, event=None):
    '''Deletes the selected criteria item'''
    idx = self.getControl('criteria').GetSelection()
    if idx >= 0:
      self.getControl('criteria').Delete(idx)
    else:
      wx.MessageDialog(self.dlg, lang('Please select the criteria to remove first.'), lang('Warning'), wx.OK | wx.ICON_WARNING).ShowModal()
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    resultstable = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable('Results Name', resultstable)
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    criteria = self.getControl('criteria')
    cols = [ self.fmt(criteria.GetString(i)) for i in range(criteria.GetCount()) ]
    incl = self.getControl('includefields')
    deletecols = [ i for i in range(incl.GetCount()) if not incl.IsChecked(i) ]
    assert len(deletecols) < incl.GetCount(), lang('Please select at least one field to include in the results table.')
    cmds = []
    cmds.append(resultstable + ' = Simple.select(' + tablename + ', \'' + (' and '.join(cols)) + '\')')
    if len(deletecols) > 0:
      cmds.append(resultstable + '.delete_column([' + ','.join([ '"' + table.column(i).name + '"' for i in deletecols ]) + '])')
    cmds.append(resultstable + '.view()')
    if self.mainframe.execute(cmds):
      self.close()
    
    
############################################
###   Select By Wildcard (or Regular Expression)

class SelectByWildcard(Dialog):
  def __init__(self, mainframe, dlgtype):
    Dialog.__init__(self, mainframe, 'SelectByWildcard')
    self.dlgtype = dlgtype

  def init(self):
    '''Initializes the dialog'''
    self.tables = self.initTables(self.getControl('tablename'), tablearrays=True)
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.refreshColumns)
    self.refreshColumns()
    for name in ( 'matchentire', 'matchpartial', 'casespecific', 'caseignore', 'includematching', 'excludematching' ):
      self.getControl(name).Bind(wx.EVT_RADIOBUTTON, self.updateResult)
    for name in ( 'pattern', 'testtext' ):
      self.getControl(name).Bind(wx.EVT_TEXT, self.updateResult)
    if self.dlgtype == 'RegEx':
      self.dlg.SetTitle("Select By Regular Expression")
      self.getControl("patternhelp").SetLabel('Use a period (.) to match any character, ^ to for the beginning and $ for end the of string, \d for a digit (0-9), \w for a leter or number, plus (+) for one or more, and star (*) for zero or more. See http://docs.python.org/dev/howto/regex.html for more information.')


  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = self.getControl('columns')
    cols.Clear()
    cols.AppendItems(table.get_column_names())
    cols.SetSelection(0)


  def updateResult(self, event=None):
    '''Updates the results field'''
    # try to compile the expression
    try:
      fullmatch = self.getControl('matchentire').GetValue()
      ignorecase = self.getControl('caseignore').GetValue()
      direction = self.getControl('includematching').GetValue()
      pattern = self.getControl('pattern').GetValue()
      assert pattern, lang('Start by entering a pattern to match')
      testtext = self.getControl('testtext').GetValue()
      assert testtext, lang('To test your pattern, enter some match text')
      if self.dlgtype == 'RegEx':
        if fullmatch and pattern[:1] != '^':
          pattern = '^' + pattern
        if fullmatch and pattern[-1:] != '$':
          pattern += '$'
        result = Simple.regex_match(pattern, testtext, ignorecase)
      else:
        result = Simple.wildcard_match(pattern, testtext, ignorecase, fullmatch)
      if not direction:
        result = not result
      if result:
        self.getControl('testresult').SetValue(lang('Successful match for this pattern'))
      else:
        self.getControl('testresult').SetValue(lang('No match for this pattern'))
    except Exception, e:
      self.getControl('testresult').SetValue(lang('Pattern error: ') + str(e))


  def ok(self, event=None):
    '''Responds to the ok button'''
    try:
      fullmatch = self.getControl('matchentire').GetValue()
      ignorecase = self.getControl('caseignore').GetValue()
      direction = self.getControl('excludematching').GetValue() and 'not ' or ''
      pattern = self.getControl('pattern').GetValue()
      assert pattern, lang('Start by entering a pattern to match')
      tablename = self.getControl('tablename').GetStringSelection()
      colname = self.getControl('columns').GetStringSelection()
      resultstable = self.fmt(self.getControl('results').GetValue())
      self.checkValidVariable('Results Name', resultstable)
      if self.dlgtype == 'RegEx':
        if fullmatch and pattern[:1] != '^':
          pattern = '^' + pattern
        if fullmatch and pattern[-1:] != '$':
          pattern += '$'
        Simple.regex_match(pattern, '', ignorecase)  # run to check for errors
        expression = direction + "Simple.regex_match('%s', %s, %s)" % (pattern.replace("'", "\\'"), colname, ignorecase)
      else:
        Simple.wildcard_match(pattern, '', ignorecase, fullmatch)  # run to check for errors
        expression = direction + "Simple.wildcard_match('%s', %s, %s, %s)" % (pattern.replace("'", "\\'"), colname, ignorecase, fullmatch)
      cmd = resultstable + ' = Simple.select(' + tablename + ', "' + expression + '")'
      if self.mainframe.execute(cmd):
        self.close()
    except Exception, e:
      raise AssertionError(lang('Pattern error: ') + unicode(e))



    
############################################
###   Select By Expression Dialog

class SelectByExpression(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'SelectByExpression')
    
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'), tablearrays=True)
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.onSelectTable)
    self.getControl('selectallbutton').Bind(wx.EVT_BUTTON, self.onSelectAll)
    self.getControl('selectnonebutton').Bind(wx.EVT_BUTTON, self.onSelectNone)
    self.getControl('expressionbuilderbutton').Bind(wx.EVT_BUTTON, self.onExpressionBuilderButton)
    self.getControl('expression').Clear()
    self.getControl('results').Clear()
    self.onSelectTable()
    
  
  def onSelectTable(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    incl = self.getControl('includefields')
    incl.Clear()
    incl.AppendItems(table.get_column_names())
    for i in range(incl.GetCount()):
      incl.Check(i)
    self.getControl('results').SetValue(tablename.replace('.', '_') + '_selected')
    
    
  def onSelectAll(self, event=None):
    '''Selects all the fields in the include fields box'''
    incl = self.getControl('includefields')
    for i in range(incl.GetCount()):
      incl.Check(i)


  def onSelectNone(self, event=None):
    '''Deselects all the fields in the include fields box'''
    incl = self.getControl('includefields')
    for i in range(incl.GetCount()):
      incl.Check(i, False)
    
  
  def onExpressionBuilderButton(self, event=None):
    '''Runs the expression builder'''
    expression = self.getControl('expression').GetValue()
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    builder = ExpressionBuilder(self.mainframe, expression=expression, local_tables=[tablename])
    if builder.runModal():
      self.getControl('expression').SetValue(builder.expression)
    
    
  def ok(self, event=None):
    '''Responds to the ok button'''
    resultstable = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable('Results Name', resultstable)
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    incl = self.getControl('includefields')
    deletecols = [ i for i in range(incl.GetCount()) if not incl.IsChecked(i) ]
    assert len(deletecols) < incl.GetCount(), lang('Please select at least one field to include in the results table.')
    expression = self.fmt(self.getControl('expression').GetValue())
    cmds = []
    cmds.append(resultstable + ' = Simple.select(' + tablename + ', "' + expression + '")')
    if len(deletecols) > 0:
      cmds.append(resultstable + '.delete_column([' + ','.join([ '"' + table.column(i).name + '"' for i in deletecols ]) + '])')
    cmds.append(resultstable + '.view()')
    if self.mainframe.execute(cmds):
      self.close()
    
    

############################################
###   Select By Index Dialog

class SelectByIndex(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'SelectByIndex')
    
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'), tablearrays=True)
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.onSelectTable)
    self.getControl('results').Clear()
    self.onSelectTable()
    
  
  def onSelectTable(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    self.getControl('fromrow').SetValue("0")
    self.getControl('torow').SetValue(str(len(table) - 1))
    
    
  def ok(self, event=None):
    '''Responds to the ok button'''
    resultstable = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable(lang('Results Name'), resultstable)
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    try:
      fromrow = int(self.getControl('fromrow').GetValue())
    except ValueError:
      assert False, lang('Please enter a valid number in the "From Row" box.')
    try:
      torow = int(self.getControl('torow').GetValue())
    except ValueError:
      assert False, lang('Please enter a valid number in the "To Row" box.')
    cmds = []
    cmds.append('%s = %s[%i:%i]' % (resultstable, tablename, fromrow, torow + 1))
    cmds.append(resultstable + '.view()')
    if self.mainframe.execute(cmds):
      self.close()
        
    
############################################
###   Select Outliers Dialog

class SelectOutliers(Dialog):
  def __init__(self, mainframe, outliers):
    Dialog.__init__(self, mainframe, 'SelectOutliers') 
    self.outliers = outliers   # whether to include outliers or exclude outliers
  
  def init(self):
    '''Initializes the dialog'''
    if self.outliers:
      self.dlg.SetTitle(lang('Select Outliers By Value'))
      self.funcname = 'select_outliers'
    else:
      self.dlg.SetTitle(lang('Exclude Outliers By Value'))
      self.funcname = 'select_nonoutliers'
      
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'), tablearrays=True)
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.refreshColumns)
    self.getControl('minvalue').Clear()
    self.getControl('maxvalue').Clear()

    # tell the columns to refresh
    self.refreshColumns()
    
  
  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = self.getControl('columns')
    cols.Clear()
    cols.AppendItems(table.get_column_names())
    cols.SetSelection(0)
    self.getControl('results').SetValue(tablename.replace('.', '_') + '_outliers')
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    resultstable = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable('Results Name', resultstable)
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    col = table.column(self.getControl('columns').GetSelection()).name
    min = self.getControl('minvalue').GetValue()
    max = self.getControl('maxvalue').GetValue()
    try:
      int(min)
      int(max)
    except ValueError:
      assert False, lang('Please ensure valid numbers are entered for the minimum and maximum values.')
    if self.mainframe.execute([
      resultstable + ' = Simple.' + self.funcname + '(' + tablename + ', "' + col + '", ' + min + ', ' + max + ')',
      resultstable + '.view()',
    ]):
      self.close()
    
    
############################################
###   Select Outliers By Z-score Dialog

class SelectOutliersByZ(Dialog):
  def __init__(self, mainframe, outliers):
    Dialog.__init__(self, mainframe, 'SelectOutliersByZ') 
    self.outliers = outliers   # whether to include outliers or exclude outliers
  
  def init(self):
    '''Initializes the dialog'''
    if self.outliers:
      self.dlg.SetTitle(lang('Select Outliers By ZScore'))
      self.funcname = 'select_outliers_z'
    else:
      self.dlg.SetTitle(lang('Exclude Outliers By ZScore'))
      self.funcname = 'select_nonoutliers_z'
      
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'), tablearrays=True)
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.refreshColumns)
    self.getControl('zscore').Clear()

    # tell the columns to refresh
    self.refreshColumns()
    
  
  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = self.getControl('columns')
    cols.Clear()
    cols.AppendItems(table.get_column_names())
    cols.SetSelection(0)
    self.getControl('results').SetValue(tablename.replace('.', '_') + '_outliers')
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    resultstable = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable('Results Name', resultstable)
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    col = table.column(self.getControl('columns').GetSelection()).name
    zscore = self.getControl('zscore').GetValue()
    try:
      float(zscore)
    except ValueError:
      assert False, lang('Please ensure valid numbers are entered for the minimum and maximum values.')
    if self.mainframe.execute([
      resultstable + ' = Simple.' + self.funcname + '(' + tablename + ', "' + col + '", ' + zscore + ')',
      resultstable + '.view()',
    ]):
      self.close()
    
    
############################################
###   Add ZScore Column Dialog

class AddZScoreCol(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'AddZScoreCol') 
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'), tablearrays=True)
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.refreshColumns)
    self.getControl('newcolname').SetValue('')
    # tell the columns to refresh
    self.refreshColumns()
    
  
  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = self.getControl('columns')
    cols.Clear()
    cols.AppendItems(table.get_column_names())
    cols.SetSelection(0)
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    newcolname = self.fmt(self.getControl('newcolname').GetValue())
    assert newcolname, lang('Please enter a name for the new zscore column.')
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    newcolname = ensure_unique_colname(table, newcolname)
    colidx = self.getControl('columns').GetSelection()
    col = table.column(colidx).name
    if self.mainframe.execute(tablename + '.insert_column(' + str(colidx+1) + ', "' + newcolname + '", number, Simple.calc_zscore(' + tablename + ', "' + col + '"))'):
      self.close()
    
     
    
############################################
###   Add Cusum Column Dialog

class AddCusumCol(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'AddCusumCol') 
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'), tablearrays=True)
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.refreshColumns)
    self.getControl('newcolname').SetValue('')
    # tell the columns to refresh
    self.refreshColumns()
    
  
  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = self.getControl('columns')
    cols.Clear()
    cols.AppendItems(table.get_column_names())
    cols.SetSelection(0)
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    newcolname = self.fmt(self.getControl('newcolname').GetValue())
    if not newcolname:
      wx.MessageDialog(self.dlg, lang('Please enter a name for the new cusum column.'), lang('Warning'), wx.OK | wx.ICON_WARNING).ShowModal()
      return False
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    newcolname = ensure_unique_colname(table, newcolname)
    colidx = self.getControl('columns').GetSelection()
    col = table.column(colidx).name
    if self.mainframe.execute(tablename + '.insert_column(' + str(colidx+1) + ', "' + newcolname + '", number, Trending.cusum(' + tablename + ', "' + col + '"))'):
      self.close()
    
    
                
############################################
###   Match By Key Dialog

class MatchByKey(Dialog):
  def __init__(self, mainframe, title, comparestr, func, resultslabel):
    Dialog.__init__(self, mainframe, 'MatchByKey') 
    self.title = title
    self.func = func
    self.comparestr = comparestr
    self.resultslabel = resultslabel
    
  
  def init(self):
    '''Initializes the dialog'''
    self.dlg.SetTitle(self.title)
    self.getControl('resultslabel').SetLabel(self.resultslabel)
    for control in [ 'equalsA', 'equalsB', 'equalsC' ]:
      self.getControl(control).SetLabel(self.comparestr)
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename1'))
    assert len(self.tables) >= 1, lang('Please open at least one table first.')
    self.getControl('tablename1').Bind(wx.EVT_CHOICE, self.refreshColumns1)
    self.tables = self.initTables(self.getControl('tablename2'))
    self.getControl('tablename2').SetSelection(1)
    self.getControl('tablename2').Bind(wx.EVT_CHOICE, self.refreshColumns2)
    # refresh the columns
    self.refreshColumns1()
    self.refreshColumns2()
    self.getControl('results').Clear()
    
  
  def refreshColumns1(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename1').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    for controlname in [ 'colA1', 'colB1', 'colC1' ]:
      cols = self.getControl(controlname)
      cols.Clear()
      cols.AppendItems(['(none)'] + table.get_column_names())
      cols.SetSelection(0)
    

  def refreshColumns2(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename2').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    for controlname in [ 'colA2', 'colB2', 'colC2' ]:
      cols = self.getControl(controlname)
      cols.Clear()
      cols.AppendItems(['(none)'] + table.get_column_names())
      cols.SetSelection(0)
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    results = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable('Results Name', results)
    if not results:
      wx.MessageDialog(self.dlg, lang('Please enter a name for each results table.'), lang('Warning'), wx.OK | wx.ICON_WARNING).ShowModal()
      return False
    tablename1 = self.tables[self.getControl('tablename1').GetSelection()]
    table1 = self.mainframe.shell.getVariable(tablename1)
    tablename2 = self.tables[self.getControl('tablename2').GetSelection()]
    table2 = self.mainframe.shell.getVariable(tablename2)
    matches = []
    for c1, c2 in [ ('colA1', 'colA2'), ('colB1', 'colB2'), ('colC1', 'colC2') ]:
      colidx1 = self.getControl(c1).GetSelection() - 1  # first element is (none)
      colidx2 = self.getControl(c2).GetSelection() - 1
      if colidx1 >= 0 and colidx2 >= 0:
        matches.append('["' + table1.column(colidx1).name + '","' + table2.column(colidx2).name + '"]') 
    assert len(matches) > 0, lang('Please designate at least one match between columns.')
    if self.mainframe.execute([
      results + ' = ' + self.func + '(' + tablename1 + ', ' + tablename2 + ', ' + ', '.join(matches) + ')',
      results + '.view()',
    ]):
      self.close()
    



############################################
###   Match By Expression Dialog

class MatchByExpression(Dialog):
  def __init__(self, mainframe, title, func, resultslabel):
    Dialog.__init__(self, mainframe, 'MatchByExpression') 
    self.title = title
    self.func = func
    self.resultslabel = resultslabel
    
  
  def init(self):
    '''Initializes the dialog'''
    self.dlg.SetTitle(self.title)
    self.getControl('resultslabel').SetLabel(self.resultslabel)
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename1'))
    self.tables = self.initTables(self.getControl('tablename2'))
    assert len(self.tables) >= 1, lang('Please open at least one table first.')
    self.getControl('expression').Clear()
    self.getControl('results').Clear()
    self.getControl('expressionbuilderbutton').Bind(wx.EVT_BUTTON, self.onExpressionBuilderButton)
    
  
  def onExpressionBuilderButton(self, event=None):
    '''Runs the expression builder'''
    expression = self.getControl('expression').GetValue()
    special_vars = {
      'record1': lang('Denotes the record from the first table that is being evaluated.  For example, record1.Name references the value in the Name field for the current record being evaluated.'),
      'record2': lang('Denotes the record from the second table that is being evaluated.  For example, record2.Age references the value in the Age field for the current record being evaluated.'),
    }
    builder = ExpressionBuilder(self.mainframe, expression=expression, special_vars=special_vars)
    if builder.runModal():
      self.getControl('expression').SetValue(builder.expression)


  def ok(self, event=None):
    '''Responds to the ok button'''
    results = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable(lang('Results Name'), results)
    tablename1 = self.tables[self.getControl('tablename1').GetSelection()]
    table1 = self.mainframe.shell.getVariable(tablename1)
    tablename2 = self.tables[self.getControl('tablename2').GetSelection()]
    table2 = self.mainframe.shell.getVariable(tablename2)
    expression = self.fmt(self.getControl('expression').GetValue().replace('"', '\\"'))
    assert expression, lang('Please enter an expression to match by.')
    if self.mainframe.execute([
      results + ' = ' + self.func + '(' + tablename1 + ', ' + tablename2 + ', "' + expression + '")',
      results + '.view()',
    ]):
      self.close()
    
    
############################################
###   Join By Expression Dialog

class JoinByExpression(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'JoinByExpression') 
    self.funcs = [  # these match the options in the dialog
      'Simple.join',
      'Simple.left_join',
      'Simple.right_join',
    ]
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename1'))
    self.tables = self.initTables(self.getControl('tablename2'))
    assert len(self.tables) >= 1, lang('Please open at least one table first.')
    self.getControl('expression').Clear()
    self.getControl('results').Clear()
    self.getControl('jointype').SetSelection(0)
    self.getControl('expressionbuilderbutton').Bind(wx.EVT_BUTTON, self.onExpressionBuilderButton)
    
  
  def onExpressionBuilderButton(self, event=None):
    '''Runs the expression builder'''
    expression = self.getControl('expression').GetValue()
    special_vars = {
      'record1': lang('Denotes the record from the first table that is being evaluated.  For example, record1.Name references the value in the Name field for the current record being evaluated.'),
      'record2': lang('Denotes the record from the second table that is being evaluated.  For example, record2.Age references the value in the Age field for the current record being evaluated.'),
    }
    builder = ExpressionBuilder(self.mainframe, expression=expression, special_vars=special_vars)
    if builder.runModal():
      self.getControl('expression').SetValue(builder.expression)


  def ok(self, event=None):
    '''Responds to the ok button'''
    results = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable(lang('Results Name'), results)
    tablename1 = self.tables[self.getControl('tablename1').GetSelection()]
    table1 = self.mainframe.shell.getVariable(tablename1)
    tablename2 = self.tables[self.getControl('tablename2').GetSelection()]
    table2 = self.mainframe.shell.getVariable(tablename2)
    expression = self.fmt(self.getControl('expression').GetValue().replace('"', '\\"'))
    assert expression, lang('Please enter an expression to match by.')
    func = self.funcs[self.getControl('jointype').GetSelection()]
    if self.mainframe.execute([
      results + ' = ' + func + '(' + tablename1 + ', ' + tablename2 + ', "' + expression + '")',
      results + '.view()',
    ]):
      self.close()
    
        
    
############################################
###   Join By Value Dialog

class JoinByValue(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'JoinByValue') 
    self.funcs = [  # these match the options in the dialog
      'Simple.col_join',
      'Simple.col_left_join',
      'Simple.col_right_join',
    ]
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename1'))
    assert len(self.tables) >= 1, lang('Please open at least one table first.')
    self.getControl('tablename1').Bind(wx.EVT_CHOICE, self.refreshColumns1)
    self.tables = self.initTables(self.getControl('tablename2'))
    self.getControl('tablename2').SetSelection(1)
    self.getControl('tablename2').Bind(wx.EVT_CHOICE, self.refreshColumns2)
    # refresh the columns
    self.refreshColumns1()
    self.refreshColumns2()
    self.getControl('jointype').SetSelection(0)
    self.getControl('results').Clear()
    
  
  def refreshColumns1(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename1').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    for controlname in [ 'colA1', 'colB1', 'colC1' ]:
      cols = self.getControl(controlname)
      cols.Clear()
      cols.AppendItems(['(none)'] + table.get_column_names())
      cols.SetSelection(0)
    

  def refreshColumns2(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename2').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    for controlname in [ 'colA2', 'colB2', 'colC2' ]:
      cols = self.getControl(controlname)
      cols.Clear()
      cols.AppendItems(['(none)'] + table.get_column_names())
      cols.SetSelection(0)
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    results = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable(lang('Results Name'), results)
    if not results:
      wx.MessageDialog(self.dlg, lang('Please enter a name for each results table.'), lang('Warning'), wx.OK | wx.ICON_WARNING).ShowModal()
      return False
    tablename1 = self.tables[self.getControl('tablename1').GetSelection()]
    table1 = self.mainframe.shell.getVariable(tablename1)
    tablename2 = self.tables[self.getControl('tablename2').GetSelection()]
    table2 = self.mainframe.shell.getVariable(tablename2)
    matches = []
    for c1, c2 in [ ('colA1', 'colA2'), ('colB1', 'colB2'), ('colC1', 'colC2') ]:
      colidx1 = self.getControl(c1).GetSelection() - 1  # first element is (none)
      colidx2 = self.getControl(c2).GetSelection() - 1
      if colidx1 >= 0 and colidx2 >= 0:
        matches.append('["' + table1.column(colidx1).name + '","' + table2.column(colidx2).name + '"]') 
    assert len(matches) > 0, lang('Please designate at least one match between columns.')
    func = self.funcs[self.getControl('jointype').GetSelection()]
    if self.mainframe.execute([
      results + ' = ' + func + '(' + tablename1 + ', ' + tablename2 + ', ' + ', '.join(matches) + ')',
      results + '.view()',
    ]):
      self.close()
    
############################################
###   Join by Soundex Dialog

class JoinBySoundex(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'JoinBySoundex') 
    self.funcs = [  # these match the options in the dialog
      'Simple.join',
      'Simple.left_join',
      'Simple.right_join',
    ]
    
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename1'))
    assert len(self.tables) >= 1, lang('Please open at least one table first.')
    self.getControl('tablename1').Bind(wx.EVT_CHOICE, self.refreshColumns1)
    self.tables = self.initTables(self.getControl('tablename2'))
    self.getControl('tablename2').Bind(wx.EVT_CHOICE, self.refreshColumns2)
    self.refreshColumns1()
    self.refreshColumns2()
    self.getControl('jointype').SetSelection(0)
    self.getControl('results').Clear()
    
  
  def refreshColumns1(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename1').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    for controlname in [ 'colA1', 'colB1', 'colC1' ]:
      cols = self.getControl(controlname)
      cols.Clear()
      cols.AppendItems(['(none)'] + table.get_column_names())
      cols.SetSelection(0)
    

  def refreshColumns2(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename2').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    for controlname in [ 'colA2', 'colB2', 'colC2' ]:
      cols = self.getControl(controlname)
      cols.Clear()
      cols.AppendItems(['(none)'] + table.get_column_names())
      cols.SetSelection(0)
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    results = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable(lang('Results Name'), results)
    soundexlen = str(self.getControl('soundexlen').GetValue())
    tablename1 = self.tables[self.getControl('tablename1').GetSelection()]
    table1 = self.mainframe.shell.getVariable(tablename1)
    tablename2 = self.tables[self.getControl('tablename2').GetSelection()]
    table2 = self.mainframe.shell.getVariable(tablename2)
    matches = []
    for c1, c2 in [ ('colA1', 'colA2'), ('colB1', 'colB2'), ('colC1', 'colC2') ]:
      colidx1 = self.getControl(c1).GetSelection() - 1  # first element is (none)
      colidx2 = self.getControl(c2).GetSelection() - 1
      if colidx1 >= 0 and colidx2 >= 0:
        matches.append('Simple.soundex(record1.' + table1.column(colidx1).name + ', ' + soundexlen + ') == Simple.soundex(record2.' + table2.column(colidx2).name + ', ' + soundexlen + ')') 
    assert len(matches) > 0, lang('Please designate at least one match between columns.')
    func = self.funcs[self.getControl('jointype').GetSelection()]
    if self.mainframe.execute([
      results + ' = ' + func + '(' + tablename1 + ', ' + tablename2 + ', \'' + (' and '.join(matches)) + '\')',
      results + '.view()',
    ]):
      self.close()
    

############################################
###   Join by Fuzzy Match Dialog

class JoinByFuzzyMatch(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'JoinByFuzzyMatch') 
    self.funcs = [  # these match the options in the dialog
      'Simple.join',
      'Simple.left_join',
      'Simple.right_join',
    ]
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename1'))
    assert len(self.tables) >= 1, lang('Please open at least one table first.')
    self.getControl('tablename1').Bind(wx.EVT_CHOICE, self.refreshColumns1)
    self.tables = self.initTables(self.getControl('tablename2'))
    self.getControl('tablename2').Bind(wx.EVT_CHOICE, self.refreshColumns2)
    self.refreshColumns1()
    self.refreshColumns2()
    self.getControl('jointype').SetSelection(0)
    self.getControl('results').Clear()
  
  def refreshColumns1(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename1').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    for controlname in [ 'colA1', 'colB1', 'colC1' ]:
      cols = self.getControl(controlname)
      cols.Clear()
      cols.AppendItems(['(none)'] + table.get_column_names())
      cols.SetSelection(0)
    

  def refreshColumns2(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename2').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    for controlname in [ 'colA2', 'colB2', 'colC2' ]:
      cols = self.getControl(controlname)
      cols.Clear()
      cols.AppendItems(['(none)'] + table.get_column_names())
      cols.SetSelection(0)
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    results = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable(lang('Results Name'), results)
    fuzzyval = float(self.getControl('fuzzyval').GetValue()) / 100.0
    tablename1 = self.tables[self.getControl('tablename1').GetSelection()]
    table1 = self.mainframe.shell.getVariable(tablename1)
    tablename2 = self.tables[self.getControl('tablename2').GetSelection()]
    table2 = self.mainframe.shell.getVariable(tablename2)
    matches = []
    for c1, c2 in [ ('colA1', 'colA2'), ('colB1', 'colB2'), ('colC1', 'colC2') ]:
      colidx1 = self.getControl(c1).GetSelection() - 1  # first element is (none)
      colidx2 = self.getControl(c2).GetSelection() - 1
      if colidx1 >= 0 and colidx2 >= 0:
        matches.append('Simple.fuzzymatch(record1.' + table1.column(colidx1).name + ', record2.' + table2.column(colidx2).name + ') >= ' + str(fuzzyval)) 
        if self.getControl('ignoreExact').IsChecked():
	      matches[-1] += ' and record1.' + table1.column(colidx1).name + ' != record2.' + table2.column(colidx2).name
    assert len(matches) > 0, lang('Please designate at least one match between columns.')
    func = self.funcs[self.getControl('jointype').GetSelection()]
    if self.mainframe.execute([
      results + ' = ' + func + '(' + tablename1 + ', ' + tablename2 + ', \'' + (' and '.join(matches)) + '\')',
      results + '.view()',
    ]):
      self.close()
    
    
    
    
############################################
###   StratifyByNumberOfGroups Dialog

class StratifyByNumberOfGroups(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'StratifyByNumberOfGroups') 
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'))
    self.getControl('numgroups').SetValue('')
    self.getControl('results').SetValue('')
    
  
  def ok(self, event=None):
    '''Responds to the ok button'''
    numgroups = self.getControl('numgroups').GetValue()
    try:
      int(numgroups)
    except ValueError:
      assert False, lang('Please enter a valid number of groups.')
    assert int(numgroups) >= 2, lang('You must stratify into at least two groups.')
    results = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable(lang('Results Name'), results)
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    if self.mainframe.execute([
      results + ' = Grouping.stratify(' + tablename + ', ' + numgroups + ')',
      results + '.view()',
    ]):
      self.close()
    
    
############################################
###   CombineTableList Dialog

class CombineTableList(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'CombineTableList') 
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'), tables=False, tablearrays=True)
    self.getControl('results').SetValue('')
    
  
  def ok(self, event=None):
    '''Responds to the ok button'''
    results = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable('New Table Name', results)
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    if self.mainframe.execute([
      '%s = %s.combine()' % (results, tablename),
      '%s.view()' % (results, ),
    ]):
      self.close()
    
    
    
############################################
###   Stratify By Value Dialog

class StratifyByValue(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'StratifyByValue') 
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'))
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.refreshColumns)
    self.getControl('results').SetValue('')

    # tell the columns to refresh
    self.refreshColumns()
    
  
  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = self.getControl('columns')
    cols.Clear()
    cols.AppendItems(table.get_column_names())
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    resultstable = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable('Results Name', resultstable)
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    column_names = table.get_column_names()
    cols = []
    for i, name in enumerate(column_names):
      if self.getControl('columns').IsChecked(i):
        cols.append('"' + name + '"')
    assert len(cols) > 0, lang('Please select at least on column.')
    if self.mainframe.execute([
      resultstable + ' = Grouping.stratify_by_value(' + tablename + ', ' + (', '.join(cols)) + ')',
      resultstable + '.view()',
    ]):
      self.close()
    


############################################
###   Summarize By Value Dialog

class SummarizeByValue(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'SummarizeByValue') 
    self.calculations = [
      [ 'Sum',      "sum(group['fieldname'])" ],
      [ 'Average',  "mean(group['fieldname'])" ],
      [ 'Count',    "count(group['fieldname'])" ],
      [ 'Max',      "max(group['fieldname'])" ],
      [ 'Min',      "min(group['fieldname'])" ],
      [ 'StdDev',   "stdev(group['fieldname'])" ],
      [ 'Variance', "variance(group['fieldname'])" ],
    ]
    
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'))
    self.getControl('tablename').Bind(wx.EVT_LISTBOX, self.refreshColumns)
    self.getControl('results').SetValue('')

    # tell the columns to refresh
    self.refreshColumns()

    # summarize area    
    self.getControl('expressions').Clear()
    self.getControl('expression').SetValue('')
    self.getControl('results').SetValue('')
    self.getControl('addbutton').Bind(wx.EVT_BUTTON, self.onAddButton)
    self.getControl('removebutton').Bind(wx.EVT_BUTTON, self.onRemoveButton)
    
    # calculations
    calculatechoice = self.getControl('calculatechoice')
    calculatechoice.Clear()
    calculatechoice.AppendItems([ name for name, calc in self.calculations])
    calculatechoice.SetSelection(0)
  
  
  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    for control in [ 'columns', 'fieldchoice' ]:
      cols = self.getControl(control)
      cols.Clear()
      cols.AppendItems(table.get_column_names())
    self.getControl('fieldchoice').SetSelection(0)
    self.getControl('expressions').Clear()
    

  def onAddButton(self, event=None):
    if self.getControl('expression').GetValue() == '':
      field = self.getControl('fieldchoice').GetStringSelection()
      calc = self.calculations[self.getControl('calculatechoice').GetSelection()][1]
      expr = calc.replace('fieldname', field)
    else:
      expr = self.getControl('expression').GetValue()
      self.getControl('expression').SetValue('')
    colnames = [ self.getControl('expressions').GetString(i)[:self.getControl('expressions').GetString(i).find('=')] for i in range(self.getControl('expressions').GetCount()) ]
    while True:
      try:
        colname = wx.GetTextFromUser(lang("Enter a column name for this expression:"))
        if not colname:
          break
        assert VARIABLE_RE.match(colname), lang('Please enter a valid column name (letters and numbers only).')
        assert not colname in colnames, lang('This column name has already been used.  Please try a different name.')
        self.getControl('expressions').Append(colname + '="' + expr.replace('"', '\\"') + '"')
        break
      except AssertionError, e:
        wx.MessageDialog(self.dlg, str(e), lang('Warning'), wx.OK | wx.ICON_WARNING).ShowModal()
      
  
  def onRemoveButton(self, event=None):
    idx = self.getControl('expressions').GetSelection()
    if idx >= 0:
      self.getControl('expressions').Delete(idx)
    else:
      wx.MessageDialog(self.dlg, lang('Please select the expression to remove first.'), lang('Warning'), wx.OK | wx.ICON_WARNING).ShowModal()
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    # results variable
    resultstable = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable('Results Name', resultstable)
    
    # table columns
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    column_names = table.get_column_names()
    cols = []
    for i, name in enumerate(column_names):
      if self.getControl('columns').IsChecked(i):
        cols.append('"' + name + '"')
    assert len(cols) > 0, lang('Please select at least on column to stratify by.')
    
    # expressions
    expressions = []
    for i in range(self.getControl('expressions').GetCount()):
      expressions.append(self.fmt(self.getControl('expressions').GetString(i)))
    assert len(expressions) > 0, lang('Please add at least one summarizing expression.')
    
    # execute
    if self.mainframe.execute([
      resultstable + ' = Grouping.summarize_by_value(' + tablename + ', ' + (', '.join(cols)) + ', ' + (', '.join(expressions)) + ')',
      resultstable + '.view()',
    ]):
      self.close()
    

############################################
###   Stratify By Date Dialog

class StratifyByDate(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'StratifyByDate') 
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'))
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.refreshColumns)
    self.getControl('results').SetValue('')

    # tell the columns to refresh
    self.refreshColumns()
    
  
  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = self.getControl('columns')
    cols.Clear()
    cols.AppendItems(table.get_column_names())
    cols.SetSelection(0)
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    resultstable = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable(lang('Results Name'), resultstable)
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    col = self.getControl('columns').GetStringSelection()
    assert col, lang('Please select the date column to stratify by.')
    step = self.getControl('step').GetValue()
    assert step, lang('Please enter a step value.')
    try:
      float(step)
    except ValueError:
      assert False, lang('Please enter a numerical step value.')
    if self.mainframe.execute([
      resultstable + ' = Grouping.stratify_by_date(' + tablename + ', "' + col + '", ' + step + ')',
      resultstable + '.view()',
    ]):
      self.close()
    


############################################
###   Summarize By Date Dialog

class SummarizeByDate(SummarizeByValue):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'SummarizeByDate') 
    self.calculations = [
      [ 'Sum',      "sum(group['fieldname'])" ],
      [ 'Average',  "mean(group['fieldname'])" ],
      [ 'Count',    "count(group['fieldname'])" ],
      [ 'Max',      "max(group['fieldname'])" ],
      [ 'Min',      "min(group['fieldname'])" ],
      [ 'StdDev',   "stdev(group['fieldname'])" ],
      [ 'Variance', "variance(group['fieldname'])" ],
    ]
    
  
  def init(self):
    '''Initializes the dialog'''
    SummarizeByValue.init(self)
    self.getControl('step').SetValue('')
    
    
  def ok(self, event=None):
    '''Responds to the ok button'''
    # results variable
    resultstable = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable('Results Name', resultstable)
    
    # table columns
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    col = self.getControl('columns').GetStringSelection()
    assert col, lang('Please select the date column to stratify by.')
 
    # step
    step = self.getControl('step').GetValue()
    assert step, lang('Please enter a step value.')
    try:
      float(step)
    except ValueError:
      assert False, lang('Please enter a numerical step value.')
    
    # expressions
    expressions = []
    for i in range(self.getControl('expressions').GetCount()):
      expressions.append(self.fmt(self.getControl('expressions').GetString(i)))
    assert len(expressions) > 0, lang('Please add at least one summarizing expression.')
    
    # execute
    if self.mainframe.execute([
      resultstable + ' = Grouping.summarize_by_date(' + tablename + ', "' + col + '", ' + step + ', ' + (', '.join(expressions)) + ')',
      resultstable + '.view()',
    ]):
      self.close()
    

############################################
###   Stratify By Expression Dialog

class StratifyByExpression(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'StratifyByExpression')
    
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'))
    self.getControl('expression').Clear()
    self.getControl('results').Clear()
    
  
  def ok(self, event=None):
    '''Responds to the ok button'''
    resultstable = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable(lang('Results Name'), resultstable)
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    expression = self.fmt(self.getControl('expression').GetValue())
    assert expression, lang('Please enter an expression to stratify with.')
    if self.mainframe.execute([
      resultstable + ' = Grouping.stratify_by_expression(' + tablename + ', "' + expression + '")',
      resultstable + '.view()',
    ]):
      self.close()
    


############################################
###   Summarize Dialog

class Summarize(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'Summarize')
    
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.mainframe.getTables(tables=False, tablearrays=True)
    self.tables.sort()
    assert len(self.tables) > 0, lang('There are no table lists to summarize (note that table lists are different than singular tables). Stratify a table first, then rerun this procedure.')
    tablechoice = self.getControl('tablelist')
    tablechoice.Clear()
    tablechoice.AppendItems(self.tables)
    tablechoice.SetSelection(0)
    self.getControl('summaryname').SetValue('')
    self.getControl('expression').SetValue('')
    self.getControl('expressions').Clear()
    self.getControl('results').SetValue('')
    self.getControl('addcriteriabutton').Bind(wx.EVT_BUTTON, self.onAddCriteria)
    self.getControl('deletecriteriabutton').Bind(wx.EVT_BUTTON, self.onDeleteCriteria)
    
  
  def onAddCriteria(self, event=None):
    '''Adds criteria to the criteria list box'''
    summaryname = self.getControl('summaryname').GetValue()
    expression = self.getControl('expression').GetValue()
    expressions = self.getControl('expressions')
    expressions.Append(summaryname + '="' + expression + '"')
    self.getControl('summaryname').SetValue('')
    self.getControl('expression').SetValue('')
    
    
  def onDeleteCriteria(self, event=None):
    '''Deletes the selected criteria item'''
    idx = self.getControl('expressions').GetSelection()
    if idx >= 0:
      self.getControl('expressions').Delete(idx)
    else:
      wx.MessageDialog(self.dlg, lang('Please select the expression to remove first.'), lang('Warning'), wx.OK | wx.ICON_WARNING).ShowModal()
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    resultstable = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable(lang('Results Name'), resultstable)
    TableList = self.tables[self.getControl('tablelist').GetSelection()]
    expressions = self.getControl('expressions')
    cols = []
    for i in range(expressions.GetCount()):
      cols.append(self.fmt(expressions.GetString(i)))
    assert len(cols) > 0, lang('Please add at least one expressions for the summary table.')
    if self.mainframe.execute([
      resultstable + ' = Grouping.summarize(' + TableList + ', ' + (', '.join(cols)) + ')',
      resultstable + '.view()',
    ]):
      self.close()
    
    
    
    
############################################
###   Trending with single column dialog

class TrendSingleCol(Dialog):
  def __init__(self, mainframe, title, funcname):
    Dialog.__init__(self, mainframe, 'TrendSingleCol') 
    self.title = title
    self.funcname = funcname
    
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.dlg.SetTitle(self.title)
    self.tables = self.initTables(self.getControl('tablename'), tablearrays=True)
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.refreshColumns)
    self.getControl('results').SetValue('')
    # tell the columns to refresh
    self.refreshColumns()
    
  
  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = self.getControl('columns')
    cols.Clear()
    cols.AppendItems(table.get_column_names())
    cols.SetSelection(0)
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    results = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable(lang('Results Name'), results)
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    colidx = self.getControl('columns').GetSelection()
    col = table.column(colidx).name
    if self.mainframe.execute([
      results + ' = ' + self.funcname + '(' + tablename + ', "' + col + '")',
      results + '.view()',
    ]):
      self.close()
    
    

    
    
############################################
###   Calculates Benford's expected values

class CalcBenfords(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'CalcBenfords') 
    
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'), tablearrays=True)
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.refreshColumns)
    self.getControl('results').SetValue('')
    # tell the columns to refresh
    self.refreshColumns()
    
  
  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = self.getControl('columns')
    cols.Clear()
    cols.AppendItems(table.get_column_names())
    cols.SetSelection(0)
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    sigdigits = self.getControl('sigdigits').GetValue()
    results = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable(lang('Results Name'), results)
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    colidx = self.getControl('columns').GetSelection()
    col = table.column(colidx).name
    if self.mainframe.execute([
      results + ' = Benfords.analyze(' + tablename + ', "' + col + '", + ' + str(sigdigits) + ')',
      results + '.view()',
    ]):
      self.close()
    
    


    
############################################
###  Adds a column for Benford's difference

class AddBenfordsDiffCol(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'AddBenfordsDiffCol') 
    
  
  def init(self):
    '''Initializes the dialog'''
    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'), tablearrays=True)
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.refreshColumns)
    self.getControl('newcolname').SetValue('')
    # tell the columns to refresh
    self.refreshColumns()
    
  
  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = self.getControl('columns')
    cols.Clear()
    cols.AppendItems(table.get_column_names())
    cols.SetSelection(0)
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    sigdigits = self.getControl('sigdigits').GetValue()
    newcolname = self.fmt(self.getControl('newcolname').GetValue())
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    colidx = self.getControl('columns').GetSelection()
    col = table.column(colidx).name
    if self.mainframe.execute(tablename + '.insert_column(' + str(colidx+1) + ', "' + newcolname + '", number, Benfords.analyze(' + tablename + ', "' + col + '", ' + str(sigdigits) + ').column(3))'):
      self.close()
    
    

    
############################################
###  Pivot Table dialog

class PivotTable(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'PivotTable') 
    self.calculations = [
      [ 'Sum',      "sum(group['fieldname'])" ],
      [ 'Average',  "mean(group['fieldname'])" ],
      [ 'Count',    "count(group['fieldname'])" ],
      [ 'Max',      "max(group['fieldname'])" ],
      [ 'Min',      "min(group['fieldname'])" ],
      [ 'StdDev',   "stdev(group['fieldname'])" ],
      [ 'Variance', "variance(group['fieldname'])" ],
    ]
    
  
  def init(self):
    '''Initializes the dialog'''
    calculatechoice = self.getControl('calculatechoice')
    calculatechoice.Clear()
    calculatechoice.AppendItems([ name for name, calc in self.calculations])
    calculatechoice.SetSelection(0)

    # set up the tables choice item
    self.tables = self.initTables(self.getControl('tablename'), tablearrays=True)
    self.getControl('tablename').Bind(wx.EVT_CHOICE, self.refreshColumns)
    self.refreshColumns()
    self.getControl('expressions').Clear()
    self.getControl('expression').SetValue('')
    self.getControl('results').SetValue('')
    self.getControl('addbutton').Bind(wx.EVT_BUTTON, self.onAddButton)
    self.getControl('removebutton').Bind(wx.EVT_BUTTON, self.onRemoveButton)
    
  
  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    for controlname in [ 'colfields', 'rowfields', 'fieldchoice' ]:
      cols = self.getControl(controlname)
      cols.Clear()
      cols.AppendItems(table.get_column_names())
      cols.SetSelection(0)
    self.getControl('expressions').Clear()


  def onAddButton(self, event=None):
    expr = self.getControl('expression').GetValue()
    if expr == '':
      field = self.getControl('fieldchoice').GetStringSelection()
      calc = self.calculations[self.getControl('calculatechoice').GetSelection()][1]
      self.getControl('expressions').Append(calc.replace('fieldname', field))
    else:
      expr = self.getControl('expression').GetValue()
      if expr == '':
        wx.MessageDialog(self.dlg, lang('Please enter an expression first.'), lang('Warning'), wx.OK | wx.ICON_WARNING).ShowModal()
        return
      self.getControl('expressions').Append(expr)
      self.getControl('expression').SetValue('')
  
    
  
  def onRemoveButton(self, event=None):
    idx = self.getControl('expressions').GetSelection()
    if idx >= 0:
      self.getControl('expressions').Delete(idx)
    else:
      wx.MessageDialog(self.dlg, lang('Please select the expression to remove first.'), lang('Warning'), wx.OK | wx.ICON_WARNING).ShowModal()
    

  def ok(self, event=None):
    '''Responds to the ok button'''
    tablename = self.tables[self.getControl('tablename').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    column_names = table.get_column_names()
    cols = []
    for i, name in enumerate(column_names):
      if self.getControl('colfields').IsChecked(i):
        cols.append('"' + name + '"')
    rows = []
    for i, name in enumerate(column_names):
      if self.getControl('rowfields').IsChecked(i):
        rows.append('"' + name + '"')
    assert len(cols) > 0 and len(rows) > 0, lang('Please check at least one field for the column labels and one field for the row labels.')
    expressions = []
    for i in range(self.getControl('expressions').GetCount()):
      expressions.append('"' + self.fmt(self.getControl('expressions').GetString(i)) + '"')
    assert len(expressions) > 0, lang('Please add at least one cell expression.')
    results = self.fmt(self.getControl('results').GetValue())
    self.checkValidVariable(lang('Results Name'), results)
    if self.mainframe.execute([
      '%s = Crosstable.pivot(%s, [%s], [%s], [%s])' % (results, tablename, ', '.join(cols), ', '.join(rows), ', '.join(expressions)),
      '%s.view()' % (results,),
    ]):
      self.close()
    
    
    
    
    
############################################
###   Preferences Dialog

class PreferencesDlg(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'Preferences')
    self.languages = Utils.getLanguages()
    self.oldlanguage = None
    
  
  def init(self):
    '''Initializes the dialog'''
    self.getControl('tablefont').SetValue(str(Preferences.get('tablefont', '')))
    self.getControl('font').Clear()
    for fonts in Editor.FONTS:
      self.getControl('font').Append(fonts[0])
    self.getControl('font').SetSelection(Preferences.get('editorfontname', 0))
    self.getControl('fontsize').SetValue(str(Preferences.get('editorfontsize', Editor.DEFAULT_FONT_SIZE)))
    self.getControl('showsplash').SetValue(Preferences.get('showsplash', True))
    self.getControl('checkforupdates').SetValue(Preferences.get('checkforupdates', True))
    self.getControl('useantialiasedfonts').SetValue(Preferences.get('useantialiasedfonts', True))
    self.getControl('usespacesfortabs').SetValue(Preferences.get('usespacesfortabs', True))
    self.getControl('tabwidth').SetValue(str(Preferences.get('tabwidth', 4)))
    self.getControl('automethodlist').SetValue(Preferences.get('automethodlist', True))
    self.getControl('automethodhelp').SetValue(Preferences.get('automethodhelp', True))
    self.getControl('language').Clear()
    self.getControl('language').AppendItems(self.languages)
    try:
      self.getControl('language').SetSelection(self.languages.index(Preferences.get('language')))
    except ValueError:
      self.getControl('language').SetSelection(self.languages.index('English'))
    self.oldlanguage = self.getControl('language').GetSelection()
    
    self.getControl('selecttablefontbutton').Bind(wx.EVT_BUTTON, self.onTableFontButton)
    
    
  def onTableFontButton(self, event=None):
    '''Shows the select font dialog'''
    fontdata = wx.FontData()
    if self.getControl('tablefont').GetValue():
      font = wx.Font(12, wx.MODERN, wx.NORMAL, wx.NORMAL)
      font.SetNativeFontInfoFromString(self.getControl('tablefont').GetValue())
      fontdata.SetInitialFont(font)
    fontdlg = wx.FontDialog(self.dlg, fontdata)
    if fontdlg.ShowModal() == wx.ID_OK:
      newfontst = fontdlg.GetFontData().GetChosenFont().GetNativeFontInfoDesc()
      self.getControl('tablefont').SetValue(newfontst)    
  
    
  def ok(self, event=None):
    '''Responds to the ok button'''
    try: int(self.getControl('tabwidth').GetValue())
    except ValueError: assert False, lang('Please enter a valid integer for the tab width')
    try: int(self.getControl('fontsize').GetValue())
    except ValueError: assert False, lang('Please enter a valid editor font size')
    Preferences.put('showsplash', self.getControl('showsplash').IsChecked())
    Preferences.put('checkforupdates', self.getControl('checkforupdates').IsChecked())
    Preferences.put('language', self.languages[self.getControl('language').GetSelection()])
    Preferences.put('useantialiasedfonts', self.getControl('useantialiasedfonts').IsChecked())
    Preferences.put('usespacesfortabs', self.getControl('usespacesfortabs').IsChecked())
    Preferences.put('tablefont', self.getControl('tablefont').GetValue())
    Preferences.put('editorfontname', self.getControl('font').GetSelection())
    Preferences.put('editorfontsize', int(self.getControl('fontsize').GetValue()))
    Preferences.put('useantialiasedfonts', self.getControl('useantialiasedfonts').IsChecked())
    Preferences.put('tabwidth', int(self.getControl('tabwidth').GetValue()))
    Preferences.put('automethodlist', self.getControl('automethodlist').IsChecked())
    Preferences.put('automethodhelp', self.getControl('automethodhelp').IsChecked())
    Preferences.save()
    
    # tell the user to restart if they changed the language
    if self.getControl('language').GetSelection() != self.oldlanguage:
      wx.MessageDialog(self.dlg, lang('Please restart Picalo to activate the language change.'), lang('Language Change'), wx.OK | wx.ICON_WARNING).ShowModal()
    
    # go through all the open editors and set their styles
    self.mainframe.shell.updatePreferences()
    self.mainframe.output.updatePreferences()
    self.mainframe.commandlog.updatePreferences()
    for i in range(self.mainframe.notebook.GetPageCount()):
      self.mainframe.notebook.GetPage(i).updatePreferences()
    
    # close up shop
    self.close()    
  
        

############################################
###   Find & Replace In Script Dialog

class FindInScript(Dialog):
  def __init__(self, mainframe, editor):
    Dialog.__init__(self, mainframe, 'FindInScript') 
    self.editor = editor
    self.initted = False
    
    
  def init(self):
    '''Initializes the dialog'''
    # events
    if not self.initted:
      self.getControl('findbutton').Bind(wx.EVT_BUTTON, self.onFindButton)
      self.getControl('replacebutton').Bind(wx.EVT_BUTTON, self.onReplaceButton)
      self.getControl('replaceallbutton').Bind(wx.EVT_BUTTON, self.onReplaceAllButton)
      self.initted = True
    self.getControl('findtext').SetFocus()
    self.getControl('findtext').SetSelection(-1, -1)
    

  def onFindButton(self, event=None):
    '''Find button'''
    self.doFind()
    self.hide()
    
      
  def onReplaceButton(self, event=None):
    '''Replace button'''
    self.doFind(True)
    self.hide()
    
    
  def onReplaceAllButton(self, event=None):
    '''Replace All button'''
    self.editor.Freeze()
    # do all the replacements we can find
    numreplacements = 0
    while self.doFind(True, False):
      numreplacements += 1
    # close it up
    self.close()
    # move cursor back and thaw out
    self.editor.Thaw()
    wx.MessageBox(str(numreplacements) + lang(' replacement(s) made.'), lang('Find & Replace'), wx.OK | wx.ICON_ERROR)

    
  def doFind(self, replace=False, showMessages=True):
    '''Does the actual find'''
    # start at the top?
    if self.getControl('startattop').IsChecked():
      currentpos = 0
    else:
      currentpos = self.editor.GetCurrentPos() + 1

    # find
    casesensitive = self.getControl('casesensitive').IsChecked() and wx.STC_FIND_MATCHCASE or 0
    findtext = self.getControl('findtext').GetValue()
    pos = self.editor.FindText(currentpos, self.editor.GetLength(), findtext, casesensitive)
    if pos >= 0:
      self.editor.GotoPos(pos)
      self.editor.SetSelection(pos, pos + len(findtext))
      if replace:
        self.editor.ReplaceSelection(self.getControl('replacetext').GetValue())
      return True
      
    elif showMessages:
      wx.MessageBox(lang('No matches found.'), lang('No matches found.'), wx.OK | wx.ICON_ERROR)
      return False


  def cancel(self):
    '''Responds to a cancel event.'''
    return self.hide()  # so the dialog doesn't get destroyed
    
    
    
      
############################################
###   Find & Replace In Table Dialog
###   This dialog is a little different than the others --
###   It is only created once, then we keep it in memory

class FindInTable(Dialog):
  def __init__(self, mainframe, spreadsheet):
    Dialog.__init__(self, mainframe, 'FindInTable') 
    self.spreadsheet = spreadsheet
    self.initted = False
  
  def init(self):
    '''Initializes the dialog'''
    # events
    if not self.initted:
      self.getControl('findbutton').Bind(wx.EVT_BUTTON, self.onFindButton)
      self.getControl('replacebutton').Bind(wx.EVT_BUTTON, self.onReplaceButton)
      self.getControl('replaceallbutton').Bind(wx.EVT_BUTTON, self.onReplaceAllButton)
      self.initted = True
    self.getControl('findtext').SetFocus()
    self.getControl('findtext').SetSelection(-1, -1)
    

  def onFindButton(self, event=None):
    '''Find button'''
    self.doFind()

      
  def onReplaceButton(self, event=None):
    '''Replace button'''
    if not self.getControl('matchentirecell').IsChecked():
        wx.MessageBox(lang('Because not all cells are string type, you can only replace entire cells.\n\nPlease select the "Match Entire Cell" option.'), lang('Picalo'), wx.OK | wx.ICON_ERROR)
        return False
    self.spreadsheet.Freeze()
    self.doFind(True)
    self.spreadsheet.Thaw()
    
    
  def onReplaceAllButton(self, event=None):
    '''Replace All button'''
    if not self.getControl('matchentirecell').IsChecked():
        wx.MessageBox(lang('Because not all cells are string type, you can only replace entire cells.\n\nPlease select the "Match Entire Cell" option.'), lang('Picalo'), wx.OK | wx.ICON_ERROR)
        return False
    self.spreadsheet.Freeze()
    # do all the replacements we can find
    numreplacements = 0
    while self.doFind(True, False):
      numreplacements += 1
    # close it up
    self.hide()
    self.spreadsheet.Thaw()
    wx.MessageBox(str(numreplacements) + lang(' replacement(s) made.'), lang('Find & Replace'), wx.OK | wx.ICON_ERROR)
    
    
  def doFind(self, replace=False, showMessages=True):
    '''Does the actual find'''
    # start at the top?
    if self.getControl('startattop').IsChecked():
      currentrow = 0
      currentcol = 0
    else:
      currentrow = self.spreadsheet.grid.GetGridCursorRow()
      currentcol = self.spreadsheet.grid.GetGridCursorCol() + 1

    # settings    
    entirecell = self.getControl('matchentirecell').IsChecked()
    casesensitive = self.getControl('casesensitive').IsChecked()
    findtext = self.getControl('findtext').GetValue()
    if not casesensitive:
      findtext = findtext.lower()

    # start searching from this location
    while True:
      # move the row down if we're beyond the number of columns
      if currentcol >= len(self.spreadsheet.table.columns):
        currentcol = 0
        currentrow += 1
      if currentrow >= len(self.spreadsheet.table) or len(self.spreadsheet.table) == 0:
        if showMessages:
          wx.MessageBox(lang('No matches found.'), 'Picalo', wx.OK | wx.ICON_ERROR)
        return False
      
      # check the cell
      cellvalue = str(self.spreadsheet.table[currentrow][currentcol])
      if not casesensitive:
        cellvalue = cellvalue.lower()
      if (entirecell and cellvalue == findtext) or (not entirecell and cellvalue.find(findtext) >= 0):
        self.spreadsheet.grid.SetGridCursor(currentrow, currentcol)
        self.spreadsheet.grid.MakeCellVisible(currentrow, currentcol)
        if replace:
          self.spreadsheet.table[currentrow][currentcol] = self.getControl('replacetext').GetValue()
        self.hide()
        return True
      
      # move one column to the right to check that cell
      currentcol += 1
      

  def cancel(self):
    '''Responds to a cancel event.'''
    return self.hide()
    



####################################################
###   Expression builder

class ExpressionBuilder(Dialog):
  def __init__(self, mainframe, expression='', validate_func=None, special_vars={}, local_tables=[]):
    '''Creates the expression builder.
       expression    => The initial expression to show to the user.
       validate_func => A function to validate the expression.  Can throw an exception to stop the dialog from returning.
                        Takes one parameter: the expression.
       special_vars  => A dictionary containing special variables (mapped to their description) if desired.
       local_tables  => Table whose column names are local to the expression (that don't need "table." in front of them.
    '''
    self.expression = None
    self.initial_expression = expression
    self.validate_func = validate_func
    self.special_vars = special_vars
    self.local_tables = local_tables
    Dialog.__init__(self, mainframe, 'ExpressionBuilder') 

  
  def init(self):
    '''Initializes the dialog'''
    # expression box
    self.insertionpos = 0
    self.getControl('expression').Bind(wx.EVT_KILL_FOCUS, self.onExpressionUnfocus)
    self.getControl('expression').SetValue(self.initial_expression)

    # tables and fields
    try:
      self.tables = self.initTables(self.getControl('tables'))
      self.getControl('tables').Bind(wx.EVT_CHOICE, self.refreshColumns)
      self.getControl('fields').Bind(wx.EVT_LISTBOX_DCLICK, self.onField) 
      self.refreshColumns()
    except AssertionError: # raised when no tables exist
      self.getControl('tables').Enable(False)
      self.getControl('fields').Enable(False)
  
    # add the events to the button panel
    for child in self.getControl('buttonPanel').GetChildren():
      child.Bind(wx.EVT_LEFT_DOWN, self.onButtonPanel)
      
    # populate the functions
    self.func_categories = Utils.get_documentation()
    categories = self.getControl('categories')
    categories.Clear()
    cats = self.func_categories.keys()
    if self.special_vars:
      cats.append('Special')
    cats.sort(lambda a,b: cmp(a.lower(), b.lower()))
    categories.AppendItems(cats)
    categories.SetSelection(self.special_vars and cats.index('Special') or cats.index('Core'))
    self.onCategoryChange()
    self.getControl('categories').Bind(wx.EVT_LISTBOX, self.onCategoryChange)
    self.getControl('functions').Bind(wx.EVT_LISTBOX, self.onFunctionChange)
    self.getControl('functions').Bind(wx.EVT_LISTBOX_DCLICK, self.onFunction)
    
    # validation
    self.getControl('validateButton').Bind(wx.EVT_BUTTON, self.onValidate)
    
    self.getControl('expression').SetFocus()
    

  def ok(self):
    '''Default OK method.  Subclasses should override this to provide behavior when the OK button is pressed'''
    if not self.onValidate():
      return
    self.expression = self.getControl('expression').GetValue()  # save the expression since the dlg variable is destroyed upon OK
    Dialog.ok(self)
    
  
  def onValidate(self, event=None):
    '''Validates the expression'''
    self.getControl('validateText').SetLabel('')
    text = self.getControl('expression').GetValue()
    # check the syntax first
    try:
      compile(text, '<script>', 'exec')
    except Exception, e:
      offset = e.offset or 1  # sometimes these are None
      lineno = e.lineno or 1
      wx.MessageBox(lang('You have a syntax error on line') + ' ' + str(lineno) + ', ' + lang('column') + ' ' + str(offset) + '.', lang('Syntax Error'))
      return False
    # if we have a validate func, check it
    try:
      if self.validate_func:
        validate_func(text)
    except Exception, e:
      wx.MessageBox(str(e), lang('Error'))
      return False
    self.getControl('validateText').SetLabel(lang('Your expression appears to be valid.'))
    return True
    
    
  def onExpressionUnfocus(self, event=None):
    '''Called when the expression box is unfocused'''
    self.insertionpos = self.getControl('expression').GetInsertionPoint()
    

  def refreshColumns(self, event=None):
    '''Refreshes the columns list for the current table'''
    tablename = self.tables[self.getControl('tables').GetSelection()]
    table = self.mainframe.shell.getVariable(tablename)
    cols = self.getControl('fields')
    cols.Clear()
    cols.AppendItems(table.get_column_names())
    cols.SetSelection(0)
    
  
  def onButtonPanel(self, event):
    '''Responds to a mouse click on the button panel'''
    self.getControl('expression').WriteText(event.GetEventObject().GetLabel())


  def onField(self, event):
    '''Responds to a double click on a field'''
    self.getControl('expression').SetFocus()
    self.getControl('expression').SetInsertionPoint(self.insertionpos)
    tablename = self.getControl('tables').GetStringSelection()
    fieldname = self.getControl('fields').GetStringSelection()
    if tablename in self.local_tables:
      self.getControl('expression').WriteText(fieldname)
    else:
      self.getControl('expression').WriteText('%s["%s"]' % (tablename, fieldname))
    
    
  def onCategoryChange(self, event=None):
    '''Changes the function list when the category changes'''
    categories = self.getControl('categories')
    catname = categories.GetString(categories.GetSelection())
    if catname == 'Special':
      funcs = self.special_vars.keys()
    else:
      funcs = self.func_categories[catname].keys()
    funcs.sort(lambda a,b: cmp(a.lower(), b.lower()))
    functions = self.getControl('functions')
    functions.Clear()
    functions.AppendItems(funcs)
    functions.SetSelection(0)
    self.onFunctionChange()


  def onFunctionChange(self, event=None):
    '''Changes the function information when the function changes'''
    categories = self.getControl('categories')
    catname = categories.GetString(categories.GetSelection())
    functions = self.getControl('functions')
    funcname = functions.GetString(functions.GetSelection())
    self.dlg.Freeze()
    self.getControl('functiondoc').SetEditable(True)  
    try:
      notebook = self.getControl('functionnotebook')
      while notebook.GetPageCount() > 1:
        notebook.DeletePage(1)
      if catname == 'Special':
        self.getControl('functiondoc').SetValue(self.special_vars[funcname])
      else:
        doc = self.func_categories[catname][funcname]
        self.getControl('functiondoc').SetValue('\n\n'.join([doc.getSignature()] + doc.desc))
        for i, example in enumerate(doc.examples):
          editor = wx.py.editwindow.EditWindow(notebook)
          Editor.updateStyles(editor)
          editor.AddText(example)
          editor.SetReadOnly(True)
          pagename = lang('Example') + (len(doc.examples) > 1 and (' ' + str(i+1)) or '')
          notebook.AddPage(editor, pagename)
    finally:
      self.getControl('functiondoc').SetEditable(False)  
      self.dlg.Thaw()  

    
  def onFunction(self, event=None):
    '''Responds to a double click on a function'''
    categories = self.getControl('categories')
    catname = categories.GetString(categories.GetSelection())
    functions = self.getControl('functions')
    funcname = functions.GetString(functions.GetSelection())
    if catname == 'Special':
      text = funcname
      posadjust = len(text)
    else:
      doc = self.func_categories[catname][funcname]
      s = []
      s.append(doc.getFullName())
      parms = []
      for p in doc.params:
        if p.required and p.type in ('str', 'unicode'):
          parms.append('""')
        elif p.required and p.type in ('list', 'tuple'):
          parms.append('[]')
        elif p.required:
          parms.append('')
      s.append('(' + (', '.join(parms)) + ')')
      text = ''.join(s)
      posadjust = len(doc.getFullName()) + 1
    self.getControl('expression').SetFocus()
    self.getControl('expression').SetInsertionPoint(self.insertionpos)
    self.getControl('expression').WriteText(text)
    self.insertionpos += posadjust
    self.getControl('expression').SetInsertionPoint(self.insertionpos)
    
    
    
###################################################
###   Hints Box that shows when starting Picalo

class Hints(Dialog):
  def __init__(self, mainframe):
    Dialog.__init__(self, mainframe, 'Hints') 
    # get the hints
    fp = open(Utils.getResourcePath('hints.txt'))
    self.hints = fp.readlines()
    fp.close()
    
  
  def init(self):
    '''Initializes the dialog'''
    # update the hint text
    Preferences.set('currenthintindex', Preferences.get('currenthintindex', -1) + 1)
    self.updateHintText()
    # set up events
    self.getControl('previous').Bind(wx.EVT_BUTTON, self.onPreviousButton)
    self.getControl('next').Bind(wx.EVT_BUTTON, self.onNextButton)
    self.getControl('showatstartup').Bind(wx.EVT_CHECKBOX, self.onShowAtStartup)
    self.getControl('showatstartup').SetValue(Preferences.get('showhintsatstartup') != False)
    
    
  def updateHintText(self):
    '''Sets the hint text based on the current hint index'''
    # wrap if the hint index is less than 0 or greater than number of hints
    if Preferences.get('currenthintindex', 0) < 0:
      Preferences.set('currenthintindex', len(self.hints) - 1)
    if Preferences.get('currenthintindex', 0) >= len(self.hints):
      Preferences.set('currenthintindex', 0)
    self.getControl('hinttext').SetValue(self.hints[Preferences.get('currenthintindex', 0)].strip())
    
  
  def onPreviousButton(self, event):
    '''Called when the previous button is pressed'''
    Preferences.set('currenthintindex', Preferences.get('currenthintindex', 0) - 1)
    self.updateHintText()
    
  
  def onNextButton(self, event):
    '''Called when the next button is pressed'''
    Preferences.set('currenthintindex', Preferences.get('currenthintindex', 0) + 1)
    self.updateHintText()
    
    
  def onShowAtStartup(self, event):
    '''Called when the show at startup checkbox is changed'''
    if event.IsChecked():
      Preferences.put('showhintsatstartup', True)
    else:
      Preferences.put('showhintsatstartup', False)
    
    
    
############################################
###   ValueEditor dialog

class ValueEditor(Dialog):
  def __init__(self, mainframe, value):
    Dialog.__init__(self, mainframe, 'ValueEditor')
    self.value = value
    
  
  def init(self):
    '''Initializes the dialog'''
    self.getControl('value').SetValue(unicode(self.value))
    
  
  def ok(self, event=None):
    '''Responds to the ok button'''
    self.value = self.getControl('value').GetValue()
    self.close(wx.OK)
    
    
    
############################################
###   FormatEditor dialog

class FormatEditor(Dialog):
  def __init__(self, mainframe, typ, format):
    Dialog.__init__(self, mainframe, 'FormatEditor')
    self.typ = typ
    self.format = format
    
  
  def init(self):
    '''Initializes the dialog'''
    self.getControl('format').Clear()
    if self.typ in ('DateTime', 'Date'):
      self.getControl('helptext').SetValue('''The special characters used for Date and DateTime fields are as follows:

- %a Locale's abbreviated weekday name.  
- %A Locale's full weekday name.
- %b Locale's abbreviated month name.  
- %B Locale's full month name.  
- %c Locale's appropriate date and time representation.  
- %d Day of the month as a decimal number [01,31].  
- %H Hour (24-hour clock) as a decimal number [00,23].  
- %I Hour (12-hour clock) as a decimal number [01,12].  
- %j Day of the year as a decimal number [001,366].  
- %m Month as a decimal number [01,12].  
- %M Minute as a decimal number [00,59].  
- %p Locale's equivalent of either AM or PM. (1)
- %S Second as a decimal number [00,61]. (2)
- %U Week number of the year (Sunday as the first day of the week) as a decimal number [00,53]. All days in a new year preceding the first Sunday are considered to be in week 0. (3)
- %w Weekday as a decimal number [0(Sunday),6].  
- %W Week number of the year (Monday as the first day of the week) as a decimal number [00,53]. All days in a new year preceding the first Monday are considered to be in week 0. (3)
- %x Locale's appropriate date representation.  
- %X Locale's appropriate time representation.  
- %y Year without century as a decimal number [00,99].  
- %Y Year with century as a decimal number.  
- %Z Time zone name (no characters if no time zone exists).  
- %% A literal '%' character.

The following are examples of date and time formats:

- %a, %d %b %Y (Thu, 28 Jun 2001)
- %B %d, %Y (June 28, 2001)
- %Y-%m-%d (2015-12-30)
- %Y/%m/%d (2015/12/30)
- %a, %d %b %Y %H:%M:%S (Thu, 28 Jun 2001 14:17:15)
- %B %d, %Y %H:%M:%S (June 28, 2001 14:17:15)
- %Y-%m-%d %H:%M:%S (2015-12-30 14:17:15)
- %Y/%m/%d %H:%M:%S (2015/12/30 14:17:15)
      ''')
      self.getControl('testtext').SetValue('2015-05-30 13:15:22')
      self.getControl('format').Append('%a, %d %b %Y (Thu, 28 Jun 2001)')
      self.getControl('format').Append('%B %d, %Y (June 28, 2001)')
      self.getControl('format').Append('%Y-%m-%d (2015-12-30)')
      self.getControl('format').Append('%Y/%m/%d (2015/12/30)')
      self.getControl('format').Append('%a, %d %b %Y %H:%M:%S (Thu, 28 Jun 2001 14:17:15)')
      self.getControl('format').Append('%B %d, %Y %H:%M:%S (June 28, 2001 14:17:15)')
      self.getControl('format').Append('%Y-%m-%d %H:%M:%S (2015-12-30 14:17:15)')
      self.getControl('format').Append('%Y/%m/%d %H:%M:%S (2015/12/30 14:17:15)')
    else:
      self.getControl('helptext').SetValue(''' The special characters used for all number fields (int, long, float, number, etc.) are as follows:
       
- Prefix the format with any character(s) (like $) to add to the front of the number.
- End the format with a zero (0) to round to the nearest whole number.
- Use a period (.) to denote decimal portions of the number.
- Use a pound (#) to denote a regular number.
- Use a comma (,) to denote a thousands separator (use with # signs to place it every three numbers)
- Use a percent (%) to denote the value should be displayed as a percent (multiplied by 100 to when displayed and divide by 100 when parsing input and the number ends with a %)
- Use the letter E+ followed by zeros to denote scientific notation. 

The following are examples of number formats:

- 0 (round to the nearest whole number;  10.99 is displayed as 11; 12.3 is displayed as 12)
- 0.00 (rounds to the nearest hundredth; 10.99 is formatted as 10.99; 12.309 is formatted as 12.31; 13 is formatted as 13.00)
- $0.00 (rounds to the nearest hundredth and adds a dollar sign to the front of the number; you can also use any other character, such as the euro glyph)
- #,### (formats thousands with a comma;  1234.56 is formatted as 1,234.56)
- #,##0 (formats thousands with a comma and rounds to the nearest whole number; 1234.56 is formatted as 1,235)
- #,##0.000 (formats thousands with a comma and rounds to the nearest thousandth; 1234.56 is formatted as 1,234.560)
- 0% (rounds to the nearest whole number, multiplies by 100 (for display only), and adds a percent sign)
- 0.00% (rounds to the nearest hundredth, multiplies by 100 (for display only), and adds a percent sign)
- #E+000 (shows the number in scientific notation to the given number of decimal places)

As of right now, the following formatting options are not available:
- Numbers with a comma for the decimal separator and a period as the thousands separator.
- Parentheses for negative numbers.
      ''')
      self.getControl('format').Append('0 (round to the nearest whole number;  10.99 is displayed as 11; 12.3 is displayed as 12)')
      self.getControl('format').Append('0.00 (rounds to the nearest hundredth; 10.99 is formatted as 10.99; 12.309 is formatted as 12.31; 13 is formatted as 13.00)')
      self.getControl('format').Append('$0.00 (rounds to the nearest hundredth and adds a dollar sign to the front of the number; you can also use any other character, such as the euro glyph)')
      self.getControl('format').Append('#,### (formats thousands with a comma;  1234.56 is formatted as 1,234.56)')
      self.getControl('format').Append('#,##0 (formats thousands with a comma and rounds to the nearest whole number; 1234.56 is formatted as 1,235)')
      self.getControl('format').Append('#,##0.000 (formats thousands with a comma and rounds to the nearest thousandth; 1234.56 is formatted as 1,234.560)')
      self.getControl('format').Append('0% (rounds to the nearest whole number, multiplies by 100 (for display only), and adds a percent sign)')
      self.getControl('format').Append('0.00% (rounds to the nearest hundredth, multiplies by 100 (for display only), and adds a percent sign)')
      self.getControl('format').Append('#E+000 (shows the number in scientific notation to the given number of decimal places)')
      self.getControl('testtext').SetValue('3.1415')
    self.getControl('format').SetValue(self.format)
    self.getControl('format').Bind(wx.EVT_COMBOBOX, self.onFormatSelect)
    self.getControl('format').Bind(wx.EVT_TEXT, self.onFormatChange)
    self.onFormatChange()
    
    
  def onFormatSelect(self, event):
    '''Responds to a number format combo box change'''
    idx = self.getControl('format').GetValue().find(' (')
    if idx > 0:
      self.getControl('format').SetValue(self.getControl('format').GetValue()[:idx])
    self.onFormatChange()
        
      
  def onFormatChange(self, event=None):
    '''Responds to a change in the number format box'''
    if not self.getControl('testtext').GetValue():
      self.getControl('testresult').SetValue("")
      return
    if self.typ in ('Date', 'DateTime'): 
      value = DateTime(self.getControl('testtext').GetValue())
    else:
      value = number(self.getControl('testtext').GetValue())
    try:
      formatted = format_value_from_type(value, eval(self.typ), self.getControl('format').GetValue())
    except Exception, e:
      formatted = str(e)
    self.getControl('testresult').SetValue(formatted)
        
  
  def ok(self, event=None):
    '''Responds to the ok button'''
    self.format = self.getControl('format').GetValue()
    self.close(wx.OK)
    
    
    