####################################################################################
#                                                                                  #
# Copyright (c) 2006 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This file is part of Picalo.                                                     #
#                                                                                  #
# Picalo is free software; you can redistribute it and/or modify                   #
# it under the terms of the GNU General Public License as published by             #
# the Free Software Foundation; either version 2 of the License, or                # 
# (at your option) any later version.                                              #
#                                                                                  #
# Picalo is distributed in the hope that it will be useful,                        #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                   #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    #
# GNU General Public License for more details.                                     #
#                                                                                  #
# You should have received a copy of the GNU General Public License                #
# along with Foobar; if not, write to the Free Software                            #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA        #
#                                                                                  #
####################################################################################
#                                                                                  #
#  This file is globally imported into Picalo.  See picalo/__init__.py.            # 
#  When you write "from picalo import *", these functions all get imported.        #
#                                                                                  #
####################################################################################

from Global import check_valid_table
from Error import error
import types

####################################################################
###   Represents a row of data



class Record(list):
  '''An individual record of a Table. Individual fields in a record can be accessed 
     via column number or name.  In the table above, this is done with the following:
       >>> rec1 = mytable3[0]
       >>> print rec1[1]
       'Bart'
       >>> print rec1['Name'] 
       'Bart'
  '''
  def __init__(self, table):
    '''Called by Table's append or insert method.  Do not create Records directly.
       table         => The Picalo table that owns this record
    '''
    self.__dict__['_table'] = table  # can't use regular syntax because we override __setitem__ below
    if table != None:  # when picalo first loads, we create some records directly (without a table)  See Global.py
      for i in range(len(table.columns)):
        list.append(self, None)


  def __repr__(self):
    '''For debugging'''
    return '<Record instance: ' + list.__repr__(self) + '>'
    
    
  def __str__(self):
    '''For debugging'''
    return '<Record instance: ' + list.__repr__(self) + '>'
    
    
  def __iter__(self):
    '''Returns an iterator to this record (using the table's column order)'''
    return RecordIterator(self)
    
    
  def __contains__(self, item):
    '''Returns whether the given item is in the record'''
    for value in self:  # go through my cell values
      try:
        if item in value:
          return True
      except TypeError:  # occurs when we can't do the "in" keyword in the given value
        if item == value:
          return True
    return False
    
    
  ##### CONTAINER METHODS ####
  
  
  def __len__(self):
    '''Returns the length of this record'''
    return len(self._table.columns)
    
  
  def has_key(self, key):
    '''Returns whether the record has the given column name'''
    return self._table.columns_map.has_key(key)


  def __getattr__(self, col):
    '''Retrieves the value of a field in the record.  This method allows
       record.colname type of access to values.
       
       @param col: The column name or index
       @type  col: str
       @return:    The value in the given field
       @rtype:     returns
    '''
    if self.__dict__.has_key(col):
      return self.__dict__[col]
    return self.__get_value__(col, [])
    

  def __getitem__(self, col):
    '''Retrieves the value of a field in the record.  This method allows
       record['colname'] type of access to values.
       
       @param col: The column name or index
       @type  col: str
       @return:    The value in the given field
       @rtype:     returns
    '''
    return self.__get_value__(col, [])
    
    
  def __get_value__(self, col, expression_backtrack):
    '''Internal method to retrieve the value of a cell.
       col                  => The column name or index
       expression_backtrack => A list of expression ids that have been run so far, to catch circular formulas
    '''
    # IMPORTANT: This function is called as much as any function in
    # Picalo.  It must be coded in the most efficient way possible.
    # (this would be a good candidate to rewrite in C)
    
    # if a slice, recursively call me to get the items of the slice
    if isinstance(col, types.SliceType):
      return [ self.__getitem__(col.name) for col in self._table.columns[col] ]
    
    else:  # a regular single request
      # for speed, I don't call Table.deref_column, but I include the code directly here
      if isinstance(col, int): # convert to the column name
        col = self._table.columns[col].name
        
      # get the column definition
      assert self._table.columns_map.has_key(col), 'This table has no column named ' + str(col)
      idx = self._table.columns_map[col]
      coldef = self._table.columns[idx]
      
      # is it a calculated column?
      if coldef.expression:
        return coldef.expression.evaluate([{'record': self}, self], expression_backtrack)
        
      # if not a calculated column, just return the value (or None if the value for this column hasn't ever been set)
      return list.__getitem__(self, idx)
      
      
  def __setattr__(self, col, value):
    '''Sets the value of a field in the record.
      
       @param col:   The column name or index
       @type  col:   str/int
       @param value: The value to save in the field
       @type  value: value
    '''
    if self.__dict__.has_key(col):
      self.__dict__[col] = value
    else:
      self.__setitem__(col, value)
    

  def __setitem__(self, col, value):
    '''Sets the value of a field in the record.
      
       @param col:   The column name or index
       @type  col:   str/int
       @param value: The value to save in the field
       @type  value: value
    '''
    assert not self._table.is_readonly(), 'This table is set as read-only and cannot be modified.'
    
    # if a slice, repeatedly call this method to set each member
    if isinstance(col, types.SliceType):
      for i in range(self._table.deref_column(col.start), self._table.deref_column(col.stop)):
        valueidx = i - self._table.deref_column(col.start)
        if valueidx >= 0 and valueidx < len(value):
          self.__setitem__(i, value[valueidx])
      return
    
    # coerce the value into the right type for this column
    idx = self._table.deref_column(col)
    col_def = self._table.columns[idx]
    if value != None and not isinstance(value, col_def.column_type):
      try:
        value = col_def.parse_value(value)
      except Exception, e:
        value = error(e)

    # invalidate the indices now that we've changed data in this table
    self._table._invalidate_indexes()
    
    # set the value 
    list.__setitem__(self, idx, value)
    
    # notify listeners that we've had changes
    self._table._notify_listeners()
    
  
  def __setslice__(self, i, j, values):
    '''Sets the value of several items in this record.  Use slice notation:
       table[rec][0:2] = [ 1, 2 ]   # sets first two items
       
       @param i:  The starting index (inclusive)
       @type  i:  int
       @param j:  The ending index (exclusive)
       @type  j:  int
       @param values: A sequence of values to set
       @type  values: list or tuple
    '''
    # note: although setslice is deprecated, it is still in our superclass, so we have to override it
    self.__setitem__(slice(i,j), values)
   
   
###############################
###   Iterator for a record
   
def RecordIterator(record):     
  '''Returns a generator object to iterate over the columns of a record'''
  index = 0
  numcols = len(record._table.columns)
  while index < numcols:
    yield record[index]
    index += 1  


