####################################################################################
#                                                                                  
# Copyright (c) 2008 Travis Kulbeth <tkulbethATgmailDOTcom>               
#                                                                                  
# This deteclet is part of the Bid Rigging detectlets library.                     
# It is open source and can be modified as needed.                                    
#                                                                                  
####################################################################################
# UPDATES:
#
# April 2008  First version of the detectlet
#
####################################################################################
#
# IDEAS/QUESTIONS
#
# April 17, 2008:
# Due to a bug in Picalo at the time this detectlet was created, this project was coded to use a String comparison on what 
# would be a boolean (True/False) comparison for the winning bid.  The sample data 'WinningBid' column that should be a boolean 
# type is presently set as a string type. When reading and processing a table using boolean values, Picalo sets all values in a 
# boolean column to 'False', which results in inaccurate results.  This bug has been reported through www.picalo.org.
# 
# The following updates to this detectlet will be needed once this bug is fixed: 
# - The comparison "if winner != 'True':" in the run method should be updated to "if winner is not True:" 
#       (the correct code is commented out above the currently functioning line)
# - The example data set will need to have the 'WinningBid' column set to a boolean datatype 
#       (just uncomment the following line of code: "table.set_type('WinningBid', boolean)" )
# 
####################################################################################

DETECTLET_STANDARD = 1.0

from picalo import *                            # import the Picalo libraries
import sys, re, random, os, os.path, urllib     # import commonly-used Python libraries

wizard = '''
<wizard>
  <page>
    Select the TABLE containing project bid information:  
    
    The table must contain 
    columns for Project ID numbers, Vendor IDs, Bid Amounts, and a column to indicate whether a 
    bid was the winning bid or not (using True or False).
    
    <parameter type="Table" variable="project_bid_table"/>
  </page>
  <page>
    Select the COLUMN from the project bid table containing the project ID numbers:  

    It could be called "ProjectID", "proj_id", "project_id", "ProjectNum", or something similar.
    
    <parameter type="Column" table="project_bid_table" variable="project_id_col"/>
  </page>
  <page>
    Select the COLUMN from the project bid table containing the project bid amounts: 
    
    They could be called "bid_amount," "BidAmount," "Amount," or something similar.
    
    <parameter type="Column" table="project_bid_table" variable="bid_amount_col"/>

    Select the COLUMN from the project bid table containing the winning bid indicator (containing True or False):
    
    They could be called "is_winner," "IsWinningBid," "Won," or something similar.
    
    Winning Bid Indicator Column: <parameter type="Column" table="project_bid_table" variable="is_winner_col"/>
  </page>
</wizard>
'''


RESULTS_TEXT = '''
    The displayed table shows all of the project records for which the winning bid amount was not the lowest bid 
    for that project.  These were found by checking the winning bid indicator column as compared to the 
    bid amounts for each project.  If the lowest bid for a project does not coincide with the winning bid, as 
    indicated, the list of bids for that project are shown in a sorted order.
    
    These results may indicate where certain vendors or contractors are being favored over others in 
    the bid selection process.  One reason for this may be that representatives from a winning
    vendor are personally connected/related to someone who selected their bid. Another reason could 
    be that the bid selector has some financial kick-back or other benefit from the winning contract.  
    The process and individuals involved in this selection process should be verified and checked for 
    any fraudulent connections or activity.
'''

def run(project_bid_table, project_id_col, bid_amount_col, is_winner_col):
    '''In the bid selection process, there are often cases where a winning bid for a project (i.e., contracted labor, 
    construction, software development, etc.) is not the lowest.  Perhaps there is valid reasoning behind 
    the selection of a more expensive vendor, but in most cases, the lowest bid is sought.  
    
    When the lowest bid is not selected, it could indicate fraudulent relationships between the 
    proposing company and the chosen vendor.  The winning vendors or contractors may be favored 
    over others in the bid selection process.  One reason for this may be that representatives from a winning
    vendor are personally connected/related to someone within the proposing company. Another case may suggest 
    the presence of financial kick-backs or other benefit to the proposing company's representative.

    This detectlet searches for all winning bids that were not the lowest offered bid for a project.

    The detectlet goes through the following process:
    - Breaks down the project list by unique project ID
    - Locates the lowest bid for each project
    - Determines if the lowest bid amount was the winning bid
    - Displays all bid records for the project whose winning bid was not the lowest.

    The process and individuals involved in this selection process should be verified and checked for 
    any fraudulent connections or activity.
    '''
    #break the project bid table down into a Table List by project
    stratified_list = Grouping.stratify_by_value(project_bid_table, project_id_col)
    
    #define results table
    results_table = Table(project_bid_table.get_column_names())
    results_table.set_type(bid_amount_col, currency)
    
    #sort the list to find the lowest bid, then determine if that bid was the winner
    for project in stratified_list:
        Simple.sort(project, True, bid_amount_col)
        #if the lowest bid (the first index) is not the winner
        iswinner = project[is_winner_col][0]
        
        # if winner is not True:
        if iswinner != 'True':
            results_table.extend(project)   
    
    return results_table, RESULTS_TEXT
    
    
def example_input():
    import StringIO  # to emulate a file for load_csv
    table = load_csv(StringIO.StringIO(csvdata))
    table.set_type('BidAmount', currency)
    # table.set_type('IsWinningBid', boolean)
    return table

csvdata = '''\
"ProjectID","VendorID","BidAmount","IsWinningBid"
1212000,10235012,3393290,True
1212000,10235008,4161235,False
1212000,10235009,5756513,False
1212000,10235007,8376753,False
1212000,10235010,9635871,False
1212000,10235011,9947921,False
1211095,10235097,2853447,False
1211095,10235098,3266269,True
1211095,10235095,3327188,False
1211095,10235099,6591213,False
1211095,10235096,9416901,False
1211087,10235087,446486,False
1211087,10235094,1069312,False
1211087,10235089,2817335,False
1211087,10235090,4409451,False
1211087,10235092,5194875,False
1211087,10235088,6931910,False
1211087,10235093,9271576,True
1211087,10235091,9301648,False
1211076,10235078,1567232,True
1211076,10235082,2644816,False
1211076,10235083,2946890,False
1211076,10235079,4561052,False
1211076,10235080,5612758,False
1211076,10235077,6023072,False
1211076,10235076,6350300,False
1211076,10235084,7365989,False
1211076,10235085,8345625,False
1211076,10235086,8359528,False
1211076,10235081,9374039,False
1211057,10235058,576248,False
1211057,10235074,1121923,False
1211057,10235075,1334980,False
1211057,10235062,1820275,False
1211057,10235057,2285016,False
1211057,10235063,3252507,False
1211057,10235071,5401285,False
1211057,10235067,5578467,False
1211057,10235065,5723538,False
1211057,10235061,6166798,False
1211057,10235069,6320598,False
1211057,10235064,6464670,False
1211057,10235073,7193095,False
1211057,10235059,7346786,False
1211057,10235072,7776194,False
1211057,10235066,8957829,False
1211057,10235070,8966828,True
1211057,10235068,9132005,False
1211057,10235060,9321959,False
1211049,10235049,906935,False
1211049,10235050,3548993,True
1211049,10235056,4808848,False
1211049,10235055,4843618,False
1211049,10235054,6449320,False
1211049,10235053,7338434,False
1211049,10235051,7477630,False
1211049,10235052,7960046,False
1211048,10235045,421606,True
1211048,10235044,1838838,False
1211048,10235046,4454021,False
1211048,10235048,6712473,False
1211048,10235047,7107235,False
1211034,10235041,177559,False
1211034,10235042,814662,True
1211034,10235038,2528561,False
1211034,10235037,4992260,False
1211034,10235035,6034431,False
1211034,10235040,6228479,False
1211034,10235036,7127276,False
1211034,10235034,7267305,False
1211034,10235043,8452286,False
1211034,10235039,8994388,False
1211019,10235024,1189496,True
1211019,10235022,1786586,False
1211019,10235027,1951665,False
1211019,10235020,3310704,False
1211019,10235025,3421573,False
1211019,10235033,4067639,False
1211019,10235031,4678953,False
1211019,10235021,5350659,False
1211019,10235032,6345878,False
1211019,10235023,7803384,False
1211019,10235028,8229633,False
1211019,10235019,8699159,False
1211019,10235029,8868249,False
1211019,10235026,9337801,False
1211019,10235030,9736244,False
1211013,10235013,1174874,False
1211013,10235018,1448245,False
1211013,10235016,5090403,False
1211013,10235015,5478895,False
1211013,10235017,8377930,False
1211013,10235014,9609407,True
1211000,10235001,1124833,True
1211000,10235000,3094369,False
1211000,10235002,4500841,False
1211000,10235006,5506454,False
1211000,10235005,5967093,False
1211000,10235003,7017835,False
1211000,10235004,8565550,False
'''
    
