####################################################################################
#
# Name: Westin Hatch
# Email: <westinATbyuDOTedu>
# 
# 
####################################################################################
# UPDATES
# 17 April 2008   First edition of this detectlet
#
####################################################################################
# STATUS: PRODUCTION
#
#   IDEAS/QUESTIONS/NOTES:
#
#
####################################################################################
from picalo import *

# The wizard asks for two variables from two tables which are passed into the run
# method. Those variables are:
#
#    1) The table where the bid data is stored
#    2) The column that contains the bid ID numbers
#    3) The column that contains the contract ID numbers
#    4) The column that contains the total price amount of the bid

DETECTLET_STANDARD = 1.0

wizard = '''
<wizard>
  <page>
    Select the TABLE containing the bidding data:

    It should look similar to the example input data (see
    the previous page), and it should have columns for
    the contracts and amount.

    <parameter type="Table" variable="table"/>
  </page>
  <page>
    Select the COLUMN in the bid table containing bid numbers:
    
    (Example: BidID, BidNumber, bid_number)
    <parameter type="Column" table="table" variable="bidcol"/>
  </page>
  <page>
    Select the COLUMN in the bid table containing contract numbers:

    This is the number that identifies the contract with which 
    the bid was done.
    
    (Example:ContractID, ContractNumber, contract_number)
    <parameter type="Column" table="table" variable="contractcol"/>
  </page>
  <page>
    Select the COLUMN in the bid table containing the bid prices:  
    
    This is the total price of the entire bid.
    
    (Example:total_amount, bid_price)
    <parameter type="Column" table="table" variable="amountcol"/>
  </page>
</wizard>
'''


# This text will be displayed in a window of its own once the results
# are returned.
RESULTS_TEXT = '''\
The resulting table from this detectlet shows all contracts whose lowest bid 
was ninety-nine percent or more than the amount of the second lowest bid. This means
that the range between the winning bid and the second lowest bid was extremely 
small and perhaps involved fraud.

The difference column displays the difference in the two bids' amounts. This is 
not a certain indicator of fraud, but the smaller the difference the more likely 
that the lowest bidder knew the amounts of the other bidders making it possible 
to just barely outbid all of the other bidders.
'''


# This is the main function where the algorithm and logic for the program
# is embedded.
def run(table, bidcol, contractcol, amountcol):
  '''In the bidding process, bidders are blind to the bid amounts put forth 
  by other bidders for a particular contract. This is done so that the bid 
  process can result in truthful bids.
  
  Once all bids are received, they are put through a standards test to make sure 
  they measure up to the required strandards for the contract. All bids that pass 
  then move on to a financial-selection round in which the bid with the lowest 
  price is selected.
  
  A possible fraud is to find out the other prices and then bid just a hair 
  under the lowest one. Then, as long as your bid passes the technical test, it will 
  be chosen.
  
  This detectlet will check all the lowest bid prices of every contract for any 
  prices that are within one percent of each other. The contract number with the 
  bid numbers, and bid amounts that fit this criteria will be flagged and 
  displayed.
  '''
  # validate the data
  assert bidcol != contractcol != amountcol, 'The bid number, contract number, and amount columns must be different.  Please ensure you haven\'t selected the same column for two of these items.'
  
  # run the analysis
  # Create the table where the results will be stored
  results = Table([
    ( 'ContractNumber', unicode ),
    ( 'LowestBid', unicode ),
    ( 'LowestAmount', unicode ),
    ( 'SecondLowestBid', unicode ),
    ( 'SecondLowestAmount', unicode ),
    ( 'Difference', unicode),
  ])
  # Instantiate the variables to be used in the code
  lowest = 0.0
  low = 0.0
  lowestBid = ''
  lowBid = ''
  counter = 0
  # Stratify by contract so that only one contract is evaluated at a time
  strat = Grouping.stratify_by_value(table, contractcol)
  # Not each contract of the stratified table is eavaluated in a for loop
  for tract in strat:
    counter = counter + 1
    # Shows the progress to the user
    show_progress('Analyzing (' + str(counter) + ' of ' + str(len(strat)) + ')...', float(counter) / len(strat))
    # Now each record of the contract is evaluated
    for rec in tract:
      # Store the records amount and bid number
      tempAm = rec[amountcol]
      tempBid = rec[bidcol]
      # If it is the first record it will go here because the lowest variable has 
      # not been set yet
      if lowest == 0.0:          
        lowest = tempAm
        lowestBid = tempBid
      # If the lowest variable has been set but the record's amount is still lower
      # than the stored lowest variable, then the lowest changes to the second lowest
      # and the record's amount becomes the lowest
      elif lowest > tempAm:
        low = lowest
        lowest = tempAm
        lowBid = lowestBid
        lowestBid = tempBid
      # If the low variable hasn't been set yet but the lowest has and this record isn't
      # smaller than it, then the record's amount is stored as the second lowest amount
      elif low == 0.0:
        low = tempAm
        lowBid = tempBid
      # If the record's amount is bigger than lowest but smaller than the second lowest,
      # it is stored as the second lowest
      elif low > tempAm:
        low = tempAm
        lowBid = tempBid
    # A percentage is equated between the lowest and second lowest bids for the contract
    temp = lowest/low
    # If the percentage is above 99%, then the bids are added to the results table
    if temp >= 0.99:
      # This temp is the difference between the bid prices and is added as the last
      # column of the results table
      temp = low - lowest
      results.append(tract[0][contractcol], lowestBid, lowest, lowBid, low, temp)
    # The lowest and second lowest variables are reset for the next contract to be
    # evaluated
    lowest = 0.0
    low = 0.0
  
  # The results table and text are sent back for the user to see
  return results, RESULTS_TEXT


# This is the method for example data from which an analysis can be run
# to show how the detectlet works
def example_input():
  import StringIO  # to emulate a file for load_csv
  table = load_csv(StringIO.StringIO(csvdata))
  table.set_type('BidNumber', int)
  table.set_type('ContractNumber', int)
  table.set_type('BidAmount', currency)
  return table



# This next part is not required, but it's easier to put the example data
# directly in this file so I don't have to worry about directories.
csvdata = '''\
BidNumber,ContractNumber,BidAmount
1000973,116,459317.56
1005370,94,542759.58
1012345,26,1488658.71
1012876,92,1267949.88
1017557,17,1059809.17
1022118,20,1205563.99
1023226,114,435338.66
1032573,106,965074.02
1048215,130,1391107.28
1048615,141,675749.18
1049789,130,400064.5
1055032,131,237739.02
1055550,116,362327.05
1067292,35,597349.83
1081691,58,314888.58
1083775,28,1111511.81
1084054,10,615570.68
1089664,14,584130.26
1090976,120,523205.02
1092122,148,416568.71
1092634,60,1405066.08
1094185,82,810143.98
1103575,112,1323042.91
1104992,81,159725.05
1106483,58,424436.33
1109598,69,1401738.57
1113294,100,1264659.75
1130252,109,556785.7
1130529,102,1260708.46
1130795,86,569820.71
1135328,120,72706.61
1137819,107,128515.31
1141177,83,1299275.48
1141571,62,827919.83
1142847,101,1239836.99
1143839,116,59168.98
1147238,88,1077375.72
1152034,96,531773.38
1152592,21,553028.9
1156640,18,510794.33
1158166,143,682115.69
1165732,55,131701.25
1170961,135,616800.52
1182811,117,1265498.75
1187715,144,368528.6
1201343,6,804799.08
1204061,38,421678.54
1209696,3,958755.4
1213858,65,108710.74
1231082,141,604055.75
1232280,57,863952.19
1233358,107,1380890.68
1234434,76,1399309.08
1243304,41,898335.52
1248622,81,343575.44
1249094,83,1261929.43
1252985,90,193341.98
1257084,45,1494231.37
1257880,141,626528.5
1264281,86,894927.96
1273732,19,197520.3
1282897,142,769773.89
1283997,109,737169.75
1298919,149,230505.55
1305610,38,297944.09
1307466,114,1377623.5
1314615,76,233185.08
1319320,95,383914.21
1319835,26,949586.78
1328094,40,759616.62
1328777,11,21529.84
1334858,67,566165.56
1338672,59,405840.76
1340621,3,51608.26
1345086,122,49700.89
1347410,5,185432.4
1350825,7,20601.24
1351902,106,632655.57
1351922,124,1214505.67
1352047,24,787242.76
1355947,70,496507.23
1357910,32,170160.81
1359817,54,1105065.43
1360407,123,440836.39
1372150,12,1458402.33
1374794,139,708341.02
1375169,106,911253.38
1378296,118,1277710.08
1378924,36,441284.21
1385720,61,552689.96
1385749,117,619639.06
1398882,54,395145.15
1401726,105,1214173.57
1402363,149,360532.8
1405700,44,90885.14
1415481,51,1221870.43
1421666,24,1400876.37
1422110,86,15643.23
1424717,138,316891.07
1426198,122,348475.26
1427312,80,299410.51
1430848,127,1183015.45
1459129,62,717340.19
1462310,112,665407.39
1462839,64,1277957.09
1465124,94,314579.3
1467686,91,581661.51
1470290,101,385573.59
1475303,73,7871.94
1490473,104,819503.11
1492335,18,286796.85
1504714,124,593448.61
1513098,83,1498794.22
1513273,26,661304.23
1519326,149,1154985.12
1525732,145,936608.33
1532039,42,1005306.76
1532437,111,416992.74
1533348,61,1287262.99
1537823,79,730155.47
1545222,53,989753.32
1545844,5,1259404.07
1556059,125,1055723.48
1559737,145,1266017.67
1571242,109,1347967.93
1578927,134,110373.49
1592131,143,417060.87
1598204,115,1122228.85
1600316,83,174861.64
1614924,94,393879.22
1617180,95,201008.16
1624459,64,1140093.95
1626766,108,333915.89
1627732,105,950569.43
1628747,118,780896.12
1639502,138,521545.47
1644953,137,328539.43
1645694,1,396501.23
1650087,45,1206112.76
1654676,79,894027.91
1656343,39,372757.89
1659165,4,775885.72
1665619,33,121279.54
1668743,79,820186.72
1675352,51,288770.3
1678375,49,1442608.83
1683308,111,577806.93
1696480,23,264158.94
1696815,14,764186.51
1701500,140,410997.94
1703857,76,936755.98
1706730,133,1223991.08
1718351,4,1330789.59
1721091,77,826943.36
1721461,105,808228.09
1722211,30,1240037.95
1726110,36,640731.83
1726836,89,521510.97
1726878,137,830565.88
1727446,6,509729.8
1744977,23,1377911.06
1760852,80,491894.52
1768813,33,1164422.68
1775569,47,672865.43
1785142,86,1395842.31
1788380,139,982610.77
1789321,141,1017339.47
1792758,91,1048914.6
1796167,128,1358474.94
1802746,21,1477655.33
1807085,110,27587.17
1817364,25,1385352.39
1823597,6,474067.79
1829337,78,1245605.97
1831061,65,1415689.87
1831066,35,1348284.64
1833661,36,1409408.92
1833817,133,146172.06
1840094,67,100282.61
1844009,141,461145.15
1844062,42,857056.03
1846358,11,296238.06
1849554,71,484857.14
1851214,100,246455.53
1853389,19,561095.51
1855532,37,499655.24
1862375,10,1230445.54
1876226,80,214644.33
1886836,103,990348.19
1904689,92,826838.52
1910627,48,875663.71
1917307,48,1094418.93
1922927,71,1378188.44
1923431,118,556725.55
1925607,81,1097095.82
1927529,34,61564.27
1933649,77,582627.89
1935085,140,1009767.54
1937030,13,1430849.07
1947255,65,367998.06
1956875,99,324279.85
1962209,19,1449459.78
1962974,56,181995.38
1964278,42,1370734.59
1981839,70,579165.69
1982129,143,505268.71
1989617,130,669128.78
1995181,13,240971.02
1996082,120,164400.63
1997066,7,674120.89
2000723,29,1270098.98
2007679,27,343113.62
2008536,122,720921.66
2009413,132,407195.86
2015170,107,1336982.62
2023522,93,289905.28
2025938,146,802657.72
2027653,18,1216648.94
2032319,20,576260.43
2045040,87,1372850.98
2045118,111,608927.78
2045935,148,337070.63
2048256,36,1023405.19
2049373,64,830673.79
2054887,96,734206.23
2055544,77,1377135.92
2059702,10,689979.53
2063842,86,977149.53
2066153,64,538497.01
2066283,121,1395700.98
2071914,66,529133.08
2077893,145,71461.23
2080161,101,403762.59
2082061,102,345516.89
2087800,58,903341.6
2090960,143,274910.6
2093121,39,18325.97
2094490,33,1026107.71
2115157,83,1142404.3
2118994,14,1215435.92
2125449,82,1420654.6
2129670,55,92781.14
2130545,137,976379.23
2137614,88,718268.92
2138777,2,690869.12
2153183,118,350209.19
2153554,10,1454205.9
2173526,122,596925.08
2175896,45,1299148.55
2187349,67,606505.25
2188643,92,1304966.18
2195024,139,387728.86
2196729,140,993545.28
2203425,68,1397572.88
2206548,116,597029.45
2208656,28,201754.25
2216910,91,1446326.86
2220129,38,1067357.67
2220166,52,58978.76
2236230,47,794721.46
2242676,135,361213.87
2245978,34,1190651.11
2248060,87,584057.7
2252249,123,335525.84
2264583,142,1365148.41
2266580,87,986525.03
2272380,7,498310.7
2276213,45,1272652.68
2279520,57,1300554.99
2284251,133,1262549.86
2290171,120,1361391.53
2300561,114,678376.3
2313020,88,125883.09
2315426,21,719759.94
2317538,83,1355283.8
2317727,22,437271.61
2321224,42,1272305.07
2331108,26,967931.73
2331175,17,722186.31
2343241,110,578503.76
2351247,90,1434971.35
2352961,61,994949.03
2354349,120,1453920.93
2362012,14,35635.22
2376458,35,792221.48
2380360,128,1368002.47
2383204,7,1455586.66
2383807,102,995302.8
2387761,83,735994.59
2388624,26,784247.38
2398291,61,173508.27
2409814,103,120178.96
2412853,63,942953.38
2424473,117,1159648.27
2425047,117,65678.35
2426784,45,776351.72
2442203,62,664232.96
2444880,17,735808.84
2446724,47,380702.28
2449815,35,1289337.48
2451366,33,1293918.79
2456261,97,1290420.11
2466281,92,625748.23
2467274,44,1445203.59
2472569,111,96455.67
2472968,117,1026885.06
2477504,65,414288.36
2480670,39,1215855.39
2482708,148,915034.24
2483271,83,751747.22
2491158,145,967885.32
2499874,127,778011.34
2501965,44,788140.6
2505292,122,707858.72
2508905,48,950345.49
2510738,114,837141.96
2511882,2,984101.62
2514430,21,653951.46
2518046,106,8526.18
2518437,24,1134748.24
2519405,69,793465.92
2521132,4,1137902.44
2526221,122,800675.41
2526341,142,130693.6
2527119,96,1305257.39
2530867,47,1452234.71
2531740,22,1223995.52
2533073,15,572072.36
2534323,118,251614.71
2536211,122,1002714.58
2544125,143,344968.27
2546545,30,594632.44
2555784,87,536099.99
2564339,83,333891.5
2566019,25,159284.08
2570954,81,639502.62
2581618,94,571202.83
2584711,113,824647.14
2587098,28,104052.15
2592040,117,267523.98
2597745,94,30250.11
2600783,96,329921.79
2602762,45,30069.32
2604488,148,1367707.5
2607736,47,794998.44
2608458,117,704751.23
2613697,96,1471549.23
2619348,88,34261.25
2628816,117,958258.83
2631963,149,519068.67
2638605,74,792589.73
2642333,63,1000249.85
2643876,85,1316573.11
2648651,8,881326.62
2649351,122,1322708.6
2651082,19,1210061.06
2660847,46,137538.5
2672791,73,775821.04
2681553,35,851878.51
2692400,47,762641.57
2693742,148,894273.67
2699344,47,640203.09
2712622,34,1321417.56
2714116,86,863148.75
2718966,50,709927.63
2719489,5,454622.19
2722148,80,954886.05
2727000,109,166875.06
2728710,126,621804.65
2729705,68,679073.36
2736654,20,753950.86
2736909,49,253141.48
2737912,101,1056335.47
2740555,61,1048457.29
2747965,21,1436642.23
2754500,43,1265424.54
2765683,41,539617.26
2765834,75,1296661.91
2780680,32,1296028.05
2781344,22,558381.27
2782538,95,271292.79
2784497,141,448246.39
2797608,93,1149136.12
2807299,97,459369.3
2813832,59,129841.82
2814083,18,1306799.4
2818860,100,1483952.66
2820121,7,1457378.29
2821859,54,303585.67
2823676,78,1242939.63
2829676,33,233168.57
2834631,110,817367.98
2834903,41,508121.07
2838019,91,1032960.78
2853737,21,389190.02
2856901,54,1442057.18
2857787,38,1356556.72
2864402,118,1150776.59
2869213,23,580494.2
2870192,66,965458
2872788,20,1453274.27
2873548,5,1232586.35
2886577,45,483440
2888787,107,404884.41
2892865,84,1385343.71
2895126,140,590550.38
2895971,98,271068.52
2897441,32,599257.47
2897522,63,583256.75
2902292,73,1198031.34
2913875,60,837225.01
2916576,142,351621.07
2921889,62,419758.03
2929740,69,237261.48
2940654,46,1039110.86
2946337,116,442303.66
2958661,88,441995.39
2959887,110,554158.71
2965911,138,12109.91
2966703,3,107863.5
2967493,81,694743.05
2971044,131,1237166.85
2972334,103,1092642.61
2973305,10,1326687.86
2974766,118,272534.91
2975799,88,1286876.27
2976723,63,964884.96
2976906,39,210735.73
2977403,89,934674.44
2978205,137,1010754.23
2978990,54,1460885.47
2982922,66,449805.33
2987389,80,1453281.72
2993518,108,1307980.87
2999783,92,1407091.08
3001731,123,259653.34
3009196,100,546610.17
3012222,112,793667.64
3017638,120,1052091.01
3018955,93,1454676.76
3021623,118,1488631.54
3024805,9,971417.38
3029032,34,1066629.28
3029407,32,1495310.2
3033423,76,294169.95
3037934,11,391023.98
3046820,12,532107.96
3058538,96,212628.47
3058601,126,727223.7
3066787,27,1449549.52
3069561,105,151392.59
3071698,128,1245097.9
3079911,51,923354.83
3081133,147,523829.09
3084292,111,638564.34
3085212,132,919650.71
3088150,69,394974.25
3101498,79,1208758.71
3106744,101,1131325.45
3116237,31,1384455.78
3135305,29,995848.15
3135612,28,1032254.38
3136831,9,428202.49
3137059,129,927000.07
3148348,148,745381.04
3148967,71,1473159.71
3150780,103,920803.17
3151113,76,1277883.96
3153914,85,1038487.94
3169413,16,313644.99
3170657,29,1099424.4
3175849,130,1104067.08
3188361,102,923356.99
3190595,57,1319658.2
3196140,55,709810.65
3198271,83,1444270.02
3208848,69,1374269.05
3216546,76,71103.06
3218491,1,1323493.28
3220267,67,1132543.06
3221191,114,763429.98
3226053,141,1465766.85
3228351,57,687611.04
3235420,68,342367.2
3237891,92,654380.04
3238668,149,349730.67
3239917,133,139066.15
3246329,35,339215.85
3246439,10,897014.73
3247273,119,1068909.99
3249742,147,608283.07
3252731,124,437869.72
3257215,31,771971.4
3258477,108,1249293.04
3260481,37,68625.79
3268673,132,959777.55
3271754,131,385107.89
3271836,50,1300480.33
3281089,54,1345289.38
3287966,37,1428625.39
3290367,108,678634.73
3292538,38,1174862.21
3295941,46,53997.77
3298434,91,348858.2
3309252,33,315845.28
3318122,3,1063426.3
3323648,14,197794.21
3323828,84,109036.57
3324441,74,925478.53
3325473,30,125224.13
3340667,108,521214.92
3344225,61,953850.6
3344561,43,770206.17
3344912,64,35525.42
3346998,149,570708.47
3361088,20,1109136.66
3364474,68,536008.99
3377285,94,1285873.57
3378097,113,1155414.63
3378724,28,178071.38
3379697,126,411654.14
3381535,107,1424768.26
3383756,89,770833.51
3391183,40,76357.23
3394649,37,1453223.57
3395457,19,246347.69
3396153,137,743304.72
3398115,29,529830.05
3406351,27,720437.31
3412633,112,1070724.13
3418670,118,528145.69
3424191,139,1128052.75
3432943,72,1046034.23
3442848,51,1080947.08
3448496,33,1450388.65
3452016,135,321824.25
3460109,75,1193950.94
3460424,137,1216971.88
3467832,118,179311.43
3472504,34,893212.55
3482598,57,1480153.09
3482805,94,517177.79
3482953,70,1361547.19
3490300,132,513438.69
3491323,31,965013.45
3497423,59,755270.77
3498063,3,1037885.03
3510068,97,340150.09
3513934,148,423424.55
3515314,140,888999.58
3517791,130,1120661.85
3518227,44,1130769.84
3520430,128,129704.78
3528640,18,647866.66
3532249,126,1013587
3537630,131,468041.63
3538464,118,150998.82
3546972,102,956472.73
3549469,28,748534.45
3550861,107,48755.92
3566286,130,187781.41
3567002,113,1257032.6
3567670,81,329068.1
3569141,143,404605.2
3584451,63,796723.66
3594004,100,66903.04
3597777,76,1391638.02
3602208,54,1015738.14
3603357,69,1015430
3607524,104,1089227.74
3611966,103,1036543.28
3618254,136,1377518.44
3625592,147,1425564.26
3631900,54,346481.09
3632390,9,1040046.82
3632750,142,177886.06
3633361,37,1106030.08
3640646,71,700338.99
3642991,131,685577.42
3643250,86,870710.49
3644346,18,491817.58
3654444,62,475205.61
3659864,82,73729.6
3661137,54,801863.51
3662115,135,506807.91
3665006,77,117731.85
3672978,111,1341902.98
3679515,15,867408.56
3682357,84,996732.9
3692009,106,431309.28
3695471,44,572275.04
3698005,82,871946.32
3701014,76,411824.96
3701148,33,177380.21
3703118,74,1070361.22
3703325,130,613314.44
3704301,133,254498.38
3709878,32,530399.29
3712041,122,973376.31
3720527,19,228341.75
3721048,42,683137.42
3722730,125,12676.16
3734900,23,1205709.79
3736323,64,1103352.71
3736391,94,233393.43
3740963,18,312280.71
3743675,6,1085636.21
3748444,130,282994.23
3754063,93,1356221.6
3755210,76,559867.42
3757430,3,1053540.82
3762501,12,645229.45
3765910,148,511728.89
3773745,70,1274326.19
3777361,143,1140873.36
3781627,63,660280.88
3793927,32,1243636.05
3795878,60,1136112.14
3798637,137,1021544.71
3808383,21,1393939.48
3813141,27,939351.5
3814154,26,1086130.93
3816975,29,876832.97
3820904,98,350693.14
3828685,116,60848.23
3836883,29,601182.99
3839443,16,1245322.17
3848747,66,1349693.07
3850406,145,162933.22
3854827,90,877914.95
3862211,78,995699.3
3863064,37,938060.04
3863285,74,59803.48
3867036,99,16252.6
3867556,68,1227722.47
3872037,36,1292180.89
3882544,27,1193935.98
3882591,85,1282989.63
3888073,147,665807.85
3889694,8,1394723.28
3913466,62,186170.43
3938735,107,964920.54
3946492,95,294774.45
3946738,5,1151399.89
3948990,125,373249.6
3949929,50,1294097.99
3950590,123,1295681.4
3963145,64,325988.02
3969439,53,1134716.45
3979501,143,793137.17
3982269,97,26904.16
4017591,122,1332364.47
4020129,18,7938.81
4031060,126,827329.27
4032855,51,1052349.49
4036755,65,52692.82
4036843,92,1184950.55
4044235,51,1371812.71
4052631,47,490974.71
4054867,140,1089924.22
4056055,111,1453036.58
4058370,138,1315414.85
4063534,70,537971.94
4064638,96,624568.71
4065231,117,625355.29
4065555,68,211599.31
4071185,6,752963.49
4082604,86,213037
4087566,67,956386.66
4095720,55,80794.16
4096439,118,318243.16
4126039,99,201552.33
4127451,62,1228094.89
4130123,44,191189.87
4130514,44,952242.98
4133414,30,900890.65
4134647,137,972504.94
4135977,54,1075169.58
4137571,75,1249373.65
4138953,9,473345.75
4140051,80,1162419.71
4144633,142,486648.13
4146902,9,1490799.71
4160031,113,321314.59
4162756,110,158551.35
4165715,147,1124522.96
4169406,129,821328.38
4170995,9,524924.77
4174405,39,966868.09
4179824,3,910467.62
4199770,90,648914.65
4200453,5,198950.58
4203874,22,996326.72
4210531,43,751216.19
4212955,149,1256759.63
4216870,130,1387475.18
4217355,5,1379081.28
4230414,95,1151252.47
4234838,121,1430590.24
4237377,35,513288.74
4237935,8,251110.55
4248180,72,919600.52
4250954,39,596416.93
4255606,58,438627.86
4255911,106,89877.96
4260461,118,1316654.57
4261882,120,1015456.38
4263929,69,896524.29
4278197,104,798410.25
4279098,73,792917.81
4288101,42,891881.78
4293078,3,1260125.3
4298114,104,1432348.87
4301703,115,528602.08
4302818,97,1470775.31
4308044,92,1434500.56
4310619,88,606361.51
4313109,8,176814.43
4315886,19,1495197.72
4317799,107,292752.76
4319413,5,320090.13
4333540,65,1307960.05
4333762,122,679890
4342296,136,79111.59
4347458,47,347406.25
4348358,125,564213.79
4365592,64,1249931.3
4366492,131,483021.81
4369653,57,1307566.25
4370585,87,26168.52
4371536,78,85567.54
4372510,31,941541.5
4374784,93,725643.49
4381257,119,674892.5
4398947,43,842888.48
4405580,58,1107795.95
4409385,143,1454178.88
4416599,134,985469.01
4419213,72,67093.5
4421210,133,995048.2
4441717,83,716996.35
4442072,127,970378.99
4452666,134,934854.4
4454391,24,92364.03
4454952,95,494754.06
4456138,129,1170169.15
4460892,97,5061.62
4465018,18,1222163.32
4468064,90,699325.03
4468472,117,1029058.16
4469187,36,1465395.13
4471737,13,820111.55
4477982,74,1260376.41
4478960,119,962844.65
4480182,38,72017.95
4487417,138,1112132.56
4487782,105,1098775.98
4488384,125,942289.34
4489261,39,1281540.38
4491148,26,1138310.69
4492128,1,794601.12
4494815,69,663402.37
4495108,104,1156416.49
4500866,131,1291115.94
4502719,41,83162.56
4508706,48,1105221.32
4510002,70,1444451.34
4510270,130,544292.61
4511575,129,1302070.36
4511744,47,1117716.77
4515546,102,1172548.73
4515661,114,446775.4
4515897,109,1353142.95
4517195,79,1150264.19
4521320,79,1467672.08
4526410,136,1233927.55
4527591,27,331256.86
4532527,23,977057.09
4535652,45,940746.29
4536995,46,839207.21
4547701,36,541173.87
4553482,68,625231.72
4556555,96,855284.48
4560212,148,388999.87
4566904,108,271896.86
4566981,127,553218.06
4572591,127,266059.71
4579747,138,623721.34
4580192,113,1375878.21
4595574,80,499136.57
4603509,70,882968.68
4604025,123,604830.44
4605953,129,1005273.44
4615210,106,1283656.38
4616656,16,1230854.36
4621327,109,185347.14
4626083,103,384077.82
4630084,110,62901.1
4636607,39,492288.73
4638562,44,839435.13
4648355,108,866895.11
4650608,146,570121.12
4654518,106,77400.22
4676074,33,241510.41
4680455,139,512265.48
4681825,101,1368015.82
4681932,142,769025.4
4706901,93,240489.07
4715136,28,765325.11
4722619,42,369936.93
4722681,70,1499322.38
4735659,65,684149.55
4737049,135,1018744.17
4737829,59,402429.15
4746382,145,830452.43
4760377,145,1352483.5
4768099,119,1174825.2
4773014,123,575782.65
4773638,5,115751.4
4774243,62,617469.8
4775755,71,668114.39
4776187,104,1474576.85
4782015,42,1300118.8
4790928,2,1221103.31
4798451,75,1035632.97
4808374,84,30369.24
4835337,119,378908.54
4854938,53,918233.4
4859924,9,1399070.83
4866202,120,1210193.27
4877348,133,59743.32
4878889,92,354242.7
4885427,121,976019.55
4889545,10,428743.16
4890011,26,1063350.84
4892664,73,54238.35
4894596,44,1155109.58
4895854,111,46343.15
4905346,64,99950.83
4906999,38,959141.79
4907759,5,463342.5
4911367,68,934863
4914606,48,192814.49
4917800,107,337877.71
4918598,90,924055.53
4924904,38,519417.34
4927105,45,247257.35
4929600,7,1024941.32
4933758,38,838975.12
4940087,72,1201436.88
4940438,37,149063.36
4944229,10,146264.47
4948200,123,472327.41
4950290,38,425950.03
4951133,97,1226034.74
4953260,136,1149326.17
4969780,93,1213754.78
4974050,28,839539.63
4980161,18,348292.99
4985966,38,990498.23
4999159,57,211928.57
5001609,52,1124219.92
5004568,91,644852.15
5013849,17,1245945.23
5013976,33,494437.91
5015810,84,149693.6
5025283,40,1260417.26
5026752,103,337623.02
5028686,73,675884.49
5030850,65,846689.44
5036561,98,538156.14
5046695,19,1243147.09
5049766,115,728871.51
5049829,68,1029228.9
5052513,68,1199983.54
5057286,1,520727.39
5059184,117,824300.66
5068079,13,1058024.36
5070196,53,59278.25
5085681,24,652629.71
5086904,120,1400506.16
5088351,50,1302094.11
5090917,115,264552.17
5092945,5,755975.01
5097189,105,1064988.89
5101329,85,97077.02
5110573,103,672941.39
5115782,122,1317053.86
5133111,35,193228.51
5144802,122,1291921.69
5151186,31,570386.12
5152972,85,1257548.99
5156724,21,729225.17
5173078,88,760422.54
5182772,98,243083.68
5185636,123,1207602.07
5190265,141,1326158.9
5193601,68,377709.06
5193627,24,1172619.86
5193854,4,31836.07
5196617,124,626414.02
5201147,3,877779.07
5202296,40,624719.51
5206077,93,524854.44
5212078,70,1143525.69
5216501,112,289311.01
5217560,64,716459.82
5219545,79,767908.71
5220815,91,730871.58
5220924,83,617493.51
5241979,66,59105.97
5245334,25,751146.1
5246400,69,90735.22
5249168,52,1165860.54
5253858,74,84166.18
5260795,113,1030933.48
5261498,50,729463.17
5269249,69,1289536.46
5275814,71,1322975.3
5277184,93,543625.58
5277354,74,1039321.82
5281769,80,1482765.41
5292359,60,299349.69
5294419,117,368110.86
5302624,115,110685.94
5302992,107,760744.63
5303188,54,1231830.1
5305442,95,1059872.95
5306092,91,1143606.28
5312104,46,1005294.42
5313973,5,1469362.27
5322972,34,857363.12
5326530,122,914573.58
5329572,41,707824.68
5330240,34,740804.17
5335919,133,733992.64
5344476,16,1087009.56
5349481,111,648337.4
5353176,103,511531.77
5359964,18,180968.5
5368029,125,1495696.26
5379601,39,463506.28
5383580,5,433504.74
5386072,47,245592.71
5396683,41,612271.65
5410021,35,117888.57
5410379,120,1258253.19
5417417,102,1499048.55
5421790,90,1143053.92
5425005,140,934535.88
5430313,99,855833.24
5433458,126,326163.32
5458457,103,451013.45
5460313,131,1292416.17
5463045,115,875276.22
5463784,79,1307476.32
5463999,116,575836.82
5473915,110,887004.76
5476363,145,1303699.26
5479056,55,558850.83
5490360,131,260238.16
5490906,26,842610.07
5491348,39,528674.46
5493769,114,227439.18
5494157,130,1052324.26
5499827,144,101420.13
5501639,108,303393.73
5502709,73,1378938.52
5503112,123,643077.25
5503375,80,279791.82
5507609,83,710609.85
5516009,2,1333610.18
5516026,144,743853.67
5521987,63,613990.55
5539595,118,826215.19
5541706,31,569255.16
5552211,67,816918.03
5553720,146,1087213.96
5554321,139,789260.89
5559672,68,1232343.3
5572736,138,753207.24
5574074,43,924987.87
5578456,24,817965.27
5579288,116,599515.59
5581959,38,279984.23
5588445,55,737912.85
5589848,14,595653.96
5591241,130,1448468.04
5592614,19,785294.69
5595253,102,1477755.38
5596892,88,1352124.54
5616691,147,865452.62
5617192,57,364907.42
5619132,96,104160.78
5623011,38,170838.82
5626471,9,742991.84
5633596,73,1068130.82
5635429,80,568828.31
5635454,118,1226612.08
5636356,40,406470.82
5636866,110,250837.55
5655935,94,962104.99
5668357,57,1218695.37
5669497,37,764466.76
5669549,94,8253.65
5671200,3,718776.27
5678162,56,1044107.69
5683195,55,669335.84
5696646,93,1490201.91
5701815,30,18989.41
5702209,35,212543.33
5704909,143,767922.29
5713907,95,492022.73
5714650,132,452395.57
5717101,59,88978.91
5721938,47,606366.01
5726177,146,177634.43
5726247,3,379727.57
5726337,65,1438153.92
5726383,137,948813.9
5729756,28,1484268.05
5734379,108,618696.72
5734651,84,6271.67
5735509,102,135251.94
5736924,16,72014.52
5739357,11,608747.15
5741242,104,252653
5746117,82,637451.68
5746518,45,410716.39
5755850,26,223408.49
5758607,48,1166620.12
5758620,38,896127.22
5762421,144,444217.14
5765283,104,584745.1
5768179,10,1187919.41
5769747,147,1106182.05
5769815,51,662630.16
5771517,84,428571.77
5772151,127,584467.93
5772561,46,1366427.96
5777555,119,996278.47
5780784,54,1059763.31
5781511,111,1062180.84
5789833,16,570706.27
5791414,114,1030267.82
5803439,86,728520.96
5806053,62,866288.05
5806669,127,35308.41
5808409,91,81563.11
5810769,104,69290.72
5812770,72,333455.53
5838567,39,1077010.75
5844114,146,60346.94
5853857,34,984098.15
5856333,138,898272.4
5856839,27,482307.51
5859708,107,310444.67
5863425,57,624128.19
5863601,7,912744.56
5866608,58,1113333.65
5883198,127,234069.13
5890621,75,971907.93
5894842,37,87195.43
5899816,78,1174413.13
5904055,133,144393.07
5905267,16,1392299.52
5907488,97,481325.52
5909983,124,1111618.09
5922255,101,777196.2
5923377,116,1044950.26
5926833,54,974177.59
5930731,27,148052.84
5932331,40,269804.65
5935983,139,57497.87
5938557,146,1079673
5939618,67,607221.58
5939748,140,728860.53
5946606,31,985339.23
5952700,49,1442804.22
5961092,23,1467935.87
5965326,20,1022813.42
5967978,141,1291507.15
5972815,81,992113.32
5985272,69,1470754.67
5986347,136,333597
5997866,39,198213.28
5998932,15,286647.27
6003005,29,433550.83
6004494,109,425733.66
6010515,17,500642.75
6013094,74,265413.15
6017317,115,1461650.77
6024342,6,1336482
6027032,129,1035306.15
6028262,99,1089611.49
6032616,137,552953.78
6033522,134,210985.59
6043336,1,1381851.47
6043590,66,632221
6046699,115,1488221.86
6048153,108,581903.79
6052751,33,1255803.78
6053553,124,1388519.35
6057576,134,509929.46
6061778,107,983795.57
6063045,58,1276171.84
6066541,72,1468747.81
6066941,118,633744.72
6072978,39,1018433
6083936,44,1041148.22
6087737,77,1351802.99
6092874,137,21231.67
6093603,3,926742.87
6095450,107,748686.62
6097037,38,1081333.74
6101414,146,464510.78
6113068,108,1229337.63
6115592,109,1354672.62
6122597,117,1307166.03
6130877,2,179288.06
6133302,119,741097.83
6139764,38,1332786.73
6139898,135,547312.55
6147778,142,122333.95
6153452,95,697674.52
6156033,37,1072818.9
6165251,107,1032343.57
6171963,51,1343532.87
6176678,146,160207.75
6177058,102,1166998.07
6179188,47,665954.76
6181984,11,11085.17
6183572,134,1224650.39
6184489,45,829613.69
6187042,137,188897.78
6187254,126,688471.92
6196430,135,896727.32
6197060,62,1455210.39
6206807,103,1404163.23
6216117,72,483278.21
6216768,77,534593.63
6236569,101,106151.33
6239365,21,1108218.8
6239410,46,1413625.71
6243323,94,903018.11
6243686,9,873280.6
6248316,17,1212271.62
6250780,137,756317.15
6252519,57,599428.16
6253681,48,1133836.21
6259161,110,1307233.08
6261749,15,759798.59
6265116,133,920337.1
6265222,131,641843.52
6274636,116,1065547.73
6278312,134,246555.93
6284563,21,90370.69
6285105,130,741244.88
6285473,35,269005.02
6291874,112,1423899.36
6293970,76,933615.7
6295263,89,1249140.2
6303065,125,295086.67
6305668,138,885786.64
6307541,51,421464.03
6310591,92,340855.08
6311922,12,427899
6313266,124,1488213.41
6315063,25,435971.9
6338079,91,441947.22
6339317,30,703767.54
6343133,141,1211996.43
6343742,54,94639.81
6346409,77,706018.55
6347502,22,97719.11
6347934,13,1433136.11
6348298,131,416835.69
6357233,108,222654.11
6358165,11,1007125.92
6359699,148,1454102.05
6363387,30,173841.89
6367432,46,1040419.59
6372394,17,851683.8
6375385,119,1313197.17
6378163,21,1484284.86
6379097,122,861462.62
6384283,49,890886.72
6387091,71,99627.52
6388810,70,1494866.8
6391476,72,1281158.63
6403976,95,1382985.54
6411434,4,653326.72
6414864,48,261977.78
6417574,8,1307686.47
6425465,49,685407.31
6430252,53,1292870.11
6442043,18,979334.01
6444571,143,1289521.64
6445591,138,615123.9
6447369,112,425966.92
6464493,62,1026291.51
6464984,33,565073
6465054,117,804642.24
6465906,34,540357.33
6467939,91,884661.34
6474474,38,1086687.01
6480261,20,528433.25
6484472,78,1320922.21
6486101,19,384837.9
6486729,130,1244188.66
6493055,96,57288
6501880,1,105310.14
6508535,26,1473035.13
6509970,141,550229.19
6513778,120,362631.41
6515552,137,1437500.86
6519336,93,1352475.79
6520707,30,387002.77
6524914,140,106153.41
6525423,105,1287751.02
6525677,92,546441.92
6527394,111,691052.23
6528832,19,387554.75
6531373,100,9467.68
6537169,25,448612.73
6545131,57,1061804.01
6564865,123,806812.76
6569861,123,976058.72
6571572,53,973645.21
6571741,116,176319.04
6576621,35,614137.72
6576827,116,1441348.48
6582408,126,1208976.63
6584361,8,492996.95
6590329,2,1230808.72
6590640,54,287522.86
6591929,64,1181985.71
6595590,114,376206.55
6597446,5,1413610.4
6606240,132,1058524.55
6607791,137,1496952.23
6610901,117,878991.28
6612892,63,516139.89
6627143,107,386602.73
6630649,30,1135716.69
6635083,23,1406996.95
6635648,127,1097374.08
6637041,7,66106.34
6641627,66,1091788.45
6643078,82,829564.52
6644643,75,1218354.01
6647804,80,846154.98
6658066,139,713796.92
6671041,94,574381.93
6676309,26,868889.29
6678296,142,834995.21
6679130,5,421207.54
6679627,61,767620.96
6679895,62,428451.71
6684794,101,954381.66
6686254,40,1260061.25
6698946,91,1422748.19
6699023,119,462001.96
6705008,38,171492.65
6714610,46,1233295.81
6719532,25,1317268.29
6723872,74,925302.23
6724176,127,593129.15
6736988,45,110446.68
6737665,4,1419755.64
6752108,18,752359.56
6756137,27,1356099.72
6760266,102,256548.79
6761550,49,459887.14
6768826,64,1285073.9
6770152,26,1415728.01
6770441,124,225161.46
6771346,117,525222.06
6771398,23,983782.24
6778562,128,1374027.45
6781343,102,364474.96
6786722,62,703401.88
6809447,134,1044564.05
6816654,27,734520.62
6823340,23,71223.53
6823371,63,716476.95
6829131,149,522741.15
6829273,54,656412.29
6829713,71,133823.12
6831012,66,1400832.18
6836680,12,168276.55
6851077,87,160404.91
6851600,92,838656.58
6853568,39,323042.31
6856126,138,266648.78
6856677,102,87489.29
6858855,7,868091.89
6860739,142,314796.99
6872256,114,606767.29
6874765,11,231370.79
6874963,98,1450508.12
6877033,134,911822.96
6877470,128,607075.82
6880884,77,1021845.7
6882218,70,1071929.81
6882224,111,1455789.57
6886944,108,898509.76
6893000,76,1061613.67
6893946,33,549781.5
6899641,23,537234.62
6904814,126,474976.78
6905952,63,745293.57
6906946,60,1459500
6913610,88,375651.88
6927391,9,696633
6931297,94,369356.38
6933274,82,366709.09
6958693,6,1073886.43
6965088,22,1126679.91
6969985,122,149778.6
6979869,107,510988.6
6983373,34,730044.81
6983427,87,787384.04
6984976,40,594371.97
6985399,87,474983.75
6986488,59,559835.91
6988113,83,786023.27
6989274,84,862218.45
6993250,67,1275008.18
6995221,27,532836.11
7002287,90,982427.3
7014469,21,666148.2
7016933,60,212514.75
7019433,1,595571.68
7024927,2,52341.19
7028915,133,764446.6
7035579,22,878192.85
7038182,5,1176481.69
7048401,83,115451.61
7049967,42,1425209.2
7071288,12,1347198.19
7072626,82,1429367.8
7074935,97,679443.83
7078718,54,1382063.37
7083097,5,415178.1
7088190,68,1188553.97
7102497,66,422509.41
7103801,78,175513.83
7110365,119,12379.83
7114471,109,1024170.5
7115661,89,434608.44
7117758,32,1385689.91
7119820,4,845169.97
7120186,16,1279170.79
7124086,125,27954.78
7133940,122,13024.33
7142995,15,1403947.4
7147648,126,719556.6
7148419,144,60021.62
7152291,94,365992.67
7155732,88,1041935.62
7172273,20,385929.84
7177105,143,1363111.73
7177623,62,1279110.11
7185746,117,984702.9
7186016,42,1403962.93
7189062,84,548995.46
7190430,137,939854.82
7193099,145,732591.02
7195433,78,479314.34
7197370,128,333911.65
7204370,52,1066985.42
7221671,101,289315.17
7231623,9,233511.07
7244286,85,114651.83
7245141,117,142583.69
7247270,148,916278.82
7252382,92,443252.74
7257225,29,1252662.47
7266785,110,560649.02
7277217,65,1114661.22
7279505,21,741961.89
7284205,58,1048971.77
7286180,20,1212895.99
7288957,97,765889.18
7290112,111,1245946.91
7300780,23,659538.54
7302503,101,151189.75
7305637,124,972993.29
7307578,90,336528.34
7312037,67,608902.06
7312152,41,366358.7
7313451,142,1084336.65
7328755,123,1398157.22
7331324,34,982649.95
7336123,1,1029824.18
7348789,97,844880.5
7348813,31,315559.86
7353028,99,593513.06
7355758,123,443278.23
7357581,18,1401958.21
7363430,49,513950.35
7365165,126,885731.14
7370632,126,777819.8
7372098,14,61859.88
7372145,9,805255.02
7386866,6,712209.91
7391312,140,1109562.87
7410119,110,1037512.4
7410380,24,1019531.6
7416195,83,665646.6
7420329,133,1363175.72
7424544,5,932630.41
7425003,93,581844.06
7427295,135,1451714.11
7447516,81,212336.71
7449714,62,1070975.21
7457914,25,436908.93
7458490,137,540812.39
7470800,53,1133919.36
7472963,38,931422.63
7474209,14,886114.17
7477403,128,1150544.22
7479057,12,1202545.63
7487431,108,479221.14
7487613,108,48314.12
7488259,48,730516.04
7488703,15,1020909.42
7489380,5,1399638.41
7494373,25,828634.99
7495837,19,352065.01
7497376,23,1318193.19
7497616,139,555141.55
7509504,87,1206897.89
7520506,142,1091390.98
7520751,110,1376177.6
7528001,21,1017905.92
7528753,38,211400.34
7528804,104,754495.72
7534848,74,1281377.26
7559997,114,288937.77
7562493,4,1394669.71
7562895,48,1344311.32
7574949,35,247268.88
7582511,49,1246480.28
7584421,114,805094.28
7597979,29,212848.95
7601258,13,754817.44
7602401,29,325409.42
7602546,82,763403.44
7605438,5,288089.8
7606757,137,1245697.69
7611853,58,994498.63
7612320,26,1184281.63
7620086,32,1236281.54
7622628,27,1276134.86
7624405,20,1058146.7
7636810,117,403378.06
7637070,82,1178642.7
7639203,55,151800.4
7661083,73,1313363.08
7662619,24,840791.4
7666084,55,1111187.14
7667944,54,1166216.26
7667992,56,834983.38
7671801,149,1127026.67
7674432,139,933076.92
7682655,125,243562.98
7682992,144,768304.36
7694804,109,965032.22
7694895,81,1357960.77
7705407,6,583337.72
7716230,25,1045351.19
7718551,67,152108.75
7724413,64,1003016.99
7736708,109,378843.82
7736743,40,1243394.98
7740293,18,839481.52
7749219,70,331021.76
7752292,68,963682.07
7770381,21,373074.36
7771122,46,1263814.99
7771226,23,1073553.57
7779594,37,1232077.08
7806823,128,322219.97
7809468,52,374550.67
7821459,72,108622.62
7822299,121,877857.31
7822783,95,760816.4
7826135,36,545774.1
7827290,133,359699.53
7832123,60,590096.83
7834446,25,190661.27
7837258,1,1190358.07
7837691,146,312532.62
7839337,130,1258006.92
7839744,142,1397613
7840136,34,500691.28
7842229,56,21537.42
7843571,1,795755.02
7843766,38,1130210.34
7844882,39,208798.19
7854845,121,641297.75
7855209,136,1354531.04
7862886,48,65756.82
7867940,135,215131.07
7868567,51,486139.1
7876160,38,708566.26
7880072,21,706717.64
7896598,7,1370304.23
7899371,67,1303524
7902746,67,1212133.26
7906142,59,739690.07
7908576,123,1395161.88
7912240,122,1426375.64
7912501,29,838697.24
7914996,21,125779.72
7920043,17,1383174.22
7924428,75,1265971.81
7925851,37,531366.72
7925904,98,721336.91
7926762,31,1297822.85
7929647,16,825191.28
7930010,105,1185482.58
7939876,142,596072.58
7940810,16,479282.71
7953040,24,1176458.68
7954238,85,103352.42
7983441,133,493769.22
7986762,145,273795.34
7987344,26,931985.26
7999013,84,10911
8001902,10,1026283.84
8002019,124,307649.75
8024808,143,1407826.65
8035110,120,1056945.03
8039763,128,1498980.4
8042690,102,652248.31
8049201,133,645644.64
8050053,123,991267
8052361,32,1090977.53
8059794,16,1267851.78
8070712,34,1200467.52
8071261,125,428792.13
8075946,20,178748.84
8076608,58,950385.87
8077259,68,272338.17
8078094,20,859219.81
8090492,143,229660.52
8094628,45,202174.86
8096357,26,166472.54
8101244,32,265960.61
8108457,57,1087724.26
8113182,84,609476.7
8114283,121,682062.25
8114653,108,623123.56
8115788,146,1140591.52
8117871,24,451935.02
8118040,48,1027009.49
8120094,33,215708.73
8120127,13,931287.88
8122376,133,253516.81
8122448,103,121682.19
8124814,24,954523.3
8125132,74,1424394.13
8132792,54,291403.79
8136675,16,1415726.94
8144144,113,1366026.47
8144613,43,543987.18
8152756,37,614650.14
8158313,128,279943.42
8159909,147,43404.18
8160917,64,1006828.1
8167286,127,1181271.93
8171956,73,826255.76
8177357,68,1184760.01
8180351,43,1452804.5
8180609,6,930191.21
8193923,99,513555.35
8196754,123,787071.16
8198780,43,102795.54
8216693,91,289330.95
8217213,114,764398.57
8218462,77,1219213.59
8223127,96,1464932.43
8228333,121,722029.94
8229917,57,561288.37
8232317,58,1438197.41
8242993,112,854353.54
8247840,147,1130442.54
8249267,64,78966.39
8250138,15,285227.38
8250907,46,88595.31
8252830,137,745130.89
8258977,75,823389.09
8260997,79,1372290.7
8261828,121,703714.17
8261894,56,1430933.04
8262572,30,971345.95
8268499,58,873955.72
8269182,52,160819.85
8275919,112,503426.49
8277320,97,815148.01
8280728,125,84516.67
8282732,37,89193.81
8285955,16,128069.31
8287450,147,1271714.45
8298358,51,1115641.17
8299387,39,1472839.24
8311743,1,303068.01
8313756,26,787162.81
8329806,115,72405.59
8330776,78,39013.14
8331250,83,176980.41
8331398,148,926437.91
8338705,106,1456905.53
8341181,27,1392631.16
8343219,46,1396334.9
8359402,83,743456.96
8361147,35,28399.19
8362746,17,656568.85
8362868,20,200186.6
8364280,131,774595.26
8366126,106,1214254.34
8373523,120,347214.71
8373828,36,390909.99
8379115,117,1310592.04
8380973,73,127281.18
8383217,9,1287699.2
8384453,2,1122583.57
8384893,19,564697.59
8386202,40,1389117.41
8387846,42,1496644.18
8394883,1,903904.75
8396104,1,630180.93
8396452,28,1198395.4
8400750,121,761799.66
8402793,23,188704.73
8411772,117,573530.53
8414006,105,69343.15
8414777,4,422897.45
8415375,58,1432007.29
8418342,7,556272.36
8418693,21,163586.47
8423916,138,304726.84
8424225,37,630218.18
8433886,47,708911.65
8433919,134,1053081.73
8438985,116,469125.4
8443662,102,568219.32
8448617,130,1379101.2
8448982,5,464907.83
8453464,136,24063.54
8456809,35,513967.42
8462507,85,1200488.41
8479464,111,64695.42
8486273,69,700574.81
8490130,34,363342.77
8497977,33,383660.06
8511327,102,1308320.14
8511958,45,735279.29
8512874,63,213601.14
8513488,51,453669.44
8515960,36,527744.22
8517549,77,427362.69
8527289,143,519964.55
8527929,126,254218.13
8535089,71,987271.75
8537562,65,477482.55
8540176,64,1159462.61
8542327,118,1364293.59
8543074,104,1150494.65
8550933,64,877801.14
8566005,78,223174.46
8567639,126,567086.4
8568585,74,433572.15
8568875,109,1265005.2
8569951,120,413273.84
8575019,11,1302287.06
8578972,36,1088965.4
8579860,127,1261482.54
8584979,39,1069607.82
8590072,113,668203.19
8590178,3,1249546.61
8602737,63,817265.62
8604837,119,413944.99
8605731,81,217794.51
8607451,10,846583.75
8610670,106,1376116.08
8615672,135,69133.93
8618497,96,995288.05
8625652,54,1182811.85
8625691,14,1269149.56
8626360,80,1215223.06
8646035,101,128840.32
8646842,51,826203.94
8648936,104,243570
8652728,38,78684.12
8653410,2,1342511.19
8662358,86,421372.33
8670952,57,527521.62
8671263,63,1274972.32
8675049,95,1348920.93
8677709,108,14902.55
8680364,5,225140.79
8684727,100,303547.24
8689481,139,831205.94
8695543,22,853336.36
8695745,76,763922.28
8696593,86,1207780.16
8699184,63,737072.51
8704262,15,717243.23
8707700,9,1423520.16
8708725,41,1079986.49
8716457,24,725424.48
8727647,84,1228465.24
8735268,55,221011.7
8737379,108,1270693.47
8739485,56,1182703.98
8740451,58,1388352.35
8743489,43,1254310.62
8756708,33,169030.41
8759917,56,123928.94
8764435,139,1489058.3
8769519,92,590634.81
8770407,85,442765.03
8770599,143,459035.61
8778940,137,1296595.61
8779409,4,1081511.45
8787100,84,1066729.95
8787431,41,856452.08
8794423,76,847571.85
8801376,44,970455.43
8809961,105,1487844.83
8816659,71,813792.49
8820529,125,1047475.66
8827501,48,1486218.48
8829203,27,209669.68
8837459,96,1314101.82
8846387,89,245209.35
8847914,60,1030705.74
8860562,30,819225.4
8865034,26,121668.89
8891934,114,113672.49
8895705,146,164042.02
8900348,121,1130726.01
8908051,74,891553.38
8912714,61,1466132.41
8919986,95,645948.44
8934286,148,1182089.07
8934895,39,1177285.31
8938803,23,862527.8
8946649,11,942693.53
8946874,14,212324.58
8947294,91,1172137.68
8949098,110,447397.21
8964583,67,649438.84
8976746,109,1286662.24
8988205,21,69566.02
8991703,141,1254159.15
8992549,3,341503.75
9002878,10,428205.99
9011686,71,185657.08
9020827,64,277656.14
9030747,80,1118489.18
9032561,127,82833.05
9032777,118,139239.92
9033208,147,1376854.53
9041851,2,1245250.07
9043859,31,242206.19
9049040,59,940506.88
9049119,9,565513.26
9050266,18,1372734.62
9050643,61,1207471.72
9052567,126,664798.31
9057067,58,506868.83
9062995,111,1010740.42
9065507,136,976587.14
9067513,107,283054.98
9079118,90,1486541.16
9079596,61,364245.9
9081493,39,117771.07
9081729,128,447869.14
9089216,123,597624.83
9101097,80,622269.96
9103800,146,975095.11
9110074,94,862292.97
9114930,58,149596.91
9115322,57,1195038.52
9119843,57,1463268.73
9123420,108,1104414.22
9129090,113,218324.63
9132305,94,1273556.52
9135842,14,1020279.07
9141779,122,754934.88
9148302,108,688458.58
9151806,131,1284881.39
9154485,145,569640.03
9155756,29,541122.72
9157214,100,1360199.9
9157315,10,1448545.54
9158466,40,837136.84
9163099,70,11931.35
9165911,115,1209494.34
9175842,96,304921.95
9176359,117,967149.51
9177046,24,534648.17
9177697,56,220655.47
9178452,60,244382.02
9184188,11,777065.92
9186052,135,721075.24
9189763,13,1462799.93
9199215,103,120177.96
9212798,104,328516.85
9226740,31,498729.75
9240203,49,1476519.75
9242556,54,1119571.69
9242557,139,669703.51
9248026,70,728629.36
9249307,3,1147860.8
9250434,77,331403.41
9252366,57,425137.24
9253727,66,1295858.81
9256424,109,243276.31
9260378,62,304052.61
9268164,147,1262266.22
9274557,4,187316.26
9277458,9,1213832.2
9282607,76,782766.3
9297725,91,274523.84
9311599,24,285464.87
9316840,77,915823.09
9318352,143,698600.79
9318802,139,1052690.79
9320559,144,1047055.47
9336866,4,120456.49
9338259,94,390596.89
9355588,35,1022932.38
9358456,144,116007.6
9358727,120,554665.4
9366132,130,64329.88
9367021,8,398182.5
9370246,10,1069254.42
9371680,38,526387.47
9373128,31,1221166.45
9377642,40,1372143.82
9377863,8,1154215.16
9389911,42,1259486.01
9390640,89,969890.13
9395176,97,1351573.18
9399379,25,776507.49
9399626,28,1041149.08
9402792,109,8589.54
9403548,65,57131.59
9407329,60,1372503.98
9407854,136,120768.32
9410846,131,1125423.48
9413523,100,1221517.41
9417968,9,857324.44
9421679,22,860811.76
9427070,98,1449809.35
9429796,68,1260677.46
9438122,87,1066521.49
9442460,3,208673.6
9443792,66,454037.69
9447613,107,1440465.05
9455789,72,1268038.01
9458106,122,1289340.39
9466732,89,1027150.44
9470009,22,124059.21
9471209,106,649138.38
9471903,117,559695.13
9479369,101,1329072.54
9479838,95,726291.5
9491380,105,464173.75
9510591,24,951229.09
9515299,107,332978.05
9526540,9,1054005.5
9532630,116,613237.31
9535170,137,1274651.06
9548915,98,1486519.95
9550612,133,470383.08
9562077,146,1435375.24
9572428,119,327042.73
9575541,32,938050.2
9578763,14,1339040.83
9579069,9,1017647.68
9581697,5,1408240.3
9582520,139,1488130.6
9590682,94,1290251.5
9599889,75,823389.08
9601295,91,1110333.74
9604736,44,592325.78
9606982,82,655253.92
9607406,70,1069383.79
9607688,82,1131452.75
9612131,100,1132555
9615634,58,563483.31
9619162,34,1318708.21
9619789,144,437808.07
9625417,33,204765.95
9625543,82,356333.51
9626314,40,1475472.64
9627550,138,1297383.58
9630991,75,1258379.45
9640771,30,1333924.33
9642228,124,1298480.39
9642900,21,531112.32
9643539,17,1394685.55
9643765,141,567101.39
9645937,13,656380.38
9652893,26,253367.48
9653549,5,1307385.27
9655679,78,351858.21
9658403,129,648034.41
9660742,65,460070.59
9661049,135,807835.3
9662187,121,887506.01
9664252,16,868632.2
9669281,134,678200.94
9682164,129,278533.95
9682235,98,679395.98
9682751,121,957593.67
9683147,33,1438144.38
9692313,140,1202996.68
9695855,126,325084.8
9704291,144,1297304.2
9714827,1,918813.08
9716621,135,670732.68
9717670,3,16131.56
9721167,48,760800.23
9721454,26,913234.41
9723986,73,1332375.41
9725617,27,475925.12
9725804,72,620192.96
9735505,31,914387.62
9756516,91,1135578.38
9757611,88,1188038.02
9762650,31,913171.11
9765771,137,1199328.37
9768027,16,384996.17
9768716,22,1252954.7
9776729,138,1131215.6
9776939,14,219036.53
9777437,98,1326382.06
9780231,140,638383.33
9784301,118,139240.92
9785214,140,1054417.97
9790069,80,479668.22
9792661,131,238629.47
9799776,21,114011.94
9804293,20,1022503.42
9805206,51,1302869.73
9812332,89,1235970.9
9818269,78,968684.08
9822979,97,275901.87
9824357,100,642326.52
9825931,39,1387713.24
9841041,31,1424592.53
9848186,99,941342.21
9849746,144,123887.28
9851470,59,293709.28
9856940,108,1352085.4
9860554,79,227830.24
9865656,121,376924.61
9866791,13,456308.39
9868624,14,1459138.13
9868757,63,460243.82
9872258,49,563228.24
9873542,92,26385.32
9890066,76,1005165.79
9893310,99,949799.61
9894587,11,1430864.82
9898094,6,817925.99
9898154,121,1409136.45
9900067,32,552795.46
9902015,125,1388550.11
9912003,147,851820.45
9921696,120,977427.41
9925364,44,672201.42
9925558,107,685641.47
9933240,95,968973.61
9934470,97,718222.34
9937696,119,430509.21
9940853,75,1405914.11
9941024,134,602568.1
9941062,96,933042.62
9947987,22,491968
9951223,27,375777.02
9952504,44,1047938.9
9952967,139,1419050.19
9956556,38,174692.01
9959831,145,301331.86
9963566,66,1465061.23
9968428,133,862102.48
9982686,81,1498169.95
9982738,48,1216685.32
9987964,61,283801.97
9988710,16,1324025.82
9998378,107,1402782.87
'''

