####################################################################################
#                                                                                  
# Copyright (c) 2005 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             
#                                                                                  
# This deteclet is part of the Bid Rigging wizards library.                     
# It requires a license and is not open source.                                    
#                                                                                  
####################################################################################
# UPDATES:
# 20 JUL 2006  Upated header
# 03 NOV 2005  First version of the wizard
#
####################################################################################
#
#   STATUS: Review
#           
#
#    IDEAS/QUESTIONS/NOTES:
#
####################################################################################
from picalo import *

DETECTLET_STANDARD = 1.0

wizard = '''
<wizard>
  <page>
    Please select which table contains the bidding data.
    It should look similar to the example input data (see
    the previous page), and it should have columns for
    the contracts, bidder, line items, and amount.
    
    If you need to join two or more tables together to get
    your data into this form, close the wizard and do so.
    <parameter type="Table" variable="table"/>
  </page>
  <page>
    Your input table should contain the records for many
    contracts (if it were only one contract, you could do this 
    analysis in a spreadsheet).  This wizard will stratify
    the table into one table for each contract, then run 
    the analysis on each contract table.
    
    Which column should the table be stratified by?  This column
    should uniquely identify each contract.  It might be called
    "contract id", "project id", or something similar.
    <parameter type="Column" table="table" variable="stratifycol"/>
  </page>
  <page>
    Each contract will have two or more bidders.  Which column
    specifies the bidder?  This column should contain data
    similar to a "bidder id".
    <parameter type="Column" table="table" variable="biddercol"/>
  </page>
  <page>
    Each contract will have two or more line items that specify
    the individual items on the contract.  For example, a
    contract might have one line item for labor, one line item
    for construction equipment, and another line item for
    a certain type of product.
    
    Which column specifies the line item?  This column should contain
    data similar to a "line item id", "bid item id", "item number",
    etc.
    <parameter type="Column" table="table" variable="lineitemcol"/>
  </page>
  <page>
    Which column contains the bid price of each line item? This is
    not the total price of the entire bid, but the price for each
    line item.
    <parameter type="Column" table="table" variable="amountcol"/>
  </page>
</wizard>
'''

RESULTS_TEXT = '''
<html><body>
<h3>Results:</h3>
    The displayed table shows the standard deviation of price differences from
    bidders on each contract.  Bidders with small standard deviations
    have line items that are very close to another bid.  
    
    The table is sorted by standard deviation to highlight those bidders
    who have very small standard deviations.  Focus on the records at the 
    top of the table because their standard deviations will be closer to
    zero.  

<h3>DiffStdDev:</h3>
    The DiffStdDev 
    column provides the most important number.  It tells you the standard 
    deviation of the bid differences.  If two contractors, A and B, have 
    a standard deviation close to zero (0), it means their items were exact
    percentages of one another.  
    
    For example, a standard deviation of zero (0) indicates that every
    line on one bid is an exact percentage of the lines on the winning bid.
    At most, it indicates that one person probably prepared both bids, and at
    least, it indicates that two bidders knew about each other's bids.  Both
    are good symptoms of bid rigging fraud.
    
<h3>DiffAverage:</h3>
    The DiffAverage column is the average difference between 
    line items on two bids.  If two contractors, A and B, have an average 
    difference of $10, it means that their line items were about (on average) $10
    away from one another.
    
</body></html>
'''



def run(table, stratifycol, biddercol, lineitemcol, amountcol):
  '''Most organizations require multiple bids to increase competition between
     suppliers.  Purchasers may want to circumvent this control when they
     are getting kickbacks from a certain vendor, have family or friends they
     want to funnel work to, or for many other reasons.
     
     One of the ways to fabricate extra bids is to copy and then
     slightly change a single bid to look like multiple bids from different
     companies.  A purchaser loads the real bid into a spreadsheet and
     creates a new column for the second bid.  Each line item of the second
     bid is a certain percentage above the first bid.  If every line item 
     is a specific percentage above the winning bid, the line items were
     likely calculated.
     
     This wizard searches for bids with line items that are all the 
     same percentage above the winning bid. It assumes that you have 
     potentially thousands of projects with 2 or more bids on each project.  
     
     The wizard goes through the following process:
     - Stratifies the file into a table for each project.  
     - Calculates the percentage difference between the line items
       of each bid.
     - Calculates the average and standard deviation of the differences,
       resulting in a single average and standard deviation value
       for each project.  
     - Sorts the results by standard deviation.  
       
     Projects with a very small standard deviation value have nearly
     identical differences between the line items.  Focus on the
     records that have small differences.
     
     The example input table contains the following four columns:
     - Contract: The contract id
     - Bidder: The bidder id - there are three bidders on each of the
       example bids.
     - Item: The line item id.  Most contracts have line items.
     - Amount: The amount of each line item.
     
     In the example dataset, contract #1 has been fabricated.  The
     amounts between bids are exact percentages of each other.
  '''
  # validate the data
  assert stratifycol != biddercol != lineitemcol != amountcol, 'The stratification, bidder, line item, and amount columns must be different.  Please ensure you haven\'t selected the same column for two of these items.'
  
  # make the groups
  groups = Grouping.stratify_by_value(table, stratifycol)
  results = Table([
    ( 'Contract',    unicode ),
    ( 'Bidder1',     unicode ),
    ( 'Bidder2',     unicode ),
    ( 'DiffStdDev',  number ),
    ( 'DiffAverage', number ),
  ])
  for counter, group in enumerate(groups):
    show_progress('Analyzing (' + str(counter) + ' of ' + str(len(groups)) + ')...', float(counter) / len(groups))
    xtab = Crosstable.pivot(group, biddercol, lineitemcol, ['sum(group.column("' + amountcol + '"))'])[:-1]  # remove the totals row at the bottom
    cols = xtab.get_column_names()
    for i in range(1, len(cols)-1):  # compare every col against every other col
      for j in range(i+1, len(cols)-1):
        diff = []
        for rec in xtab:
          try:
            diff.append((rec[cols[i]] - rec[cols[j]]) / rec[cols[j]])
          except:
            pass
        sd = stdev(diff)
        avg = mean(diff)
        rec = results.append()
        rec['Contract'] = group.startvalue
        rec['Bidder1'] = cols[i]
        rec['Bidder2'] = cols[j]
        rec['DiffStdDev'] = round(sd, 2)
        rec['DiffAverage'] = round(avg, 2)

  clear_progress()
  Simple.sort(results, True, 'DiffStdDev')
  return results, RESULTS_TEXT 

  
def example_input():
  import StringIO  # to emulate a file for load_csv
  table = load_csv(StringIO.StringIO(csvdata))
  table.set_type('Contract', int)
  table.set_type('Amount',   number)
  return table



# This next part is not required, but it's easier to put the example data
# directly in this file so I don't have to worry about directories.
csvdata = '''\
Contract,Bidder,Item,Amount
1,BidderA,1.1,663.63
1,BidderA,1.2,4736.74
1,BidderA,1.3,8754.76
1,BidderA,2.1,8491.77
1,BidderA,2.2,10563.76
1,BidderA,2.3,11001.7
1,BidderA,2.4,1478.44
1,BidderA,3.1,596.71
1,BidderA,3.2,7071.23
1,BidderB,1.1,675.53
1,BidderB,1.2,4821.64
1,BidderB,1.3,8911.67
1,BidderB,2.1,8643.96
1,BidderB,2.2,10753.09
1,BidderB,2.3,11198.87
1,BidderB,2.4,1504.94
1,BidderB,3.1,607.4
1,BidderB,3.2,7197.95
1,BidderC,1.1,705.19
1,BidderC,1.2,5033.38
1,BidderC,1.3,9303.02
1,BidderC,2.1,9023.55
1,BidderC,2.2,11225.3
1,BidderC,2.3,11690.66
1,BidderC,2.4,1571.03
1,BidderC,3.1,634.07
1,BidderC,3.2,7514.05
10,BidderA,1.1,10622.41
10,BidderA,1.2,6961.75
10,BidderA,1.3,7344.35
10,BidderA,1.4,4180.43
10,BidderA,2.1,693.57
10,BidderA,2.2,820.29
10,BidderA,2.3,6502.77
10,BidderA,2.4,3503.7
10,BidderA,2.5,3049.7
10,BidderA,2.6,8488.31
10,BidderA,2.7,10165.96
10,BidderA,3.1,5465.59
10,BidderA,3.2,2066.17
10,BidderB,1.1,9400.1
10,BidderB,1.2,6382.71
10,BidderB,1.3,7775.68
10,BidderB,1.4,4003.3
10,BidderB,2.1,757.84
10,BidderB,2.2,809.73
10,BidderB,2.3,6679.61
10,BidderB,2.4,3963.94
10,BidderB,2.5,3031.5
10,BidderB,2.6,9134.64
10,BidderB,2.7,9916.21
10,BidderB,3.1,5403.21
10,BidderB,3.2,1649.85
10,BidderC,1.1,11219.31
10,BidderC,1.2,7011.08
10,BidderC,1.3,7526.3
10,BidderC,1.4,3949.62
10,BidderC,2.1,650.0
10,BidderC,2.2,841.88
10,BidderC,2.3,7173.26
10,BidderC,2.4,3896.24
10,BidderC,2.5,3380.66
10,BidderC,2.6,9433.95
10,BidderC,2.7,10409.59
10,BidderC,3.1,5474.42
10,BidderC,3.2,1848.25
12,BidderA,1.1,10781.24
12,BidderA,1.2,3030.46
12,BidderA,1.3,10048.74
12,BidderA,1.4,125.11
12,BidderA,1.5,9212.31
12,BidderA,1.6,6261.97
12,BidderA,1.7,3829.97
12,BidderA,1.8,2079.09
12,BidderA,1.9,1430.05
12,BidderA,2.1,8328.13
12,BidderA,2.2,9533.72
12,BidderA,2.3,9919.42
12,BidderA,2.4,1585.7
12,BidderA,2.5,2331.02
12,BidderA,3.1,6959.2
12,BidderA,3.2,7750.23
12,BidderA,3.3,9886.74
12,BidderA,3.4,2907.31
12,BidderA,3.5,648.14
12,BidderA,4.1,4162.33
12,BidderA,4.2,7806.38
12,BidderB,1.1,10294.36
12,BidderB,1.2,3206.4
12,BidderB,1.3,9707.78
12,BidderB,1.4,120.35
12,BidderB,1.5,8975.44
12,BidderB,1.6,6837.65
12,BidderB,1.7,4075.54
12,BidderB,1.8,2074.88
12,BidderB,1.9,1452.33
12,BidderB,2.1,7225.69
12,BidderB,2.2,8951.47
12,BidderB,2.3,9034.04
12,BidderB,2.4,1836.6
12,BidderB,2.5,2591.29
12,BidderB,3.1,7446.19
12,BidderB,3.2,7851.13
12,BidderB,3.3,9846.64
12,BidderB,3.4,3082.47
12,BidderB,3.5,815.47
12,BidderB,4.1,3607.82
12,BidderB,4.2,8082.47
12,BidderC,1.1,11235.32
12,BidderC,1.2,3135.61
12,BidderC,1.3,10147.71
12,BidderC,1.4,128.31
12,BidderC,1.5,9669.11
12,BidderC,1.6,6415.7
12,BidderC,1.7,3733.31
12,BidderC,1.8,2169.84
12,BidderC,1.9,1460.36
12,BidderC,2.1,8234.87
12,BidderC,2.2,9263.48
12,BidderC,2.3,9104.86
12,BidderC,2.4,1624.27
12,BidderC,2.5,2335.88
12,BidderC,3.1,6922.79
12,BidderC,3.2,7820.31
12,BidderC,3.3,9432.21
12,BidderC,3.4,2903.41
12,BidderC,3.5,714.9
12,BidderC,4.1,3925.26
12,BidderC,4.2,8002.82
17,BidderA,1.1,7955.55
17,BidderA,1.2,10569.49
17,BidderA,1.3,1721.17
17,BidderA,1.4,900.52
17,BidderA,1.5,10051.87
17,BidderA,1.6,5480.16
17,BidderA,1.7,395.63
17,BidderA,2.1,7769.2
17,BidderA,2.2,1505.08
17,BidderA,2.3,5230.38
17,BidderA,2.4,6916.54
17,BidderA,2.5,10815.46
17,BidderA,2.6,2937.17
17,BidderA,2.7,9763.99
17,BidderA,3.1,222.75
17,BidderA,3.2,8978.6
17,BidderA,3.3,1805.62
17,BidderA,3.4,9354.45
17,BidderA,3.5,9705.54
17,BidderA,3.6,11791.27
17,BidderA,3.7,8813.41
17,BidderA,4.1,2445.39
17,BidderA,4.2,6796.07
17,BidderA,5.1,4635.75
17,BidderA,5.2,8301.31
17,BidderA,5.3,5424.87
17,BidderA,5.4,10137.8
17,BidderA,5.5,5025.16
17,BidderA,5.6,3462.84
17,BidderA,5.7,3664.29
17,BidderA,5.8,820.06
17,BidderA,5.9,9165.76
17,BidderA,6.1,10473.34
17,BidderA,6.2,1072.44
17,BidderA,6.3,7291.86
17,BidderA,6.4,8506.02
17,BidderA,6.5,9173.46
17,BidderA,6.6,2109.47
17,BidderA,6.7,10370.34
17,BidderA,7.1,5823.67
17,BidderA,7.2,11767.19
17,BidderA,7.3,1529.73
17,BidderA,7.4,5082.53
17,BidderA,7.5,10355.74
17,BidderA,7.6,7776.32
17,BidderA,8.1,8409.26
17,BidderA,8.2,5361.9
17,BidderA,8.3,1575.43
17,BidderA,8.4,3748.75
17,BidderA,8.5,9688.21
17,BidderB,1.1,8492.83
17,BidderB,1.2,10653.23
17,BidderB,1.3,1675.68
17,BidderB,1.4,868.83
17,BidderB,1.5,9916.24
17,BidderB,1.6,6167.31
17,BidderB,1.7,384.67
17,BidderB,2.1,8118.15
17,BidderB,2.2,1378.19
17,BidderB,2.3,5204.37
17,BidderB,2.4,6816.05
17,BidderB,2.5,9813.95
17,BidderB,2.6,3135.25
17,BidderB,2.7,9661.55
17,BidderB,3.1,216.21
17,BidderB,3.2,8007.43
17,BidderB,3.3,1715.58
17,BidderB,3.4,10120.16
17,BidderB,3.5,9866.88
17,BidderB,3.6,10141.25
17,BidderB,3.7,9657.75
17,BidderB,4.1,2429.25
17,BidderB,4.2,6407.58
17,BidderB,5.1,4116.06
17,BidderB,5.2,7349.5
17,BidderB,5.3,5675.49
17,BidderB,5.4,10219.14
17,BidderB,5.5,4788.4
17,BidderB,5.6,3141.97
17,BidderB,5.7,3266.03
17,BidderB,5.8,901.74
17,BidderB,5.9,9555.88
17,BidderB,6.1,9484.17
17,BidderB,6.2,1080.22
17,BidderB,6.3,7186.79
17,BidderB,6.4,7688.31
17,BidderB,6.5,9315.46
17,BidderB,6.6,2456.25
17,BidderB,6.7,10939.1
17,BidderB,7.1,6097.8
17,BidderB,7.2,10924.85
17,BidderB,7.3,1747.94
17,BidderB,7.4,5258.97
17,BidderB,7.5,9467.58
17,BidderB,7.6,7986.57
17,BidderB,8.1,8077.99
17,BidderB,8.2,5486.22
17,BidderB,8.3,1592.94
17,BidderB,8.4,3570.47
17,BidderB,8.5,11090.76
17,BidderC,1.1,9130.3
17,BidderC,1.2,9920.6
17,BidderC,1.3,1633.66
17,BidderC,1.4,864.65
17,BidderC,1.5,8982.95
17,BidderC,1.6,5776.33
17,BidderC,1.7,415.75
17,BidderC,2.1,7207.72
17,BidderC,2.2,1693.91
17,BidderC,2.3,5188.74
17,BidderC,2.4,6779.25
17,BidderC,2.5,11229.53
17,BidderC,2.6,2667.82
17,BidderC,2.7,8991.28
17,BidderC,3.1,205.81
17,BidderC,3.2,8943.09
17,BidderC,3.3,1939.92
17,BidderC,3.4,10486.05
17,BidderC,3.5,9932.66
17,BidderC,3.6,10447.15
17,BidderC,3.7,10454.79
17,BidderC,4.1,2381.23
17,BidderC,4.2,7163.94
17,BidderC,5.1,4435.47
17,BidderC,5.2,8116.43
17,BidderC,5.3,6123.24
17,BidderC,5.4,10608.4
17,BidderC,5.5,4890.71
17,BidderC,5.6,3488.19
17,BidderC,5.7,3776.53
17,BidderC,5.8,856.14
17,BidderC,5.9,9420.07
17,BidderC,6.1,10264.78
17,BidderC,6.2,1003.75
17,BidderC,6.3,7068.29
17,BidderC,6.4,8320.16
17,BidderC,6.5,8985.11
17,BidderC,6.6,2294.32
17,BidderC,6.7,8899.13
17,BidderC,7.1,6209.7
17,BidderC,7.2,10596.37
17,BidderC,7.3,1712.79
17,BidderC,7.4,5051.92
17,BidderC,7.5,10144.32
17,BidderC,7.6,7278.83
17,BidderC,8.1,8285.03
17,BidderC,8.2,5918.9
17,BidderC,8.3,1457.67
17,BidderC,8.4,4098.47
17,BidderC,8.5,10499.06
25,BidderA,1.1,5958.13
25,BidderA,1.2,10987.69
25,BidderA,1.3,10011.11
25,BidderA,1.4,9607.35
25,BidderA,1.5,2304.16
25,BidderA,1.6,2595.23
25,BidderA,2.1,5262.8
25,BidderA,2.2,9376.0
25,BidderA,2.3,3240.02
25,BidderA,2.4,9546.1
25,BidderA,2.5,3834.95
25,BidderA,2.6,3838.46
25,BidderA,2.7,10318.74
25,BidderA,3.1,2367.65
25,BidderA,3.2,10662.15
25,BidderA,3.3,7795.13
25,BidderA,3.4,11747.92
25,BidderA,3.5,922.9
25,BidderA,4.1,6085.51
25,BidderA,4.2,9109.13
25,BidderA,5.1,2730.03
25,BidderA,5.2,1530.89
25,BidderA,5.3,4369.12
25,BidderA,5.4,11261.41
25,BidderA,5.5,9493.57
25,BidderA,5.6,3367.64
25,BidderA,5.7,2055.33
25,BidderA,6.1,5440.31
25,BidderA,6.2,3823.65
25,BidderA,6.3,3499.61
25,BidderA,6.4,5776.72
25,BidderA,6.5,1404.36
25,BidderB,1.1,6130.07
25,BidderB,1.2,9769.89
25,BidderB,1.3,10706.45
25,BidderB,1.4,9786.63
25,BidderB,1.5,1988.05
25,BidderB,1.6,2563.49
25,BidderB,2.1,4637.16
25,BidderB,2.2,9820.02
25,BidderB,2.3,3528.22
25,BidderB,2.4,9470.5
25,BidderB,2.5,3238.37
25,BidderB,2.6,4392.68
25,BidderB,2.7,11254.19
25,BidderB,3.1,2698.23
25,BidderB,3.2,10250.32
25,BidderB,3.3,6493.13
25,BidderB,3.4,10407.03
25,BidderB,3.5,943.8
25,BidderB,4.1,5490.37
25,BidderB,4.2,8900.92
25,BidderB,5.1,3162.58
25,BidderB,5.2,1518.13
25,BidderB,5.3,4602.38
25,BidderB,5.4,10489.49
25,BidderB,5.5,9232.13
25,BidderB,5.6,3415.36
25,BidderB,5.7,2030.75
25,BidderB,6.1,5568.96
25,BidderB,6.2,3839.91
25,BidderB,6.3,3589.6
25,BidderB,6.4,5364.11
25,BidderB,6.5,1460.41
25,BidderC,1.1,5881.56
25,BidderC,1.2,10816.85
25,BidderC,1.3,11173.7
25,BidderC,1.4,9168.1
25,BidderC,1.5,2220.55
25,BidderC,1.6,2508.76
25,BidderC,2.1,5076.36
25,BidderC,2.2,9575.49
25,BidderC,2.3,2995.82
25,BidderC,2.4,9901.4
25,BidderC,2.5,4055.44
25,BidderC,2.6,3893.12
25,BidderC,2.7,10496.68
25,BidderC,3.1,2697.27
25,BidderC,3.2,10034.98
25,BidderC,3.3,7272.39
25,BidderC,3.4,11635.71
25,BidderC,3.5,869.04
25,BidderC,4.1,6168.85
25,BidderC,4.2,8944.35
25,BidderC,5.1,3093.65
25,BidderC,5.2,1553.98
25,BidderC,5.3,4295.06
25,BidderC,5.4,10180.83
25,BidderC,5.5,8763.06
25,BidderC,5.6,3900.3
25,BidderC,5.7,1997.2
25,BidderC,6.1,5532.03
25,BidderC,6.2,3088.22
25,BidderC,6.3,3485.89
25,BidderC,6.4,5344.69
25,BidderC,6.5,1494.27
27,BidderA,1.1,6314.17
27,BidderA,1.2,5696.43
27,BidderA,1.3,10588.56
27,BidderA,1.4,1901.72
27,BidderA,1.5,6784.84
27,BidderA,1.6,5801.3
27,BidderA,1.7,6884.57
27,BidderA,2.1,9319.98
27,BidderA,2.2,6959.29
27,BidderA,2.3,2011.78
27,BidderA,2.4,2083.69
27,BidderA,2.5,5847.3
27,BidderA,2.6,135.73
27,BidderA,2.7,6334.08
27,BidderA,3.1,9963.11
27,BidderA,3.2,2239.52
27,BidderA,3.3,2465.37
27,BidderA,3.4,6237.08
27,BidderA,3.5,5704.32
27,BidderA,3.6,3294.59
27,BidderA,3.7,9395.23
27,BidderA,3.8,11104.78
27,BidderA,3.9,11221.31
27,BidderA,4.1,9016.28
27,BidderA,4.2,6816.37
27,BidderA,4.3,11511.38
27,BidderA,4.4,2272.77
27,BidderA,4.5,5594.47
27,BidderA,4.6,9120.28
27,BidderA,4.7,3719.72
27,BidderA,5.1,1452.24
27,BidderA,5.2,145.9
27,BidderA,5.3,2268.64
27,BidderA,6.1,8287.28
27,BidderA,6.2,7227.04
27,BidderA,6.3,6500.28
27,BidderA,6.4,2570.56
27,BidderA,6.5,325.12
27,BidderA,6.6,6082.75
27,BidderA,6.7,8523.67
27,BidderA,6.8,5639.12
27,BidderA,6.9,4578.19
27,BidderB,1.1,5963.61
27,BidderB,1.2,5323.64
27,BidderB,1.3,10563.38
27,BidderB,1.4,1744.15
27,BidderB,1.5,7313.5
27,BidderB,1.6,5643.96
27,BidderB,1.7,6978.28
27,BidderB,2.1,9220.57
27,BidderB,2.2,7840.25
27,BidderB,2.3,1977.09
27,BidderB,2.4,2046.5
27,BidderB,2.5,5811.85
27,BidderB,2.6,134.25
27,BidderB,2.7,6026.47
27,BidderB,3.1,9684.89
27,BidderB,3.2,2218.18
27,BidderB,3.3,2297.34
27,BidderB,3.4,6736.51
27,BidderB,3.5,5314.8
27,BidderB,3.6,3166.73
27,BidderB,3.7,8671.01
27,BidderB,3.8,9725.49
27,BidderB,3.9,11024.33
27,BidderB,4.1,9863.08
27,BidderB,4.2,6783.06
27,BidderB,4.3,10457.26
27,BidderB,4.4,2252.5
27,BidderB,4.5,5494.05
27,BidderB,4.6,8944.7
27,BidderB,4.7,4142.37
27,BidderB,5.1,1297.05
27,BidderB,5.2,142.96
27,BidderB,5.3,2506.15
27,BidderB,6.1,7372.96
27,BidderB,6.2,7359.49
27,BidderB,6.3,7899.37
27,BidderB,6.4,2292.21
27,BidderB,6.5,363.13
27,BidderB,6.6,6469.34
27,BidderB,6.7,7288.7
27,BidderB,6.8,6136.8
27,BidderB,6.9,5388.83
27,BidderC,1.1,5566.37
27,BidderC,1.2,5630.35
27,BidderC,1.3,9849.31
27,BidderC,1.4,1710.14
27,BidderC,1.5,7182.12
27,BidderC,1.6,6253.26
27,BidderC,1.7,6772.32
27,BidderC,2.1,8360.68
27,BidderC,2.2,8008.04
27,BidderC,2.3,1791.02
27,BidderC,2.4,2065.28
27,BidderC,2.5,6373.42
27,BidderC,2.6,121.15
27,BidderC,2.7,6151.38
27,BidderC,3.1,9531.29
27,BidderC,3.2,2271.83
27,BidderC,3.3,2514.56
27,BidderC,3.4,6579.27
27,BidderC,3.5,5699.8
27,BidderC,3.6,3366.97
27,BidderC,3.7,9786.6
27,BidderC,3.8,11281.39
27,BidderC,3.9,10034.07
27,BidderC,4.1,9494.14
27,BidderC,4.2,7521.21
27,BidderC,4.3,11029.03
27,BidderC,4.4,2325.47
27,BidderC,4.5,5191.46
27,BidderC,4.6,9761.03
27,BidderC,4.7,4105.78
27,BidderC,5.1,1385.32
27,BidderC,5.2,150.57
27,BidderC,5.3,2509.14
27,BidderC,6.1,7693.34
27,BidderC,6.2,6820.33
27,BidderC,6.3,7867.77
27,BidderC,6.4,2538.77
27,BidderC,6.5,345.63
27,BidderC,6.6,5948.12
27,BidderC,6.7,7385.89
27,BidderC,6.8,6151.11
27,BidderC,6.9,5244.69
32,BidderA,1.1,6624.78
32,BidderA,1.2,3171.84
32,BidderA,1.3,4248.99
32,BidderA,1.4,8201.57
32,BidderA,2.1,8936.47
32,BidderA,2.2,2053.73
32,BidderA,3.1,4197.22
32,BidderA,3.2,4548.02
32,BidderA,3.3,10434.27
32,BidderA,3.4,7957.13
32,BidderA,3.5,1440.06
32,BidderA,3.6,1274.99
32,BidderA,3.7,3574.43
32,BidderA,3.8,5078.75
32,BidderA,3.9,4414.52
32,BidderA,4.1,1451.61
32,BidderA,4.2,10366.35
32,BidderA,4.3,1093.91
32,BidderB,1.1,6096.11
32,BidderB,1.2,3091.34
32,BidderB,1.3,4085.75
32,BidderB,1.4,8806.87
32,BidderB,2.1,8236.76
32,BidderB,2.2,1897.09
32,BidderB,3.1,3949.26
32,BidderB,3.2,4598.24
32,BidderB,3.3,9824.75
32,BidderB,3.4,8041.05
32,BidderB,3.5,1477.93
32,BidderB,3.6,1453.4
32,BidderB,3.7,4066.8
32,BidderB,3.8,4731.72
32,BidderB,3.9,4720.23
32,BidderB,4.1,1439.36
32,BidderB,4.2,9659.23
32,BidderB,4.3,1225.53
32,BidderC,1.1,6210.96
32,BidderC,1.2,3326.4
32,BidderC,1.3,4483.0
32,BidderC,1.4,8547.48
32,BidderC,2.1,9552.81
32,BidderC,2.2,2040.02
32,BidderC,3.1,3753.96
32,BidderC,3.2,5237.34
32,BidderC,3.3,10882.39
32,BidderC,3.4,8703.01
32,BidderC,3.5,1433.56
32,BidderC,3.6,1401.25
32,BidderC,3.7,4094.03
32,BidderC,3.8,5086.92
32,BidderC,3.9,4660.99
32,BidderC,4.1,1492.8
32,BidderC,4.2,10423.22
32,BidderC,4.3,1268.21
39,BidderA,1.1,4363.21
39,BidderA,1.2,10478.03
39,BidderA,1.3,9743.99
39,BidderA,2.1,6129.4
39,BidderA,2.2,11221.14
39,BidderA,2.3,8967.28
39,BidderA,3.1,9651.66
39,BidderA,3.2,7997.29
39,BidderA,3.3,2522.39
39,BidderA,3.4,3392.59
39,BidderA,3.5,8744.78
39,BidderA,3.6,5471.13
39,BidderA,3.7,7897.01
39,BidderA,3.8,4680.97
39,BidderA,3.9,2331.89
39,BidderA,4.1,3934.95
39,BidderA,4.2,8495.67
39,BidderA,4.3,5613.93
39,BidderA,4.4,9754.33
39,BidderA,4.5,383.63
39,BidderA,4.6,10719.39
39,BidderA,4.7,8048.92
39,BidderA,5.1,3552.49
39,BidderA,5.2,7207.13
39,BidderA,5.3,8714.01
39,BidderA,5.4,10978.75
39,BidderA,5.5,3303.66
39,BidderA,5.6,7011.41
39,BidderA,5.7,8208.21
39,BidderA,5.8,5203.56
39,BidderA,6.1,512.89
39,BidderA,6.2,5190.89
39,BidderA,6.3,434.59
39,BidderA,6.4,4235.99
39,BidderA,6.5,7294.81
39,BidderA,6.6,7895.85
39,BidderB,1.1,4132.96
39,BidderB,1.2,9466.86
39,BidderB,1.3,10400.15
39,BidderB,2.1,6200.53
39,BidderB,2.2,10219.51
39,BidderB,2.3,7258.03
39,BidderB,3.1,9619.47
39,BidderB,3.2,6443.61
39,BidderB,3.3,2466.81
39,BidderB,3.4,3468.38
39,BidderB,3.5,9830.63
39,BidderB,3.6,5366.33
39,BidderB,3.7,8358.1
39,BidderB,3.8,4400.19
39,BidderB,3.9,2923.78
39,BidderB,4.1,4213.88
39,BidderB,4.2,8873.2
39,BidderB,4.3,5822.73
39,BidderB,4.4,10871.49
39,BidderB,4.5,381.04
39,BidderB,4.6,10254.51
39,BidderB,4.7,8942.55
39,BidderB,5.1,3401.97
39,BidderB,5.2,7505.78
39,BidderB,5.3,8116.96
39,BidderB,5.4,11579.32
39,BidderB,5.5,3329.57
39,BidderB,5.6,6314.01
39,BidderB,5.7,7727.76
39,BidderB,5.8,5607.65
39,BidderB,6.1,540.92
39,BidderB,6.2,5439.68
39,BidderB,6.3,399.51
39,BidderB,6.4,3751.19
39,BidderB,6.5,6906.38
39,BidderB,6.6,8405.6
39,BidderC,1.1,4458.76
39,BidderC,1.2,10404.17
39,BidderC,1.3,10703.87
39,BidderC,2.1,6228.07
39,BidderC,2.2,11736.33
39,BidderC,2.3,8554.84
39,BidderC,3.1,9920.51
39,BidderC,3.2,7928.68
39,BidderC,3.3,2584.13
39,BidderC,3.4,3799.1
39,BidderC,3.5,9242.93
39,BidderC,3.6,5261.07
39,BidderC,3.7,6981.59
39,BidderC,3.8,4445.15
39,BidderC,3.9,2923.63
39,BidderC,4.1,3893.47
39,BidderC,4.2,8925.55
39,BidderC,4.3,6097.33
39,BidderC,4.4,10301.19
39,BidderC,4.5,368.23
39,BidderC,4.6,9637.2
39,BidderC,4.7,9182.22
39,BidderC,5.1,3660.32
39,BidderC,5.2,7516.86
39,BidderC,5.3,9539.36
39,BidderC,5.4,10075.08
39,BidderC,5.5,3025.03
39,BidderC,5.6,6779.11
39,BidderC,5.7,7697.14
39,BidderC,5.8,5605.02
39,BidderC,6.1,513.74
39,BidderC,6.2,5455.04
39,BidderC,6.3,416.86
39,BidderC,6.4,3782.44
39,BidderC,6.5,6521.27
39,BidderC,6.6,8909.3
44,BidderA,1.1,5878.47
44,BidderA,1.2,5113.76
44,BidderA,1.3,10917.27
44,BidderA,1.4,5316.63
44,BidderA,1.5,5370.13
44,BidderA,2.1,6031.81
44,BidderA,2.2,1807.62
44,BidderA,2.3,7270.0
44,BidderA,3.1,5676.0
44,BidderA,3.2,8431.04
44,BidderA,3.3,7406.06
44,BidderA,3.4,2709.43
44,BidderA,3.5,7672.65
44,BidderA,4.1,3276.34
44,BidderA,4.2,10071.98
44,BidderA,4.3,2727.1
44,BidderA,5.1,6200.64
44,BidderA,5.2,3590.95
44,BidderA,5.3,1821.23
44,BidderA,5.4,4030.45
44,BidderA,5.5,7652.86
44,BidderA,5.6,7801.62
44,BidderA,5.7,2110.44
44,BidderA,6.1,8102.64
44,BidderA,6.2,932.62
44,BidderA,6.3,6620.73
44,BidderA,7.1,3420.48
44,BidderA,7.2,10120.31
44,BidderA,7.3,2101.54
44,BidderA,7.4,9087.75
44,BidderA,7.5,316.67
44,BidderB,1.1,5808.49
44,BidderB,1.2,5201.89
44,BidderB,1.3,11424.45
44,BidderB,1.4,5053.69
44,BidderB,1.5,6152.05
44,BidderB,2.1,6695.27
44,BidderB,2.2,1897.86
44,BidderB,2.3,7046.84
44,BidderB,3.1,4915.16
44,BidderB,3.2,8647.69
44,BidderB,3.3,8232.67
44,BidderB,3.4,2590.08
44,BidderB,3.5,7057.14
44,BidderB,4.1,3487.82
44,BidderB,4.2,10692.1
44,BidderB,4.3,2741.66
44,BidderB,5.1,5999.79
44,BidderB,5.2,3120.54
44,BidderB,5.3,1728.98
44,BidderB,5.4,3788.68
44,BidderB,5.5,7840.06
44,BidderB,5.6,7102.16
44,BidderB,5.7,1714.97
44,BidderB,6.1,8594.45
44,BidderB,6.2,1022.01
44,BidderB,6.3,7049.92
44,BidderB,7.1,3135.14
44,BidderB,7.2,10246.54
44,BidderB,7.3,2312.73
44,BidderB,7.4,8874.05
44,BidderB,7.5,294.09
44,BidderC,1.1,6701.05
44,BidderC,1.2,5355.95
44,BidderC,1.3,9996.85
44,BidderC,1.4,5538.1
44,BidderC,1.5,5034.46
44,BidderC,2.1,6204.09
44,BidderC,2.2,2022.68
44,BidderC,2.3,7919.05
44,BidderC,3.1,5750.06
44,BidderC,3.2,7814.71
44,BidderC,3.3,7766.09
44,BidderC,3.4,2612.0
44,BidderC,3.5,6729.94
44,BidderC,4.1,3406.02
44,BidderC,4.2,10686.82
44,BidderC,4.3,3058.26
44,BidderC,5.1,6324.65
44,BidderC,5.2,3399.25
44,BidderC,5.3,1915.07
44,BidderC,5.4,4275.89
44,BidderC,5.5,7793.04
44,BidderC,5.6,7242.83
44,BidderC,5.7,1866.49
44,BidderC,6.1,9214.84
44,BidderC,6.2,934.02
44,BidderC,6.3,6919.89
44,BidderC,7.1,3068.84
44,BidderC,7.2,11821.05
44,BidderC,7.3,2403.95
44,BidderC,7.4,9062.36
44,BidderC,7.5,330.65
46,BidderA,1.1,5796.31
46,BidderA,1.2,2404.65
46,BidderA,1.3,1375.32
46,BidderA,1.4,4392.21
46,BidderA,1.5,9639.26
46,BidderA,1.6,3405.06
46,BidderA,2.1,2851.6
46,BidderA,2.2,551.15
46,BidderA,2.3,10282.72
46,BidderA,2.4,4779.64
46,BidderA,2.5,4157.08
46,BidderA,2.6,1554.45
46,BidderA,2.7,6709.67
46,BidderA,2.8,5426.86
46,BidderA,2.9,3110.55
46,BidderA,3.1,4516.37
46,BidderA,3.2,10952.9
46,BidderA,3.3,5764.21
46,BidderA,3.4,9303.85
46,BidderA,3.5,2591.68
46,BidderA,3.6,5096.59
46,BidderA,4.1,4150.2
46,BidderA,4.2,8970.13
46,BidderA,4.3,6832.28
46,BidderA,4.4,2303.73
46,BidderA,5.1,9618.53
46,BidderA,5.2,6242.52
46,BidderA,5.3,3090.24
46,BidderA,5.4,8870.2
46,BidderA,5.5,1199.89
46,BidderA,5.6,5405.82
46,BidderA,6.1,3328.78
46,BidderA,6.2,3039.71
46,BidderA,6.3,9137.89
46,BidderA,6.4,265.5
46,BidderA,6.5,10493.53
46,BidderA,6.6,3335.95
46,BidderA,6.7,2754.73
46,BidderA,6.8,10364.21
46,BidderA,6.9,196.85
46,BidderA,7.1,2023.83
46,BidderA,7.2,4351.45
46,BidderA,7.3,4850.11
46,BidderA,7.4,7428.02
46,BidderA,8.1,4987.36
46,BidderA,8.2,11080.79
46,BidderA,8.3,6303.41
46,BidderA,8.4,1128.24
46,BidderA,8.5,3931.36
46,BidderA,8.6,5280.98
46,BidderB,1.1,6243.63
46,BidderB,1.2,2554.29
46,BidderB,1.3,1606.44
46,BidderB,1.4,4161.89
46,BidderB,1.5,8793.15
46,BidderB,1.6,3405.08
46,BidderB,2.1,3039.7
46,BidderB,2.2,540.88
46,BidderB,2.3,11089.26
46,BidderB,2.4,4779.64
46,BidderB,2.5,3653.22
46,BidderB,2.6,1761.9
46,BidderB,2.7,6208.19
46,BidderB,2.8,5323.74
46,BidderB,2.9,3035.63
46,BidderB,3.1,4759.97
46,BidderB,3.2,10218.42
46,BidderB,3.3,6351.9
46,BidderB,3.4,9581.03
46,BidderB,3.5,2601.08
46,BidderB,3.6,5087.53
46,BidderB,4.1,4670.98
46,BidderB,4.2,8136.67
46,BidderB,4.3,7476.64
46,BidderB,4.4,2188.9
46,BidderB,5.1,8851.87
46,BidderB,5.2,6690.87
46,BidderB,5.3,3500.1
46,BidderB,5.4,8983.23
46,BidderB,5.5,1234.22
46,BidderB,5.6,5446.55
46,BidderB,6.1,3499.29
46,BidderB,6.2,3038.01
46,BidderB,6.3,9039.28
46,BidderB,6.4,259.95
46,BidderB,6.5,10277.95
46,BidderB,6.6,3853.18
46,BidderB,6.7,3192.3
46,BidderB,6.8,9846.87
46,BidderB,6.9,184.38
46,BidderB,7.1,2039.76
46,BidderB,7.2,4552.39
46,BidderB,7.3,5481.12
46,BidderB,7.4,7452.09
46,BidderB,8.1,4831.94
46,BidderB,8.2,10363.84
46,BidderB,8.3,6558.7
46,BidderB,8.4,1000.52
46,BidderB,8.5,4092.06
46,BidderB,8.6,5823.64
46,BidderC,1.1,6522.08
46,BidderC,1.2,2477.06
46,BidderC,1.3,1505.52
46,BidderC,1.4,4268.97
46,BidderC,1.5,8625.64
46,BidderC,1.6,3826.91
46,BidderC,2.1,3335.82
46,BidderC,2.2,485.74
46,BidderC,2.3,10080.09
46,BidderC,2.4,4708.63
46,BidderC,2.5,3839.21
46,BidderC,2.6,1735.79
46,BidderC,2.7,6348.31
46,BidderC,2.8,5365.28
46,BidderC,2.9,2800.7
46,BidderC,3.1,4759.97
46,BidderC,3.2,10311.09
46,BidderC,3.3,5606.4
46,BidderC,3.4,10092.69
46,BidderC,3.5,2634.9
46,BidderC,3.6,4882.43
46,BidderC,4.1,3827.07
46,BidderC,4.2,9258.59
46,BidderC,4.3,6757.31
46,BidderC,4.4,2012.06
46,BidderC,5.1,9844.29
46,BidderC,5.2,6147.75
46,BidderC,5.3,3078.71
46,BidderC,5.4,8648.12
46,BidderC,5.5,1282.92
46,BidderC,5.6,5794.41
46,BidderC,6.1,3312.45
46,BidderC,6.2,2903.2
46,BidderC,6.3,9874.82
46,BidderC,6.4,253.8
46,BidderC,6.5,9888.29
46,BidderC,6.6,3860.51
46,BidderC,6.7,3356.01
46,BidderC,6.8,9027.09
46,BidderC,6.9,199.0
46,BidderC,7.1,2011.83
46,BidderC,7.2,4534.79
46,BidderC,7.3,5023.07
46,BidderC,7.4,7356.88
46,BidderC,8.1,4483.29
46,BidderC,8.2,10660.47
46,BidderC,8.3,6667.65
46,BidderC,8.4,1035.1
46,BidderC,8.5,4274.12
46,BidderC,8.6,5886.96
55,BidderA,1.1,10671.95
55,BidderA,1.2,3910.04
55,BidderA,1.3,1384.26
55,BidderA,1.4,2109.6
55,BidderA,1.5,5088.2
55,BidderA,1.6,8917.71
55,BidderA,1.7,4547.74
55,BidderA,2.1,11128.95
55,BidderA,2.2,6934.97
55,BidderB,1.1,10909.03
55,BidderB,1.2,3961.68
55,BidderB,1.3,1247.76
55,BidderB,1.4,2169.15
55,BidderB,1.5,4541.55
55,BidderB,1.6,8745.73
55,BidderB,1.7,4398.72
55,BidderB,2.1,11404.91
55,BidderB,2.2,8264.54
55,BidderC,1.1,11261.48
55,BidderC,1.2,3501.65
55,BidderC,1.3,1306.07
55,BidderC,1.4,2039.9
55,BidderC,1.5,4568.3
55,BidderC,1.6,8420.18
55,BidderC,1.7,4534.63
55,BidderC,2.1,11630.96
55,BidderC,2.2,7819.23
59,BidderA,1.1,2435.38
59,BidderA,1.2,6503.32
59,BidderA,1.3,890.61
59,BidderA,1.4,3735.55
59,BidderA,1.5,694.72
59,BidderA,1.6,5474.0
59,BidderA,1.7,5963.97
59,BidderA,2.1,5597.64
59,BidderA,2.2,6393.49
59,BidderA,2.3,2116.59
59,BidderA,2.4,6991.03
59,BidderA,3.1,9195.43
59,BidderA,3.2,3443.17
59,BidderA,3.3,4013.56
59,BidderA,3.4,4622.07
59,BidderA,4.1,9571.35
59,BidderA,4.2,4235.26
59,BidderA,4.3,1365.17
59,BidderA,4.4,5706.48
59,BidderA,4.5,1179.55
59,BidderA,5.1,7758.84
59,BidderA,5.2,3670.7
59,BidderA,5.3,3151.47
59,BidderA,5.4,8688.85
59,BidderA,5.5,6337.24
59,BidderA,5.6,2821.14
59,BidderA,5.7,1754.86
59,BidderA,5.8,7946.09
59,BidderA,6.1,11188.84
59,BidderA,6.2,10982.85
59,BidderA,6.3,7571.35
59,BidderA,7.1,2704.49
59,BidderA,7.2,2499.68
59,BidderA,7.3,7127.37
59,BidderA,7.4,2178.39
59,BidderA,7.5,2783.45
59,BidderA,7.6,2107.5
59,BidderB,1.1,2385.29
59,BidderB,1.2,6076.74
59,BidderB,1.3,890.61
59,BidderB,1.4,3618.95
59,BidderB,1.5,680.91
59,BidderB,1.6,5683.22
59,BidderB,1.7,6112.87
59,BidderB,2.1,5654.62
59,BidderB,2.2,6762.49
59,BidderB,2.3,2290.3
59,BidderB,2.4,6074.72
59,BidderB,3.1,8689.87
59,BidderB,3.2,3736.38
59,BidderB,3.3,3916.42
59,BidderB,3.4,4853.12
59,BidderB,4.1,9325.4
59,BidderB,4.2,3908.39
59,BidderB,4.3,1440.5
59,BidderB,4.4,5282.24
59,BidderB,4.5,1036.05
59,BidderB,5.1,7402.99
59,BidderB,5.2,3508.72
59,BidderB,5.3,2569.03
59,BidderB,5.4,9312.02
59,BidderB,5.5,5557.72
59,BidderB,5.6,2603.2
59,BidderB,5.7,1720.53
59,BidderB,5.8,8273.59
59,BidderB,6.1,10434.87
59,BidderB,6.2,10870.22
59,BidderB,6.3,8072.35
59,BidderB,7.1,2542.42
59,BidderB,7.2,2467.34
59,BidderB,7.3,7618.19
59,BidderB,7.4,2459.53
59,BidderB,7.5,2698.77
59,BidderB,7.6,2103.28
59,BidderC,1.1,2113.04
59,BidderC,1.2,5870.07
59,BidderC,1.3,1021.58
59,BidderC,1.4,3392.95
59,BidderC,1.5,701.86
59,BidderC,1.6,5840.72
59,BidderC,1.7,6778.25
59,BidderC,2.1,4997.17
59,BidderC,2.2,6428.07
59,BidderC,2.3,2340.65
59,BidderC,2.4,5611.79
59,BidderC,3.1,9452.53
59,BidderC,3.2,3536.06
59,BidderC,3.3,4333.79
59,BidderC,3.4,4148.98
59,BidderC,4.1,10562.53
59,BidderC,4.2,4291.34
59,BidderC,4.3,1425.31
59,BidderC,4.4,5449.76
59,BidderC,4.5,958.43
59,BidderC,5.1,7616.98
59,BidderC,5.2,3844.15
59,BidderC,5.3,3117.36
59,BidderC,5.4,10112.68
59,BidderC,5.5,6410.23
59,BidderC,5.6,3283.01
59,BidderC,5.7,1897.26
59,BidderC,5.8,7949.39
59,BidderC,6.1,10892.05
59,BidderC,6.2,9447.4
59,BidderC,6.3,6924.85
59,BidderC,7.1,2395.91
59,BidderC,7.2,2241.8
59,BidderC,7.3,7798.4
59,BidderC,7.4,2241.48
59,BidderC,7.5,2531.11
59,BidderC,7.6,1992.42
61,BidderA,1.1,2586.96
61,BidderA,1.2,5969.76
61,BidderA,1.3,8871.61
61,BidderA,1.4,4954.92
61,BidderA,1.5,10055.84
61,BidderA,1.6,8698.29
61,BidderA,1.7,426.96
61,BidderA,1.8,2828.24
61,BidderA,2.1,7484.5
61,BidderA,2.2,10276.5
61,BidderA,2.3,4901.29
61,BidderA,2.4,8209.57
61,BidderA,2.5,9790.69
61,BidderA,2.6,6379.78
61,BidderA,2.7,1167.6
61,BidderA,2.8,4430.63
61,BidderA,2.9,9431.4
61,BidderA,3.1,1265.15
61,BidderA,3.2,10316.27
61,BidderA,3.3,250.46
61,BidderA,3.4,3522.49
61,BidderA,3.5,6654.57
61,BidderA,3.6,2818.78
61,BidderA,4.1,3342.23
61,BidderA,4.2,3160.89
61,BidderA,4.3,2992.34
61,BidderA,4.4,10629.09
61,BidderA,4.5,5995.87
61,BidderA,4.6,7568.18
61,BidderA,4.7,1197.55
61,BidderA,5.1,380.69
61,BidderA,5.2,6690.24
61,BidderA,5.3,225.27
61,BidderA,5.4,753.82
61,BidderA,5.5,1107.0
61,BidderA,5.6,1389.07
61,BidderA,5.7,3663.4
61,BidderA,6.1,10693.93
61,BidderA,6.2,5148.39
61,BidderA,6.3,571.2
61,BidderA,6.4,5750.84
61,BidderA,6.5,5426.69
61,BidderA,6.6,4103.82
61,BidderA,6.7,7216.96
61,BidderA,6.8,5296.7
61,BidderA,6.9,6176.75
61,BidderA,7.1,5906.2
61,BidderA,7.2,8118.44
61,BidderA,7.3,2908.34
61,BidderA,7.4,8730.86
61,BidderA,7.5,7569.52
61,BidderA,7.6,9526.41
61,BidderA,7.7,4185.39
61,BidderA,8.1,4579.64
61,BidderA,8.2,8878.19
61,BidderA,8.3,7325.41
61,BidderA,8.4,4818.63
61,BidderA,8.5,3339.06
61,BidderA,8.6,8783.68
61,BidderA,8.7,8346.86
61,BidderA,8.8,8678.59
61,BidderA,8.9,814.17
61,BidderB,1.1,2312.17
61,BidderB,1.2,5355.54
61,BidderB,1.3,8701.23
61,BidderB,1.4,4514.14
61,BidderB,1.5,9555.57
61,BidderB,1.6,9706.0
61,BidderB,1.7,482.4
61,BidderB,1.8,2579.99
61,BidderB,2.1,7131.07
61,BidderB,2.2,10165.54
61,BidderB,2.3,4477.7
61,BidderB,2.4,8092.67
61,BidderB,2.5,9963.84
61,BidderB,2.6,7340.15
61,BidderB,2.7,1294.95
61,BidderB,2.8,4932.54
61,BidderB,2.9,8268.27
61,BidderB,3.1,1197.93
61,BidderB,3.2,10788.65
61,BidderB,3.3,246.33
61,BidderB,3.4,3481.75
61,BidderB,3.5,5781.89
61,BidderB,3.6,3169.76
61,BidderB,4.1,3510.46
61,BidderB,4.2,3128.16
61,BidderB,4.3,2856.45
61,BidderB,4.4,9980.22
61,BidderB,4.5,6411.93
61,BidderB,4.6,8259.22
61,BidderB,4.7,1168.63
61,BidderB,5.1,340.1
61,BidderB,5.2,7476.17
61,BidderB,5.3,227.7
61,BidderB,5.4,751.97
61,BidderB,5.5,1045.84
61,BidderB,5.6,1325.85
61,BidderB,5.7,3583.15
61,BidderB,6.1,10557.67
61,BidderB,6.2,4886.09
61,BidderB,6.3,695.15
61,BidderB,6.4,5839.07
61,BidderB,6.5,5525.84
61,BidderB,6.6,3921.54
61,BidderB,6.7,6725.98
61,BidderB,6.8,5711.94
61,BidderB,6.9,6831.83
61,BidderB,7.1,5803.76
61,BidderB,7.2,8569.3
61,BidderB,7.3,2499.5
61,BidderB,7.4,10107.78
61,BidderB,7.5,7949.05
61,BidderB,7.6,9837.98
61,BidderB,7.7,4404.09
61,BidderB,8.1,4322.49
61,BidderB,8.2,8202.91
61,BidderB,8.3,7562.35
61,BidderB,8.4,4610.59
61,BidderB,8.5,3461.47
61,BidderB,8.6,9563.76
61,BidderB,8.7,8733.73
61,BidderB,8.8,9323.23
61,BidderB,8.9,804.96
61,BidderC,1.1,2498.84
61,BidderC,1.2,5139.74
61,BidderC,1.3,8295.75
61,BidderC,1.4,4503.5
61,BidderC,1.5,10859.46
61,BidderC,1.6,8666.88
61,BidderC,1.7,443.81
61,BidderC,1.8,2817.29
61,BidderC,2.1,7408.5
61,BidderC,2.2,9636.31
61,BidderC,2.3,4858.09
61,BidderC,2.4,7414.1
61,BidderC,2.5,10847.24
61,BidderC,2.6,6541.29
61,BidderC,2.7,1252.3
61,BidderC,2.8,4453.06
61,BidderC,2.9,8407.61
61,BidderC,3.1,1213.66
61,BidderC,3.2,9728.9
61,BidderC,3.3,223.06
61,BidderC,3.4,4331.26
61,BidderC,3.5,6485.97
61,BidderC,3.6,3079.87
61,BidderC,4.1,3873.66
61,BidderC,4.2,3287.33
61,BidderC,4.3,2751.91
61,BidderC,4.4,11479.64
61,BidderC,4.5,5614.45
61,BidderC,4.6,7800.64
61,BidderC,4.7,1126.99
61,BidderC,5.1,376.86
61,BidderC,5.2,6867.63
61,BidderC,5.3,241.17
61,BidderC,5.4,698.51
61,BidderC,5.5,986.67
61,BidderC,5.6,1285.86
61,BidderC,5.7,3716.89
61,BidderC,6.1,11109.89
61,BidderC,6.2,4607.55
61,BidderC,6.3,708.76
61,BidderC,6.4,6026.95
61,BidderC,6.5,5225.93
61,BidderC,6.6,3897.99
61,BidderC,6.7,6417.31
61,BidderC,6.8,5726.18
61,BidderC,6.9,5997.58
61,BidderC,7.1,5506.67
61,BidderC,7.2,8250.39
61,BidderC,7.3,2702.69
61,BidderC,7.4,8920.81
61,BidderC,7.5,7533.2
61,BidderC,7.6,11125.34
61,BidderC,7.7,4312.16
61,BidderC,8.1,4220.55
61,BidderC,8.2,7062.51
61,BidderC,8.3,7358.72
61,BidderC,8.4,3830.5
61,BidderC,8.5,3778.76
61,BidderC,8.6,8877.4
61,BidderC,8.7,8462.58
61,BidderC,8.8,8589.9
61,BidderC,8.9,814.17
70,BidderA,1.1,8998.11
70,BidderA,1.2,2577.0
70,BidderA,1.3,8686.43
70,BidderA,1.4,10996.85
70,BidderA,1.5,2191.12
70,BidderA,1.6,1074.48
70,BidderA,2.1,6604.47
70,BidderA,2.2,1736.32
70,BidderA,2.3,6995.95
70,BidderA,2.4,4554.28
70,BidderB,1.1,8245.27
70,BidderB,1.2,2666.22
70,BidderB,1.3,9434.92
70,BidderB,1.4,11881.13
70,BidderB,1.5,1909.88
70,BidderB,1.6,1110.63
70,BidderB,2.1,6660.32
70,BidderB,2.2,1861.19
70,BidderB,2.3,7890.95
70,BidderB,2.4,4720.21
70,BidderC,1.1,8245.27
70,BidderC,1.2,2609.14
70,BidderC,1.3,9389.19
70,BidderC,1.4,11441.45
70,BidderC,1.5,2119.36
70,BidderC,1.6,967.35
70,BidderC,2.1,7078.14
70,BidderC,2.2,1736.32
70,BidderC,2.3,7238.15
70,BidderC,2.4,4912.0
73,BidderA,1.1,3199.25
73,BidderA,1.2,5387.51
73,BidderA,1.3,10091.24
73,BidderA,1.4,2774.24
73,BidderA,1.5,189.02
73,BidderA,1.6,5371.93
73,BidderA,1.7,6653.0
73,BidderA,1.8,2866.26
73,BidderA,1.9,10875.19
73,BidderA,2.1,493.97
73,BidderA,2.2,8990.76
73,BidderB,1.1,3405.76
73,BidderB,1.2,6020.49
73,BidderB,1.3,10162.42
73,BidderB,1.4,2692.47
73,BidderB,1.5,188.06
73,BidderB,1.6,5729.41
73,BidderB,1.7,6969.03
73,BidderB,1.8,2453.5
73,BidderB,1.9,10634.41
73,BidderB,2.1,505.5
73,BidderB,2.2,8001.49
73,BidderC,1.1,3065.31
73,BidderC,1.2,5603.13
73,BidderC,1.3,10176.29
73,BidderC,1.4,2805.88
73,BidderC,1.5,184.46
73,BidderC,1.6,5364.31
73,BidderC,1.7,6360.7
73,BidderC,1.8,2520.97
73,BidderC,1.9,10489.39
73,BidderC,2.1,455.65
73,BidderC,2.2,8284.76
78,BidderA,1.1,5692.62
78,BidderA,1.2,2460.51
78,BidderA,1.3,1280.92
78,BidderA,1.4,7292.5
78,BidderA,2.1,3600.68
78,BidderA,2.2,2366.19
78,BidderA,3.1,6187.11
78,BidderA,3.2,1626.7
78,BidderA,3.3,8472.62
78,BidderA,3.4,2248.4
78,BidderA,3.5,1326.02
78,BidderA,4.1,1189.25
78,BidderA,4.2,8117.14
78,BidderA,4.3,10101.43
78,BidderA,4.4,10048.97
78,BidderA,4.5,8066.93
78,BidderA,5.1,1477.86
78,BidderA,5.2,8628.66
78,BidderA,5.3,8104.04
78,BidderA,5.4,7775.77
78,BidderA,5.5,8864.71
78,BidderA,5.6,4320.6
78,BidderA,5.7,5316.55
78,BidderA,5.8,3336.16
78,BidderA,6.1,6340.9
78,BidderA,6.2,8182.93
78,BidderA,6.3,8464.98
78,BidderA,6.4,430.45
78,BidderA,6.5,6845.15
78,BidderA,6.6,6580.98
78,BidderA,6.7,3432.74
78,BidderA,6.8,5531.18
78,BidderA,6.9,223.13
78,BidderA,7.1,4332.6
78,BidderA,7.2,3053.11
78,BidderA,7.3,9378.84
78,BidderA,7.4,10690.74
78,BidderA,7.5,11540.76
78,BidderA,7.6,3481.56
78,BidderA,7.7,1658.63
78,BidderA,7.8,2908.29
78,BidderA,7.9,6782.94
78,BidderB,1.1,5335.57
78,BidderB,1.2,2170.12
78,BidderB,1.3,1412.14
78,BidderB,1.4,8370.93
78,BidderB,2.1,3301.92
78,BidderB,2.2,2676.16
78,BidderB,3.1,7135.58
78,BidderB,3.2,1604.44
78,BidderB,3.3,8374.7
78,BidderB,3.4,2224.7
78,BidderB,3.5,1359.92
78,BidderB,4.1,1380.27
78,BidderB,4.2,8278.79
78,BidderB,4.3,9331.44
78,BidderB,4.4,11558.55
78,BidderB,4.5,7963.07
78,BidderB,5.1,1459.95
78,BidderB,5.2,9191.97
78,BidderB,5.3,9474.22
78,BidderB,5.4,8245.25
78,BidderB,5.5,8734.5
78,BidderB,5.6,4644.11
78,BidderB,5.7,5853.72
78,BidderB,5.8,3174.03
78,BidderB,6.1,7137.05
78,BidderB,6.2,8706.48
78,BidderB,6.3,8778.54
78,BidderB,6.4,464.94
78,BidderB,6.5,7188.62
78,BidderB,6.6,7514.09
78,BidderB,6.7,3051.51
78,BidderB,6.8,6874.84
78,BidderB,6.9,205.42
78,BidderB,7.1,4373.27
78,BidderB,7.2,3166.41
78,BidderB,7.3,8702.71
78,BidderB,7.4,9780.5
78,BidderB,7.5,11498.16
78,BidderB,7.6,3565.51
78,BidderB,7.7,1777.38
78,BidderB,7.8,2661.39
78,BidderB,7.9,7232.04
78,BidderC,1.1,6562.86
78,BidderC,1.2,2305.94
78,BidderC,1.3,1274.15
78,BidderC,1.4,7878.54
78,BidderC,2.1,3533.76
78,BidderC,2.2,2646.37
78,BidderC,3.1,6347.64
78,BidderC,3.2,1649.96
78,BidderC,3.3,8357.1
78,BidderC,3.4,2350.35
78,BidderC,3.5,1399.91
78,BidderC,4.1,1202.54
78,BidderC,4.2,7783.65
78,BidderC,4.3,10646.24
78,BidderC,4.4,11291.04
78,BidderC,4.5,7587.5
78,BidderC,5.1,1357.4
78,BidderC,5.2,9499.24
78,BidderC,5.3,9321.21
78,BidderC,5.4,8278.61
78,BidderC,5.5,8191.01
78,BidderC,5.6,4052.73
78,BidderC,5.7,5858.18
78,BidderC,5.8,3654.62
78,BidderC,6.1,6213.09
78,BidderC,6.2,8566.32
78,BidderC,6.3,8998.08
78,BidderC,6.4,498.78
78,BidderC,6.5,5955.16
78,BidderC,6.6,6883.98
78,BidderC,6.7,3113.75
78,BidderC,6.8,6880.11
78,BidderC,6.9,197.21
78,BidderC,7.1,4352.2
78,BidderC,7.2,3517.32
78,BidderC,7.3,9316.48
78,BidderC,7.4,10006.21
78,BidderC,7.5,10529.73
78,BidderC,7.6,3554.91
78,BidderC,7.7,1965.38
78,BidderC,7.8,2575.53
78,BidderC,7.9,7227.88
83,BidderA,1.1,5047.63
83,BidderA,1.2,8066.13
83,BidderA,1.3,5173.57
83,BidderA,2.1,3893.42
83,BidderA,2.2,6165.92
83,BidderA,2.3,2881.83
83,BidderA,2.4,1378.96
83,BidderA,2.5,1264.58
83,BidderA,3.1,7018.74
83,BidderA,3.2,7139.89
83,BidderA,3.3,7188.11
83,BidderA,3.4,4757.28
83,BidderA,3.5,2379.46
83,BidderA,3.6,4232.54
83,BidderB,1.1,5105.35
83,BidderB,1.2,7514.18
83,BidderB,1.3,4699.19
83,BidderB,2.1,4142.19
83,BidderB,2.2,7108.4
83,BidderB,2.3,2899.47
83,BidderB,2.4,1366.32
83,BidderB,2.5,1470.62
83,BidderB,3.1,7342.26
83,BidderB,3.2,8063.55
83,BidderB,3.3,7188.11
83,BidderB,3.4,4669.57
83,BidderB,3.5,2127.92
83,BidderB,3.6,4269.64
83,BidderC,1.1,5445.71
83,BidderC,1.2,7302.31
83,BidderC,1.3,4768.24
83,BidderC,2.1,4170.65
83,BidderC,2.2,6174.59
83,BidderC,2.3,2977.85
83,BidderC,2.4,1450.88
83,BidderC,2.5,1418.1
83,BidderC,3.1,7359.61
83,BidderC,3.2,7841.09
83,BidderC,3.3,7129.63
83,BidderC,3.4,4873.71
83,BidderC,3.5,2215.47
83,BidderC,3.6,3849.59
87,BidderA,1.1,7902.58
87,BidderA,1.2,3447.04
87,BidderA,1.3,1232.13
87,BidderA,2.1,10162.74
87,BidderA,2.2,9516.92
87,BidderA,2.3,5670.61
87,BidderA,2.4,2384.82
87,BidderA,2.5,4631.87
87,BidderA,2.6,8162.11
87,BidderA,2.7,1661.96
87,BidderA,3.1,3356.77
87,BidderA,3.2,1644.78
87,BidderA,3.3,1999.12
87,BidderB,1.1,8814.03
87,BidderB,1.2,3439.32
87,BidderB,1.3,1272.62
87,BidderB,2.1,11772.64
87,BidderB,2.2,8795.89
87,BidderB,2.3,5808.62
87,BidderB,2.4,2464.47
87,BidderB,2.5,4099.74
87,BidderB,2.6,8466.16
87,BidderB,2.7,1662.15
87,BidderB,3.1,3178.61
87,BidderB,3.2,1560.82
87,BidderB,3.3,2004.06
87,BidderC,1.1,8273.35
87,BidderC,1.2,3595.7
87,BidderC,1.3,1165.82
87,BidderC,2.1,11100.56
87,BidderC,2.2,10071.65
87,BidderC,2.3,4864.8
87,BidderC,2.4,2286.79
87,BidderC,2.5,4464.19
87,BidderC,2.6,9389.52
87,BidderC,2.7,1916.01
87,BidderC,3.1,3370.49
87,BidderC,3.2,1688.36
87,BidderC,3.3,1776.0
93,BidderA,1.1,3247.93
93,BidderA,1.2,4652.68
93,BidderA,1.3,4812.22
93,BidderA,1.4,9109.95
93,BidderA,1.5,7644.56
93,BidderA,2.1,9959.94
93,BidderA,2.2,9042.04
93,BidderB,1.1,3772.04
93,BidderB,1.2,4568.6
93,BidderB,1.3,5008.34
93,BidderB,1.4,9638.98
93,BidderB,1.5,6623.09
93,BidderB,2.1,8798.83
93,BidderB,2.2,9538.24
93,BidderC,1.1,3409.03
93,BidderC,1.2,4339.16
93,BidderC,1.3,5112.58
93,BidderC,1.4,8979.39
93,BidderC,1.5,6660.94
93,BidderC,2.1,8880.35
93,BidderC,2.2,9917.75
99,BidderA,1.1,11036.19
99,BidderA,1.2,4996.22
99,BidderA,1.3,9270.49
99,BidderA,2.1,3429.76
99,BidderA,2.2,7927.19
99,BidderA,2.3,3833.99
99,BidderA,2.4,10069.05
99,BidderA,2.5,2412.54
99,BidderA,2.6,1194.9
99,BidderA,2.7,3848.81
99,BidderA,3.1,3643.99
99,BidderA,3.2,6622.01
99,BidderA,4.1,9490.51
99,BidderA,4.2,8717.6
99,BidderB,1.1,10181.87
99,BidderB,1.2,4257.83
99,BidderB,1.3,9856.11
99,BidderB,2.1,3872.39
99,BidderB,2.2,6994.34
99,BidderB,2.3,3526.15
99,BidderB,2.4,9012.28
99,BidderB,2.5,2600.23
99,BidderB,2.6,1239.28
99,BidderB,2.7,3672.76
99,BidderB,3.1,3318.11
99,BidderB,3.2,6738.47
99,BidderB,4.1,9238.18
99,BidderB,4.2,10379.74
99,BidderC,1.1,10107.24
99,BidderC,1.2,4285.09
99,BidderC,1.3,9358.1
99,BidderC,2.1,3909.11
99,BidderC,2.2,7465.33
99,BidderC,2.3,3408.36
99,BidderC,2.4,9039.43
99,BidderC,2.5,2575.69
99,BidderC,2.6,1148.29
99,BidderC,2.7,3605.55
99,BidderC,3.1,3851.08
99,BidderC,3.2,6035.44
99,BidderC,4.1,10540.02
99,BidderC,4.2,10652.52
'''

