####################################################################################
#                                                                                  #
# Copyright (c) 2005 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This deteclet is part of the Bid Rigging wizards library.                     #
# It requires a license and is not open source.                                    #
#                                                                                  #
####################################################################################
# UPDATES:
# 20 Jul 2006   Updated Header
# 03 NOV 2005  First version of the wizard
#
####################################################################################
#
#   STATUS: Review
#           
#
#    IDEAS/QUESTIONS/NOTES:
#
####################################################################################
from picalo import *

DETECTLET_STANDARD = 1.0

wizard = '''
<wizard>
  <page>
    Please select which table contains the bidding data.
    It should look similar to the example input data (see
    the previous page), and it should have columns for
    the contracts, bidder, line items, and amount.
    
    If you need to join two or more tables together to get
    your data into this form, close the wizard and do so.
    <parameter type="Table" variable="table"/>
  </page>
  <page>
    Your input table should contain the records for many
    contracts (if it were only one contract, you could do this 
    analysis in a spreadsheet).  This wizard will stratify
    the table into one table for each contract, then run 
    the analysis on each contract table.
    
    Which column should the table be stratified by?  This column
    should uniquely identify each contract/project.  It might be 
    called "id", "project id", "contract id", or something similar.
    <parameter type="Column" table="table" variable="stratifycol"/>
  </page>
  <page>
    Each contract will have two or more bidders.  Which column
    specifies the bidder?  This column should contain data
    similar to a "bidder id" or "vendor id".
    <parameter type="Column" table="table" variable="biddercol"/>
  </page>
  <page>
    Each contract will have two or more line items that specify
    the individual items on the contract.  For example, a
    contract might have one line item for labor, one line item
    for construction equipment, and another line item for
    a certain type of product.
    
    This number should be identical from bid to bid.  For example,
    one line item might be for labor.  If this item has item number
    5 in bid A, it should also have item number 5 in bid B.  The wizard
    will use this field to match items in project bids.
    
    Which column specifies the line item?  This column should contain
    data similar to a "line item id", "bid item id", "item number",
    etc.
    <parameter type="Column" table="table" variable="lineitemcol"/>
  </page>
  <page>
    Which column contains the bid price of each line item?  This is not
    the total price of the entire bid, but the price of individual 
    line items.
    <parameter type="Column" table="table" variable="amountcol"/>
  </page>
</wizard>
'''

RESULTS_TEXT = '''\
    The displayed table shows the percentage of identical item amounts
    between bidders on a given project.  Bidders should not normally
    have identical amounts because the point of asking for bids is to evaluate 
    the lowest cost bidder.  If bid line items are identical, it means
    the two bidder probably colluded with one another.
    
    One type of scheme this might indicate is where one vendor creates
    two or three of the bids.  After creating the first bid, the vendor
    simply copies the bid and changes a few line items to form the second
    bid.  This produces a difference in overall cost, but the bids
    are identical in many items.
    
    One exception to this is where line item amounts are published
    or standardized in an industry. 
    
    When looking at these results, look for any vendors who have 
    a high percentage of identical line item amounts.  The
    PercentageIdentical column gives this percentage.  The table 
    is sorted by this column, so look at the records at the top
    of the table. Those with a high percentage
    of identical item amounts may be fraudulent.
'''


def run(table, stratifycol, biddercol, lineitemcol, amountcol):
  '''Most organizations require multiple bids to increase competition between
     suppliers.  Purchasers may want to circumvent this control when they
     are getting kickbacks from a certain vendor, have family or friends they
     want to funnel work to, or for many other reasons.
     
     One of the ways to fabricate extra bids is to copy and then
     slightly change a single bid to look like multiple bids from different
     companies.  A purchaser loads the real bid into a spreadsheet and
     creates a new column for the second bid.  He or she copies the values
     from the real bid column and then changes one or two line items with
     large amounts.  Since the line items with large amounts are changed,
     the bid total is significantly different than the real bid.
     
     This wizard searches for bids with line items that have many line items
     as exact copies of other bids.  It assumes that you have 
     potentially thousands of projects with 2 or more bids on each project.  
     
     The wizard goes through the following process:
     - Stratifies the file into a table for each project.  
     - Calculates the number of line items between competing bids
       that are identical.
     - Creates a "percentage similar" value by dividing the number
       of identical lines by the number of total lines in the bid.
     - Sorts the results by the percentage.
       
     Projects with a significant number of identical line items
     may be fraudulent.
     
     The example input table contains the following four columns:
     - Contract: The contract id
     - Bidder: The bidder id - there are three bidders on each of the
       example bids.
     - Item: The line item id.  Most contracts have line items.
     - Amount: The amount of each line item.
     
     In the example dataset, contract #1 has been fabricated.  The line items
     between bids are nearly identical.
  '''
  # validate the data
  assert stratifycol != biddercol != lineitemcol != amountcol, 'The stratification, bidder, line item, and amount columns must be different.  Please ensure you haven\'t selected the same column for two of these items.'
  
  # run the analysis
  groups = Grouping.stratify_by_value(table, stratifycol)
  results = Table([
    ( 'Contract',            unicode ),
    ( 'Bidder1',             unicode ),
    ( 'Bidder2',             unicode ),
    ( 'PercentageIdentical', number),
  ])
  for counter, group in enumerate(groups):
    show_progress('Analyzing (' + str(counter) + ' of ' + str(len(groups)) + ')...', float(counter) / len(groups))
    xtab = Crosstable.pivot(group, biddercol, lineitemcol, ['sum(group.column("' + amountcol + '"))'])[:-1]  # remove the totals row at the bottom
    cols = xtab.get_column_names()
    for i in range(1, len(cols)-1):  # compare every col against every other col
      for j in range(i+1, len(cols)-1):
        identical = 0
        for rec in xtab:
          try:
            if rec[cols[i]] == rec[cols[j]]:
              identical += 1
          except:
            pass
        rec = results.append()
        rec['Contract'] = group.startvalue
        rec['Bidder1'] = cols[i]
        rec['Bidder2'] = cols[j]
        rec['PercentageIdentical'] = round(float(identical) / len(xtab) * 100, 2)

  show_progress("Sorting results...", 0)
  Simple.sort(results, False, 'PercentageIdentical')
  clear_progress()
  return results, RESULTS_TEXT


def example_input():
  import StringIO  # to emulate a file for load_csv
  table = load_csv(StringIO.StringIO(csvdata))
  table.set_type('Contract', int)
  table.set_type('Amount', number)
  return table



# This next part is not required, but it's easier to put the example data
# directly in this file so I don't have to worry about directories.
csvdata = '''\
Contract,Bidder,Item,Amount
1,BidderA,1.1,9908.01
1,BidderA,1.2,4147.38
1,BidderA,1.3,2675.48
1,BidderA,2.1,7612.40
1,BidderA,2.2,4444.31
1,BidderA,3.1,9712.15
1,BidderA,3.2,3784.11
1,BidderB,1.1,11024.03
1,BidderB,1.2,4147.38
1,BidderB,1.3,2675.48
1,BidderB,2.1,7710.04
1,BidderB,2.2,4444.31
1,BidderB,3.1,9712.15
1,BidderB,3.2,3784.11
1,BidderC,1.1,13682.51
1,BidderC,1.2,4147.38
1,BidderC,1.3,2675.48
1,BidderC,2.1,9610.04
1,BidderC,2.2,4444.31
1,BidderC,3.1,9712.15
1,BidderC,3.2,3784.11
9,BidderA,1.1,4725.31
9,BidderA,1.2,386.36
9,BidderA,1.3,4429.6
9,BidderA,1.4,10300.63
9,BidderA,1.5,2963.85
9,BidderA,1.6,7206.11
9,BidderA,1.7,6565.34
9,BidderA,1.8,2450.38
9,BidderA,2.1,5137.69
9,BidderA,2.2,5383.91
9,BidderA,2.3,6851.32
9,BidderA,2.4,4899.27
9,BidderA,2.5,9314.92
9,BidderA,2.6,2340.83
9,BidderA,2.7,6573.02
9,BidderA,2.8,8794.33
9,BidderA,3.1,6903.6
9,BidderA,3.2,866.89
9,BidderA,3.3,305.22
9,BidderA,3.4,3168.95
9,BidderA,3.5,6979.86
9,BidderA,3.6,4052.23
9,BidderA,3.7,6734.51
9,BidderA,4.1,3418.17
9,BidderA,4.2,3164.31
9,BidderA,4.3,4172.04
9,BidderA,4.4,7946.88
9,BidderA,5.1,5829.54
9,BidderA,5.2,10537.48
9,BidderA,5.3,7681.75
9,BidderA,5.4,10973.42
9,BidderA,5.5,10398.93
9,BidderA,5.6,1842.88
9,BidderA,5.7,5588.91
9,BidderA,5.8,11256.57
9,BidderA,6.1,8823.86
9,BidderA,6.2,3588.83
9,BidderA,6.3,2469.82
9,BidderA,6.4,141.33
9,BidderA,6.5,1730.16
9,BidderA,6.6,8291.06
9,BidderA,6.7,1905.45
9,BidderA,6.8,1133.94
9,BidderB,1.1,5007.19
9,BidderB,1.2,390.05
9,BidderB,1.3,4606.82
9,BidderB,1.4,10598.91
9,BidderB,1.5,3267.26
9,BidderB,1.6,7678.35
9,BidderB,1.7,8061.86
9,BidderB,1.8,2689.86
9,BidderB,2.1,5580.32
9,BidderB,2.2,4721.73
9,BidderB,2.3,5914.4
9,BidderB,2.4,4995.45
9,BidderB,2.5,8893.46
9,BidderB,2.6,2278.73
9,BidderB,2.7,6753.88
9,BidderB,2.8,8292.58
9,BidderB,3.1,6998.37
9,BidderB,3.2,962.02
9,BidderB,3.3,308.66
9,BidderB,3.4,3314.07
9,BidderB,3.5,7295.16
9,BidderB,3.6,4074.41
9,BidderB,3.7,7811.22
9,BidderB,4.1,3554.74
9,BidderB,4.2,3217.02
9,BidderB,4.3,4245.18
9,BidderB,4.4,8288.39
9,BidderB,5.1,5726.55
9,BidderB,5.2,10033.01
9,BidderB,5.3,6871.48
9,BidderB,5.4,9748.64
9,BidderB,5.5,9305.53
9,BidderB,5.6,1967.47
9,BidderB,5.7,5244.7
9,BidderB,5.8,10590.45
9,BidderB,6.1,8396.95
9,BidderB,6.2,3624.89
9,BidderB,6.3,2340.19
9,BidderB,6.4,126.4
9,BidderB,6.5,1858.06
9,BidderB,6.6,8929.97
9,BidderB,6.7,1871.86
9,BidderB,6.8,1198.32
9,BidderC,1.1,5192.32
9,BidderC,1.2,412.33
9,BidderC,1.3,4825.95
9,BidderC,1.4,10506.58
9,BidderC,1.5,3149.19
9,BidderC,1.6,7559.98
9,BidderC,1.7,7563.36
9,BidderC,1.8,2610.1
9,BidderC,2.1,5239.49
9,BidderC,2.2,5317.8
9,BidderC,2.3,5517.1
9,BidderC,2.4,4600.38
9,BidderC,2.5,9484.5
9,BidderC,2.6,2638.02
9,BidderC,2.7,6236.39
9,BidderC,2.8,8892.29
9,BidderC,3.1,6813.29
9,BidderC,3.2,897.51
9,BidderC,3.3,307.84
9,BidderC,3.4,3420.3
9,BidderC,3.5,7247.53
9,BidderC,3.6,4024.56
9,BidderC,3.7,6844.37
9,BidderC,4.1,3447.06
9,BidderC,4.2,3273.02
9,BidderC,4.3,4313.52
9,BidderC,4.4,7897.32
9,BidderC,5.1,5963.05
9,BidderC,5.2,10231.31
9,BidderC,5.3,6995.77
9,BidderC,5.4,10227.52
9,BidderC,5.5,8344.54
9,BidderC,5.6,1955.1
9,BidderC,5.7,5066.07
9,BidderC,5.8,11438.57
9,BidderC,6.1,8705.62
9,BidderC,6.2,3539.02
9,BidderC,6.3,2271.18
9,BidderC,6.4,144.29
9,BidderC,6.5,1559.17
9,BidderC,6.6,9067.69
9,BidderC,6.7,1782.04
9,BidderC,6.8,1303.89
13,BidderA,1.1,4851.56
13,BidderA,1.2,5675.24
13,BidderA,1.3,7304.02
13,BidderA,1.4,8903.29
13,BidderA,1.5,9056.14
13,BidderA,1.6,6699.82
13,BidderA,2.1,2271.95
13,BidderA,2.2,4114.38
13,BidderA,2.3,8327.2
13,BidderA,2.4,1296.5
13,BidderA,2.5,1697.19
13,BidderA,2.6,3053.76
13,BidderA,3.1,10410.07
13,BidderA,3.2,9617.29
13,BidderA,3.3,5285.87
13,BidderA,3.4,8841.36
13,BidderA,4.1,9712.66
13,BidderA,4.2,480.15
13,BidderA,4.3,1350.61
13,BidderA,4.4,5211.45
13,BidderA,5.1,2508.61
13,BidderA,5.2,3224.88
13,BidderA,5.3,7391.32
13,BidderA,6.1,2941.53
13,BidderA,6.2,7939.02
13,BidderA,6.3,5518.9
13,BidderA,6.4,670.43
13,BidderA,6.5,6370.01
13,BidderA,6.6,632.94
13,BidderA,6.7,9974.69
13,BidderA,6.8,8335.26
13,BidderA,7.1,5678.15
13,BidderA,7.2,6358.89
13,BidderA,7.3,5621.26
13,BidderA,8.1,10990.6
13,BidderA,8.2,5435.46
13,BidderB,1.1,5056.85
13,BidderB,1.2,5501.66
13,BidderB,1.3,6459.88
13,BidderB,1.4,8975.2
13,BidderB,1.5,8486.22
13,BidderB,1.6,6616.11
13,BidderB,2.1,2728.03
13,BidderB,2.2,4494.7
13,BidderB,2.3,6567.95
13,BidderB,2.4,1297.76
13,BidderB,2.5,1860.8
13,BidderB,2.6,2842.93
13,BidderB,3.1,11246.7
13,BidderB,3.2,10079.84
13,BidderB,3.3,5446.8
13,BidderB,3.4,9604.16
13,BidderB,4.1,10221.05
13,BidderB,4.2,456.78
13,BidderB,4.3,1419.73
13,BidderB,4.4,5697.33
13,BidderB,5.1,2480.86
13,BidderB,5.2,3204.37
13,BidderB,5.3,6714.8
13,BidderB,6.1,3137.04
13,BidderB,6.2,8197.02
13,BidderB,6.3,5068.9
13,BidderB,6.4,719.4
13,BidderB,6.5,6858.35
13,BidderB,6.6,681.62
13,BidderB,6.7,9459.96
13,BidderB,6.8,7741.96
13,BidderB,7.1,6172.41
13,BidderB,7.2,6258.83
13,BidderB,7.3,5975.52
13,BidderB,8.1,10089.37
13,BidderB,8.2,5896.1
13,BidderC,1.1,5152.22
13,BidderC,1.2,5951.69
13,BidderC,1.3,7245.17
13,BidderC,1.4,9180.84
13,BidderC,1.5,9400.4
13,BidderC,1.6,6695.76
13,BidderC,2.1,2530.01
13,BidderC,2.2,4422.63
13,BidderC,2.3,7451.89
13,BidderC,2.4,1212.38
13,BidderC,2.5,1622.46
13,BidderC,2.6,2939.68
13,BidderC,3.1,10065.88
13,BidderC,3.2,9688.98
13,BidderC,3.3,5500.89
13,BidderC,3.4,9501.09
13,BidderC,4.1,9826.39
13,BidderC,4.2,461.32
13,BidderC,4.3,1534.6
13,BidderC,4.4,5077.83
13,BidderC,5.1,2404.98
13,BidderC,5.2,3380.41
13,BidderC,5.3,6110.96
13,BidderC,6.1,3192.76
13,BidderC,6.2,8597.45
13,BidderC,6.3,5728.01
13,BidderC,6.4,692.68
13,BidderC,6.5,5603.0
13,BidderC,6.6,719.72
13,BidderC,6.7,9326.84
13,BidderC,6.8,7623.2
13,BidderC,7.1,6945.19
13,BidderC,7.2,6624.67
13,BidderC,7.3,6538.68
13,BidderC,8.1,11120.34
13,BidderC,8.2,5684.76
17,BidderA,1.1,2404.92
17,BidderA,1.2,9120.24
17,BidderA,1.3,5393.56
17,BidderA,1.4,3847.9
17,BidderA,1.5,8425.33
17,BidderA,2.1,296.77
17,BidderA,2.2,123.83
17,BidderA,2.3,5425.72
17,BidderA,2.4,4726.93
17,BidderA,2.5,10493.03
17,BidderA,2.6,1911.18
17,BidderA,2.7,7363.42
17,BidderA,2.8,4919.24
17,BidderA,2.9,7579.52
17,BidderB,1.1,2456.27
17,BidderB,1.2,8913.09
17,BidderB,1.3,5100.71
17,BidderB,1.4,4092.24
17,BidderB,1.5,8519.31
17,BidderB,2.1,265.02
17,BidderB,2.2,122.01
17,BidderB,2.3,5879.68
17,BidderB,2.4,4548.88
17,BidderB,2.5,10015.87
17,BidderB,2.6,1770.53
17,BidderB,2.7,7434.79
17,BidderB,2.8,5204.09
17,BidderB,2.9,6969.73
17,BidderC,1.1,2323.43
17,BidderC,1.2,8191.27
17,BidderC,1.3,5776.91
17,BidderC,1.4,3721.07
17,BidderC,1.5,7991.32
17,BidderC,2.1,300.0
17,BidderC,2.2,124.76
17,BidderC,2.3,5176.08
17,BidderC,2.4,4056.6
17,BidderC,2.5,9592.94
17,BidderC,2.6,1728.5
17,BidderC,2.7,7418.81
17,BidderC,2.8,4339.75
17,BidderC,2.9,6571.27
24,BidderA,1.1,2073.53
24,BidderA,1.2,9790.59
24,BidderA,1.3,7902.52
24,BidderA,1.4,4523.7
24,BidderA,1.5,8939.42
24,BidderA,1.6,5504.4
24,BidderA,1.7,876.44
24,BidderA,1.8,4361.5
24,BidderA,1.9,8366.03
24,BidderA,2.1,5723.15
24,BidderA,2.2,3406.05
24,BidderA,2.3,5723.06
24,BidderA,2.4,9014.65
24,BidderA,2.5,11531.9
24,BidderA,2.6,5867.87
24,BidderA,2.7,6399.3
24,BidderA,2.8,9087.09
24,BidderA,3.1,3308.8
24,BidderA,3.2,6177.43
24,BidderA,3.3,8284.72
24,BidderA,3.4,2247.8
24,BidderA,3.5,131.5
24,BidderA,3.6,6122.47
24,BidderA,3.7,4406.28
24,BidderA,3.8,9578.2
24,BidderA,4.1,1016.07
24,BidderA,4.2,6266.46
24,BidderA,4.3,10659.88
24,BidderA,5.1,3863.73
24,BidderA,5.2,852.65
24,BidderA,5.3,3342.53
24,BidderA,5.4,6729.58
24,BidderA,5.5,10387.84
24,BidderA,5.6,1925.68
24,BidderA,6.1,8171.74
24,BidderA,6.2,3153.69
24,BidderA,6.3,7165.46
24,BidderA,6.4,7387.05
24,BidderA,6.5,7424.54
24,BidderA,6.6,8123.76
24,BidderA,6.7,6759.94
24,BidderA,6.8,2333.66
24,BidderA,6.9,9900.82
24,BidderA,7.1,9356.32
24,BidderA,7.2,8003.3
24,BidderA,7.3,7457.46
24,BidderA,7.4,4875.44
24,BidderA,7.5,7719.16
24,BidderA,7.6,9220.12
24,BidderA,7.7,3896.48
24,BidderA,7.8,3240.37
24,BidderA,8.1,9231.32
24,BidderA,8.2,8454.52
24,BidderA,8.3,4966.78
24,BidderA,8.4,6435.59
24,BidderA,8.5,6340.32
24,BidderA,8.6,2571.75
24,BidderA,8.7,4571.43
24,BidderA,8.8,3987.07
24,BidderB,1.1,2050.44
24,BidderB,1.2,9283.04
24,BidderB,1.3,7823.56
24,BidderB,1.4,4073.25
24,BidderB,1.5,8885.62
24,BidderB,1.6,5093.12
24,BidderB,1.7,1012.1
24,BidderB,1.8,4508.68
24,BidderB,1.9,8172.85
24,BidderB,2.1,5317.14
24,BidderB,2.2,3549.27
24,BidderB,2.3,6893.23
24,BidderB,2.4,9131.88
24,BidderB,2.5,11057.5
24,BidderB,2.6,5904.19
24,BidderB,2.7,6126.99
24,BidderB,2.8,8458.18
24,BidderB,3.1,3337.05
24,BidderB,3.2,6592.47
24,BidderB,3.3,8821.98
24,BidderB,3.4,2332.74
24,BidderB,3.5,130.16
24,BidderB,3.6,6790.25
24,BidderB,3.7,4897.28
24,BidderB,3.8,9310.81
24,BidderB,4.1,1008.21
24,BidderB,4.2,6342.99
24,BidderB,4.3,9905.72
24,BidderB,5.1,4355.91
24,BidderB,5.2,799.16
24,BidderB,5.3,3526.81
24,BidderB,5.4,6286.36
24,BidderB,5.5,9282.3
24,BidderB,5.6,1610.99
24,BidderB,6.1,7059.26
24,BidderB,6.2,2955.39
24,BidderB,6.3,5963.44
24,BidderB,6.4,7401.46
24,BidderB,6.5,6822.04
24,BidderB,6.6,7793.4
24,BidderB,6.7,6677.26
24,BidderB,6.8,2432.31
24,BidderB,6.9,9393.33
24,BidderB,7.1,8998.55
24,BidderB,7.2,8542.97
24,BidderB,7.3,7074.33
24,BidderB,7.4,4646.76
24,BidderB,7.5,7767.6
24,BidderB,7.6,9760.88
24,BidderB,7.7,3790.24
24,BidderB,7.8,3277.69
24,BidderB,8.1,9340.89
24,BidderB,8.2,8963.93
24,BidderB,8.3,5051.84
24,BidderB,8.4,5605.0
24,BidderB,8.5,6005.71
24,BidderB,8.6,2634.47
24,BidderB,8.7,4976.27
24,BidderB,8.8,3843.03
24,BidderC,1.1,1892.03
24,BidderC,1.2,10361.99
24,BidderC,1.3,7153.86
24,BidderC,1.4,3524.56
24,BidderC,1.5,7821.45
24,BidderC,1.6,5040.02
24,BidderC,1.7,867.39
24,BidderC,1.8,4393.18
24,BidderC,1.9,7173.43
24,BidderC,2.1,5317.87
24,BidderC,2.2,3437.72
24,BidderC,2.3,6168.29
24,BidderC,2.4,9400.68
24,BidderC,2.5,11669.89
24,BidderC,2.6,5686.18
24,BidderC,2.7,6404.16
24,BidderC,2.8,8578.71
24,BidderC,3.1,3352.27
24,BidderC,3.2,6320.98
24,BidderC,3.3,8366.22
24,BidderC,3.4,2446.8
24,BidderC,3.5,131.87
24,BidderC,3.6,6774.93
24,BidderC,3.7,4909.56
24,BidderC,3.8,9253.93
24,BidderC,4.1,1048.99
24,BidderC,4.2,6124.29
24,BidderC,4.3,10266.44
24,BidderC,5.1,4004.56
24,BidderC,5.2,807.31
24,BidderC,5.3,3453.22
24,BidderC,5.4,6632.73
24,BidderC,5.5,9620.14
24,BidderC,5.6,1929.71
24,BidderC,6.1,7343.62
24,BidderC,6.2,2935.59
24,BidderC,6.3,6475.67
24,BidderC,6.4,7367.54
24,BidderC,6.5,7598.43
24,BidderC,6.6,7522.7
24,BidderC,6.7,7317.46
24,BidderC,6.8,2570.28
24,BidderC,6.9,9841.63
24,BidderC,7.1,8804.31
24,BidderC,7.2,8172.64
24,BidderC,7.3,7484.83
24,BidderC,7.4,4530.77
24,BidderC,7.5,8249.82
24,BidderC,7.6,10693.31
24,BidderC,7.7,3561.04
24,BidderC,7.8,3125.89
24,BidderC,8.1,9848.53
24,BidderC,8.2,7824.52
24,BidderC,8.3,4916.69
24,BidderC,8.4,6767.13
24,BidderC,8.5,6672.47
24,BidderC,8.6,2461.35
24,BidderC,8.7,4863.77
24,BidderC,8.8,4179.63
28,BidderA,1.1,8709.01
28,BidderA,1.2,10991.64
28,BidderA,1.3,1608.87
28,BidderA,1.4,831.58
28,BidderA,2.1,7094.3
28,BidderA,2.2,9097.58
28,BidderA,2.3,4207.58
28,BidderA,2.4,1962.96
28,BidderA,2.5,6194.79
28,BidderA,2.6,6729.42
28,BidderA,2.7,4011.67
28,BidderA,2.8,4099.33
28,BidderA,3.1,5035.83
28,BidderA,3.2,9831.26
28,BidderA,3.3,3495.6
28,BidderA,3.4,8041.26
28,BidderA,3.5,9518.96
28,BidderA,3.6,2476.64
28,BidderA,3.7,8103.03
28,BidderA,4.1,2348.67
28,BidderA,4.2,9552.93
28,BidderA,4.3,8367.78
28,BidderA,5.1,9409.65
28,BidderA,5.2,5996.61
28,BidderA,6.1,2750.79
28,BidderA,6.2,4116.94
28,BidderA,6.3,6271.92
28,BidderA,6.4,3292.19
28,BidderA,6.5,1305.15
28,BidderA,6.6,2682.43
28,BidderA,6.7,3092.77
28,BidderB,1.1,7655.58
28,BidderB,1.2,11298.93
28,BidderB,1.3,1628.43
28,BidderB,1.4,856.28
28,BidderB,2.1,7155.53
28,BidderB,2.2,10154.1
28,BidderB,2.3,4004.64
28,BidderB,2.4,1813.58
28,BidderB,2.5,6748.05
28,BidderB,2.6,7748.7
28,BidderB,2.7,3743.35
28,BidderB,2.8,4234.56
28,BidderB,3.1,4987.44
28,BidderB,3.2,9822.33
28,BidderB,3.3,3068.49
28,BidderB,3.4,7367.85
28,BidderB,3.5,9219.81
28,BidderB,3.6,2659.56
28,BidderB,3.7,8174.18
28,BidderB,4.1,2179.15
28,BidderB,4.2,10197.9
28,BidderB,4.3,9338.41
28,BidderB,5.1,9826.33
28,BidderB,5.2,6659.02
28,BidderB,6.1,3030.14
28,BidderB,6.2,4585.8
28,BidderB,6.3,6118.89
28,BidderB,6.4,3300.91
28,BidderB,6.5,1360.08
28,BidderB,6.6,2767.05
28,BidderB,6.7,2736.97
28,BidderC,1.1,8494.16
28,BidderC,1.2,9294.74
28,BidderC,1.3,1527.49
28,BidderC,1.4,750.87
28,BidderC,2.1,6918.53
28,BidderC,2.2,8945.81
28,BidderC,2.3,3976.55
28,BidderC,2.4,1855.06
28,BidderC,2.5,6251.56
28,BidderC,2.6,7554.66
28,BidderC,2.7,3941.61
28,BidderC,2.8,4014.64
28,BidderC,3.1,4752.84
28,BidderC,3.2,10612.75
28,BidderC,3.3,3374.12
28,BidderC,3.4,7153.92
28,BidderC,3.5,8816.71
28,BidderC,3.6,2778.87
28,BidderC,3.7,8448.7
28,BidderC,4.1,2330.7
28,BidderC,4.2,9333.23
28,BidderC,4.3,9470.76
28,BidderC,5.1,10717.93
28,BidderC,5.2,7254.56
28,BidderC,6.1,3030.14
28,BidderC,6.2,4279.49
28,BidderC,6.3,7022.13
28,BidderC,6.4,3288.13
28,BidderC,6.5,1268.32
28,BidderC,6.6,2989.17
28,BidderC,6.7,2952.5
34,BidderA,1.1,7671.66
34,BidderA,1.2,7023.85
34,BidderA,1.3,5526.92
34,BidderA,1.4,3735.56
34,BidderA,1.5,7709.47
34,BidderA,2.1,8457.35
34,BidderA,2.2,3522.27
34,BidderA,2.3,2448.72
34,BidderA,2.4,4379.77
34,BidderA,2.5,1128.99
34,BidderA,2.6,4850.24
34,BidderA,2.7,1615.02
34,BidderA,2.8,419.16
34,BidderA,3.1,4211.99
34,BidderA,3.2,4438.96
34,BidderA,3.3,5231.16
34,BidderA,3.4,7473.17
34,BidderA,3.5,2159.41
34,BidderA,3.6,7773.35
34,BidderA,3.7,1246.17
34,BidderA,3.8,8039.18
34,BidderA,4.1,8819.25
34,BidderA,4.2,1509.12
34,BidderA,5.1,3691.93
34,BidderA,5.2,5306.2
34,BidderA,5.3,3142.09
34,BidderA,5.4,5546.12
34,BidderA,5.5,1141.26
34,BidderA,5.6,2692.83
34,BidderA,5.7,9780.51
34,BidderA,5.8,6364.81
34,BidderA,5.9,311.86
34,BidderA,6.1,4591.14
34,BidderA,6.2,6332.77
34,BidderA,6.3,10386.27
34,BidderA,6.4,2871.03
34,BidderA,6.5,1620.23
34,BidderA,6.6,9603.97
34,BidderA,6.7,11737.7
34,BidderA,7.1,3607.8
34,BidderA,7.2,6002.03
34,BidderA,7.3,9600.46
34,BidderA,7.4,8983.86
34,BidderA,7.5,4249.54
34,BidderA,7.6,5571.25
34,BidderA,7.7,10137.21
34,BidderA,7.8,10707.18
34,BidderA,8.1,2232.49
34,BidderA,8.2,9433.9
34,BidderA,8.3,2895.05
34,BidderA,8.4,5925.51
34,BidderA,8.5,2134.73
34,BidderA,8.6,1494.83
34,BidderA,8.7,6793.79
34,BidderA,8.8,4880.75
34,BidderB,1.1,7855.11
34,BidderB,1.2,7449.33
34,BidderB,1.3,5103.87
34,BidderB,1.4,4052.49
34,BidderB,1.5,7829.94
34,BidderB,2.1,8312.76
34,BidderB,2.2,3973.38
34,BidderB,2.3,2401.39
34,BidderB,2.4,4416.48
34,BidderB,2.5,995.44
34,BidderB,2.6,4978.87
34,BidderB,2.7,1580.95
34,BidderB,2.8,402.13
34,BidderB,3.1,4484.4
34,BidderB,3.2,4625.11
34,BidderB,3.3,4743.16
34,BidderB,3.4,6871.65
34,BidderB,3.5,2252.52
34,BidderB,3.6,7773.41
34,BidderB,3.7,1269.68
34,BidderB,3.8,7683.51
34,BidderB,4.1,9263.66
34,BidderB,4.2,1580.32
34,BidderB,5.1,3611.84
34,BidderB,5.2,5492.77
34,BidderB,5.3,2992.96
34,BidderB,5.4,5177.19
34,BidderB,5.5,1246.05
34,BidderB,5.6,2731.01
34,BidderB,5.7,9591.68
34,BidderB,5.8,6514.1
34,BidderB,5.9,309.8
34,BidderB,6.1,4292.39
34,BidderB,6.2,6183.49
34,BidderB,6.3,10639.17
34,BidderB,6.4,2743.67
34,BidderB,6.5,1630.11
34,BidderB,6.6,9938.27
34,BidderB,6.7,11101.05
34,BidderB,7.1,3516.03
34,BidderB,7.2,6132.25
34,BidderB,7.3,10306.58
34,BidderB,7.4,9740.2
34,BidderB,7.5,4236.06
34,BidderB,7.6,5889.62
34,BidderB,7.7,9628.39
34,BidderB,7.8,10191.12
34,BidderB,8.1,2307.64
34,BidderB,8.2,9217.4
34,BidderB,8.3,2840.61
34,BidderB,8.4,6190.48
34,BidderB,8.5,2139.83
34,BidderB,8.6,1407.99
34,BidderB,8.7,6923.72
34,BidderB,8.8,4666.48
34,BidderC,1.1,7279.57
34,BidderC,1.2,6781.59
34,BidderC,1.3,5267.0
34,BidderC,1.4,3729.41
34,BidderC,1.5,7815.23
34,BidderC,2.1,8061.13
34,BidderC,2.2,4436.95
34,BidderC,2.3,2145.21
34,BidderC,2.4,4647.89
34,BidderC,2.5,1086.1
34,BidderC,2.6,4464.89
34,BidderC,2.7,1617.65
34,BidderC,2.8,458.66
34,BidderC,3.1,4176.1
34,BidderC,3.2,4322.34
34,BidderC,3.3,5045.18
34,BidderC,3.4,7174.21
34,BidderC,3.5,2005.9
34,BidderC,3.6,7674.18
34,BidderC,3.7,1309.78
34,BidderC,3.8,8547.67
34,BidderC,4.1,9069.57
34,BidderC,4.2,1489.74
34,BidderC,5.1,3490.49
34,BidderC,5.2,4503.78
34,BidderC,5.3,3336.45
34,BidderC,5.4,4888.0
34,BidderC,5.5,1112.68
34,BidderC,5.6,2942.47
34,BidderC,5.7,9518.73
34,BidderC,5.8,7148.32
34,BidderC,5.9,288.37
34,BidderC,6.1,4259.9
34,BidderC,6.2,6489.3
34,BidderC,6.3,9873.25
34,BidderC,6.4,2835.04
34,BidderC,6.5,1507.75
34,BidderC,6.6,9985.97
34,BidderC,6.7,10579.34
34,BidderC,7.1,3405.73
34,BidderC,7.2,5996.27
34,BidderC,7.3,9923.2
34,BidderC,7.4,9176.5
34,BidderC,7.5,4461.35
34,BidderC,7.6,5716.24
34,BidderC,7.7,10195.93
34,BidderC,7.8,10806.51
34,BidderC,8.1,2271.56
34,BidderC,8.2,9282.67
34,BidderC,8.3,2571.53
34,BidderC,8.4,6707.93
34,BidderC,8.5,2124.86
34,BidderC,8.6,1418.95
34,BidderC,8.7,7169.63
34,BidderC,8.8,5263.8
36,BidderA,1.1,6686.05
36,BidderA,1.2,3359.18
36,BidderA,1.3,1988.16
36,BidderA,1.4,2920.35
36,BidderA,1.5,8477.31
36,BidderA,1.6,10765.05
36,BidderA,1.7,10909.19
36,BidderA,1.8,6430.45
36,BidderA,1.9,241.49
36,BidderA,2.1,9150.45
36,BidderA,2.2,1963.58
36,BidderA,2.3,8633.76
36,BidderA,2.4,9985.02
36,BidderA,2.5,10873.34
36,BidderB,1.1,7473.05
36,BidderB,1.2,3431.31
36,BidderB,1.3,2032.65
36,BidderB,1.4,2919.54
36,BidderB,1.5,8407.22
36,BidderB,1.6,11444.5
36,BidderB,1.7,11428.34
36,BidderB,1.8,6809.09
36,BidderB,1.9,270.73
36,BidderB,2.1,9127.53
36,BidderB,2.2,1903.09
36,BidderB,2.3,9203.96
36,BidderB,2.4,10446.69
36,BidderB,2.5,9775.73
36,BidderC,1.1,7038.97
36,BidderC,1.2,3541.23
36,BidderC,1.3,1956.35
36,BidderC,1.4,2853.17
36,BidderC,1.5,9064.25
36,BidderC,1.6,10399.07
36,BidderC,1.7,11504.26
36,BidderC,1.8,6816.59
36,BidderC,1.9,264.49
36,BidderC,2.1,9744.08
36,BidderC,2.2,2073.21
36,BidderC,2.3,8538.9
36,BidderC,2.4,10076.15
36,BidderC,2.5,10358.28
44,BidderA,1.1,3663.56
44,BidderA,1.2,8265.46
44,BidderA,1.3,2475.36
44,BidderA,1.4,7268.22
44,BidderA,1.5,7486.57
44,BidderA,2.1,8483.91
44,BidderA,2.2,6892.58
44,BidderA,2.3,3675.62
44,BidderA,2.4,6248.89
44,BidderA,2.5,6762.41
44,BidderA,2.6,7663.38
44,BidderA,3.1,7659.58
44,BidderA,3.2,5047.72
44,BidderA,3.3,959.79
44,BidderA,3.4,10146.5
44,BidderA,4.1,10027.82
44,BidderA,4.2,7114.11
44,BidderA,4.3,4453.8
44,BidderA,4.4,7025.05
44,BidderA,4.5,4058.6
44,BidderA,5.1,2622.31
44,BidderA,5.2,4796.25
44,BidderA,5.3,10517.51
44,BidderA,6.1,1070.52
44,BidderA,6.2,7336.65
44,BidderA,6.3,6332.09
44,BidderA,6.4,4706.93
44,BidderA,7.1,2492.01
44,BidderA,7.2,6915.52
44,BidderA,7.3,8978.66
44,BidderA,7.4,6845.87
44,BidderA,7.5,2933.36
44,BidderA,7.6,1567.31
44,BidderA,8.1,4554.4
44,BidderA,8.2,8871.78
44,BidderA,8.3,6391.79
44,BidderA,8.4,7636.59
44,BidderA,8.5,510.92
44,BidderA,8.6,1188.09
44,BidderA,8.7,783.92
44,BidderB,1.1,3529.04
44,BidderB,1.2,8958.48
44,BidderB,1.3,2431.7
44,BidderB,1.4,6744.89
44,BidderB,1.5,8472.98
44,BidderB,2.1,7508.62
44,BidderB,2.2,7343.96
44,BidderB,2.3,3339.58
44,BidderB,2.4,5385.85
44,BidderB,2.5,5811.85
44,BidderB,2.6,7677.15
44,BidderB,3.1,7201.34
44,BidderB,3.2,5316.83
44,BidderB,3.3,1095.85
44,BidderB,3.4,9316.0
44,BidderB,4.1,9334.38
44,BidderB,4.2,7514.69
44,BidderB,4.3,4809.66
44,BidderB,4.4,7272.29
44,BidderB,4.5,3926.02
44,BidderB,5.1,2727.97
44,BidderB,5.2,4404.6
44,BidderB,5.3,10302.34
44,BidderB,6.1,929.04
44,BidderB,6.2,7366.44
44,BidderB,6.3,6820.57
44,BidderB,6.4,5155.71
44,BidderB,7.1,2341.1
44,BidderB,7.2,5634.39
44,BidderB,7.3,8724.49
44,BidderB,7.4,7150.79
44,BidderB,7.5,3325.76
44,BidderB,7.6,1411.24
44,BidderB,8.1,4541.15
44,BidderB,8.2,10172.2
44,BidderB,8.3,5965.9
44,BidderB,8.4,7139.37
44,BidderB,8.5,535.41
44,BidderB,8.6,1186.78
44,BidderB,8.7,765.43
44,BidderC,1.1,3720.82
44,BidderC,1.2,8020.42
44,BidderC,1.3,2209.46
44,BidderC,1.4,6859.66
44,BidderC,1.5,7607.06
44,BidderC,2.1,7025.39
44,BidderC,2.2,7150.08
44,BidderC,2.3,3371.3
44,BidderC,2.4,5360.3
44,BidderC,2.5,6656.31
44,BidderC,2.6,7439.7
44,BidderC,3.1,8254.51
44,BidderC,3.2,5451.73
44,BidderC,3.3,1084.39
44,BidderC,3.4,10146.5
44,BidderC,4.1,9791.54
44,BidderC,4.2,7083.1
44,BidderC,4.3,4227.06
44,BidderC,4.4,6372.54
44,BidderC,4.5,4172.58
44,BidderC,5.1,2719.48
44,BidderC,5.2,4157.97
44,BidderC,5.3,10283.85
44,BidderC,6.1,1091.22
44,BidderC,6.2,7896.67
44,BidderC,6.3,6961.23
44,BidderC,6.4,5043.17
44,BidderC,7.1,2412.03
44,BidderC,7.2,6058.04
44,BidderC,7.3,7981.72
44,BidderC,7.4,6717.36
44,BidderC,7.5,3334.88
44,BidderC,7.6,1565.96
44,BidderC,8.1,4734.67
44,BidderC,8.2,9881.35
44,BidderC,8.3,6552.5
44,BidderC,8.4,7308.46
44,BidderC,8.5,567.91
44,BidderC,8.6,1297.18
44,BidderC,8.7,718.59
49,BidderA,1.1,5811.45
49,BidderA,1.2,9241.22
49,BidderA,1.3,6695.03
49,BidderA,1.4,5829.9
49,BidderA,1.5,3054.99
49,BidderA,1.6,6642.52
49,BidderA,1.7,5032.34
49,BidderA,1.8,11257.56
49,BidderA,1.9,6587.11
49,BidderA,2.1,2527.12
49,BidderA,2.2,8514.16
49,BidderA,3.1,456.84
49,BidderA,3.2,9586.12
49,BidderA,3.3,10294.22
49,BidderA,3.4,2190.79
49,BidderA,3.5,7231.78
49,BidderA,3.6,9809.69
49,BidderA,4.1,9574.13
49,BidderA,4.2,10256.1
49,BidderA,4.3,10215.95
49,BidderA,4.4,5576.06
49,BidderA,5.1,8839.79
49,BidderA,5.2,8810.35
49,BidderA,5.3,9766.97
49,BidderA,6.1,11255.23
49,BidderA,6.2,2476.32
49,BidderA,6.3,2928.4
49,BidderA,6.4,3607.22
49,BidderA,6.5,732.44
49,BidderA,6.6,4150.11
49,BidderA,6.7,8471.18
49,BidderA,7.1,2108.49
49,BidderA,7.2,4569.65
49,BidderA,7.3,11282.55
49,BidderA,7.4,5551.77
49,BidderA,7.5,486.08
49,BidderA,7.6,8009.62
49,BidderA,7.7,1474.76
49,BidderA,7.8,2735.69
49,BidderA,7.9,1551.2
49,BidderB,1.1,5977.17
49,BidderB,1.2,10107.38
49,BidderB,1.3,6930.38
49,BidderB,1.4,6162.47
49,BidderB,1.5,3345.64
49,BidderB,1.6,7096.61
49,BidderB,1.7,4876.79
49,BidderB,1.8,10400.04
49,BidderB,1.9,6237.46
49,BidderB,2.1,2513.01
49,BidderB,2.2,9844.67
49,BidderB,3.1,453.66
49,BidderB,3.2,9814.2
49,BidderB,3.3,10841.07
49,BidderB,3.4,2132.36
49,BidderB,3.5,7422.1
49,BidderB,3.6,10117.5
49,BidderB,4.1,10127.52
49,BidderB,4.2,11579.39
49,BidderB,4.3,9997.87
49,BidderB,4.4,4860.83
49,BidderB,5.1,7838.45
49,BidderB,5.2,8266.82
49,BidderB,5.3,9798.12
49,BidderB,6.1,10913.58
49,BidderB,6.2,2707.83
49,BidderB,6.3,3220.99
49,BidderB,6.4,3393.69
49,BidderB,6.5,749.22
49,BidderB,6.6,4382.96
49,BidderB,6.7,7912.73
49,BidderB,7.1,2095.69
49,BidderB,7.2,4361.32
49,BidderB,7.3,10211.34
49,BidderB,7.4,5646.62
49,BidderB,7.5,471.24
49,BidderB,7.6,8169.72
49,BidderB,7.7,1442.05
49,BidderB,7.8,2699.09
49,BidderB,7.9,1617.82
49,BidderC,1.1,6189.49
49,BidderC,1.2,9398.79
49,BidderC,1.3,7458.01
49,BidderC,1.4,6255.75
49,BidderC,1.5,3369.57
49,BidderC,1.6,7071.09
49,BidderC,1.7,4830.45
49,BidderC,1.8,10888.99
49,BidderC,1.9,5936.77
49,BidderC,2.1,2716.79
49,BidderC,2.2,10058.73
49,BidderC,3.1,458.27
49,BidderC,3.2,9303.58
49,BidderC,3.3,11337.64
49,BidderC,3.4,2167.66
49,BidderC,3.5,8069.35
49,BidderC,3.6,10524.76
49,BidderC,4.1,9240.02
49,BidderC,4.2,10727.36
49,BidderC,4.3,9136.39
49,BidderC,4.4,5624.27
49,BidderC,5.1,8365.52
49,BidderC,5.2,7623.15
49,BidderC,5.3,9838.02
49,BidderC,6.1,10879.66
49,BidderC,6.2,2768.97
49,BidderC,6.3,3375.74
49,BidderC,6.4,3361.6
49,BidderC,6.5,797.17
49,BidderC,6.6,4601.24
49,BidderC,6.7,7798.78
49,BidderC,7.1,2108.49
49,BidderC,7.2,4190.96
49,BidderC,7.3,10078.1
49,BidderC,7.4,5366.86
49,BidderC,7.5,476.27
49,BidderC,7.6,7887.39
49,BidderC,7.7,1478.52
49,BidderC,7.8,2541.09
49,BidderC,7.9,1593.69
53,BidderA,1.1,4781.35
53,BidderA,1.2,1213.58
53,BidderA,1.3,4539.54
53,BidderA,1.4,6925.78
53,BidderA,2.1,6369.21
53,BidderA,2.2,424.48
53,BidderA,2.3,2962.48
53,BidderA,3.1,3586.64
53,BidderA,3.2,7582.57
53,BidderA,3.3,8683.55
53,BidderA,3.4,5789.34
53,BidderA,3.5,3778.29
53,BidderA,3.6,10140.74
53,BidderA,3.7,3252.31
53,BidderA,4.1,8732.7
53,BidderA,4.2,6742.99
53,BidderA,4.3,11016.01
53,BidderA,4.4,4462.31
53,BidderA,4.5,5124.13
53,BidderA,4.6,9102.75
53,BidderA,4.7,7662.61
53,BidderA,4.8,1261.38
53,BidderA,4.9,1391.11
53,BidderA,5.1,5940.38
53,BidderA,5.2,652.19
53,BidderA,5.3,1022.9
53,BidderA,5.4,2946.17
53,BidderA,5.5,6875.77
53,BidderA,5.6,2673.62
53,BidderA,6.1,6819.42
53,BidderA,6.2,437.36
53,BidderA,6.3,4636.76
53,BidderA,6.4,1780.4
53,BidderA,6.5,9399.63
53,BidderA,6.6,6168.41
53,BidderA,6.7,9929.92
53,BidderA,6.8,3176.92
53,BidderA,6.9,11194.19
53,BidderB,1.1,5088.29
53,BidderB,1.2,1157.1
53,BidderB,1.3,4622.42
53,BidderB,1.4,7131.57
53,BidderB,2.1,5133.38
53,BidderB,2.2,412.65
53,BidderB,2.3,2763.7
53,BidderB,3.1,4122.67
53,BidderB,3.2,8034.18
53,BidderB,3.3,8574.37
53,BidderB,3.4,5667.11
53,BidderB,3.5,4200.78
53,BidderB,3.6,10701.37
53,BidderB,3.7,3025.95
53,BidderB,4.1,8855.48
53,BidderB,4.2,6853.56
53,BidderB,4.3,11001.5
53,BidderB,4.4,3971.93
53,BidderB,4.5,5123.12
53,BidderB,4.6,8711.36
53,BidderB,4.7,7697.01
53,BidderB,4.8,1354.3
53,BidderB,4.9,1468.48
53,BidderB,5.1,6062.43
53,BidderB,5.2,592.8
53,BidderB,5.3,951.21
53,BidderB,5.4,2790.81
53,BidderB,5.5,6490.1
53,BidderB,5.6,2698.12
53,BidderB,6.1,6502.28
53,BidderB,6.2,454.01
53,BidderB,6.3,4769.13
53,BidderB,6.4,1995.29
53,BidderB,6.5,9903.25
53,BidderB,6.6,5870.09
53,BidderB,6.7,9858.32
53,BidderB,6.8,3180.16
53,BidderB,6.9,10512.57
53,BidderC,1.1,5259.19
53,BidderC,1.2,1074.82
53,BidderC,1.3,4522.5
53,BidderC,1.4,8062.65
53,BidderC,2.1,5714.44
53,BidderC,2.2,435.52
53,BidderC,2.3,2635.19
53,BidderC,3.1,3745.07
53,BidderC,3.2,7185.28
53,BidderC,3.3,8869.99
53,BidderC,3.4,6144.6
53,BidderC,3.5,4240.31
53,BidderC,3.6,11592.16
53,BidderC,3.7,3076.6
53,BidderC,4.1,8684.87
53,BidderC,4.2,7065.65
53,BidderC,4.3,11770.13
53,BidderC,4.4,4173.71
53,BidderC,4.5,4838.62
53,BidderC,4.6,8946.58
53,BidderC,4.7,8021.19
53,BidderC,4.8,1419.69
53,BidderC,4.9,1360.74
53,BidderC,5.1,5512.33
53,BidderC,5.2,608.0
53,BidderC,5.3,987.4
53,BidderC,5.4,2900.71
53,BidderC,5.5,7026.51
53,BidderC,5.6,2555.46
53,BidderC,6.1,6702.37
53,BidderC,6.2,437.57
53,BidderC,6.3,4497.0
53,BidderC,6.4,1730.74
53,BidderC,6.5,8921.79
53,BidderC,6.6,5971.12
53,BidderC,6.7,10982.67
53,BidderC,6.8,3257.12
53,BidderC,6.9,11792.63
57,BidderA,1.1,751.61
57,BidderA,1.2,10212.49
57,BidderA,1.3,3957.99
57,BidderA,1.4,618.67
57,BidderA,1.5,1448.32
57,BidderA,1.6,6838.99
57,BidderA,1.7,9151.93
57,BidderA,2.1,1077.64
57,BidderA,2.2,9646.6
57,BidderA,2.3,8605.86
57,BidderA,2.4,6752.34
57,BidderA,2.5,4286.21
57,BidderA,2.6,4949.79
57,BidderA,3.1,10427.55
57,BidderA,3.2,9770.9
57,BidderA,3.3,4894.95
57,BidderA,3.4,9168.27
57,BidderA,4.1,9362.56
57,BidderA,4.2,202.13
57,BidderA,4.3,8040.46
57,BidderA,4.4,119.94
57,BidderA,5.1,434.25
57,BidderA,5.2,5656.84
57,BidderA,5.3,7124.18
57,BidderA,5.4,5620.21
57,BidderA,5.5,10030.7
57,BidderA,5.6,7982.91
57,BidderA,5.7,8656.03
57,BidderA,5.8,901.57
57,BidderA,6.1,3151.47
57,BidderA,6.2,4570.75
57,BidderA,6.3,4512.18
57,BidderA,7.1,9472.89
57,BidderA,7.2,9620.14
57,BidderA,7.3,1068.98
57,BidderA,7.4,5379.84
57,BidderA,7.5,9417.3
57,BidderA,8.1,7283.35
57,BidderA,8.2,2669.52
57,BidderA,8.3,2059.56
57,BidderA,8.4,4141.26
57,BidderA,8.5,1075.59
57,BidderA,8.6,835.48
57,BidderA,8.7,7683.25
57,BidderA,8.8,6840.26
57,BidderA,8.9,8228.9
57,BidderA,9.1,2027.32
57,BidderA,9.2,2087.64
57,BidderB,1.1,658.41
57,BidderB,1.2,8845.31
57,BidderB,1.3,3629.99
57,BidderB,1.4,611.35
57,BidderB,1.5,1460.44
57,BidderB,1.6,7087.54
57,BidderB,1.7,9844.81
57,BidderB,2.1,1058.1
57,BidderB,2.2,8712.53
57,BidderB,2.3,8327.48
57,BidderB,2.4,6883.09
57,BidderB,2.5,4551.59
57,BidderB,2.6,4660.65
57,BidderB,3.1,9451.52
57,BidderB,3.2,9763.39
57,BidderB,3.3,4853.79
57,BidderB,3.4,9988.49
57,BidderB,4.1,9510.96
57,BidderB,4.2,192.67
57,BidderB,4.3,7403.7
57,BidderB,4.4,124.83
57,BidderB,5.1,391.54
57,BidderB,5.2,5696.82
57,BidderB,5.3,7735.83
57,BidderB,5.4,6516.71
57,BidderB,5.5,9854.14
57,BidderB,5.6,7873.97
57,BidderB,5.7,8811.3
57,BidderB,5.8,859.61
57,BidderB,6.1,2834.33
57,BidderB,6.2,5053.49
57,BidderB,6.3,4412.51
57,BidderB,7.1,9902.23
57,BidderB,7.2,9868.05
57,BidderB,7.3,863.2
57,BidderB,7.4,5262.26
57,BidderB,7.5,9758.27
57,BidderB,8.1,7952.07
57,BidderB,8.2,2734.21
57,BidderB,8.3,2255.87
57,BidderB,8.4,4875.58
57,BidderB,8.5,1010.92
57,BidderB,8.6,821.34
57,BidderB,8.7,7395.05
57,BidderB,8.8,7871.67
57,BidderB,8.9,8236.61
57,BidderB,9.1,2338.2
57,BidderB,9.2,2111.66
57,BidderC,1.1,668.43
57,BidderC,1.2,9071.12
57,BidderC,1.3,3902.98
57,BidderC,1.4,674.16
57,BidderC,1.5,1508.78
57,BidderC,1.6,7087.54
57,BidderC,1.7,10638.38
57,BidderC,2.1,1119.77
57,BidderC,2.2,8373.53
57,BidderC,2.3,7950.74
57,BidderC,2.4,6213.6
57,BidderC,2.5,3753.02
57,BidderC,2.6,4516.34
57,BidderC,3.1,10495.74
57,BidderC,3.2,8645.47
57,BidderC,3.3,4773.24
57,BidderC,3.4,8780.27
57,BidderC,4.1,9737.58
57,BidderC,4.2,181.67
57,BidderC,4.3,8219.6
57,BidderC,4.4,132.1
57,BidderC,5.1,404.2
57,BidderC,5.2,6380.54
57,BidderC,5.3,7781.91
57,BidderC,5.4,5852.02
57,BidderC,5.5,10158.5
57,BidderC,5.6,8512.72
57,BidderC,5.7,8328.49
57,BidderC,5.8,961.12
57,BidderC,6.1,2905.74
57,BidderC,6.2,4880.87
57,BidderC,6.3,4221.97
57,BidderC,7.1,9732.36
57,BidderC,7.2,9395.25
57,BidderC,7.3,1060.68
57,BidderC,7.4,5549.0
57,BidderC,7.5,9515.8
57,BidderC,8.1,6691.44
57,BidderC,8.2,2845.89
57,BidderC,8.3,2287.99
57,BidderC,8.4,4896.22
57,BidderC,8.5,1080.34
57,BidderC,8.6,888.78
57,BidderC,8.7,7854.98
57,BidderC,8.8,7315.88
57,BidderC,8.9,9234.78
57,BidderC,9.1,2025.8
57,BidderC,9.2,2151.44
64,BidderA,1.1,4402.39
64,BidderA,1.2,7089.06
64,BidderA,1.3,1245.04
64,BidderA,2.1,374.08
64,BidderA,2.2,6712.77
64,BidderA,2.3,3726.21
64,BidderA,2.4,2323.14
64,BidderA,3.1,9210.36
64,BidderA,3.2,1846.34
64,BidderA,3.3,2364.35
64,BidderA,3.4,8062.88
64,BidderA,3.5,2681.45
64,BidderA,3.6,8599.72
64,BidderA,3.7,5078.68
64,BidderA,3.8,3061.22
64,BidderA,4.1,7117.26
64,BidderA,4.2,6440.14
64,BidderA,4.3,4061.29
64,BidderA,4.4,7646.08
64,BidderA,4.5,1795.81
64,BidderA,5.1,9447.78
64,BidderA,5.2,7303.77
64,BidderA,5.3,9746.61
64,BidderA,5.4,6471.25
64,BidderA,5.5,6337.85
64,BidderA,5.6,7897.89
64,BidderA,5.7,11071.48
64,BidderA,5.8,5195.59
64,BidderA,5.9,465.21
64,BidderA,6.1,2773.96
64,BidderA,6.2,3755.06
64,BidderA,7.1,6708.56
64,BidderA,7.2,1423.01
64,BidderA,7.3,7820.78
64,BidderA,8.1,10156.14
64,BidderA,8.2,1009.07
64,BidderA,8.3,4560.71
64,BidderA,8.4,4013.82
64,BidderA,8.5,7149.19
64,BidderA,8.6,9562.36
64,BidderA,8.7,4941.2
64,BidderA,8.8,236.37
64,BidderA,8.9,6731.67
64,BidderA,9.1,5772.73
64,BidderA,9.2,4477.91
64,BidderA,9.3,9333.15
64,BidderA,9.4,3812.78
64,BidderA,9.5,4976.38
64,BidderA,9.6,3433.37
64,BidderA,9.7,623.98
64,BidderB,1.1,4401.35
64,BidderB,1.2,8195.54
64,BidderB,1.3,1430.38
64,BidderB,2.1,379.07
64,BidderB,2.2,7371.37
64,BidderB,2.3,4046.8
64,BidderB,2.4,2213.8
64,BidderB,3.1,9388.97
64,BidderB,3.2,1930.53
64,BidderB,3.3,2195.35
64,BidderB,3.4,7199.93
64,BidderB,3.5,2613.39
64,BidderB,3.6,8281.68
64,BidderB,3.7,4928.66
64,BidderB,3.8,2888.74
64,BidderB,4.1,6689.25
64,BidderB,4.2,6704.91
64,BidderB,4.3,4011.78
64,BidderB,4.4,7152.29
64,BidderB,4.5,1676.29
64,BidderB,5.1,9453.29
64,BidderB,5.2,6721.42
64,BidderB,5.3,9145.82
64,BidderB,5.4,6296.61
64,BidderB,5.5,6182.38
64,BidderB,5.6,7879.31
64,BidderB,5.7,10747.07
64,BidderB,5.8,4776.84
64,BidderB,5.9,494.82
64,BidderB,6.1,2849.18
64,BidderB,6.2,3637.97
64,BidderB,7.1,6783.91
64,BidderB,7.2,1380.94
64,BidderB,7.3,8207.59
64,BidderB,8.1,10511.8
64,BidderB,8.2,893.53
64,BidderB,8.3,4938.04
64,BidderB,8.4,4321.32
64,BidderB,8.5,7376.76
64,BidderB,8.6,10047.68
64,BidderB,8.7,5615.86
64,BidderB,8.8,250.69
64,BidderB,8.9,7250.8
64,BidderB,9.1,5674.6
64,BidderB,9.2,4257.35
64,BidderB,9.3,11535.59
64,BidderB,9.4,4869.33
64,BidderB,9.5,4885.01
64,BidderB,9.6,3353.95
64,BidderB,9.7,623.98
64,BidderC,1.1,4418.87
64,BidderC,1.2,7702.25
64,BidderC,1.3,1292.67
64,BidderC,2.1,417.21
64,BidderC,2.2,6886.13
64,BidderC,2.3,3955.1
64,BidderC,2.4,2228.39
64,BidderC,3.1,9927.72
64,BidderC,3.2,1910.78
64,BidderC,3.3,2235.84
64,BidderC,3.4,8946.2
64,BidderC,3.5,2599.46
64,BidderC,3.6,8768.74
64,BidderC,3.7,4938.88
64,BidderC,3.8,3272.44
64,BidderC,4.1,6695.48
64,BidderC,4.2,6308.94
64,BidderC,4.3,3881.48
64,BidderC,4.4,6982.62
64,BidderC,4.5,1720.51
64,BidderC,5.1,10072.75
64,BidderC,5.2,5791.78
64,BidderC,5.3,9721.97
64,BidderC,5.4,6589.76
64,BidderC,5.5,6520.26
64,BidderC,5.6,7788.92
64,BidderC,5.7,11209.53
64,BidderC,5.8,5264.32
64,BidderC,5.9,495.22
64,BidderC,6.1,2838.82
64,BidderC,6.2,3728.53
64,BidderC,7.1,7523.91
64,BidderC,7.2,1443.23
64,BidderC,7.3,8883.96
64,BidderC,8.1,10777.53
64,BidderC,8.2,929.13
64,BidderC,8.3,4477.68
64,BidderC,8.4,3456.11
64,BidderC,8.5,7012.9
64,BidderC,8.6,9438.71
64,BidderC,8.7,5150.94
64,BidderC,8.8,227.71
64,BidderC,8.9,7372.29
64,BidderC,9.1,5038.95
64,BidderC,9.2,4201.52
64,BidderC,9.3,10124.91
64,BidderC,9.4,4260.3
64,BidderC,9.5,4437.08
64,BidderC,9.6,3498.66
64,BidderC,9.7,534.15
68,BidderA,1.1,8615.61
68,BidderA,1.2,5609.15
68,BidderA,1.3,8744.63
68,BidderA,1.4,1979.05
68,BidderA,1.5,5744.5
68,BidderA,1.6,604.15
68,BidderA,1.7,9392.54
68,BidderA,1.8,4791.99
68,BidderA,2.1,5344.49
68,BidderA,2.2,2180.07
68,BidderA,2.3,6430.48
68,BidderA,2.4,5155.57
68,BidderA,2.5,3431.21
68,BidderA,2.6,5857.45
68,BidderA,3.1,4356.91
68,BidderA,3.2,1535.01
68,BidderA,4.1,2861.23
68,BidderA,4.2,6711.61
68,BidderA,4.3,7547.46
68,BidderA,5.1,10662.68
68,BidderA,5.2,5865.67
68,BidderA,5.3,8417.57
68,BidderA,6.1,9079.57
68,BidderA,6.2,4638.52
68,BidderA,6.3,2031.46
68,BidderA,6.4,1990.68
68,BidderA,7.1,9638.83
68,BidderA,7.2,2176.18
68,BidderB,1.1,8337.64
68,BidderB,1.2,5701.56
68,BidderB,1.3,10218.77
68,BidderB,1.4,2144.0
68,BidderB,1.5,5958.43
68,BidderB,1.6,649.06
68,BidderB,1.7,10016.19
68,BidderB,1.8,4911.03
68,BidderB,2.1,5310.39
68,BidderB,2.2,2175.81
68,BidderB,2.3,7074.73
68,BidderB,2.4,5394.89
68,BidderB,2.5,3230.93
68,BidderB,2.6,6427.69
68,BidderB,3.1,4709.66
68,BidderB,3.2,1400.29
68,BidderB,4.1,2962.3
68,BidderB,4.2,7056.58
68,BidderB,4.3,7372.97
68,BidderB,5.1,8906.79
68,BidderB,5.2,5638.09
68,BidderB,5.3,9466.64
68,BidderB,6.1,10162.05
68,BidderB,6.2,4500.55
68,BidderB,6.3,1844.63
68,BidderB,6.4,1879.88
68,BidderB,7.1,8332.82
68,BidderB,7.2,2061.43
68,BidderC,1.1,7602.57
68,BidderC,1.2,5605.98
68,BidderC,1.3,10532.14
68,BidderC,1.4,2127.91
68,BidderC,1.5,5976.93
68,BidderC,1.6,585.41
68,BidderC,1.7,8956.9
68,BidderC,1.8,5409.75
68,BidderC,2.1,5530.0
68,BidderC,2.2,2152.04
68,BidderC,2.3,7377.38
68,BidderC,2.4,5012.91
68,BidderC,2.5,3330.74
68,BidderC,2.6,6179.57
68,BidderC,3.1,4357.62
68,BidderC,3.2,1466.95
68,BidderC,4.1,2894.79
68,BidderC,4.2,6843.98
68,BidderC,4.3,7537.65
68,BidderC,5.1,11245.1
68,BidderC,5.2,5601.34
68,BidderC,5.3,9074.47
68,BidderC,6.1,8925.93
68,BidderC,6.2,4509.66
68,BidderC,6.3,1952.81
68,BidderC,6.4,1906.43
68,BidderC,7.1,8896.09
68,BidderC,7.2,2106.01
75,BidderA,1.1,8071.6
75,BidderA,1.2,5734.55
75,BidderA,1.3,2066.48
75,BidderA,1.4,1080.14
75,BidderA,2.1,9535.12
75,BidderA,2.2,7585.52
75,BidderA,2.3,4144.74
75,BidderA,2.4,6853.24
75,BidderA,2.5,365.03
75,BidderA,2.6,8483.47
75,BidderA,2.7,4580.78
75,BidderA,2.8,385.29
75,BidderA,2.9,9391.49
75,BidderA,3.1,3399.64
75,BidderA,3.2,9940.93
75,BidderA,3.3,8077.66
75,BidderA,3.4,5445.32
75,BidderA,3.5,8850.93
75,BidderA,3.6,1176.2
75,BidderA,3.7,8249.89
75,BidderA,4.1,3033.06
75,BidderA,4.2,3435.41
75,BidderA,4.3,10463.19
75,BidderA,4.4,6419.32
75,BidderA,4.5,4006.09
75,BidderA,4.6,11933.96
75,BidderA,4.7,816.17
75,BidderA,4.8,9053.75
75,BidderA,4.9,11143.05
75,BidderA,5.1,9195.93
75,BidderA,5.2,10428.59
75,BidderA,5.3,377.78
75,BidderA,5.4,7253.55
75,BidderA,5.5,10256.31
75,BidderA,5.6,7949.71
75,BidderA,5.7,3416.69
75,BidderA,5.8,8924.35
75,BidderB,1.1,8332.88
75,BidderB,1.2,6042.95
75,BidderB,1.3,2478.11
75,BidderB,1.4,1132.74
75,BidderB,2.1,9590.25
75,BidderB,2.2,8171.42
75,BidderB,2.3,4012.43
75,BidderB,2.4,6754.7
75,BidderB,2.5,413.6
75,BidderB,2.6,9066.75
75,BidderB,2.7,4349.33
75,BidderB,2.8,358.15
75,BidderB,2.9,9700.01
75,BidderB,3.1,3347.89
75,BidderB,3.2,8989.36
75,BidderB,3.3,8101.38
75,BidderB,3.4,5577.09
75,BidderB,3.5,10173.89
75,BidderB,3.6,1265.87
75,BidderB,3.7,8429.64
75,BidderB,4.1,3078.63
75,BidderB,4.2,3081.19
75,BidderB,4.3,11522.62
75,BidderB,4.4,7212.94
75,BidderB,4.5,4108.17
75,BidderB,4.6,11147.23
75,BidderB,4.7,886.14
75,BidderB,4.8,9899.24
75,BidderB,4.9,10953.54
75,BidderB,5.1,9183.55
75,BidderB,5.2,9446.37
75,BidderB,5.3,417.87
75,BidderB,5.4,7526.97
75,BidderB,5.5,10802.03
75,BidderB,5.6,6894.61
75,BidderB,5.7,3679.85
75,BidderB,5.8,8968.03
75,BidderC,1.1,8088.51
75,BidderC,1.2,5928.93
75,BidderC,1.3,2299.19
75,BidderC,1.4,1148.9
75,BidderC,2.1,9177.36
75,BidderC,2.2,7732.85
75,BidderC,2.3,3753.74
75,BidderC,2.4,6029.27
75,BidderC,2.5,414.93
75,BidderC,2.6,8833.05
75,BidderC,2.7,4375.9
75,BidderC,2.8,402.35
75,BidderC,2.9,10092.65
75,BidderC,3.1,3422.48
75,BidderC,3.2,9023.52
75,BidderC,3.3,7978.96
75,BidderC,3.4,5211.7
75,BidderC,3.5,9100.75
75,BidderC,3.6,1328.82
75,BidderC,3.7,9312.01
75,BidderC,4.1,2881.8
75,BidderC,4.2,3484.42
75,BidderC,4.3,10770.82
75,BidderC,4.4,6636.97
75,BidderC,4.5,3913.17
75,BidderC,4.6,10673.73
75,BidderC,4.7,903.05
75,BidderC,4.8,9371.96
75,BidderC,4.9,11550.71
75,BidderC,5.1,8887.09
75,BidderC,5.2,10076.78
75,BidderC,5.3,403.7
75,BidderC,5.4,7379.05
75,BidderC,5.5,10686.42
75,BidderC,5.6,8038.33
75,BidderC,5.7,3672.73
75,BidderC,5.8,8396.59
77,BidderA,1.1,5655.85
77,BidderA,1.2,5753.21
77,BidderA,1.3,1678.19
77,BidderA,1.4,1137.26
77,BidderA,1.5,10180.82
77,BidderA,1.6,7884.11
77,BidderA,2.1,6862.06
77,BidderA,2.2,362.86
77,BidderA,2.3,7683.39
77,BidderA,3.1,5813.94
77,BidderA,3.2,5153.92
77,BidderA,3.3,9183.34
77,BidderA,3.4,10126.13
77,BidderA,3.5,4196.85
77,BidderA,3.6,4847.34
77,BidderA,4.1,1020.11
77,BidderA,4.2,1992.41
77,BidderA,4.3,5762.44
77,BidderA,4.4,7199.62
77,BidderA,4.5,2466.32
77,BidderA,5.1,5284.54
77,BidderA,5.2,213.21
77,BidderA,6.1,3790.88
77,BidderA,6.2,6142.5
77,BidderA,6.3,8993.32
77,BidderA,6.4,6028.98
77,BidderA,6.5,3614.33
77,BidderA,6.6,9149.59
77,BidderA,6.7,8026.78
77,BidderA,6.8,3380.66
77,BidderA,6.9,7341.59
77,BidderA,7.1,5418.33
77,BidderA,7.2,3020.04
77,BidderA,7.3,10631.36
77,BidderA,7.4,11111.76
77,BidderA,7.5,2586.92
77,BidderA,8.1,7019.75
77,BidderA,8.2,2836.49
77,BidderA,8.3,5078.17
77,BidderB,1.1,5582.93
77,BidderB,1.2,5715.42
77,BidderB,1.3,1642.17
77,BidderB,1.4,1056.44
77,BidderB,1.5,9903.86
77,BidderB,1.6,7544.98
77,BidderB,2.1,6513.12
77,BidderB,2.2,339.29
77,BidderB,2.3,7086.36
77,BidderB,3.1,5813.39
77,BidderB,3.2,4937.61
77,BidderB,3.3,8208.46
77,BidderB,3.4,10702.33
77,BidderB,3.5,3944.14
77,BidderB,3.6,4644.16
77,BidderB,4.1,990.19
77,BidderB,4.2,2210.71
77,BidderB,4.3,5486.09
77,BidderB,4.4,7138.29
77,BidderB,4.5,2400.36
77,BidderB,5.1,5236.6
77,BidderB,5.2,211.99
77,BidderB,6.1,3388.05
77,BidderB,6.2,6931.6
77,BidderB,6.3,9747.36
77,BidderB,6.4,6755.89
77,BidderB,6.5,3370.96
77,BidderB,6.6,8727.4
77,BidderB,6.7,7030.18
77,BidderB,6.8,3408.48
77,BidderB,6.9,7561.94
77,BidderB,7.1,6198.73
77,BidderB,7.2,2780.35
77,BidderB,7.3,9547.31
77,BidderB,7.4,10688.49
77,BidderB,7.5,2600.45
77,BidderB,8.1,7407.68
77,BidderB,8.2,2825.18
77,BidderB,8.3,5217.35
77,BidderC,1.1,5517.54
77,BidderC,1.2,5379.5
77,BidderC,1.3,1527.14
77,BidderC,1.4,1096.12
77,BidderC,1.5,9729.36
77,BidderC,1.6,7504.35
77,BidderC,2.1,5832.15
77,BidderC,2.2,364.0
77,BidderC,2.3,7379.76
77,BidderC,3.1,5386.0
77,BidderC,3.2,5372.01
77,BidderC,3.3,9411.87
77,BidderC,3.4,9460.56
77,BidderC,3.5,4023.62
77,BidderC,3.6,4546.68
77,BidderC,4.1,997.21
77,BidderC,4.2,2042.53
77,BidderC,4.3,5875.8
77,BidderC,4.4,7671.87
77,BidderC,4.5,2470.2
77,BidderC,5.1,4722.09
77,BidderC,5.2,223.94
77,BidderC,6.1,3661.46
77,BidderC,6.2,6243.43
77,BidderC,6.3,8816.12
77,BidderC,6.4,5907.71
77,BidderC,6.5,3522.45
77,BidderC,6.6,8271.99
77,BidderC,6.7,7720.16
77,BidderC,6.8,3623.41
77,BidderC,6.9,7772.81
77,BidderC,7.1,5534.36
77,BidderC,7.2,2725.68
77,BidderC,7.3,10524.43
77,BidderC,7.4,10841.62
77,BidderC,7.5,2409.0
77,BidderC,8.1,7512.66
77,BidderC,8.2,2916.18
77,BidderC,8.3,4934.66
84,BidderA,1.1,4754.96
84,BidderA,1.2,7505.0
84,BidderA,2.1,2914.99
84,BidderA,2.2,2510.93
84,BidderA,2.3,3563.3
84,BidderA,2.4,8206.03
84,BidderA,2.5,4018.73
84,BidderA,2.6,7453.23
84,BidderA,2.7,4106.37
84,BidderA,2.8,9522.8
84,BidderA,3.1,2123.4
84,BidderA,3.2,8900.66
84,BidderA,3.3,6865.29
84,BidderA,3.4,3689.05
84,BidderA,3.5,2087.63
84,BidderA,3.6,1002.31
84,BidderA,3.7,6011.1
84,BidderA,3.8,3004.53
84,BidderA,3.9,2458.97
84,BidderA,4.1,8273.98
84,BidderA,4.2,6302.51
84,BidderA,5.1,5750.72
84,BidderA,5.2,7895.54
84,BidderA,5.3,1336.14
84,BidderA,5.4,1856.65
84,BidderA,6.1,10453.04
84,BidderA,6.2,170.64
84,BidderA,6.3,9034.39
84,BidderA,7.1,9537.9
84,BidderA,7.2,7978.1
84,BidderA,7.3,4483.75
84,BidderA,8.1,8234.33
84,BidderA,8.2,9862.61
84,BidderA,8.3,2580.68
84,BidderA,9.1,2025.25
84,BidderA,9.2,398.34
84,BidderA,9.3,1542.87
84,BidderB,1.1,4777.16
84,BidderB,1.2,7592.7
84,BidderB,2.1,2545.46
84,BidderB,2.2,1931.59
84,BidderB,2.3,3599.14
84,BidderB,2.4,7974.7
84,BidderB,2.5,3998.14
84,BidderB,2.6,7395.63
84,BidderB,2.7,3871.26
84,BidderB,2.8,8720.67
84,BidderB,3.1,2192.72
84,BidderB,3.2,9915.31
84,BidderB,3.3,7275.8
84,BidderB,3.4,3885.43
84,BidderB,3.5,1821.37
84,BidderB,3.6,1008.99
84,BidderB,3.7,4791.78
84,BidderB,3.8,2974.13
84,BidderB,3.9,2276.71
84,BidderB,4.1,8466.77
84,BidderB,4.2,6532.88
84,BidderB,5.1,5125.14
84,BidderB,5.2,8172.69
84,BidderB,5.3,1334.21
84,BidderB,5.4,1835.16
84,BidderB,6.1,10555.05
84,BidderB,6.2,169.11
84,BidderB,6.3,9902.88
84,BidderB,7.1,9569.35
84,BidderB,7.2,7317.2
84,BidderB,7.3,4323.51
84,BidderB,8.1,7650.18
84,BidderB,8.2,9254.59
84,BidderB,8.3,2757.9
84,BidderB,9.1,1834.99
84,BidderB,9.2,365.95
84,BidderB,9.3,1360.93
84,BidderC,1.1,4306.19
84,BidderC,1.2,7053.67
84,BidderC,2.1,2844.11
84,BidderC,2.2,2166.49
84,BidderC,2.3,3435.47
84,BidderC,2.4,8031.2
84,BidderC,2.5,3989.06
84,BidderC,2.6,7796.67
84,BidderC,2.7,4161.47
84,BidderC,2.8,9429.07
84,BidderC,3.1,2055.62
84,BidderC,3.2,10149.99
84,BidderC,3.3,7415.17
84,BidderC,3.4,3691.47
84,BidderC,3.5,1838.22
84,BidderC,3.6,1027.77
84,BidderC,3.7,5591.86
84,BidderC,3.8,2866.58
84,BidderC,3.9,2362.86
84,BidderC,4.1,7052.87
84,BidderC,4.2,6625.53
84,BidderC,5.1,5576.16
84,BidderC,5.2,7485.2
84,BidderC,5.3,1388.34
84,BidderC,5.4,1997.62
84,BidderC,6.1,10292.04
84,BidderC,6.2,172.6
84,BidderC,6.3,9591.38
84,BidderC,7.1,10184.11
84,BidderC,7.2,7311.74
84,BidderC,7.3,4750.25
84,BidderC,8.1,8646.51
84,BidderC,8.2,10058.57
84,BidderC,8.3,2648.98
84,BidderC,9.1,1846.65
84,BidderC,9.2,358.74
84,BidderC,9.3,1625.79
88,BidderA,1.1,1151.18
88,BidderA,1.2,6265.88
88,BidderA,1.3,4084.85
88,BidderA,1.4,8621.53
88,BidderA,2.1,5394.32
88,BidderA,2.2,3578.7
88,BidderA,2.3,7634.26
88,BidderA,2.4,3589.13
88,BidderA,3.1,7549.15
88,BidderA,3.2,3466.91
88,BidderA,3.3,2450.67
88,BidderA,3.4,8904.95
88,BidderA,3.5,1598.54
88,BidderA,3.6,5557.74
88,BidderA,4.1,4601.39
88,BidderA,4.2,966.91
88,BidderA,4.3,6568.03
88,BidderA,4.4,5849.71
88,BidderA,4.5,5277.96
88,BidderA,4.6,2964.69
88,BidderA,4.7,6323.15
88,BidderA,5.1,3267.24
88,BidderA,5.2,11037.37
88,BidderA,6.1,4032.38
88,BidderA,6.2,5297.59
88,BidderA,6.3,3988.16
88,BidderA,6.4,8573.88
88,BidderB,1.1,1103.1
88,BidderB,1.2,6938.53
88,BidderB,1.3,3885.15
88,BidderB,1.4,8586.59
88,BidderB,2.1,5496.13
88,BidderB,2.2,3424.36
88,BidderB,2.3,8169.01
88,BidderB,2.4,3182.17
88,BidderB,3.1,6098.83
88,BidderB,3.2,3860.49
88,BidderB,3.3,2449.96
88,BidderB,3.4,7305.04
88,BidderB,3.5,1599.26
88,BidderB,3.6,6258.4
88,BidderB,4.1,4458.68
88,BidderB,4.2,1034.74
88,BidderB,4.3,6653.07
88,BidderB,4.4,5803.13
88,BidderB,4.5,5803.77
88,BidderB,4.6,3501.67
88,BidderB,4.7,5637.86
88,BidderB,5.1,3002.53
88,BidderB,5.2,10277.98
88,BidderB,6.1,3452.39
88,BidderB,6.2,4734.6
88,BidderB,6.3,3465.38
88,BidderB,6.4,8974.32
88,BidderC,1.1,1172.18
88,BidderC,1.2,6313.08
88,BidderC,1.3,3859.72
88,BidderC,1.4,8613.72
88,BidderC,2.1,6199.38
88,BidderC,2.2,3398.72
88,BidderC,2.3,8533.82
88,BidderC,2.4,3545.04
88,BidderC,3.1,7341.16
88,BidderC,3.2,3573.23
88,BidderC,3.3,2415.09
88,BidderC,3.4,8979.72
88,BidderC,3.5,1431.2
88,BidderC,3.6,6346.89
88,BidderC,4.1,4821.74
88,BidderC,4.2,1101.85
88,BidderC,4.3,6233.7
88,BidderC,4.4,5959.78
88,BidderC,4.5,5110.43
88,BidderC,4.6,3379.15
88,BidderC,4.7,5875.88
88,BidderC,5.1,2988.54
88,BidderC,5.2,10264.14
88,BidderC,6.1,4236.09
88,BidderC,6.2,4815.33
88,BidderC,6.3,3925.94
88,BidderC,6.4,8145.56
93,BidderA,1.1,4546.14
93,BidderA,1.2,9969.83
93,BidderA,1.3,8337.38
93,BidderA,1.4,1923.31
93,BidderA,1.5,4773.66
93,BidderA,1.6,10958.25
93,BidderA,1.7,6363.83
93,BidderA,1.8,8215.7
93,BidderA,1.9,10017.8
93,BidderA,2.1,9862.46
93,BidderA,2.2,7595.9
93,BidderA,2.3,2899.73
93,BidderA,2.4,7409.77
93,BidderB,1.1,4391.7
93,BidderB,1.2,9676.08
93,BidderB,1.3,8734.6
93,BidderB,1.4,1958.73
93,BidderB,1.5,4857.27
93,BidderB,1.6,11009.41
93,BidderB,1.7,6840.35
93,BidderB,1.8,8003.96
93,BidderB,1.9,10344.84
93,BidderB,2.1,11269.92
93,BidderB,2.2,7265.92
93,BidderB,2.3,2908.71
93,BidderB,2.4,8124.49
93,BidderC,1.1,4422.15
93,BidderC,1.2,9669.83
93,BidderC,1.3,8342.74
93,BidderC,1.4,1926.98
93,BidderC,1.5,4857.67
93,BidderC,1.6,9803.09
93,BidderC,1.7,6135.43
93,BidderC,1.8,7871.33
93,BidderC,1.9,9264.57
93,BidderC,2.1,10320.27
93,BidderC,2.2,7098.49
93,BidderC,2.3,2539.21
93,BidderC,2.4,7160.3
96,BidderA,1.1,873.71
96,BidderA,1.2,7835.23
96,BidderA,1.3,4693.61
96,BidderA,1.4,7497.69
96,BidderA,1.5,7663.76
96,BidderA,1.6,4197.68
96,BidderA,2.1,6782.23
96,BidderA,2.2,2987.5
96,BidderA,2.3,3628.64
96,BidderA,3.1,1764.44
96,BidderA,3.2,112.18
96,BidderA,3.3,7490.03
96,BidderA,3.4,4245.36
96,BidderA,3.5,8046.73
96,BidderA,3.6,9371.52
96,BidderA,4.1,8974.23
96,BidderA,4.2,158.1
96,BidderA,4.3,4333.11
96,BidderA,4.4,7312.0
96,BidderA,4.5,5369.38
96,BidderA,4.6,2104.88
96,BidderA,4.7,8573.77
96,BidderA,5.1,11067.01
96,BidderA,5.2,6406.49
96,BidderA,5.3,3689.46
96,BidderA,6.1,7805.59
96,BidderA,6.2,4826.57
96,BidderA,6.3,6060.04
96,BidderA,6.4,10398.28
96,BidderA,6.5,5162.29
96,BidderB,1.1,868.68
96,BidderB,1.2,8029.24
96,BidderB,1.3,4239.4
96,BidderB,1.4,7220.74
96,BidderB,1.5,7293.75
96,BidderB,1.6,4489.73
96,BidderB,2.1,7306.41
96,BidderB,2.2,2725.72
96,BidderB,2.3,3250.01
96,BidderB,3.1,1596.82
96,BidderB,3.2,121.76
96,BidderB,3.3,6675.07
96,BidderB,3.4,4670.22
96,BidderB,3.5,7668.66
96,BidderB,3.6,9474.86
96,BidderB,4.1,10077.45
96,BidderB,4.2,143.82
96,BidderB,4.3,4445.55
96,BidderB,4.4,7116.61
96,BidderB,4.5,5237.49
96,BidderB,4.6,2346.84
96,BidderB,4.7,8713.56
96,BidderB,5.1,10284.21
96,BidderB,5.2,6281.25
96,BidderB,5.3,3266.68
96,BidderB,6.1,7550.05
96,BidderB,6.2,4196.06
96,BidderB,6.3,6510.59
96,BidderB,6.4,8588.73
96,BidderB,6.5,5403.39
96,BidderC,1.1,840.08
96,BidderC,1.2,8072.34
96,BidderC,1.3,3710.36
96,BidderC,1.4,7879.56
96,BidderC,1.5,6776.14
96,BidderC,1.6,4954.92
96,BidderC,2.1,6958.29
96,BidderC,2.2,3111.23
96,BidderC,2.3,3654.14
96,BidderC,3.1,1665.42
96,BidderC,3.2,105.08
96,BidderC,3.3,7119.98
96,BidderC,3.4,4432.12
96,BidderC,3.5,7872.12
96,BidderC,3.6,10395.22
96,BidderC,4.1,10073.77
96,BidderC,4.2,147.07
96,BidderC,4.3,4247.1
96,BidderC,4.4,7628.87
96,BidderC,4.5,5502.09
96,BidderC,4.6,2090.09
96,BidderC,4.7,9716.64
96,BidderC,5.1,9737.49
96,BidderC,5.2,5728.31
96,BidderC,5.3,3365.55
96,BidderC,6.1,7422.32
96,BidderC,6.2,4391.22
96,BidderC,6.3,5926.68
96,BidderC,6.4,9396.51
96,BidderC,6.5,4956.62
'''

