from picalo import *

DETECTLET_STANDARD = 1.0

wizard = '''
<wizard>
  <page>
    Select the TABLE containing your received invoices.  
    
    The table should include an invoice number column and a vendor column   
    <parameter type="Table" variable="table"/>
  </page>
  <page>
    Select the COLUMN containing the invoice number:  
    
    This column should have the invoice numbers used by the vendors and your company's assigned invoice number.
    <parameter type="Column" table="table" variable="invoice"/>
  </page>
  <page>
    Select the COLUMN containing the vendor name or ID:
    <parameter type="Column" table="table" variable="vendor"/>
  </page>
</wizard>
'''

RESULTS_TEXT = '''\
    The displayed results table shows how many duplicate invoices have been submitted by
    each vendor. Some vendors try submitting their invoices multiple times in hopes that
    the receiving company will pay the invoice multiple times. 
    
    This results table shows all the invoices that appear multiple times.
'''


def run(table, invoice, vendor):
  '''A common fraud occurs when a company submits its invoice
     multiple times in hopes that the receiving company will pay the bill multiple
     times.   
     
     This wizard takes an invoice file (or similar billing file) and 
     identifies invoices that have been submitted more than one time
     by a given vendor.
        
     In the results table, all duplicate invoices are identified along with the
     vendors who submited them.
  '''
  # validate the input data
  assert invoice != vendor, 'The invoice number and vendor id columns must be different.  Please make sure you have selected different columns.'
  
  # check the vendors for duplicate invoices

  results = Simple.find_duplicates(table, vendor, invoice)
  
  Simple.sort(results, False, vendor)
  return results, RESULTS_TEXT

  
  
def example_input():
  '''Returns the eaxmple input table so the user can
     see what their input should look like'''
  import StringIO  # to emulate a file for load_csv
  table = load_csv(StringIO.StringIO(csvdata))
  table.set_type('Invoice', int)
  table.set_type('Date',    DateTime)
  table.set_type('Amount',  number)
  return table  
  
  
# This next part is not required, but it's easier to put the example data
# directly in this file so I don't have to worry about directories.
csvdata = '''\
Vendor,Invoice,Date,Amount
"McKenzie Supplies",1000,2006-01-01,7636.73
"McKenzie Supplies",1001,2006-01-01,3401.31
"McKenzie Supplies",1002,2006-01-07,2121.84
"McKenzie Supplies",1003,2006-01-07,3592.86
"McKenzie Supplies",1004,2006-01-08,5602.18
"McKenzie Supplies",1005,2006-01-08,3610.11
"McKenzie Supplies",1006,2006-01-08,4569.14
"McKenzie Supplies",1007,2006-01-09,3643.24
"McKenzie Supplies",1008,2006-01-19,2566.29
"McKenzie Supplies",1009,2006-01-27,2456.09
"McKenzie Supplies",1010,2006-02-02,1321.04
"McKenzie Supplies",1011,2006-02-03,5490.54
"McKenzie Supplies",1012,2006-02-04,5943.35
"McKenzie Supplies",1013,2006-02-05,5940.22
"McKenzie Supplies",1014,2006-02-13,9721.48
"McKenzie Supplies",1015,2006-02-14,9668.81
"McKenzie Supplies",1016,2006-02-24,4580.49
"McKenzie Supplies",1017,2006-03-02,8346.86
"McKenzie Supplies",1018,2006-03-02,2869.63
"McKenzie Supplies",1019,2006-03-12,2751.64
"McKenzie Supplies",1020,2006-03-20,9217.69
"McKenzie Supplies",1021,2006-03-26,9442.18
"McKenzie Supplies",1022,2006-04-05,6419.64
"McKenzie Supplies",1023,2006-04-11,3530.55
"McKenzie Supplies",1024,2006-04-12,1901.52
"McKenzie Supplies",1025,2006-04-22,3707.74
"McKenzie Supplies",1026,2006-04-28,9751.59
"McKenzie Supplies",1027,2006-05-06,7870.72
"McKenzie Supplies",1028,2006-05-07,2272.03
"McKenzie Supplies",1029,2006-05-08,3839.99
"McKenzie Supplies",1032,2006-05-16,7900.78
"McKenzie Supplies",1034,2006-05-22,4476.24
"McKenzie Supplies",1035,2006-05-28,6900.39
"McKenzie Supplies",1036,2006-05-28,8819.22
"McKenzie Supplies",1037,2006-05-28,7764.3
"McKenzie Supplies",1038,2006-06-07,6127.03
"McKenzie Supplies",1039,2006-06-13,5427.38
"McKenzie Supplies",1040,2006-06-19,2499.69
"McKenzie Supplies",1041,2006-06-19,9548.2
"McKenzie Supplies",1042,2006-06-19,4183.61
"McKenzie Supplies",1043,2006-06-19,8387.14
"McKenzie Supplies",1045,2006-06-20,9110.9
"McKenzie Supplies",1046,2006-06-20,4621.04
"McKenzie Supplies",1047,2006-06-28,4982.35
"McKenzie Supplies",1048,2006-07-04,7758.23
"McKenzie Supplies",1049,2006-07-05,5513.6
"McKenzie Supplies",1050,2006-07-11,8016.41
"McKenzie Supplies",1051,2006-07-21,7629.48
"McKenzie Supplies",1054,2006-07-29,3784.99
"McKenzie Supplies",1055,2006-08-06,2339.43
"McKenzie Supplies",1056,2006-08-12,1131.89
"McKenzie Supplies",1057,2006-08-12,4484.97
"McKenzie Supplies",1058,2006-08-22,8622.86
"McKenzie Supplies",1061,2006-08-22,2645.9
"McKenzie Supplies",1062,2006-08-30,5005.78
"McKenzie Supplies",1063,2006-08-30,9211.57
"McKenzie Supplies",1064,2006-09-09,4618.49
"McKenzie Supplies",1065,2006-09-17,4623.71
"McKenzie Supplies",1066,2006-09-25,3559.3
"McKenzie Supplies",1067,2006-09-26,9672.57
"McKenzie Supplies",1068,2006-09-26,8755.43
"McKenzie Supplies",1069,2006-10-02,2860.2
"McKenzie Supplies",1070,2006-10-12,5208.54
"McKenzie Supplies",1071,2006-10-12,8705.76
"McKenzie Supplies",1072,2006-10-18,3715.96
"McKenzie Supplies",1073,2006-10-18,5730.77
"McKenzie Supplies",1074,2006-10-28,2503.24
"McKenzie Supplies",1075,2006-11-03,5888.42
"McKenzie Supplies",1076,2006-11-09,1636.75
"McKenzie Supplies",1077,2006-11-15,8343.28
"McKenzie Supplies",1078,2006-11-16,7477.09
"McKenzie Supplies",1079,2006-11-17,6879.27
"McKenzie Supplies",1080,2006-11-18,9046.18
"McKenzie Supplies",1081,2006-11-24,4656.82
"McKenzie Supplies",1082,2006-12-02,6539.31
"McKenzie Supplies",1083,2006-12-03,4694.23
"McKenzie Supplies",1084,2006-12-04,7165.63
"McKenzie Supplies",1085,2006-12-10,3447.87
"McKenzie Supplies",1086,2006-12-18,6283.92
"McKenzie Supplies",1087,2006-12-18,9220.86
"McKenzie Supplies",1088,2006-12-26,2465.32
"McKenzie Supplies",1089,2006-12-26,4796.84
"McKenzie Supplies",1090,2007-01-01,5458.56
"McKenzie Supplies",1091,2007-01-02,7708.66
"McKenzie Supplies",1092,2007-01-03,4648.93
"McKenzie Supplies",1093,2007-01-03,1441.17
"McKenzie Supplies",1094,2007-01-04,7944.14
"McKenzie Supplies",1095,2007-01-05,2488.96
"McKenzie Supplies",1096,2007-01-06,2120.96
"McKenzie Supplies",1097,2007-01-06,1152.81
"McKenzie Supplies",1098,2007-01-06,4485.16
"McKenzie Supplies",1099,2007-01-07,1235.22
"McKenzie Supplies",1100,2007-01-13,5623.0
"McKenzie Supplies",1101,2007-01-14,7375.46
"McKenzie Supplies",1102,2007-01-20,5721.16
"McKenzie Supplies",1103,2007-01-21,2129.51
"McKenzie Supplies",1104,2007-01-21,8809.53
"McKenzie Supplies",1105,2007-01-27,6549.25
"McKenzie Supplies",1106,2007-02-02,3199.5
"McKenzie Supplies",1107,2007-02-03,9623.6
"McKenzie Supplies",1108,2007-02-04,6677.08
"McKenzie Supplies",1109,2007-02-04,6514.19
"McKenzie Supplies",1110,2007-02-10,7625.21
"McKenzie Supplies",1111,2007-02-20,1951.44
"McKenzie Supplies",1112,2007-02-20,3431.8
"McKenzie Supplies",1113,2007-02-21,1022.47
"McKenzie Supplies",1114,2007-02-22,9083.12
"McKenzie Supplies",1115,2007-02-23,1073.89
"McKenzie Supplies",1116,2007-02-24,8183.51
"McKenzie Supplies",1117,2007-02-25,5061.08
"McKenzie Supplies",1118,2007-02-26,2223.38
"McKenzie Supplies",1119,2007-02-27,5958.96
"McKenzie Supplies",1120,2007-03-05,4104.07
"McKenzie Supplies",1121,2007-03-06,4041.25
"McKenzie Supplies",1122,2007-03-06,6971.99
"McKenzie Supplies",1123,2007-03-06,5594.44
"McKenzie Supplies",1124,2007-03-07,1935.3
"McKenzie Supplies",1125,2007-03-17,7452.12
"McKenzie Supplies",1126,2007-03-17,1827.69
"McKenzie Supplies",1127,2007-03-25,2151.04
"McKenzie Supplies",1128,2007-03-25,6293.46
"McKenzie Supplies",1129,2007-03-26,7435.71
"McKenzie Supplies",1130,2007-03-26,5192.77
"McKenzie Supplies",1131,2007-03-26,5501.99
"McKenzie Supplies",1132,2007-04-05,1087.76
"McKenzie Supplies",1133,2007-04-06,4176.53
"McKenzie Supplies",1134,2007-04-06,3368.02
"McKenzie Supplies",1135,2007-04-06,8056.19
"McKenzie Supplies",1136,2007-04-16,1353.98
"McKenzie Supplies",1137,2007-04-17,6569.06
"McKenzie Supplies",1138,2007-04-18,3663.87
"McKenzie Supplies",1139,2007-04-18,2198.48
"McKenzie Supplies",1140,2007-04-18,9645.38
"McKenzie Supplies",1141,2007-04-19,9431.16
"McKenzie Supplies",1144,2007-04-25,7217.79
"McKenzie Supplies",1145,2007-05-03,5089.6
"McKenzie Supplies",1148,2007-05-09,2727.42
"McKenzie Supplies",1149,2007-05-10,5638.87
"McKenzie Supplies",1150,2007-05-10,2296.14
"McKenzie Supplies",1151,2007-05-16,5410.36
"McKenzie Supplies",1152,2007-05-22,3875.91
"McKenzie Supplies",1153,2007-06-01,9103.99
"McKenzie Supplies",1154,2007-06-07,1260.62
"McKenzie Supplies",1155,2007-06-15,6479.23
"McKenzie Supplies",1156,2007-06-16,9324.63
"McKenzie Supplies",1158,2007-06-17,4230.66
"McKenzie Supplies",1159,2007-06-25,8115.67
"McKenzie Supplies",1160,2007-07-03,7263.01
"McKenzie Supplies",1161,2007-07-11,6812.01
"McKenzie Supplies",1162,2007-07-12,3081.92
"McKenzie Supplies",1163,2007-07-13,8081.83
"McKenzie Supplies",1164,2007-07-14,4145.46
"McKenzie Supplies",1165,2007-07-22,7753.5
"McKenzie Supplies",1166,2007-07-30,7029.05
"McKenzie Supplies",1167,2007-08-07,6769.79
"McKenzie Supplies",1168,2007-08-08,7279.88
"McKenzie Supplies",1170,2007-08-08,7901.97
"McKenzie Supplies",1171,2007-08-18,1323.05
"McKenzie Supplies",1172,2007-08-18,2620.91
"McKenzie Supplies",1173,2007-08-19,9807.14
"McKenzie Supplies",1174,2007-08-20,6972.48
"McKenzie Supplies",1175,2007-08-21,8233.55
"McKenzie Supplies",1176,2007-08-22,1645.22
"McKenzie Supplies",1177,2007-08-22,3604.14
"McKenzie Supplies",1180,2007-09-01,8472.98
"McKenzie Supplies",1182,2007-09-09,4359.33
"McKenzie Supplies",1183,2007-09-10,9330.82
"McKenzie Supplies",1184,2007-09-11,3146.27
"McKenzie Supplies",1185,2007-09-11,1314.65
"McKenzie Supplies",1186,2007-09-21,7219.29
"McKenzie Supplies",1188,2007-09-21,4240.57
"McKenzie Supplies",1189,2007-09-21,2138.09
"McKenzie Supplies",1190,2007-10-01,7346.02
"McKenzie Supplies",1191,2007-10-11,5549.49
"McKenzie Supplies",1192,2007-10-12,7136.01
"McKenzie Supplies",1193,2007-10-12,1371.36
"McKenzie Supplies",1195,2007-10-13,6538.26
"McKenzie Supplies",1196,2007-10-21,9444.62
"McKenzie Supplies",1197,2007-10-21,2461.63
"McKenzie Supplies",1198,2007-10-29,4619.71
"McKenzie Supplies",1199,2007-11-06,4476.56
"McKenzie Supplies",1200,2007-11-06,5826.27
"McKenzie Supplies",1201,2007-11-07,2286.57
"McKenzie Supplies",1202,2007-11-15,2515.76
"McKenzie Supplies",1204,2007-11-16,1386.48
"McKenzie Supplies",1206,2007-11-16,9107.16
"McKenzie Supplies",1207,2007-11-24,1683.01
"McKenzie Supplies",1208,2007-11-24,8764.57
"McKenzie Supplies",1209,2007-11-24,7392.72
"McKenzie Supplies",1210,2007-11-25,9795.69
"McKenzie Supplies",1211,2007-12-05,2135.23
"McKenzie Supplies",1212,2007-12-11,4134.71
"McKenzie Supplies",1213,2007-12-21,1586.88
"McKenzie Supplies",1214,2007-12-27,2235.48
"McKenzie Supplies",1215,2008-01-02,5046.97
"McKenzie Supplies",1216,2008-01-03,5722.6
"McKenzie Supplies",1217,2008-01-03,4218.56
"McKenzie Supplies",1219,2008-01-04,6626.11
"McKenzie Supplies",1220,2008-01-12,8634.01
"McKenzie Supplies",1221,2008-01-18,3793.39
"McKenzie Supplies",1222,2008-01-18,5384.92
"McKenzie Supplies",1223,2008-01-28,3527.81
"McKenzie Supplies",1224,2008-01-28,4211.31
"McKenzie Supplies",1225,2008-01-28,5304.53
"McKenzie Supplies",1226,2008-02-07,6659.65
"McKenzie Supplies",1227,2008-02-15,3103.98
"McKenzie Supplies",1228,2008-02-21,5690.2
"McKenzie Supplies",1229,2008-02-27,2641.78
"McKenzie Supplies",1230,2008-02-27,9356.18
"McKenzie Supplies",1232,2008-03-06,6338.06
"McKenzie Supplies",1233,2008-03-06,8063.05
"McKenzie Supplies",1234,2008-03-07,3686.17
"McKenzie Supplies",1235,2008-03-08,9370.52
"McKenzie Supplies",1236,2008-03-14,7297.99
"McKenzie Supplies",1237,2008-03-20,4773.66
"McKenzie Supplies",1238,2008-03-20,4982.97
"McKenzie Supplies",1239,2008-03-20,3735.48
"McKenzie Supplies",1240,2008-03-20,5072.85
"McKenzie Supplies",1242,2008-03-30,2677.58
"McKenzie Supplies",1245,2008-03-30,1654.07
"McKenzie Supplies",1246,2008-03-31,6953.08
"McKenzie Supplies",1247,2008-04-01,1855.56
"McKenzie Supplies",1248,2008-04-01,6850.51
"McKenzie Supplies",1249,2008-04-02,4076.97
"McKenzie Supplies",1250,2008-04-02,3801.86
"McKenzie Supplies",1251,2008-04-12,8603.99
"McKenzie Supplies",1252,2008-04-13,5047.62
"McKenzie Supplies",1253,2008-04-19,8282.79
"McKenzie Supplies",1254,2008-04-20,1526.37
"McKenzie Supplies",1257,2008-04-20,7676.95
"McKenzie Supplies",1258,2008-04-20,9624.96
"McKenzie Supplies",1259,2008-04-30,6054.74
"McKenzie Supplies",1260,2008-04-30,9879.22
"McKenzie Supplies",1261,2008-05-06,6764.22
"McKenzie Supplies",1262,2008-05-06,4886.16
"McKenzie Supplies",1263,2008-05-07,5824.12
"McKenzie Supplies",1264,2008-05-07,3683.69
"McKenzie Supplies",1265,2008-05-08,3086.13
"McKenzie Supplies",1266,2008-05-18,4808.14
"McKenzie Supplies",1267,2008-05-24,7739.02
"McKenzie Supplies",1269,2008-05-24,8149.21
"McKenzie Supplies",1270,2008-05-24,9978.23
"McKenzie Supplies",1273,2008-05-25,2770.28
"McKenzie Supplies",1274,2008-05-31,8816.17
"McKenzie Supplies",1277,2008-05-31,4317.33
"McKenzie Supplies",1278,2008-06-10,1990.19
"McKenzie Supplies",1279,2008-06-11,3910.9
"McKenzie Supplies",1280,2008-06-11,7324.63
"McKenzie Supplies",1281,2008-06-12,8833.86
"McKenzie Supplies",1282,2008-06-12,4859.52
"McKenzie Supplies",1283,2008-06-13,5561.37
"McKenzie Supplies",1284,2008-06-13,3467.45
"McKenzie Supplies",1285,2008-06-13,7403.31
"McKenzie Supplies",1286,2008-06-14,3567.1
"McKenzie Supplies",1287,2008-06-15,8137.26
"McKenzie Supplies",1288,2008-06-15,8119.31
"McKenzie Supplies",1289,2008-06-21,6404.01
"McKenzie Supplies",1290,2008-06-27,9318.03
"McKenzie Supplies",1291,2008-06-27,1876.48
"McKenzie Supplies",1292,2008-07-07,2011.37
"McKenzie Supplies",1293,2008-07-17,6441.07
"McKenzie Supplies",1294,2008-07-17,5315.62
"McKenzie Supplies",1295,2008-07-27,4611.18
"McKenzie Supplies",1296,2008-07-27,5521.38
"McKenzie Supplies",1297,2008-07-28,5374.47
"McKenzie Supplies",1298,2008-07-28,2259.17
"McKenzie Supplies",1299,2008-07-29,9492.51
"McKenzie Supplies",1300,2008-07-29,9646.14
"McKenzie Supplies",1301,2008-08-08,7259.35
"McKenzie Supplies",1302,2008-08-18,9609.05
"McKenzie Supplies",1303,2008-08-19,6437.2
"McKenzie Supplies",1304,2008-08-29,1839.17
"McKenzie Supplies",1305,2008-09-06,5654.49
"McKenzie Supplies",1306,2008-09-06,2761.28
"McKenzie Supplies",1307,2008-09-06,6354.12
"McKenzie Supplies",1308,2008-09-06,6558.72
"McKenzie Supplies",1309,2008-09-06,7689.06
"McKenzie Supplies",1310,2008-09-07,7593.34
"McKenzie Supplies",1311,2008-09-08,1692.03
"McKenzie Supplies",1314,2008-09-14,5612.99
"McKenzie Supplies",1315,2008-09-14,7584.7
"McKenzie Supplies",1316,2008-09-22,2852.58
"McKenzie Supplies",1317,2008-09-22,1566.43
"McKenzie Supplies",1320,2008-09-23,6261.69
"McKenzie Supplies",1321,2008-09-24,3246.64
"McKenzie Supplies",1322,2008-09-25,8533.9
"McKenzie Supplies",1323,2008-10-01,9917.32
"McKenzie Supplies",1325,2008-10-07,4162.84
"McKenzie Supplies",1326,2008-10-17,8856.88
"McKenzie Supplies",1327,2008-10-27,2963.68
"McKenzie Supplies",1328,2008-10-27,1910.62
"McKenzie Supplies",1329,2008-11-06,6732.13
"McKenzie Supplies",1330,2008-11-07,5986.18
"McKenzie Supplies",1331,2008-11-08,9981.03
"McKenzie Supplies",1332,2008-11-09,7430.32
"McKenzie Supplies",1333,2008-11-15,1321.33
"McKenzie Supplies",1334,2008-11-25,7684.03
"McKenzie Supplies",1335,2008-11-26,2085.44
"McKenzie Supplies",1336,2008-12-06,3450.41
"McKenzie Supplies",1337,2008-12-06,2407.98
"McKenzie Supplies",1338,2008-12-16,8428.6
"McKenzie Supplies",1339,2008-12-16,3293.75
"McKenzie Supplies",1341,2008-12-17,5052.16
"McKenzie Supplies",1342,2008-12-17,6044.26
"McKenzie Supplies",1343,2008-12-25,7309.18
"McKenzie Supplies",1344,2009-01-02,8686.81
"McKenzie Supplies",1345,2009-01-02,7267.35
"McKenzie Supplies",1346,2009-01-03,1557.9
"McKenzie Supplies",1347,2009-01-13,8773.4
"McKenzie Supplies",1348,2009-01-14,3932.73
"McKenzie Supplies",1349,2009-01-15,8557.94
"McKenzie Supplies",1350,2009-01-15,8819.02
"McKenzie Supplies",1351,2009-01-15,1773.98
"McKenzie Supplies",1354,2009-01-21,3983.24
"McKenzie Supplies",1355,2009-01-29,3236.4
"McKenzie Supplies",1356,2009-02-04,6106.91
"McKenzie Supplies",1357,2009-02-14,6624.66
"McKenzie Supplies",1358,2009-02-15,4183.74
"McKenzie Supplies",1359,2009-02-21,7600.32
"McKenzie Supplies",1360,2009-03-03,7949.5
"McKenzie Supplies",1361,2009-03-04,1046.15
"McKenzie Supplies",1362,2009-03-04,6591.83
"McKenzie Supplies",1363,2009-03-04,2270.22
"McKenzie Supplies",1364,2009-03-05,8989.92
"McKenzie Supplies",1365,2009-03-06,7935.72
"McKenzie Supplies",1366,2009-03-07,4239.56
"McKenzie Supplies",1367,2009-03-08,1579.39
"McKenzie Supplies",1369,2009-03-08,9184.35
"McKenzie Supplies",1370,2009-03-08,5374.44
"McKenzie Supplies",1371,2009-03-09,7706.33
"McKenzie Supplies",1372,2009-03-09,1999.21
"McKenzie Supplies",1373,2009-03-09,6915.09
"McKenzie Supplies",1374,2009-03-09,6593.97
"McKenzie Supplies",1375,2009-03-17,8283.64
"McKenzie Supplies",1376,2009-03-23,1237.38
"McKenzie Supplies",1377,2009-03-24,1640.82
"McKenzie Supplies",1378,2009-04-03,2850.24
"McKenzie Supplies",1379,2009-04-13,7530.64
"McKenzie Supplies",1380,2009-04-14,6509.03
"McKenzie Supplies",1381,2009-04-20,3587.43
"McKenzie Supplies",1382,2009-04-20,7889.57
"McKenzie Supplies",1383,2009-04-30,3146.76
"McKenzie Supplies",1384,2009-05-01,8769.6
"McKenzie Supplies",1385,2009-05-01,3343.17
"McKenzie Supplies",1386,2009-05-07,7731.6
"McKenzie Supplies",1387,2009-05-08,3459.16
"McKenzie Supplies",1390,2009-05-14,6786.99
"McKenzie Supplies",1391,2009-05-22,7754.68
"McKenzie Supplies",1392,2009-05-28,7204.14
"McKenzie Supplies",1393,2009-05-29,7162.73
"McKenzie Supplies",1394,2009-06-06,8686.13
"McKenzie Supplies",1395,2009-06-14,3399.21
"McKenzie Supplies",1396,2009-06-14,7458.19
"McKenzie Supplies",1397,2009-06-14,5662.08
"McKenzie Supplies",1398,2009-06-15,6900.61
"McKenzie Supplies",1399,2009-06-25,9255.53
"McKenzie Supplies",1402,2009-07-03,1591.2
"McKenzie Supplies",1403,2009-07-04,7049.87
"McKenzie Supplies",1404,2009-07-12,9631.83
"McKenzie Supplies",1405,2009-07-12,1745.63
"McKenzie Supplies",1406,2009-07-20,6385.03
"McKenzie Supplies",1407,2009-07-20,8068.03
"McKenzie Supplies",1408,2009-07-28,3944.4
"McKenzie Supplies",1409,2009-07-29,2143.66
"McKenzie Supplies",1410,2009-07-30,5896.4
"McKenzie Supplies",1411,2009-08-05,6366.38
"McKenzie Supplies",1414,2009-08-15,1654.3
"McKenzie Supplies",1415,2009-08-23,3138.35
"McKenzie Supplies",1416,2009-08-31,3778.93
"McKenzie Supplies",1417,2009-09-01,2333.91
"McKenzie Supplies",1418,2009-09-02,7942.91
"McKenzie Supplies",1419,2009-09-02,8141.58
"McKenzie Supplies",1420,2009-09-03,8607.29
"McKenzie Supplies",1421,2009-09-04,4747.79
"McKenzie Supplies",1422,2009-09-14,1848.07
"McKenzie Supplies",1423,2009-09-14,8479.79
"McKenzie Supplies",1424,2009-09-14,6970.09
"McKenzie Supplies",1425,2009-09-14,7904.14
"McKenzie Supplies",1426,2009-09-14,9127.54
"McKenzie Supplies",1427,2009-09-15,5458.76
"McKenzie Supplies",1428,2009-09-16,8535.36
"McKenzie Supplies",1429,2009-09-17,7429.61
"McKenzie Supplies",1431,2009-09-17,7368.74
"McKenzie Supplies",1432,2009-09-17,4810.08
"McKenzie Supplies",1433,2009-09-17,9650.11
"McKenzie Supplies",1434,2009-09-23,8557.97
"McKenzie Supplies",1435,2009-09-24,5185.56
"McKenzie Supplies",1436,2009-09-25,9445.13
"McKenzie Supplies",1437,2009-10-05,8548.64
"McKenzie Supplies",1438,2009-10-05,8315.6
"McKenzie Supplies",1439,2009-10-05,1824.22
"McKenzie Supplies",1440,2009-10-13,3822.7
"McKenzie Supplies",1441,2009-10-19,5800.2
"McKenzie Supplies",1442,2009-10-19,4233.91
"McKenzie Supplies",1443,2009-10-20,7560.96
"McKenzie Supplies",1446,2009-10-26,4060.61
"McKenzie Supplies",1447,2009-11-05,9537.65
"McKenzie Supplies",1448,2009-11-06,8468.2
"McKenzie Supplies",1449,2009-11-12,1792.77
"McKenzie Supplies",1450,2009-11-18,6786.1
"McKenzie Supplies",1451,2009-11-28,3138.22
"McKenzie Supplies",1452,2009-12-08,4142.4
"McKenzie Supplies",1453,2009-12-08,8172.16
"McKenzie Supplies",1454,2009-12-09,4985.61
"McKenzie Supplies",1455,2009-12-10,8558.55
"McKenzie Supplies",1456,2009-12-11,7976.31
"McKenzie Supplies",1457,2009-12-12,8451.75
"McKenzie Supplies",1458,2009-12-18,6165.72
"McKenzie Supplies",1459,2009-12-24,4059.18
"McKenzie Supplies",1460,2009-12-24,5921.35
"McKenzie Supplies",1461,2009-12-24,6654.38
"McKenzie Supplies",1464,2009-12-25,8913.62
"McKenzie Supplies",1465,2009-12-25,8799.48
"McKenzie Supplies",1466,2009-12-26,3306.47
"McKenzie Supplies",1467,2010-01-01,1091.02
"McKenzie Supplies",1468,2010-01-11,7367.79
"McKenzie Supplies",1469,2010-01-12,8090.36
"McKenzie Supplies",1472,2010-01-18,8829.61
"McKenzie Supplies",1473,2010-01-18,6609.01
"McKenzie Supplies",1474,2010-01-19,8901.67
"McKenzie Supplies",1475,2010-01-25,2411.06
"McKenzie Supplies",1476,2010-01-25,4619.43
"McKenzie Supplies",1477,2010-02-02,6242.62
"McKenzie Supplies",1478,2010-02-08,9054.66
"McKenzie Supplies",1479,2010-02-08,9734.28
"McKenzie Supplies",1480,2010-02-09,7581.18
"McKenzie Supplies",1482,2010-02-10,2711.79
"McKenzie Supplies",1483,2010-02-18,6700.35
"McKenzie Supplies",1484,2010-02-19,4285.65
"McKenzie Supplies",1485,2010-02-25,3538.4
"McKenzie Supplies",1486,2010-03-03,5232.18
"McKenzie Supplies",1487,2010-03-03,5603.95
"McKenzie Supplies",1488,2010-03-03,5621.47
"McKenzie Supplies",1489,2010-03-03,7882.33
"McKenzie Supplies",1490,2010-03-11,7777.75
"McKenzie Supplies",1491,2010-03-12,3628.91
"McKenzie Supplies",1492,2010-03-12,1233.4
"McKenzie Supplies",1493,2010-03-13,9372.94
"McKenzie Supplies",1495,2010-03-13,8515.21
"McKenzie Supplies",1496,2010-03-19,4295.66
"McKenzie Supplies",1497,2010-03-27,3222.67
"McKenzie Supplies",1498,2010-03-28,5436.72
"McKenzie Supplies",1499,2010-04-05,3756.04
"McKenzie Supplies",1500,2010-04-15,8135.51
"McKenzie Supplies",1501,2010-04-25,6124.69
"McKenzie Supplies",1502,2010-04-25,2810.56
"McKenzie Supplies",1504,2010-05-01,9174.44
"McKenzie Supplies",1505,2010-05-11,1519.21
"McKenzie Supplies",1506,2010-05-21,4953.7
"McKenzie Supplies",1507,2010-05-29,7264.47
"McKenzie Supplies",1508,2010-05-29,5804.92
"McKenzie Supplies",1509,2010-05-29,9221.24
"McKenzie Supplies",1510,2010-05-30,9542.06
"McKenzie Supplies",1511,2010-06-05,2828.65
"McKenzie Supplies",1512,2010-06-06,7726.9
"McKenzie Supplies",1513,2010-06-06,5277.13
"McKenzie Supplies",1514,2010-06-16,8282.29
"McKenzie Supplies",1515,2010-06-16,3589.21
"McKenzie Supplies",1516,2010-06-24,7974.62
"McKenzie Supplies",1517,2010-06-24,6273.67
"McKenzie Supplies",1518,2010-07-04,8116.6
"McKenzie Supplies",1519,2010-07-04,9970.64
"McKenzie Supplies",1520,2010-07-04,3575.76
"McKenzie Supplies",1521,2010-07-04,5159.58
"McKenzie Supplies",1522,2010-07-12,9344.01
"McKenzie Supplies",1523,2010-07-18,5182.09
"McKenzie Supplies",1524,2010-07-19,9982.4
"McKenzie Supplies",1526,2010-07-27,6098.55
"McKenzie Supplies",1527,2010-07-27,8975.28
"McKenzie Supplies",1528,2010-08-04,4670.78
"McKenzie Supplies",1529,2010-08-10,7888.3
"McKenzie Supplies",1530,2010-08-16,9467.7
"McKenzie Supplies",1531,2010-08-17,9716.54
"McKenzie Supplies",1532,2010-08-17,8787.26
"McKenzie Supplies",1533,2010-08-25,5784.89
"McKenzie Supplies",1534,2010-08-26,2774.57
"McKenzie Supplies",1535,2010-09-01,9208.05
"McKenzie Supplies",1537,2010-09-02,4527.88
"McKenzie Supplies",1538,2010-09-02,4688.93
"McKenzie Supplies",1539,2010-09-02,4058.99
"McKenzie Supplies",1540,2010-09-12,4055.15
"McKenzie Supplies",1541,2010-09-12,9701.12
"McKenzie Supplies",1544,2010-09-20,4150.35
"McKenzie Supplies",1545,2010-09-21,7071.96
"McKenzie Supplies",1546,2010-09-27,2441.39
"McKenzie Supplies",1549,2010-10-05,3096.6
"McKenzie Supplies",1550,2010-10-13,1393.19
"McKenzie Supplies",1553,2010-10-19,9232.83
"McKenzie Supplies",1554,2010-10-25,8312.88
"McKenzie Supplies",1555,2010-10-31,4367.24
"McKenzie Supplies",1556,2010-11-06,4234.26
"McKenzie Supplies",1557,2010-11-12,1914.82
"McKenzie Supplies",1558,2010-11-12,9541.58
"McKenzie Supplies",1559,2010-11-20,2983.46
"McKenzie Supplies",1560,2010-11-21,3857.92
"McKenzie Supplies",1561,2010-11-27,5785.04
"McKenzie Supplies",1562,2010-11-28,5025.29
"McKenzie Supplies",1563,2010-11-28,8538.88
"McKenzie Supplies",1564,2010-11-29,8794.56
"McKenzie Supplies",1565,2010-11-30,8005.18
"McKenzie Supplies",1566,2010-12-08,3227.58
"McKenzie Supplies",1567,2010-12-09,5569.47
"McKenzie Supplies",1568,2010-12-15,6058.41
"McKenzie Supplies",1569,2010-12-16,3489.36
"McKenzie Supplies",1571,2010-12-22,4223.36
"McKenzie Supplies",1572,2010-12-30,4891.46
"McKenzie Supplies",1573,2010-12-31,6017.58
"McKenzie Supplies",1574,2011-01-06,1957.33
"McKenzie Supplies",1577,2011-01-06,6059.96
"McKenzie Supplies",1578,2011-01-07,8083.89
"McKenzie Supplies",1579,2011-01-13,4527.74
"McKenzie Supplies",1580,2011-01-14,7747.46
"McKenzie Supplies",1581,2011-01-24,7262.15
"McKenzie Supplies",1582,2011-02-03,8273.91
"McKenzie Supplies",1583,2011-02-03,2955.37
"McKenzie Supplies",1584,2011-02-09,2898.86
"McKenzie Supplies",1585,2011-02-10,4618.01
"McKenzie Supplies",1586,2011-02-11,6197.22
"McKenzie Supplies",1587,2011-02-12,6638.96
"McKenzie Supplies",1588,2011-02-13,4440.5
"McKenzie Supplies",1589,2011-02-19,8838.66
"McKenzie Supplies",1590,2011-02-20,1651.94
"McKenzie Supplies",1591,2011-02-20,5348.09
"McKenzie Supplies",1592,2011-02-26,3586.11
"McKenzie Supplies",1593,2011-02-27,8402.78
"McKenzie Supplies",1594,2011-02-27,1041.94
"McKenzie Supplies",1595,2011-02-27,1477.3
"McKenzie Supplies",1596,2011-03-07,3500.24
"McKenzie Supplies",1597,2011-03-08,6967.62
"McKenzie Supplies",1598,2011-03-16,9230.7
"McKenzie Supplies",1599,2011-03-26,6439.34
"McKenzie Supplies",1600,2011-04-01,2028.94
"McKenzie Supplies",1601,2011-04-01,6480.62
"McKenzie Supplies",1604,2011-04-01,4657.12
"McKenzie Supplies",1605,2011-04-01,6894.96
"McKenzie Supplies",1606,2011-04-02,5418.18
"McKenzie Supplies",1607,2011-04-02,8477.84
"McKenzie Supplies",1609,2011-04-03,2795.1
"McKenzie Supplies",1610,2011-04-04,2876.61
"McKenzie Supplies",1611,2011-04-04,1415.21
"McKenzie Supplies",1612,2011-04-04,2740.75
"McKenzie Supplies",1613,2011-04-05,7133.19
"McKenzie Supplies",1614,2011-04-05,7784.62
"McKenzie Supplies",1615,2011-04-13,5494.47
"McKenzie Supplies",1616,2011-04-14,6192.43
"McKenzie Supplies",1619,2011-04-14,5525.37
"McKenzie Supplies",1620,2011-04-14,1820.63
"McKenzie Supplies",1621,2011-04-15,1728.72
"McKenzie Supplies",1623,2011-04-16,9321.15
"McKenzie Supplies",1624,2011-04-24,6394.35
"McKenzie Supplies",1625,2011-05-02,2137.12
"McKenzie Supplies",1626,2011-05-03,2097.67
"McKenzie Supplies",1628,2011-05-03,7326.25
"McKenzie Supplies",1629,2011-05-04,9491.17
"McKenzie Supplies",1630,2011-05-14,9186.77
"McKenzie Supplies",1631,2011-05-15,1414.12
"McKenzie Supplies",1632,2011-05-16,9567.74
"McKenzie Supplies",1633,2011-05-17,2312.13
"McKenzie Supplies",1634,2011-05-17,7682.78
"McKenzie Supplies",1635,2011-05-17,7202.9
"McKenzie Supplies",1636,2011-05-17,3622.18
"McKenzie Supplies",1637,2011-05-17,6844.7
"McKenzie Supplies",1638,2011-05-18,3831.71
"McKenzie Supplies",1639,2011-05-24,7662.15
"McKenzie Supplies",1642,2011-06-01,4995.44
"McKenzie Supplies",1643,2011-06-07,3395.79
"McKenzie Supplies",1646,2011-06-07,6718.02
"McKenzie Supplies",1647,2011-06-15,5958.79
"McKenzie Supplies",1648,2011-06-25,8141.51
"McKenzie Supplies",1650,2011-06-26,8164.83
"McKenzie Supplies",1651,2011-06-27,1977.73
"McKenzie Supplies",1652,2011-06-28,1954.48
"McKenzie Supplies",1653,2011-06-28,6839.43
"McKenzie Supplies",1654,2011-06-29,3507.02
"McKenzie Supplies",1655,2011-07-07,9025.11
"McKenzie Supplies",1656,2011-07-08,1133.81
"McKenzie Supplies",1657,2011-07-08,1555.07
"McKenzie Supplies",1658,2011-07-09,1983.07
"McKenzie Supplies",1659,2011-07-10,2838.4
"McKenzie Supplies",1660,2011-07-10,4612.77
"McKenzie Supplies",1661,2011-07-11,4306.37
"McKenzie Supplies",1662,2011-07-17,8017.3
"McKenzie Supplies",1663,2011-07-18,2068.28
"McKenzie Supplies",1665,2011-07-24,9778.34
"McKenzie Supplies",1666,2011-07-24,4297.08
"McKenzie Supplies",1667,2011-07-24,9083.65
"McKenzie Supplies",1668,2011-07-30,9534.35
"McKenzie Supplies",1669,2011-07-30,7406.88
"McKenzie Supplies",1670,2011-07-30,4870.7
"McKenzie Supplies",1671,2011-07-31,9587.3
"McKenzie Supplies",1672,2011-08-06,1496.11
"McKenzie Supplies",1673,2011-08-06,5250.62
"McKenzie Supplies",1674,2011-08-06,8121.95
"McKenzie Supplies",1677,2011-08-06,4524.88
"McKenzie Supplies",1678,2011-08-07,1874.46
"McKenzie Supplies",1680,2011-08-07,7192.5
"McKenzie Supplies",1681,2011-08-07,5461.72
"McKenzie Supplies",1682,2011-08-15,6109.21
"McKenzie Supplies",1683,2011-08-16,5765.75
"McKenzie Supplies",1684,2011-08-26,4419.89
"McKenzie Supplies",1687,2011-09-03,2615.43
"McKenzie Supplies",1688,2011-09-04,6834.3
"McKenzie Supplies",1689,2011-09-04,2637.27
"McKenzie Supplies",1690,2011-09-05,6519.86
"McKenzie Supplies",1691,2011-09-06,9907.33
"McKenzie Supplies",1692,2011-09-06,9373.27
"McKenzie Supplies",1693,2011-09-06,8217.67
"McKenzie Supplies",1694,2011-09-06,1994.31
"McKenzie Supplies",1695,2011-09-12,5507.99
"McKenzie Supplies",1696,2011-09-13,1407.34
"McKenzie Supplies",1697,2011-09-13,1141.81
"McKenzie Supplies",1698,2011-09-23,9983.76
"McKenzie Supplies",1699,2011-09-29,7015.15
"McKenzie Supplies",1700,2011-10-05,6302.04
"McKenzie Supplies",1701,2011-10-05,6282.62
"McKenzie Supplies",1704,2011-10-15,2909.23
"McKenzie Supplies",1705,2011-10-15,8890.22
"McKenzie Supplies",1706,2011-10-15,8915.16
"McKenzie Supplies",1707,2011-10-16,7872.7
"McKenzie Supplies",1708,2011-10-16,4608.22
"McKenzie Supplies",1709,2011-10-17,4818.71
"McKenzie Supplies",1710,2011-10-27,4076.53
"McKenzie Supplies",1711,2011-11-02,6935.83
"McKenzie Supplies",1712,2011-11-08,5540.54
"McKenzie Supplies",1715,2011-11-09,8571.38
"McKenzie Supplies",1716,2011-11-09,6276.22
"McKenzie Supplies",1717,2011-11-17,2952.92
"McKenzie Supplies",1718,2011-11-25,9044.05
"McKenzie Supplies",1719,2011-11-25,4660.6
"McKenzie Supplies",1720,2011-12-01,1043.88
"McKenzie Supplies",1721,2011-12-09,9880.82
"McKenzie Supplies",1722,2011-12-09,4764.14
"McKenzie Supplies",1723,2011-12-17,2496.43
"McKenzie Supplies",1724,2011-12-17,1384.3
"McKenzie Supplies",1725,2011-12-17,9065.11
"McKenzie Supplies",1726,2011-12-17,6585.39
"McKenzie Supplies",1727,2011-12-18,6924.53
"McKenzie Supplies",1728,2011-12-19,9953.27
"McKenzie Supplies",1729,2011-12-20,1682.71
"McKenzie Supplies",1731,2011-12-20,6816.5
"McKenzie Supplies",1733,2011-12-20,7298.47
"McKenzie Supplies",1734,2011-12-20,5595.28
"McKenzie Supplies",1735,2011-12-28,2586.26
"McKenzie Supplies",1736,2011-12-28,3586.2
"McKenzie Supplies",1737,2012-01-07,9913.55
"McKenzie Supplies",1738,2012-01-17,3263.82
"McKenzie Supplies",1739,2012-01-17,2175.92
"McKenzie Supplies",1740,2012-01-17,9064.27
"McKenzie Supplies",1741,2012-01-17,2872.48
"McKenzie Supplies",1742,2012-01-17,7305.57
"McKenzie Supplies",1743,2012-01-27,1198.02
"McKenzie Supplies",1744,2012-02-02,1669.85
"McKenzie Supplies",1745,2012-02-10,4049.1
"McKenzie Supplies",1746,2012-02-18,6768.83
"McKenzie Supplies",1747,2012-02-18,7107.64
"McKenzie Supplies",1748,2012-02-19,6572.61
"McKenzie Supplies",1749,2012-02-25,4462.34
"McKenzie Supplies",1750,2012-02-25,8044.07
"McKenzie Supplies",1751,2012-02-25,6272.71
"McKenzie Supplies",1752,2012-03-06,2225.9
"McKenzie Supplies",1753,2012-03-07,9340.34
"McKenzie Supplies",1754,2012-03-07,5748.04
"McKenzie Supplies",1755,2012-03-07,2535.73
"McKenzie Supplies",1756,2012-03-17,6792.29
"McKenzie Supplies",1757,2012-03-25,7342.55
"McKenzie Supplies",1758,2012-04-04,9677.89
"McKenzie Supplies",1759,2012-04-10,7037.92
"McKenzie Supplies",1760,2012-04-10,7402.19
"McKenzie Supplies",1761,2012-04-16,8929.82
"McKenzie Supplies",1762,2012-04-22,1029.71
"McKenzie Supplies",1763,2012-04-23,9611.98
"McKenzie Supplies",1764,2012-04-23,6459.25
"McKenzie Supplies",1767,2012-05-03,8183.82
"McKenzie Supplies",1768,2012-05-04,2791.84
"McKenzie Supplies",1770,2012-05-05,7910.21
"McKenzie Supplies",1771,2012-05-11,2180.47
"McKenzie Supplies",1772,2012-05-11,5448.73
"McKenzie Supplies",1773,2012-05-12,9893.35
"McKenzie Supplies",1774,2012-05-13,5395.58
"McKenzie Supplies",1775,2012-05-21,5184.75
"McKenzie Supplies",1776,2012-05-27,4853.87
"McKenzie Supplies",1777,2012-06-04,8124.19
"McKenzie Supplies",1779,2012-06-05,4650.42
"McKenzie Supplies",1780,2012-06-06,3331.45
"McKenzie Supplies",1782,2012-06-06,5729.31
"McKenzie Supplies",1783,2012-06-14,2868.95
"McKenzie Supplies",1784,2012-06-20,7830.15
"McKenzie Supplies",1785,2012-06-21,7006.5
"McKenzie Supplies",1786,2012-06-22,7281.86
"McKenzie Supplies",1787,2012-06-28,1456.39
"McKenzie Supplies",1788,2012-06-29,9099.35
"McKenzie Supplies",1789,2012-07-07,7689.09
"McKenzie Supplies",1790,2012-07-07,8265.47
"McKenzie Supplies",1792,2012-07-15,7220.38
"McKenzie Supplies",1793,2012-07-16,8888.16
"McKenzie Supplies",1794,2012-07-17,7928.51
"McKenzie Supplies",1795,2012-07-23,4019.72
"McKenzie Supplies",1796,2012-07-23,6647.89
"McKenzie Supplies",1797,2012-07-29,3158.31
"McKenzie Supplies",1798,2012-08-06,1217.47
"McKenzie Supplies",1799,2012-08-06,7408.22
"McKenzie Supplies",1800,2012-08-12,1901.55
"McKenzie Supplies",1801,2012-08-12,7846.57
"McKenzie Supplies",1802,2012-08-12,2556.16
"McKenzie Supplies",1803,2012-08-12,1234.74
"McKenzie Supplies",1804,2012-08-12,1493.82
"McKenzie Supplies",1805,2012-08-12,4475.78
"McKenzie Supplies",1806,2012-08-12,4726.43
"McKenzie Supplies",1807,2012-08-13,8833.37
"McKenzie Supplies",1808,2012-08-14,8840.53
"McKenzie Supplies",1809,2012-08-15,4031.11
"McKenzie Supplies",1812,2012-08-16,6595.94
"McKenzie Supplies",1813,2012-08-16,1655.78
"McKenzie Supplies",1814,2012-08-26,9461.71
"McKenzie Supplies",1815,2012-08-27,3641.27
"McKenzie Supplies",1816,2012-08-27,2652.18
"McKenzie Supplies",1817,2012-09-06,6518.78
"McKenzie Supplies",1818,2012-09-12,5371.49
"McKenzie Supplies",1819,2012-09-13,8092.66
"McKenzie Supplies",1820,2012-09-21,9605.35
"McKenzie Supplies",1821,2012-09-27,9203.51
"McKenzie Supplies",1824,2012-09-27,3310.93
"McKenzie Supplies",1825,2012-10-03,2564.62
"McKenzie Supplies",1826,2012-10-09,2335.27
"McKenzie Supplies",1828,2012-10-15,4935.06
"McKenzie Supplies",1829,2012-10-16,2644.12
"McKenzie Supplies",1830,2012-10-16,7627.99
"McKenzie Supplies",1831,2012-10-22,7332.92
"McKenzie Supplies",1832,2012-10-22,3292.62
"McKenzie Supplies",1834,2012-10-22,3634.27
"McKenzie Supplies",1835,2012-10-22,3729.83
"McKenzie Supplies",1836,2012-11-01,8833.18
"McKenzie Supplies",1837,2012-11-02,1742.04
"McKenzie Supplies",1838,2012-11-03,2997.61
"McKenzie Supplies",1839,2012-11-03,1721.93
"McKenzie Supplies",1840,2012-11-04,4361.96
"McKenzie Supplies",1841,2012-11-10,4348.88
"McKenzie Supplies",1842,2012-11-11,2471.49
"McKenzie Supplies",1843,2012-11-12,7603.22
"McKenzie Supplies",1844,2012-11-12,8615.41
"McKenzie Supplies",1845,2012-11-22,8960.49
"McKenzie Supplies",1846,2012-11-23,8602.24
"McKenzie Supplies",1847,2012-12-03,1082.94
"McKenzie Supplies",1848,2012-12-03,8185.45
"McKenzie Supplies",1849,2012-12-09,7113.02
"McKenzie Supplies",1850,2012-12-15,7222.67
"McKenzie Supplies",1851,2012-12-16,5453.88
"McKenzie Supplies",1852,2012-12-16,6570.74
"McKenzie Supplies",1853,2012-12-24,4999.06
"McKenzie Supplies",1854,2013-01-01,3948.07
"McKenzie Supplies",1855,2013-01-11,6552.25
"McKenzie Supplies",1856,2013-01-17,4261.36
"McKenzie Supplies",1857,2013-01-17,7667.57
"McKenzie Supplies",1858,2013-01-23,6187.62
"McKenzie Supplies",1859,2013-01-24,3069.48
"McKenzie Supplies",1860,2013-01-30,7826.99
"McKenzie Supplies",1861,2013-02-09,5503.5
"McKenzie Supplies",1862,2013-02-10,5902.95
"McKenzie Supplies",1863,2013-02-18,7363.98
"McKenzie Supplies",1864,2013-02-24,9653.66
"McKenzie Supplies",1865,2013-03-04,8455.33
"McKenzie Supplies",1866,2013-03-12,2957.29
"McKenzie Supplies",1867,2013-03-12,4911.62
"McKenzie Supplies",1868,2013-03-12,3650.28
"McKenzie Supplies",1870,2013-03-13,7759.92
"McKenzie Supplies",1871,2013-03-21,4302.2
"McKenzie Supplies",1872,2013-03-21,8255.53
"McKenzie Supplies",1873,2013-03-22,2703.35
"McKenzie Supplies",1874,2013-03-30,1945.45
"McKenzie Supplies",1875,2013-03-31,3062.22
"McKenzie Supplies",1876,2013-04-06,4877.44
"McKenzie Supplies",1877,2013-04-12,8745.88
"McKenzie Supplies",1878,2013-04-13,3005.84
"McKenzie Supplies",1879,2013-04-14,3271.88
"McKenzie Supplies",1881,2013-04-15,2707.39
"McKenzie Supplies",1883,2013-04-15,5176.54
"McKenzie Supplies",1884,2013-04-23,1615.62
"McKenzie Supplies",1885,2013-04-24,8476.26
"McKenzie Supplies",1886,2013-04-25,4490.28
"McKenzie Supplies",1887,2013-04-26,8931.74
"McKenzie Supplies",1889,2013-05-02,2688.79
"McKenzie Supplies",1890,2013-05-02,5296.06
"McKenzie Supplies",1891,2013-05-12,3721.84
"McKenzie Supplies",1893,2013-05-12,9768.52
"McKenzie Supplies",1894,2013-05-13,1821.2
"McKenzie Supplies",1895,2013-05-19,9552.32
"McKenzie Supplies",1896,2013-05-25,1585.23
"McKenzie Supplies",1899,2013-05-26,5967.53
"McKenzie Supplies",1900,2013-06-05,2089.77
"McKenzie Supplies",1901,2013-06-06,6342.83
"McKenzie Supplies",1902,2013-06-12,8715.99
"McKenzie Supplies",1903,2013-06-13,7459.8
"McKenzie Supplies",1904,2013-06-19,5902.46
"McKenzie Supplies",1905,2013-06-20,6674.55
"McKenzie Supplies",1906,2013-06-20,1487.67
"McKenzie Supplies",1908,2013-06-21,6838.96
"McKenzie Supplies",1909,2013-06-21,2695.62
"McKenzie Supplies",1910,2013-06-27,5249.09
"McKenzie Supplies",1911,2013-07-03,5970.94
"McKenzie Supplies",1912,2013-07-03,9817.91
"McKenzie Supplies",1913,2013-07-03,9735.56
"McKenzie Supplies",1914,2013-07-09,5709.81
"McKenzie Supplies",1915,2013-07-09,4289.58
"McKenzie Supplies",1916,2013-07-09,4885.53
"McKenzie Supplies",1917,2013-07-19,8056.78
"McKenzie Supplies",1918,2013-07-27,3771.91
"McKenzie Supplies",1919,2013-07-28,9843.44
"McKenzie Supplies",1920,2013-07-29,1312.74
"McKenzie Supplies",1921,2013-07-29,5573.85
"McKenzie Supplies",1922,2013-08-08,2544.46
"McKenzie Supplies",1923,2013-08-08,1366.88
"McKenzie Supplies",1924,2013-08-09,6026.48
"McKenzie Supplies",1925,2013-08-15,9894.49
"McKenzie Supplies",1926,2013-08-25,6916.52
"McKenzie Supplies",1928,2013-08-26,7368.24
"McKenzie Supplies",1930,2013-08-27,8093.77
"McKenzie Supplies",1931,2013-08-27,7706.76
"McKenzie Supplies",1932,2013-08-27,9198.35
"McKenzie Supplies",1934,2013-08-27,9623.62
"McKenzie Supplies",1935,2013-08-28,9871.8
"McKenzie Supplies",1936,2013-09-03,2809.75
"McKenzie Supplies",1937,2013-09-13,9860.04
"McKenzie Supplies",1938,2013-09-13,5737.75
"McKenzie Supplies",1939,2013-09-13,7413.71
"McKenzie Supplies",1940,2013-09-13,8928.33
"McKenzie Supplies",1941,2013-09-14,2941.72
"McKenzie Supplies",1942,2013-09-20,3396.73
"McKenzie Supplies",1943,2013-09-30,3723.43
"McKenzie Supplies",1944,2013-10-06,2782.53
"McKenzie Supplies",1945,2013-10-06,3989.82
"McKenzie Supplies",1946,2013-10-06,3214.64
"McKenzie Supplies",1947,2013-10-14,1395.88
"McKenzie Supplies",1948,2013-10-20,8249.17
"McKenzie Supplies",1949,2013-10-21,1428.68
"McKenzie Supplies",1950,2013-10-21,7039.85
"McKenzie Supplies",1951,2013-10-21,4878.68
"McKenzie Supplies",1952,2013-10-27,6206.63
"McKenzie Supplies",1953,2013-10-27,1879.92
"McKenzie Supplies",1954,2013-10-27,8493.03
"McKenzie Supplies",1955,2013-10-28,1645.17
"McKenzie Supplies",1956,2013-10-29,8327.53
"McKenzie Supplies",1957,2013-10-30,3529.15
"McKenzie Supplies",1958,2013-10-30,9060.09
"McKenzie Supplies",1959,2013-11-05,2013.25
"McKenzie Supplies",1960,2013-11-11,9082.7
"McKenzie Supplies",1961,2013-11-12,5897.74
"McKenzie Supplies",1962,2013-11-12,5566.02
"McKenzie Supplies",1963,2013-11-13,2455.67
"McKenzie Supplies",1965,2013-11-19,2322.06
"McKenzie Supplies",1966,2013-11-19,3426.64
"McKenzie Supplies",1967,2013-11-19,3128.08
"McKenzie Supplies",1968,2013-11-19,5311.4
"McKenzie Supplies",1969,2013-11-20,4197.34
"McKenzie Supplies",1970,2013-11-21,9919.92
"McKenzie Supplies",1971,2013-11-21,4227.47
"McKenzie Supplies",1972,2013-11-22,6197.89
"McKenzie Supplies",1973,2013-11-28,6531.79
"McKenzie Supplies",1975,2013-11-28,3055.59
"McKenzie Supplies",1976,2013-12-04,6680.83
"McKenzie Supplies",1977,2013-12-04,6145.59
"McKenzie Supplies",1978,2013-12-12,6630.1
"McKenzie Supplies",1979,2013-12-13,8677.1
"McKenzie Supplies",1980,2013-12-14,7534.2
"McKenzie Supplies",1981,2013-12-14,3527.55
"McKenzie Supplies",1982,2013-12-14,2060.46
"McKenzie Supplies",1983,2013-12-20,3912.99
"McKenzie Supplies",1984,2013-12-28,4729.25
"McKenzie Supplies",1985,2014-01-07,3106.77
"McKenzie Supplies",1986,2014-01-07,9091.81
"McKenzie Supplies",1987,2014-01-08,4901.46
"McKenzie Supplies",1988,2014-01-09,8404.79
"McKenzie Supplies",1989,2014-01-10,1961.19
"McKenzie Supplies",1990,2014-01-16,8744.77
"McKenzie Supplies",1991,2014-01-26,3486.13
"McKenzie Supplies",1992,2014-02-05,7553.04
"McKenzie Supplies",1993,2014-02-06,9402.99
"McKenzie Supplies",1994,2014-02-06,4804.59
"McKenzie Supplies",1995,2014-02-07,9204.86
"McKenzie Supplies",1996,2014-02-15,5227.62
"McKenzie Supplies",1997,2014-02-16,5077.8
"McKenzie Supplies",1998,2014-02-17,2906.12
"McKenzie Supplies",1999,2014-02-18,7535.15
"McKenzie Supplies",2001,2014-02-24,2518.84
"McKenzie Supplies",2002,2014-02-25,1030.98
"McKenzie Supplies",2003,2014-02-26,1977.53
"McKenzie Supplies",2004,2014-03-04,8153.3
"McKenzie Supplies",2005,2014-03-04,5204.59
"McKenzie Supplies",2007,2014-03-05,6237.36
"McKenzie Supplies",2008,2014-03-06,5719.85
"McKenzie Supplies",2009,2014-03-07,8955.68
"McKenzie Supplies",2010,2014-03-07,9194.77
"McKenzie Supplies",2011,2014-03-07,3699.87
"McKenzie Supplies",2012,2014-03-17,4380.93
"McKenzie Supplies",2013,2014-03-17,6930.55
"McKenzie Supplies",2015,2014-03-23,8410.96
"McKenzie Supplies",2016,2014-03-23,6772.65
"McKenzie Supplies",2019,2014-03-24,1022.65
"McKenzie Supplies",2020,2014-03-25,6255.43
"McKenzie Supplies",2021,2014-04-04,3476.55
"McKenzie Supplies",2022,2014-04-04,7695.47
"McKenzie Supplies",2023,2014-04-04,5145.64
"McKenzie Supplies",2024,2014-04-04,3381.09
"McKenzie Supplies",2025,2014-04-04,9463.48
"McKenzie Supplies",2026,2014-04-04,2337.55
"McKenzie Supplies",2027,2014-04-10,2049.05
"McKenzie Supplies",2028,2014-04-10,8119.4
"McKenzie Supplies",2029,2014-04-11,4518.47
"McKenzie Supplies",2030,2014-04-21,6192.6
"McKenzie Supplies",2031,2014-04-21,6570.95
"McKenzie Supplies",2033,2014-04-22,9826.87
"McKenzie Supplies",2034,2014-05-02,5997.17
"McKenzie Supplies",2035,2014-05-08,7965.93
"McKenzie Supplies",2038,2014-05-09,1357.7
"McKenzie Supplies",2039,2014-05-10,4135.4
"McKenzie Supplies",2040,2014-05-18,4205.01
"McKenzie Supplies",2041,2014-05-18,4852.48
"McKenzie Supplies",2042,2014-05-28,4925.95
"McKenzie Supplies",2043,2014-05-29,7281.19
"McKenzie Supplies",2044,2014-05-29,2310.54
"McKenzie Supplies",2045,2014-06-04,2390.86
"McKenzie Supplies",2046,2014-06-04,9064.53
"McKenzie Supplies",2047,2014-06-10,5946.07
"Printer Mart",8,2006-01-01,1014.14
"Printer Mart",14,2006-01-02,6029.25
"Printer Mart",18,2006-01-12,7528.88
"Printer Mart",23,2006-01-12,7368.73
"Printer Mart",29,2006-01-12,8580.51
"Printer Mart",32,2006-01-12,7442.71
"Printer Mart",36,2006-01-12,1369.86
"Printer Mart",42,2006-01-12,3323.41
"Printer Mart",47,2006-01-13,1219.99
"Printer Mart",51,2006-01-14,2211.84
"Printer Mart",60,2006-01-14,4577.15
"Printer Mart",65,2006-01-24,1776.87
"Printer Mart",69,2006-01-25,1061.12
"Printer Mart",73,2006-01-26,3472.97
"Printer Mart",77,2006-01-27,4159.8
"Printer Mart",81,2006-02-04,1888.98
"Printer Mart",88,2006-02-14,9112.54
"Printer Mart",93,2006-02-24,2973.85
"Printer Mart",97,2006-02-24,4908.43
"Printer Mart",101,2006-03-06,3333.82
"Printer Mart",110,2006-03-16,1165.33
"Printer Mart",112,2006-03-17,1125.47
"Printer Mart",116,2006-03-17,7631.98
"Printer Mart",125,2006-03-23,1040.82
"Printer Mart",126,2006-03-24,4537.73
"Printer Mart",134,2006-03-25,3201.07
"Printer Mart",138,2006-04-02,5723.02
"Printer Mart",143,2006-04-02,4503.87
"Printer Mart",148,2006-04-02,2975.02
"Printer Mart",151,2006-04-03,1791.63
"Printer Mart",159,2006-04-03,1704.57
"Printer Mart",161,2006-04-04,2138.12
"Printer Mart",167,2006-04-10,2364.91
"Printer Mart",171,2006-04-10,1447.87
"Printer Mart",180,2006-04-10,8072.82
"Printer Mart",184,2006-04-10,1443.26
"Printer Mart",189,2006-04-11,2170.75
"Printer Mart",191,2006-04-17,2701.28
"Printer Mart",199,2006-04-17,5187.26
"Printer Mart",201,2006-04-17,5081.3
"Printer Mart",208,2006-04-23,2410.47
"Printer Mart",214,2006-04-24,2777.1
"Printer Mart",220,2006-04-30,2607.94
"Printer Mart",223,2006-04-30,4638.31
"Printer Mart",228,2006-05-08,1309.61
"Printer Mart",235,2006-05-09,1719.18
"Printer Mart",239,2006-05-10,1292.03
"Printer Mart",244,2006-05-16,1542.87
"Printer Mart",248,2006-05-22,1821.42
"Printer Mart",255,2006-05-22,6839.84
"Printer Mart",259,2006-05-23,1027.42
"Printer Mart",261,2006-05-23,1130.15
"Printer Mart",267,2006-05-23,1705.56
"Printer Mart",271,2006-05-23,8450.78
"Printer Mart",277,2006-05-24,8843.1
"Printer Mart",281,2006-05-24,1171.96
"Printer Mart",289,2006-05-30,1300.67
"Printer Mart",291,2006-06-07,7340.29
"Printer Mart",297,2006-06-13,2612.96
"Printer Mart",304,2006-06-14,2735.54
"Printer Mart",306,2006-06-20,2413.67
"Printer Mart",314,2006-06-28,6446.77
"Printer Mart",320,2006-06-28,4840.07
"Printer Mart",322,2006-06-29,1285.19
"Printer Mart",326,2006-07-05,3026.8
"Printer Mart",334,2006-07-15,1784.8
"Printer Mart",340,2006-07-15,3941.41
"Printer Mart",341,2006-07-15,2240.56
"Printer Mart",349,2006-07-16,1938.29
"Printer Mart",355,2006-07-26,2734.38
"Printer Mart",358,2006-07-26,2184.42
"Printer Mart",363,2006-07-26,2426.13
"Printer Mart",367,2006-08-03,1956.67
"Printer Mart",374,2006-08-04,4522.71
"Printer Mart",377,2006-08-04,1083.6
"Printer Mart",385,2006-08-10,2849.48
"Printer Mart",388,2006-08-16,5784.31
"Printer Mart",392,2006-08-17,2070.18
"Printer Mart",397,2006-08-17,1088.38
"Printer Mart",405,2006-08-23,6960.39
"Printer Mart",408,2006-08-29,4148.65
"Printer Mart",411,2006-09-04,7382.67
"Printer Mart",417,2006-09-04,2248.69
"Printer Mart",421,2006-09-10,7410.17
"Printer Mart",427,2006-09-10,4343.0
"Printer Mart",432,2006-09-10,1732.92
"Printer Mart",440,2006-09-20,1257.27
"Printer Mart",441,2006-09-21,3082.03
"Printer Mart",446,2006-09-21,1852.51
"Printer Mart",455,2006-09-21,1255.65
"Printer Mart",457,2006-09-21,4435.7
"Printer Mart",461,2006-09-22,3800.42
"Printer Mart",469,2006-09-28,3088.66
"Printer Mart",473,2006-09-29,3652.28
"Printer Mart",480,2006-09-29,1916.65
"Printer Mart",484,2006-09-29,3198.96
"Printer Mart",487,2006-09-30,2563.18
"Printer Mart",493,2006-09-30,5136.67
"Printer Mart",496,2006-10-08,8275.35
"Printer Mart",502,2006-10-14,3917.69
"Printer Mart",510,2006-10-15,4830.95
"Printer Mart",511,2006-10-23,5111.71
"Printer Mart",520,2006-10-24,1692.39
"Printer Mart",523,2006-10-24,1748.04
"Printer Mart",529,2006-10-24,7767.26
"Printer Mart",533,2006-10-24,1055.65
"Printer Mart",539,2006-10-30,3829.5
"Printer Mart",543,2006-10-31,1483.04
"Printer Mart",549,2006-11-06,4099.11
"Printer Mart",554,2006-11-06,1713.32
"Printer Mart",556,2006-11-07,6969.51
"Printer Mart",561,2006-11-07,2075.45
"Printer Mart",568,2006-11-15,6307.01
"Printer Mart",574,2006-11-21,6601.49
"Printer Mart",579,2006-11-27,1053.95
"Printer Mart",583,2006-11-27,1132.2
"Printer Mart",590,2006-11-28,9753.19
"Printer Mart",593,2006-11-28,4882.15
"Printer Mart",599,2006-11-28,4509.7
"Printer Mart",603,2006-12-04,3629.5
"Printer Mart",609,2006-12-12,1033.56
"Printer Mart",611,2006-12-18,1024.74
"Printer Mart",617,2006-12-18,2204.59
"Printer Mart",621,2006-12-18,6632.31
"Printer Mart",629,2006-12-19,8740.64
"Printer Mart",632,2006-12-29,4991.74
"Printer Mart",637,2007-01-08,1134.67
"Printer Mart",644,2007-01-14,5665.69
"Printer Mart",647,2007-01-20,1654.61
"Printer Mart",655,2007-01-26,1631.25
"Printer Mart",658,2007-01-26,5412.13
"Printer Mart",665,2007-02-03,9677.21
"Printer Mart",670,2007-02-03,1721.97
"Printer Mart",674,2007-02-13,2965.87
"Printer Mart",677,2007-02-19,8525.91
"Printer Mart",682,2007-02-25,5060.95
"Printer Mart",688,2007-03-03,1453.46
"Printer Mart",693,2007-03-13,2851.82
"Printer Mart",696,2007-03-14,4004.88
"Printer Mart",704,2007-03-20,4619.11
"Printer Mart",710,2007-03-21,6955.1
"Printer Mart",711,2007-03-22,1553.37
"Printer Mart",720,2007-03-23,3926.4
"Printer Mart",723,2007-04-02,2465.18
"Printer Mart",727,2007-04-08,3359.13
"Printer Mart",732,2007-04-09,9822.55
"Printer Mart",737,2007-04-09,1522.15
"Printer Mart",744,2007-04-09,8165.08
"Printer Mart",747,2007-04-09,7006.5
"Printer Mart",755,2007-04-19,8098.67
"Printer Mart",757,2007-04-20,3443.25
"Printer Mart",762,2007-04-26,1930.74
"Printer Mart",766,2007-05-02,1462.57
"Printer Mart",774,2007-05-10,1703.44
"Printer Mart",780,2007-05-20,1446.53
"Printer Mart",781,2007-05-21,6897.05
"Printer Mart",789,2007-05-21,2125.0
"Printer Mart",793,2007-05-22,2203.03
"Printer Mart",796,2007-05-22,6485.69
"Printer Mart",801,2007-05-22,5909.47
"Printer Mart",810,2007-05-28,1860.46
"Printer Mart",814,2007-06-07,2944.72
"Printer Mart",817,2007-06-07,1734.79
"Printer Mart",822,2007-06-13,1812.17
"Printer Mart",828,2007-06-19,2003.37
"Printer Mart",831,2007-06-19,5585.4
"Printer Mart",837,2007-06-27,1644.97
"Printer Mart",845,2007-07-05,1152.7
"Printer Mart",846,2007-07-11,8855.52
"Printer Mart",855,2007-07-11,6218.47
"Printer Mart",860,2007-07-11,3160.42
"Printer Mart",862,2007-07-21,9857.81
"Printer Mart",870,2007-07-31,3237.37
"Printer Mart",872,2007-08-01,1834.24
"Printer Mart",879,2007-08-07,4296.95
"Printer Mart",885,2007-08-13,4966.54
"Printer Mart",886,2007-08-14,1145.51
"Printer Mart",893,2007-08-24,6425.17
"Printer Mart",897,2007-08-25,5628.83
"Printer Mart",905,2007-09-04,5312.01
"Printer Mart",907,2007-09-04,3614.37
"Printer Mart",914,2007-09-14,5225.01
"Printer Mart",916,2007-09-15,1676.5
"Printer Mart",923,2007-09-16,1963.53
"Printer Mart",927,2007-09-24,2777.07
"Printer Mart",933,2007-09-24,1150.85
"Printer Mart",938,2007-09-25,1260.87
"Printer Mart",942,2007-09-26,3544.46
"Printer Mart",948,2007-10-04,1455.94
"Printer Mart",951,2007-10-04,1371.39
"Printer Mart",959,2007-10-04,3312.72
"Printer Mart",963,2007-10-05,1349.32
"Printer Mart",968,2007-10-13,1075.88
"Printer Mart",975,2007-10-19,1333.43
"Printer Mart",977,2007-10-29,2033.44
"Printer Mart",983,2007-10-29,8722.44
"Printer Mart",988,2007-10-29,6193.13
"Printer Mart",994,2007-10-30,3477.46
"Printer Mart",996,2007-11-09,8968.58
"Printer Mart",1003,2007-11-10,5239.48
"Printer Mart",1006,2007-11-20,2533.7
"Printer Mart",1012,2007-11-26,1974.86
"Printer Mart",1018,2007-12-02,4775.85
"Printer Mart",1021,2007-12-08,5969.67
"Printer Mart",1030,2007-12-08,7732.94
"Printer Mart",1034,2007-12-16,3253.93
"Printer Mart",1038,2007-12-16,8180.8
"Printer Mart",1045,2007-12-22,1677.75
"Printer Mart",1050,2007-12-23,2789.22
"Printer Mart",1053,2007-12-24,1384.03
"Printer Mart",1058,2007-12-30,9769.87
"Printer Mart",1063,2007-12-30,1450.51
"Printer Mart",1069,2007-12-30,7313.4
"Printer Mart",1073,2007-12-31,1500.69
"Printer Mart",1076,2008-01-01,5266.6
"Printer Mart",1085,2008-01-01,1233.15
"Printer Mart",1087,2008-01-11,7119.78
"Printer Mart",1092,2008-01-11,9227.63
"Printer Mart",1098,2008-01-11,1278.29
"Printer Mart",1102,2008-01-12,1486.21
"Printer Mart",1110,2008-01-18,8362.73
"Printer Mart",1113,2008-01-24,2916.6
"Printer Mart",1117,2008-01-25,2184.51
"Printer Mart",1121,2008-01-25,1820.55
"Printer Mart",1129,2008-01-26,1516.18
"Printer Mart",1131,2008-02-05,1069.54
"Printer Mart",1140,2008-02-06,2670.08
"Printer Mart",1141,2008-02-07,7234.59
"Printer Mart",1149,2008-02-07,3017.05
"Printer Mart",1152,2008-02-17,4919.59
"Printer Mart",1160,2008-02-17,9810.75
"Printer Mart",1161,2008-02-27,1343.03
"Printer Mart",1169,2008-02-28,1299.73
"Printer Mart",1171,2008-02-28,1953.98
"Printer Mart",1179,2008-02-28,3364.01
"Printer Mart",1183,2008-03-07,4745.74
"Printer Mart",1189,2008-03-13,2225.78
"Printer Mart",1194,2008-03-23,1691.08
"Printer Mart",1199,2008-03-29,1185.89
"Printer Mart",1202,2008-04-04,3457.53
"Printer Mart",1209,2008-04-04,2367.47
"Printer Mart",1213,2008-04-04,4257.41
"Printer Mart",1216,2008-04-04,2482.33
"Printer Mart",1222,2008-04-10,2671.28
"Printer Mart",1230,2008-04-10,4265.18
"Printer Mart",1231,2008-04-10,3033.42
"Printer Mart",1236,2008-04-20,5225.96
"Printer Mart",1243,2008-04-21,3744.25
"Printer Mart",1249,2008-04-27,1017.93
"Printer Mart",1254,2008-04-27,9153.2
"Printer Mart",1256,2008-04-28,3436.46
"Printer Mart",1265,2008-05-08,8973.56
"Printer Mart",1268,2008-05-08,2788.07
"Printer Mart",1275,2008-05-14,5168.33
"Printer Mart",1280,2008-05-14,3797.31
"Printer Mart",1284,2008-05-15,6900.31
"Printer Mart",1290,2008-05-21,3086.31
"Printer Mart",1292,2008-05-27,1680.55
"Printer Mart",1298,2008-06-06,5914.91
"Printer Mart",1305,2008-06-07,8698.74
"Printer Mart",1306,2008-06-13,2319.71
"Printer Mart",1311,2008-06-13,9388.15
"Printer Mart",1318,2008-06-23,2368.93
"Printer Mart",1321,2008-07-01,3276.9
"Printer Mart",1326,2008-07-02,2403.4
"Printer Mart",1333,2008-07-03,2095.37
"Printer Mart",1337,2008-07-03,7052.55
"Printer Mart",1344,2008-07-04,1880.0
"Printer Mart",1350,2008-07-05,1756.73
"Printer Mart",1351,2008-07-06,2273.76
"Printer Mart",1358,2008-07-14,3010.76
"Printer Mart",1361,2008-07-15,1400.76
"Printer Mart",1370,2008-07-16,3032.34
"Printer Mart",1375,2008-07-24,1799.5
"Printer Mart",1376,2008-07-30,5123.52
"Printer Mart",1382,2008-07-31,5127.98
"Printer Mart",1386,2008-08-10,3295.2
"Printer Mart",1392,2008-08-10,1407.73
"Printer Mart",1396,2008-08-18,2064.6
"Printer Mart",1401,2008-08-18,6602.36
"Printer Mart",1409,2008-08-18,4368.02
"Printer Mart",1414,2008-08-28,2725.55
"Printer Mart",1419,2008-08-29,6091.34
"Printer Mart",1421,2008-09-04,1344.21
"Printer Mart",1427,2008-09-04,1995.23
"Printer Mart",1434,2008-09-14,7561.93
"Printer Mart",1436,2008-09-14,8156.66
"Printer Mart",1444,2008-09-20,1731.75
"Printer Mart",1447,2008-09-20,3016.41
"Printer Mart",1451,2008-09-21,3145.08
"Printer Mart",1458,2008-10-01,3670.45
"Printer Mart",1462,2008-10-01,1152.08
"Printer Mart",1467,2008-10-11,7664.1
"Printer Mart",1475,2008-10-19,3654.54
"Printer Mart",1476,2008-10-25,1302.38
"Printer Mart",1483,2008-10-26,2727.24
"Printer Mart",1490,2008-10-27,2386.72
"Printer Mart",1495,2008-10-28,4770.11
"Printer Mart",1500,2008-11-03,5480.38
"Printer Mart",1502,2008-11-03,5013.85
"Printer Mart",1507,2008-11-04,9762.07
"Printer Mart",1513,2008-11-05,1466.85
"Printer Mart",1519,2008-11-11,1036.1
"Printer Mart",1523,2008-11-11,1140.52
"Printer Mart",1527,2008-11-21,1277.07
"Printer Mart",1534,2008-11-22,1508.11
"Printer Mart",1538,2008-11-22,1212.3
"Printer Mart",1545,2008-11-22,2052.22
"Printer Mart",1546,2008-11-23,3340.13
"Printer Mart",1555,2008-12-03,1034.01
"Printer Mart",1559,2008-12-13,2213.65
"Printer Mart",1561,2008-12-13,2193.52
"Printer Mart",1569,2008-12-14,3297.57
"Printer Mart",1575,2008-12-15,1175.54
"Printer Mart",1579,2008-12-16,2182.12
"Printer Mart",1581,2008-12-17,5000.1
"Printer Mart",1590,2008-12-25,5370.91
"Printer Mart",1595,2008-12-25,2712.5
"Printer Mart",1600,2008-12-25,2806.44
"Printer Mart",1601,2008-12-25,2320.03
"Printer Mart",1609,2008-12-26,4726.16
"Printer Mart",1613,2009-01-03,1165.59
"Printer Mart",1620,2009-01-04,2717.92
"Printer Mart",1624,2009-01-05,6501.73
"Printer Mart",1626,2009-01-05,3987.69
"Printer Mart",1635,2009-01-05,3986.96
"Printer Mart",1639,2009-01-05,3625.38
"Printer Mart",1641,2009-01-06,5990.85
"Printer Mart",1650,2009-01-07,4483.16
"Printer Mart",1654,2009-01-13,8495.51
"Printer Mart",1660,2009-01-23,1459.03
"Printer Mart",1664,2009-01-24,1308.18
"Printer Mart",1669,2009-01-24,1631.09
"Printer Mart",1672,2009-02-03,1431.75
"Printer Mart",1680,2009-02-04,2892.15
"Printer Mart",1682,2009-02-12,6940.19
"Printer Mart",1688,2009-02-22,4855.53
"Printer Mart",1691,2009-02-22,3127.87
"Printer Mart",1697,2009-03-04,2343.79
"Printer Mart",1703,2009-03-14,3477.02
"Printer Mart",1708,2009-03-15,5994.61
"Printer Mart",1715,2009-03-21,4685.06
"Printer Mart",1720,2009-03-29,1066.71
"Printer Mart",1724,2009-03-30,5677.96
"Printer Mart",1727,2009-03-31,6620.21
"Printer Mart",1731,2009-04-10,4805.94
"Printer Mart",1739,2009-04-11,5002.5
"Printer Mart",1744,2009-04-17,5963.98
"Printer Mart",1750,2009-04-18,3245.36
"Printer Mart",1751,2009-04-19,1138.56
"Printer Mart",1756,2009-04-19,2343.6
"Printer Mart",1764,2009-04-27,4400.93
"Printer Mart",1769,2009-04-27,2473.08
"Printer Mart",1771,2009-05-03,5992.54
"Printer Mart",1780,2009-05-03,9161.13
"Printer Mart",1781,2009-05-03,1348.76
"Printer Mart",1787,2009-05-09,6676.96
"Printer Mart",1794,2009-05-19,1300.01
"Printer Mart",1800,2009-05-29,1943.69
"Printer Mart",1803,2009-05-30,1650.74
"Printer Mart",1806,2009-06-09,5511.8
"Printer Mart",1814,2009-06-17,1331.33
"Printer Mart",1816,2009-06-17,7097.95
"Printer Mart",1822,2009-06-18,2643.95
"Printer Mart",1828,2009-06-24,4083.02
"Printer Mart",1832,2009-06-25,5717.75
"Printer Mart",1836,2009-06-25,4440.95
"Printer Mart",1842,2009-06-26,1856.65
"Printer Mart",1850,2009-07-06,7395.81
"Printer Mart",1852,2009-07-07,8839.13
"Printer Mart",1860,2009-07-08,5418.04
"Printer Mart",1864,2009-07-14,1651.46
"Printer Mart",1867,2009-07-24,2493.08
"Printer Mart",1871,2009-08-03,9413.54
"Printer Mart",1876,2009-08-04,2022.51
"Printer Mart",1881,2009-08-10,3171.77
"Printer Mart",1888,2009-08-10,3408.46
"Printer Mart",1892,2009-08-16,6057.57
"Printer Mart",1897,2009-08-24,1811.52
"Printer Mart",1905,2009-08-25,1029.71
"Printer Mart",1909,2009-08-31,3672.35
"Printer Mart",1913,2009-09-06,2612.32
"Printer Mart",1919,2009-09-16,1486.55
"Printer Mart",1923,2009-09-24,1291.01
"Printer Mart",1929,2009-09-24,2806.06
"Printer Mart",1931,2009-09-24,1335.58
"Printer Mart",1940,2009-09-25,2030.99
"Printer Mart",1944,2009-10-01,2156.6
"Printer Mart",1949,2009-10-11,2860.33
"Printer Mart",1955,2009-10-11,8565.39
"Printer Mart",1960,2009-10-17,1575.42
"Printer Mart",1963,2009-10-18,5407.04
"Printer Mart",1970,2009-10-24,1670.83
"Printer Mart",1971,2009-10-30,2197.72
"Printer Mart",1979,2009-10-31,1766.7
"Printer Mart",1981,2009-11-01,1649.7
"Printer Mart",1988,2009-11-01,2062.36
"Printer Mart",1992,2009-11-11,1557.76
"Printer Mart",2000,2009-11-12,1629.08
"Printer Mart",2002,2009-11-18,5139.56
"Printer Mart",2010,2009-11-18,1811.26
"Printer Mart",2011,2009-11-19,1291.18
"Printer Mart",2019,2009-11-29,4313.85
"Printer Mart",2024,2009-12-07,2990.39
"Printer Mart",2030,2009-12-17,6834.98
"Printer Mart",2033,2009-12-18,4360.79
"Printer Mart",2037,2009-12-19,1882.53
"Printer Mart",2042,2009-12-25,3987.89
"Printer Mart",2049,2009-12-31,4237.05
"Printer Mart",2053,2009-12-31,4376.6
"Printer Mart",2059,2009-12-31,3647.12
"Printer Mart",2062,2010-01-08,4510.94
"Printer Mart",2067,2010-01-16,5514.29
"Printer Mart",2073,2010-01-16,5749.38
"Printer Mart",2078,2010-01-17,2328.95
"Printer Mart",2083,2010-01-17,7617.42
"Printer Mart",2089,2010-01-23,1124.26
"Printer Mart",2092,2010-02-02,3336.33
"Printer Mart",2098,2010-02-03,1032.44
"Printer Mart",2104,2010-02-03,4947.52
"Printer Mart",2107,2010-02-11,2267.17
"Printer Mart",2111,2010-02-21,1531.23
"Printer Mart",2120,2010-02-22,2466.19
"Printer Mart",2121,2010-02-23,9113.29
"Printer Mart",2129,2010-02-23,1920.29
"Printer Mart",2135,2010-03-01,1118.05
"Printer Mart",2139,2010-03-02,9763.1
"Printer Mart",2143,2010-03-02,9243.52
"Printer Mart",2147,2010-03-08,2135.23
"Printer Mart",2155,2010-03-08,3745.61
"Printer Mart",2160,2010-03-09,1911.47
"Printer Mart",2162,2010-03-19,3012.67
"Printer Mart",2169,2010-03-20,1331.69
"Printer Mart",2172,2010-03-30,7163.14
"Printer Mart",2178,2010-04-07,1686.8
"Printer Mart",2185,2010-04-17,2424.25
"Printer Mart",2186,2010-04-18,2383.59
"Printer Mart",2195,2010-04-18,9522.98
"Printer Mart",2197,2010-04-24,1712.48
"Printer Mart",2205,2010-04-25,6354.56
"Printer Mart",2207,2010-05-03,1165.25
"Printer Mart",2213,2010-05-03,4206.75
"Printer Mart",2219,2010-05-03,3116.57
"Printer Mart",2225,2010-05-04,1098.88
"Printer Mart",2227,2010-05-04,4197.05
"Printer Mart",2235,2010-05-05,6082.19
"Printer Mart",2240,2010-05-15,1799.67
"Printer Mart",2244,2010-05-16,2908.69
"Printer Mart",2246,2010-05-26,2693.34
"Printer Mart",2254,2010-06-03,7714.61
"Printer Mart",2256,2010-06-04,7112.45
"Printer Mart",2261,2010-06-10,2071.19
"Printer Mart",2268,2010-06-11,7889.89
"Printer Mart",2271,2010-06-11,9996.03
"Printer Mart",2276,2010-06-12,2025.1
"Printer Mart",2281,2010-06-18,1912.6
"Printer Mart",2288,2010-06-26,1739.87
"Printer Mart",2292,2010-06-27,6855.17
"Printer Mart",2296,2010-06-27,3867.15
"Printer Mart",2305,2010-06-27,1004.1
"Printer Mart",2306,2010-06-28,6703.36
"Printer Mart",2314,2010-07-04,4761.81
"Printer Mart",2318,2010-07-12,1767.54
"Printer Mart",2323,2010-07-13,1111.12
"Printer Mart",2330,2010-07-14,5990.27
"Printer Mart",2333,2010-07-14,1940.31
"Printer Mart",2337,2010-07-15,2874.48
"Printer Mart",2345,2010-07-16,1690.61
"Printer Mart",2348,2010-07-26,1694.07
"Printer Mart",2353,2010-07-27,1291.01
"Printer Mart",2359,2010-07-28,1734.82
"Printer Mart",2361,2010-08-03,9116.68
"Printer Mart",2369,2010-08-04,9714.72
"Printer Mart",2372,2010-08-05,4451.23
"Printer Mart",2377,2010-08-11,2360.51
"Printer Mart",2385,2010-08-12,4344.42
"Printer Mart",2386,2010-08-13,6943.39
"Printer Mart",2395,2010-08-14,2089.86
"Printer Mart",2400,2010-08-15,9687.74
"Printer Mart",2403,2010-08-15,2020.05
"Printer Mart",2409,2010-08-16,9254.82
"Printer Mart",2411,2010-08-22,6919.79
"Printer Mart",2418,2010-08-23,1425.22
"Printer Mart",2423,2010-08-31,7136.78
"Printer Mart",2429,2010-09-01,2145.59
"Printer Mart",2435,2010-09-01,1993.73
"Printer Mart",2437,2010-09-11,1646.75
"Printer Mart",2441,2010-09-17,2440.74
"Printer Mart",2450,2010-09-17,2853.75
"Printer Mart",2451,2010-09-23,3099.92
"Printer Mart",2460,2010-09-23,3056.92
"Printer Mart",2462,2010-10-03,4322.19
"Printer Mart",2468,2010-10-03,1422.62
"Printer Mart",2474,2010-10-03,2744.9
"Printer Mart",2480,2010-10-11,9438.85
"Printer Mart",2483,2010-10-21,2152.04
"Printer Mart",2487,2010-10-21,2712.97
"Printer Mart",2495,2010-10-22,2315.97
"Printer Mart",2497,2010-10-22,6911.2
"Printer Mart",2505,2010-11-01,6301.61
"Printer Mart",2506,2010-11-01,1656.88
"Printer Mart",2512,2010-11-01,5483.96
"Printer Mart",2518,2010-11-01,9553.02
"Printer Mart",2525,2010-11-02,1497.75
"Printer Mart",2530,2010-11-02,9643.92
"Printer Mart",2533,2010-11-08,5695.43
"Printer Mart",2539,2010-11-09,5188.32
"Printer Mart",2543,2010-11-15,4849.27
"Printer Mart",2549,2010-11-21,7364.02
"Printer Mart",2552,2010-11-27,5834.06
"Printer Mart",2556,2010-11-28,5728.97
"Printer Mart",2564,2010-11-29,7307.98
"Printer Mart",2568,2010-12-07,5441.24
"Printer Mart",2573,2010-12-17,5167.33
"Printer Mart",2576,2010-12-18,3836.1
"Printer Mart",2582,2010-12-28,8613.92
"Printer Mart",2587,2010-12-29,4333.68
"Printer Mart",2593,2010-12-30,1286.38
"Printer Mart",2596,2010-12-30,2165.04
"Printer Mart",2601,2011-01-05,6242.89
"Printer Mart",2608,2011-01-11,2928.88
"Printer Mart",2615,2011-01-19,1043.29
"Printer Mart",2617,2011-01-20,7032.98
"Printer Mart",2625,2011-01-20,5859.34
"Printer Mart",2629,2011-01-21,8564.53
"Printer Mart",2632,2011-01-27,3619.18
"Printer Mart",2638,2011-01-28,3261.09
"Printer Mart",2644,2011-01-29,1258.69
"Printer Mart",2649,2011-02-06,6745.76
"Printer Mart",2655,2011-02-07,2153.34
"Printer Mart",2659,2011-02-13,2773.21
"Printer Mart",2664,2011-02-19,5324.4
"Printer Mart",2666,2011-03-01,4211.69
"Printer Mart",2671,2011-03-02,7730.66
"Printer Mart",2676,2011-03-03,4389.7
"Printer Mart",2685,2011-03-04,2442.86
"Printer Mart",2689,2011-03-14,7341.47
"Printer Mart",2693,2011-03-15,1083.97
"Printer Mart",2697,2011-03-16,1113.58
"Printer Mart",2701,2011-03-24,8360.18
"Printer Mart",2707,2011-03-25,2755.24
"Printer Mart",2715,2011-03-26,8279.38
"Printer Mart",2717,2011-03-27,2079.26
"Printer Mart",2721,2011-04-02,7128.92
"Printer Mart",2727,2011-04-02,2824.37
"Printer Mart",2734,2011-04-02,1762.59
"Printer Mart",2739,2011-04-03,5380.42
"Printer Mart",2744,2011-04-04,2269.43
"Printer Mart",2748,2011-04-04,3573.81
"Printer Mart",2755,2011-04-10,5246.48
"Printer Mart",2760,2011-04-10,5507.06
"Printer Mart",2763,2011-04-16,9501.54
"Printer Mart",2769,2011-04-17,2671.94
"Printer Mart",2773,2011-04-27,1241.58
"Printer Mart",2780,2011-04-27,1202.62
"Printer Mart",2785,2011-04-27,4517.29
"Printer Mart",2788,2011-04-27,1636.0
"Printer Mart",2795,2011-04-28,1306.58
"Printer Mart",2800,2011-05-04,2466.39
"Printer Mart",2801,2011-05-14,3408.3
"Printer Mart",2810,2011-05-24,2544.06
"Printer Mart",2814,2011-05-24,3375.76
"Printer Mart",2819,2011-05-24,2431.09
"Printer Mart",2824,2011-05-30,2973.59
"Printer Mart",2827,2011-06-09,2185.89
"Printer Mart",2835,2011-06-09,5414.94
"Printer Mart",2837,2011-06-10,5912.82
"Printer Mart",2844,2011-06-20,2456.27
"Printer Mart",2848,2011-06-26,6937.77
"Printer Mart",2851,2011-06-27,1141.77
"Printer Mart",2856,2011-06-28,1836.51
"Printer Mart",2862,2011-06-29,4827.57
"Printer Mart",2867,2011-06-30,6007.41
"Printer Mart",2873,2011-06-30,1229.14
"Printer Mart",2876,2011-06-30,7431.45
"Printer Mart",2885,2011-06-30,2825.99
"Printer Mart",2890,2011-07-01,2214.41
"Printer Mart",2895,2011-07-01,7380.96
"Printer Mart",2896,2011-07-11,2370.9
"Printer Mart",2903,2011-07-12,1477.72
"Printer Mart",2908,2011-07-13,5357.51
"Printer Mart",2915,2011-07-14,1801.84
"Printer Mart",2917,2011-07-14,4611.49
"Printer Mart",2921,2011-07-15,6402.14
"Printer Mart",2930,2011-07-16,1625.36
"Printer Mart",2932,2011-07-24,1054.14
"Printer Mart",2939,2011-07-30,5380.08
"Printer Mart",2943,2011-07-30,2453.6
"Printer Mart",2950,2011-07-31,1894.82
"Printer Mart",2953,2011-08-08,1479.37
"Printer Mart",2957,2011-08-14,1418.4
"Printer Mart",2965,2011-08-24,6144.03
"Printer Mart",2967,2011-08-25,1144.63
"Printer Mart",2972,2011-09-02,4717.86
"Printer Mart",2976,2011-09-10,1345.59
"Printer Mart",2982,2011-09-11,5342.17
"Printer Mart",2987,2011-09-17,1531.86
"Printer Mart",2991,2011-09-18,2735.81
"Printer Mart",3000,2011-09-19,2688.75
"Printer Mart",3005,2011-09-29,3890.93
"Printer Mart",3009,2011-10-09,1591.53
"Printer Mart",3012,2011-10-15,2706.09
"Printer Mart",3020,2011-10-15,1582.63
"Printer Mart",3024,2011-10-16,8166.22
"Printer Mart",3026,2011-10-17,2388.68
"Printer Mart",3033,2011-10-17,9519.39
"Printer Mart",3038,2011-10-18,5913.48
"Printer Mart",3045,2011-10-18,3919.67
"Printer Mart",3050,2011-10-24,9251.18
"Printer Mart",3052,2011-10-25,1601.12
"Printer Mart",3056,2011-10-26,4196.66
"Printer Mart",3063,2011-10-27,7917.95
"Printer Mart",3067,2011-10-28,4767.91
"Printer Mart",3071,2011-10-28,2143.1
"Printer Mart",3078,2011-10-29,9276.2
"Printer Mart",3082,2011-10-29,1184.72
"Printer Mart",3086,2011-11-04,5933.01
"Printer Mart",3091,2011-11-05,3731.24
"Printer Mart",3096,2011-11-06,3607.25
"Printer Mart",3102,2011-11-07,1984.33
"Printer Mart",3108,2011-11-08,4115.44
"Printer Mart",3113,2011-11-14,1987.75
"Printer Mart",3120,2011-11-20,1233.2
"Printer Mart",3121,2011-11-26,1915.29
"Printer Mart",3126,2011-11-26,2350.39
"Printer Mart",3131,2011-12-06,1522.25
"Printer Mart",3137,2011-12-07,6929.06
"Printer Mart",3141,2011-12-07,9193.7
"Printer Mart",3146,2011-12-08,3429.0
"Printer Mart",3155,2011-12-14,1011.29
"Printer Mart",3160,2011-12-20,3569.74
"Printer Mart",3161,2011-12-28,3259.57
"Printer Mart",3168,2012-01-03,6258.35
"Printer Mart",3173,2012-01-11,4508.04
"Printer Mart",3179,2012-01-11,5049.88
"Printer Mart",3184,2012-01-17,4588.79
"Printer Mart",3189,2012-01-18,3744.35
"Printer Mart",3192,2012-01-24,6228.87
"Printer Mart",3200,2012-01-25,1590.07
"Printer Mart",3202,2012-02-04,1795.59
"Printer Mart",3207,2012-02-04,2643.11
"Printer Mart",3212,2012-02-04,8218.76
"Printer Mart",3219,2012-02-10,2633.08
"Printer Mart",3224,2012-02-16,1720.85
"Printer Mart",3227,2012-02-16,2014.11
"Printer Mart",3233,2012-02-22,4275.65
"Printer Mart",3236,2012-02-23,6134.59
"Printer Mart",3244,2012-03-04,2002.45
"Printer Mart",3246,2012-03-04,5865.59
"Printer Mart",3255,2012-03-04,8869.23
"Printer Mart",3256,2012-03-04,1743.0
"Printer Mart",3263,2012-03-14,1300.17
"Printer Mart",3268,2012-03-22,4499.82
"Printer Mart",3274,2012-03-22,2275.43
"Printer Mart",3277,2012-03-28,2640.56
"Printer Mart",3285,2012-03-29,4772.46
"Printer Mart",3289,2012-03-29,1496.33
"Printer Mart",3293,2012-03-30,1882.12
"Printer Mart",3296,2012-04-09,1494.68
"Printer Mart",3303,2012-04-19,2105.8
"Printer Mart",3306,2012-04-19,6088.88
"Printer Mart",3315,2012-04-29,6633.3
"Printer Mart",3319,2012-05-07,1014.75
"Printer Mart",3322,2012-05-13,7652.78
"Printer Mart",3330,2012-05-14,2081.7
"Printer Mart",3335,2012-05-15,4661.99
"Printer Mart",3337,2012-05-16,1588.9
"Printer Mart",3342,2012-05-24,1779.57
"Printer Mart",3348,2012-05-25,1622.4
"Printer Mart",3353,2012-06-04,1966.85
"Printer Mart",3359,2012-06-04,3068.83
"Printer Mart",3363,2012-06-04,1976.51
"Printer Mart",3369,2012-06-05,3428.63
"Printer Mart",3372,2012-06-11,1360.42
"Printer Mart",3376,2012-06-21,7556.79
"Printer Mart",3382,2012-06-27,3308.32
"Printer Mart",3387,2012-06-27,5191.22
"Printer Mart",3394,2012-07-03,1164.86
"Printer Mart",3397,2012-07-09,9596.18
"Printer Mart",3402,2012-07-19,5156.44
"Printer Mart",3410,2012-07-20,2308.47
"Printer Mart",3413,2012-07-30,1421.05
"Printer Mart",3417,2012-07-30,1839.34
"Printer Mart",3424,2012-07-30,4638.39
"Printer Mart",3428,2012-07-31,1314.65
"Printer Mart",3431,2012-07-31,1081.78
"Printer Mart",3438,2012-07-31,4826.44
"Printer Mart",3445,2012-08-08,2065.52
"Printer Mart",3447,2012-08-08,1628.05
"Printer Mart",3454,2012-08-16,4613.12
"Printer Mart",3457,2012-08-22,8431.31
"Printer Mart",3463,2012-08-22,1071.78
"Printer Mart",3466,2012-08-23,1619.89
"Printer Mart",3472,2012-09-02,1870.58
"Printer Mart",3477,2012-09-12,4460.21
"Printer Mart",3484,2012-09-12,1609.1
"Printer Mart",3490,2012-09-22,2039.22
"Printer Mart",3491,2012-09-30,1817.9
"Printer Mart",3498,2012-09-30,2932.28
"Printer Mart",3501,2012-09-30,4052.72
"Printer Mart",3509,2012-10-10,4885.52
"Printer Mart",3511,2012-10-18,3782.43
"Printer Mart",3520,2012-10-19,1914.58
"Printer Mart",3524,2012-10-19,1028.14
"Printer Mart",3527,2012-10-25,1353.31
"Printer Mart",3535,2012-10-26,8397.69
"Printer Mart",3537,2012-10-27,1801.09
"Printer Mart",3544,2012-10-27,8579.31
"Printer Mart",3547,2012-11-04,3217.24
"Printer Mart",3554,2012-11-10,5057.22
"Printer Mart",3556,2012-11-10,1803.15
"Printer Mart",3562,2012-11-20,7034.74
"Printer Mart",3567,2012-11-28,1644.34
"Printer Mart",3573,2012-11-28,2280.82
"Printer Mart",3579,2012-11-28,3848.57
"Printer Mart",3584,2012-11-28,4286.05
"Printer Mart",3590,2012-11-29,6268.88
"Printer Mart",3592,2012-11-29,5992.54
"Printer Mart",3598,2012-11-30,8489.24
"Printer Mart",3601,2012-12-01,4770.83
"Printer Mart",3610,2012-12-07,3796.01
"Printer Mart",3613,2012-12-08,4614.88
"Printer Mart",3616,2012-12-08,2293.2
"Printer Mart",3623,2012-12-09,1709.31
"Printer Mart",3627,2012-12-15,4292.01
"Printer Mart",3632,2012-12-16,1036.74
"Printer Mart",3637,2012-12-24,1450.39
"Printer Mart",3644,2012-12-24,3514.41
"Printer Mart",3647,2012-12-30,1372.59
"Printer Mart",3652,2013-01-05,5138.81
"Printer Mart",3660,2013-01-11,1053.42
"Printer Mart",3663,2013-01-12,7479.19
"Printer Mart",3668,2013-01-18,1426.61
"Printer Mart",3673,2013-01-28,1529.67
"Printer Mart",3678,2013-02-03,1698.1
"Printer Mart",3682,2013-02-03,1401.77
"Printer Mart",3686,2013-02-04,5928.25
"Printer Mart",3691,2013-02-04,9004.16
"Printer Mart",3696,2013-02-05,1885.82
"Printer Mart",3705,2013-02-05,8123.78
"Printer Mart",3708,2013-02-05,8692.98
"Printer Mart",3715,2013-02-06,2433.56
"Printer Mart",3718,2013-02-07,9501.53
"Printer Mart",3723,2013-02-07,1818.69
"Printer Mart",3726,2013-02-08,6086.58
"Printer Mart",3734,2013-02-18,4295.79
"Printer Mart",3738,2013-02-18,3772.01
"Printer Mart",3744,2013-02-18,3126.83
"Printer Mart",3749,2013-02-28,5912.92
"Printer Mart",3751,2013-02-28,6099.36
"Printer Mart",3757,2013-02-28,2001.74
"Printer Mart",3762,2013-03-01,1500.92
"Printer Mart",3769,2013-03-07,3862.91
"Printer Mart",3773,2013-03-13,2434.05
"Printer Mart",3776,2013-03-13,1020.29
"Printer Mart",3783,2013-03-13,5437.9
"Printer Mart",3790,2013-03-14,1805.5
"Printer Mart",3794,2013-03-15,3978.57
"Printer Mart",3796,2013-03-21,1924.58
"Printer Mart",3805,2013-03-27,2988.53
"Printer Mart",3806,2013-04-04,5496.54
"Printer Mart",3815,2013-04-05,8027.1
"Printer Mart",3818,2013-04-11,1398.83
"Printer Mart",3823,2013-04-17,6800.17
"Printer Mart",3827,2013-04-23,5113.23
"Printer Mart",3832,2013-04-24,6045.17
"Printer Mart",3837,2013-04-25,7571.59
"Printer Mart",3845,2013-05-01,6955.13
"Printer Mart",3850,2013-05-07,7166.94
"Printer Mart",3852,2013-05-07,6005.21
"Printer Mart",3856,2013-05-07,3765.47
"Printer Mart",3865,2013-05-08,2432.23
"Printer Mart",3869,2013-05-09,2928.99
"Printer Mart",3873,2013-05-10,5047.56
"Printer Mart",3879,2013-05-10,1228.49
"Printer Mart",3881,2013-05-16,8211.45
"Printer Mart",3888,2013-05-22,1441.53
"Printer Mart",3891,2013-05-28,2088.47
"Printer Mart",3897,2013-06-05,8484.12
"Printer Mart",3903,2013-06-13,7564.0
"Printer Mart",3907,2013-06-23,4828.47
"Printer Mart",3911,2013-06-23,3916.94
"Printer Mart",3918,2013-06-23,4259.14
"Printer Mart",3922,2013-07-01,1373.93
"Printer Mart",3927,2013-07-01,8098.87
"Printer Mart",3934,2013-07-07,1067.12
"Printer Mart",3939,2013-07-17,1108.37
"Printer Mart",3941,2013-07-27,5712.59
"Printer Mart",3947,2013-07-28,1586.37
"Printer Mart",3952,2013-07-28,1332.92
"Printer Mart",3956,2013-07-29,1210.33
"Printer Mart",3962,2013-07-29,2201.49
"Printer Mart",3968,2013-07-29,1291.74
"Printer Mart",3975,2013-08-04,2891.57
"Printer Mart",3977,2013-08-05,2958.31
"Printer Mart",3982,2013-08-05,2178.1
"Printer Mart",3986,2013-08-11,2347.51
"Printer Mart",3994,2013-08-12,1227.97
"Printer Mart",3996,2013-08-13,1882.13
"Printer Mart",4001,2013-08-19,1204.29
"Printer Mart",4006,2013-08-19,1370.65
"Printer Mart",4014,2013-08-19,2563.27
"Printer Mart",4017,2013-08-19,1695.53
"Printer Mart",4022,2013-08-29,9377.2
"Printer Mart",4030,2013-09-04,4863.78
"Printer Mart",4031,2013-09-10,5149.47
"Printer Mart",4039,2013-09-16,5534.66
"Printer Mart",4043,2013-09-16,1435.59
"Printer Mart",4048,2013-09-24,8178.13
"Printer Mart",4052,2013-10-02,7934.24
"Printer Mart",4056,2013-10-10,4228.29
"Printer Mart",4062,2013-10-18,2190.63
"Printer Mart",4067,2013-10-19,1770.96
"Printer Mart",4075,2013-10-19,6528.59
"Printer Mart",4078,2013-10-20,7287.15
"Printer Mart",4084,2013-10-21,1285.24
"Printer Mart",4089,2013-10-31,8049.16
"Printer Mart",4091,2013-11-08,2289.78
"Printer Mart",4098,2013-11-16,5551.18
"Printer Mart",4104,2013-11-26,5028.34
"Printer Mart",4109,2013-11-27,8256.17
"Printer Mart",4115,2013-12-07,8176.91
"Printer Mart",4119,2013-12-07,1443.66
"Printer Mart",4121,2013-12-07,3669.07
"Printer Mart",4130,2013-12-07,1070.98
"Printer Mart",4135,2013-12-07,8811.5
"Printer Mart",4140,2013-12-13,1851.69
"Printer Mart",4145,2013-12-13,5021.35
"Printer Mart",4149,2013-12-14,2234.76
"Printer Mart",4155,2013-12-14,6382.37
"Printer Mart",4157,2013-12-20,2669.95
"Printer Mart",4162,2013-12-28,3884.4
"Printer Mart",4169,2014-01-03,3105.11
"Printer Mart",4172,2014-01-09,1252.07
"Printer Mart",4179,2014-01-10,3573.39
"Printer Mart",4184,2014-01-11,2644.71
"Printer Mart",4188,2014-01-19,1345.64
"Printer Mart",4193,2014-01-19,5609.51
"Printer Mart",4197,2014-01-19,3366.07
"Printer Mart",4202,2014-01-20,4083.88
"Printer Mart",4208,2014-01-26,9960.85
"Printer Mart",4213,2014-01-26,1030.19
"Printer Mart",4217,2014-01-26,1368.02
"Printer Mart",4225,2014-02-01,1995.74
"Printer Mart",4228,2014-02-02,2744.92
"Printer Mart",4233,2014-02-08,1176.97
"Printer Mart",4237,2014-02-14,2676.88
"Printer Mart",4243,2014-02-14,3368.89
"Printer Mart",4249,2014-02-14,4990.17
"Printer Mart",4253,2014-02-15,8536.74
"Printer Mart",4260,2014-02-15,7309.06
"Printer Mart",4265,2014-02-15,9893.62
"Printer Mart",4269,2014-02-16,3078.7
"Printer Mart",4273,2014-02-22,1663.74
"Printer Mart",4278,2014-02-23,1985.43
"Printer Mart",4284,2014-03-01,8272.82
"Printer Mart",4290,2014-03-07,4779.55
"Printer Mart",4294,2014-03-13,1200.01
"Printer Mart",4300,2014-03-14,4965.92
"Printer Mart",4301,2014-03-20,2609.0
"Printer Mart",4306,2014-03-20,1971.27
"Printer Mart",4311,2014-03-21,9156.95
"Printer Mart",4317,2014-03-21,5008.35
"Printer Mart",4324,2014-03-27,4308.02
"Printer Mart",4330,2014-03-27,2736.64
"Printer Mart",4333,2014-03-27,7143.41
"Printer Mart",4336,2014-04-06,2709.38
"Printer Mart",4344,2014-04-07,4754.52
"Printer Mart",4346,2014-04-08,3052.51
"Printer Mart",4354,2014-04-14,4182.98
"Printer Mart",4358,2014-04-15,1672.4
"Printer Mart",4363,2014-04-15,1903.05
"Printer Mart",4367,2014-04-15,2094.26
"Printer Mart",4371,2014-04-21,6255.92
"Printer Mart",4377,2014-04-29,1396.53
"Printer Mart",4381,2014-05-05,1004.03
"Printer Mart",4386,2014-05-05,3960.87
"Videoeletrix",9,2006-01-02,8763.91
"Videoeletrix",13,2006-01-02,1292.94
"Videoeletrix",17,2006-01-12,8194.42
"Videoeletrix",22,2006-01-13,7025.67
"Videoeletrix",30,2006-01-14,1200.95
"Videoeletrix",33,2006-01-20,1514.54
"Videoeletrix",39,2006-01-30,8909.76
"Videoeletrix",43,2006-01-30,1853.27
"Videoeletrix",46,2006-02-05,6682.52
"Videoeletrix",55,2006-02-11,2130.02
"Videoeletrix",57,2006-02-17,4623.93
"Videoeletrix",65,2006-02-25,2780.18
"Videoeletrix",68,2006-03-07,9131.65
"Videoeletrix",74,2006-03-07,2080.44
"Videoeletrix",78,2006-03-07,1404.37
"Videoeletrix",85,2006-03-07,1159.31
"Videoeletrix",87,2006-03-07,8593.58
"Videoeletrix",92,2006-03-08,5863.19
"Videoeletrix",97,2006-03-08,3260.08
"Videoeletrix",105,2006-03-18,4374.97
"Videoeletrix",107,2006-03-19,1897.08
"Videoeletrix",112,2006-03-27,1942.15
"Videoeletrix",116,2006-03-27,1156.0
"Videoeletrix",122,2006-03-28,5964.45
"Videoeletrix",127,2006-03-28,4489.19
"Videoeletrix",134,2006-04-03,6817.48
"Videoeletrix",136,2006-04-04,2537.05
"Videoeletrix",145,2006-04-04,5642.34
"Videoeletrix",148,2006-04-10,1374.95
"Videoeletrix",153,2006-04-20,2244.71
"Videoeletrix",158,2006-04-26,1331.11
"Videoeletrix",164,2006-04-27,9228.17
"Videoeletrix",166,2006-04-28,3603.03
"Videoeletrix",175,2006-05-04,1110.06
"Videoeletrix",176,2006-05-04,4131.77
"Videoeletrix",182,2006-05-14,6964.39
"Videoeletrix",188,2006-05-14,8251.2
"Videoeletrix",195,2006-05-15,1327.47
"Videoeletrix",200,2006-05-21,9257.09
"Videoeletrix",205,2006-05-22,2947.04
"Videoeletrix",208,2006-05-23,5950.91
"Videoeletrix",215,2006-05-23,4186.44
"Videoeletrix",217,2006-05-29,2009.22
"Videoeletrix",224,2006-05-30,7717.8
"Videoeletrix",226,2006-05-30,6930.34
"Videoeletrix",234,2006-06-09,4056.07
"Videoeletrix",236,2006-06-10,8843.59
"Videoeletrix",245,2006-06-11,3883.45
"Videoeletrix",247,2006-06-12,4543.83
"Videoeletrix",252,2006-06-13,1241.68
"Videoeletrix",256,2006-06-14,2457.06
"Videoeletrix",261,2006-06-20,2895.02
"Videoeletrix",268,2006-06-30,5317.13
"Videoeletrix",273,2006-07-01,6859.35
"Videoeletrix",276,2006-07-01,1767.58
"Videoeletrix",282,2006-07-01,1560.47
"Videoeletrix",286,2006-07-07,3462.79
"Videoeletrix",293,2006-07-08,1510.43
"Videoeletrix",300,2006-07-08,4968.72
"Videoeletrix",305,2006-07-14,2629.3
"Videoeletrix",307,2006-07-14,3171.46
"Videoeletrix",315,2006-07-14,4428.51
"Videoeletrix",320,2006-07-20,3492.43
"Videoeletrix",325,2006-07-26,3541.7
"Videoeletrix",329,2006-07-27,2336.25
"Videoeletrix",332,2006-08-06,1595.23
"Videoeletrix",340,2006-08-12,2228.2
"Videoeletrix",343,2006-08-12,1854.91
"Videoeletrix",347,2006-08-18,3945.21
"Videoeletrix",355,2006-08-19,1400.06
"Videoeletrix",359,2006-08-20,4827.51
"Videoeletrix",361,2006-08-30,4048.21
"Videoeletrix",366,2006-08-31,6975.89
"Videoeletrix",372,2006-09-10,1041.03
"Videoeletrix",380,2006-09-16,2587.06
"Videoeletrix",385,2006-09-17,6096.26
"Videoeletrix",387,2006-09-17,2177.01
"Videoeletrix",395,2006-09-17,7291.52
"Videoeletrix",399,2006-09-17,2725.07
"Videoeletrix",404,2006-09-17,7813.31
"Videoeletrix",406,2006-09-23,1615.94
"Videoeletrix",412,2006-09-23,5224.76
"Videoeletrix",420,2006-09-29,4812.18
"Videoeletrix",423,2006-10-05,2913.13
"Videoeletrix",428,2006-10-05,4353.77
"Videoeletrix",431,2006-10-06,3892.18
"Videoeletrix",438,2006-10-12,1363.12
"Videoeletrix",441,2006-10-13,1242.51
"Videoeletrix",450,2006-10-23,3760.97
"Videoeletrix",455,2006-10-31,9996.87
"Videoeletrix",458,2006-11-06,2482.62
"Videoeletrix",461,2006-11-06,5613.64
"Videoeletrix",467,2006-11-07,1096.05
"Videoeletrix",474,2006-11-08,3348.95
"Videoeletrix",480,2006-11-14,7296.3
"Videoeletrix",485,2006-11-22,2183.04
"Videoeletrix",490,2006-11-23,2237.65
"Videoeletrix",491,2006-12-03,8860.86
"Videoeletrix",499,2006-12-03,1653.77
"Videoeletrix",504,2006-12-04,6665.95
"Videoeletrix",506,2006-12-12,6051.07
"Videoeletrix",513,2006-12-13,1532.83
"Videoeletrix",518,2006-12-21,8493.01
"Videoeletrix",524,2006-12-31,4151.62
"Videoeletrix",528,2006-12-31,9708.81
"Videoeletrix",535,2007-01-01,3743.99
"Videoeletrix",538,2007-01-02,3308.5
"Videoeletrix",542,2007-01-02,4072.87
"Videoeletrix",549,2007-01-08,1603.61
"Videoeletrix",551,2007-01-09,4700.08
"Videoeletrix",557,2007-01-09,3852.17
"Videoeletrix",565,2007-01-09,2045.88
"Videoeletrix",567,2007-01-19,5237.08
"Videoeletrix",574,2007-01-19,4355.33
"Videoeletrix",580,2007-01-20,3101.79
"Videoeletrix",585,2007-01-21,1685.37
"Videoeletrix",589,2007-01-31,5911.54
"Videoeletrix",593,2007-02-08,3465.96
"Videoeletrix",596,2007-02-08,1086.13
"Videoeletrix",602,2007-02-18,1943.44
"Videoeletrix",610,2007-02-18,3234.11
"Videoeletrix",612,2007-02-24,1691.12
"Videoeletrix",620,2007-02-25,6638.22
"Videoeletrix",625,2007-03-07,7138.08
"Videoeletrix",626,2007-03-07,7990.06
"Videoeletrix",631,2007-03-17,3467.91
"Videoeletrix",639,2007-03-17,1717.49
"Videoeletrix",644,2007-03-18,8417.31
"Videoeletrix",649,2007-03-18,3194.88
"Videoeletrix",655,2007-03-18,3942.86
"Videoeletrix",660,2007-03-18,8430.18
"Videoeletrix",661,2007-03-28,2715.22
"Videoeletrix",666,2007-03-29,4497.58
"Videoeletrix",675,2007-03-30,5314.93
"Videoeletrix",676,2007-03-31,3004.68
"Videoeletrix",682,2007-03-31,2883.31
"Videoeletrix",687,2007-04-01,7431.5
"Videoeletrix",694,2007-04-02,1320.55
"Videoeletrix",696,2007-04-02,1602.85
"Videoeletrix",701,2007-04-03,1142.81
"Videoeletrix",706,2007-04-03,1996.22
"Videoeletrix",711,2007-04-09,3534.68
"Videoeletrix",720,2007-04-09,2660.87
"Videoeletrix",721,2007-04-10,1686.16
"Videoeletrix",727,2007-04-11,1871.68
"Videoeletrix",734,2007-04-12,6934.82
"Videoeletrix",737,2007-04-22,3343.34
"Videoeletrix",742,2007-04-22,2611.96
"Videoeletrix",750,2007-04-23,1181.79
"Videoeletrix",751,2007-04-23,3704.99
"Videoeletrix",759,2007-04-24,3718.74
"Videoeletrix",765,2007-04-24,1099.97
"Videoeletrix",768,2007-04-25,3982.61
"Videoeletrix",774,2007-05-01,1054.53
"Videoeletrix",777,2007-05-02,1847.6
"Videoeletrix",784,2007-05-02,1670.08
"Videoeletrix",787,2007-05-03,7152.2
"Videoeletrix",791,2007-05-09,3342.77
"Videoeletrix",798,2007-05-09,1400.17
"Videoeletrix",804,2007-05-15,4595.4
"Videoeletrix",806,2007-05-25,1331.64
"Videoeletrix",812,2007-05-26,8088.04
"Videoeletrix",818,2007-05-26,8749.75
"Videoeletrix",821,2007-05-26,6794.29
"Videoeletrix",826,2007-05-26,2150.85
"Videoeletrix",833,2007-05-27,3266.98
"Videoeletrix",837,2007-06-06,9351.48
"Videoeletrix",842,2007-06-06,2348.44
"Videoeletrix",847,2007-06-16,3293.95
"Videoeletrix",854,2007-06-16,1436.87
"Videoeletrix",860,2007-06-17,8993.57
"Videoeletrix",863,2007-06-27,6831.34
"Videoeletrix",868,2007-06-28,7246.48
"Videoeletrix",875,2007-07-04,9351.39
"Videoeletrix",879,2007-07-12,9392.57
"Videoeletrix",883,2007-07-13,3803.18
"Videoeletrix",886,2007-07-21,1146.16
"Videoeletrix",894,2007-07-22,2127.23
"Videoeletrix",900,2007-07-28,1571.55
"Videoeletrix",905,2007-07-28,1829.75
"Videoeletrix",906,2007-08-03,6652.46
"Videoeletrix",911,2007-08-03,1217.8
"Videoeletrix",917,2007-08-11,5069.45
"Videoeletrix",924,2007-08-19,2396.88
"Videoeletrix",930,2007-08-25,4515.35
"Videoeletrix",933,2007-09-02,8948.7
"Videoeletrix",936,2007-09-02,6522.04
"Videoeletrix",943,2007-09-02,8145.01
"Videoeletrix",949,2007-09-02,3138.19
"Videoeletrix",952,2007-09-02,2735.85
"Videoeletrix",959,2007-09-03,1401.22
"Videoeletrix",962,2007-09-09,3245.65
"Videoeletrix",968,2007-09-17,1713.18
"Videoeletrix",974,2007-09-18,4897.82
"Videoeletrix",977,2007-09-24,1006.03
"Videoeletrix",983,2007-10-02,7630.71
"Videoeletrix",990,2007-10-08,8389.28
"Videoeletrix",995,2007-10-16,1713.93
"Videoeletrix",1000,2007-10-24,1222.39
"Videoeletrix",1004,2007-10-30,1211.22
"Videoeletrix",1006,2007-11-09,1684.06
"Videoeletrix",1014,2007-11-10,1781.95
"Videoeletrix",1019,2007-11-11,1235.61
"Videoeletrix",1023,2007-11-21,3302.06
"Videoeletrix",1026,2007-11-21,6919.95
"Videoeletrix",1035,2007-11-27,2112.47
"Videoeletrix",1040,2007-12-03,1654.54
"Videoeletrix",1041,2007-12-09,1558.84
"Videoeletrix",1050,2007-12-15,1960.85
"Videoeletrix",1052,2007-12-23,9259.68
"Videoeletrix",1058,2007-12-29,1111.7
"Videoeletrix",1065,2007-12-29,2841.16
"Videoeletrix",1070,2007-12-30,1026.28
"Videoeletrix",1074,2007-12-31,4284.6
"Videoeletrix",1079,2008-01-08,2060.3
"Videoeletrix",1083,2008-01-09,8729.12
"Videoeletrix",1086,2008-01-10,1077.3
"Videoeletrix",1093,2008-01-10,1239.72
"Videoeletrix",1096,2008-01-11,1948.76
"Videoeletrix",1104,2008-01-11,1963.57
"Videoeletrix",1109,2008-01-19,3364.68
"Videoeletrix",1113,2008-01-27,6025.43
"Videoeletrix",1120,2008-02-06,8771.55
"Videoeletrix",1121,2008-02-12,6217.46
"Videoeletrix",1128,2008-02-20,1375.43
"Videoeletrix",1134,2008-02-20,1063.05
"Videoeletrix",1139,2008-02-20,6169.63
"Videoeletrix",1142,2008-02-20,1250.55
"Videoeletrix",1150,2008-02-20,8635.19
"Videoeletrix",1152,2008-02-28,1503.27
"Videoeletrix",1156,2008-02-28,1631.82
"Videoeletrix",1165,2008-02-28,1034.9
"Videoeletrix",1170,2008-03-05,3136.54
"Videoeletrix",1173,2008-03-06,1118.76
"Videoeletrix",1180,2008-03-07,8987.3
"Videoeletrix",1181,2008-03-08,2000.12
"Videoeletrix",1187,2008-03-14,3190.68
"Videoeletrix",1192,2008-03-22,6958.42
"Videoeletrix",1200,2008-03-28,8713.55
"Videoeletrix",1202,2008-03-29,3748.41
"Videoeletrix",1207,2008-04-04,6166.46
"Videoeletrix",1213,2008-04-04,8096.58
"Videoeletrix",1220,2008-04-10,6474.25
"Videoeletrix",1223,2008-04-11,2883.88
"Videoeletrix",1229,2008-04-19,1694.85
"Videoeletrix",1233,2008-04-20,6725.6
"Videoeletrix",1240,2008-04-20,8761.2
"Videoeletrix",1242,2008-04-28,3345.56
"Videoeletrix",1246,2008-05-06,1932.82
"Videoeletrix",1252,2008-05-12,7013.03
"Videoeletrix",1260,2008-05-12,5527.06
"Videoeletrix",1263,2008-05-18,1624.08
"Videoeletrix",1266,2008-05-18,5933.55
"Videoeletrix",1275,2008-05-28,3707.76
"Videoeletrix",1279,2008-05-28,2711.22
"Videoeletrix",1283,2008-06-05,7919.39
"Videoeletrix",1287,2008-06-11,8199.84
"Videoeletrix",1291,2008-06-17,5935.64
"Videoeletrix",1300,2008-06-18,2428.61
"Videoeletrix",1303,2008-06-18,1944.13
"Videoeletrix",1306,2008-06-24,9968.25
"Videoeletrix",1315,2008-07-02,3097.67
"Videoeletrix",1318,2008-07-02,2310.74
"Videoeletrix",1325,2008-07-10,4486.19
"Videoeletrix",1330,2008-07-11,3870.7
"Videoeletrix",1333,2008-07-21,1343.99
"Videoeletrix",1338,2008-07-22,3386.0
"Videoeletrix",1345,2008-07-22,1642.48
"Videoeletrix",1350,2008-07-22,3978.08
"Videoeletrix",1355,2008-07-23,1563.3
"Videoeletrix",1358,2008-07-23,2883.95
"Videoeletrix",1363,2008-07-23,1937.35
"Videoeletrix",1368,2008-07-29,1375.57
"Videoeletrix",1375,2008-08-04,6486.35
"Videoeletrix",1376,2008-08-04,3647.41
"Videoeletrix",1384,2008-08-10,2063.77
"Videoeletrix",1386,2008-08-11,1047.97
"Videoeletrix",1394,2008-08-17,1592.54
"Videoeletrix",1399,2008-08-18,6785.83
"Videoeletrix",1404,2008-08-19,4826.04
"Videoeletrix",1408,2008-08-25,2870.23
"Videoeletrix",1413,2008-09-02,8435.69
"Videoeletrix",1418,2008-09-10,1184.63
"Videoeletrix",1423,2008-09-11,4862.89
"Videoeletrix",1427,2008-09-12,2264.35
"Videoeletrix",1433,2008-09-12,1726.41
"Videoeletrix",1436,2008-09-20,7859.13
"Videoeletrix",1445,2008-09-20,5534.45
"Videoeletrix",1450,2008-09-21,3446.37
"Videoeletrix",1455,2008-09-22,4542.01
"Videoeletrix",1458,2008-09-28,5856.72
"Videoeletrix",1464,2008-09-29,4024.53
"Videoeletrix",1466,2008-09-29,2773.96
"Videoeletrix",1474,2008-09-30,3696.74
"Videoeletrix",1478,2008-09-30,3942.14
"Videoeletrix",1485,2008-10-10,6324.18
"Videoeletrix",1488,2008-10-11,1929.1
"Videoeletrix",1493,2008-10-11,6284.63
"Videoeletrix",1499,2008-10-12,7766.57
"Videoeletrix",1504,2008-10-22,9410.98
"Videoeletrix",1506,2008-10-30,1249.98
"Videoeletrix",1512,2008-10-30,2636.32
"Videoeletrix",1517,2008-10-31,8417.34
"Videoeletrix",1525,2008-10-31,6427.24
"Videoeletrix",1528,2008-11-06,7893.49
"Videoeletrix",1535,2008-11-12,2912.74
"Videoeletrix",1539,2008-11-13,1347.26
"Videoeletrix",1542,2008-11-13,2709.4
"Videoeletrix",1549,2008-11-13,3274.55
"Videoeletrix",1554,2008-11-19,1330.47
"Videoeletrix",1559,2008-11-20,7141.88
"Videoeletrix",1562,2008-11-30,4703.5
"Videoeletrix",1570,2008-11-30,2930.2
"Videoeletrix",1572,2008-11-30,1405.75
"Videoeletrix",1576,2008-12-01,4599.46
"Videoeletrix",1585,2008-12-09,2303.65
"Videoeletrix",1587,2008-12-19,5493.65
"Videoeletrix",1592,2008-12-19,1133.85
"Videoeletrix",1599,2008-12-20,1462.13
"Videoeletrix",1601,2008-12-20,3508.49
"Videoeletrix",1608,2008-12-20,4805.27
"Videoeletrix",1611,2008-12-28,2684.58
"Videoeletrix",1620,2008-12-28,7853.96
"Videoeletrix",1623,2009-01-05,4420.07
"Videoeletrix",1628,2009-01-05,2885.75
"Videoeletrix",1635,2009-01-05,2624.45
"Videoeletrix",1639,2009-01-13,8828.57
"Videoeletrix",1645,2009-01-13,2062.74
"Videoeletrix",1646,2009-01-13,2481.49
"Videoeletrix",1655,2009-01-14,7262.0
"Videoeletrix",1660,2009-01-24,5476.65
"Videoeletrix",1664,2009-01-25,1530.83
"Videoeletrix",1670,2009-01-25,4509.09
"Videoeletrix",1673,2009-02-02,3876.98
"Videoeletrix",1680,2009-02-10,6554.38
"Videoeletrix",1685,2009-02-11,9125.15
"Videoeletrix",1689,2009-02-12,1977.88
"Videoeletrix",1695,2009-02-13,1222.73
"Videoeletrix",1699,2009-02-19,7953.88
"Videoeletrix",1703,2009-02-19,2647.42
"Videoeletrix",1706,2009-02-25,7588.13
"Videoeletrix",1711,2009-03-03,2444.51
"Videoeletrix",1718,2009-03-04,2415.28
"Videoeletrix",1721,2009-03-14,2236.02
"Videoeletrix",1729,2009-03-15,2010.39
"Videoeletrix",1733,2009-03-21,7835.71
"Videoeletrix",1739,2009-03-29,1785.84
"Videoeletrix",1741,2009-03-29,2080.18
"Videoeletrix",1747,2009-03-29,5252.41
"Videoeletrix",1751,2009-04-08,1120.69
"Videoeletrix",1757,2009-04-14,7451.07
"Videoeletrix",1761,2009-04-14,3893.47
"Videoeletrix",1767,2009-04-24,1844.92
"Videoeletrix",1774,2009-04-30,1722.46
"Videoeletrix",1776,2009-05-10,4210.49
"Videoeletrix",1783,2009-05-20,1732.37
"Videoeletrix",1786,2009-05-20,1981.86
"Videoeletrix",1794,2009-05-20,3055.0
"Videoeletrix",1796,2009-05-21,2264.0
"Videoeletrix",1805,2009-05-21,1800.9
"Videoeletrix",1808,2009-05-22,1269.95
"Videoeletrix",1812,2009-05-23,9805.38
"Videoeletrix",1819,2009-05-24,1664.34
"Videoeletrix",1821,2009-05-24,1945.09
"Videoeletrix",1829,2009-05-25,2214.55
"Videoeletrix",1835,2009-05-26,8098.51
"Videoeletrix",1837,2009-05-27,2007.54
"Videoeletrix",1842,2009-06-02,8719.46
"Videoeletrix",1846,2009-06-03,1534.2
"Videoeletrix",1854,2009-06-03,1755.24
"Videoeletrix",1857,2009-06-09,5641.94
"Videoeletrix",1861,2009-06-10,1391.39
"Videoeletrix",1866,2009-06-11,1496.04
"Videoeletrix",1872,2009-06-19,1254.91
"Videoeletrix",1880,2009-06-25,4379.82
"Videoeletrix",1884,2009-07-01,4616.61
"Videoeletrix",1889,2009-07-02,2303.77
"Videoeletrix",1892,2009-07-02,1029.7
"Videoeletrix",1900,2009-07-08,1919.09
"Videoeletrix",1904,2009-07-14,1611.0
"Videoeletrix",1906,2009-07-24,5583.26
"Videoeletrix",1912,2009-08-01,4120.95
"Videoeletrix",1920,2009-08-09,1080.2
"Videoeletrix",1925,2009-08-17,5110.12
"Videoeletrix",1927,2009-08-17,9259.12
"Videoeletrix",1932,2009-08-27,1010.28
"Videoeletrix",1937,2009-09-06,1014.74
"Videoeletrix",1941,2009-09-14,1547.76
"Videoeletrix",1946,2009-09-24,1582.33
"Videoeletrix",1953,2009-09-24,3038.0
"Videoeletrix",1957,2009-09-30,3705.23
"Videoeletrix",1961,2009-09-30,3563.1
"Videoeletrix",1969,2009-10-06,1476.89
"Videoeletrix",1971,2009-10-06,1404.82
"Videoeletrix",1978,2009-10-12,1065.64
"Videoeletrix",1984,2009-10-12,1330.57
"Videoeletrix",1988,2009-10-22,1310.47
"Videoeletrix",1995,2009-10-23,7312.52
"Videoeletrix",1999,2009-10-24,1479.59
"Videoeletrix",2003,2009-10-24,8040.15
"Videoeletrix",2007,2009-10-24,2959.2
"Videoeletrix",2015,2009-10-25,6939.57
"Videoeletrix",2019,2009-10-26,4294.06
"Videoeletrix",2025,2009-10-26,4624.48
"Videoeletrix",2030,2009-10-27,9487.91
"Videoeletrix",2033,2009-10-27,1745.08
"Videoeletrix",2040,2009-11-04,7392.03
"Videoeletrix",2045,2009-11-14,4105.12
"Videoeletrix",2049,2009-11-20,7110.27
"Videoeletrix",2054,2009-11-21,4387.66
"Videoeletrix",2060,2009-11-29,5349.64
"Videoeletrix",2063,2009-12-05,4135.87
"Videoeletrix",2067,2009-12-05,1348.91
"Videoeletrix",2075,2009-12-13,3245.15
"Videoeletrix",2079,2009-12-13,2305.58
"Videoeletrix",2085,2009-12-19,9191.8
"Videoeletrix",2089,2009-12-19,1065.81
"Videoeletrix",2091,2009-12-29,3473.46
"Videoeletrix",2099,2010-01-04,4082.84
"Videoeletrix",2103,2010-01-05,1783.76
"Videoeletrix",2110,2010-01-05,6701.47
"Videoeletrix",2113,2010-01-06,2996.81
"Videoeletrix",2118,2010-01-12,3235.17
"Videoeletrix",2123,2010-01-12,4577.32
"Videoeletrix",2126,2010-01-12,8722.05
"Videoeletrix",2133,2010-01-12,9338.67
"Videoeletrix",2138,2010-01-13,1075.03
"Videoeletrix",2145,2010-01-14,7608.24
"Videoeletrix",2149,2010-01-20,1109.19
"Videoeletrix",2152,2010-01-20,9227.26
"Videoeletrix",2157,2010-01-20,6085.6
"Videoeletrix",2161,2010-01-21,1594.83
"Videoeletrix",2170,2010-01-21,1022.99
"Videoeletrix",2175,2010-01-21,2599.01
"Videoeletrix",2176,2010-01-22,1572.16
"Videoeletrix",2181,2010-01-22,1059.45
"Videoeletrix",2188,2010-01-23,3880.82
"Videoeletrix",2195,2010-01-24,1651.52
"Videoeletrix",2196,2010-01-24,8960.73
"Videoeletrix",2201,2010-01-25,7613.87
"Videoeletrix",2209,2010-01-31,1536.51
"Videoeletrix",2212,2010-02-08,2706.03
"Videoeletrix",2218,2010-02-14,2371.21
"Videoeletrix",2225,2010-02-20,1341.26
"Videoeletrix",2228,2010-02-21,1790.32
"Videoeletrix",2234,2010-02-22,3985.5
"Videoeletrix",2238,2010-02-28,5652.4
"Videoeletrix",2244,2010-03-06,3154.57
"Videoeletrix",2249,2010-03-12,6265.79
"Videoeletrix",2251,2010-03-12,5604.73
"Videoeletrix",2260,2010-03-12,1230.95
"Videoeletrix",2261,2010-03-18,1524.97
"Videoeletrix",2270,2010-03-28,3378.37
"Videoeletrix",2274,2010-04-03,2392.7
"Videoeletrix",2279,2010-04-03,4012.08
"Videoeletrix",2284,2010-04-13,8344.57
"Videoeletrix",2289,2010-04-14,2201.55
"Videoeletrix",2294,2010-04-20,3760.13
"Videoeletrix",2296,2010-04-21,7719.71
"Videoeletrix",2305,2010-04-27,2911.96
"Videoeletrix",2306,2010-04-27,1002.3
"Videoeletrix",2313,2010-05-05,2135.97
"Videoeletrix",2318,2010-05-06,1432.64
"Videoeletrix",2322,2010-05-07,3118.38
"Videoeletrix",2326,2010-05-13,3656.02
"Videoeletrix",2335,2010-05-13,5317.36
"Videoeletrix",2337,2010-05-14,2064.64
"Videoeletrix",2345,2010-05-14,4520.89
"Videoeletrix",2348,2010-05-15,9413.18
"Videoeletrix",2352,2010-05-16,7608.16
"Videoeletrix",2359,2010-05-22,1817.05
"Videoeletrix",2361,2010-05-23,1475.15
"Videoeletrix",2366,2010-05-24,1996.82
"Videoeletrix",2371,2010-05-24,7462.18
"Videoeletrix",2380,2010-06-01,6388.33
"Videoeletrix",2381,2010-06-07,2329.41
"Videoeletrix",2387,2010-06-07,3016.08
"Videoeletrix",2393,2010-06-08,1400.76
"Videoeletrix",2399,2010-06-09,4337.25
"Videoeletrix",2401,2010-06-09,3934.98
"Videoeletrix",2407,2010-06-19,7409.66
"Videoeletrix",2414,2010-06-20,8005.79
"Videoeletrix",2416,2010-06-20,3931.36
"Videoeletrix",2425,2010-06-20,5326.64
"Videoeletrix",2428,2010-06-21,2110.74
"Videoeletrix",2433,2010-06-21,2253.28
"Videoeletrix",2437,2010-06-27,2677.25
"Videoeletrix",2441,2010-06-27,2630.89
"Videoeletrix",2449,2010-07-07,2005.16
"Videoeletrix",2451,2010-07-08,1984.35
"Videoeletrix",2460,2010-07-09,6609.56
"Videoeletrix",2461,2010-07-17,3106.92
"Videoeletrix",2466,2010-07-23,1446.21
"Videoeletrix",2474,2010-07-23,3568.42
"Videoeletrix",2479,2010-07-23,4554.19
"Videoeletrix",2482,2010-07-29,5364.74
"Videoeletrix",2490,2010-07-29,1173.66
"Videoeletrix",2494,2010-07-30,3659.34
"Videoeletrix",2499,2010-07-30,1330.26
"Videoeletrix",2503,2010-08-07,2223.24
"Videoeletrix",2506,2010-08-13,1693.47
"Videoeletrix",2511,2010-08-14,1013.55
"Videoeletrix",2516,2010-08-24,8398.57
"Videoeletrix",2522,2010-09-03,6658.49
"Videoeletrix",2529,2010-09-04,7623.14
"Videoeletrix",2531,2010-09-10,4704.62
"Videoeletrix",2536,2010-09-20,4347.46
"Videoeletrix",2542,2010-09-26,3060.87
"Videoeletrix",2548,2010-09-27,5541.51
"Videoeletrix",2552,2010-09-28,1116.54
"Videoeletrix",2556,2010-09-29,1459.78
"Videoeletrix",2565,2010-10-09,4995.85
"Videoeletrix",2568,2010-10-15,1485.51
"Videoeletrix",2575,2010-10-16,2657.18
"Videoeletrix",2576,2010-10-17,4879.93
"Videoeletrix",2585,2010-10-27,4104.99
"Videoeletrix",2588,2010-10-28,3099.36
"Videoeletrix",2593,2010-10-28,9781.48
"Videoeletrix",2600,2010-10-29,2470.57
"Videoeletrix",2602,2010-10-29,1774.27
"Videoeletrix",2609,2010-11-06,1397.76
"Videoeletrix",2612,2010-11-06,2580.87
"Videoeletrix",2616,2010-11-06,1796.53
"Videoeletrix",2621,2010-11-16,2490.86
"Videoeletrix",2628,2010-11-17,8284.16
"Videoeletrix",2631,2010-11-17,5562.44
"Videoeletrix",2636,2010-11-18,3278.01
"Videoeletrix",2641,2010-11-28,3713.39
"Videoeletrix",2648,2010-11-29,1233.78
"Videoeletrix",2653,2010-11-29,4459.7
"Videoeletrix",2659,2010-11-29,2248.9
"Videoeletrix",2663,2010-11-30,6624.19
"Videoeletrix",2670,2010-11-30,2287.72
"Videoeletrix",2675,2010-12-08,1737.39
"Videoeletrix",2676,2010-12-09,1082.06
"Videoeletrix",2681,2010-12-09,3243.52
"Videoeletrix",2688,2010-12-10,2138.78
"Videoeletrix",2693,2010-12-10,1903.5
"Videoeletrix",2698,2010-12-10,1111.95
"Videoeletrix",2703,2010-12-11,4867.76
"Videoeletrix",2707,2010-12-11,9437.21
"Videoeletrix",2713,2010-12-17,2150.05
"Videoeletrix",2719,2010-12-17,3202.35
"Videoeletrix",2722,2010-12-27,6220.9
"Videoeletrix",2727,2010-12-28,3782.53
"Videoeletrix",2735,2010-12-28,5887.88
"Videoeletrix",2739,2011-01-07,7220.02
"Videoeletrix",2743,2011-01-07,4883.3
"Videoeletrix",2748,2011-01-15,4285.68
"Videoeletrix",2755,2011-01-15,3912.41
"Videoeletrix",2758,2011-01-16,1884.58
"Videoeletrix",2761,2011-01-17,6331.87
"Videoeletrix",2769,2011-01-25,2365.43
"Videoeletrix",2773,2011-01-25,1513.07
"Videoeletrix",2776,2011-01-25,1225.49
"Videoeletrix",2782,2011-01-25,2159.0
"Videoeletrix",2788,2011-01-31,1056.52
"Videoeletrix",2794,2011-02-06,2282.91
"Videoeletrix",2799,2011-02-12,6349.42
"Videoeletrix",2803,2011-02-13,3682.44
"Videoeletrix",2809,2011-02-14,2416.72
"Videoeletrix",2814,2011-02-15,8381.28
"Videoeletrix",2819,2011-02-16,8126.67
"Videoeletrix",2821,2011-02-22,1600.36
"Videoeletrix",2828,2011-02-22,4325.19
"Videoeletrix",2832,2011-02-23,1035.03
"Videoeletrix",2839,2011-03-03,3987.06
"Videoeletrix",2842,2011-03-11,2312.63
"Videoeletrix",2847,2011-03-17,1317.05
"Videoeletrix",2852,2011-03-18,1664.14
"Videoeletrix",2859,2011-03-18,8972.78
"Videoeletrix",2863,2011-03-26,2724.18
"Videoeletrix",2868,2011-04-05,4470.83
"Videoeletrix",2872,2011-04-11,6287.83
"Videoeletrix",2878,2011-04-12,5142.57
"Videoeletrix",2884,2011-04-13,3741.87
"Videoeletrix",2889,2011-04-13,1125.64
"Videoeletrix",2892,2011-04-13,1115.56
"Videoeletrix",2899,2011-04-23,9653.42
"Videoeletrix",2903,2011-04-24,4801.5
"Videoeletrix",2906,2011-04-25,3689.77
"Videoeletrix",2912,2011-04-25,1788.93
"Videoeletrix",2919,2011-05-01,2479.75
"Videoeletrix",2922,2011-05-01,6095.08
"Videoeletrix",2927,2011-05-01,3559.33
"Videoeletrix",2931,2011-05-02,2001.54
"Videoeletrix",2936,2011-05-03,3094.86
"Videoeletrix",2941,2011-05-09,1133.77
"Videoeletrix",2950,2011-05-19,2031.02
"Videoeletrix",2953,2011-05-20,8428.03
"Videoeletrix",2958,2011-05-21,3843.84
"Videoeletrix",2964,2011-05-22,3330.9
"Videoeletrix",2970,2011-05-28,1449.66
"Videoeletrix",2973,2011-06-05,2812.3
"Videoeletrix",2977,2011-06-05,2865.34
"Videoeletrix",2983,2011-06-13,1846.03
"Videoeletrix",2987,2011-06-13,2415.43
"Videoeletrix",2993,2011-06-13,2934.78
"Videoeletrix",2996,2011-06-14,4536.9
"Videoeletrix",3004,2011-06-15,2784.57
"Videoeletrix",3009,2011-06-21,1140.55
"Videoeletrix",3011,2011-07-01,1488.9
"Videoeletrix",3019,2011-07-07,9164.76
"Videoeletrix",3024,2011-07-13,1086.61
"Videoeletrix",3027,2011-07-14,5957.02
"Videoeletrix",3033,2011-07-14,8102.63
"Videoeletrix",3036,2011-07-15,5470.56
"Videoeletrix",3045,2011-07-23,1785.86
"Videoeletrix",3048,2011-07-23,5611.56
"Videoeletrix",3053,2011-07-24,2095.25
"Videoeletrix",3056,2011-08-01,8057.45
"Videoeletrix",3063,2011-08-09,2283.37
"Videoeletrix",3068,2011-08-09,1408.35
"Videoeletrix",3073,2011-08-15,4325.61
"Videoeletrix",3079,2011-08-16,8243.56
"Videoeletrix",3085,2011-08-22,3183.43
"Videoeletrix",3088,2011-08-22,3192.05
"Videoeletrix",3095,2011-08-23,2850.02
"Videoeletrix",3096,2011-08-23,1739.22
"Videoeletrix",3101,2011-08-24,2914.85
"Videoeletrix",3108,2011-09-01,4116.48
"Videoeletrix",3111,2011-09-02,4584.29
"Videoeletrix",3117,2011-09-02,4638.77
"Videoeletrix",3123,2011-09-08,5456.15
"Videoeletrix",3129,2011-09-08,6090.41
"Videoeletrix",3135,2011-09-09,4521.46
"Videoeletrix",3138,2011-09-09,2433.11
"Videoeletrix",3144,2011-09-17,7867.61
"Videoeletrix",3149,2011-09-23,3008.26
"Videoeletrix",3154,2011-09-23,4621.65
"Videoeletrix",3158,2011-09-23,1804.8
"Videoeletrix",3163,2011-10-03,1565.2
"Videoeletrix",3170,2011-10-09,2142.04
"Videoeletrix",3171,2011-10-15,8006.27
"Videoeletrix",3180,2011-10-16,2941.98
"Videoeletrix",3181,2011-10-22,6157.81
"Videoeletrix",3187,2011-10-30,5055.95
"Videoeletrix",3191,2011-10-30,6516.88
"Videoeletrix",3196,2011-10-31,3128.27
"Videoeletrix",3201,2011-11-10,2755.39
"Videoeletrix",3207,2011-11-10,9450.92
"Videoeletrix",3212,2011-11-11,4293.5
"Videoeletrix",3218,2011-11-11,5786.2
"Videoeletrix",3224,2011-11-12,2905.15
"Videoeletrix",3230,2011-11-12,1829.49
"Videoeletrix",3232,2011-11-22,7539.4
"Videoeletrix",3237,2011-11-23,6450.05
"Videoeletrix",3245,2011-11-29,8234.58
"Videoeletrix",3250,2011-12-09,7159.6
"Videoeletrix",3253,2011-12-09,6226.55
"Videoeletrix",3257,2011-12-09,3590.8
"Videoeletrix",3261,2011-12-10,1203.36
"Videoeletrix",3268,2011-12-18,4795.26
"Videoeletrix",3272,2011-12-19,3944.04
"Videoeletrix",3280,2011-12-19,1454.69
"Videoeletrix",3283,2011-12-27,1878.61
"Videoeletrix",3289,2012-01-04,1530.69
"Videoeletrix",3291,2012-01-05,2431.19
"Videoeletrix",3300,2012-01-15,5206.04
"Videoeletrix",3304,2012-01-15,8848.47
"Videoeletrix",3309,2012-01-23,1477.08
"Videoeletrix",3312,2012-01-29,3658.18
"Videoeletrix",3316,2012-01-29,5757.47
"Videoeletrix",3323,2012-02-06,6874.07
"Videoeletrix",3329,2012-02-14,3422.82
"Videoeletrix",3331,2012-02-24,3420.38
"Videoeletrix",3337,2012-02-24,9617.93
"Videoeletrix",3345,2012-03-01,1417.24
"Videoeletrix",3346,2012-03-01,2780.51
"Videoeletrix",3352,2012-03-11,2416.18
"Videoeletrix",3357,2012-03-19,6268.41
"Videoeletrix",3363,2012-03-19,6350.6
"Videoeletrix",3367,2012-03-20,4118.17
"Videoeletrix",3372,2012-03-20,5205.28
"Videoeletrix",3379,2012-03-21,1358.1
"Videoeletrix",3385,2012-03-31,2604.41
"Videoeletrix",3388,2012-04-10,2864.15
"Videoeletrix",3393,2012-04-20,6501.84
"Videoeletrix",3398,2012-04-28,4879.67
"Videoeletrix",3401,2012-05-08,3091.34
"Videoeletrix",3408,2012-05-09,5062.16
"Videoeletrix",3412,2012-05-19,7210.46
"Videoeletrix",3419,2012-05-20,1021.28
"Videoeletrix",3424,2012-05-20,1036.24
"Videoeletrix",3430,2012-05-20,4784.86
"Videoeletrix",3431,2012-05-26,8224.64
"Videoeletrix",3438,2012-06-01,4888.94
"Videoeletrix",3443,2012-06-01,1789.22
"Videoeletrix",3448,2012-06-09,8287.03
"Videoeletrix",3452,2012-06-09,5499.07
"Videoeletrix",3457,2012-06-15,2232.26
"Videoeletrix",3465,2012-06-16,1059.4
"Videoeletrix",3467,2012-06-16,2489.29
"Black Knight Co.",10,2006-01-01,2502.07
"Black Knight Co.",12,2006-01-07,5605.47
"Black Knight Co.",17,2006-01-15,1635.74
"Black Knight Co.",21,2006-01-15,7248.57
"Black Knight Co.",29,2006-01-16,2518.46
"Black Knight Co.",33,2006-01-16,9257.03
"Black Knight Co.",39,2006-01-16,1706.22
"Black Knight Co.",41,2006-01-16,2801.2
"Black Knight Co.",46,2006-01-26,1512.82
"Black Knight Co.",52,2006-01-26,1585.34
"Black Knight Co.",57,2006-02-01,8519.91
"Black Knight Co.",63,2006-02-02,1892.6
"Black Knight Co.",66,2006-02-08,8035.34
"Black Knight Co.",72,2006-02-14,2766.01
"Black Knight Co.",80,2006-02-15,5023.91
"Black Knight Co.",82,2006-02-15,5314.63
"Black Knight Co.",88,2006-02-16,3271.81
"Black Knight Co.",94,2006-02-16,3130.54
"Black Knight Co.",97,2006-02-17,4400.64
"Black Knight Co.",103,2006-02-23,3553.86
"Black Knight Co.",108,2006-03-05,6596.27
"Black Knight Co.",113,2006-03-05,3125.35
"Black Knight Co.",118,2006-03-06,5129.68
"Black Knight Co.",121,2006-03-14,1123.94
"Black Knight Co.",127,2006-03-15,9133.9
"Black Knight Co.",133,2006-03-15,3030.17
"Black Knight Co.",139,2006-03-16,1183.57
"Black Knight Co.",143,2006-03-16,3310.27
"Black Knight Co.",147,2006-03-16,1081.58
"Black Knight Co.",153,2006-03-17,1868.49
"Black Knight Co.",159,2006-03-23,3246.12
"Black Knight Co.",164,2006-03-23,1754.85
"Black Knight Co.",167,2006-03-31,1213.48
"Black Knight Co.",173,2006-03-31,1244.67
"Black Knight Co.",177,2006-03-31,1386.29
"Black Knight Co.",181,2006-04-01,7594.55
"Black Knight Co.",187,2006-04-01,2640.98
"Black Knight Co.",194,2006-04-02,4300.06
"Black Knight Co.",198,2006-04-02,8568.12
"Black Knight Co.",204,2006-04-12,9353.82
"Black Knight Co.",208,2006-04-12,1040.66
"Black Knight Co.",212,2006-04-13,8745.15
"Black Knight Co.",220,2006-04-19,1286.87
"Black Knight Co.",221,2006-04-20,1441.18
"Black Knight Co.",228,2006-04-21,2037.77
"Black Knight Co.",234,2006-04-27,4805.08
"Black Knight Co.",237,2006-05-03,4072.42
"Black Knight Co.",243,2006-05-03,1097.53
"Black Knight Co.",246,2006-05-13,3836.9
"Black Knight Co.",253,2006-05-13,1394.61
"Black Knight Co.",258,2006-05-13,2875.81
"Black Knight Co.",264,2006-05-14,2211.04
"Black Knight Co.",267,2006-05-15,6224.45
"Black Knight Co.",273,2006-05-21,5697.63
"Black Knight Co.",280,2006-05-21,4494.25
"Black Knight Co.",285,2006-05-22,8434.03
"Black Knight Co.",287,2006-05-23,6938.29
"Black Knight Co.",295,2006-05-23,2694.16
"Black Knight Co.",296,2006-05-24,5536.16
"Black Knight Co.",303,2006-05-24,3732.27
"Black Knight Co.",307,2006-06-01,1362.87
"Black Knight Co.",315,2006-06-02,9950.21
"Black Knight Co.",320,2006-06-03,1116.58
"Black Knight Co.",325,2006-06-04,4821.84
"Black Knight Co.",328,2006-06-04,3301.83
"Black Knight Co.",332,2006-06-10,3186.48
"Black Knight Co.",336,2006-06-10,1495.79
"Black Knight Co.",342,2006-06-11,2825.09
"Black Knight Co.",349,2006-06-11,2068.68
"Black Knight Co.",352,2006-06-17,1819.36
"Black Knight Co.",359,2006-06-18,2444.64
"Black Knight Co.",361,2006-06-19,2020.15
"Black Knight Co.",369,2006-06-20,1198.15
"Black Knight Co.",371,2006-06-20,4659.46
"Black Knight Co.",380,2006-06-26,4405.28
"Black Knight Co.",381,2006-06-27,6072.59
"Black Knight Co.",387,2006-06-28,1204.32
"Black Knight Co.",392,2006-06-29,2133.98
"Black Knight Co.",397,2006-06-29,6129.31
"Black Knight Co.",403,2006-07-07,1011.89
"Black Knight Co.",406,2006-07-13,7818.36
"Black Knight Co.",415,2006-07-19,2942.91
"Black Knight Co.",417,2006-07-19,2175.14
"Black Knight Co.",424,2006-07-19,1968.73
"Black Knight Co.",428,2006-07-27,1476.35
"Black Knight Co.",433,2006-08-04,1850.35
"Black Knight Co.",436,2006-08-05,3690.48
"Black Knight Co.",444,2006-08-15,5404.58
"Black Knight Co.",450,2006-08-15,1209.09
"Black Knight Co.",453,2006-08-15,1978.03
"Black Knight Co.",459,2006-08-15,4103.98
"Black Knight Co.",461,2006-08-16,1318.08
"Black Knight Co.",466,2006-08-22,1771.72
"Black Knight Co.",471,2006-08-23,2649.72
"Black Knight Co.",478,2006-08-23,2244.1
"Black Knight Co.",485,2006-08-24,2440.87
"Black Knight Co.",490,2006-08-25,6391.77
"Black Knight Co.",491,2006-09-04,1201.32
"Black Knight Co.",497,2006-09-05,9722.66
"Black Knight Co.",504,2006-09-05,7625.2
"Black Knight Co.",509,2006-09-11,9942.67
"Black Knight Co.",513,2006-09-12,9899.5
"Black Knight Co.",517,2006-09-22,1227.81
"Black Knight Co.",522,2006-09-23,2993.43
"Black Knight Co.",529,2006-09-29,6308.68
"Black Knight Co.",534,2006-09-29,7402.03
"Black Knight Co.",539,2006-09-29,2688.44
"Black Knight Co.",544,2006-09-30,9475.6
"Black Knight Co.",548,2006-09-30,8792.72
"Black Knight Co.",552,2006-10-08,1600.27
"Black Knight Co.",559,2006-10-08,1535.73
"Black Knight Co.",563,2006-10-14,1865.11
"Black Knight Co.",567,2006-10-14,2187.79
"Black Knight Co.",573,2006-10-20,9238.66
"Black Knight Co.",577,2006-10-20,1653.49
"Black Knight Co.",581,2006-10-20,1838.18
"Black Knight Co.",587,2006-10-28,8211.63
"Black Knight Co.",593,2006-10-29,7219.61
"Black Knight Co.",599,2006-10-29,3918.16
"Black Knight Co.",601,2006-10-30,5028.07
"Black Knight Co.",609,2006-10-31,1796.86
"Black Knight Co.",611,2006-11-06,6520.51
"Black Knight Co.",616,2006-11-06,1616.18
"Black Knight Co.",622,2006-11-07,1994.5
"Black Knight Co.",627,2006-11-07,3758.03
"Black Knight Co.",631,2006-11-15,5044.61
"Black Knight Co.",637,2006-11-21,8838.9
"Black Knight Co.",641,2006-11-22,1277.12
"Black Knight Co.",650,2006-11-28,5095.47
"Black Knight Co.",651,2006-12-06,9832.5
"Black Knight Co.",660,2006-12-12,2103.3
"Black Knight Co.",665,2006-12-22,7091.24
"Black Knight Co.",668,2006-12-22,7979.16
"Black Knight Co.",671,2006-12-22,9123.64
"Black Knight Co.",677,2006-12-28,5080.67
"Black Knight Co.",683,2007-01-03,2656.99
"Black Knight Co.",687,2007-01-04,8608.2
"Black Knight Co.",695,2007-01-04,1356.88
"Black Knight Co.",700,2007-01-05,9234.94
"Black Knight Co.",701,2007-01-11,2404.5
"Black Knight Co.",706,2007-01-17,1111.32
"Black Knight Co.",711,2007-01-18,1510.55
"Black Knight Co.",718,2007-01-18,1629.16
"Black Knight Co.",723,2007-01-19,7548.01
"Black Knight Co.",730,2007-01-19,4054.97
"Black Knight Co.",733,2007-01-25,1151.89
"Black Knight Co.",736,2007-01-25,7406.08
"Black Knight Co.",741,2007-01-26,3741.22
"Black Knight Co.",748,2007-01-26,3712.16
"Black Knight Co.",753,2007-01-27,1143.12
"Black Knight Co.",757,2007-01-28,4315.66
"Black Knight Co.",763,2007-01-29,3504.88
"Black Knight Co.",769,2007-02-04,3211.55
"Black Knight Co.",773,2007-02-12,1994.79
"Black Knight Co.",776,2007-02-18,1869.77
"Black Knight Co.",785,2007-02-28,2213.4
"Black Knight Co.",789,2007-02-28,1610.4
"Black Knight Co.",793,2007-03-01,2485.05
"Black Knight Co.",799,2007-03-02,2096.44
"Black Knight Co.",803,2007-03-08,2628.01
"Black Knight Co.",806,2007-03-14,1096.81
"Black Knight Co.",813,2007-03-14,5104.46
"Black Knight Co.",818,2007-03-20,2423.09
"Black Knight Co.",823,2007-03-26,2233.82
"Black Knight Co.",829,2007-03-26,6758.25
"Black Knight Co.",834,2007-04-03,1964.17
"Black Knight Co.",840,2007-04-04,1158.6
"Black Knight Co.",845,2007-04-04,3405.41
"Black Knight Co.",847,2007-04-04,8821.73
"Black Knight Co.",854,2007-04-05,2543.41
"Black Knight Co.",856,2007-04-06,3509.84
"Black Knight Co.",861,2007-04-06,3595.52
"Black Knight Co.",870,2007-04-12,4079.42
"Black Knight Co.",874,2007-04-13,3935.7
"Black Knight Co.",880,2007-04-13,7870.85
"Black Knight Co.",883,2007-04-19,1221.55
"Black Knight Co.",890,2007-04-19,1295.97
"Black Knight Co.",894,2007-04-20,8554.17
"Black Knight Co.",896,2007-04-20,2838.74
"Black Knight Co.",902,2007-04-28,1992.3
"Black Knight Co.",907,2007-04-28,2693.2
"Black Knight Co.",914,2007-04-29,8365.78
"Black Knight Co.",917,2007-04-30,6930.89
"Black Knight Co.",923,2007-05-08,1889.79
"Black Knight Co.",929,2007-05-14,5846.6
"Black Knight Co.",931,2007-05-15,4265.64
"Black Knight Co.",937,2007-05-16,3724.52
"Black Knight Co.",944,2007-05-17,7183.92
"Black Knight Co.",947,2007-05-23,6895.36
"Black Knight Co.",954,2007-05-24,5006.16
"Black Knight Co.",960,2007-05-24,4848.22
"Black Knight Co.",962,2007-05-25,2469.45
"Black Knight Co.",967,2007-05-25,6703.15
"Black Knight Co.",975,2007-06-02,8560.51
"Black Knight Co.",977,2007-06-02,3791.49
"Black Knight Co.",981,2007-06-02,1161.32
"Black Knight Co.",989,2007-06-03,2055.87
"Black Knight Co.",994,2007-06-11,7352.37
"Black Knight Co.",997,2007-06-12,4637.06
"Black Knight Co.",1004,2007-06-20,6701.64
"Black Knight Co.",1009,2007-06-28,2473.17
"Black Knight Co.",1015,2007-07-08,6804.03
"Black Knight Co.",1020,2007-07-14,1438.74
"Black Knight Co.",1025,2007-07-14,5918.98
"Black Knight Co.",1030,2007-07-14,8808.35
"Black Knight Co.",1035,2007-07-20,5717.8
"Black Knight Co.",1036,2007-07-20,1580.03
"Black Knight Co.",1045,2007-07-21,2969.09
"Black Knight Co.",1050,2007-07-21,4756.65
"Black Knight Co.",1055,2007-07-22,7260.11
"Black Knight Co.",1059,2007-08-01,1858.45
"Black Knight Co.",1062,2007-08-01,5791.92
"Black Knight Co.",1066,2007-08-01,4659.33
"Black Knight Co.",1073,2007-08-02,1131.44
"Black Knight Co.",1079,2007-08-02,2436.32
"Black Knight Co.",1083,2007-08-12,2767.51
"Black Knight Co.",1090,2007-08-13,5587.59
"Black Knight Co.",1092,2007-08-14,6580.0
"Black Knight Co.",1099,2007-08-24,1924.26
"Black Knight Co.",1104,2007-08-25,2509.9
"Black Knight Co.",1109,2007-08-25,1064.01
"Black Knight Co.",1113,2007-08-26,1244.71
"Black Knight Co.",1117,2007-09-01,1680.0
"Black Knight Co.",1124,2007-09-07,1712.87
"Black Knight Co.",1126,2007-09-08,1872.22
"Black Knight Co.",1132,2007-09-18,3504.98
"Black Knight Co.",1137,2007-09-19,6731.66
"Black Knight Co.",1142,2007-09-25,1792.45
"Black Knight Co.",1149,2007-10-05,9885.37
"Black Knight Co.",1153,2007-10-06,1693.61
"Black Knight Co.",1156,2007-10-07,4151.33
"Black Knight Co.",1164,2007-10-13,7121.18
"Black Knight Co.",1169,2007-10-14,1073.31
"Black Knight Co.",1174,2007-10-20,9362.85
"Black Knight Co.",1179,2007-10-26,8730.61
"Black Knight Co.",1184,2007-11-05,4459.89
"Black Knight Co.",1187,2007-11-13,4585.21
"Black Knight Co.",1193,2007-11-21,1423.04
"Black Knight Co.",1198,2007-11-21,7841.01
"Black Knight Co.",1201,2007-11-22,1053.54
"Black Knight Co.",1209,2007-11-22,1708.54
"Black Knight Co.",1214,2007-11-28,3759.61
"Black Knight Co.",1220,2007-12-06,8626.01
"Black Knight Co.",1221,2007-12-12,2520.46
"Black Knight Co.",1230,2007-12-12,1005.39
"Black Knight Co.",1233,2007-12-20,2167.89
"Black Knight Co.",1239,2007-12-20,1673.61
"Black Knight Co.",1244,2007-12-26,2991.2
"Black Knight Co.",1248,2007-12-26,1787.05
"Black Knight Co.",1255,2008-01-05,2142.12
"Black Knight Co.",1256,2008-01-05,5200.46
"Black Knight Co.",1265,2008-01-13,4491.46
"Black Knight Co.",1267,2008-01-13,2604.99
"Black Knight Co.",1273,2008-01-13,1727.0
"Black Knight Co.",1276,2008-01-14,3718.63
"Black Knight Co.",1284,2008-01-24,5028.28
"Black Knight Co.",1286,2008-02-03,7578.23
"Black Knight Co.",1293,2008-02-04,6524.38
"Black Knight Co.",1297,2008-02-14,1950.96
"Black Knight Co.",1302,2008-02-15,4342.92
"Black Knight Co.",1307,2008-02-25,2650.62
"Black Knight Co.",1311,2008-02-26,1417.55
"Black Knight Co.",1320,2008-03-03,1405.08
"Black Knight Co.",1323,2008-03-04,1673.81
"Black Knight Co.",1330,2008-03-14,3341.48
"Black Knight Co.",1334,2008-03-20,5725.37
"Black Knight Co.",1336,2008-03-20,1089.67
"Black Knight Co.",1344,2008-03-28,5612.2
"Black Knight Co.",1350,2008-04-07,2691.31
"Black Knight Co.",1354,2008-04-13,3595.03
"Black Knight Co.",1359,2008-04-21,7414.54
"Black Knight Co.",1363,2008-04-29,1359.42
"Black Knight Co.",1370,2008-04-30,2260.82
"Black Knight Co.",1372,2008-04-30,3007.17
"Black Knight Co.",1380,2008-05-01,4288.0
"Black Knight Co.",1383,2008-05-07,1217.92
"Black Knight Co.",1389,2008-05-13,7947.07
"Black Knight Co.",1395,2008-05-13,5405.13
"Black Knight Co.",1398,2008-05-13,2006.84
"Black Knight Co.",1403,2008-05-23,1759.03
"Black Knight Co.",1410,2008-05-24,1547.66
"Black Knight Co.",1414,2008-05-30,9661.47
"Black Knight Co.",1418,2008-05-30,4440.17
"Black Knight Co.",1421,2008-06-07,5732.81
"Black Knight Co.",1427,2008-06-07,1612.2
"Black Knight Co.",1434,2008-06-08,9342.36
"Black Knight Co.",1440,2008-06-09,1842.39
"Black Knight Co.",1443,2008-06-15,5385.2
"Black Knight Co.",1447,2008-06-15,9003.81
"Black Knight Co.",1453,2008-06-16,1456.26
"Black Knight Co.",1459,2008-06-16,5089.69
"Black Knight Co.",1464,2008-06-24,5771.24
"Black Knight Co.",1467,2008-06-25,4824.78
"Black Knight Co.",1473,2008-06-25,2528.62
"Black Knight Co.",1479,2008-07-01,1414.91
"Black Knight Co.",1481,2008-07-07,4320.65
"Black Knight Co.",1488,2008-07-17,8452.21
"Black Knight Co.",1494,2008-07-17,8794.18
"Black Knight Co.",1498,2008-07-25,2423.43
"Black Knight Co.",1501,2008-07-31,1033.65
"Black Knight Co.",1507,2008-08-01,1739.35
"Black Knight Co.",1513,2008-08-02,1199.64
"Black Knight Co.",1520,2008-08-03,1554.66
"Black Knight Co.",1522,2008-08-04,1315.43
"Black Knight Co.",1529,2008-08-10,1830.81
"Black Knight Co.",1531,2008-08-10,4980.84
"Black Knight Co.",1540,2008-08-11,3507.4
"Black Knight Co.",1542,2008-08-17,7516.14
"Black Knight Co.",1549,2008-08-23,2260.08
"Black Knight Co.",1552,2008-08-24,8343.3
"Black Knight Co.",1558,2008-08-25,8218.9
"Black Knight Co.",1564,2008-08-31,4612.7
"Black Knight Co.",1568,2008-08-31,9185.27
"Black Knight Co.",1575,2008-09-06,1869.84
"Black Knight Co.",1580,2008-09-07,3076.76
"Black Knight Co.",1582,2008-09-17,2294.3
"Black Knight Co.",1589,2008-09-25,1778.62
"Black Knight Co.",1593,2008-09-25,3954.62
"Black Knight Co.",1600,2008-09-26,5431.35
"Black Knight Co.",1601,2008-10-06,7932.58
"Black Knight Co.",1607,2008-10-14,3021.03
"Black Knight Co.",1613,2008-10-15,6043.27
"Black Knight Co.",1616,2008-10-15,1087.1
"Black Knight Co.",1621,2008-10-15,1405.09
"Black Knight Co.",1628,2008-10-16,2224.93
"Black Knight Co.",1634,2008-10-26,6616.03
"Black Knight Co.",1637,2008-11-01,4413.12
"Black Knight Co.",1641,2008-11-01,1644.8
"Black Knight Co.",1647,2008-11-01,9792.91
"Black Knight Co.",1651,2008-11-01,5240.2
"Black Knight Co.",1659,2008-11-02,6703.29
"Black Knight Co.",1664,2008-11-10,7801.86
"Black Knight Co.",1669,2008-11-20,3220.65
"Black Knight Co.",1672,2008-11-20,2753.33
"Black Knight Co.",1679,2008-11-20,2475.01
"Black Knight Co.",1684,2008-11-21,2620.31
"Black Knight Co.",1689,2008-11-22,4720.97
"Black Knight Co.",1695,2008-11-22,6829.27
"Black Knight Co.",1700,2008-11-28,1262.43
"Black Knight Co.",1705,2008-11-28,5128.88
"Black Knight Co.",1709,2008-11-29,6474.28
"Black Knight Co.",1712,2008-12-07,7560.14
"Black Knight Co.",1720,2008-12-13,1411.39
"Black Knight Co.",1724,2008-12-14,3594.15
"Black Knight Co.",1729,2008-12-20,2607.91
"Black Knight Co.",1732,2008-12-26,2638.46
"Black Knight Co.",1736,2009-01-05,1789.7
"Black Knight Co.",1743,2009-01-06,2129.45
"Black Knight Co.",1747,2009-01-16,1265.98
"Black Knight Co.",1752,2009-01-24,1757.64
"Black Knight Co.",1759,2009-02-03,1013.02
"Black Knight Co.",1761,2009-02-04,1663.01
"Black Knight Co.",1769,2009-02-04,7403.51
"Black Knight Co.",1771,2009-02-14,1181.26
"Black Knight Co.",1780,2009-02-22,3309.51
"Black Knight Co.",1782,2009-02-28,1339.28
"Black Knight Co.",1787,2009-03-06,1081.52
"Black Knight Co.",1792,2009-03-07,2194.54
"Black Knight Co.",1799,2009-03-08,2157.99
"Black Knight Co.",1802,2009-03-09,3003.2
"Black Knight Co.",1806,2009-03-09,1213.95
"Black Knight Co.",1814,2009-03-15,6443.71
"Black Knight Co.",1816,2009-03-23,1895.26
"Black Knight Co.",1823,2009-03-24,2323.62
"Black Knight Co.",1830,2009-04-01,6275.93
"Black Knight Co.",1831,2009-04-11,5358.81
"Black Knight Co.",1839,2009-04-21,5399.2
"Black Knight Co.",1845,2009-05-01,6803.52
"Black Knight Co.",1846,2009-05-09,6604.06
"Black Knight Co.",1854,2009-05-09,2148.61
"Black Knight Co.",1859,2009-05-10,9417.42
"Black Knight Co.",1864,2009-05-20,7575.3
"Black Knight Co.",1870,2009-05-30,6811.18
"Black Knight Co.",1872,2009-05-30,1998.22
"Black Knight Co.",1878,2009-06-05,1103.35
"Black Knight Co.",1885,2009-06-06,9882.52
"Black Knight Co.",1888,2009-06-12,1376.53
"Black Knight Co.",1888,2009-06-13,3180.92
"Black Knight Co.",1896,2009-06-13,5381.56
"Black Knight Co.",1902,2009-06-19,3929.52
"Black Knight Co.",1902,2009-06-19,2522.25
"Black Knight Co.",1913,2009-06-19,1258.13
"Black Knight Co.",1913,2009-06-20,9853.49
"Black Knight Co.",1925,2009-06-30,1812.4
"Black Knight Co.",1928,2009-07-01,2153.57
"Black Knight Co.",1933,2009-07-07,1220.89
"Black Knight Co.",1937,2009-07-07,4245.26
"Black Knight Co.",1943,2009-07-07,5275.31
"Black Knight Co.",1950,2009-07-08,7961.72
"Black Knight Co.",1951,2009-07-09,1939.07
"Black Knight Co.",1956,2009-07-19,1399.82
"Black Knight Co.",1961,2009-07-20,5245.52
"Black Knight Co.",1968,2009-07-21,3854.79
"Black Knight Co.",1972,2009-07-31,1964.6
"Black Knight Co.",1976,2009-08-08,1193.83
"Black Knight Co.",1985,2009-08-09,1773.04
"Black Knight Co.",1989,2009-08-10,2581.74
"Black Knight Co.",1993,2009-08-16,2833.75
"Black Knight Co.",1999,2009-08-16,2374.88
"Black Knight Co.",2005,2009-08-16,4746.82
"Black Knight Co.",2006,2009-08-22,1013.96
"Black Knight Co.",2011,2009-08-23,3597.13
"Black Knight Co.",2019,2009-08-23,1119.53
"Black Knight Co.",2024,2009-08-23,1606.29
"Black Knight Co.",2029,2009-08-24,6760.69
"Black Knight Co.",2033,2009-08-25,3992.88
"Black Knight Co.",2033,2009-08-25,2807.96
"Black Knight Co.",2033,2009-08-25,1973.19
"Black Knight Co.",2049,2009-08-31,3740.99
"Black Knight Co.",2051,2009-08-31,3045.45
"Black Knight Co.",2060,2009-09-10,1134.95
"Black Knight Co.",2063,2009-09-11,3430.39
"Black Knight Co.",2070,2009-09-19,3121.13
"Black Knight Co.",2072,2009-09-20,6194.19
"Black Knight Co.",2078,2009-09-26,6453.87
"Black Knight Co.",2085,2009-09-26,4624.71
"Black Knight Co.",2090,2009-09-26,3710.93
"Black Knight Co.",2095,2009-09-27,1341.2
"Black Knight Co.",2097,2009-09-27,1721.65
"Black Knight Co.",2104,2009-09-27,2617.98
"Black Knight Co.",2110,2009-09-27,3804.92
"Black Knight Co.",2112,2009-09-28,1052.89
"Black Knight Co.",2117,2009-10-04,7181.6
"Black Knight Co.",2124,2009-10-04,7279.69
"Black Knight Co.",2127,2009-10-10,9654.83
"Black Knight Co.",2133,2009-10-10,8475.44
"Black Knight Co.",2136,2009-10-16,5682.29
"Black Knight Co.",2141,2009-10-22,4270.13
"Black Knight Co.",2150,2009-10-30,8085.49
"Black Knight Co.",2154,2009-10-31,4900.32
"Black Knight Co.",2159,2009-11-10,5868.82
"Black Knight Co.",2159,2009-11-10,6154.58
"Black Knight Co.",2159,2009-11-20,1608.68
"Black Knight Co.",2174,2009-11-20,2823.42
"Black Knight Co.",2179,2009-11-20,1029.46
"Black Knight Co.",2181,2009-11-26,8105.88
"Black Knight Co.",2187,2009-12-02,1661.0
"Black Knight Co.",2193,2009-12-10,3323.16
"Black Knight Co.",2198,2009-12-11,4113.93
"Black Knight Co.",2203,2009-12-11,1288.55
"Black Knight Co.",2207,2009-12-19,9313.82
"Black Knight Co.",2211,2009-12-19,2958.03
"Black Knight Co.",2216,2009-12-20,2033.82
"Black Knight Co.",2223,2009-12-20,1909.7
"Black Knight Co.",2230,2009-12-26,7519.44
"Black Knight Co.",2233,2010-01-01,2779.93
"Black Knight Co.",2238,2010-01-01,3043.53
"Black Knight Co.",2242,2010-01-07,1232.06
"Black Knight Co.",2250,2010-01-08,1409.83
"Black Knight Co.",2254,2010-01-09,2525.1
"Black Knight Co.",2258,2010-01-19,1696.28
"Black Knight Co.",2261,2010-01-19,4742.92
"Black Knight Co.",2268,2010-01-20,1520.07
"Black Knight Co.",2271,2010-01-26,3397.37
"Black Knight Co.",2276,2010-02-03,1068.38
"Black Knight Co.",2284,2010-02-09,3081.04
"Black Knight Co.",2290,2010-02-19,2754.93
"Black Knight Co.",2292,2010-02-20,2267.48
"Black Knight Co.",2296,2010-03-02,4141.62
"Black Knight Co.",2305,2010-03-10,8713.78
"Black Knight Co.",2307,2010-03-11,3725.17
"Black Knight Co.",2311,2010-03-12,2708.72
"Black Knight Co.",2318,2010-03-13,5967.89
"Black Knight Co.",2321,2010-03-19,8590.8
"Black Knight Co.",2328,2010-03-19,2445.16
"Black Knight Co.",2331,2010-03-29,8402.75
"Black Knight Co.",2337,2010-03-30,7961.0
"Black Knight Co.",2342,2010-03-30,2565.09
"Black Knight Co.",2350,2010-04-05,2188.29
"Black Knight Co.",2354,2010-04-05,1461.0
"Black Knight Co.",2358,2010-04-06,1608.22
"Black Knight Co.",2365,2010-04-07,1174.51
"Black Knight Co.",2370,2010-04-13,6199.07
"Black Knight Co.",2371,2010-04-14,6512.21
"Black Knight Co.",2377,2010-04-14,2430.12
"Black Knight Co.",2385,2010-04-14,4594.66
"Black Knight Co.",2390,2010-04-24,7271.43
"Black Knight Co.",2394,2010-04-30,4224.32
"Black Knight Co.",2399,2010-04-30,9687.45
"Black Knight Co.",2403,2010-05-06,1111.66
"Black Knight Co.",2410,2010-05-06,1404.9
"Black Knight Co.",2415,2010-05-12,8047.65
"Black Knight Co.",2417,2010-05-12,9171.12
"Black Knight Co.",2424,2010-05-13,1158.24
"Black Knight Co.",2429,2010-05-13,6606.76
"Black Knight Co.",2433,2010-05-14,5992.85
"Black Knight Co.",2438,2010-05-14,8242.72
"Black Knight Co.",2442,2010-05-14,5690.6
"Black Knight Co.",2450,2010-05-14,2320.45
"Black Knight Co.",2454,2010-05-14,3746.53
"Black Knight Co.",2460,2010-05-14,1428.44
"Black Knight Co.",2461,2010-05-20,1726.39
"Black Knight Co.",2468,2010-05-21,3572.52
"Black Knight Co.",2474,2010-05-21,1171.84
"Black Knight Co.",2479,2010-05-27,8780.57
"Black Knight Co.",2481,2010-06-06,1204.61
"Black Knight Co.",2490,2010-06-12,1563.17
"Black Knight Co.",2492,2010-06-18,5720.09
"Black Knight Co.",2497,2010-06-19,1960.48
"Black Knight Co.",2501,2010-06-20,2404.41
"Black Knight Co.",2509,2010-06-21,2847.47
"Black Knight Co.",2515,2010-07-01,3593.06
"Black Knight Co.",2516,2010-07-01,6236.77
"Black Knight Co.",2523,2010-07-02,2316.31
"Black Knight Co.",2529,2010-07-02,2700.84
"Black Knight Co.",2533,2010-07-03,1420.92
"Black Knight Co.",2536,2010-07-09,3267.19
"Black Knight Co.",2541,2010-07-19,1207.82
"Black Knight Co.",2550,2010-07-25,2262.73
"Black Knight Co.",2555,2010-07-26,2017.27
"Black Knight Co.",2556,2010-07-26,4678.02
"Black Knight Co.",2565,2010-07-26,2695.56
"Black Knight Co.",2567,2010-07-26,1943.49
"Black Knight Co.",2572,2010-07-27,1904.38
"Black Knight Co.",2578,2010-07-28,1343.88
"Black Knight Co.",2583,2010-08-07,1866.21
"Black Knight Co.",2587,2010-08-08,6953.66
"Black Knight Co.",2595,2010-08-09,4428.72
"Black Knight Co.",2598,2010-08-09,3645.72
"Black Knight Co.",2601,2010-08-10,1822.58
"Black Knight Co.",2608,2010-08-10,1097.99
"Black Knight Co.",2614,2010-08-10,1206.79
"Black Knight Co.",2617,2010-08-11,2035.04
"Black Knight Co.",2621,2010-08-11,2790.12
"Black Knight Co.",2629,2010-08-17,1714.6
"Black Knight Co.",2634,2010-08-23,3667.84
"Black Knight Co.",2636,2010-08-24,4328.44
"Black Knight Co.",2642,2010-08-30,4290.46
"Black Knight Co.",2649,2010-09-05,4148.26
"Black Knight Co.",2651,2010-09-06,2547.33
"Black Knight Co.",2659,2010-09-07,1240.14
"Black Knight Co.",2663,2010-09-08,9053.37
"Black Knight Co.",2669,2010-09-14,5332.99
"Black Knight Co.",2673,2010-09-20,3675.68
"Black Knight Co.",2680,2010-09-21,3874.59
"Black Knight Co.",2683,2010-09-29,5489.16
"Black Knight Co.",2690,2010-10-07,2188.99
"Black Knight Co.",2693,2010-10-07,2932.56
"Black Knight Co.",2697,2010-10-08,1125.5
"Black Knight Co.",2701,2010-10-16,3352.79
"Black Knight Co.",2709,2010-10-26,3034.39
"Black Knight Co.",2714,2010-11-01,9811.04
"Black Knight Co.",2718,2010-11-01,1563.47
"Black Knight Co.",2723,2010-11-07,9702.2
"Black Knight Co.",2723,2010-11-17,2074.96
"Black Knight Co.",2733,2010-11-18,2001.97
"Black Knight Co.",2738,2010-11-19,1768.71
"Black Knight Co.",2741,2010-11-25,1301.21
"Black Knight Co.",2746,2010-12-03,3591.0
"Black Knight Co.",2752,2010-12-03,1638.55
"Black Knight Co.",2756,2010-12-11,7483.56
"Black Knight Co.",2765,2010-12-12,4405.43
"Black Knight Co.",2768,2010-12-20,1148.97
"Black Knight Co.",2772,2010-12-30,4421.65
"Black Knight Co.",2777,2011-01-05,2108.42
"Black Knight Co.",2784,2011-01-05,1275.79
"Black Knight Co.",2789,2011-01-06,4310.68
"Black Knight Co.",2792,2011-01-07,7038.27
"Black Knight Co.",2798,2011-01-13,4207.26
"Black Knight Co.",2801,2011-01-19,5843.26
"Black Knight Co.",2806,2011-01-27,6301.45
"Black Knight Co.",2814,2011-02-04,1749.8
"Black Knight Co.",2816,2011-02-10,6645.13
"Black Knight Co.",2825,2011-02-10,1417.6
"Black Knight Co.",2827,2011-02-11,3549.47
"Black Knight Co.",2835,2011-02-12,4846.27
"Black Knight Co.",2838,2011-02-13,2648.07
"Black Knight Co.",2844,2011-02-14,9133.3
"Black Knight Co.",2848,2011-02-24,2053.47
"Black Knight Co.",2851,2011-02-25,8634.27
"Black Knight Co.",2857,2011-02-26,4553.26
"Black Knight Co.",2862,2011-03-04,1280.96
"Black Knight Co.",2870,2011-03-12,3177.32
"Black Knight Co.",2875,2011-03-12,1630.97
"Black Knight Co.",2880,2011-03-18,4677.35
"Black Knight Co.",2885,2011-03-28,6414.02
"Black Knight Co.",2888,2011-03-29,1864.46
"Black Knight Co.",2891,2011-04-08,3560.26
"Black Knight Co.",2899,2011-04-16,1504.06
"Black Knight Co.",2903,2011-04-16,3993.65
"Black Knight Co.",2906,2011-04-24,1231.45
"Black Knight Co.",2912,2011-04-30,2715.25
"Black Knight Co.",2916,2011-05-01,2528.65
"Black Knight Co.",2922,2011-05-07,1406.61
"Black Knight Co.",2930,2011-05-08,1931.17
"Black Knight Co.",2935,2011-05-08,1266.32
"Black Knight Co.",2938,2011-05-18,1448.49
"Black Knight Co.",2944,2011-05-19,3245.62
"Black Knight Co.",2947,2011-05-29,8538.97
"Black Knight Co.",2955,2011-05-29,1999.19
"Black Knight Co.",2958,2011-05-29,1658.04
"Black Knight Co.",2963,2011-05-29,5796.18
"Black Knight Co.",2970,2011-06-06,2604.1
"Black Knight Co.",2971,2011-06-12,1586.32
"Black Knight Co.",2978,2011-06-12,1013.22
"Black Knight Co.",2982,2011-06-18,2967.24
"Black Knight Co.",2990,2011-06-24,3320.12
"Black Knight Co.",2992,2011-07-02,6223.06
"Black Knight Co.",3000,2011-07-08,4170.62
"Black Knight Co.",3002,2011-07-08,4737.88
"Black Knight Co.",3008,2011-07-18,1856.47
"Black Knight Co.",3012,2011-07-18,6206.96
"Black Knight Co.",3016,2011-07-28,8928.08
"Black Knight Co.",3021,2011-08-03,2539.88
"Black Knight Co.",3030,2011-08-03,2307.66
"Black Knight Co.",3033,2011-08-11,9827.41
"Black Knight Co.",3038,2011-08-11,7405.03
"Black Knight Co.",3041,2011-08-11,5919.99
"Black Knight Co.",3046,2011-08-17,4196.64
"Black Knight Co.",3053,2011-08-23,1118.73
"Black Knight Co.",3057,2011-08-23,5103.86
"Black Knight Co.",3061,2011-08-23,2094.13
"Black Knight Co.",3070,2011-08-24,1489.64
"Black Knight Co.",3071,2011-08-24,2266.84
"Black Knight Co.",3076,2011-08-24,3701.59
"Black Knight Co.",3082,2011-08-30,1344.59
"Black Knight Co.",3087,2011-09-05,8514.04
"Black Knight Co.",3093,2011-09-13,8643.22
"Black Knight Co.",3100,2011-09-13,1376.98
"Black Knight Co.",3102,2011-09-19,3827.45
"Black Knight Co.",3106,2011-09-29,9862.28
"Black Knight Co.",3112,2011-09-30,1398.55
"Black Knight Co.",3116,2011-09-30,4730.92
"Black Knight Co.",3122,2011-10-01,2899.02
"Black Knight Co.",3130,2011-10-01,5661.32
"Black Knight Co.",3133,2011-10-01,1093.32
"Black Knight Co.",3136,2011-10-07,1784.89
"Black Knight Co.",3142,2011-10-07,2096.05
"Black Knight Co.",3147,2011-10-13,7614.31
"Black Knight Co.",3152,2011-10-19,2230.94
"Black Knight Co.",3160,2011-10-29,6194.5
"Black Knight Co.",3161,2011-10-29,1340.42
"Black Knight Co.",3169,2011-10-30,3768.84
"Black Knight Co.",3175,2011-10-31,1966.28
"Black Knight Co.",3178,2011-11-01,1955.35
"Black Knight Co.",3181,2011-11-01,2300.84
"Black Knight Co.",3188,2011-11-02,4080.02
"Black Knight Co.",3191,2011-11-08,2688.23
"Black Knight Co.",3197,2011-11-09,1114.1
"Black Knight Co.",3202,2011-11-10,6069.73
"Black Knight Co.",3209,2011-11-16,2589.71
"Black Knight Co.",3214,2011-11-16,1305.18
"Black Knight Co.",3217,2011-11-17,2667.39
"Black Knight Co.",3222,2011-11-17,4341.91
"Black Knight Co.",3227,2011-11-17,3709.6
"Black Knight Co.",3232,2011-11-25,1421.15
"Black Knight Co.",3240,2011-11-26,4685.59
"Black Knight Co.",3242,2011-12-06,1731.76
"Black Knight Co.",3247,2011-12-16,3315.89
"Black Knight Co.",3255,2011-12-26,6242.34
"Black Knight Co.",3257,2012-01-05,4124.3
"Black Knight Co.",3261,2012-01-11,4471.54
"Black Knight Co.",3270,2012-01-12,1086.13
"Black Knight Co.",3272,2012-01-18,5431.93
"Black Knight Co.",3276,2012-01-18,4795.74
"Black Knight Co.",3285,2012-01-18,8948.48
"Black Knight Co.",3287,2012-01-26,4974.77
"Black Knight Co.",3292,2012-01-26,1089.44
"Black Knight Co.",3296,2012-01-26,1873.59
"Black Knight Co.",3304,2012-01-26,1014.27
"Black Knight Co.",3310,2012-01-26,5111.72
"Black Knight Co.",3313,2012-01-26,1681.37
"Black Knight Co.",3319,2012-01-27,3000.24
"Black Knight Co.",3323,2012-01-28,4941.77
"Black Knight Co.",3326,2012-01-28,1111.21
"Black Knight Co.",3335,2012-02-03,5592.12
"Black Knight Co.",3337,2012-02-13,4735.55
"Black Knight Co.",3344,2012-02-19,6094.78
"Black Knight Co.",3350,2012-02-20,3888.99
"Black Knight Co.",3355,2012-02-21,4655.44
"Black Knight Co.",3357,2012-02-22,6333.88
"Black Knight Co.",3364,2012-03-03,4236.56
"Black Knight Co.",3366,2012-03-04,1316.21
"Black Knight Co.",3372,2012-03-04,2249.78
"Black Knight Co.",3378,2012-03-10,4674.43
"Black Knight Co.",3382,2012-03-11,4489.18
"Black Knight Co.",3390,2012-03-17,1461.1
"Black Knight Co.",3392,2012-03-18,2792.62
"Black Knight Co.",3399,2012-03-28,3879.2
"Black Knight Co.",3399,2012-03-28,5460.69
"Black Knight Co.",3410,2012-03-28,2514.29
"Black Knight Co.",3414,2012-03-29,4744.41
"Black Knight Co.",3416,2012-03-29,3283.3
"Black Knight Co.",3423,2012-03-29,4873.06
"Black Knight Co.",3426,2012-03-30,3508.81
"Black Knight Co.",3433,2012-03-31,4951.88
"Black Knight Co.",3440,2012-04-01,7250.22
"Black Knight Co.",3445,2012-04-02,6822.47
"Black Knight Co.",3448,2012-04-12,5077.25
"Black Knight Co.",3453,2012-04-18,7778.8
"Black Knight Co.",3458,2012-04-28,5976.1
"Black Knight Co.",3461,2012-04-28,1256.99
"Black Knight Co.",3468,2012-05-08,1314.63
"Black Knight Co.",3472,2012-05-08,2221.09
"Black Knight Co.",3478,2012-05-14,1701.31
"Black Knight Co.",3483,2012-05-14,1844.62
"Black Knight Co.",3486,2012-05-14,2579.94
"Black Knight Co.",3494,2012-05-20,1645.27
"Black Knight Co.",3497,2012-05-21,1295.56
"Black Knight Co.",3503,2012-05-22,3277.94
"Black Knight Co.",3506,2012-05-22,1123.33
"Black Knight Co.",3512,2012-05-22,2941.92
"Black Knight Co.",3518,2012-05-23,1221.44
"Black Knight Co.",3523,2012-05-24,2175.86
"Black Knight Co.",3527,2012-05-25,9147.65
"Black Knight Co.",3535,2012-06-02,7796.95
"Black Knight Co.",3540,2012-06-02,2321.55
"Black Knight Co.",3544,2012-06-12,1960.35
"Black Knight Co.",3549,2012-06-22,4890.03
"Black Knight Co.",3553,2012-07-02,2215.3
"Black Knight Co.",3557,2012-07-02,1191.03
"Black Knight Co.",3562,2012-07-02,3375.71
"Black Knight Co.",3567,2012-07-02,4818.97
"Black Knight Co.",3575,2012-07-12,5956.8
"Black Knight Co.",3580,2012-07-13,7327.11
"Black Knight Co.",3583,2012-07-14,3305.47
"Black Knight Co.",3589,2012-07-24,1795.7
"Black Knight Co.",3592,2012-08-03,9458.1
"Black Knight Co.",3597,2012-08-09,1472.26
"Black Knight Co.",3603,2012-08-09,1368.85
"Black Knight Co.",3607,2012-08-09,4547.12
"Black Knight Co.",3611,2012-08-19,6881.06
"Black Knight Co.",3617,2012-08-25,3504.03
"Black Knight Co.",3621,2012-09-02,3222.93
"Black Knight Co.",3626,2012-09-08,1161.87
"Black Knight Co.",3635,2012-09-08,3199.12
"Black Knight Co.",3639,2012-09-09,2262.85
"Black Knight Co.",3645,2012-09-19,6525.29
"Black Knight Co.",3646,2012-09-27,2937.67
"Black Knight Co.",3651,2012-09-27,8478.77
"Black Knight Co.",3656,2012-10-07,8982.37
"Black Knight Co.",3664,2012-10-13,3122.8
"Black Knight Co.",3669,2012-10-14,1592.34
"Black Knight Co.",3672,2012-10-20,7006.04
"Black Knight Co.",3677,2012-10-20,1137.45
"Black Knight Co.",3682,2012-10-26,1441.66
"Black Knight Co.",3687,2012-10-27,1379.32
"Black Knight Co.",3695,2012-11-02,1872.33
"Black Knight Co.",3696,2012-11-02,3581.35
"Black Knight Co.",3704,2012-11-10,2200.34
"Black Knight Co.",3709,2012-11-10,9328.06
"Black Knight Co.",3712,2012-11-20,3796.86
"Black Knight Co.",3717,2012-11-20,1106.92
"Black Knight Co.",3723,2012-11-20,1991.06
"Black Knight Co.",3726,2012-11-20,1962.12
"Black Knight Co.",3733,2012-11-30,6374.01
"Black Knight Co.",3740,2012-12-10,6987.5
"Black Knight Co.",3744,2012-12-10,3285.97
"Black Knight Co.",3748,2012-12-10,1001.07
"Black Knight Co.",3752,2012-12-11,5205.09
"Black Knight Co.",3758,2012-12-12,1534.71
"Black Knight Co.",3764,2012-12-12,5134.64
"Black Knight Co.",3767,2012-12-12,2814.5
"Black Knight Co.",3772,2012-12-13,6858.35
"Black Knight Co.",3778,2012-12-19,3624.51
"Black Knight Co.",3782,2012-12-19,2333.37
"Black Knight Co.",3786,2012-12-19,2979.66
"Black Knight Co.",3794,2012-12-19,7327.47
"Black Knight Co.",3796,2012-12-20,1121.23
"Black Knight Co.",3803,2012-12-26,1062.79
"Black Knight Co.",3809,2013-01-01,2969.75
"Black Knight Co.",3813,2013-01-07,5217.68
"Black Knight Co.",3816,2013-01-07,8623.23
"Black Knight Co.",3825,2013-01-13,3816.07
"Black Knight Co.",3827,2013-01-21,3186.12
"Black Knight Co.",3834,2013-01-21,1633.2
"Black Knight Co.",3840,2013-01-22,3242.83
"Black Knight Co.",3842,2013-01-22,6043.84
"Black Knight Co.",3850,2013-01-22,3433.99
"Black Knight Co.",3855,2013-02-01,2277.78
"Black Knight Co.",3860,2013-02-02,3974.25
"Black Knight Co.",3862,2013-02-12,1658.46
"Black Knight Co.",3869,2013-02-20,8380.52
"Black Knight Co.",3873,2013-02-21,7745.86
"Black Knight Co.",3879,2013-03-01,8703.39
"Black Knight Co.",3883,2013-03-01,8248.16
"Black Knight Co.",3887,2013-03-02,4067.42
"Black Knight Co.",3891,2013-03-12,5807.79
"Black Knight Co.",3900,2013-03-18,4557.4
"Black Knight Co.",3901,2013-03-19,5249.79
"Black Knight Co.",3907,2013-03-29,7429.4
"Black Knight Co.",3914,2013-04-08,1137.52
"Black Knight Co.",3918,2013-04-18,9442.89
"Black Knight Co.",3921,2013-04-19,6080.87
"Black Knight Co.",3921,2013-04-20,2636.83
"Black Knight Co.",3931,2013-04-20,3098.1
"Black Knight Co.",3931,2013-04-28,2356.46
"Black Knight Co.",3943,2013-04-28,2731.06
"Black Knight Co.",3943,2013-05-04,4484.42
"Black Knight Co.",3954,2013-05-10,7465.71
"Black Knight Co.",3957,2013-05-16,1885.25
"Black Knight Co.",3961,2013-05-17,1092.99
"Black Knight Co.",3970,2013-05-17,6094.83
"Black Knight Co.",3974,2013-05-25,1785.13
"Black Knight Co.",3979,2013-05-31,1302.33
"Black Knight Co.",3981,2013-06-10,5430.04
"Black Knight Co.",3987,2013-06-16,3710.82
"Black Knight Co.",3995,2013-06-16,4534.9
"Black Knight Co.",3999,2013-06-17,2380.22
"Black Knight Co.",4003,2013-06-23,3509.51
"Black Knight Co.",4010,2013-06-29,7316.32
"Black Knight Co.",4011,2013-06-29,3647.49
"Black Knight Co.",4020,2013-07-05,2729.47
"Black Knight Co.",4022,2013-07-05,6837.63
"Black Knight Co.",4028,2013-07-13,4381.66
"Black Knight Co.",4031,2013-07-13,2060.94
"Black Knight Co.",4036,2013-07-13,1194.82
"Black Knight Co.",4041,2013-07-19,8107.42
"Black Knight Co.",4050,2013-07-20,6852.74
"Black Knight Co.",4052,2013-07-21,2580.79
"Black Knight Co.",4058,2013-07-27,2387.69
"Black Knight Co.",4062,2013-08-06,1481.97
"Black Knight Co.",4069,2013-08-14,2019.9
"Black Knight Co.",4075,2013-08-20,4418.51
"Black Knight Co.",4078,2013-08-26,4195.1
"Black Knight Co.",4082,2013-09-01,9554.77
"Black Knight Co.",4088,2013-09-02,9791.31
"Black Knight Co.",4091,2013-09-02,1582.15
"Black Knight Co.",4096,2013-09-10,1469.22
"Black Knight Co.",4104,2013-09-10,9297.36
"Limozeen",9,2006-01-07,9077.87
"Limozeen",15,2006-01-15,1214.69
"Limozeen",20,2006-01-15,2632.45
"Limozeen",22,2006-01-15,5267.88
"Limozeen",30,2006-01-21,2642.71
"Limozeen",31,2006-01-27,3095.29
"Limozeen",36,2006-01-28,6167.58
"Limozeen",44,2006-01-29,1313.27
"Limozeen",47,2006-02-06,3022.74
"Limozeen",52,2006-02-07,2635.32
"Limozeen",60,2006-02-08,4934.72
"Limozeen",64,2006-02-08,1857.91
"Limozeen",66,2006-02-14,6310.71
"Limozeen",72,2006-02-24,1407.75
"Limozeen",76,2006-03-02,2063.11
"Limozeen",83,2006-03-08,6332.03
"Limozeen",89,2006-03-09,5678.84
"Limozeen",91,2006-03-17,7225.91
"Limozeen",98,2006-03-18,1610.7
"Limozeen",105,2006-03-18,4255.44
"Limozeen",107,2006-03-18,1427.91
"Limozeen",115,2006-03-28,3529.1
"Limozeen",120,2006-04-03,2315.33
"Limozeen",122,2006-04-04,9704.82
"Limozeen",126,2006-04-05,5033.84
"Limozeen",133,2006-04-05,7092.76
"Limozeen",136,2006-04-06,2698.55
"Limozeen",141,2006-04-06,8088.75
"Limozeen",147,2006-04-14,1754.36
"Limozeen",151,2006-04-15,6542.91
"Limozeen",157,2006-04-16,1859.37
"Limozeen",161,2006-04-22,2451.92
"Limozeen",166,2006-04-23,7113.95
"Limozeen",173,2006-04-23,1353.87
"Limozeen",179,2006-05-01,5847.69
"Limozeen",183,2006-05-09,1604.96
"Limozeen",188,2006-05-19,1025.92
"Limozeen",195,2006-05-19,1625.42
"Limozeen",199,2006-05-20,1635.27
"Limozeen",201,2006-05-21,2993.17
"Limozeen",210,2006-05-27,8839.97
"Limozeen",213,2006-05-27,7196.81
"Limozeen",217,2006-05-27,2803.39
"Limozeen",221,2006-06-02,6408.05
"Limozeen",228,2006-06-03,1105.07
"Limozeen",235,2006-06-04,1990.77
"Limozeen",237,2006-06-14,5841.41
"Limozeen",245,2006-06-22,5158.6
"Limozeen",246,2006-06-30,8237.25
"Limozeen",254,2006-06-30,2641.02
"Limozeen",258,2006-07-06,6201.49
"Limozeen",265,2006-07-12,5223.55
"Limozeen",267,2006-07-22,1040.23
"Limozeen",271,2006-07-22,1340.74
"Limozeen",279,2006-07-22,5140.06
"Limozeen",282,2006-07-22,9042.25
"Limozeen",289,2006-07-23,1553.09
"Limozeen",295,2006-08-02,1021.65
"Limozeen",300,2006-08-02,1004.94
"Limozeen",301,2006-08-12,7832.71
"Limozeen",308,2006-08-20,5143.18
"Limozeen",312,2006-08-20,9325.66
"Limozeen",317,2006-08-20,3685.2
"Limozeen",323,2006-08-26,2790.05
"Limozeen",330,2006-09-01,2676.34
"Limozeen",331,2006-09-01,3491.97
"Limozeen",337,2006-09-01,2573.92
"Limozeen",343,2006-09-02,7860.71
"Limozeen",347,2006-09-03,7431.39
"Limozeen",351,2006-09-04,2827.41
"Limozeen",357,2006-09-05,4152.66
"Limozeen",364,2006-09-13,4728.82
"Limozeen",367,2006-09-19,1883.26
"Limozeen",371,2006-09-25,4318.92
"Limozeen",379,2006-10-03,1020.7
"Limozeen",382,2006-10-04,1828.87
"Limozeen",388,2006-10-04,1363.22
"Limozeen",393,2006-10-10,6101.79
"Limozeen",398,2006-10-11,6333.38
"Limozeen",404,2006-10-17,2130.0
"Limozeen",408,2006-10-27,6612.28
"Limozeen",412,2006-10-28,1927.37
"Limozeen",419,2006-10-29,4974.4
"Limozeen",425,2006-11-04,2699.14
"Limozeen",429,2006-11-10,8699.84
"Limozeen",435,2006-11-16,6736.26
"Limozeen",436,2006-11-16,6299.63
"Limozeen",445,2006-11-17,8500.32
"Limozeen",449,2006-11-25,4502.83
"Limozeen",453,2006-11-25,5654.77
"Limozeen",458,2006-11-26,4058.72
"Limozeen",464,2006-11-26,2694.48
"Limozeen",467,2006-12-06,1075.36
"Limozeen",472,2006-12-12,7182.45
"Limozeen",476,2006-12-12,2693.22
"Limozeen",481,2006-12-12,8803.55
"Limozeen",486,2006-12-20,1934.97
"Limozeen",491,2006-12-26,2690.97
"Limozeen",496,2006-12-26,2761.59
"Limozeen",501,2006-12-26,4172.62
"Limozeen",507,2007-01-05,1373.05
"Limozeen",514,2007-01-06,4853.63
"Limozeen",518,2007-01-12,3779.37
"Limozeen",523,2007-01-18,1600.19
"Limozeen",530,2007-01-19,4646.82
"Limozeen",531,2007-01-29,5536.24
"Limozeen",536,2007-01-29,7543.44
"Limozeen",541,2007-02-04,1864.42
"Limozeen",549,2007-02-04,2738.24
"Limozeen",555,2007-02-12,3104.34
"Limozeen",557,2007-02-13,6787.27
"Limozeen",562,2007-02-13,6410.85
"Limozeen",570,2007-02-23,3628.9
"Limozeen",575,2007-02-24,7829.95
"Limozeen",580,2007-03-02,6162.73
"Limozeen",581,2007-03-08,2586.3
"Limozeen",586,2007-03-08,1075.37
"Limozeen",591,2007-03-09,1287.07
"Limozeen",596,2007-03-10,9802.1
"Limozeen",605,2007-03-11,1370.91
"Limozeen",606,2007-03-11,1147.34
"Limozeen",613,2007-03-21,1229.76
"Limozeen",617,2007-03-21,7823.06
"Limozeen",624,2007-03-27,1025.76
"Limozeen",630,2007-04-02,5512.2
"Limozeen",634,2007-04-02,3841.88
"Limozeen",636,2007-04-03,5365.45
"Limozeen",641,2007-04-04,3810.27
"Limozeen",649,2007-04-05,1034.46
"Limozeen",655,2007-04-05,2547.31
"Limozeen",658,2007-04-11,2252.22
"Limozeen",662,2007-04-12,2679.61
"Limozeen",666,2007-04-13,5151.55
"Limozeen",674,2007-04-14,2966.05
"Limozeen",680,2007-04-20,1963.48
"Limozeen",682,2007-04-30,1949.1
"Limozeen",689,2007-04-30,1736.91
"Limozeen",695,2007-04-30,2628.19
"Limozeen",697,2007-05-06,2305.97
"Limozeen",704,2007-05-06,1278.77
"Limozeen",710,2007-05-12,4148.3
"Limozeen",714,2007-05-13,9847.02
"Limozeen",718,2007-05-13,1034.34
"Limozeen",722,2007-05-14,2835.76
"Limozeen",730,2007-05-20,6358.07
"Limozeen",734,2007-05-21,2686.79
"Limozeen",740,2007-05-27,1415.82
"Limozeen",745,2007-06-04,2773.65
"Limozeen",750,2007-06-05,7071.7
"Limozeen",753,2007-06-06,1849.41
"Limozeen",759,2007-06-06,1972.92
"Limozeen",763,2007-06-07,1165.99
"Limozeen",767,2007-06-13,1276.06
"Limozeen",771,2007-06-14,2962.21
"Limozeen",777,2007-06-15,6234.67
"Limozeen",781,2007-06-16,7375.15
"Limozeen",787,2007-06-22,3448.13
"Limozeen",791,2007-06-22,1281.75
"Limozeen",799,2007-06-22,2475.27
"Limozeen",801,2007-06-23,7332.02
"Limozeen",808,2007-06-24,7099.74
"Limozeen",814,2007-06-25,1794.23
"Limozeen",818,2007-06-26,7229.57
"Limozeen",821,2007-06-27,9974.12
"Limozeen",827,2007-07-07,4198.28
"Limozeen",834,2007-07-08,2733.17
"Limozeen",838,2007-07-09,3635.35
"Limozeen",843,2007-07-10,4728.65
"Limozeen",847,2007-07-16,2675.04
"Limozeen",854,2007-07-17,3033.13
"Limozeen",857,2007-07-25,1731.16
"Limozeen",863,2007-07-25,6718.84
"Limozeen",867,2007-07-25,1946.19
"Limozeen",871,2007-07-26,1303.48
"Limozeen",879,2007-07-26,4176.63
"Limozeen",883,2007-08-05,2149.99
"Limozeen",888,2007-08-05,2374.93
"Limozeen",894,2007-08-06,4477.26
"Limozeen",897,2007-08-12,8701.81
"Limozeen",903,2007-08-13,9449.74
"Limozeen",906,2007-08-14,5624.99
"Limozeen",912,2007-08-24,8246.61
"Limozeen",918,2007-08-25,3347.06
"Limozeen",921,2007-08-25,1982.5
"Limozeen",926,2007-08-26,1283.79
"Limozeen",932,2007-08-26,6328.04
"Limozeen",937,2007-08-26,1321.18
"Limozeen",941,2007-09-05,5764.65
"Limozeen",946,2007-09-15,1234.0
"Limozeen",953,2007-09-25,1723.05
"Limozeen",958,2007-09-25,2916.14
"Limozeen",965,2007-10-01,2427.25
"Limozeen",968,2007-10-02,2077.56
"Limozeen",971,2007-10-03,2594.04
"Limozeen",978,2007-10-03,6581.15
"Limozeen",983,2007-10-13,2734.52
"Limozeen",986,2007-10-14,8078.4
"Limozeen",995,2007-10-14,9255.37
"Limozeen",1000,2007-10-15,4690.98
"Limozeen",1001,2007-10-16,1316.17
"Limozeen",1010,2007-10-17,3010.87
"Limozeen",1011,2007-10-18,1993.16
"Limozeen",1019,2007-10-18,9838.34
"Limozeen",1021,2007-10-19,4951.55
"Limozeen",1029,2007-10-25,1214.7
"Limozeen",1035,2007-10-25,1263.58
"Limozeen",1039,2007-10-31,1651.64
"Limozeen",1044,2007-10-31,7527.82
"Limozeen",1046,2007-11-06,8236.81
"Limozeen",1054,2007-11-07,1038.86
"Limozeen",1060,2007-11-07,2524.71
"Limozeen",1064,2007-11-13,2099.24
"Limozeen",1068,2007-11-13,2445.91
"Limozeen",1075,2007-11-13,2375.47
"Limozeen",1076,2007-11-19,8704.32
"Limozeen",1084,2007-11-27,1274.83
"Limozeen",1086,2007-12-03,5918.14
"Limozeen",1094,2007-12-03,1089.51
"Limozeen",1099,2007-12-04,1853.25
"Limozeen",1101,2007-12-04,2245.21
"Limozeen",1107,2007-12-05,1602.02
"Limozeen",1113,2007-12-05,4861.41
"Limozeen",1116,2007-12-05,2516.97
"Limozeen",1123,2007-12-05,2943.55
"Limozeen",1129,2007-12-11,6601.31
"Limozeen",1133,2007-12-11,1262.75
"Limozeen",1139,2007-12-11,7407.65
"Limozeen",1142,2007-12-17,1710.96
"Limozeen",1149,2007-12-17,7139.11
"Limozeen",1152,2007-12-18,3334.2
"Limozeen",1160,2007-12-19,1660.78
"Limozeen",1165,2007-12-20,3989.53
"Limozeen",1168,2007-12-21,1344.35
"Limozeen",1171,2007-12-31,1617.16
"Limozeen",1177,2008-01-01,1849.17
"Limozeen",1184,2008-01-01,4016.1
"Limozeen",1187,2008-01-07,5135.13
"Limozeen",1193,2008-01-08,6134.12
"Limozeen",1199,2008-01-08,4103.48
"Limozeen",1204,2008-01-16,5624.48
"Limozeen",1209,2008-01-24,2612.09
"Limozeen",1213,2008-01-30,2458.11
"Limozeen",1218,2008-02-05,1857.96
"Limozeen",1223,2008-02-06,1180.41
"Limozeen",1228,2008-02-06,6018.31
"Limozeen",1231,2008-02-07,1135.17
"Limozeen",1238,2008-02-07,1532.98
"Limozeen",1241,2008-02-08,2475.73
"Limozeen",1249,2008-02-18,3532.34
"Limozeen",1253,2008-02-26,9490.82
"Limozeen",1259,2008-03-05,2619.5
"Limozeen",1261,2008-03-05,7379.49
"Limozeen",1270,2008-03-13,7042.23
"Limozeen",1274,2008-03-13,6033.21
"Limozeen",1279,2008-03-19,4485.91
"Limozeen",1282,2008-03-25,4965.04
"Limozeen",1286,2008-03-25,1840.09
"Limozeen",1293,2008-03-31,1272.88
"Limozeen",1297,2008-04-06,1344.86
"Limozeen",1305,2008-04-16,3832.13
"Limozeen",1309,2008-04-16,3126.81
"Limozeen",1311,2008-04-16,1745.2
"Limozeen",1318,2008-04-24,1326.21
"Limozeen",1324,2008-04-30,1681.27
"Limozeen",1328,2008-05-10,5448.18
"Limozeen",1334,2008-05-11,1862.19
"Limozeen",1340,2008-05-11,3461.6
"Limozeen",1341,2008-05-11,2305.93
"Limozeen",1348,2008-05-19,3909.29
"Limozeen",1355,2008-05-20,4248.01
"Limozeen",1358,2008-05-20,3247.32
"Limozeen",1364,2008-05-26,2668.98
"Limozeen",1368,2008-06-01,2582.95
"Limozeen",1373,2008-06-11,6515.86
"Limozeen",1378,2008-06-11,4529.46
"Limozeen",1382,2008-06-12,2440.26
"Limozeen",1389,2008-06-18,1371.27
"Limozeen",1394,2008-06-18,4120.24
"Limozeen",1400,2008-06-18,5664.62
"Limozeen",1401,2008-06-24,3734.42
"Limozeen",1408,2008-07-04,2898.59
"Limozeen",1411,2008-07-04,2114.44
"Limozeen",1416,2008-07-12,9082.59
"Limozeen",1424,2008-07-12,2917.68
"Limozeen",1428,2008-07-13,2920.09
"Limozeen",1432,2008-07-13,9872.63
"Limozeen",1437,2008-07-13,3900.88
"Limozeen",1443,2008-07-21,1126.1
"Limozeen",1447,2008-07-22,1986.13
"Limozeen",1455,2008-08-01,1738.38
"Limozeen",1456,2008-08-01,1073.8
"Limozeen",1463,2008-08-01,8937.58
"Limozeen",1470,2008-08-09,3684.07
"Limozeen",1471,2008-08-15,1761.13
"Limozeen",1479,2008-08-16,1936.08
"Limozeen",1483,2008-08-16,7923.51
"Limozeen",1487,2008-08-16,1012.43
"Limozeen",1491,2008-08-16,1953.57
"Limozeen",1496,2008-08-22,5680.5
"Limozeen",1505,2008-08-28,8073.45
"Limozeen",1506,2008-08-28,6092.91
"Limozeen",1515,2008-08-28,1845.14
"Limozeen",1517,2008-09-05,5266.61
"Limozeen",1523,2008-09-06,7953.19
"Limozeen",1527,2008-09-06,8565.44
"Limozeen",1532,2008-09-12,4419.47
"Limozeen",1537,2008-09-13,3407.36
"Limozeen",1544,2008-09-13,1149.0
"Limozeen",1548,2008-09-23,1495.3
"Limozeen",1551,2008-09-24,4861.39
"Limozeen",1559,2008-09-24,2202.64
"Limozeen",1563,2008-09-25,5404.91
"Limozeen",1566,2008-09-25,1225.93
"Limozeen",1571,2008-09-26,2617.37
"Limozeen",1580,2008-09-26,5329.48
"Limozeen",1583,2008-09-26,3227.03
"Limozeen",1586,2008-09-26,1095.15
"Limozeen",1593,2008-09-27,1941.46
"Limozeen",1598,2008-10-05,2085.57
"Limozeen",1602,2008-10-06,6642.38
"Limozeen",1607,2008-10-16,2136.76
"Limozeen",1612,2008-10-26,1454.85
"Limozeen",1618,2008-10-26,5153.09
"Limozeen",1624,2008-10-26,8086.56
"Limozeen",1628,2008-11-01,2000.31
"Limozeen",1633,2008-11-01,2802.32
"Limozeen",1638,2008-11-07,2427.06
"Limozeen",1642,2008-11-13,5430.35
"Limozeen",1647,2008-11-19,9401.81
"Limozeen",1655,2008-11-19,7445.62
"Limozeen",1656,2008-11-19,4602.66
"Limozeen",1665,2008-11-19,1203.01
"Limozeen",1667,2008-11-20,4315.3
"Limozeen",1672,2008-11-21,2772.63
"Limozeen",1676,2008-11-21,5180.28
"Limozeen",1685,2008-11-22,7508.4
"Limozeen",1686,2008-11-23,1844.26
"Limozeen",1693,2008-12-01,1604.73
"Limozeen",1698,2008-12-09,6603.75
"Limozeen",1703,2008-12-15,1913.53
"Limozeen",1707,2008-12-16,1032.39
"Limozeen",1714,2008-12-22,3230.79
"Limozeen",1719,2008-12-30,8306.05
"Limozeen",1725,2008-12-30,4652.15
"Limozeen",1728,2008-12-31,1118.42
"Limozeen",1735,2008-12-31,1272.92
"Limozeen",1738,2009-01-01,1408.96
"Limozeen",1741,2009-01-07,5551.95
"Limozeen",1749,2009-01-07,5309.43
"Limozeen",1755,2009-01-08,4494.84
"Limozeen",1757,2009-01-09,5852.17
"Limozeen",1763,2009-01-19,1320.16
"Limozeen",1766,2009-01-20,3622.45
"Limozeen",1772,2009-01-20,1312.56
"Limozeen",1779,2009-01-21,2841.1
"Limozeen",1781,2009-01-29,6494.04
"Limozeen",1787,2009-01-30,2520.83
"Limozeen",1795,2009-02-05,1924.23
"Limozeen",1798,2009-02-11,1644.77
"Limozeen",1802,2009-02-19,1750.44
"Limozeen",1806,2009-02-20,5294.29
"Limozeen",1815,2009-02-26,2254.88
"Limozeen",1817,2009-02-27,5837.84
"Limozeen",1822,2009-03-07,1789.9
"Limozeen",1826,2009-03-07,1190.21
"Limozeen",1832,2009-03-07,3464.48
"Limozeen",1840,2009-03-13,2195.1
"Limozeen",1842,2009-03-19,8472.81
"Limozeen",1848,2009-03-19,1151.5
"Limozeen",1851,2009-03-19,1801.22
"Limozeen",1858,2009-03-25,1886.0
"Limozeen",1865,2009-03-25,9244.59
"Limozeen",1868,2009-03-31,3520.98
"Limozeen",1875,2009-04-01,8054.64
"Limozeen",1877,2009-04-07,1260.71
"Limozeen",1885,2009-04-15,3882.75
"Limozeen",1889,2009-04-16,7641.39
"Limozeen",1893,2009-04-17,8555.59
"Limozeen",1898,2009-04-27,5645.96
"Limozeen",1902,2009-04-28,3964.27
"Limozeen",1906,2009-04-28,2644.16
"Limozeen",1912,2009-04-29,1083.81
"Limozeen",1916,2009-04-29,3181.98
"Limozeen",1925,2009-04-29,2643.45
"Limozeen",1927,2009-04-29,1838.59
"Limozeen",1931,2009-04-29,6789.22
"Limozeen",1938,2009-04-29,1346.0
"Limozeen",1944,2009-04-29,2311.39
"Limozeen",1947,2009-04-30,6887.17
"Limozeen",1952,2009-04-30,6749.43
"Limozeen",1959,2009-05-06,1904.73
"Limozeen",1964,2009-05-07,1509.33
"Limozeen",1970,2009-05-15,8898.86
"Limozeen",1975,2009-05-15,5244.72
"Limozeen",1979,2009-05-15,2989.94
"Limozeen",1981,2009-05-15,7165.96
"Limozeen",1989,2009-05-25,9168.32
"Limozeen",1995,2009-05-25,5435.69
"Limozeen",1997,2009-05-31,1194.74
"Limozeen",2001,2009-06-01,9799.49
"Limozeen",2006,2009-06-09,2739.64
"Limozeen",2014,2009-06-17,5730.45
"Limozeen",2019,2009-06-18,1039.41
"Limozeen",2024,2009-06-19,1678.31
"Limozeen",2026,2009-06-25,2252.54
"Limozeen",2032,2009-06-25,4995.63
"Limozeen",2038,2009-06-25,1363.94
"Limozeen",2043,2009-06-25,8902.42
"Limozeen",2050,2009-07-01,2885.32
"Limozeen",2051,2009-07-01,8672.07
"Limozeen",2056,2009-07-02,1165.5
"Limozeen",2061,2009-07-08,2596.62
"Limozeen",2069,2009-07-14,1708.29
"Limozeen",2074,2009-07-20,1448.6
"Limozeen",2077,2009-07-20,1109.65
"Limozeen",2083,2009-07-26,2128.64
"Limozeen",2088,2009-07-27,1308.5
"Limozeen",2091,2009-07-28,1079.33
"Limozeen",2097,2009-07-29,3974.07
"Limozeen",2101,2009-07-29,6736.33
"Limozeen",2107,2009-07-29,1275.07
"Limozeen",2111,2009-08-08,7338.03
"Limozeen",2120,2009-08-09,1721.72
"Limozeen",2122,2009-08-17,3007.31
"Limozeen",2127,2009-08-27,1167.15
"Limozeen",2132,2009-08-28,1049.31
"Limozeen",2137,2009-08-28,6077.87
"Limozeen",2143,2009-08-28,8222.97
"Limozeen",2146,2009-09-03,2683.01
"Limozeen",2152,2009-09-03,1184.32
"Limozeen",2158,2009-09-03,1441.42
"Limozeen",2164,2009-09-09,2484.63
"Limozeen",2168,2009-09-09,7009.42
"Limozeen",2175,2009-09-15,2545.01
"Limozeen",2180,2009-09-16,1501.95
"Limozeen",2181,2009-09-24,5553.84
"Limozeen",2188,2009-09-24,2473.19
"Limozeen",2192,2009-09-25,4620.83
"Limozeen",2199,2009-09-25,2825.31
"Limozeen",2201,2009-09-26,3364.24
"Limozeen",2210,2009-09-27,4312.61
"Limozeen",2212,2009-09-28,1216.5
"Limozeen",2216,2009-09-29,2106.3
"Limozeen",2224,2009-10-07,5088.29
"Limozeen",2230,2009-10-13,8753.63
"Limozeen",2234,2009-10-13,1371.1
"Limozeen",2238,2009-10-13,7457.67
"Limozeen",2242,2009-10-14,3701.84
"Limozeen",2250,2009-10-20,9372.42
"Limozeen",2251,2009-10-20,8320.43
"Limozeen",2258,2009-10-30,5430.12
"Limozeen",2261,2009-10-31,4520.52
"Limozeen",2266,2009-11-06,2786.45
"Limozeen",2272,2009-11-14,4346.44
"Limozeen",2280,2009-11-15,8995.86
"Limozeen",2284,2009-11-16,5138.36
"Limozeen",2289,2009-11-17,1931.41
"Limozeen",2293,2009-11-18,3183.76
"Limozeen",2300,2009-11-24,1930.28
"Limozeen",2302,2009-12-02,4240.05
"Limozeen",2307,2009-12-10,2584.2
"Limozeen",2312,2009-12-11,7974.65
"Limozeen",2316,2009-12-11,5173.26
"Limozeen",2322,2009-12-12,4174.47
"Limozeen",2327,2009-12-13,1761.28
"Limozeen",2331,2009-12-14,7069.84
"Limozeen",2338,2009-12-15,7056.92
"Limozeen",2343,2009-12-15,9616.49
"Limozeen",2346,2009-12-21,1859.74
"Limozeen",2354,2009-12-22,8169.68
"Limozeen",2359,2009-12-22,5143.24
"Limozeen",2364,2009-12-28,1156.05
"Limozeen",2367,2010-01-03,8943.86
"Limozeen",2374,2010-01-09,4042.98
"Limozeen",2380,2010-01-19,1383.37
"Limozeen",2385,2010-01-20,2674.85
"Limozeen",2389,2010-01-20,1893.44
"Limozeen",2394,2010-01-20,2507.82
"Limozeen",2399,2010-01-30,3420.4
"Limozeen",2404,2010-01-30,9627.89
"Limozeen",2410,2010-01-30,1564.84
"Limozeen",2412,2010-02-09,1122.35
"Limozeen",2416,2010-02-10,1335.25
"Limozeen",2424,2010-02-16,1004.43
"Limozeen",2430,2010-02-17,1598.31
"Limozeen",2432,2010-02-17,2225.61
"Limozeen",2439,2010-02-17,5793.73
"Limozeen",2442,2010-02-18,3351.24
"Limozeen",2449,2010-02-26,2329.48
"Limozeen",2452,2010-03-08,5985.5
"Limozeen",2456,2010-03-18,1115.61
"Limozeen",2463,2010-03-24,7834.13
"Limozeen",2468,2010-04-03,1292.85
"Limozeen",2473,2010-04-09,1094.45
"Limozeen",2480,2010-04-09,2300.12
"Limozeen",2482,2010-04-15,4314.95
"Limozeen",2486,2010-04-23,1901.69
"Limozeen",2493,2010-04-24,2020.12
"Limozeen",2500,2010-04-25,1792.67
"Limozeen",2505,2010-05-01,1131.19
"Limozeen",2508,2010-05-07,3829.65
"Limozeen",2514,2010-05-07,4845.8
"Limozeen",2519,2010-05-13,5146.26
"Limozeen",2522,2010-05-14,2062.24
"Limozeen",2527,2010-05-20,3433.14
"Limozeen",2535,2010-05-26,1955.67
"Limozeen",2537,2010-06-03,5096.37
"Limozeen",2543,2010-06-03,8419.89
"Limozeen",2546,2010-06-09,5614.48
"Limozeen",2554,2010-06-19,7881.75
"Limozeen",2560,2010-06-25,7378.36
"Limozeen",2561,2010-07-03,2193.67
"Limozeen",2569,2010-07-03,2455.85
"Limozeen",2572,2010-07-03,2181.69
"Limozeen",2579,2010-07-03,1509.11
"Limozeen",2582,2010-07-04,1239.84
"Limozeen",2589,2010-07-05,2503.83
"Limozeen",2592,2010-07-05,1259.58
"Limozeen",2599,2010-07-06,5316.19
"Limozeen",2601,2010-07-16,3045.59
"Limozeen",2606,2010-07-26,1528.67
"Limozeen",2612,2010-07-26,1172.84
"Limozeen",2617,2010-08-01,1851.44
"Limozeen",2621,2010-08-02,6760.02
"Limozeen",2626,2010-08-08,1007.48
"Limozeen",2634,2010-08-08,2900.91
"Limozeen",2639,2010-08-16,1555.28
"Limozeen",2641,2010-08-16,1056.58
"Limozeen",2648,2010-08-17,2055.95
"Limozeen",2651,2010-08-18,5423.81
"Limozeen",2656,2010-08-24,3237.85
"Limozeen",2661,2010-08-24,2252.16
"Limozeen",2668,2010-08-30,3242.34
"Limozeen",2674,2010-08-31,3700.75
"Limozeen",2680,2010-09-08,4327.97
"Limozeen",2683,2010-09-08,4341.33
"Limozeen",2687,2010-09-09,9468.67
"Limozeen",2692,2010-09-09,1660.92
"Limozeen",2698,2010-09-10,1133.47
"Limozeen",2705,2010-09-16,4952.13
"Limozeen",2710,2010-09-17,7787.71
"Limozeen",2712,2010-09-17,5491.35
"Limozeen",2718,2010-09-18,2967.72
"Limozeen",2725,2010-09-24,8378.9
"Limozeen",2726,2010-09-30,5710.97
"Limozeen",2735,2010-10-08,6545.03
"Limozeen",2736,2010-10-09,9450.3
"Limozeen",2744,2010-10-19,7151.96
"Limozeen",2747,2010-10-19,1274.63
"Limozeen",2751,2010-10-20,2198.86
"Limozeen",2757,2010-10-26,2313.73
"Limozeen",2762,2010-11-05,4859.39
"Limozeen",2767,2010-11-06,4930.59
"Limozeen",2773,2010-11-07,1665.23
"Limozeen",2780,2010-11-17,1807.98
"Limozeen",2782,2010-11-18,3704.1
"Limozeen",2787,2010-11-19,1076.61
"Limozeen",2791,2010-11-19,1280.78
"Limozeen",2796,2010-11-25,2344.76
"Limozeen",2804,2010-12-01,9045.03
"Limozeen",2807,2010-12-07,8282.69
"Limozeen",2812,2010-12-17,4324.9
"Limozeen",2820,2010-12-17,6177.37
"Limozeen",2824,2010-12-17,4824.7
"Limozeen",2830,2010-12-17,8207.23
"Limozeen",2832,2010-12-17,1621.6
"Limozeen",2839,2010-12-18,1683.22
"Limozeen",2844,2010-12-18,1440.78
"Limozeen",2846,2010-12-19,8165.97
"Limozeen",2854,2010-12-25,4876.85
"Limozeen",2858,2011-01-02,6417.17
"Limozeen",2862,2011-01-08,1785.44
"Limozeen",2868,2011-01-14,7884.17
"Limozeen",2871,2011-01-15,1525.97
"Limozeen",2880,2011-01-21,9863.08
"Limozeen",2882,2011-01-22,2124.44
"Limozeen",2886,2011-01-28,2107.6
"Limozeen",2895,2011-02-07,4132.88
"Limozeen",2896,2011-02-07,4835.36
"Limozeen",2905,2011-02-17,1020.55
"Limozeen",2906,2011-02-27,4431.05
"Limozeen",2912,2011-03-07,4829.53
"Limozeen",2918,2011-03-07,5552.16
"Limozeen",2921,2011-03-13,6385.69
"Limozeen",2928,2011-03-13,4568.93
"Limozeen",2931,2011-03-13,4754.25
"Limozeen",2937,2011-03-21,1907.65
"Limozeen",2942,2011-03-22,8822.45
"Limozeen",2947,2011-03-28,1729.12
"Limozeen",2952,2011-04-07,2431.43
"Limozeen",2958,2011-04-07,9339.68
"Limozeen",2962,2011-04-13,1123.57
"Limozeen",2966,2011-04-19,5546.63
"Limozeen",2974,2011-04-25,2362.04
"Limozeen",2978,2011-05-01,1671.14
"Limozeen",2982,2011-05-02,1927.14
"Limozeen",2988,2011-05-03,1933.91
"Limozeen",2992,2011-05-04,4532.95
"Limozeen",2999,2011-05-12,2173.32
"Limozeen",3001,2011-05-20,3621.01
"Limozeen",3008,2011-05-21,5890.06
"Limozeen",3013,2011-05-21,2002.92
"Limozeen",3020,2011-05-22,1473.16
"Limozeen",3024,2011-05-30,1070.97
"Limozeen",3027,2011-05-30,3631.87
"Limozeen",3033,2011-06-05,1606.9
"Limozeen",3038,2011-06-06,1362.43
"Limozeen",3045,2011-06-06,1087.24
"Limozeen",3047,2011-06-12,1980.67
"Limozeen",3052,2011-06-12,6814.57
"Limozeen",3056,2011-06-13,2768.17
"Limozeen",3065,2011-06-14,2618.84
"Limozeen",3066,2011-06-20,2403.5
"Limozeen",3071,2011-06-20,1364.97
"Limozeen",3080,2011-06-20,4535.26
"Limozeen",3082,2011-06-21,8084.62
"Limozeen",3090,2011-06-22,5730.69
"Limozeen",3092,2011-06-22,4417.39
"Limozeen",3098,2011-06-23,2814.3
"Limozeen",3103,2011-07-01,1220.63
"Limozeen",3107,2011-07-02,3098.56
"Limozeen",3114,2011-07-02,4064.8
"Limozeen",3118,2011-07-08,6915.87
"Limozeen",3125,2011-07-08,5616.84
"Limozeen",3130,2011-07-08,4929.88
"Limozeen",3132,2011-07-09,1551.99
"Limozeen",3139,2011-07-19,1018.1
"Limozeen",3141,2011-07-20,6371.71
"Limozeen",3147,2011-07-20,1618.94
"Limozeen",3153,2011-07-30,1915.15
"Limozeen",3156,2011-08-05,1767.01
"Limozeen",3162,2011-08-05,2977.73
"Limozeen",3168,2011-08-05,5054.14
"Limozeen",3172,2011-08-05,1981.55
"Limozeen",3177,2011-08-06,2019.57
"Limozeen",3182,2011-08-06,3245.57
"Limozeen",3189,2011-08-12,2880.05
"Limozeen",3193,2011-08-13,9678.82
"Limozeen",3198,2011-08-14,2439.26
"Limozeen",3201,2011-08-20,1216.11
"Limozeen",3207,2011-08-26,1732.15
"Limozeen",3213,2011-08-27,2211.22
"Limozeen",3220,2011-09-02,8696.05
"Limozeen",3225,2011-09-02,6310.15
"Limozeen",3229,2011-09-03,1344.45
"Limozeen",3232,2011-09-04,5721.3
"Limozeen",3236,2011-09-12,6567.07
"Limozeen",3241,2011-09-18,4329.61
"Limozeen",3250,2011-09-19,1839.95
"Limozeen",3252,2011-09-19,2443.8
"Limozeen",3259,2011-09-19,5603.86
"Limozeen",3262,2011-09-19,2940.6
"Limozeen",3268,2011-09-20,3829.32
"Limozeen",3272,2011-09-21,7599.7
"Limozeen",3280,2011-09-21,1194.95
"Limozeen",3284,2011-09-21,5951.49
"Limozeen",3288,2011-09-29,7359.34
"Limozeen",3294,2011-09-30,1357.52
"Limozeen",3300,2011-10-01,1439.28
"Limozeen",3304,2011-10-01,6089.26
"Limozeen",3306,2011-10-11,7416.84
"Limozeen",3314,2011-10-21,2814.33
"Limozeen",3317,2011-10-21,8685.19
"Limozeen",3324,2011-10-21,2632.85
"Limozeen",3327,2011-10-22,2391.61
"Limozeen",3331,2011-10-30,9472.25
"Limozeen",3340,2011-10-30,1157.33
"Limozeen",3344,2011-10-30,1783.86
"Limozeen",3349,2011-10-30,1550.29
"Limozeen",3355,2011-11-05,3192.77
"Limozeen",3359,2011-11-05,1792.56
"Limozeen",3361,2011-11-06,1363.54
"Limozeen",3370,2011-11-06,8792.22
"Limozeen",3374,2011-11-12,3239.21
"Limozeen",3380,2011-11-13,2560.97
"Limozeen",3382,2011-11-13,5077.62
"Limozeen",3390,2011-11-23,2508.21
"Limozeen",3391,2011-11-23,1830.55
"Limozeen",3399,2011-11-24,1971.8
"Limozeen",3404,2011-11-24,1772.5
"Limozeen",3406,2011-11-25,8514.5
"Limozeen",3415,2011-11-25,5803.93
"Limozeen",3420,2011-12-05,7214.52
"Limozeen",3423,2011-12-11,2079.75
"Limozeen",3429,2011-12-17,3785.46
"Limozeen",3433,2011-12-17,3444.0
"Limozeen",3438,2011-12-18,6675.1
"Limozeen",3442,2011-12-18,1115.3
"Limozeen",3446,2011-12-18,7043.82
"Limozeen",3455,2011-12-26,1998.7
"Limozeen",3460,2011-12-26,5095.69
"Limozeen",3464,2012-01-03,1974.55
"Limozeen",3469,2012-01-03,5378.18
"Limozeen",3471,2012-01-13,4421.13
"Limozeen",3480,2012-01-13,8956.64
"Limozeen",3482,2012-01-14,4879.0
"Limozeen",3488,2012-01-15,5004.62
"Limozeen",3491,2012-01-15,3524.44
"Limozeen",3497,2012-01-23,2307.68
"Limozeen",3503,2012-01-24,1931.41
"Limozeen",3506,2012-01-24,8243.64
"Limozeen",3514,2012-01-25,7500.41
"Limozeen",3519,2012-02-04,1460.48
"Limozeen",3525,2012-02-04,1603.55
"Limozeen",3527,2012-02-04,1566.02
"Limozeen",3532,2012-02-14,1563.1
"Limozeen",3539,2012-02-20,6279.29
"Limozeen",3541,2012-02-26,2834.48
"Limozeen",3549,2012-03-03,1224.58
"Limozeen",3553,2012-03-09,2486.41
"Limozeen",3558,2012-03-10,3389.53
"Limozeen",3565,2012-03-16,8218.47
"Limozeen",3570,2012-03-24,4339.53
"Limozeen",3573,2012-03-24,9809.66
"Limozeen",3578,2012-03-25,3647.98
"Limozeen",3583,2012-04-02,6946.01
"Limozeen",3586,2012-04-08,3026.25
"Limozeen",3595,2012-04-09,1557.4
"Limozeen",3600,2012-04-09,5411.93
"Limozeen",3602,2012-04-09,7899.71
"Limozeen",3608,2012-04-09,1062.98
"Limozeen",3614,2012-04-09,3611.26
"Limozeen",3618,2012-04-10,4054.44
"Limozeen",3623,2012-04-11,1467.34
"Limozeen",3629,2012-04-12,4000.25
"Limozeen",3633,2012-04-12,5678.68
"Limozeen",3640,2012-04-12,6342.47
"Limozeen",3644,2012-04-22,3707.9
"Limozeen",3650,2012-04-22,4075.46
"Limozeen",3653,2012-04-28,5373.08
"Limozeen",3658,2012-04-29,7286.13
"Limozeen",3661,2012-04-30,3395.06
"Limozeen",3668,2012-05-06,4869.15
"Limozeen",3671,2012-05-06,7891.09
"Limozeen",3680,2012-05-07,1972.03
"Limozeen",3681,2012-05-13,2802.26
"Limozeen",3689,2012-05-14,8432.47
"Limozeen",3695,2012-05-20,2366.95
"Limozeen",3696,2012-05-26,2722.79
"Limozeen",3701,2012-05-26,9469.83
"Limozeen",3710,2012-05-27,1103.61
"Limozeen",3714,2012-05-27,3649.97
"Limozeen",3719,2012-05-28,1452.31
"Limozeen",3724,2012-05-29,3620.85
"Limozeen",3727,2012-05-30,5747.46
"Limozeen",3731,2012-06-05,3929.14
"Limozeen",3739,2012-06-05,4632.38
"Limozeen",3743,2012-06-05,1380.6
"Limozeen",3750,2012-06-05,1753.29
"Limozeen",3751,2012-06-05,3226.94
"Limozeen",3759,2012-06-06,1157.11
"Limozeen",3763,2012-06-16,8055.32
"Limozeen",3766,2012-06-22,3108.62
"Limozeen",3774,2012-06-23,1186.64
"Limozeen",3778,2012-06-24,3551.02
"Limozeen",3783,2012-06-24,1703.13
"Limozeen",3789,2012-06-25,3591.01
"Limozeen",3791,2012-07-05,8922.27
"Limozeen",3798,2012-07-05,7858.66
"Limozeen",3802,2012-07-05,1158.6
"Limozeen",3808,2012-07-06,1602.12
"Limozeen",3814,2012-07-07,3304.14
"Limozeen",3819,2012-07-07,9283.1
"Limozeen",3822,2012-07-07,3147.92
"Limozeen",3827,2012-07-13,5552.0
"Limozeen",3832,2012-07-19,1737.38
"Limozeen",3837,2012-07-20,1549.84
"Limozeen",3843,2012-07-28,4644.27
"Limozeen",3847,2012-08-03,1540.83
"Limozeen",3853,2012-08-03,1750.71
"Limozeen",3856,2012-08-11,2360.4
"Limozeen",3862,2012-08-17,2556.8
"Limozeen",3866,2012-08-23,7639.99
"Limozeen",3872,2012-08-24,3448.74
"Limozeen",3878,2012-08-25,6784.29
"Limozeen",3884,2012-09-04,3178.0
"Limozeen",3889,2012-09-10,2126.58
"Limozeen",3892,2012-09-11,1909.02
"Limozeen",3898,2012-09-17,5226.6
"Limozeen",3902,2012-09-23,6283.64
"Limozeen",3907,2012-09-23,1158.83
"Limozeen",3914,2012-09-23,2735.21
"Limozeen",3919,2012-09-23,1630.11
"Limozeen",3923,2012-10-01,6900.24
"Limozeen",3927,2012-10-09,1830.51
"Limozeen",3935,2012-10-10,8385.42
"Limozeen",3937,2012-10-11,4040.43
"Limozeen",3942,2012-10-12,1676.09
"Limozeen",3946,2012-10-12,2905.8
"Limozeen",3953,2012-10-12,9065.73
"Limozeen",3959,2012-10-12,2889.26
"Limozeen",3962,2012-10-12,3346.18
"Limozeen",3968,2012-10-12,6751.57
"Limozeen",3971,2012-10-18,4805.97
"Limozeen",3976,2012-10-24,5892.59
"Limozeen",3983,2012-10-30,3523.51
"Limozeen",3989,2012-10-31,1024.22
"Limozeen",3994,2012-11-06,1114.93
"Limozeen",3997,2012-11-16,1503.39
"Limozeen",4005,2012-11-16,7242.72
"Limozeen",4008,2012-11-26,1433.78
"Limozeen",4012,2012-11-26,3777.59
"Limozeen",4016,2012-12-02,2430.08
"Limozeen",4024,2012-12-12,5371.1
"Limozeen",4027,2012-12-12,1466.13
"Limozeen",4035,2012-12-18,1843.31
"Limozeen",4039,2012-12-19,2245.34
"Limozeen",4042,2012-12-20,1633.98
"Limozeen",4047,2012-12-21,5961.77
"Limozeen",4051,2012-12-21,5859.18
"Limozeen",4059,2012-12-31,5642.63
"Limozeen",4062,2013-01-06,8830.24
"Limozeen",4066,2013-01-16,4860.2
"Limozeen",4074,2013-01-24,1605.67
"Limozeen",4076,2013-02-03,1369.68
"Limozeen",4083,2013-02-03,8103.98
"Limozeen",4088,2013-02-04,3744.84
"Limozeen",4091,2013-02-04,2283.07
"Limozeen",4099,2013-02-05,4937.61
"Limozeen",4105,2013-02-05,1880.52
"Limozeen",4110,2013-02-11,6103.15
"Limozeen",4112,2013-02-12,1854.41
"Limozeen",4118,2013-02-13,1277.52
"Limozeen",4124,2013-02-14,3439.29
"Limozeen",4126,2013-02-20,3242.89
"Limozeen",4133,2013-02-20,6024.78
"Limozeen",4139,2013-03-02,3939.51
"Limozeen",4141,2013-03-10,1679.0
"Limozeen",4150,2013-03-20,6948.12
"Limozeen",4154,2013-03-20,4590.41
"Limozeen",4158,2013-03-21,5095.31
"Limozeen",4161,2013-03-29,9372.92
"Limozeen",4166,2013-04-06,1013.43
"Limozeen",4175,2013-04-06,1169.16
"Limozeen",4176,2013-04-14,1876.48
"Limozeen",4181,2013-04-15,3401.46
"Limozeen",4187,2013-04-25,8739.95
"Limozeen",4193,2013-05-05,1593.38
"Limozeen",4200,2013-05-06,1993.71
"Limozeen",4202,2013-05-14,1956.22
"Limozeen",4208,2013-05-20,4457.17
"Limozeen",4214,2013-05-26,6685.28
"Limozeen",4219,2013-05-26,1046.78
"Limozeen",4223,2013-05-27,4521.59
"Limozeen",4229,2013-05-28,5074.32
"Limozeen",4234,2013-06-05,2113.71
"Limozeen",4240,2013-06-11,1544.76
"Limozeen",4245,2013-06-12,9678.26
"Limozeen",4249,2013-06-12,5432.01
"Limozeen",4251,2013-06-18,7889.01
"Limozeen",4258,2013-06-24,5035.25
"Limozeen",4263,2013-06-24,7101.8
"Limozeen",4270,2013-06-25,7093.1
"Limozeen",4274,2013-07-05,1306.51
"Limozeen",4276,2013-07-06,1721.87
"Limozeen",4284,2013-07-06,3876.27
"Limozeen",4287,2013-07-07,1581.42
"Limozeen",4291,2013-07-13,2348.39
"Limozeen",4298,2013-07-19,3143.18
"Limozeen",4305,2013-07-20,4537.22
"Limozeen",4308,2013-07-20,6915.15
"Limozeen",4312,2013-07-30,2009.81
"Limozeen",4317,2013-08-07,7279.7
"Limozeen",4323,2013-08-07,9121.7
"Limozeen",4329,2013-08-07,3128.46
"Limozeen",4335,2013-08-13,2158.32
"Limozeen",4338,2013-08-14,3945.55
"Limozeen",4343,2013-08-14,2696.59
"Limozeen",4350,2013-08-14,6506.32
"Limozeen",4355,2013-08-15,2494.63
"Limozeen",4356,2013-08-21,4192.77
"NNP Inc",7,2006-01-01,1753.22
"NNP Inc",15,2006-01-07,1760.53
"NNP Inc",16,2006-01-07,4136.16
"NNP Inc",25,2006-01-13,1002.11
"NNP Inc",27,2006-01-14,3458.96
"NNP Inc",33,2006-01-20,4130.27
"NNP Inc",40,2006-01-20,8191.66
"NNP Inc",45,2006-01-26,5012.68
"NNP Inc",46,2006-01-27,5401.3
"NNP Inc",52,2006-01-28,2007.29
"NNP Inc",59,2006-01-29,2071.4
"NNP Inc",61,2006-01-30,3759.4
"NNP Inc",67,2006-01-30,7832.38
"NNP Inc",72,2006-01-30,3061.93
"NNP Inc",79,2006-01-31,9754.95
"NNP Inc",81,2006-01-31,6649.24
"NNP Inc",89,2006-02-08,8150.9
"NNP Inc",92,2006-02-09,2506.92
"NNP Inc",97,2006-02-10,1675.83
"NNP Inc",103,2006-02-16,5521.75
"NNP Inc",109,2006-02-22,4782.98
"NNP Inc",113,2006-03-02,1677.41
"NNP Inc",117,2006-03-02,2067.33
"NNP Inc",123,2006-03-10,4710.15
"NNP Inc",128,2006-03-16,1131.99
"NNP Inc",133,2006-03-16,1733.1
"NNP Inc",137,2006-03-22,1665.42
"NNP Inc",141,2006-03-28,1083.12
"NNP Inc",147,2006-04-03,1683.59
"NNP Inc",154,2006-04-13,1284.58
"NNP Inc",159,2006-04-19,7098.46
"NNP Inc",164,2006-04-20,1356.93
"NNP Inc",168,2006-04-28,1771.32
"NNP Inc",174,2006-05-04,1458.95
"NNP Inc",180,2006-05-04,2310.12
"NNP Inc",184,2006-05-05,3624.29
"NNP Inc",187,2006-05-06,2043.02
"NNP Inc",194,2006-05-07,3464.01
"NNP Inc",200,2006-05-07,1081.46
"NNP Inc",205,2006-05-07,4940.58
"NNP Inc",210,2006-05-07,2569.9
"NNP Inc",211,2006-05-15,1402.39
"NNP Inc",217,2006-05-15,9833.31
"NNP Inc",222,2006-05-15,4939.76
"NNP Inc",228,2006-05-21,4675.75
"NNP Inc",235,2006-05-31,4131.35
"NNP Inc",238,2006-06-06,1847.89
"NNP Inc",241,2006-06-06,3105.0
"NNP Inc",250,2006-06-06,2242.68
"NNP Inc",254,2006-06-07,4004.66
"NNP Inc",257,2006-06-07,2840.13
"NNP Inc",264,2006-06-08,4640.05
"NNP Inc",268,2006-06-14,5376.07
"NNP Inc",271,2006-06-14,1613.04
"NNP Inc",279,2006-06-22,3733.61
"NNP Inc",282,2006-06-23,5322.04
"NNP Inc",289,2006-06-23,4890.96
"NNP Inc",292,2006-06-24,3871.6
"NNP Inc",296,2006-06-30,1132.05
"NNP Inc",305,2006-07-08,3755.36
"NNP Inc",310,2006-07-08,7694.16
"NNP Inc",314,2006-07-08,5599.13
"NNP Inc",320,2006-07-09,1575.46
"NNP Inc",323,2006-07-19,2021.06
"NNP Inc",327,2006-07-29,1581.17
"NNP Inc",333,2006-08-04,6051.3
"NNP Inc",336,2006-08-12,1517.48
"NNP Inc",345,2006-08-12,4424.45
"NNP Inc",349,2006-08-13,7984.57
"NNP Inc",351,2006-08-13,9175.09
"NNP Inc",357,2006-08-14,1418.26
"NNP Inc",365,2006-08-14,3197.85
"NNP Inc",369,2006-08-15,1920.41
"NNP Inc",375,2006-08-16,3731.42
"NNP Inc",378,2006-08-22,6789.64
"NNP Inc",384,2006-08-28,1117.91
"NNP Inc",387,2006-08-28,1790.56
"NNP Inc",391,2006-08-28,3514.7
"NNP Inc",396,2006-09-05,1085.06
"NNP Inc",404,2006-09-05,5676.06
"NNP Inc",410,2006-09-06,9523.56
"NNP Inc",414,2006-09-12,4501.68
"NNP Inc",417,2006-09-20,1185.91
"NNP Inc",423,2006-09-20,4950.72
"NNP Inc",429,2006-09-26,1321.49
"NNP Inc",434,2006-10-04,4187.5
"NNP Inc",440,2006-10-05,2658.03
"NNP Inc",444,2006-10-06,3151.94
"NNP Inc",450,2006-10-07,1174.7
"NNP Inc",453,2006-10-17,3655.76
"NNP Inc",456,2006-10-23,1377.81
"NNP Inc",462,2006-11-02,1501.3
"NNP Inc",469,2006-11-08,9170.73
"NNP Inc",471,2006-11-09,2662.81
"NNP Inc",480,2006-11-19,1448.47
"NNP Inc",482,2006-11-20,1367.0
"NNP Inc",487,2006-11-20,2697.43
"NNP Inc",495,2006-11-26,2959.63
"NNP Inc",498,2006-11-26,3322.4
"NNP Inc",502,2006-12-02,7571.13
"NNP Inc",510,2006-12-03,1256.17
"NNP Inc",513,2006-12-04,3148.63
"NNP Inc",516,2006-12-14,9460.17
"NNP Inc",524,2006-12-15,3428.22
"NNP Inc",526,2006-12-21,1440.13
"NNP Inc",531,2006-12-27,4365.01
"NNP Inc",540,2006-12-28,5099.66
"NNP Inc",545,2007-01-03,7665.12
"NNP Inc",546,2007-01-03,3608.65
"NNP Inc",553,2007-01-04,3673.09
"NNP Inc",558,2007-01-04,3913.75
"NNP Inc",564,2007-01-10,1193.48
"NNP Inc",569,2007-01-11,5219.73
"NNP Inc",575,2007-01-11,9329.65
"NNP Inc",579,2007-01-19,3974.63
"NNP Inc",585,2007-01-27,3693.31
"NNP Inc",590,2007-01-27,1350.26
"NNP Inc",595,2007-02-06,5828.17
"NNP Inc",598,2007-02-12,7120.3
"NNP Inc",604,2007-02-18,8281.59
"NNP Inc",607,2007-02-24,6877.77
"NNP Inc",612,2007-03-06,2318.53
"NNP Inc",619,2007-03-07,9546.75
"NNP Inc",625,2007-03-07,1672.6
"NNP Inc",626,2007-03-13,8455.93
"NNP Inc",635,2007-03-23,2188.11
"NNP Inc",638,2007-03-29,1422.96
"NNP Inc",642,2007-04-04,2638.4
"NNP Inc",647,2007-04-04,9684.66
"NNP Inc",653,2007-04-14,1524.44
"NNP Inc",659,2007-04-14,8294.8
"NNP Inc",661,2007-04-14,3990.09
"NNP Inc",669,2007-04-14,7129.56
"NNP Inc",675,2007-04-20,1181.25
"NNP Inc",678,2007-04-21,1742.89
"NNP Inc",685,2007-04-22,1515.67
"NNP Inc",690,2007-04-23,7490.66
"NNP Inc",693,2007-04-23,1361.05
"NNP Inc",697,2007-04-29,6064.62
"NNP Inc",701,2007-05-05,2105.19
"NNP Inc",709,2007-05-11,4158.65
"NNP Inc",713,2007-05-17,4243.75
"NNP Inc",717,2007-05-18,1401.82
"NNP Inc",721,2007-05-18,5750.31
"NNP Inc",728,2007-05-28,1129.39
"NNP Inc",732,2007-05-29,3090.27
"NNP Inc",737,2007-06-04,6049.3
"NNP Inc",744,2007-06-10,4516.29
"NNP Inc",748,2007-06-20,4554.97
"NNP Inc",752,2007-06-26,1818.73
"NNP Inc",757,2007-06-26,9448.23
"NNP Inc",761,2007-06-26,1300.64
"NNP Inc",768,2007-06-27,8907.66
"NNP Inc",775,2007-07-05,6517.05
"NNP Inc",779,2007-07-05,3232.47
"NNP Inc",783,2007-07-05,4439.84
"NNP Inc",790,2007-07-11,4224.04
"NNP Inc",795,2007-07-11,5265.73
"NNP Inc",796,2007-07-21,7400.02
"NNP Inc",801,2007-07-27,2855.94
"NNP Inc",810,2007-07-28,2107.0
"NNP Inc",812,2007-07-28,6443.65
"NNP Inc",818,2007-07-29,5565.13
"NNP Inc",825,2007-07-29,1227.08
"NNP Inc",829,2007-07-30,4592.35
"NNP Inc",835,2007-07-30,1820.36
"NNP Inc",836,2007-07-30,1344.48
"NNP Inc",844,2007-08-07,3380.94
"NNP Inc",846,2007-08-08,8294.25
"NNP Inc",854,2007-08-18,1366.78
"NNP Inc",856,2007-08-18,1518.31
"NNP Inc",861,2007-08-24,6747.66
"NNP Inc",870,2007-08-25,4441.65
"NNP Inc",875,2007-09-04,3089.23
"NNP Inc",878,2007-09-10,1180.92
"NNP Inc",881,2007-09-18,1469.83
"NNP Inc",890,2007-09-18,3328.37
"NNP Inc",891,2007-09-18,9097.12
"NNP Inc",900,2007-09-24,1390.72
"NNP Inc",901,2007-09-25,4384.82
"NNP Inc",910,2007-09-25,2928.09
"NNP Inc",913,2007-09-26,1162.63
"NNP Inc",918,2007-09-27,9497.37
"NNP Inc",922,2007-09-28,1992.2
"NNP Inc",928,2007-10-04,1828.74
"NNP Inc",932,2007-10-04,1076.12
"NNP Inc",938,2007-10-04,3103.46
"NNP Inc",945,2007-10-10,1284.9
"NNP Inc",946,2007-10-20,5128.96
"NNP Inc",953,2007-10-20,1445.33
"NNP Inc",957,2007-10-26,1059.01
"NNP Inc",964,2007-10-26,6897.11
"NNP Inc",968,2007-10-26,1593.01
"NNP Inc",974,2007-11-05,8514.64
"NNP Inc",980,2007-11-11,9283.73
"NNP Inc",982,2007-11-17,3780.21
"NNP Inc",990,2007-11-17,3355.02
"NNP Inc",992,2007-11-17,1544.56
"NNP Inc",999,2007-11-25,1836.1
"NNP Inc",1005,2007-12-03,1914.31
"NNP Inc",1010,2007-12-13,2173.69
"NNP Inc",1015,2007-12-14,2957.77
"NNP Inc",1016,2007-12-14,9267.07
"NNP Inc",1021,2007-12-24,2385.43
"NNP Inc",1026,2007-12-25,6219.3
"NNP Inc",1034,2008-01-04,8336.88
"NNP Inc",1040,2008-01-12,6482.28
"NNP Inc",1043,2008-01-12,2463.63
"NNP Inc",1048,2008-01-20,3455.01
"NNP Inc",1053,2008-01-20,4470.85
"NNP Inc",1060,2008-01-26,3267.78
"NNP Inc",1061,2008-01-27,5196.47
"NNP Inc",1066,2008-01-28,6920.68
"NNP Inc",1073,2008-01-29,1180.34
"NNP Inc",1077,2008-02-04,9965.21
"NNP Inc",1082,2008-02-05,7605.88
"NNP Inc",1087,2008-02-13,3289.12
"NNP Inc",1092,2008-02-21,5688.96
"NNP Inc",1099,2008-02-27,4299.13
"NNP Inc",1101,2008-02-27,6322.12
"NNP Inc",1109,2008-03-06,4907.3
"NNP Inc",1114,2008-03-06,3067.36
"NNP Inc",1118,2008-03-07,2556.99
"NNP Inc",1121,2008-03-15,5249.99
"NNP Inc",1126,2008-03-16,3715.83
"NNP Inc",1131,2008-03-22,1890.7
"NNP Inc",1138,2008-03-23,3552.03
"NNP Inc",1143,2008-03-23,2215.32
"NNP Inc",1147,2008-03-23,2047.93
"NNP Inc",1155,2008-03-23,2230.11
"NNP Inc",1159,2008-03-29,1105.1
"NNP Inc",1163,2008-04-08,1097.82
"NNP Inc",1166,2008-04-09,2157.06
"NNP Inc",1173,2008-04-09,4450.59
"NNP Inc",1179,2008-04-15,1218.58
"NNP Inc",1181,2008-04-16,1269.85
"NNP Inc",1188,2008-04-17,9204.02
"NNP Inc",1194,2008-04-17,3595.14
"NNP Inc",1198,2008-04-17,2690.91
"NNP Inc",1202,2008-04-17,1023.49
"NNP Inc",1208,2008-04-25,1093.02
"NNP Inc",1215,2008-04-25,1612.23
"NNP Inc",1216,2008-04-26,3165.95
"NNP Inc",1221,2008-04-26,1623.97
"NNP Inc",1229,2008-05-06,1123.94
"NNP Inc",1231,2008-05-06,8795.71
"NNP Inc",1240,2008-05-07,9105.45
"NNP Inc",1245,2008-05-15,1372.11
"NNP Inc",1248,2008-05-25,2493.82
"NNP Inc",1252,2008-05-26,3608.54
"NNP Inc",1256,2008-05-27,1538.19
"NNP Inc",1261,2008-05-27,4212.45
"NNP Inc",1269,2008-06-02,3210.7
"NNP Inc",1273,2008-06-02,6182.63
"NNP Inc",1278,2008-06-03,5249.37
"NNP Inc",1281,2008-06-13,5433.43
"NNP Inc",1290,2008-06-14,4875.35
"NNP Inc",1291,2008-06-14,4661.57
"NNP Inc",1298,2008-06-14,6981.5
"NNP Inc",1302,2008-06-14,5647.42
"NNP Inc",1306,2008-06-14,2844.91
"NNP Inc",1314,2008-06-14,3528.95
"NNP Inc",1317,2008-06-15,1726.52
"NNP Inc",1321,2008-06-15,8740.13
"NNP Inc",1330,2008-06-15,2525.72
"NNP Inc",1335,2008-06-21,4800.75
"NNP Inc",1336,2008-06-29,5281.26
"NNP Inc",1342,2008-06-29,1034.24
"NNP Inc",1346,2008-06-29,4761.15
"NNP Inc",1351,2008-06-30,3414.52
"NNP Inc",1356,2008-07-01,1217.78
"NNP Inc",1363,2008-07-02,3759.07
"NNP Inc",1366,2008-07-08,1608.63
"NNP Inc",1371,2008-07-09,6026.23
"NNP Inc",1380,2008-07-17,8923.4
"NNP Inc",1385,2008-07-23,2032.48
"NNP Inc",1389,2008-07-29,5222.46
"NNP Inc",1394,2008-08-06,2150.09
"NNP Inc",1400,2008-08-07,2244.68
"NNP Inc",1402,2008-08-17,7305.09
"NNP Inc",1407,2008-08-18,2736.8
"NNP Inc",1414,2008-08-24,1176.28
"NNP Inc",1417,2008-08-25,5630.48
"NNP Inc",1425,2008-08-31,3003.93
"NNP Inc",1429,2008-09-01,2038.97
"NNP Inc",1435,2008-09-07,1648.57
"NNP Inc",1440,2008-09-07,2029.07
"NNP Inc",1443,2008-09-07,1732.09
"NNP Inc",1450,2008-09-08,6337.53
"NNP Inc",1452,2008-09-08,6296.74
"NNP Inc",1457,2008-09-16,6243.42
"NNP Inc",1463,2008-09-17,5443.44
"NNP Inc",1467,2008-09-17,1860.2
"NNP Inc",1475,2008-09-25,1027.52
"NNP Inc",1478,2008-09-26,1617.48
"NNP Inc",1481,2008-10-02,3384.54
"NNP Inc",1489,2008-10-10,4249.94
"NNP Inc",1491,2008-10-11,4705.09
"NNP Inc",1500,2008-10-11,1841.74
"NNP Inc",1502,2008-10-17,1044.47
"NNP Inc",1506,2008-10-23,3452.8
"NNP Inc",1512,2008-10-24,1282.87
"NNP Inc",1520,2008-10-24,5814.82
"NNP Inc",1524,2008-10-25,3546.19
"NNP Inc",1527,2008-10-26,3724.31
"NNP Inc",1535,2008-10-27,3660.48
"NNP Inc",1537,2008-11-04,1009.94
"NNP Inc",1541,2008-11-05,1047.99
"NNP Inc",1547,2008-11-11,1337.36
"NNP Inc",1551,2008-11-11,2799.42
"NNP Inc",1556,2008-11-12,1456.93
"NNP Inc",1561,2008-11-13,6251.31
"NNP Inc",1568,2008-11-14,1772.62
"NNP Inc",1573,2008-11-20,6710.49
"NNP Inc",1576,2008-11-20,3910.99
"NNP Inc",1583,2008-11-28,1178.0
"NNP Inc",1586,2008-11-29,4663.26
"NNP Inc",1592,2008-11-29,8675.88
"NNP Inc",1598,2008-11-29,6607.69
"NNP Inc",1602,2008-12-07,5280.0
"NNP Inc",1607,2008-12-13,1619.32
"NNP Inc",1612,2008-12-21,4127.24
"NNP Inc",1619,2008-12-31,4828.41
"NNP Inc",1624,2009-01-01,2635.01
"NNP Inc",1629,2009-01-07,1930.68
"NNP Inc",1632,2009-01-17,4570.23
"NNP Inc",1640,2009-01-23,1479.13
"NNP Inc",1644,2009-01-23,1576.94
"NNP Inc",1650,2009-01-24,8139.4
"NNP Inc",1655,2009-02-01,1401.78
"NNP Inc",1658,2009-02-01,5634.63
"NNP Inc",1665,2009-02-07,1092.85
"NNP Inc",1669,2009-02-13,2959.51
"NNP Inc",1674,2009-02-14,1890.51
"NNP Inc",1676,2009-02-14,3527.2
"NNP Inc",1684,2009-02-14,1620.22
"NNP Inc",1687,2009-02-15,2556.27
"NNP Inc",1692,2009-02-21,8971.69
"NNP Inc",1696,2009-02-21,1088.44
"NNP Inc",1701,2009-02-27,1412.03
"NNP Inc",1710,2009-02-28,1375.72
"NNP Inc",1712,2009-02-28,1763.92
"NNP Inc",1717,2009-03-01,4461.24
"NNP Inc",1724,2009-03-07,1393.16
"NNP Inc",1726,2009-03-08,8943.84
"NNP Inc",1732,2009-03-16,2884.95
"NNP Inc",1740,2009-03-16,1155.59
"NNP Inc",1744,2009-03-22,1421.73
"NNP Inc",1748,2009-03-23,7617.85
"NNP Inc",1753,2009-03-24,3264.21
"NNP Inc",1759,2009-03-24,4101.41
"NNP Inc",1765,2009-03-24,1589.17
"NNP Inc",1768,2009-03-24,4112.76
"NNP Inc",1771,2009-03-25,3899.26
"NNP Inc",1780,2009-03-31,1955.3
"NNP Inc",1785,2009-04-10,4197.52
"NNP Inc",1786,2009-04-18,3795.42
"NNP Inc",1794,2009-04-18,1293.95
"NNP Inc",1796,2009-04-24,9092.19
"NNP Inc",1801,2009-04-25,2946.64
"NNP Inc",1810,2009-04-26,9612.63
"NNP Inc",1813,2009-05-02,1287.49
"NNP Inc",1818,2009-05-12,7891.94
"NNP Inc",1825,2009-05-22,2630.71
"NNP Inc",1829,2009-05-22,2995.35
"NNP Inc",1833,2009-05-28,2671.7
"NNP Inc",1840,2009-06-03,2885.81
"NNP Inc",1842,2009-06-03,6727.71
"NNP Inc",1849,2009-06-03,2212.57
"NNP Inc",1855,2009-06-04,2793.66
"NNP Inc",1856,2009-06-05,4134.5
"NNP Inc",1865,2009-06-15,3087.47
"NNP Inc",1867,2009-06-15,1512.31
"NNP Inc",1873,2009-06-21,4667.94
"NNP Inc",1879,2009-07-01,2444.35
"NNP Inc",1884,2009-07-02,1104.9
"NNP Inc",1888,2009-07-03,1433.68
"NNP Inc",1892,2009-07-09,1384.07
"NNP Inc",1900,2009-07-09,6770.1
"NNP Inc",1903,2009-07-09,5419.77
"NNP Inc",1910,2009-07-09,4514.58
"NNP Inc",1914,2009-07-17,2101.55
"NNP Inc",1919,2009-07-18,9144.07
"NNP Inc",1921,2009-07-26,3350.14
"NNP Inc",1929,2009-08-01,1627.11
"NNP Inc",1934,2009-08-11,4440.56
"NNP Inc",1940,2009-08-11,2609.93
"NNP Inc",1944,2009-08-17,1613.37
"NNP Inc",1950,2009-08-17,1377.07
"NNP Inc",1955,2009-08-23,2569.25
"NNP Inc",1956,2009-08-23,2913.29
"NNP Inc",1965,2009-08-29,5643.34
"NNP Inc",1968,2009-08-30,5482.1
"NNP Inc",1974,2009-09-05,4320.64
"NNP Inc",1976,2009-09-06,1430.86
"NNP Inc",1981,2009-09-07,2112.63
"NNP Inc",1987,2009-09-13,9743.46
"NNP Inc",1994,2009-09-14,7387.8
"NNP Inc",2000,2009-09-24,1231.05
"NNP Inc",2004,2009-10-02,3196.23
"NNP Inc",2010,2009-10-02,2057.38
"NNP Inc",2011,2009-10-08,4798.81
"NNP Inc",2017,2009-10-08,1563.37
"NNP Inc",2022,2009-10-18,9941.24
"NNP Inc",2028,2009-10-18,1475.14
"NNP Inc",2035,2009-10-26,2459.1
"NNP Inc",2038,2009-10-26,2740.67
"NNP Inc",2042,2009-11-01,2637.16
"NNP Inc",2050,2009-11-02,8649.66
"NNP Inc",2055,2009-11-03,2074.83
"NNP Inc",2059,2009-11-04,4846.94
"NNP Inc",2064,2009-11-10,6443.69
"NNP Inc",2069,2009-11-16,2583.14
"NNP Inc",2075,2009-11-16,1701.27
"NNP Inc",2076,2009-11-16,1591.32
"NNP Inc",2084,2009-11-16,9346.58
"NNP Inc",2090,2009-11-17,1133.1
"NNP Inc",2092,2009-11-17,4617.47
"NNP Inc",2099,2009-11-17,1230.84
"NNP Inc",2104,2009-11-17,6612.13
"NNP Inc",2106,2009-11-17,4696.51
"NNP Inc",2114,2009-11-18,1870.31
"NNP Inc",2117,2009-11-18,4247.04
"NNP Inc",2124,2009-11-28,9911.74
"NNP Inc",2128,2009-11-29,1112.62
"NNP Inc",2135,2009-11-30,3778.31
"NNP Inc",2138,2009-12-01,4067.97
"NNP Inc",2144,2009-12-02,1318.44
"NNP Inc",2147,2009-12-02,6979.75
"NNP Inc",2154,2009-12-10,3565.75
"NNP Inc",2158,2009-12-10,9933.92
"NNP Inc",2163,2009-12-16,2317.49
"NNP Inc",2168,2009-12-22,2127.89
"NNP Inc",2172,2009-12-23,6413.99
"NNP Inc",2176,2009-12-23,9420.87
"NNP Inc",2181,2010-01-02,1882.21
"NNP Inc",2189,2010-01-02,1755.94
"NNP Inc",2195,2010-01-03,1347.65
"NNP Inc",2197,2010-01-11,2779.29
"NNP Inc",2204,2010-01-17,1044.83
"NNP Inc",2209,2010-01-18,2706.77
"NNP Inc",2212,2010-01-18,3651.3
"NNP Inc",2217,2010-01-18,5091.91
"NNP Inc",2224,2010-01-19,8357.93
"NNP Inc",2226,2010-01-25,3317.84
"NNP Inc",2233,2010-01-31,2241.0
"NNP Inc",2238,2010-02-01,5205.0
"NNP Inc",2243,2010-02-09,4031.89
"NNP Inc",2250,2010-02-10,7704.49
"NNP Inc",2253,2010-02-10,7856.97
"NNP Inc",2256,2010-02-11,4540.66
"NNP Inc",2263,2010-02-21,2375.62
"NNP Inc",2270,2010-02-27,5402.08
"NNP Inc",2273,2010-03-09,5459.11
"NNP Inc",2276,2010-03-10,4047.68
"NNP Inc",2285,2010-03-18,1854.53
"NNP Inc",2289,2010-03-19,8368.13
"NNP Inc",2294,2010-03-20,2339.74
"NNP Inc",2298,2010-03-20,9117.85
"NNP Inc",2304,2010-03-21,1846.86
"NNP Inc",2310,2010-03-27,1122.95
"NNP Inc",2315,2010-03-28,3043.55
"NNP Inc",2316,2010-03-28,1187.34
"NNP Inc",2325,2010-03-29,1027.51
"NNP Inc",2329,2010-03-30,4768.16
"NNP Inc",2332,2010-03-30,2438.12
"NNP Inc",2339,2010-04-05,2321.97
"NNP Inc",2345,2010-04-05,4470.78
"NNP Inc",2347,2010-04-11,2760.2
"NNP Inc",2353,2010-04-21,1275.14
"NNP Inc",2356,2010-04-22,9825.97
"NNP Inc",2362,2010-04-22,6704.79
"NNP Inc",2366,2010-04-22,2338.41
"NNP Inc",2375,2010-05-02,1372.47
"NNP Inc",2377,2010-05-03,1607.17
"NNP Inc",2383,2010-05-04,1296.47
"NNP Inc",2386,2010-05-10,3262.88
"NNP Inc",2393,2010-05-11,2156.51
"NNP Inc",2396,2010-05-11,3446.59
"NNP Inc",2404,2010-05-12,4104.44
"NNP Inc",2410,2010-05-18,1434.72
"NNP Inc",2412,2010-05-18,1994.01
"NNP Inc",2416,2010-05-19,9548.58
"NNP Inc",2422,2010-05-19,6063.07
"NNP Inc",2429,2010-05-20,1591.67
"NNP Inc",2432,2010-05-26,3721.66
"NNP Inc",2439,2010-05-27,1010.07
"NNP Inc",2443,2010-05-28,2786.65
"NNP Inc",2450,2010-05-29,2100.02
"NNP Inc",2452,2010-06-08,8919.14
"NNP Inc",2458,2010-06-18,7568.54
"NNP Inc",2462,2010-06-24,2043.38
"NNP Inc",2467,2010-06-25,6394.12
"NNP Inc",2472,2010-07-01,9118.69
"NNP Inc",2480,2010-07-02,8501.44
"NNP Inc",2481,2010-07-12,1795.17
"NNP Inc",2486,2010-07-20,4177.17
"NNP Inc",2495,2010-07-20,3400.76
"NNP Inc",2497,2010-07-20,4979.81
"NNP Inc",2504,2010-07-21,9332.28
"NNP Inc",2507,2010-07-29,1551.5
"NNP Inc",2514,2010-08-04,9351.16
"NNP Inc",2517,2010-08-10,5782.28
"NNP Inc",2525,2010-08-10,5082.06
"NNP Inc",2530,2010-08-11,2688.38
"NNP Inc",2535,2010-08-12,2481.4
"NNP Inc",2540,2010-08-13,3039.48
"NNP Inc",2543,2010-08-14,7095.79
"NNP Inc",2549,2010-08-15,6603.98
"NNP Inc",2552,2010-08-16,1347.27
"NNP Inc",2558,2010-08-26,2108.27
"NNP Inc",2564,2010-09-03,2339.66
"NNP Inc",2567,2010-09-11,3340.98
"NNP Inc",2574,2010-09-11,9971.86
"NNP Inc",2578,2010-09-19,3077.98
"NNP Inc",2585,2010-09-25,2301.21
"NNP Inc",2587,2010-09-25,4488.87
"NNP Inc",2592,2010-09-26,2498.36
"NNP Inc",2597,2010-09-27,7472.32
"NNP Inc",2603,2010-09-28,2934.1
"NNP Inc",2609,2010-10-06,4345.75
"NNP Inc",2612,2010-10-06,5522.15
"NNP Inc",2618,2010-10-07,1069.74
"NNP Inc",2623,2010-10-13,6538.99
"NNP Inc",2627,2010-10-21,4510.9
"NNP Inc",2634,2010-10-27,1853.74
"NNP Inc",2638,2010-11-04,1180.53
"NNP Inc",2642,2010-11-10,1043.79
"NNP Inc",2646,2010-11-10,2668.57
"NNP Inc",2653,2010-11-16,9896.08
"NNP Inc",2656,2010-11-16,2496.74
"NNP Inc",2663,2010-11-22,1007.18
"NNP Inc",2666,2010-12-02,1276.91
"NNP Inc",2672,2010-12-03,2048.2
"NNP Inc",2680,2010-12-13,6882.32
"NNP Inc",2681,2010-12-13,1537.48
"NNP Inc",2690,2010-12-14,1410.35
"NNP Inc",2693,2010-12-22,6599.82
"NNP Inc",2698,2010-12-28,2443.08
"NNP Inc",2701,2011-01-05,1028.25
"NNP Inc",2709,2011-01-05,1872.31
"NNP Inc",2715,2011-01-11,1205.44
"NNP Inc",2719,2011-01-11,9587.6
"NNP Inc",2725,2011-01-19,8854.51
"NNP Inc",2727,2011-01-19,3381.85
"NNP Inc",2733,2011-01-20,1335.86
"NNP Inc",2739,2011-01-20,2229.97
"NNP Inc",2741,2011-01-20,5703.6
"NNP Inc",2749,2011-01-21,6766.55
"NNP Inc",2755,2011-01-21,3861.39
"NNP Inc",2757,2011-01-21,1943.96
"NNP Inc",2761,2011-01-22,2429.67
"NNP Inc",2767,2011-01-28,3742.52
"NNP Inc",2774,2011-02-05,2029.68
"NNP Inc",2776,2011-02-06,6791.0
"NNP Inc",2781,2011-02-16,1967.61
"NNP Inc",2788,2011-02-17,1125.01
"NNP Inc",2792,2011-02-18,1798.43
"NNP Inc",2796,2011-02-18,5227.38
"NNP Inc",2804,2011-02-18,3427.61
"NNP Inc",2808,2011-02-26,9253.59
"NNP Inc",2814,2011-02-26,1229.52
"NNP Inc",2819,2011-03-04,3206.06
"NNP Inc",2825,2011-03-10,3249.22
"NNP Inc",2827,2011-03-10,1092.13
"NNP Inc",2835,2011-03-20,5035.98
"NNP Inc",2836,2011-03-21,2763.73
"NNP Inc",2844,2011-03-27,7781.65
"NNP Inc",2846,2011-03-28,8016.07
"NNP Inc",2852,2011-03-29,3935.96
"NNP Inc",2859,2011-03-29,3373.11
"NNP Inc",2862,2011-03-30,1933.52
"NNP Inc",2869,2011-03-31,3220.72
"NNP Inc",2873,2011-04-10,1160.03
"NNP Inc",2877,2011-04-18,8937.51
"NNP Inc",2883,2011-04-18,2538.69
"NNP Inc",2888,2011-04-26,1993.2
"NNP Inc",2893,2011-04-27,2445.14
"NNP Inc",2896,2011-04-27,7080.71
"NNP Inc",2905,2011-04-28,1095.04
"NNP Inc",2910,2011-04-28,1846.16
"NNP Inc",2913,2011-05-04,2718.15
"NNP Inc",2918,2011-05-05,1238.0
"NNP Inc",2924,2011-05-15,1357.41
"NNP Inc",2930,2011-05-25,5705.5
"NNP Inc",2932,2011-06-02,2144.91
"NNP Inc",2940,2011-06-02,4498.37
"NNP Inc",2945,2011-06-02,1216.43
"NNP Inc",2949,2011-06-03,2672.18
"NNP Inc",2951,2011-06-09,2489.96
"NNP Inc",2960,2011-06-10,8384.85
"NNP Inc",2964,2011-06-16,3259.83
"NNP Inc",2968,2011-06-17,8414.29
"NNP Inc",2971,2011-06-27,5733.47
"NNP Inc",2977,2011-06-27,1464.84
"NNP Inc",2981,2011-07-03,6401.71
"NNP Inc",2987,2011-07-04,1838.73
"NNP Inc",2995,2011-07-12,4935.13
"NNP Inc",2996,2011-07-13,9250.18
"NNP Inc",3004,2011-07-19,2256.32
"NNP Inc",3006,2011-07-25,1654.88
"NNP Inc",3014,2011-07-25,1884.39
"NNP Inc",3016,2011-07-26,5617.64
"NNP Inc",3024,2011-07-27,3473.42
"NNP Inc",3029,2011-08-04,2945.37
"NNP Inc",3033,2011-08-10,8731.74
"NNP Inc",3038,2011-08-11,8314.02
"NNP Inc",3043,2011-08-12,1029.76
"NNP Inc",3049,2011-08-18,3001.09
"NNP Inc",3052,2011-08-19,1438.81
"NNP Inc",3058,2011-08-25,3368.02
"NNP Inc",3061,2011-08-25,6096.2
"NNP Inc",3066,2011-08-25,2713.86
"NNP Inc",3075,2011-08-25,1499.02
"NNP Inc",3080,2011-08-31,6361.42
"NNP Inc",3084,2011-08-31,3513.7
"NNP Inc",3090,2011-08-31,1528.93
"NNP Inc",3093,2011-08-31,3589.17
"NNP Inc",3097,2011-09-08,1073.65
"NNP Inc",3104,2011-09-09,2513.54
"NNP Inc",3107,2011-09-10,2819.8
"NNP Inc",3114,2011-09-18,1454.62
"NNP Inc",3117,2011-09-19,3241.54
"NNP Inc",3122,2011-09-25,1406.27
"NNP Inc",3129,2011-10-03,1943.7
"NNP Inc",3133,2011-10-03,8224.01
"NNP Inc",3137,2011-10-09,1160.64
"NNP Inc",3141,2011-10-10,1672.59
"NNP Inc",3147,2011-10-11,2902.21
"NNP Inc",3155,2011-10-11,6593.32
"NNP Inc",3156,2011-10-12,7466.44
"NNP Inc",3162,2011-10-18,2172.76
"NNP Inc",3167,2011-10-19,3713.22
"NNP Inc",3172,2011-10-25,1548.19
"NNP Inc",3177,2011-10-25,2224.71
"NNP Inc",3181,2011-11-02,6283.59
"NNP Inc",3189,2011-11-02,7211.12
"NNP Inc",3193,2011-11-10,3779.1
"NNP Inc",3197,2011-11-10,6891.55
"NNP Inc",3203,2011-11-11,1417.66
"NNP Inc",3206,2011-11-11,1362.73
"NNP Inc",3212,2011-11-12,5523.31
"NNP Inc",3220,2011-11-22,1268.44
"NNP Inc",3221,2011-12-02,1494.95
"NNP Inc",3228,2011-12-02,5234.69
"NNP Inc",3233,2011-12-12,2471.55
"NNP Inc",3236,2011-12-18,5292.75
"NNP Inc",3244,2011-12-19,5737.81
"NNP Inc",3250,2011-12-29,1401.91
"NNP Inc",3253,2011-12-29,2056.69
"NNP Inc",3260,2012-01-06,4734.25
"NNP Inc",3265,2012-01-16,1203.43
"NNP Inc",3269,2012-01-22,4985.06
"NNP Inc",3271,2012-02-01,4617.8
"NNP Inc",3277,2012-02-07,6193.7
"NNP Inc",3284,2012-02-17,1529.5
"NNP Inc",3289,2012-02-27,2044.02
"NNP Inc",3292,2012-03-08,2387.43
"NNP Inc",3296,2012-03-09,3310.7
"NNP Inc",3301,2012-03-19,2207.94
"NNP Inc",3307,2012-03-19,1753.51
"NNP Inc",3314,2012-03-25,1155.55
"NNP Inc",3319,2012-03-26,8709.93
"NNP Inc",3321,2012-04-03,1881.73
"NNP Inc",3327,2012-04-04,1581.51
"NNP Inc",3334,2012-04-10,5863.27
"NNP Inc",3336,2012-04-16,3983.14
"NNP Inc",3345,2012-04-17,6843.21
"NNP Inc",3346,2012-04-17,2633.02
"NNP Inc",3354,2012-04-27,7193.38
"NNP Inc",3358,2012-05-03,2223.54
"NNP Inc",3364,2012-05-03,4232.7
"NNP Inc",3369,2012-05-04,3370.14
"NNP Inc",3371,2012-05-10,1001.51
"NNP Inc",3379,2012-05-20,1224.92
"NNP Inc",3385,2012-05-28,1605.87
"NNP Inc",3389,2012-05-29,9425.93
"NNP Inc",3395,2012-06-08,5833.3
"NNP Inc",3398,2012-06-14,1139.01
"NNP Inc",3405,2012-06-14,8382.44
"NNP Inc",3409,2012-06-14,3222.03
"NNP Inc",3414,2012-06-14,3106.16
"NNP Inc",3416,2012-06-14,9539.89
"NNP Inc",3422,2012-06-15,2339.69
"NNP Inc",3426,2012-06-15,4343.97
"NNP Inc",3433,2012-06-21,2108.34
"NNP Inc",3437,2012-06-22,1866.65
"NNP Inc",3445,2012-06-23,6859.46
"NNP Inc",3450,2012-07-03,1213.16
"NNP Inc",3455,2012-07-11,1677.79
"NNP Inc",3459,2012-07-12,1753.72
"NNP Inc",3465,2012-07-13,2656.73
"NNP Inc",3469,2012-07-13,4055.43
"NNP Inc",3472,2012-07-13,2112.56
"NNP Inc",3479,2012-07-19,3319.66
"NNP Inc",3481,2012-07-25,7520.68
"NNP Inc",3490,2012-08-02,1442.1
"NNP Inc",3494,2012-08-02,4723.18
"NNP Inc",3500,2012-08-02,5099.4
"NNP Inc",3505,2012-08-08,2727.47
"NNP Inc",3510,2012-08-09,1561.59
"NNP Inc",3513,2012-08-17,3083.49
"Vandeley Ind",10,2006-01-02,6180.26
"Vandeley Ind",13,2006-01-03,1070.31
"Vandeley Ind",17,2006-01-09,4621.69
"Vandeley Ind",23,2006-01-10,1763.76
"Vandeley Ind",29,2006-01-18,7637.12
"Vandeley Ind",34,2006-01-18,3089.71
"Vandeley Ind",38,2006-01-28,3600.46
"Vandeley Ind",45,2006-01-28,2633.42
"Vandeley Ind",48,2006-02-05,4145.15
"Vandeley Ind",53,2006-02-05,7401.4
"Vandeley Ind",58,2006-02-05,3344.65
"Vandeley Ind",64,2006-02-11,4342.06
"Vandeley Ind",66,2006-02-11,1257.33
"Vandeley Ind",72,2006-02-11,3112.58
"Vandeley Ind",79,2006-02-17,2806.49
"Vandeley Ind",85,2006-02-25,2727.75
"Vandeley Ind",88,2006-02-25,5793.65
"Vandeley Ind",93,2006-02-25,7645.88
"Vandeley Ind",98,2006-02-26,1165.67
"Vandeley Ind",104,2006-03-04,7782.2
"Vandeley Ind",106,2006-03-10,2578.81
"Vandeley Ind",112,2006-03-11,9748.69
"Vandeley Ind",120,2006-03-17,2851.22
"Vandeley Ind",122,2006-03-17,2694.83
"Vandeley Ind",128,2006-03-17,2852.42
"Vandeley Ind",131,2006-03-25,2568.52
"Vandeley Ind",136,2006-03-31,4200.36
"Vandeley Ind",144,2006-04-08,2486.03
"Vandeley Ind",150,2006-04-08,2081.16
"Vandeley Ind",152,2006-04-09,7303.37
"Vandeley Ind",160,2006-04-09,1044.87
"Vandeley Ind",164,2006-04-10,1538.29
"Vandeley Ind",167,2006-04-10,4142.71
"Vandeley Ind",173,2006-04-11,3355.83
"Vandeley Ind",179,2006-04-12,4609.13
"Vandeley Ind",184,2006-04-12,3205.42
"Vandeley Ind",189,2006-04-12,2619.51
"Vandeley Ind",191,2006-04-13,8693.35
"Vandeley Ind",200,2006-04-13,3204.12
"Vandeley Ind",201,2006-04-14,8576.92
"Vandeley Ind",210,2006-04-15,1739.07
"Vandeley Ind",213,2006-04-21,1504.14
"Vandeley Ind",216,2006-04-27,1914.17
"Vandeley Ind",221,2006-05-05,1333.68
"Vandeley Ind",229,2006-05-11,2472.92
"Vandeley Ind",235,2006-05-11,1740.59
"Vandeley Ind",236,2006-05-11,2340.63
"Vandeley Ind",243,2006-05-11,1122.94
"Vandeley Ind",247,2006-05-12,1347.08
"Vandeley Ind",252,2006-05-18,2163.64
"Vandeley Ind",260,2006-05-24,2656.06
"Vandeley Ind",261,2006-05-25,5393.05
"Vandeley Ind",266,2006-05-26,5143.75
"Vandeley Ind",275,2006-05-26,3170.15
"Vandeley Ind",276,2006-06-05,7173.76
"Vandeley Ind",285,2006-06-05,4178.64
"Vandeley Ind",290,2006-06-05,1589.88
"Vandeley Ind",291,2006-06-11,2059.98
"Vandeley Ind",297,2006-06-19,1661.74
"Vandeley Ind",302,2006-06-19,1016.64
"Vandeley Ind",310,2006-06-29,1502.87
"Vandeley Ind",315,2006-07-07,3635.19
"Vandeley Ind",320,2006-07-07,3732.9
"Vandeley Ind",325,2006-07-08,8412.76
"Vandeley Ind",329,2006-07-16,2611.37
"Vandeley Ind",335,2006-07-16,1675.64
"Vandeley Ind",340,2006-07-16,9646.41
"Vandeley Ind",342,2006-07-24,3117.87
"Vandeley Ind",347,2006-07-25,1834.7
"Vandeley Ind",352,2006-07-25,7136.51
"Vandeley Ind",356,2006-08-04,3648.03
"Vandeley Ind",364,2006-08-05,1438.34
"Vandeley Ind",368,2006-08-06,2404.35
"Vandeley Ind",374,2006-08-12,1188.57
"Vandeley Ind",378,2006-08-20,2005.75
"Vandeley Ind",381,2006-08-26,1757.4
"Vandeley Ind",388,2006-08-27,8342.28
"Vandeley Ind",392,2006-08-27,1425.57
"Vandeley Ind",400,2006-08-27,1252.78
"Vandeley Ind",403,2006-08-27,3552.12
"Vandeley Ind",409,2006-08-27,4032.86
"Vandeley Ind",415,2006-09-02,1170.85
"Vandeley Ind",420,2006-09-02,2312.91
"Vandeley Ind",424,2006-09-02,3123.47
"Vandeley Ind",428,2006-09-02,3117.07
"Vandeley Ind",435,2006-09-03,1594.78
"Vandeley Ind",439,2006-09-11,4044.05
"Vandeley Ind",443,2006-09-11,3137.97
"Vandeley Ind",449,2006-09-17,6515.8
"Vandeley Ind",455,2006-09-23,4944.54
"Vandeley Ind",457,2006-09-29,2789.98
"Vandeley Ind",462,2006-09-30,1518.61
"Vandeley Ind",468,2006-10-10,3402.66
"Vandeley Ind",471,2006-10-10,3356.5
"Vandeley Ind",477,2006-10-16,2635.81
"Vandeley Ind",482,2006-10-17,4986.4
"Vandeley Ind",488,2006-10-17,2712.23
"Vandeley Ind",492,2006-10-23,1335.51
"Vandeley Ind",500,2006-10-29,5089.02
"Vandeley Ind",504,2006-11-06,3298.03
"Vandeley Ind",507,2006-11-14,1145.37
"Vandeley Ind",514,2006-11-22,2728.89
"Vandeley Ind",516,2006-11-28,2039.44
"Vandeley Ind",525,2006-12-04,2798.96
"Vandeley Ind",528,2006-12-04,7469.9
"Vandeley Ind",532,2006-12-10,1214.39
"Vandeley Ind",538,2006-12-16,1047.29
"Vandeley Ind",542,2006-12-16,7190.95
"Vandeley Ind",549,2006-12-16,1178.85
"Vandeley Ind",555,2006-12-17,1480.41
"Vandeley Ind",560,2006-12-18,2177.4
"Vandeley Ind",562,2006-12-28,1092.53
"Vandeley Ind",567,2007-01-03,5292.84
"Vandeley Ind",572,2007-01-09,8664.41
"Vandeley Ind",579,2007-01-15,4401.87
"Vandeley Ind",583,2007-01-15,1777.78
"Vandeley Ind",589,2007-01-23,4599.37
"Vandeley Ind",593,2007-01-29,1094.42
"Vandeley Ind",599,2007-01-29,4495.07
"Vandeley Ind",604,2007-01-29,1754.78
"Vandeley Ind",610,2007-01-30,3419.03
"Vandeley Ind",614,2007-01-31,2106.02
"Vandeley Ind",620,2007-01-31,4903.8
"Vandeley Ind",623,2007-01-31,1307.67
"Vandeley Ind",629,2007-02-01,4668.78
"Vandeley Ind",633,2007-02-01,5731.28
"Vandeley Ind",637,2007-02-07,1308.42
"Vandeley Ind",645,2007-02-13,5037.64
"Vandeley Ind",650,2007-02-14,1708.27
"Vandeley Ind",655,2007-02-14,1827.25
"Vandeley Ind",656,2007-02-15,9706.64
"Vandeley Ind",662,2007-02-23,1315.6
"Vandeley Ind",669,2007-03-01,3141.82
"Vandeley Ind",672,2007-03-01,3385.62
"Vandeley Ind",676,2007-03-09,1048.39
"Vandeley Ind",684,2007-03-15,1917.3
"Vandeley Ind",690,2007-03-16,1735.99
"Vandeley Ind",692,2007-03-16,7142.99
"Vandeley Ind",698,2007-03-17,1342.57
"Vandeley Ind",701,2007-03-18,5942.42
"Vandeley Ind",710,2007-03-24,7527.75
"Vandeley Ind",715,2007-03-24,2654.76
"Vandeley Ind",719,2007-03-25,6943.81
"Vandeley Ind",724,2007-04-04,3720.36
"Vandeley Ind",729,2007-04-12,3085.09
"Vandeley Ind",734,2007-04-22,6991.97
"Vandeley Ind",740,2007-04-23,5831.34
"Vandeley Ind",744,2007-04-23,3855.3
"Vandeley Ind",747,2007-05-01,2000.07
"Vandeley Ind",751,2007-05-01,5143.36
"Vandeley Ind",756,2007-05-07,7630.44
"Vandeley Ind",764,2007-05-07,2438.97
"Vandeley Ind",768,2007-05-07,9085.99
"Vandeley Ind",773,2007-05-07,7122.76
"Vandeley Ind",780,2007-05-08,2461.12
"Vandeley Ind",784,2007-05-16,3433.95
"Vandeley Ind",790,2007-05-24,1342.71
"Vandeley Ind",795,2007-05-30,5735.91
"Vandeley Ind",796,2007-06-05,2063.7
"Vandeley Ind",804,2007-06-06,2716.7
"Vandeley Ind",809,2007-06-12,3582.03
"Vandeley Ind",815,2007-06-13,3151.21
"Vandeley Ind",816,2007-06-13,2558.26
"Vandeley Ind",821,2007-06-14,3598.16
"Vandeley Ind",829,2007-06-14,8018.7
"Vandeley Ind",835,2007-06-15,3703.72
"Vandeley Ind",837,2007-06-15,8122.82
"Vandeley Ind",844,2007-06-16,4934.72
"Vandeley Ind",849,2007-06-22,6040.77
"Vandeley Ind",851,2007-06-22,1073.87
"Vandeley Ind",860,2007-06-22,1769.52
"Vandeley Ind",862,2007-06-23,5637.42
"Vandeley Ind",866,2007-06-23,5513.85
"Vandeley Ind",872,2007-06-24,8639.48
"Vandeley Ind",877,2007-06-30,1161.09
"Vandeley Ind",884,2007-07-06,1483.25
"Vandeley Ind",889,2007-07-12,3725.42
"Vandeley Ind",893,2007-07-13,2252.76
"Vandeley Ind",900,2007-07-14,2217.14
"Vandeley Ind",901,2007-07-14,2619.25
"Vandeley Ind",908,2007-07-24,6229.48
"Vandeley Ind",912,2007-07-30,1507.07
"Vandeley Ind",917,2007-08-09,1844.83
"Vandeley Ind",925,2007-08-19,1328.91
"Vandeley Ind",928,2007-08-20,2298.12
"Vandeley Ind",933,2007-08-21,9731.75
"Vandeley Ind",939,2007-08-27,1550.9
"Vandeley Ind",943,2007-09-06,8491.28
"Vandeley Ind",948,2007-09-06,3899.55
"Vandeley Ind",953,2007-09-06,1463.21
"Vandeley Ind",960,2007-09-06,3811.42
"Vandeley Ind",964,2007-09-07,3932.24
"Vandeley Ind",966,2007-09-07,9498.3
"Vandeley Ind",975,2007-09-17,3769.6
"Vandeley Ind",978,2007-09-17,1354.79
"Vandeley Ind",982,2007-09-25,2821.23
"Vandeley Ind",989,2007-10-01,3340.46
"Vandeley Ind",992,2007-10-07,6601.52
"Vandeley Ind",997,2007-10-13,8737.4
"Vandeley Ind",1002,2007-10-23,1459.85
"Vandeley Ind",1006,2007-10-23,4445.81
"Vandeley Ind",1014,2007-10-23,8279.43
"Vandeley Ind",1016,2007-10-29,5721.45
"Vandeley Ind",1025,2007-11-04,7306.35
"Vandeley Ind",1030,2007-11-05,5042.71
"Vandeley Ind",1034,2007-11-11,1917.0
"Vandeley Ind",1039,2007-11-12,1566.55
"Vandeley Ind",1045,2007-11-12,1354.48
"Vandeley Ind",1050,2007-11-22,6148.91
"Vandeley Ind",1055,2007-11-22,3722.95
"Vandeley Ind",1059,2007-11-23,8084.25
"Vandeley Ind",1062,2007-11-24,4661.91
"Vandeley Ind",1067,2007-11-30,7051.9
"Vandeley Ind",1072,2007-11-30,1711.84
"Vandeley Ind",1080,2007-12-06,1811.77
"Vandeley Ind",1084,2007-12-07,1669.66
"Vandeley Ind",1089,2007-12-15,1755.15
"Vandeley Ind",1094,2007-12-16,1209.8
"Vandeley Ind",1100,2007-12-16,1780.71
"Vandeley Ind",1102,2007-12-17,1133.33
"Vandeley Ind",1110,2007-12-18,1396.83
"Vandeley Ind",1111,2007-12-18,1895.38
"Vandeley Ind",1116,2007-12-18,1791.16
"Vandeley Ind",1124,2007-12-24,7857.6
"Vandeley Ind",1127,2008-01-03,3471.32
"Vandeley Ind",1132,2008-01-13,2782.22
"Vandeley Ind",1138,2008-01-13,2296.75
"Vandeley Ind",1144,2008-01-19,1064.05
"Vandeley Ind",1147,2008-01-29,3632.33
"Vandeley Ind",1151,2008-02-04,1043.25
"Vandeley Ind",1158,2008-02-04,1618.44
"Vandeley Ind",1163,2008-02-10,6080.01
"Vandeley Ind",1167,2008-02-18,6445.62
"Vandeley Ind",1175,2008-02-18,1039.86
"Vandeley Ind",1176,2008-02-28,6389.11
"Vandeley Ind",1185,2008-03-05,1273.7
"Vandeley Ind",1189,2008-03-05,2054.0
"Vandeley Ind",1192,2008-03-06,2752.57
"Vandeley Ind",1199,2008-03-14,5319.78
"Vandeley Ind",1202,2008-03-20,1154.78
"Vandeley Ind",1206,2008-03-20,1285.5
"Vandeley Ind",1214,2008-03-20,3211.01
"Vandeley Ind",1218,2008-03-21,1302.06
"Vandeley Ind",1224,2008-03-27,5779.81
"Vandeley Ind",1230,2008-03-27,1025.35
"Vandeley Ind",1235,2008-03-28,6698.29
"Vandeley Ind",1236,2008-04-05,3029.68
"Vandeley Ind",1242,2008-04-11,2145.12
"Vandeley Ind",1246,2008-04-17,1561.04
"Vandeley Ind",1251,2008-04-18,2829.2
"Vandeley Ind",1256,2008-04-18,4115.58
"Vandeley Ind",1261,2008-04-24,4618.26
"Vandeley Ind",1267,2008-04-24,2366.29
"Vandeley Ind",1271,2008-04-24,5510.07
"Vandeley Ind",1276,2008-04-30,3273.86
"Vandeley Ind",1285,2008-05-10,5478.23
"Vandeley Ind",1290,2008-05-20,4445.57
"Vandeley Ind",1295,2008-05-21,1101.28
"Vandeley Ind",1299,2008-05-22,3979.81
"Vandeley Ind",1302,2008-05-22,5420.3
"Vandeley Ind",1310,2008-05-22,4155.53
"Vandeley Ind",1314,2008-05-28,3358.3
"Vandeley Ind",1320,2008-05-29,1672.32
"Vandeley Ind",1324,2008-05-30,4112.75
"Vandeley Ind",1329,2008-06-05,3846.36
"Vandeley Ind",1333,2008-06-05,6619.21
"Vandeley Ind",1337,2008-06-06,9846.9
"Vandeley Ind",1345,2008-06-12,4404.16
"Vandeley Ind",1347,2008-06-12,9586.62
"Vandeley Ind",1352,2008-06-18,2473.0
"Vandeley Ind",1357,2008-06-18,1528.39
"Vandeley Ind",1364,2008-06-28,2712.44
"Vandeley Ind",1367,2008-06-29,3896.18
"Vandeley Ind",1374,2008-06-29,1974.23
"Vandeley Ind",1380,2008-06-29,5857.41
"Vandeley Ind",1381,2008-06-30,2661.65
"Vandeley Ind",1386,2008-06-30,4193.56
"Vandeley Ind",1392,2008-07-10,4827.43
"Vandeley Ind",1400,2008-07-20,7917.79
"Vandeley Ind",1401,2008-07-30,2207.43
"Vandeley Ind",1410,2008-08-07,1063.04
"Vandeley Ind",1413,2008-08-13,6501.25
"Vandeley Ind",1419,2008-08-19,1637.07
"Vandeley Ind",1423,2008-08-19,9445.73
"Vandeley Ind",1427,2008-08-20,3472.68
"Vandeley Ind",1432,2008-08-20,5199.46
"Vandeley Ind",1436,2008-08-20,4652.45
"Vandeley Ind",1443,2008-08-26,1032.06
"Vandeley Ind",1448,2008-08-27,1774.19
"Vandeley Ind",1454,2008-09-02,3813.92
"Vandeley Ind",1459,2008-09-12,2435.19
"Vandeley Ind",1464,2008-09-20,1924.1
"Vandeley Ind",1470,2008-09-26,3478.76
"Vandeley Ind",1472,2008-09-27,1094.97
"Vandeley Ind",1480,2008-09-28,2955.24
"Vandeley Ind",1481,2008-10-08,2176.86
"Vandeley Ind",1489,2008-10-16,2236.84
"Vandeley Ind",1491,2008-10-22,7119.55
"Vandeley Ind",1498,2008-10-22,1695.56
"Vandeley Ind",1503,2008-10-23,1033.2
"Vandeley Ind",1506,2008-10-29,1671.45
"Vandeley Ind",1511,2008-11-04,1568.36
"Vandeley Ind",1516,2008-11-04,1683.53
"Vandeley Ind",1525,2008-11-04,7328.33
"Vandeley Ind",1529,2008-11-10,2309.53
"Vandeley Ind",1535,2008-11-16,2851.59
"Vandeley Ind",1537,2008-11-22,1595.87
"Vandeley Ind",1544,2008-11-23,1202.67
"Vandeley Ind",1547,2008-12-01,1783.11
"Vandeley Ind",1553,2008-12-01,2276.36
"Vandeley Ind",1560,2008-12-11,2709.28
"Vandeley Ind",1565,2008-12-11,3505.3
"Vandeley Ind",1569,2008-12-11,4160.54
"Vandeley Ind",1573,2008-12-12,2361.74
"Vandeley Ind",1580,2008-12-12,1974.88
"Vandeley Ind",1585,2008-12-18,7351.98
"Vandeley Ind",1587,2008-12-24,4609.48
"Vandeley Ind",1593,2008-12-30,7717.35
"Vandeley Ind",1596,2009-01-09,6099.75
"Vandeley Ind",1602,2009-01-15,6070.91
"Vandeley Ind",1606,2009-01-16,1534.55
"Vandeley Ind",1612,2009-01-16,3636.16
"Vandeley Ind",1619,2009-01-17,3148.39
"Vandeley Ind",1622,2009-01-27,2851.15
"Vandeley Ind",1628,2009-02-04,1460.19
"Vandeley Ind",1635,2009-02-04,6003.94
"Vandeley Ind",1640,2009-02-04,1869.02
"Vandeley Ind",1641,2009-02-05,2771.52
"Vandeley Ind",1648,2009-02-06,8660.33
"Vandeley Ind",1653,2009-02-16,8248.84
"Vandeley Ind",1656,2009-02-22,2406.14
"Vandeley Ind",1662,2009-02-22,3492.82
"Vandeley Ind",1669,2009-03-04,1374.94
"Vandeley Ind",1675,2009-03-14,1105.86
"Vandeley Ind",1679,2009-03-15,1432.84
"Vandeley Ind",1681,2009-03-23,3946.36
"Vandeley Ind",1689,2009-03-23,4990.02
"Vandeley Ind",1692,2009-03-29,1104.57
"Vandeley Ind",1698,2009-04-08,3458.32
"Vandeley Ind",1705,2009-04-08,7337.9
"Vandeley Ind",1709,2009-04-08,8810.56
"Vandeley Ind",1713,2009-04-09,3937.37
"Vandeley Ind",1716,2009-04-19,4163.95
"Vandeley Ind",1721,2009-04-29,2102.08
"Vandeley Ind",1729,2009-05-05,1415.48
"Vandeley Ind",1734,2009-05-11,4097.17
"Vandeley Ind",1739,2009-05-21,1447.31
"Vandeley Ind",1741,2009-05-22,3863.03
"Vandeley Ind",1746,2009-05-28,1078.06
"Vandeley Ind",1751,2009-06-03,5877.56
"Vandeley Ind",1758,2009-06-03,2680.51
"Vandeley Ind",1763,2009-06-03,6723.6
"Vandeley Ind",1770,2009-06-09,8882.73
"Vandeley Ind",1774,2009-06-09,2334.37
"Vandeley Ind",1776,2009-06-15,1334.99
"Vandeley Ind",1783,2009-06-16,1809.18
"Vandeley Ind",1787,2009-06-17,7017.29
"Vandeley Ind",1794,2009-06-17,2136.35
"Vandeley Ind",1800,2009-06-17,4324.94
"Vandeley Ind",1803,2009-06-18,1151.83
"Vandeley Ind",1806,2009-06-19,5330.79
"Vandeley Ind",1813,2009-06-20,3341.18
"Vandeley Ind",1816,2009-06-26,1430.07
"Vandeley Ind",1825,2009-06-27,7742.95
"Vandeley Ind",1826,2009-07-07,3835.32
"Vandeley Ind",1834,2009-07-08,4256.51
"Vandeley Ind",1836,2009-07-18,2410.01
"Vandeley Ind",1841,2009-07-26,1310.4
"Vandeley Ind",1850,2009-07-26,1554.35
"Vandeley Ind",1854,2009-08-01,1090.83
"Vandeley Ind",1857,2009-08-02,7715.21
"Vandeley Ind",1865,2009-08-02,8259.34
"Vandeley Ind",1869,2009-08-10,2125.62
"Vandeley Ind",1873,2009-08-10,6005.37
"Vandeley Ind",1878,2009-08-11,3287.02
"Vandeley Ind",1881,2009-08-21,3578.33
"Vandeley Ind",1889,2009-08-22,2504.75
"Vandeley Ind",1892,2009-08-30,1072.95
"Vandeley Ind",1897,2009-08-30,8674.11
"Vandeley Ind",1902,2009-09-05,3046.89
"Vandeley Ind",1906,2009-09-06,5403.92
"Vandeley Ind",1914,2009-09-06,7020.35
"Vandeley Ind",1918,2009-09-12,2171.34
"Vandeley Ind",1924,2009-09-20,8544.08
"Vandeley Ind",1929,2009-09-21,5124.69
"Vandeley Ind",1934,2009-09-21,1494.28
"Vandeley Ind",1940,2009-09-21,3100.83
"Vandeley Ind",1941,2009-09-29,6953.57
"Vandeley Ind",1950,2009-10-05,2791.51
"Vandeley Ind",1952,2009-10-06,6425.65
"Vandeley Ind",1960,2009-10-12,1767.84
"Vandeley Ind",1963,2009-10-13,2644.03
"Vandeley Ind",1967,2009-10-14,1272.88
"Vandeley Ind",1975,2009-10-20,1858.08
"Vandeley Ind",1976,2009-10-20,4106.83
"Vandeley Ind",1982,2009-10-21,1339.09
"Vandeley Ind",1986,2009-10-22,3327.94
"Vandeley Ind",1993,2009-10-22,1041.71
"Vandeley Ind",1997,2009-10-22,4067.99
"Vandeley Ind",2003,2009-10-22,5816.46
"Vandeley Ind",2009,2009-10-28,8035.11
"Vandeley Ind",2015,2009-11-03,1726.48
"Vandeley Ind",2020,2009-11-03,6691.58
"Vandeley Ind",2024,2009-11-13,1740.29
"Vandeley Ind",2030,2009-11-13,5938.0
"Vandeley Ind",2031,2009-11-14,1977.12
"Vandeley Ind",2037,2009-11-22,1013.05
"Vandeley Ind",2043,2009-11-22,4456.49
"Vandeley Ind",2050,2009-11-23,5938.83
"Vandeley Ind",2051,2009-11-24,1542.49
"Vandeley Ind",2057,2009-11-24,5939.79
"Vandeley Ind",2061,2009-11-24,1734.97
"Vandeley Ind",2067,2009-11-25,6295.99
"Vandeley Ind",2071,2009-11-25,1272.75
"Vandeley Ind",2080,2009-11-26,1051.44
"Vandeley Ind",2082,2009-12-02,9783.31
"Vandeley Ind",2090,2009-12-12,6743.82
"Vandeley Ind",2092,2009-12-22,2176.28
"Vandeley Ind",2100,2009-12-22,8678.79
"Vandeley Ind",2102,2009-12-23,1548.99
"Vandeley Ind",2109,2009-12-31,1120.69
"Vandeley Ind",2113,2010-01-01,1996.21
"Vandeley Ind",2117,2010-01-01,1732.36
"Vandeley Ind",2124,2010-01-07,2580.13
"Vandeley Ind",2126,2010-01-13,1060.08
"Vandeley Ind",2133,2010-01-13,3408.3
"Vandeley Ind",2140,2010-01-21,2385.03
"Vandeley Ind",2145,2010-01-27,6142.16
"Vandeley Ind",2150,2010-02-06,1044.09
"Vandeley Ind",2155,2010-02-14,1961.4
"Vandeley Ind",2157,2010-02-15,1386.59
"Vandeley Ind",2165,2010-02-15,2402.51
"Vandeley Ind",2168,2010-02-16,1614.55
"Vandeley Ind",2175,2010-02-17,3342.76
"Vandeley Ind",2180,2010-02-18,2760.06
"Vandeley Ind",2185,2010-02-24,4932.16
"Vandeley Ind",2188,2010-02-25,3170.44
"Vandeley Ind",2194,2010-02-26,3657.47
"Vandeley Ind",2197,2010-02-26,7532.74
"Vandeley Ind",2204,2010-02-26,3391.95
"Vandeley Ind",2210,2010-02-27,7688.59
"Vandeley Ind",2215,2010-02-27,1706.03
"Vandeley Ind",2219,2010-03-05,7707.27
"Vandeley Ind",2221,2010-03-06,1062.27
"Vandeley Ind",2226,2010-03-07,2165.38
"Vandeley Ind",2232,2010-03-15,4903.44
"Vandeley Ind",2239,2010-03-15,2023.96
"Vandeley Ind",2243,2010-03-21,3920.06
"Vandeley Ind",2246,2010-03-21,2553.27
"Vandeley Ind",2254,2010-03-27,3838.05
"Vandeley Ind",2259,2010-03-27,2833.33
"Vandeley Ind",2264,2010-03-27,3043.36
"Vandeley Ind",2266,2010-04-04,2665.41
"Vandeley Ind",2273,2010-04-10,1767.09
"Vandeley Ind",2279,2010-04-16,7063.85
"Vandeley Ind",2282,2010-04-22,8620.65
"Vandeley Ind",2290,2010-04-30,2917.2
"Vandeley Ind",2295,2010-05-01,1193.5
"Vandeley Ind",2299,2010-05-11,1609.93
"Vandeley Ind",2305,2010-05-12,7698.96
"Vandeley Ind",2309,2010-05-18,5718.9
"Vandeley Ind",2314,2010-05-18,1751.51
"Vandeley Ind",2319,2010-05-19,2134.58
"Vandeley Ind",2325,2010-05-19,5050.44
"Vandeley Ind",2326,2010-05-29,1039.59
"Vandeley Ind",2332,2010-06-08,8221.61
"Vandeley Ind",2338,2010-06-08,4209.22
"Vandeley Ind",2342,2010-06-14,1182.94
"Vandeley Ind",2346,2010-06-22,5734.28
"Vandeley Ind",2354,2010-06-28,3530.47
"Vandeley Ind",2357,2010-07-08,2512.23
"Vandeley Ind",2365,2010-07-08,2737.77
"Vandeley Ind",2368,2010-07-08,2745.39
"Vandeley Ind",2373,2010-07-14,6913.86
"Vandeley Ind",2378,2010-07-20,1606.4
"Vandeley Ind",2385,2010-07-26,4607.61
"Vandeley Ind",2388,2010-07-26,9773.25
"Vandeley Ind",2394,2010-08-01,8305.82
"Vandeley Ind",2398,2010-08-02,2786.34
"Vandeley Ind",2402,2010-08-03,7077.85
"Vandeley Ind",2406,2010-08-09,4547.28
"Vandeley Ind",2413,2010-08-09,1892.07
"Vandeley Ind",2416,2010-08-17,7249.03
"Vandeley Ind",2423,2010-08-25,1198.21
"Vandeley Ind",2430,2010-08-31,7465.42
"Vandeley Ind",2435,2010-09-10,1195.37
"Vandeley Ind",2439,2010-09-10,5249.34
"Vandeley Ind",2441,2010-09-20,1201.93
"Vandeley Ind",2447,2010-09-20,3617.08
"Vandeley Ind",2453,2010-09-21,1479.46
"Vandeley Ind",2460,2010-09-22,8300.4
"Vandeley Ind",2465,2010-09-23,3551.12
"Vandeley Ind",2467,2010-09-23,4042.1
"Vandeley Ind",2472,2010-09-23,3816.44
"Vandeley Ind",2477,2010-09-24,5754.03
"Vandeley Ind",2485,2010-09-25,2156.83
"Vandeley Ind",2490,2010-10-01,9675.4
"Vandeley Ind",2491,2010-10-11,2452.58
"Vandeley Ind",2498,2010-10-17,4554.99
"Vandeley Ind",2501,2010-10-27,4755.92
"Vandeley Ind",2507,2010-10-28,8807.02
"Vandeley Ind",2513,2010-10-28,1684.4
"Vandeley Ind",2518,2010-10-28,2668.4
"Vandeley Ind",2524,2010-10-28,9043.92
"Vandeley Ind",2528,2010-10-28,1525.4
"Vandeley Ind",2535,2010-11-07,8693.88
"Vandeley Ind",2539,2010-11-07,4391.27
"Vandeley Ind",2542,2010-11-08,1320.95
"Vandeley Ind",2547,2010-11-14,4357.89
"Vandeley Ind",2555,2010-11-14,4212.33
"Vandeley Ind",2556,2010-11-14,1380.03
"Vandeley Ind",2563,2010-11-24,4106.09
"Vandeley Ind",2566,2010-11-24,6554.74
"Vandeley Ind",2572,2010-12-02,2659.69
"Vandeley Ind",2577,2010-12-08,7889.07
"Vandeley Ind",2584,2010-12-14,1660.07
"Vandeley Ind",2587,2010-12-15,3013.77
"Vandeley Ind",2594,2010-12-16,2310.55
"Vandeley Ind",2599,2010-12-17,6746.49
"Vandeley Ind",2604,2010-12-27,3377.25
"Vandeley Ind",2610,2010-12-27,6721.08
"Vandeley Ind",2611,2010-12-27,4157.97
"Vandeley Ind",2618,2011-01-02,2670.07
"Vandeley Ind",2622,2011-01-03,1515.45
"Vandeley Ind",2626,2011-01-03,4543.5
"Vandeley Ind",2635,2011-01-04,7607.26
"Vandeley Ind",2640,2011-01-05,1137.12
"Vandeley Ind",2643,2011-01-13,5131.92
"Vandeley Ind",2649,2011-01-23,9175.74
"Vandeley Ind",2653,2011-01-23,2470.49
"Vandeley Ind",2659,2011-01-24,3363.67
"Vandeley Ind",2661,2011-01-30,1654.58
"Vandeley Ind",2670,2011-02-07,1139.21
"Vandeley Ind",2672,2011-02-07,4542.07
"Vandeley Ind",2678,2011-02-08,2878.52
"Vandeley Ind",2685,2011-02-08,5983.14
"Vandeley Ind",2688,2011-02-08,1011.83
"Vandeley Ind",2695,2011-02-18,2512.34
"Vandeley Ind",2700,2011-02-19,1405.6
"Vandeley Ind",2705,2011-02-19,3240.1
"Vandeley Ind",2707,2011-02-25,1921.47
"Vandeley Ind",2715,2011-02-26,8140.82
"Vandeley Ind",2720,2011-02-27,9045.82
"Vandeley Ind",2725,2011-02-27,2911.14
"Vandeley Ind",2728,2011-03-07,7770.01
"Vandeley Ind",2735,2011-03-17,2055.52
"Vandeley Ind",2737,2011-03-25,6672.71
"Vandeley Ind",2741,2011-04-02,3060.89
"Vandeley Ind",2746,2011-04-10,1142.21
"Vandeley Ind",2752,2011-04-18,1211.66
"Vandeley Ind",2758,2011-04-24,6197.84
"Vandeley Ind",2764,2011-04-24,7317.67
"Vandeley Ind",2767,2011-05-02,5530.12
"Vandeley Ind",2773,2011-05-03,2291.36
"Vandeley Ind",2777,2011-05-03,1348.52
"Vandeley Ind",2782,2011-05-03,5397.1
"Vandeley Ind",2789,2011-05-09,5213.71
"Vandeley Ind",2792,2011-05-09,1279.74
"Vandeley Ind",2800,2011-05-15,2948.66
"Vandeley Ind",2805,2011-05-15,1761.29
"Vandeley Ind",2808,2011-05-16,7806.81
"Vandeley Ind",2812,2011-05-17,1627.37
"Vandeley Ind",2818,2011-05-25,2518.54
"Vandeley Ind",2823,2011-05-31,3783.48
"Vandeley Ind",2830,2011-06-06,2036.59
"Vandeley Ind",2831,2011-06-14,5443.18
"Vandeley Ind",2836,2011-06-20,1452.23
"Vandeley Ind",2843,2011-06-20,1888.65
"Vandeley Ind",2850,2011-06-30,1438.48
"Vandeley Ind",2851,2011-07-01,1480.0
"Vandeley Ind",2858,2011-07-09,6313.36
"Vandeley Ind",2862,2011-07-09,5186.61
"Vandeley Ind",2867,2011-07-15,5421.25
"Vandeley Ind",2871,2011-07-16,4246.77
"Vandeley Ind",2876,2011-07-24,6802.23
"Vandeley Ind",2884,2011-07-25,1690.2
"Vandeley Ind",2886,2011-07-25,6065.99
"Vandeley Ind",2891,2011-07-26,1522.94
"Vandeley Ind",2898,2011-07-27,9807.24
"Vandeley Ind",2901,2011-08-04,2362.14
"Vandeley Ind",2909,2011-08-04,1896.32
"Vandeley Ind",2913,2011-08-05,1326.95
"Vandeley Ind",2917,2011-08-05,2445.66
"Vandeley Ind",2925,2011-08-06,1426.56
"Vandeley Ind",2929,2011-08-07,2603.82
"Vandeley Ind",2931,2011-08-07,3469.28
"Vandeley Ind",2937,2011-08-13,2497.29
"Vandeley Ind",2942,2011-08-14,8403.77
"Vandeley Ind",2947,2011-08-24,8937.6
"Vandeley Ind",2951,2011-08-25,7385.39
"Vandeley Ind",2956,2011-08-31,1639.61
"Vandeley Ind",2965,2011-09-10,1470.26
"Vandeley Ind",2969,2011-09-16,1072.14
"Vandeley Ind",2971,2011-09-26,3749.4
"Vandeley Ind",2979,2011-09-26,2180.26
"Vandeley Ind",2984,2011-10-02,4108.94
"Vandeley Ind",2989,2011-10-03,2267.52
"Vandeley Ind",2993,2011-10-03,6025.17
"Vandeley Ind",2996,2011-10-13,1377.41
"Vandeley Ind",3004,2011-10-21,3960.75
"Vandeley Ind",3006,2011-10-27,3842.16
"Vandeley Ind",3014,2011-10-28,1507.76
"Vandeley Ind",3016,2011-11-05,8385.21
"Vandeley Ind",3021,2011-11-11,2565.31
"Vandeley Ind",3030,2011-11-11,3348.32
"Vandeley Ind",3031,2011-11-19,3292.07
"Vandeley Ind",3038,2011-11-29,1146.08
"Vandeley Ind",3041,2011-11-29,5786.06
"Vandeley Ind",3050,2011-11-29,1236.26
"Vandeley Ind",3055,2011-12-09,3810.85
"Vandeley Ind",3059,2011-12-10,1573.41
"Vandeley Ind",3064,2011-12-10,5420.38
"Vandeley Ind",3070,2011-12-10,1830.81
"Vandeley Ind",3072,2011-12-18,5222.09
"Vandeley Ind",3077,2011-12-24,1577.35
"Vandeley Ind",3083,2011-12-30,4964.75
"Vandeley Ind",3090,2012-01-07,6130.72
"Vandeley Ind",3093,2012-01-17,7908.73
"Vandeley Ind",3097,2012-01-18,3653.55
"Vandeley Ind",3103,2012-01-19,1518.01
"Vandeley Ind",3108,2012-01-27,2634.48
"Vandeley Ind",3114,2012-01-27,1766.2
"Vandeley Ind",3117,2012-01-28,1593.11
"Vandeley Ind",3123,2012-01-29,2244.86
"Vandeley Ind",3127,2012-01-29,1623.44
"Vandeley Ind",3134,2012-02-06,1066.24
"Vandeley Ind",3140,2012-02-07,6220.44
"Vandeley Ind",3144,2012-02-08,3404.49
"Vandeley Ind",3146,2012-02-16,3198.95
"Vandeley Ind",3151,2012-02-22,7418.7
"Vandeley Ind",3159,2012-02-28,8929.58
"Vandeley Ind",3165,2012-03-09,2220.82
"Vandeley Ind",3170,2012-03-10,1259.21
"Vandeley Ind",3172,2012-03-16,1225.64
"Vandeley Ind",3177,2012-03-16,4265.55
"Vandeley Ind",3185,2012-03-22,9519.51
"Vandeley Ind",3187,2012-03-23,5137.21
"Vandeley Ind",3191,2012-03-31,4355.99
"Vandeley Ind",3200,2012-04-08,6823.52
"Vandeley Ind",3205,2012-04-09,3223.78
"Vandeley Ind",3207,2012-04-10,1163.92
"Vandeley Ind",3214,2012-04-11,6521.49
"Vandeley Ind",3220,2012-04-21,9675.94
"Vandeley Ind",3222,2012-04-22,7129.38
"Vandeley Ind",3227,2012-04-30,4023.9
"Vandeley Ind",3233,2012-05-10,6521.57
"Vandeley Ind",3237,2012-05-11,1260.5
"Vandeley Ind",3242,2012-05-19,7851.96
"Vandeley Ind",3250,2012-05-20,1994.49
"Pens R Us",7,2006-01-09,9714.45
"Pens R Us",13,2006-01-09,3499.58
"Pens R Us",20,2006-01-15,8433.86
"Pens R Us",25,2006-01-23,3463.4
"Pens R Us",29,2006-01-23,1189.17
"Pens R Us",32,2006-01-24,9134.26
"Pens R Us",38,2006-01-30,5302.12
"Pens R Us",44,2006-01-31,5487.22
"Pens R Us",50,2006-01-31,1346.87
"Pens R Us",51,2006-02-01,1288.38
"Pens R Us",57,2006-02-02,1181.19
"Pens R Us",65,2006-02-08,3632.68
"Pens R Us",68,2006-02-08,1513.69
"Pens R Us",74,2006-02-09,8967.13
"Pens R Us",76,2006-02-09,1006.58
"Pens R Us",81,2006-02-17,1251.07
"Pens R Us",87,2006-02-25,9745.27
"Pens R Us",92,2006-02-26,2882.37
"Pens R Us",99,2006-02-26,1190.78
"Pens R Us",104,2006-02-26,1281.47
"Pens R Us",107,2006-03-06,4438.73
"Pens R Us",114,2006-03-06,6513.46
"Pens R Us",118,2006-03-07,5106.08
"Pens R Us",121,2006-03-07,8458.59
"Pens R Us",127,2006-03-08,6548.95
"Pens R Us",135,2006-03-14,4009.21
"Pens R Us",139,2006-03-15,3825.57
"Pens R Us",144,2006-03-25,1567.6
"Pens R Us",150,2006-03-25,8156.96
"Pens R Us",155,2006-03-31,1845.52
"Pens R Us",156,2006-04-01,4028.73
"Pens R Us",165,2006-04-01,1128.79
"Pens R Us",167,2006-04-07,5548.2
"Pens R Us",173,2006-04-07,2167.0
"Pens R Us",177,2006-04-17,3401.81
"Pens R Us",184,2006-04-18,3611.32
"Pens R Us",187,2006-04-28,1763.54
"Pens R Us",195,2006-04-28,1344.16
"Pens R Us",196,2006-04-29,1938.57
"Pens R Us",205,2006-05-09,1531.09
"Pens R Us",210,2006-05-10,5473.64
"Pens R Us",214,2006-05-10,3791.5
"Pens R Us",216,2006-05-10,1142.35
"Pens R Us",223,2006-05-11,2232.8
"Pens R Us",226,2006-05-11,1772.07
"Pens R Us",231,2006-05-21,6232.81
"Pens R Us",240,2006-05-21,2587.12
"Pens R Us",242,2006-05-22,2779.35
"Pens R Us",247,2006-05-22,1584.18
"Pens R Us",254,2006-05-28,4718.37
"Pens R Us",257,2006-05-29,4353.69
"Pens R Us",263,2006-05-29,7788.21
"Pens R Us",268,2006-06-08,5177.39
"Pens R Us",273,2006-06-14,1943.23
"Pens R Us",279,2006-06-15,1029.53
"Pens R Us",281,2006-06-25,1412.55
"Pens R Us",289,2006-06-26,2410.37
"Pens R Us",292,2006-07-02,7139.85
"Pens R Us",296,2006-07-12,3982.8
"Pens R Us",302,2006-07-12,6916.76
"Pens R Us",308,2006-07-20,6473.88
"Pens R Us",314,2006-07-26,1099.85
"Pens R Us",316,2006-08-05,1340.22
"Pens R Us",325,2006-08-13,6134.04
"Pens R Us",330,2006-08-14,9971.44
"Pens R Us",331,2006-08-15,9381.48
"Pens R Us",339,2006-08-15,8684.99
"Pens R Us",341,2006-08-16,1486.62
"Pens R Us",347,2006-08-16,1844.73
"Pens R Us",351,2006-08-16,1784.4
"Pens R Us",357,2006-08-22,1147.07
"Pens R Us",362,2006-08-23,2939.96
"Pens R Us",366,2006-08-29,6957.6
"Pens R Us",373,2006-09-06,1694.94
"Pens R Us",376,2006-09-07,3266.28
"Pens R Us",382,2006-09-15,2989.37
"Pens R Us",390,2006-09-25,9589.21
"Pens R Us",392,2006-09-25,7793.38
"Pens R Us",396,2006-10-05,8133.83
"Pens R Us",404,2006-10-06,5857.45
"Pens R Us",408,2006-10-06,1815.62
"Pens R Us",412,2006-10-07,1520.56
"Pens R Us",417,2006-10-13,4712.06
"Pens R Us",425,2006-10-14,1277.77
"Pens R Us",428,2006-10-24,9872.85
"Pens R Us",431,2006-10-30,6523.18
"Pens R Us",437,2006-10-30,1455.2
"Pens R Us",442,2006-11-09,4033.66
"Pens R Us",450,2006-11-10,4743.18
"Pens R Us",452,2006-11-10,3649.16
"Pens R Us",459,2006-11-10,1623.68
"Pens R Us",465,2006-11-10,1296.43
"Pens R Us",466,2006-11-16,3285.9
"Pens R Us",474,2006-11-16,7410.84
"Pens R Us",480,2006-11-17,7654.64
"Pens R Us",483,2006-11-23,1514.12
"Pens R Us",486,2006-11-23,2272.68
"Pens R Us",491,2006-12-01,1192.82
"Pens R Us",499,2006-12-07,2552.99
"Pens R Us",501,2006-12-08,5274.15
"Pens R Us",508,2006-12-16,2085.88
"Pens R Us",514,2006-12-26,1187.43
"Pens R Us",518,2007-01-03,5111.67
"Pens R Us",524,2007-01-11,6703.56
"Pens R Us",526,2007-01-11,3244.88
"Pens R Us",533,2007-01-21,2701.35
"Pens R Us",540,2007-01-27,8002.63
"Pens R Us",544,2007-02-06,9892.95
"Pens R Us",546,2007-02-12,5111.09
"Pens R Us",551,2007-02-12,1069.11
"Pens R Us",557,2007-02-13,7895.29
"Pens R Us",561,2007-02-19,3473.81
"Pens R Us",566,2007-02-19,8539.8
"Pens R Us",575,2007-02-27,4225.36
"Pens R Us",579,2007-03-07,1856.3
"Pens R Us",582,2007-03-17,3125.41
"Pens R Us",588,2007-03-23,2204.88
"Pens R Us",594,2007-03-29,9000.36
"Pens R Us",599,2007-03-30,6443.54
"Pens R Us",601,2007-03-31,1031.88
"Pens R Us",606,2007-04-01,2710.21
"Pens R Us",612,2007-04-07,2759.46
"Pens R Us",616,2007-04-13,8645.0
"Pens R Us",623,2007-04-23,5612.43
"Pens R Us",628,2007-04-24,1858.49
"Pens R Us",633,2007-05-04,2814.72
"Pens R Us",640,2007-05-10,2179.46
"Pens R Us",643,2007-05-16,2092.43
"Pens R Us",646,2007-05-17,2739.84
"Pens R Us",653,2007-05-25,4325.99
"Pens R Us",658,2007-05-31,8421.95
"Pens R Us",664,2007-06-01,1235.0
"Pens R Us",670,2007-06-02,9476.91
"Pens R Us",673,2007-06-03,3330.95
"Pens R Us",677,2007-06-03,9647.0
"Pens R Us",683,2007-06-04,3720.66
"Pens R Us",686,2007-06-05,3158.68
"Pens R Us",692,2007-06-05,9330.27
"Pens R Us",696,2007-06-05,1962.58
"Pens R Us",704,2007-06-11,4212.3
"Pens R Us",710,2007-06-11,6599.02
"Pens R Us",715,2007-06-17,5204.9
"Pens R Us",718,2007-06-18,5537.28
"Pens R Us",725,2007-06-18,7477.3
"Pens R Us",728,2007-06-19,1143.38
"Pens R Us",731,2007-06-20,5670.05
"Pens R Us",738,2007-06-21,1062.58
"Pens R Us",745,2007-06-22,3671.98
"Pens R Us",746,2007-06-23,2865.98
"Pens R Us",752,2007-06-23,8871.02
"Pens R Us",759,2007-06-24,1552.6
"Pens R Us",763,2007-06-30,9723.12
"Pens R Us",768,2007-06-30,9147.56
"Pens R Us",771,2007-07-01,6307.97
"Pens R Us",777,2007-07-07,1648.78
"Pens R Us",783,2007-07-07,4850.23
"Pens R Us",789,2007-07-13,2012.85
"Pens R Us",795,2007-07-13,1314.34
"Pens R Us",797,2007-07-14,6036.05
"Pens R Us",802,2007-07-15,3045.27
"Pens R Us",809,2007-07-16,3592.2
"Pens R Us",811,2007-07-22,5291.14
"Pens R Us",820,2007-07-28,8724.39
"Pens R Us",822,2007-08-03,9324.27
"Pens R Us",827,2007-08-03,4745.89
"Pens R Us",833,2007-08-03,2550.73
"Pens R Us",836,2007-08-03,4041.17
"Pens R Us",841,2007-08-04,2051.33
"Pens R Us",846,2007-08-10,2402.38
"Pens R Us",854,2007-08-10,7704.21
"Pens R Us",859,2007-08-10,8174.77
"Pens R Us",864,2007-08-10,1034.38
"Pens R Us",866,2007-08-18,9053.44
"Pens R Us",875,2007-08-19,1511.79
"Pens R Us",876,2007-08-19,3206.56
"Pens R Us",882,2007-08-20,9981.17
"Pens R Us",890,2007-08-28,1152.39
"Pens R Us",891,2007-09-03,2078.74
"Pens R Us",898,2007-09-09,1344.32
"Pens R Us",905,2007-09-10,7236.27
"Pens R Us",910,2007-09-10,3140.37
"Pens R Us",911,2007-09-16,9449.02
"Pens R Us",917,2007-09-22,4249.48
"Pens R Us",924,2007-09-23,2648.15
"Pens R Us",927,2007-09-23,5807.44
"Pens R Us",935,2007-09-24,2204.33
"Pens R Us",940,2007-09-25,8419.21
"Pens R Us",941,2007-09-26,3029.1
"Pens R Us",946,2007-10-04,2381.06
"Pens R Us",952,2007-10-05,9163.13
"Pens R Us",958,2007-10-06,4107.22
"Pens R Us",964,2007-10-07,3545.31
"Pens R Us",966,2007-10-07,3532.69
"Pens R Us",973,2007-10-07,1162.67
"Pens R Us",978,2007-10-08,2988.79
"Pens R Us",982,2007-10-08,2737.52
"Pens R Us",989,2007-10-14,3977.47
"Pens R Us",993,2007-10-22,1138.51
"Pens R Us",997,2007-10-30,2317.77
"Pens R Us",1001,2007-10-30,1352.61
"Pens R Us",1008,2007-10-30,1888.62
"Pens R Us",1011,2007-10-30,1196.83
"Pens R Us",1018,2007-11-05,3341.02
"Pens R Us",1024,2007-11-13,1667.75
"Pens R Us",1027,2007-11-14,2611.04
"Pens R Us",1034,2007-11-14,5105.36
"Pens R Us",1039,2007-11-14,7272.38
"Pens R Us",1042,2007-11-14,1382.61
"Pens R Us",1049,2007-11-14,1214.13
"Pens R Us",1055,2007-11-20,7962.25
"Pens R Us",1058,2007-11-21,1559.27
"Pens R Us",1065,2007-11-27,5062.78
"Pens R Us",1068,2007-12-05,7344.39
"Pens R Us",1074,2007-12-05,7543.75
"Pens R Us",1076,2007-12-06,4368.75
"Pens R Us",1081,2007-12-07,5049.1
"Pens R Us",1086,2007-12-07,3845.28
"Pens R Us",1095,2007-12-07,8255.74
"Pens R Us",1100,2007-12-17,1933.9
"Pens R Us",1104,2007-12-17,6075.66
"Pens R Us",1106,2007-12-23,1999.91
"Pens R Us",1112,2008-01-02,4258.34
"Pens R Us",1117,2008-01-02,2470.95
"Pens R Us",1122,2008-01-03,1414.3
"Pens R Us",1127,2008-01-13,2039.21
"Pens R Us",1133,2008-01-13,6725.3
"Pens R Us",1138,2008-01-14,1021.66
"Pens R Us",1141,2008-01-15,2442.07
"Pens R Us",1147,2008-01-15,1689.58
"Pens R Us",1154,2008-01-25,1645.14
"Pens R Us",1156,2008-01-31,5145.65
"Pens R Us",1162,2008-02-10,1924.11
"Pens R Us",1170,2008-02-20,2175.35
"Pens R Us",1171,2008-02-28,5065.31
"Pens R Us",1176,2008-02-29,1235.14
"Pens R Us",1184,2008-03-01,2194.51
"Pens R Us",1187,2008-03-02,1216.79
"Pens R Us",1192,2008-03-10,3400.47
"Pens R Us",1200,2008-03-10,2790.11
"Pens R Us",1201,2008-03-16,5200.73
"Pens R Us",1210,2008-03-24,3113.47
"Pens R Us",1215,2008-03-25,2881.77
"Pens R Us",1219,2008-03-26,6052.07
"Pens R Us",1221,2008-03-26,4698.46
"Pens R Us",1227,2008-04-05,2047.06
"Pens R Us",1231,2008-04-05,4284.75
"Pens R Us",1239,2008-04-11,1076.0
"Pens R Us",1243,2008-04-19,5260.84
"Pens R Us",1250,2008-04-27,1589.74
"Pens R Us",1254,2008-05-05,1249.88
"Pens R Us",1260,2008-05-11,5382.7
"Pens R Us",1265,2008-05-12,7424.4
"Pens R Us",1269,2008-05-18,2555.67
"Pens R Us",1274,2008-05-28,2288.87
"Pens R Us",1279,2008-05-29,6322.38
"Pens R Us",1282,2008-05-29,4026.41
"Pens R Us",1286,2008-05-29,2509.99
"Pens R Us",1291,2008-05-29,1200.88
"Pens R Us",1297,2008-06-04,6310.08
"Pens R Us",1303,2008-06-04,6841.13
"Pens R Us",1310,2008-06-04,3860.15
"Pens R Us",1314,2008-06-04,2697.86
"Pens R Us",1316,2008-06-04,2462.86
"Pens R Us",1324,2008-06-10,7702.15
"Pens R Us",1330,2008-06-11,2184.51
"Pens R Us",1333,2008-06-12,1055.24
"Pens R Us",1337,2008-06-18,4514.91
"Pens R Us",1343,2008-06-24,6774.91
"Pens R Us",1349,2008-06-30,1731.5
"Pens R Us",1353,2008-06-30,1504.82
"Pens R Us",1357,2008-07-01,1800.78
"Pens R Us",1364,2008-07-02,2126.13
"Pens R Us",1366,2008-07-02,9272.97
"Pens R Us",1374,2008-07-03,7114.63
"Pens R Us",1377,2008-07-04,5777.03
"Pens R Us",1383,2008-07-12,4674.85
"Pens R Us",1387,2008-07-12,2616.45
"Pens R Us",1393,2008-07-18,1781.95
"Pens R Us",1400,2008-07-28,1463.22
"Pens R Us",1404,2008-07-28,1845.79
"Pens R Us",1410,2008-08-05,1816.62
"Pens R Us",1414,2008-08-06,1451.32
"Pens R Us",1416,2008-08-06,1691.82
"Pens R Us",1421,2008-08-12,5529.26
"Pens R Us",1429,2008-08-12,3986.14
"Pens R Us",1435,2008-08-12,4871.27
"Pens R Us",1439,2008-08-12,1668.29
"Pens R Us",1442,2008-08-22,9538.67
"Pens R Us",1449,2008-08-22,1932.8
"Pens R Us",1451,2008-09-01,6734.6
"Pens R Us",1459,2008-09-01,5593.22
"Pens R Us",1465,2008-09-01,8072.27
"Pens R Us",1469,2008-09-01,1512.71
"Pens R Us",1471,2008-09-01,5307.89
"Pens R Us",1479,2008-09-01,8100.65
"Pens R Us",1481,2008-09-11,1342.56
"Pens R Us",1489,2008-09-17,2125.93
"Pens R Us",1494,2008-09-23,8075.94
"Pens R Us",1498,2008-09-23,1179.53
"Pens R Us",1504,2008-09-23,2748.54
"Pens R Us",1506,2008-09-23,5730.81
"Pens R Us",1512,2008-09-24,8745.55
"Pens R Us",1518,2008-09-24,4936.15
"Pens R Us",1525,2008-09-25,3342.33
"Pens R Us",1530,2008-09-26,1517.14
"Pens R Us",1534,2008-10-02,2159.37
"Pens R Us",1538,2008-10-02,6284.74
"Pens R Us",1544,2008-10-02,5461.45
"Pens R Us",1547,2008-10-02,3231.11
"Pens R Us",1551,2008-10-02,5215.6
"Pens R Us",1558,2008-10-08,6847.58
"Pens R Us",1561,2008-10-08,6163.57
"Pens R Us",1569,2008-10-08,8089.76
"Pens R Us",1571,2008-10-09,8046.4
"Pens R Us",1577,2008-10-15,6018.32
"Pens R Us",1581,2008-10-21,1812.91
"Pens R Us",1588,2008-10-31,3538.87
"Pens R Us",1594,2008-11-06,5966.13
"Pens R Us",1597,2008-11-07,1225.46
"Pens R Us",1603,2008-11-08,8762.99
"Pens R Us",1606,2008-11-18,2825.09
"Pens R Us",1615,2008-11-19,1456.54
"Pens R Us",1618,2008-11-29,1851.18
"Pens R Us",1621,2008-12-07,1486.57
"Pens R Us",1627,2008-12-13,2532.05
"Pens R Us",1633,2008-12-19,3726.45
"Pens R Us",1640,2008-12-20,5152.66
"Pens R Us",1641,2008-12-21,1351.18
"Pens R Us",1647,2008-12-22,2524.05
"Pens R Us",1655,2009-01-01,8787.68
"Pens R Us",1656,2009-01-01,6815.33
"Pens R Us",1663,2009-01-07,1169.16
"Pens R Us",1670,2009-01-07,1152.95
"Pens R Us",1671,2009-01-13,1931.08
"Pens R Us",1677,2009-01-14,9840.1
"Pens R Us",1682,2009-01-14,7136.75
"Pens R Us",1690,2009-01-15,6296.45
"Pens R Us",1695,2009-01-15,5160.64
"Pens R Us",1700,2009-01-15,1312.31
"Pens R Us",1701,2009-01-25,3031.97
"Pens R Us",1710,2009-01-25,3396.85
"Pens R Us",1711,2009-01-26,9878.24
"Pens R Us",1716,2009-01-27,5615.63
"Pens R Us",1721,2009-02-06,4454.32
"Pens R Us",1726,2009-02-06,7482.43
"Pens R Us",1734,2009-02-06,2991.08
"Pens R Us",1736,2009-02-06,2534.04
"Pens R Us",1743,2009-02-07,1102.07
"Pens R Us",1748,2009-02-07,8325.97
"Pens R Us",1753,2009-02-13,4061.29
"Pens R Us",1757,2009-02-19,8663.69
"Pens R Us",1763,2009-02-20,2751.28
"Pens R Us",1767,2009-02-21,1338.93
"Pens R Us",1775,2009-03-01,1102.98
"Pens R Us",1779,2009-03-01,1025.22
"Pens R Us",1784,2009-03-09,2331.23
"Pens R Us",1786,2009-03-19,3924.59
"Pens R Us",1791,2009-03-25,6204.29
"Pens R Us",1799,2009-03-26,9645.44
"Pens R Us",1803,2009-03-26,4585.86
"Pens R Us",1806,2009-04-05,7709.22
"Pens R Us",1812,2009-04-05,1072.29
"Pens R Us",1816,2009-04-15,4982.07
"Pens R Us",1821,2009-04-16,3464.36
"Pens R Us",1829,2009-04-17,1560.99
"Pens R Us",1834,2009-04-17,2391.0
"Pens R Us",1839,2009-04-17,4031.51
"Pens R Us",1845,2009-04-18,1565.91
"Pens R Us",1847,2009-04-24,4420.95
"Pens R Us",1851,2009-04-25,8167.83
"Pens R Us",1857,2009-04-26,6605.25
"Pens R Us",1862,2009-05-06,6709.38
"Pens R Us",1866,2009-05-16,1815.95
"Pens R Us",1872,2009-05-22,1659.27
"Pens R Us",1878,2009-05-28,4383.16
"Pens R Us",1881,2009-05-28,1383.53
"Pens R Us",1887,2009-05-29,6264.3
"Pens R Us",1893,2009-05-30,1599.69
"Pens R Us",1896,2009-06-05,1549.24
"Pens R Us",1901,2009-06-05,1295.97
"Pens R Us",1906,2009-06-06,1360.84
"Pens R Us",1911,2009-06-12,2424.62
"Pens R Us",1918,2009-06-22,1212.17
"Pens R Us",1921,2009-06-30,7053.55
"Pens R Us",1928,2009-07-06,6268.75
"Pens R Us",1935,2009-07-06,1363.04
"Pens R Us",1937,2009-07-12,1560.95
"Pens R Us",1943,2009-07-18,2747.47
"Pens R Us",1950,2009-07-24,3142.98
"Pens R Us",1951,2009-07-25,1108.16
"Pens R Us",1956,2009-07-25,2897.48
"Pens R Us",1964,2009-07-25,1407.64
"Pens R Us",1967,2009-07-31,7462.78
"Pens R Us",1975,2009-08-08,6250.56
"Pens R Us",1980,2009-08-14,6005.24
"Pens R Us",1983,2009-08-15,1329.29
"Pens R Us",1986,2009-08-23,2853.13
"Pens R Us",1995,2009-08-31,3153.62
"Pens R Us",1996,2009-09-06,5738.39
"Pens R Us",2004,2009-09-12,4762.64
"Pens R Us",2008,2009-09-12,1303.01
"Pens R Us",2012,2009-09-22,1881.1
"Pens R Us",2017,2009-09-28,1514.25
"Pens R Us",2022,2009-09-28,3424.17
"Pens R Us",2028,2009-09-28,1899.76
"Pens R Us",2033,2009-10-06,7984.25
"Pens R Us",2040,2009-10-06,1383.21
"Pens R Us",2043,2009-10-16,4962.14
"Pens R Us",2049,2009-10-22,6241.95
"Pens R Us",2055,2009-10-28,1786.21
"Pens R Us",2060,2009-10-28,1882.72
"Pens R Us",2062,2009-10-28,2591.22
"Pens R Us",2067,2009-10-28,3735.93
"Pens R Us",2073,2009-11-03,1665.21
"Pens R Us",2077,2009-11-09,3499.36
"Pens R Us",2083,2009-11-19,6503.54
"Pens R Us",2088,2009-11-20,1427.17
"Pens R Us",2094,2009-11-21,1384.92
"Pens R Us",2097,2009-12-01,7449.22
"Pens R Us",2104,2009-12-09,7654.48
"Pens R Us",2106,2009-12-10,5521.5
"Pens R Us",2113,2009-12-10,1542.39
"Pens R Us",2117,2009-12-16,8009.13
"Pens R Us",2123,2009-12-17,2049.54
"Pens R Us",2130,2009-12-18,1712.5
"Pens R Us",2135,2009-12-28,1806.6
"Pens R Us",2137,2009-12-28,3458.35
"Pens R Us",2141,2009-12-28,3155.49
"Pens R Us",2148,2009-12-29,1977.27
"Pens R Us",2154,2010-01-04,2398.68
"Pens R Us",2160,2010-01-10,2504.34
"Pens R Us",2165,2010-01-10,1875.97
"Pens R Us",2169,2010-01-11,8307.26
"Pens R Us",2174,2010-01-11,1647.07
"Pens R Us",2176,2010-01-12,1138.21
"Pens R Us",2183,2010-01-12,5751.43
"Pens R Us",2189,2010-01-20,3850.97
"Pens R Us",2193,2010-01-26,1361.31
"Pens R Us",2199,2010-01-27,2054.98
"Pens R Us",2201,2010-02-04,2417.98
"Pens R Us",2206,2010-02-14,4014.25
"Pens R Us",2214,2010-02-15,2699.25
"Pens R Us",2219,2010-02-15,1178.88
"Pens R Us",2221,2010-02-15,3313.8
"Pens R Us",2228,2010-02-16,1749.11
"Pens R Us",2231,2010-02-22,2396.24
"Pens R Us",2238,2010-02-22,4581.23
"Pens R Us",2243,2010-02-23,1687.8
"Pens R Us",2247,2010-02-24,2156.75
"Pens R Us",2254,2010-02-25,3704.84
"Pens R Us",2256,2010-02-26,1423.34
"Pens R Us",2261,2010-02-27,9594.89
"Pens R Us",2269,2010-02-28,4752.53
"Pens R Us",2275,2010-03-01,3185.86
"Pens R Us",2276,2010-03-07,2095.77
"Pens R Us",2284,2010-03-13,6015.03
"Pens R Us",2288,2010-03-19,8767.29
"Pens R Us",2295,2010-03-20,8315.45
"Pens R Us",2297,2010-03-26,1750.58
"Pens R Us",2304,2010-03-26,5090.45
"Pens R Us",2309,2010-04-01,4129.85
"Pens R Us",2311,2010-04-09,4182.53
"Pens R Us",2320,2010-04-15,1049.3
"Pens R Us",2324,2010-04-23,4455.74
"Pens R Us",2328,2010-04-29,2105.4
"Pens R Us",2335,2010-04-29,1711.98
"Pens R Us",2340,2010-04-30,3408.37
"Pens R Us",2344,2010-05-01,3734.95
"Pens R Us",2348,2010-05-02,1285.89
"Pens R Us",2353,2010-05-03,8655.51
"Pens R Us",2358,2010-05-03,1416.41
"Pens R Us",2364,2010-05-09,2446.56
"Pens R Us",2367,2010-05-10,2311.41
"Pens R Us",2374,2010-05-16,1321.88
"Pens R Us",2378,2010-05-22,4874.86
"Pens R Us",2381,2010-06-01,1349.38
"Pens R Us",2388,2010-06-11,6099.34
"Pens R Us",2395,2010-06-11,5809.12
"Pens R Us",2396,2010-06-11,1278.05
"Pens R Us",2403,2010-06-12,8015.51
"Pens R Us",2409,2010-06-18,6702.64
"Pens R Us",2414,2010-06-18,2603.71
"Pens R Us",2420,2010-06-24,7685.91
"Pens R Us",2423,2010-06-24,2822.54
"Pens R Us",2427,2010-06-24,9295.81
"Pens R Us",2435,2010-06-24,5506.93
"Pens R Us",2438,2010-06-24,7510.38
"Pens R Us",2443,2010-06-24,2975.93
"Pens R Us",2446,2010-06-25,1900.76
"Pens R Us",2455,2010-06-25,1787.55
"Pens R Us",2460,2010-06-26,1781.16
"Pens R Us",2463,2010-07-02,3875.76
"Pens R Us",2466,2010-07-08,9134.01
"Pens R Us",2474,2010-07-09,6665.53
"Pens R Us",2478,2010-07-09,1068.58
"Pens R Us",2482,2010-07-19,1992.86
"Pens R Us",2487,2010-07-19,6245.86
"Pens R Us",2493,2010-07-19,2972.44
"Pens R Us",2497,2010-07-29,1737.77
"Pens R Us",2501,2010-07-29,4127.96
"Pens R Us",2506,2010-07-29,1332.92
"Pens R Us",2515,2010-07-30,1642.86
"Pens R Us",2517,2010-07-31,6670.67
"Pens R Us",2524,2010-08-08,1077.98
"Pens R Us",2530,2010-08-09,2157.18
"Pens R Us",2532,2010-08-15,1144.64
"Pens R Us",2539,2010-08-16,3607.94
"Pens R Us",2543,2010-08-17,2132.39
"Pens R Us",2550,2010-08-25,1400.91
"Pens R Us",2555,2010-08-25,4517.89
"Pens R Us",2559,2010-09-02,3540.78
"Pens R Us",2562,2010-09-12,1057.65
"Pens R Us",2568,2010-09-12,7155.76
"Pens R Us",2574,2010-09-18,3535.7
"Pens R Us",2579,2010-09-26,4269.58
"Pens R Us",2584,2010-09-26,4329.7
"Pens R Us",2590,2010-09-26,1600.84
"Pens R Us",2591,2010-09-27,5678.49
"Pens R Us",2598,2010-10-05,5514.66
"Pens R Us",2604,2010-10-05,1933.98
"Pens R Us",2609,2010-10-11,8804.75
"Pens R Us",2612,2010-10-12,1270.4
"Pens R Us",2618,2010-10-22,3062.02
"Pens R Us",2624,2010-10-22,2735.84
"Pens R Us",2626,2010-10-22,4101.99
"Pens R Us",2632,2010-10-28,3431.25
"Pens R Us",2636,2010-10-29,4570.27
"Pens R Us",2642,2010-10-29,9220.62
"Pens R Us",2647,2010-10-30,2017.03
"Pens R Us",2654,2010-10-31,4853.15
"Pens R Us",2658,2010-11-10,7544.38
"Pens R Us",2661,2010-11-20,1205.31
"Pens R Us",2668,2010-11-21,8159.32
"Pens R Us",2673,2010-11-22,4447.34
"Pens R Us",2676,2010-11-23,1350.84
"Pens R Us",2682,2010-11-24,1047.64
"Pens R Us",2688,2010-11-25,6087.99
"Pens R Us",2691,2010-12-01,6583.89
"Pens R Us",2700,2010-12-01,3577.96
"Pens R Us",2701,2010-12-01,5625.87
"Pens R Us",2708,2010-12-02,3037.57
"Pens R Us",2713,2010-12-03,2316.79
"Pens R Us",2718,2010-12-11,4554.54
"Pens R Us",2722,2010-12-17,1253.55
"Pens R Us",2727,2010-12-27,1630.67
"Pens R Us",2731,2010-12-27,3964.95
"Pens R Us",2737,2010-12-27,2818.75
"Pens R Us",2743,2011-01-02,5084.01
"Pens R Us",2747,2011-01-03,1707.11
"Pens R Us",2752,2011-01-09,5248.02
"Pens R Us",2759,2011-01-10,8164.05
"Pens R Us",2763,2011-01-10,2682.43
"Pens R Us",2768,2011-01-18,1060.92
"Pens R Us",2771,2011-01-19,2857.09
"Pens R Us",2780,2011-01-19,6600.9
"Pens R Us",2785,2011-01-27,1512.22
"Pens R Us",2787,2011-01-27,1063.78
"Pens R Us",2792,2011-01-27,6462.78
"Pens R Us",2797,2011-01-27,1127.51
"Pens R Us",2801,2011-01-28,2147.4
"Pens R Us",2807,2011-01-29,9022.94
"Pens R Us",2813,2011-02-04,2428.14
"Pens R Us",2816,2011-02-12,1367.5
"Pens R Us",2824,2011-02-20,5138.46
"Pens R Us",2828,2011-02-20,4590.3
"Pens R Us",2831,2011-02-20,8698.93
"Pens R Us",2837,2011-03-02,7403.71
"Pens R Us",2845,2011-03-08,4475.89
"Pens R Us",2850,2011-03-08,9182.18
"Pens R Us",2854,2011-03-16,6649.05
"Pens R Us",2856,2011-03-22,1194.15
"Pens R Us",2865,2011-03-28,1424.06
"Pens R Us",2868,2011-03-29,9222.77
"Pens R Us",2871,2011-03-29,1298.05
"Pens R Us",2878,2011-04-04,3749.61
"Pens R Us",2882,2011-04-05,1949.17
"Pens R Us",2886,2011-04-05,1689.76
"Pens R Us",2892,2011-04-05,6494.3
"Pens R Us",2899,2011-04-11,7284.44
"Pens R Us",2901,2011-04-21,2651.17
"Pens R Us",2908,2011-04-27,1032.24
"Pens R Us",2914,2011-04-28,4776.92
"Pens R Us",2917,2011-05-04,5880.74
"Pens R Us",2924,2011-05-04,2223.56
"Pens R Us",2926,2011-05-14,1784.7
"Pens R Us",2931,2011-05-14,1823.44
"Pens R Us",2937,2011-05-14,2227.31
"Pens R Us",2944,2011-05-14,4513.5
"Pens R Us",2948,2011-05-15,2131.94
"Pens R Us",2955,2011-05-21,9302.66
"Pens R Us",2956,2011-05-27,9136.91
"Pens R Us",2962,2011-05-27,2724.65
"Pens R Us",2966,2011-05-27,3643.32
"Pens R Us",2972,2011-06-02,1087.77
"Pens R Us",2978,2011-06-12,1510.55
"Pens R Us",2981,2011-06-12,3332.69
"Pens R Us",2986,2011-06-12,3593.89
"Pens R Us",2994,2011-06-22,3083.11
"Pens R Us",2999,2011-06-28,6334.13
"Pens R Us",3001,2011-06-29,1123.84
"Pens R Us",3008,2011-06-30,1925.05
"Pens R Us",3012,2011-07-06,4403.95
"Pens R Us",3018,2011-07-06,8305.12
"Pens R Us",3022,2011-07-14,9965.84
"Pens R Us",3026,2011-07-22,4568.05
"Pens R Us",3031,2011-07-23,6500.21
"Pens R Us",3039,2011-07-24,1330.11
"Pens R Us",3043,2011-07-24,9585.88
"Pens R Us",3050,2011-07-25,2448.91
"Pens R Us",3053,2011-07-25,3522.1
"Pens R Us",3059,2011-07-31,3561.29
"Pens R Us",3064,2011-07-31,1325.99
"Pens R Us",3066,2011-08-10,4001.6
"Pens R Us",3071,2011-08-11,2129.2
"Pens R Us",3078,2011-08-11,1815.47
"Pens R Us",3081,2011-08-11,1624.49
"Pens R Us",3088,2011-08-19,3632.45
"Pens R Us",3092,2011-08-29,1616.37
"Pens R Us",3098,2011-08-30,8011.36
"Pens R Us",3104,2011-09-05,4698.49
"Pens R Us",3106,2011-09-06,1285.6
"Pens R Us",3111,2011-09-14,4361.76
"Pens R Us",3118,2011-09-14,9239.56
"Pens R Us",3121,2011-09-20,2517.19
"Pens R Us",3129,2011-09-28,1833.48
"Pens R Us",3135,2011-09-29,1022.78
"Pens R Us",3140,2011-09-30,2447.78
"Pens R Us",3145,2011-10-06,3326.41
"Pens R Us",3147,2011-10-07,7869.27
"Pens R Us",3152,2011-10-17,1370.59
"Pens R Us",3156,2011-10-18,1181.78
"Pens R Us",3162,2011-10-26,4496.94
"Pens R Us",3168,2011-10-26,2253.93
"Pens R Us",3171,2011-10-27,1690.58
"Pens R Us",3177,2011-11-02,2857.41
"Pens R Us",3182,2011-11-03,1430.49
"Pens R Us",3187,2011-11-09,2159.75
"Pens R Us",3193,2011-11-19,6722.01
"Pens R Us",3200,2011-11-20,9849.4
"Pens R Us",3205,2011-11-20,1379.3
"Pens R Us",3207,2011-11-21,7541.48
"Pens R Us",3215,2011-11-22,1032.3
"Pens R Us",3217,2011-12-02,2820.37
"Pens R Us",3222,2011-12-02,3185.15
"Pens R Us",3229,2011-12-03,5637.41
"Pens R Us",3232,2011-12-04,1910.36
"Pens R Us",3237,2011-12-14,3086.16
"Pens R Us",3241,2011-12-22,2072.26
"Pens R Us",3246,2011-12-23,3176.19
"Pens R Us",3251,2011-12-23,1316.89
"Pens R Us",3257,2011-12-23,2988.56
"Pens R Us",3262,2011-12-23,1154.51
"Pens R Us",3270,2011-12-29,1749.41
"Pens R Us",3273,2011-12-29,3353.79
"Pens R Us",3277,2011-12-29,3029.88
"Pens R Us",3284,2011-12-29,2695.8
"Pens R Us",3289,2012-01-04,2675.79
"Pens R Us",3295,2012-01-05,7493.25
"Pens R Us",3299,2012-01-06,2863.71
"Pens R Us",3304,2012-01-06,3965.61
"Pens R Us",3309,2012-01-12,2537.59
"Pens R Us",3315,2012-01-22,2281.44
"Pens R Us",3317,2012-01-28,1614.76
"Pens R Us",3323,2012-01-28,8099.14
"Pens R Us",3329,2012-01-29,1682.04
"Pens R Us",3332,2012-01-29,3420.67
"Pens R Us",3338,2012-02-04,8088.94
"Pens R Us",3345,2012-02-05,6929.42
"Pens R Us",3350,2012-02-05,2490.86
"Pens R Us",3354,2012-02-05,1193.3
"Pens R Us",3356,2012-02-05,9076.35
"Pens R Us",3362,2012-02-13,3497.89
"Pens R Us",3369,2012-02-19,1958.1
"Pens R Us",3373,2012-02-19,3229.34
"Pens R Us",3377,2012-02-20,3998.93
"Pens R Us",3385,2012-02-26,3882.06
"Pens R Us",3386,2012-02-26,2529.95
"Pens R Us",3395,2012-02-26,3877.14
"Pens R Us",3399,2012-02-27,9827.43
"Pens R Us",3401,2012-03-08,6550.75
"Pens R Us",3409,2012-03-14,9219.36
"Pens R Us",3411,2012-03-15,1557.31
"Pens R Us",3417,2012-03-16,7051.37
"Pens R Us",3425,2012-03-26,4301.31
"Pens R Us",3430,2012-03-26,2394.02
"Pens R Us",3431,2012-04-03,1492.56
"Pens R Us",3440,2012-04-03,4803.33
"Pens R Us",3441,2012-04-03,5066.84
"Pens R Us",3447,2012-04-09,2692.85
"Pens R Us",3452,2012-04-10,7753.7
"Pens R Us",3458,2012-04-16,3784.16
"Pens R Us",3462,2012-04-17,1877.76
"Pens R Us",3468,2012-04-17,2416.83
"Pens R Us",3473,2012-04-25,3824.94
"Pens R Us",3479,2012-04-26,3710.82
"Pens R Us",3483,2012-04-26,4609.89
"Pens R Us",3488,2012-04-26,3311.85
"Pens R Us",3491,2012-04-27,1749.03
"Pens R Us",3498,2012-04-28,1443.93
"Pens R Us",3504,2012-04-28,1470.1
"Pens R Us",3506,2012-04-29,8275.35
"Pens R Us",3512,2012-05-05,7042.56
"Pens R Us",3517,2012-05-11,1840.82
"Pens R Us",3524,2012-05-11,1769.25
"Pens R Us",3528,2012-05-12,1054.18
"Pens R Us",3533,2012-05-22,2194.3
"Pens R Us",3538,2012-05-23,2643.23
"Pens R Us",3544,2012-05-24,2259.21
"Pens R Us",3550,2012-05-30,1247.16
"Pens R Us",3553,2012-05-30,6870.08
"Pens R Us",3559,2012-06-05,2713.6
"Pens R Us",3565,2012-06-05,1451.54
"Pens R Us",3569,2012-06-05,8165.27
"Pens R Us",3571,2012-06-05,2560.4
"Pens R Us",3579,2012-06-11,2220.03
"Pens R Us",3583,2012-06-12,2150.72
"Pens R Us",3590,2012-06-12,2985.36
"Pens R Us",3593,2012-06-12,5687.44
"Pens R Us",3597,2012-06-18,5290.97
"Pens R Us",3605,2012-06-19,1223.28
"Pens R Us",3607,2012-06-25,4380.85
"Pens R Us",3613,2012-06-26,3539.92
"Pens R Us",3617,2012-07-06,8477.65
"Pens R Us",3622,2012-07-06,1648.4
"Pens R Us",3630,2012-07-07,3352.36
"Pens R Us",3631,2012-07-15,1485.06
"Pens R Us",3636,2012-07-21,2662.24
"Pens R Us",3641,2012-07-22,1943.78
"DunderMifflin",7,2006-01-01,7160.76
"DunderMifflin",11,2006-01-11,1652.91
"DunderMifflin",19,2006-01-17,3552.17
"DunderMifflin",24,2006-01-27,1641.08
"DunderMifflin",28,2006-01-27,4767.16
"DunderMifflin",32,2006-01-28,3231.91
"DunderMifflin",38,2006-02-03,9512.44
"DunderMifflin",42,2006-02-13,2826.12
"DunderMifflin",50,2006-02-19,1718.99
"DunderMifflin",53,2006-02-20,2884.79
"DunderMifflin",58,2006-02-20,8558.68
"DunderMifflin",61,2006-02-21,4285.22
"DunderMifflin",68,2006-02-21,4792.16
"DunderMifflin",71,2006-02-27,6116.07
"DunderMifflin",76,2006-02-28,2797.57
"DunderMifflin",82,2006-03-10,2716.59
"DunderMifflin",90,2006-03-11,2318.36
"DunderMifflin",95,2006-03-21,2912.11
"DunderMifflin",100,2006-03-27,1400.32
"DunderMifflin",103,2006-03-27,8025.69
"DunderMifflin",106,2006-04-06,2201.0
"DunderMifflin",111,2006-04-07,1548.43
"DunderMifflin",116,2006-04-07,3338.08
"DunderMifflin",124,2006-04-07,1128.55
"DunderMifflin",127,2006-04-08,3100.14
"DunderMifflin",134,2006-04-08,1475.81
"DunderMifflin",136,2006-04-14,1096.2
"DunderMifflin",142,2006-04-15,2005.01
"DunderMifflin",147,2006-04-21,2604.92
"DunderMifflin",152,2006-04-22,8325.02
"DunderMifflin",157,2006-04-22,4414.6
"DunderMifflin",164,2006-04-30,5293.9
"DunderMifflin",166,2006-05-01,2007.34
"DunderMifflin",171,2006-05-09,1758.12
"DunderMifflin",176,2006-05-09,7602.62
"DunderMifflin",183,2006-05-15,4055.69
"DunderMifflin",188,2006-05-23,2949.6
"DunderMifflin",192,2006-05-29,1070.02
"DunderMifflin",200,2006-06-06,2502.47
"DunderMifflin",205,2006-06-07,1284.3
"DunderMifflin",207,2006-06-08,4290.99
"DunderMifflin",211,2006-06-08,6632.85
"DunderMifflin",219,2006-06-09,8695.98
"DunderMifflin",221,2006-06-09,2923.07
"DunderMifflin",230,2006-06-09,1542.68
"DunderMifflin",235,2006-06-10,9923.25
"DunderMifflin",236,2006-06-10,4184.4
"DunderMifflin",245,2006-06-11,1804.72
"DunderMifflin",250,2006-06-17,2411.62
"DunderMifflin",255,2006-06-23,7206.02
"DunderMifflin",260,2006-06-23,5318.74
"DunderMifflin",264,2006-06-24,6074.94
"DunderMifflin",268,2006-06-30,3266.94
"DunderMifflin",275,2006-06-30,4953.92
"DunderMifflin",277,2006-06-30,2712.58
"DunderMifflin",284,2006-07-01,2809.68
"DunderMifflin",288,2006-07-01,3761.27
"DunderMifflin",292,2006-07-07,4395.65
"DunderMifflin",297,2006-07-17,4192.96
"DunderMifflin",305,2006-07-23,5085.99
"DunderMifflin",307,2006-07-23,8203.6
"DunderMifflin",313,2006-07-29,1474.25
"DunderMifflin",318,2006-07-30,2281.29
"DunderMifflin",322,2006-08-07,3308.89
"DunderMifflin",330,2006-08-15,1462.93
"DunderMifflin",331,2006-08-15,3010.84
"DunderMifflin",339,2006-08-21,5050.2
"DunderMifflin",341,2006-08-27,5971.08
"DunderMifflin",347,2006-08-28,3219.78
"DunderMifflin",352,2006-08-28,2452.82
"DunderMifflin",358,2006-09-03,7724.6
"DunderMifflin",361,2006-09-13,5458.9
"DunderMifflin",370,2006-09-13,1106.54
"DunderMifflin",374,2006-09-14,1436.89
"DunderMifflin",378,2006-09-15,2776.99
"DunderMifflin",385,2006-09-23,3283.18
"DunderMifflin",388,2006-09-24,1530.64
"DunderMifflin",395,2006-09-24,1542.11
"DunderMifflin",396,2006-09-25,7391.78
"DunderMifflin",405,2006-10-03,2003.58
"DunderMifflin",407,2006-10-03,1073.55
"DunderMifflin",413,2006-10-04,5350.4
"DunderMifflin",418,2006-10-10,1572.57
"DunderMifflin",424,2006-10-10,3658.36
"DunderMifflin",428,2006-10-11,1223.52
"DunderMifflin",435,2006-10-11,8028.07
"DunderMifflin",440,2006-10-11,4341.32
"DunderMifflin",445,2006-10-11,1799.35
"DunderMifflin",447,2006-10-12,4556.27
"DunderMifflin",452,2006-10-13,6552.19
"DunderMifflin",459,2006-10-14,4566.14
"DunderMifflin",463,2006-10-22,2010.08
"DunderMifflin",467,2006-10-23,4292.1
"DunderMifflin",475,2006-10-29,6279.54
"DunderMifflin",478,2006-10-30,1519.83
"DunderMifflin",482,2006-11-09,1476.71
"DunderMifflin",489,2006-11-10,9278.41
"DunderMifflin",492,2006-11-18,9306.61
"DunderMifflin",498,2006-11-28,5449.62
"DunderMifflin",502,2006-12-08,3881.46
"DunderMifflin",509,2006-12-16,3833.07
"DunderMifflin",515,2006-12-17,8515.11
"DunderMifflin",517,2006-12-17,1332.8
"DunderMifflin",523,2006-12-23,1817.51
"DunderMifflin",529,2006-12-29,4779.53
"DunderMifflin",531,2006-12-30,8857.22
"DunderMifflin",540,2006-12-30,2456.09
"DunderMifflin",543,2007-01-05,4423.0
"DunderMifflin",548,2007-01-06,2732.7
"DunderMifflin",554,2007-01-07,3864.61
"DunderMifflin",557,2007-01-15,8751.9
"DunderMifflin",564,2007-01-15,1978.86
"DunderMifflin",566,2007-01-16,5280.93
"DunderMifflin",572,2007-01-16,1207.1
"DunderMifflin",578,2007-01-22,3490.71
"DunderMifflin",583,2007-01-22,2879.96
"DunderMifflin",589,2007-01-23,2741.86
"DunderMifflin",591,2007-01-24,1430.14
"DunderMifflin",600,2007-01-25,2174.95
"DunderMifflin",602,2007-01-25,6236.34
"DunderMifflin",606,2007-01-26,5893.33
"DunderMifflin",613,2007-01-27,2981.54
"DunderMifflin",616,2007-02-06,1485.32
"DunderMifflin",621,2007-02-06,3120.7
"DunderMifflin",628,2007-02-07,4929.93
"DunderMifflin",632,2007-02-07,8471.69
"DunderMifflin",640,2007-02-15,1599.81
"DunderMifflin",644,2007-02-21,4028.36
"DunderMifflin",646,2007-02-27,1106.37
"DunderMifflin",653,2007-03-07,5009.85
"DunderMifflin",659,2007-03-08,5143.05
"DunderMifflin",663,2007-03-09,1585.39
"DunderMifflin",669,2007-03-09,5296.67
"DunderMifflin",672,2007-03-09,3437.3
"DunderMifflin",678,2007-03-15,3325.46
"DunderMifflin",683,2007-03-16,1469.48
"DunderMifflin",687,2007-03-22,4000.2
"DunderMifflin",693,2007-03-23,9822.71
"DunderMifflin",697,2007-03-23,3565.83
"DunderMifflin",705,2007-03-29,3422.83
"DunderMifflin",708,2007-03-29,2169.03
"DunderMifflin",713,2007-03-30,3207.51
"DunderMifflin",720,2007-03-31,3337.57
"DunderMifflin",725,2007-04-01,2026.0
"DunderMifflin",726,2007-04-07,1301.89
"DunderMifflin",733,2007-04-13,2329.08
"DunderMifflin",737,2007-04-19,2494.56
"DunderMifflin",743,2007-04-20,9957.34
"DunderMifflin",750,2007-04-26,3173.22
"DunderMifflin",751,2007-04-26,8403.08
"DunderMifflin",760,2007-04-26,1195.9
"DunderMifflin",762,2007-05-06,3956.2
"DunderMifflin",768,2007-05-07,5522.69
"DunderMifflin",772,2007-05-07,2963.58
"DunderMifflin",777,2007-05-07,1599.09
"DunderMifflin",781,2007-05-07,3356.47
"DunderMifflin",787,2007-05-08,2199.04
"DunderMifflin",793,2007-05-14,1876.05
"DunderMifflin",796,2007-05-15,8913.19
"DunderMifflin",801,2007-05-15,1198.85
"DunderMifflin",806,2007-05-16,6150.96
"DunderMifflin",813,2007-05-16,7561.69
"DunderMifflin",819,2007-05-26,4070.07
"DunderMifflin",825,2007-05-27,8794.13
"DunderMifflin",827,2007-06-04,6974.31
"DunderMifflin",833,2007-06-05,1656.12
"DunderMifflin",836,2007-06-05,3183.42
"DunderMifflin",842,2007-06-06,4832.84
"DunderMifflin",848,2007-06-07,2158.74
"DunderMifflin",852,2007-06-13,6467.33
"DunderMifflin",858,2007-06-21,4029.17
"DunderMifflin",865,2007-06-27,6603.68
"DunderMifflin",867,2007-06-27,4080.33
"DunderMifflin",875,2007-06-28,3504.94
"DunderMifflin",879,2007-06-28,7822.89
"DunderMifflin",885,2007-07-04,2878.17
"DunderMifflin",886,2007-07-12,3819.83
"DunderMifflin",891,2007-07-18,4208.36
"DunderMifflin",896,2007-07-26,1406.13
"DunderMifflin",902,2007-08-05,1585.75
"DunderMifflin",908,2007-08-11,9107.58
"DunderMifflin",911,2007-08-12,1844.37
"DunderMifflin",917,2007-08-12,7967.44
"DunderMifflin",921,2007-08-18,1655.42
"DunderMifflin",927,2007-08-24,5161.1
"DunderMifflin",933,2007-08-24,3869.76
"DunderMifflin",936,2007-09-03,6409.85
"DunderMifflin",945,2007-09-11,4045.23
"DunderMifflin",946,2007-09-17,2111.1
"DunderMifflin",951,2007-09-17,4035.69
"DunderMifflin",959,2007-09-17,3913.91
"DunderMifflin",964,2007-09-18,3305.42
"DunderMifflin",970,2007-09-18,1980.69
"DunderMifflin",972,2007-09-18,3312.32
"DunderMifflin",976,2007-09-18,4989.39
"DunderMifflin",981,2007-09-19,2013.92
"DunderMifflin",989,2007-09-19,5709.36
"DunderMifflin",995,2007-09-19,3840.0
"DunderMifflin",996,2007-09-20,8844.4
"DunderMifflin",1004,2007-09-30,6864.53
"DunderMifflin",1010,2007-09-30,2876.59
"DunderMifflin",1015,2007-10-06,5065.19
"DunderMifflin",1020,2007-10-07,2015.07
"DunderMifflin",1023,2007-10-13,1740.32
"DunderMifflin",1029,2007-10-19,6471.91
"DunderMifflin",1034,2007-10-25,3795.92
"DunderMifflin",1038,2007-10-26,1532.02
"DunderMifflin",1045,2007-10-27,2138.13
"DunderMifflin",1046,2007-11-02,1091.54
"DunderMifflin",1052,2007-11-10,4611.64
"DunderMifflin",1059,2007-11-10,2275.89
"DunderMifflin",1062,2007-11-16,4058.62
"DunderMifflin",1069,2007-11-16,9143.91
"DunderMifflin",1074,2007-11-26,5303.6
"DunderMifflin",1076,2007-11-26,3196.69
"DunderMifflin",1083,2007-11-27,3362.5
"DunderMifflin",1089,2007-11-27,1440.51
"DunderMifflin",1092,2007-11-28,2907.06
"DunderMifflin",1099,2007-11-29,2741.48
"DunderMifflin",1101,2007-11-30,2182.8
"DunderMifflin",1108,2007-11-30,5811.11
"DunderMifflin",1114,2007-12-01,2343.88
"DunderMifflin",1118,2007-12-02,1763.14
"DunderMifflin",1124,2007-12-03,4000.88
"DunderMifflin",1126,2007-12-11,1235.69
"DunderMifflin",1135,2007-12-11,8084.08
"DunderMifflin",1139,2007-12-11,3074.62
"DunderMifflin",1145,2007-12-12,1560.55
"DunderMifflin",1148,2007-12-20,9216.53
"DunderMifflin",1152,2007-12-21,1026.55
"DunderMifflin",1160,2007-12-27,1841.47
"DunderMifflin",1164,2007-12-28,7959.38
"DunderMifflin",1166,2007-12-28,5934.72
"DunderMifflin",1172,2007-12-29,5417.44
"DunderMifflin",1177,2007-12-30,2725.47
"DunderMifflin",1184,2007-12-31,9503.82
"DunderMifflin",1186,2008-01-01,3430.89
"DunderMifflin",1195,2008-01-02,4483.87
"DunderMifflin",1196,2008-01-03,1244.12
"DunderMifflin",1201,2008-01-04,2110.45
"DunderMifflin",1209,2008-01-10,8507.8
"DunderMifflin",1213,2008-01-16,1337.64
"DunderMifflin",1218,2008-01-24,5114.76
"DunderMifflin",1223,2008-02-01,5239.37
"DunderMifflin",1229,2008-02-07,4389.28
"DunderMifflin",1233,2008-02-07,3507.41
"DunderMifflin",1240,2008-02-07,1960.02
"DunderMifflin",1245,2008-02-17,4455.7
"DunderMifflin",1248,2008-02-18,1746.14
"DunderMifflin",1251,2008-02-19,1446.04
"DunderMifflin",1258,2008-02-19,1941.25
"DunderMifflin",1264,2008-02-19,1117.88
"DunderMifflin",1267,2008-02-29,5817.07
"DunderMifflin",1274,2008-02-29,2696.77
"DunderMifflin",1278,2008-03-06,1501.62
"DunderMifflin",1282,2008-03-16,1023.4
"DunderMifflin",1288,2008-03-22,5182.79
"DunderMifflin",1295,2008-03-23,3500.62
"DunderMifflin",1297,2008-04-02,2803.31
"DunderMifflin",1304,2008-04-12,1145.59
"DunderMifflin",1310,2008-04-18,1310.17
"DunderMifflin",1312,2008-04-19,6451.51
"DunderMifflin",1319,2008-04-19,8780.94
"DunderMifflin",1324,2008-04-20,7653.8
"DunderMifflin",1326,2008-04-21,6801.1
"DunderMifflin",1332,2008-04-27,1361.94
"DunderMifflin",1339,2008-04-28,5728.08
"DunderMifflin",1342,2008-04-29,3537.0
"DunderMifflin",1348,2008-05-07,3823.39
"DunderMifflin",1355,2008-05-15,2391.89
"DunderMifflin",1358,2008-05-15,7513.95
"DunderMifflin",1361,2008-05-16,8050.79
"DunderMifflin",1369,2008-05-22,4217.97
"DunderMifflin",1372,2008-05-28,5257.4
"DunderMifflin",1376,2008-05-29,5934.62
"DunderMifflin",1382,2008-06-08,2631.91
"DunderMifflin",1389,2008-06-08,4278.04
"DunderMifflin",1395,2008-06-14,2786.99
"DunderMifflin",1398,2008-06-14,9155.63
"DunderMifflin",1404,2008-06-15,1084.35
"DunderMifflin",1406,2008-06-16,2021.01
"DunderMifflin",1413,2008-06-17,1608.61
"DunderMifflin",1418,2008-06-23,9048.19
"DunderMifflin",1425,2008-07-03,5600.73
"DunderMifflin",1430,2008-07-04,6640.55
"DunderMifflin",1435,2008-07-05,4684.65
"DunderMifflin",1439,2008-07-05,4136.42
"DunderMifflin",1441,2008-07-15,6150.48
"DunderMifflin",1450,2008-07-21,1190.83
"DunderMifflin",1451,2008-07-31,4219.19
"DunderMifflin",1457,2008-08-08,4089.59
"DunderMifflin",1461,2008-08-09,1933.15
"DunderMifflin",1467,2008-08-09,2008.96
"DunderMifflin",1473,2008-08-10,8651.89
"DunderMifflin",1476,2008-08-11,9614.32
"DunderMifflin",1482,2008-08-17,3140.31
"DunderMifflin",1489,2008-08-23,3527.75
"DunderMifflin",1491,2008-08-23,2359.85
"DunderMifflin",1496,2008-09-02,6412.32
"DunderMifflin",1504,2008-09-02,1126.26
"DunderMifflin",1508,2008-09-02,1407.52
"DunderMifflin",1513,2008-09-08,1773.58
"DunderMifflin",1517,2008-09-08,5257.18
"DunderMifflin",1521,2008-09-08,4028.47
"DunderMifflin",1529,2008-09-08,3446.38
"DunderMifflin",1534,2008-09-18,3072.51
"DunderMifflin",1540,2008-09-18,1532.1
"DunderMifflin",1544,2008-09-26,4546.36
"DunderMifflin",1548,2008-09-26,6502.55
"DunderMifflin",1553,2008-10-06,5469.71
"DunderMifflin",1560,2008-10-12,5949.78
"DunderMifflin",1561,2008-10-13,5043.45
"DunderMifflin",1569,2008-10-14,9309.14
"DunderMifflin",1572,2008-10-14,1748.25
"DunderMifflin",1580,2008-10-20,3571.79
"DunderMifflin",1582,2008-10-20,1176.17
"DunderMifflin",1587,2008-10-20,4337.26
"DunderMifflin",1591,2008-10-20,1475.37
"DunderMifflin",1599,2008-10-21,1197.0
"DunderMifflin",1602,2008-10-22,4530.88
"DunderMifflin",1607,2008-10-23,1652.24
"DunderMifflin",1612,2008-10-23,5500.03
"DunderMifflin",1620,2008-10-23,4864.83
"DunderMifflin",1624,2008-10-23,1394.0
"DunderMifflin",1629,2008-10-24,2585.68
"DunderMifflin",1634,2008-10-24,2464.78
"DunderMifflin",1639,2008-10-24,1714.68
"DunderMifflin",1643,2008-10-30,1500.06
"DunderMifflin",1649,2008-10-30,3237.87
"DunderMifflin",1651,2008-10-30,1502.79
"DunderMifflin",1656,2008-10-30,4456.94
"DunderMifflin",1662,2008-10-31,2612.17
"DunderMifflin",1666,2008-11-01,1402.06
"DunderMifflin",1673,2008-11-01,4206.78
"DunderMifflin",1679,2008-11-07,4441.92
"DunderMifflin",1684,2008-11-08,1813.11
"DunderMifflin",1688,2008-11-09,2970.7
"DunderMifflin",1693,2008-11-09,5535.27
"DunderMifflin",1699,2008-11-10,1803.67
"DunderMifflin",1704,2008-11-11,6380.17
"DunderMifflin",1708,2008-11-21,8008.77
"DunderMifflin",1712,2008-11-22,3025.9
"DunderMifflin",1718,2008-12-02,1068.11
"DunderMifflin",1724,2008-12-08,4163.61
"DunderMifflin",1727,2008-12-09,2202.24
"DunderMifflin",1733,2008-12-15,1539.4
"DunderMifflin",1736,2008-12-25,4111.98
"DunderMifflin",1744,2008-12-31,2054.42
"DunderMifflin",1748,2009-01-06,7916.13
"DunderMifflin",1753,2009-01-14,5234.89
"DunderMifflin",1756,2009-01-22,2812.22
"DunderMifflin",1765,2009-01-22,3655.27
"DunderMifflin",1770,2009-01-22,2866.64
"DunderMifflin",1774,2009-01-28,2639.36
"DunderMifflin",1779,2009-01-29,1638.63
"DunderMifflin",1782,2009-02-08,5323.21
"DunderMifflin",1788,2009-02-08,6176.92
"DunderMifflin",1792,2009-02-09,1189.37
"DunderMifflin",1798,2009-02-15,2459.69
"DunderMifflin",1805,2009-02-21,5434.18
"DunderMifflin",1806,2009-03-01,3225.51
"DunderMifflin",1814,2009-03-07,3415.8
"DunderMifflin",1817,2009-03-08,6256.7
"DunderMifflin",1824,2009-03-08,1127.24
"DunderMifflin",1826,2009-03-08,3381.68
"DunderMifflin",1832,2009-03-08,1267.61
"DunderMifflin",1840,2009-03-09,6298.61
"DunderMifflin",1845,2009-03-09,1005.48
"DunderMifflin",1850,2009-03-10,1439.4
"DunderMifflin",1851,2009-03-11,4207.9
"DunderMifflin",1859,2009-03-17,9623.14
"DunderMifflin",1863,2009-03-18,1044.08
"DunderMifflin",1869,2009-03-28,9312.89
"DunderMifflin",1872,2009-03-28,6714.87
"DunderMifflin",1880,2009-03-29,4619.08
"DunderMifflin",1884,2009-04-04,5815.56
"DunderMifflin",1890,2009-04-05,1097.9
"DunderMifflin",1894,2009-04-11,2036.56
"DunderMifflin",1900,2009-04-12,3074.04
"DunderMifflin",1901,2009-04-13,1385.57
"DunderMifflin",1907,2009-04-13,7084.69
"DunderMifflin",1911,2009-04-14,1724.69
"DunderMifflin",1917,2009-04-20,4798.59
"DunderMifflin",1921,2009-04-26,8343.76
"DunderMifflin",1927,2009-05-06,7253.31
"DunderMifflin",1931,2009-05-07,1958.01
"DunderMifflin",1936,2009-05-15,3690.6
"DunderMifflin",1941,2009-05-15,4440.69
"DunderMifflin",1947,2009-05-21,4590.37
"DunderMifflin",1955,2009-05-29,2230.5
"DunderMifflin",1956,2009-05-29,8195.79
"DunderMifflin",1964,2009-06-08,1788.1
"DunderMifflin",1968,2009-06-09,1724.18
"DunderMifflin",1974,2009-06-10,1360.66
"DunderMifflin",1978,2009-06-10,4826.9
"DunderMifflin",1982,2009-06-11,5874.25
"DunderMifflin",1986,2009-06-11,1158.17
"DunderMifflin",1995,2009-06-12,1042.35
"DunderMifflin",2000,2009-06-12,6219.93
"DunderMifflin",2003,2009-06-20,3698.75
"DunderMifflin",2009,2009-06-26,8846.62
"DunderMifflin",2011,2009-06-26,4107.2
"DunderMifflin",2018,2009-07-06,2137.01
"DunderMifflin",2024,2009-07-06,2691.33
"DunderMifflin",2027,2009-07-06,1393.33
"DunderMifflin",2034,2009-07-07,1655.71
"DunderMifflin",2038,2009-07-07,1023.0
"DunderMifflin",2042,2009-07-07,2475.36
"DunderMifflin",2049,2009-07-07,2268.8
"DunderMifflin",2052,2009-07-07,1728.01
"DunderMifflin",2059,2009-07-13,8470.22
"DunderMifflin",2065,2009-07-19,1536.79
"DunderMifflin",2068,2009-07-25,5063.93
"DunderMifflin",2073,2009-07-26,6368.43
"DunderMifflin",2076,2009-07-27,2240.05
"DunderMifflin",2083,2009-08-06,2800.02
"DunderMifflin",2087,2009-08-14,8692.99
"DunderMifflin",2095,2009-08-14,1398.2
"DunderMifflin",2096,2009-08-20,3057.96
"DunderMifflin",2103,2009-08-21,8342.58
"DunderMifflin",2109,2009-08-21,3360.43
"DunderMifflin",2114,2009-08-22,1645.25
"DunderMifflin",2116,2009-08-22,3257.73
"DunderMifflin",2125,2009-08-28,3809.35
"DunderMifflin",2130,2009-09-05,3952.17
"DunderMifflin",2135,2009-09-11,2210.47
"DunderMifflin",2137,2009-09-11,4446.97
"DunderMifflin",2142,2009-09-11,4789.64
"DunderMifflin",2147,2009-09-12,5915.12
"DunderMifflin",2152,2009-09-18,4922.06
"DunderMifflin",2158,2009-09-19,1045.37
"DunderMifflin",2162,2009-09-19,2932.17
"DunderMifflin",2170,2009-09-25,6842.03
"DunderMifflin",2172,2009-10-03,2013.22
"DunderMifflin",2179,2009-10-03,1710.76
"DunderMifflin",2183,2009-10-03,9987.11
"DunderMifflin",2189,2009-10-11,1868.68
"DunderMifflin",2191,2009-10-21,7720.8
"DunderMifflin",2197,2009-10-31,5669.32
"DunderMifflin",2202,2009-11-10,1673.46
"DunderMifflin",2210,2009-11-11,7034.4
"DunderMifflin",2214,2009-11-12,1216.73
"DunderMifflin",2218,2009-11-18,1107.88
"DunderMifflin",2225,2009-11-18,3640.65
"DunderMifflin",2230,2009-11-19,1144.7
"DunderMifflin",2231,2009-11-20,9014.63
"DunderMifflin",2240,2009-11-26,1835.86
"DunderMifflin",2241,2009-12-02,5079.75
"DunderMifflin",2247,2009-12-02,2627.89
"DunderMifflin",2255,2009-12-03,6414.64
"DunderMifflin",2259,2009-12-04,1125.5
"DunderMifflin",2262,2009-12-04,3345.86
"DunderMifflin",2266,2009-12-04,3070.36
"DunderMifflin",2271,2009-12-05,2513.18
"DunderMifflin",2277,2009-12-11,4288.43
"DunderMifflin",2284,2009-12-11,2061.2
"DunderMifflin",2288,2009-12-21,5554.67
"DunderMifflin",2292,2009-12-21,2978.98
"DunderMifflin",2299,2009-12-22,8622.98
"DunderMifflin",2302,2010-01-01,4480.59
"DunderMifflin",2308,2010-01-02,2741.4
"DunderMifflin",2314,2010-01-03,3001.41
"DunderMifflin",2318,2010-01-09,3222.9
"DunderMifflin",2321,2010-01-10,2775.21
"DunderMifflin",2329,2010-01-10,2426.05
"DunderMifflin",2335,2010-01-16,5702.9
"DunderMifflin",2340,2010-01-22,2129.43
"DunderMifflin",2345,2010-01-22,1661.32
"DunderMifflin",2347,2010-01-23,2795.09
"DunderMifflin",2355,2010-01-23,3796.65
"DunderMifflin",2356,2010-01-31,1698.38
"DunderMifflin",2362,2010-02-01,1335.74
"DunderMifflin",2368,2010-02-02,2340.52
"DunderMifflin",2374,2010-02-10,7142.04
"DunderMifflin",2379,2010-02-10,3019.68
"DunderMifflin",2385,2010-02-16,6288.87
"DunderMifflin",2387,2010-02-17,1807.16
"DunderMifflin",2394,2010-02-17,1409.47
"DunderMifflin",2400,2010-02-18,3490.75
"DunderMifflin",2405,2010-02-28,5327.54
"DunderMifflin",2409,2010-03-10,2508.13
"DunderMifflin",2415,2010-03-18,4030.28
"DunderMifflin",2416,2010-03-26,5155.79
"DunderMifflin",2424,2010-04-01,4249.26
"DunderMifflin",2426,2010-04-01,1316.91
"DunderMifflin",2434,2010-04-09,5612.75
"DunderMifflin",2436,2010-04-10,4123.39
"DunderMifflin",2442,2010-04-10,6042.88
"DunderMifflin",2448,2010-04-18,5664.22
"DunderMifflin",2455,2010-04-24,1153.66
"DunderMifflin",2457,2010-04-25,4813.4
"DunderMifflin",2464,2010-04-25,3288.77
"DunderMifflin",2468,2010-05-01,6653.2
"DunderMifflin",2475,2010-05-01,1898.21
"DunderMifflin",2476,2010-05-02,6652.98
"DunderMifflin",2484,2010-05-08,3326.45
"DunderMifflin",2488,2010-05-18,8239.04
"DunderMifflin",2495,2010-05-19,2508.86
"DunderMifflin",2497,2010-05-19,1879.64
"DunderMifflin",2503,2010-05-20,9978.88
"DunderMifflin",2510,2010-05-21,6359.43
"DunderMifflin",2511,2010-05-27,6485.32
"DunderMifflin",2516,2010-05-27,1202.33
"DunderMifflin",2524,2010-05-27,1952.57
"DunderMifflin",2527,2010-05-27,1676.36
"DunderMifflin",2532,2010-06-02,2664.26
"DunderMifflin",2538,2010-06-02,1414.29
"DunderMifflin",2544,2010-06-03,1349.32
"DunderMifflin",2550,2010-06-03,8363.51
"DunderMifflin",2554,2010-06-04,1047.96
"DunderMifflin",2558,2010-06-10,4725.74
"DunderMifflin",2564,2010-06-11,9462.88
"DunderMifflin",2567,2010-06-11,2012.56
"DunderMifflin",2575,2010-06-11,2995.65
"DunderMifflin",2577,2010-06-17,1039.17
"DunderMifflin",2581,2010-06-27,1815.72
"DunderMifflin",2586,2010-06-27,1474.38
"DunderMifflin",2591,2010-06-27,7461.7
"DunderMifflin",2598,2010-06-28,1414.02
"DunderMifflin",2601,2010-06-29,2422.64
"DunderMifflin",2608,2010-06-29,4688.37
"DunderMifflin",2612,2010-06-29,8606.27
"DunderMifflin",2617,2010-06-29,6631.74
"DunderMifflin",2623,2010-07-07,7422.61
"DunderMifflin",2628,2010-07-07,4122.97
"DunderMifflin",2635,2010-07-07,2347.41
"DunderMifflin",2639,2010-07-13,1807.29
"DunderMifflin",2645,2010-07-14,5495.37
"DunderMifflin",2650,2010-07-20,6745.06
"DunderMifflin",2651,2010-07-21,7078.45
"DunderMifflin",2656,2010-07-21,5097.79
"DunderMifflin",2665,2010-07-22,4028.02
"DunderMifflin",2668,2010-07-23,3339.07
"DunderMifflin",2672,2010-07-24,2629.76
"DunderMifflin",2679,2010-07-24,4672.19
"DunderMifflin",2683,2010-07-25,1250.77
"DunderMifflin",2690,2010-07-25,2499.17
"DunderMifflin",2695,2010-07-25,2145.93
"DunderMifflin",2700,2010-08-04,2137.55
"DunderMifflin",2701,2010-08-12,1946.67
"DunderMifflin",2710,2010-08-12,3705.51
"DunderMifflin",2714,2010-08-12,6836.58
"DunderMifflin",2717,2010-08-20,8973.63
"DunderMifflin",2723,2010-08-20,8895.78
"DunderMifflin",2730,2010-08-28,1492.03
"DunderMifflin",2733,2010-08-28,3918.54
"DunderMifflin",2736,2010-08-28,1209.24
"DunderMifflin",2742,2010-08-28,4632.56
"DunderMifflin",2749,2010-08-29,8800.69
"DunderMifflin",2753,2010-08-29,4682.06
"DunderMifflin",2758,2010-08-30,2222.86
"DunderMifflin",2762,2010-08-30,4602.0
"DunderMifflin",2766,2010-09-05,1158.42
"DunderMifflin",2773,2010-09-11,5859.5
"DunderMifflin",2777,2010-09-12,2325.9
"DunderMifflin",2784,2010-09-13,2558.95
"DunderMifflin",2786,2010-09-21,9841.3
"DunderMifflin",2795,2010-09-27,3065.36
"DunderMifflin",2800,2010-10-07,1048.91
"DunderMifflin",2801,2010-10-07,6579.72
"DunderMifflin",2806,2010-10-17,4031.44
"DunderMifflin",2813,2010-10-18,8202.91
"DunderMifflin",2818,2010-10-18,3754.26
"DunderMifflin",2824,2010-10-19,6648.54
"DunderMifflin",2827,2010-10-27,1200.29
"DunderMifflin",2833,2010-10-28,4493.12
"DunderMifflin",2840,2010-10-28,4315.87
"DunderMifflin",2844,2010-10-28,3650.59
"DunderMifflin",2850,2010-11-07,9388.23
"DunderMifflin",2852,2010-11-07,1893.11
"DunderMifflin",2859,2010-11-13,1804.17
"DunderMifflin",2864,2010-11-19,2556.61
"DunderMifflin",2870,2010-11-29,1714.85
"DunderMifflin",2871,2010-11-30,9504.33
"DunderMifflin",2877,2010-11-30,1990.89
"DunderMifflin",2881,2010-12-01,1281.88
"DunderMifflin",2889,2010-12-02,1648.87
"DunderMifflin",2892,2010-12-08,8260.5
"DunderMifflin",2900,2010-12-09,1329.06
"DunderMifflin",2905,2010-12-15,2237.72
"DunderMifflin",2910,2010-12-15,1732.71
"DunderMifflin",2914,2010-12-16,1262.22
"DunderMifflin",2916,2010-12-16,2077.35
"DunderMifflin",2921,2010-12-16,2034.05
"DunderMifflin",2926,2010-12-17,2631.94
"DunderMifflin",2934,2010-12-17,2177.12
"DunderMifflin",2938,2010-12-18,1819.01
"DunderMifflin",2941,2010-12-19,6829.38
"DunderMifflin",2947,2010-12-19,5933.57
"DunderMifflin",2954,2010-12-25,8175.99
"DunderMifflin",2956,2010-12-25,2548.79
"DunderMifflin",2961,2010-12-26,9148.93
"DunderMifflin",2968,2011-01-03,4447.04
"DunderMifflin",2971,2011-01-04,3449.43
"DunderMifflin",2976,2011-01-14,9264.77
"DunderMifflin",2985,2011-01-15,2604.45
"DunderMifflin",2988,2011-01-21,2847.1
"DunderMifflin",2993,2011-01-21,1431.73
"DunderMifflin",2997,2011-01-22,3513.46
"DunderMifflin",3003,2011-01-30,2863.55
"DunderMifflin",3007,2011-02-05,1829.48
"DunderMifflin",3011,2011-02-05,1002.21
"DunderMifflin",3019,2011-02-05,3616.19
"DunderMifflin",3025,2011-02-13,4476.08
"DunderMifflin",3029,2011-02-21,6609.18
"DunderMifflin",3032,2011-03-01,2820.48
"DunderMifflin",3037,2011-03-01,3406.73
"DunderMifflin",3043,2011-03-02,4491.51
"DunderMifflin",3050,2011-03-03,1144.3
"DunderMifflin",3052,2011-03-13,5995.44
"DunderMifflin",3058,2011-03-13,2859.41
"DunderMifflin",3064,2011-03-14,7303.46
"DunderMifflin",3070,2011-03-20,1581.19
"DunderMifflin",3071,2011-03-30,3063.05
"DunderMifflin",3080,2011-03-31,2440.24
"DunderMifflin",3085,2011-04-10,2914.21
"DunderMifflin",3087,2011-04-10,1156.12
"DunderMifflin",3092,2011-04-11,8871.91
"DunderMifflin",3098,2011-04-11,1391.73
"DunderMifflin",3103,2011-04-11,3043.04
"DunderMifflin",3106,2011-04-11,6165.69
"DunderMifflin",3111,2011-04-21,6138.7
"DunderMifflin",3117,2011-04-21,4422.84
"DunderMifflin",3123,2011-04-22,3272.12
"DunderMifflin",3126,2011-04-28,1982.83
"DunderMifflin",3135,2011-05-06,1091.48
"DunderMifflin",3136,2011-05-07,3294.62
"DunderMifflin",3141,2011-05-15,5262.69
"DunderMifflin",3150,2011-05-15,1803.11
"DunderMifflin",3151,2011-05-23,2979.98
"DunderMifflin",3159,2011-05-24,5520.05
"DunderMifflin",3163,2011-05-30,3565.61
"DunderMifflin",3166,2011-06-05,1063.95
"DunderMifflin",3171,2011-06-06,7877.44
"DunderMifflin",3176,2011-06-06,3583.38
"DunderMifflin",3183,2011-06-16,2628.88
"DunderMifflin",3187,2011-06-16,7385.33
"DunderMifflin",3192,2011-06-17,1250.2
"DunderMifflin",3197,2011-06-23,1930.13
"DunderMifflin",3204,2011-06-24,2962.77
"DunderMifflin",3207,2011-06-30,5653.17
"DunderMifflin",3211,2011-06-30,2285.59
"DunderMifflin",3216,2011-07-01,3865.21
"DunderMifflin",3222,2011-07-01,1836.1
"DunderMifflin",3229,2011-07-02,2114.82
"DunderMifflin",3232,2011-07-10,2558.47
"DunderMifflin",3238,2011-07-16,1563.54
"DunderMifflin",3245,2011-07-17,4501.67
"DunderMifflin",3247,2011-07-27,3516.98
"DunderMifflin",3255,2011-07-27,1762.13
"DunderMifflin",3257,2011-08-02,2106.98
"DunderMifflin",3264,2011-08-03,3534.33
"DunderMifflin",3268,2011-08-04,3697.87
"DunderMifflin",3274,2011-08-10,3667.09
"DunderMifflin",3278,2011-08-10,3042.46
"DunderMifflin",3282,2011-08-18,2004.24
"DunderMifflin",3286,2011-08-18,3630.67
"DunderMifflin",3295,2011-08-18,1223.88
"DunderMifflin",3299,2011-08-28,1794.5
"DunderMifflin",3302,2011-08-28,1702.1
"DunderMifflin",3310,2011-09-05,2553.14
"DunderMifflin",3311,2011-09-06,2558.74
"DunderMifflin",3317,2011-09-07,2442.69
"DunderMifflin",3321,2011-09-15,1065.54
"DunderMifflin",3330,2011-09-16,1306.05
"DunderMifflin",3334,2011-09-17,3360.63
"DunderMifflin",3337,2011-09-18,1815.37
"DunderMifflin",3341,2011-09-18,1125.87
"DunderMifflin",3347,2011-09-18,4280.69
"DunderMifflin",3354,2011-09-19,6268.03
"DunderMifflin",3360,2011-09-25,4830.17
"DunderMifflin",3361,2011-10-01,1845.24
"DunderMifflin",3368,2011-10-11,9130.0
"DunderMifflin",3373,2011-10-17,2000.76
"DunderMifflin",3377,2011-10-18,1390.84
"DunderMifflin",3381,2011-10-19,5715.58
"DunderMifflin",3388,2011-10-27,7186.22
"DunderMifflin",3392,2011-10-28,1198.4
"DunderMifflin",3398,2011-10-28,5100.48
"DunderMifflin",3401,2011-10-28,7447.37
"DunderMifflin",3410,2011-11-07,3277.65
"DunderMifflin",3411,2011-11-17,1312.64
"DunderMifflin",3418,2011-11-27,1191.31
"DunderMifflin",3422,2011-11-28,1381.04
"DunderMifflin",3426,2011-12-04,2258.56
"DunderMifflin",3431,2011-12-04,4491.37
"DunderMifflin",3437,2011-12-12,4589.87
"DunderMifflin",3445,2011-12-12,8827.81
"DunderMifflin",3448,2011-12-13,6378.19
"DunderMifflin",3455,2011-12-13,2455.21
"DunderMifflin",3457,2011-12-21,3039.0
"DunderMifflin",3461,2011-12-21,2065.23
"DunderMifflin",3470,2011-12-21,1285.06
"DunderMifflin",3472,2011-12-31,4028.72
"DunderMifflin",3477,2012-01-01,3125.3
"DunderMifflin",3483,2012-01-02,6429.6
"DunderMifflin",3488,2012-01-03,2571.94
"DunderMifflin",3494,2012-01-09,2168.36
"DunderMifflin",3499,2012-01-09,1997.51
"DunderMifflin",3502,2012-01-15,1317.99
"DunderMifflin",3510,2012-01-15,1009.53
"DunderMifflin",3511,2012-01-15,2540.28
"DunderMifflin",3519,2012-01-25,6057.33
"DunderMifflin",3522,2012-01-26,2136.85
"DunderMifflin",3527,2012-02-03,1387.55
"DunderMifflin",3535,2012-02-03,1472.12
"DunderMifflin",3537,2012-02-03,3305.07
"DunderMifflin",3542,2012-02-03,2681.53
"DunderMifflin",3549,2012-02-04,2050.23
"DunderMifflin",3552,2012-02-12,1710.41
"DunderMifflin",3558,2012-02-12,2264.47
"DunderMifflin",3564,2012-02-18,6632.12
"DunderMifflin",3569,2012-02-19,2246.7
"DunderMifflin",3575,2012-02-29,6540.63
"DunderMifflin",3576,2012-03-01,2982.98
"DunderMifflin",3581,2012-03-02,2831.92
"DunderMifflin",3590,2012-03-10,2431.23
"DunderMifflin",3595,2012-03-16,2223.69
"DunderMifflin",3599,2012-03-17,2299.64
"DunderMifflin",3602,2012-03-18,3543.87
"DunderMifflin",3606,2012-03-24,1456.11
"DunderMifflin",3612,2012-03-25,1161.82
"DunderMifflin",3617,2012-03-31,2576.12
"DunderMifflin",3625,2012-04-01,2028.6
"DunderMifflin",3627,2012-04-02,5977.17
"DunderMifflin",3635,2012-04-03,3779.44
"DunderMifflin",3636,2012-04-03,1624.67
"DunderMifflin",3644,2012-04-04,1057.56
"DunderMifflin",3646,2012-04-14,3321.31
"DunderMifflin",3654,2012-04-14,4678.47
"DunderMifflin",3658,2012-04-22,2562.96
"DunderMifflin",3662,2012-04-22,2579.38
"DunderMifflin",3668,2012-04-28,5832.05
"DunderMifflin",3672,2012-04-28,5798.83
"DunderMifflin",3677,2012-04-29,1312.05
"DunderMifflin",3685,2012-05-05,2479.98
"DunderMifflin",3688,2012-05-06,8616.21
"DunderMifflin",3692,2012-05-16,5968.17
"DunderMifflin",3698,2012-05-26,1658.85
"DunderMifflin",3702,2012-06-01,1555.97
"DunderMifflin",3707,2012-06-01,8327.59
"DunderMifflin",3713,2012-06-07,8568.42
"DunderMifflin",3717,2012-06-07,8384.41
"DunderMifflin",3724,2012-06-13,1987.23
"DunderMifflin",3729,2012-06-13,4178.42
"DunderMifflin",3733,2012-06-14,1978.86
"DunderMifflin",3737,2012-06-22,7081.21
"DunderMifflin",3743,2012-06-28,4054.29
"DunderMifflin",3747,2012-07-06,3256.3
"DunderMifflin",3753,2012-07-12,9128.73
"DunderMifflin",3757,2012-07-18,2964.48
"DunderMifflin",3761,2012-07-18,2018.86
"DunderMifflin",3767,2012-07-26,1056.61
"DunderMifflin",3773,2012-08-03,3972.67
"DunderMifflin",3776,2012-08-09,8800.56
"DunderMifflin",3785,2012-08-10,1165.95
"DunderMifflin",3787,2012-08-11,4658.13
"DunderMifflin",3792,2012-08-12,9857.81
"DunderMifflin",3800,2012-08-12,4983.54
"DunderMifflin",3805,2012-08-12,1911.81
"DunderMifflin",3809,2012-08-12,5181.1
"DunderMifflin",3815,2012-08-12,1301.91
"DunderMifflin",3816,2012-08-13,3427.72
"DunderMifflin",3822,2012-08-21,7065.44
"DunderMifflin",3828,2012-08-21,1484.46
"DunderMifflin",3834,2012-08-21,1646.05
"DunderMifflin",3839,2012-08-21,4676.32
"DunderMifflin",3843,2012-08-31,1835.46
"DunderMifflin",3848,2012-09-01,1607.82
"DunderMifflin",3853,2012-09-11,4363.83
"DunderMifflin",3860,2012-09-12,5070.38
"DunderMifflin",3863,2012-09-12,1617.33
"DunderMifflin",3869,2012-09-22,3332.65
"DunderMifflin",3871,2012-09-23,8318.86
"DunderMifflin",3876,2012-10-03,2023.21
"DunderMifflin",3882,2012-10-03,9047.39
"DunderMifflin",3890,2012-10-03,2669.89
"DunderMifflin",3894,2012-10-04,2732.14
"DunderMifflin",3900,2012-10-04,4219.97
"DunderMifflin",3904,2012-10-12,2504.31
"DunderMifflin",3907,2012-10-18,9002.85
"DunderMifflin",3913,2012-10-18,1687.43
"DunderMifflin",3920,2012-10-24,2323.27
"DunderMifflin",3924,2012-10-25,5413.97
"DunderMifflin",3926,2012-10-26,1448.35
"DunderMifflin",3932,2012-11-01,2485.51
"DunderMifflin",3937,2012-11-09,1216.29
"DunderMifflin",3941,2012-11-10,4421.36
"DunderMifflin",3947,2012-11-16,9189.86
"DunderMifflin",3955,2012-11-16,1859.27
"DunderMifflin",3959,2012-11-17,6816.57
"DunderMifflin",3961,2012-11-25,6428.83
"DunderMifflin",3967,2012-11-26,8008.08
"DunderMifflin",3973,2012-11-27,5670.85
"DunderMifflin",3977,2012-12-03,3324.04
"DunderMifflin",3984,2012-12-09,1922.14
"DunderMifflin",3987,2012-12-15,3846.17
"DunderMifflin",3991,2012-12-15,4545.53
"DunderMifflin",3997,2012-12-16,1104.92
"DunderMifflin",4003,2012-12-17,7053.59
"DunderMifflin",4006,2012-12-17,5108.56
"DunderMifflin",4011,2012-12-23,7780.56
"DunderMifflin",4020,2012-12-23,3917.29
"DunderMifflin",4021,2013-01-02,1063.08
"DunderMifflin",4029,2013-01-08,2521.0
"DunderMifflin",4035,2013-01-08,8864.66
"DunderMifflin",4039,2013-01-16,1756.12
"DunderMifflin",4043,2013-01-24,2852.91
"DunderMifflin",4049,2013-02-03,5275.51
"DunderMifflin",4055,2013-02-03,1163.21
"DunderMifflin",4059,2013-02-11,2253.35
"DunderMifflin",4064,2013-02-11,1553.84
"DunderMifflin",4067,2013-02-12,2646.4
"DunderMifflin",4074,2013-02-18,1533.55
"DunderMifflin",4077,2013-02-19,9843.14
"DunderMifflin",4083,2013-02-19,2076.76
"DunderMifflin",4089,2013-02-19,2198.99
"DunderMifflin",4091,2013-02-20,1083.02
"DunderMifflin",4099,2013-02-21,2404.1
"DunderMifflin",4104,2013-03-01,2694.78
"DunderMifflin",4106,2013-03-01,5216.91
"DunderMifflin",4113,2013-03-11,2070.32
"DunderMifflin",4117,2013-03-17,2687.72
"DunderMifflin",4121,2013-03-17,1656.12
"DunderMifflin",4130,2013-03-17,9511.51
"DunderMifflin",4131,2013-03-17,8513.47
"DunderMifflin",4136,2013-03-25,7014.93
"DunderMifflin",4141,2013-03-26,1034.0
"DunderMifflin",4149,2013-03-26,3814.28
"DunderMifflin",4152,2013-04-03,6808.94
"DunderMifflin",4157,2013-04-03,6005.37
"DunderMifflin",4164,2013-04-11,6621.69
"DunderMifflin",4170,2013-04-11,5308.31
"DunderMifflin",4173,2013-04-11,2248.78
"DunderMifflin",4179,2013-04-21,6148.67
"DunderMifflin",4183,2013-04-27,1677.48
"DunderMifflin",4189,2013-04-28,4509.8
"DunderMifflin",4195,2013-04-28,2881.71
"DunderMifflin",4199,2013-04-29,1029.32
"DunderMifflin",4202,2013-04-30,3006.97
"DunderMifflin",4210,2013-05-01,3922.61
"DunderMifflin",4213,2013-05-11,2834.21
"DunderMifflin",4219,2013-05-11,6222.02
"Burkman Builders",7,2006-01-11,2588.42
"Burkman Builders",15,2006-01-17,3096.99
"Burkman Builders",18,2006-01-23,1647.36
"Burkman Builders",24,2006-01-29,3085.89
"Burkman Builders",26,2006-02-06,6596.71
"Burkman Builders",32,2006-02-06,2652.36
"Burkman Builders",39,2006-02-12,3345.19
"Burkman Builders",41,2006-02-20,7708.55
"Burkman Builders",46,2006-02-21,1153.39
"Burkman Builders",51,2006-02-22,1515.57
"Burkman Builders",57,2006-02-23,1707.58
"Burkman Builders",64,2006-02-23,9387.45
"Burkman Builders",66,2006-03-03,1653.4
"Burkman Builders",73,2006-03-03,8166.87
"Burkman Builders",78,2006-03-04,1585.52
"Burkman Builders",85,2006-03-04,6026.87
"Burkman Builders",87,2006-03-05,4245.93
"Burkman Builders",93,2006-03-06,4890.25
"Burkman Builders",99,2006-03-07,1542.62
"Burkman Builders",105,2006-03-08,6125.28
"Burkman Builders",107,2006-03-08,1488.85
"Burkman Builders",114,2006-03-16,2207.51
"Burkman Builders",119,2006-03-22,7713.73
"Burkman Builders",122,2006-04-01,1435.94
"Burkman Builders",128,2006-04-07,8195.4
"Burkman Builders",133,2006-04-07,3381.25
"Burkman Builders",136,2006-04-07,9102.36
"Burkman Builders",144,2006-04-08,9873.33
"Burkman Builders",148,2006-04-18,1601.69
"Burkman Builders",151,2006-04-18,6384.75
"Burkman Builders",156,2006-04-28,5093.85
"Burkman Builders",161,2006-04-29,1008.51
"Burkman Builders",168,2006-04-30,1988.34
"Burkman Builders",173,2006-05-06,8902.82
"Burkman Builders",180,2006-05-12,9849.95
"Burkman Builders",181,2006-05-13,5495.1
"Burkman Builders",187,2006-05-13,3499.35
"Burkman Builders",192,2006-05-13,2970.18
"Burkman Builders",200,2006-05-19,6036.71
"Burkman Builders",204,2006-05-29,4368.61
"Burkman Builders",208,2006-05-30,5230.77
"Burkman Builders",213,2006-06-07,1190.98
"Burkman Builders",217,2006-06-08,1780.21
"Burkman Builders",223,2006-06-14,1795.61
"Burkman Builders",226,2006-06-22,3163.02
"Burkman Builders",232,2006-06-23,4885.57
"Burkman Builders",239,2006-06-24,5623.42
"Burkman Builders",242,2006-06-25,2188.3
"Burkman Builders",249,2006-06-26,1930.1
"Burkman Builders",255,2006-06-26,4452.98
"Burkman Builders",257,2006-06-27,9830.16
"Burkman Builders",261,2006-06-27,2276.97
"Burkman Builders",269,2006-06-28,1295.08
"Burkman Builders",271,2006-06-29,3309.39
"Burkman Builders",277,2006-07-05,8223.71
"Burkman Builders",284,2006-07-13,6418.36
"Burkman Builders",288,2006-07-13,2400.27
"Burkman Builders",291,2006-07-19,2724.67
"Burkman Builders",297,2006-07-19,5601.73
"Burkman Builders",301,2006-07-20,7011.69
"Burkman Builders",307,2006-07-30,2453.1
"Burkman Builders",312,2006-07-31,9739.9
"Burkman Builders",316,2006-07-31,1597.37
"Burkman Builders",324,2006-08-10,2801.12
"Burkman Builders",328,2006-08-10,4485.06
"Burkman Builders",331,2006-08-16,4215.3
"Burkman Builders",339,2006-08-22,3695.75
"Burkman Builders",341,2006-09-01,2595.62
"Burkman Builders",346,2006-09-02,4009.38
"Burkman Builders",353,2006-09-12,6656.8
"Burkman Builders",358,2006-09-13,3300.98
"Burkman Builders",363,2006-09-14,2162.93
"Burkman Builders",366,2006-09-14,4058.06
"Burkman Builders",375,2006-09-22,1751.82
"Burkman Builders",379,2006-09-22,5443.72
"Burkman Builders",385,2006-09-22,1645.42
"Burkman Builders",388,2006-09-23,1125.57
"Burkman Builders",391,2006-09-29,4280.33
"Burkman Builders",399,2006-10-07,7790.61
"Burkman Builders",402,2006-10-07,7001.27
"Burkman Builders",410,2006-10-07,2522.0
"Burkman Builders",413,2006-10-13,5159.14
"Burkman Builders",417,2006-10-19,4500.39
"Burkman Builders",422,2006-10-25,2859.81
"Burkman Builders",429,2006-10-25,1212.39
"Burkman Builders",432,2006-10-26,2585.43
"Burkman Builders",438,2006-10-26,5142.87
"Burkman Builders",442,2006-10-26,4737.57
"Burkman Builders",447,2006-11-03,1683.02
"Burkman Builders",454,2006-11-09,3267.23
"Burkman Builders",459,2006-11-09,1657.63
"Burkman Builders",464,2006-11-09,2518.79
"Burkman Builders",468,2006-11-15,5797.62
"Burkman Builders",471,2006-11-15,1138.75
"Burkman Builders",477,2006-11-15,3499.85
"Burkman Builders",483,2006-11-16,2911.99
"Burkman Builders",486,2006-11-24,1264.1
"Burkman Builders",492,2006-11-30,3521.58
"Burkman Builders",500,2006-12-06,1411.47
"Burkman Builders",502,2006-12-16,1541.62
"Burkman Builders",509,2006-12-26,2835.63
"Burkman Builders",514,2007-01-01,4365.91
"Burkman Builders",518,2007-01-02,4293.34
"Burkman Builders",524,2007-01-02,5164.23
"Burkman Builders",526,2007-01-12,3031.58
"Burkman Builders",535,2007-01-13,9853.03
"Burkman Builders",536,2007-01-13,1950.95
"Burkman Builders",545,2007-01-14,5217.77
"Burkman Builders",546,2007-01-14,5190.24
"Burkman Builders",552,2007-01-22,4567.23
"Burkman Builders",560,2007-01-30,9746.81
"Burkman Builders",562,2007-01-30,2003.41
"Burkman Builders",570,2007-02-05,3086.66
"Burkman Builders",573,2007-02-11,9173.65
"Burkman Builders",577,2007-02-11,1224.69
"Burkman Builders",583,2007-02-12,3014.9
"Burkman Builders",590,2007-02-13,1229.65
"Burkman Builders",591,2007-02-13,6163.37
"Burkman Builders",597,2007-02-19,3255.31
"Burkman Builders",602,2007-02-19,1662.72
"Burkman Builders",607,2007-02-20,8433.79
"Burkman Builders",615,2007-02-21,1862.38
"Burkman Builders",620,2007-02-21,3825.93
"Burkman Builders",623,2007-02-22,1121.21
"Burkman Builders",627,2007-02-23,2494.06
"Burkman Builders",633,2007-02-23,2073.45
"Burkman Builders",636,2007-02-23,2216.11
"Burkman Builders",645,2007-02-24,4492.57
"Burkman Builders",647,2007-03-04,1356.81
"Burkman Builders",651,2007-03-10,3448.86
"Burkman Builders",656,2007-03-16,2682.91
"Burkman Builders",662,2007-03-24,6168.01
"Burkman Builders",670,2007-04-01,1534.85
"Burkman Builders",672,2007-04-01,2201.46
"Burkman Builders",680,2007-04-02,2221.57
"Burkman Builders",681,2007-04-10,1598.33
"Burkman Builders",694,2007-04-16,8030.74
"Burkman Builders",694,2007-04-17,2478.71
"Burkman Builders",694,2007-04-17,4880.63
"Burkman Builders",705,2007-04-18,7871.62
"Burkman Builders",710,2007-04-28,5901.51
"Burkman Builders",714,2007-05-06,2404.98
"Burkman Builders",716,2007-05-12,1817.43
"Burkman Builders",721,2007-05-12,2154.96
"Burkman Builders",729,2007-05-13,2962.5
"Burkman Builders",732,2007-05-14,5285.64
"Burkman Builders",739,2007-05-14,1127.06
"Burkman Builders",748,2007-05-20,5762.16
"Burkman Builders",748,2007-05-28,2681.47
"Burkman Builders",757,2007-05-29,1831.74
"Burkman Builders",757,2007-05-29,2153.53
"Burkman Builders",762,2007-05-29,8677.86
"Burkman Builders",770,2007-05-29,4639.35
"Burkman Builders",772,2007-05-30,9276.75
"Burkman Builders",777,2007-06-05,4816.55
"Burkman Builders",784,2007-06-11,9573.02
"Burkman Builders",788,2007-06-12,3745.93
"Burkman Builders",791,2007-06-13,8715.89
"Burkman Builders",800,2007-06-14,3831.79
"Burkman Builders",804,2007-06-15,2536.63
"Burkman Builders",808,2007-06-15,3110.57
"Burkman Builders",814,2007-06-15,8952.17
"Burkman Builders",819,2007-06-21,1117.12
"Burkman Builders",825,2007-06-29,1411.43
"Burkman Builders",826,2007-06-29,1328.57
"Burkman Builders",833,2007-07-05,8432.94
"Burkman Builders",839,2007-07-15,9222.4
"Burkman Builders",845,2007-07-25,2501.66
"Burkman Builders",846,2007-07-31,3997.12
"Burkman Builders",854,2007-08-01,3149.23
"Burkman Builders",856,2007-08-01,1717.0
"Burkman Builders",864,2007-08-02,1232.91
"Burkman Builders",868,2007-08-02,1849.8
"Burkman Builders",872,2007-08-08,1254.66
"Burkman Builders",877,2007-08-14,7726.73
"Burkman Builders",883,2007-08-15,1333.84
"Burkman Builders",887,2007-08-16,2881.45
"Burkman Builders",893,2007-08-17,7537.34
"Burkman Builders",899,2007-08-17,1675.54
"Burkman Builders",907,2007-08-17,3600.66
"Burkman Builders",907,2007-08-17,2118.61
"Burkman Builders",912,2007-08-23,5605.11
"Burkman Builders",918,2007-08-31,1426.67
"Burkman Builders",924,2007-09-01,1963.56
"Burkman Builders",930,2007-09-09,2051.04
"Burkman Builders",932,2007-09-09,7257.83
"Burkman Builders",940,2007-09-10,2950.84
"Burkman Builders",942,2007-09-11,2210.06
"Burkman Builders",949,2007-09-11,5244.01
"Burkman Builders",955,2007-09-11,4501.02
"Burkman Builders",957,2007-09-12,1414.51
"Burkman Builders",965,2007-09-22,2581.8
"Burkman Builders",970,2007-09-28,4034.07
"Burkman Builders",972,2007-09-28,2363.38
"Burkman Builders",980,2007-09-28,2216.95
"Burkman Builders",983,2007-09-29,1699.64
"Burkman Builders",989,2007-09-30,3953.63
"Burkman Builders",992,2007-09-30,9638.06
"Burkman Builders",999,2007-10-10,7692.35
"Burkman Builders",1001,2007-10-18,1763.57
"Burkman Builders",1007,2007-10-19,3642.39
"Burkman Builders",1011,2007-10-19,2566.05
"Burkman Builders",1020,2007-10-20,7213.06
"Burkman Builders",1021,2007-10-20,2762.2
"Burkman Builders",1028,2007-10-21,7012.35
"Burkman Builders",1033,2007-10-27,3462.26
"Burkman Builders",1039,2007-10-28,2488.29
"Burkman Builders",1044,2007-10-29,7563.25
"Burkman Builders",1047,2007-11-08,3489.73
"Burkman Builders",1053,2007-11-08,1513.23
"Burkman Builders",1058,2007-11-16,7115.38
"Burkman Builders",1065,2007-11-17,9929.44
"Burkman Builders",1065,2007-11-18,3999.19
"Burkman Builders",1065,2007-11-28,1853.01
"Burkman Builders",1065,2007-12-06,8826.54
"Burkman Builders",1065,2007-12-12,1489.38
"Burkman Builders",1090,2007-12-18,4292.85
"Burkman Builders",1094,2007-12-24,6147.66
"Burkman Builders",1099,2007-12-24,1646.49
"Burkman Builders",1101,2007-12-24,9083.78
"Burkman Builders",1107,2007-12-24,2251.77
"Burkman Builders",1111,2008-01-01,3761.85
"Burkman Builders",1118,2008-01-01,1439.85
"Burkman Builders",1123,2008-01-07,7135.75
"Burkman Builders",1128,2008-01-08,2459.29
"Burkman Builders",1131,2008-01-18,1795.08
"Burkman Builders",1136,2008-01-28,2530.57
"Burkman Builders",1141,2008-01-28,3072.15
"Burkman Builders",1146,2008-01-29,1495.75
"Burkman Builders",1154,2008-01-30,2728.85
"Burkman Builders",1160,2008-02-09,7301.11
"Burkman Builders",1164,2008-02-15,2205.41
"Burkman Builders",1170,2008-02-23,1127.47
"Burkman Builders",1173,2008-02-23,4810.71
"Burkman Builders",1178,2008-02-23,3262.2
"Burkman Builders",1182,2008-02-24,1662.41
"Burkman Builders",1190,2008-02-24,3752.24
"Burkman Builders",1190,2008-03-01,5490.08
"Burkman Builders",1198,2008-03-07,2926.97
"Burkman Builders",1198,2008-03-07,7213.73
"Burkman Builders",1208,2008-03-08,3314.9
"Burkman Builders",1208,2008-03-08,6587.78
"Burkman Builders",1218,2008-03-16,2322.34
"Burkman Builders",1218,2008-03-17,6962.82
"Burkman Builders",1227,2008-03-17,2259.06
"Burkman Builders",1227,2008-03-18,4645.62
"Burkman Builders",1227,2008-03-24,5782.52
"Burkman Builders",1227,2008-04-03,1522.27
"Burkman Builders",1251,2008-04-04,2306.1
"Burkman Builders",1260,2008-04-04,1624.36
"Burkman Builders",1262,2008-04-10,8760.52
"Burkman Builders",1267,2008-04-10,1055.31
"Burkman Builders",1271,2008-04-18,4083.98
"Burkman Builders",1278,2008-04-19,1115.5
"Burkman Builders",1285,2008-04-25,2307.91
"Burkman Builders",1286,2008-05-01,3582.0
"Burkman Builders",1294,2008-05-07,2346.13
"Burkman Builders",1296,2008-05-13,1993.94
"Burkman Builders",1302,2008-05-14,3938.55
"Burkman Builders",1310,2008-05-15,6476.77
"Burkman Builders",1311,2008-05-16,2491.38
"Burkman Builders",1311,2008-05-22,9719.7
"Burkman Builders",1325,2008-05-28,1895.55
"Burkman Builders",1328,2008-05-29,1140.73
"Burkman Builders",1333,2008-06-04,1115.17
"Burkman Builders",1337,2008-06-04,1161.91
"Burkman Builders",1342,2008-06-04,9107.75
"Burkman Builders",1346,2008-06-05,3437.98
"Burkman Builders",1355,2008-06-13,4572.08
"Burkman Builders",1355,2008-06-23,6144.73
"Burkman Builders",1365,2008-06-24,2015.83
"Burkman Builders",1368,2008-06-24,8859.33
"Burkman Builders",1375,2008-06-25,1046.53
"Burkman Builders",1378,2008-07-01,9787.64
"Burkman Builders",1385,2008-07-07,3409.27
"Burkman Builders",1386,2008-07-07,2931.13
"Burkman Builders",1393,2008-07-08,3557.71
"Burkman Builders",1399,2008-07-09,9897.55
"Burkman Builders",1402,2008-07-19,1624.14
"Burkman Builders",1410,2008-07-27,1333.75
"Burkman Builders",1411,2008-08-06,5204.06
"Burkman Builders",1416,2008-08-06,1337.95
"Burkman Builders",1424,2008-08-12,1236.81
"Burkman Builders",1430,2008-08-13,1206.68
"Burkman Builders",1433,2008-08-23,1094.19
"Burkman Builders",1439,2008-08-29,9488.51
"Burkman Builders",1444,2008-09-06,7288.86
"Burkman Builders",1450,2008-09-06,9089.08
"Burkman Builders",1452,2008-09-12,3560.95
"Burkman Builders",1456,2008-09-13,2801.17
"Burkman Builders",1462,2008-09-13,2367.56
"Burkman Builders",1467,2008-09-13,2385.86
"Burkman Builders",1472,2008-09-13,8032.17
"Burkman Builders",1476,2008-09-13,7065.29
"Burkman Builders",1483,2008-09-13,5830.68
"Burkman Builders",1490,2008-09-19,2528.49
"Burkman Builders",1490,2008-09-20,1790.56
"Burkman Builders",1490,2008-09-26,4763.08
"Burkman Builders",1503,2008-09-27,2613.77
"Burkman Builders",1503,2008-10-03,2648.46
"Burkman Builders",1515,2008-10-04,1114.69
"Burkman Builders",1519,2008-10-05,6603.49
"Burkman Builders",1524,2008-10-11,5187.71
"Burkman Builders",1526,2008-10-17,2516.45
"Burkman Builders",1535,2008-10-18,2288.24
"Burkman Builders",1536,2008-10-24,6651.73
"Burkman Builders",1541,2008-11-01,7852.31
"Burkman Builders",1547,2008-11-07,2776.37
"Burkman Builders",1551,2008-11-07,1598.53
"Burkman Builders",1560,2008-11-08,2743.76
"Burkman Builders",1561,2008-11-14,4256.15
"Burkman Builders",1568,2008-11-15,3220.32
"Burkman Builders",1575,2008-11-15,1934.18
"Burkman Builders",1579,2008-11-16,5939.49
"Burkman Builders",1583,2008-11-16,1283.76
"Burkman Builders",1587,2008-11-26,2794.14
"Burkman Builders",1591,2008-11-26,2012.21
"Burkman Builders",1599,2008-11-27,2881.32
"Burkman Builders",1603,2008-11-27,1065.39
"Burkman Builders",1608,2008-12-03,1910.12
"Burkman Builders",1613,2008-12-11,2972.78
"Burkman Builders",1616,2008-12-12,1955.84
"Burkman Builders",1624,2008-12-18,1312.9
"Burkman Builders",1630,2008-12-18,1900.33
"Burkman Builders",1633,2008-12-19,3145.39
"Burkman Builders",1638,2008-12-20,9185.03
"Burkman Builders",1641,2008-12-30,2941.86
"Burkman Builders",1649,2008-12-31,1245.16
"Burkman Builders",1655,2009-01-01,1033.42
"Burkman Builders",1656,2009-01-09,4078.28
"Burkman Builders",1665,2009-01-10,8345.92
"Burkman Builders",1668,2009-01-11,3079.59
"Burkman Builders",1674,2009-01-11,1750.04
"Burkman Builders",1679,2009-01-12,1749.3
"Burkman Builders",1684,2009-01-12,9840.41
"Burkman Builders",1687,2009-01-13,1201.8
"Burkman Builders",1695,2009-01-14,8376.85
"Burkman Builders",1698,2009-01-20,1946.21
"Burkman Builders",1704,2009-01-20,2144.04
"Burkman Builders",1706,2009-01-20,2995.05
"Burkman Builders",1711,2009-01-21,1347.53
"Burkman Builders",1719,2009-01-21,2890.36
"Burkman Builders",1723,2009-01-22,5053.63
"Burkman Builders",1729,2009-01-22,4585.81
"Burkman Builders",1731,2009-01-28,8475.06
"Burkman Builders",1738,2009-01-28,2315.69
"Burkman Builders",1744,2009-02-03,5582.88
"Burkman Builders",1746,2009-02-13,1313.79
"Burkman Builders",1754,2009-02-23,2415.59
"Burkman Builders",1757,2009-03-01,3921.93
"Burkman Builders",1763,2009-03-01,7485.62
"Burkman Builders",1768,2009-03-09,3364.87
"Burkman Builders",1771,2009-03-15,7849.51
"Burkman Builders",1779,2009-03-15,7734.35
"Burkman Builders",1784,2009-03-15,1347.13
"Burkman Builders",1787,2009-03-16,7305.51
"Burkman Builders",1792,2009-03-24,1175.93
"Burkman Builders",1797,2009-03-25,1474.76
"Burkman Builders",1805,2009-03-26,1257.97
"Burkman Builders",1809,2009-03-27,2563.89
"Burkman Builders",1819,2009-03-28,3130.69
"Burkman Builders",1819,2009-04-05,1791.58
"Burkman Builders",1823,2009-04-11,3839.56
"Burkman Builders",1826,2009-04-12,1639.24
"Burkman Builders",1833,2009-04-13,2602.9
"Burkman Builders",1839,2009-04-14,1975.48
"Burkman Builders",1843,2009-04-15,9671.17
"Burkman Builders",1849,2009-04-21,9088.29
"Burkman Builders",1853,2009-04-21,2874.3
"Burkman Builders",1859,2009-05-01,6802.93
"Burkman Builders",1863,2009-05-09,3961.64
"Burkman Builders",1866,2009-05-09,2068.55
"Burkman Builders",1873,2009-05-09,8098.39
"Burkman Builders",1878,2009-05-09,1237.96
"Burkman Builders",1883,2009-05-15,4167.51
"Burkman Builders",1890,2009-05-21,5734.98
"Burkman Builders",1894,2009-05-21,1079.09
"Burkman Builders",1896,2009-05-22,6697.02
"Burkman Builders",1905,2009-05-22,1214.95
"Burkman Builders",1908,2009-05-23,7360.62
"Burkman Builders",1912,2009-06-02,4536.98
"Burkman Builders",1917,2009-06-10,1758.42
"Burkman Builders",1922,2009-06-16,2316.1
"Burkman Builders",1930,2009-06-22,2229.32
"Burkman Builders",1933,2009-07-02,7314.3
"Burkman Builders",1939,2009-07-08,3385.99
"Burkman Builders",1945,2009-07-09,5836.91
"Burkman Builders",1947,2009-07-19,1208.21
"Burkman Builders",1953,2009-07-19,2608.61
"Burkman Builders",1960,2009-07-19,5547.91
"Burkman Builders",1961,2009-07-20,1113.84
"Burkman Builders",1967,2009-07-26,6224.41
"Burkman Builders",1974,2009-08-01,1618.86
"Burkman Builders",1977,2009-08-02,3579.83
"Burkman Builders",1982,2009-08-03,5796.77
"Burkman Builders",1988,2009-08-09,2048.55
"Burkman Builders",1993,2009-08-10,2144.82
"Burkman Builders",1998,2009-08-18,1031.79
"Burkman Builders",2001,2009-08-18,1456.37
"Burkman Builders",2006,2009-08-19,5898.2
"Burkman Builders",2011,2009-08-25,5659.76
"Burkman Builders",2017,2009-08-26,5068.39
"Burkman Builders",2025,2009-08-26,6548.55
"Burkman Builders",2028,2009-08-26,3693.45
"Burkman Builders",2034,2009-09-03,1152.95
"Burkman Builders",2036,2009-09-03,2267.06
"Burkman Builders",2043,2009-09-09,7366.89
"Burkman Builders",2048,2009-09-15,4871.2
"Burkman Builders",2054,2009-09-21,2319.14
"Burkman Builders",2059,2009-09-21,1155.26
"Burkman Builders",2062,2009-09-22,9108.74
"Burkman Builders",2069,2009-10-02,2450.69
"Burkman Builders",2074,2009-10-03,1390.01
"Burkman Builders",2076,2009-10-03,1700.2
"Burkman Builders",2084,2009-10-04,6926.97
"Burkman Builders",2089,2009-10-04,1533.9
"Burkman Builders",2095,2009-10-10,5344.37
"Burkman Builders",2096,2009-10-16,6426.65
"Burkman Builders",2101,2009-10-24,4342.3
"Burkman Builders",2107,2009-11-01,6882.97
"Burkman Builders",2111,2009-11-02,3766.29
"Burkman Builders",2118,2009-11-10,7653.28
"Burkman Builders",2124,2009-11-16,2015.95
"Burkman Builders",2126,2009-11-26,2174.62
"Burkman Builders",2132,2009-12-02,1505.44
"Burkman Builders",2139,2009-12-10,3678.08
"Burkman Builders",2145,2009-12-11,6005.65
"Burkman Builders",2148,2009-12-12,3604.57
"Burkman Builders",2152,2009-12-18,1085.92
"Burkman Builders",2159,2009-12-19,9449.54
"Burkman Builders",2163,2009-12-25,2197.83
"Burkman Builders",2170,2010-01-04,1237.94
"Burkman Builders",2171,2010-01-04,2558.97
"Burkman Builders",2179,2010-01-04,3962.76
"Burkman Builders",2181,2010-01-05,1791.32
"Burkman Builders",2188,2010-01-15,2754.71
"Burkman Builders",2191,2010-01-16,2515.5
"Burkman Builders",2198,2010-01-17,2618.12
"Burkman Builders",2205,2010-01-18,4553.56
"Burkman Builders",2208,2010-01-18,3303.83
"Burkman Builders",2215,2010-01-18,2889.84
"Burkman Builders",2218,2010-01-18,7151.46
"Burkman Builders",2221,2010-01-19,7439.9
"Burkman Builders",2227,2010-01-25,4775.08
"Burkman Builders",2233,2010-02-02,2398.75
"Burkman Builders",2240,2010-02-02,5291.28
"Burkman Builders",2244,2010-02-03,1117.14
"Burkman Builders",2246,2010-02-03,3518.26
"Burkman Builders",2252,2010-02-09,6003.41
"Burkman Builders",2252,2010-02-17,1479.77
"Burkman Builders",2263,2010-02-18,2144.87
"Burkman Builders",2273,2010-02-19,2348.11
"Burkman Builders",2273,2010-02-20,8441.97
"Burkman Builders",2278,2010-02-21,2351.15
"Burkman Builders",2283,2010-03-03,3513.23
"Burkman Builders",2289,2010-03-04,1424.44
"Burkman Builders",2293,2010-03-10,1069.21
"Burkman Builders",2296,2010-03-11,1144.25
"Burkman Builders",2304,2010-03-11,1350.83
"Burkman Builders",2307,2010-03-12,4613.78
"Burkman Builders",2314,2010-03-20,2275.85
"Burkman Builders",2319,2010-03-20,1401.84
"Burkman Builders",2325,2010-03-28,1332.04
"Burkman Builders",2330,2010-04-05,9396.42
"Burkman Builders",2334,2010-04-11,4422.84
"Burkman Builders",2340,2010-04-19,1770.41
"Burkman Builders",2341,2010-04-20,1112.11
"Burkman Builders",2347,2010-04-20,4885.76
"Burkman Builders",2352,2010-04-28,4814.19
"Burkman Builders",2360,2010-04-28,8165.66
"Burkman Builders",2364,2010-05-04,1662.36
"Burkman Builders",2366,2010-05-14,7168.44
"Burkman Builders",2373,2010-05-15,4411.81
"Burkman Builders",2377,2010-05-21,4557.87
"Burkman Builders",2382,2010-05-22,1989.68
"Burkman Builders",2389,2010-05-23,4980.14
"Burkman Builders",2391,2010-05-24,1473.13
"Burkman Builders",2400,2010-05-24,1146.9
"Burkman Builders",2405,2010-05-25,4229.67
"Burkman Builders",2410,2010-05-31,1028.45
"Burkman Builders",2411,2010-06-01,1596.63
"Burkman Builders",2416,2010-06-02,2021.07
"Burkman Builders",2423,2010-06-03,1153.42
"Burkman Builders",2427,2010-06-04,5024.55
"Burkman Builders",2435,2010-06-04,2167.54
"Burkman Builders",2439,2010-06-10,1022.9
"Burkman Builders",2445,2010-06-10,8315.99
"Burkman Builders",2449,2010-06-11,1143.08
"Burkman Builders",2455,2010-06-21,3217.46
"Burkman Builders",2457,2010-06-22,1063.96
"Burkman Builders",2463,2010-06-30,1065.13
"Burkman Builders",2468,2010-06-30,9095.92
"Burkman Builders",2473,2010-06-30,1374.15
"Burkman Builders",2477,2010-06-30,5004.69
"Burkman Builders",2484,2010-07-06,4179.81
"Burkman Builders",2489,2010-07-07,1596.42
"Burkman Builders",2491,2010-07-07,1747.97
"Burkman Builders",2497,2010-07-08,1517.09
"Burkman Builders",2501,2010-07-08,5904.37
"Burkman Builders",2507,2010-07-16,4157.43
"Burkman Builders",2515,2010-07-17,1496.3
"Burkman Builders",2518,2010-07-17,2640.01
"Burkman Builders",2521,2010-07-17,7059.95
"Burkman Builders",2526,2010-07-23,2677.8
"Burkman Builders",2534,2010-07-23,6524.0
"Burkman Builders",2540,2010-07-23,1276.63
"Burkman Builders",2542,2010-07-23,8764.24
"Burkman Builders",2547,2010-07-23,4666.98
"Burkman Builders",2551,2010-07-29,3519.2
"Burkman Builders",2560,2010-07-30,1481.25
"Burkman Builders",2565,2010-07-30,1834.24
"Burkman Builders",2567,2010-07-31,3422.5
"Burkman Builders",2572,2010-07-31,4965.77
"Burkman Builders",2579,2010-08-10,2273.83
"Burkman Builders",2583,2010-08-18,3616.29
"Burkman Builders",2588,2010-08-28,1683.26
"Burkman Builders",2595,2010-09-03,7718.47
"Burkman Builders",2598,2010-09-09,5598.93
"Burkman Builders",2602,2010-09-10,4196.94
"Burkman Builders",2606,2010-09-11,1887.64
"Burkman Builders",2613,2010-09-21,2180.39
"Burkman Builders",2617,2010-09-27,4054.59
"Burkman Builders",2625,2010-09-27,1245.45
"Burkman Builders",2629,2010-10-03,1352.97
"Burkman Builders",2632,2010-10-09,2559.71
"Burkman Builders",2636,2010-10-17,1118.54
"Burkman Builders",2641,2010-10-18,2522.69
"Burkman Builders",2646,2010-10-28,5122.48
"Burkman Builders",2652,2010-11-03,4057.73
"Burkman Builders",2657,2010-11-11,4793.25
"Burkman Builders",2665,2010-11-17,1485.03
"Burkman Builders",2670,2010-11-18,1437.51
"Burkman Builders",2675,2010-11-18,2127.98
"Burkman Builders",2678,2010-11-19,1660.4
"Burkman Builders",2685,2010-11-20,6056.67
"Burkman Builders",2687,2010-11-21,1978.25
"Burkman Builders",2695,2010-11-22,6500.77
"Burkman Builders",2696,2010-11-23,7581.32
"Burkman Builders",2702,2010-11-24,4263.62
"Burkman Builders",2707,2010-11-25,1464.92
"Burkman Builders",2714,2010-11-26,1490.63
"Burkman Builders",2720,2010-12-06,9136.89
"Burkman Builders",2724,2010-12-12,8763.71
"Burkman Builders",2730,2010-12-13,5895.18
"Burkman Builders",2732,2010-12-13,6477.1
"Burkman Builders",2739,2010-12-21,6072.55
"Burkman Builders",2744,2010-12-27,3065.01
"Burkman Builders",2747,2010-12-27,1649.27
"Burkman Builders",2752,2010-12-27,3123.7
"Burkman Builders",2759,2011-01-04,1582.77
"Burkman Builders",2763,2011-01-05,1274.25
"Burkman Builders",2766,2011-01-06,3525.64
"Burkman Builders",2774,2011-01-12,2326.74
"Burkman Builders",2778,2011-01-13,8126.65
"Burkman Builders",2782,2011-01-23,7916.77
"Burkman Builders",2790,2011-01-31,1163.72
"Burkman Builders",2791,2011-01-31,3227.25
"Burkman Builders",2800,2011-02-08,3797.86
"Burkman Builders",2803,2011-02-09,3495.38
"Burkman Builders",2810,2011-02-15,5047.13
"Burkman Builders",2811,2011-02-25,2412.45
"Burkman Builders",2818,2011-02-26,1367.11
"Burkman Builders",2825,2011-03-04,2361.96
"Burkman Builders",2826,2011-03-05,1523.3
"Burkman Builders",2833,2011-03-06,4391.02
"Burkman Builders",2837,2011-03-06,1281.12
"Burkman Builders",2842,2011-03-12,4470.78
"Burkman Builders",2848,2011-03-12,6080.13
"Burkman Builders",2855,2011-03-13,7840.18
"Burkman Builders",2858,2011-03-19,1975.59
"Burkman Builders",2865,2011-03-25,1755.18
"Burkman Builders",2867,2011-03-31,1366.69
"Burkman Builders",2875,2011-04-10,7580.39
"Burkman Builders",2880,2011-04-11,3077.65
"Burkman Builders",2883,2011-04-12,4790.2
"Burkman Builders",2887,2011-04-18,1888.22
"Burkman Builders",2893,2011-04-26,2067.99
"Burkman Builders",2898,2011-05-02,3797.34
"Burkman Builders",2903,2011-05-02,3001.12
"Burkman Builders",2910,2011-05-08,1069.67
"Burkman Builders",2913,2011-05-08,4699.96
"Burkman Builders",2916,2011-05-08,3195.52
"Burkman Builders",2925,2011-05-09,8747.27
"Burkman Builders",2927,2011-05-10,3737.5
"Burkman Builders",2931,2011-05-11,2507.84
"Burkman Builders",2938,2011-05-11,5372.18
"Burkman Builders",2941,2011-05-17,2504.75
"Burkman Builders",2948,2011-05-25,1803.39
"Burkman Builders",2953,2011-05-25,2310.73
"Burkman Builders",2956,2011-05-31,9797.0
"Burkman Builders",2964,2011-06-01,4093.52
"Burkman Builders",2969,2011-06-01,6428.51
"Burkman Builders",2975,2011-06-07,1371.82
"Burkman Builders",2978,2011-06-07,7191.49
"Burkman Builders",2983,2011-06-17,1130.39
"Burkman Builders",2989,2011-06-25,4249.75
"Burkman Builders",2992,2011-06-26,2480.66
"Burkman Builders",2998,2011-06-26,3094.48
"Burkman Builders",3002,2011-06-26,1449.96
"Burkman Builders",3009,2011-07-02,1708.87
"Burkman Builders",3011,2011-07-02,2061.79
"Burkman Builders",3016,2011-07-12,9398.44
"Burkman Builders",3024,2011-07-18,2012.53
"Burkman Builders",3029,2011-07-18,7864.0
"Burkman Builders",3032,2011-07-24,1639.8
"Burkman Builders",3039,2011-07-24,1964.82
"Burkman Builders",3042,2011-07-25,1737.02
"Burkman Builders",3049,2011-08-02,7688.1
"Burkman Builders",3053,2011-08-03,2081.54
"Burkman Builders",3060,2011-08-03,1637.5
"Burkman Builders",3065,2011-08-03,3283.98
"Burkman Builders",3068,2011-08-09,6641.92
"Burkman Builders",3075,2011-08-15,4449.0
"Burkman Builders",3075,2011-08-21,1314.27
"Burkman Builders",3084,2011-08-22,3804.16
"Burkman Builders",3086,2011-08-23,7166.88
"Burkman Builders",3093,2011-08-23,5104.29
"Burkman Builders",3098,2011-08-23,2138.5
"Burkman Builders",3101,2011-08-29,6744.2
"Burkman Builders",3108,2011-08-29,2714.96
"Burkman Builders",3114,2011-08-30,1563.81
"Burkman Builders",3116,2011-08-30,2741.1
"Burkman Builders",3123,2011-09-09,5380.59
"Burkman Builders",3130,2011-09-15,1712.37
"Burkman Builders",3135,2011-09-16,4890.14
"Burkman Builders",3136,2011-09-26,2110.95
"Burkman Builders",3144,2011-09-26,1168.98
"Burkman Builders",3150,2011-10-06,2169.73
"Burkman Builders",3152,2011-10-12,3703.46
"Burkman Builders",3159,2011-10-18,6170.0
"Burkman Builders",3165,2011-10-24,3464.4
"Burkman Builders",3168,2011-10-25,1949.88
"Burkman Builders",3171,2011-10-25,2127.25
"Burkman Builders",3178,2011-10-25,1500.38
"Burkman Builders",3183,2011-10-25,8729.13
"Burkman Builders",3190,2011-10-25,1906.05
"Burkman Builders",3193,2011-10-26,1113.32
"Burkman Builders",3198,2011-10-26,3328.13
"Burkman Builders",3201,2011-11-05,6927.53
"Burkman Builders",3207,2011-11-15,3404.17
"Burkman Builders",3211,2011-11-15,2011.21
"Burkman Builders",3219,2011-11-16,2415.68
"Burkman Builders",3224,2011-11-16,1159.84
"Burkman Builders",3227,2011-11-16,1088.92
"Burkman Builders",3234,2011-11-17,1216.24
"Burkman Builders",3238,2011-11-18,6070.49
"Burkman Builders",3242,2011-11-26,7679.35
"Burkman Builders",3250,2011-12-06,3908.82
"Burkman Builders",3252,2011-12-16,9272.72
"Burkman Builders",3259,2011-12-17,1269.07
"Burkman Builders",3264,2011-12-23,2418.43
"Burkman Builders",3269,2011-12-23,2994.54
"Burkman Builders",3273,2011-12-31,5350.9
"Burkman Builders",3279,2011-12-31,5347.25
"Burkman Builders",3281,2012-01-01,1286.69
"Burkman Builders",3289,2012-01-01,3087.42
"Burkman Builders",3294,2012-01-07,2709.48
"Burkman Builders",3298,2012-01-13,2674.49
"Burkman Builders",3301,2012-01-14,4325.75
"Burkman Builders",3307,2012-01-24,7056.82
"Burkman Builders",3314,2012-01-25,1460.65
"Burkman Builders",3317,2012-01-26,2402.06
"Burkman Builders",3322,2012-01-26,1014.04
"Burkman Builders",3327,2012-01-26,5283.01
"Burkman Builders",3335,2012-02-05,1959.06
"Burkman Builders",3337,2012-02-06,1746.47
"Burkman Builders",3344,2012-02-16,5020.07
"Burkman Builders",3346,2012-02-17,6224.05
"Burkman Builders",3352,2012-02-17,2433.52
"Burkman Builders",3360,2012-02-17,2651.72
"Burkman Builders",3364,2012-02-18,1202.7
"Burkman Builders",3368,2012-02-28,1052.04
"Burkman Builders",3372,2012-03-05,4421.36
"Burkman Builders",3376,2012-03-05,1008.52
"Burkman Builders",3384,2012-03-06,9080.43
"Burkman Builders",3387,2012-03-12,5201.08
"Burkman Builders",3392,2012-03-12,4070.31
"Burkman Builders",3397,2012-03-12,5550.09
"Burkman Builders",3404,2012-03-13,1270.7
"Burkman Builders",3407,2012-03-21,1767.18
"Burkman Builders",3412,2012-03-22,3286.33
"Burkman Builders",3416,2012-04-01,7349.98
"Burkman Builders",3423,2012-04-02,1881.6
"Burkman Builders",3426,2012-04-08,7408.25
"Burkman Builders",3433,2012-04-14,9471.58
"Burkman Builders",3436,2012-04-20,1327.64
"Burkman Builders",3441,2012-04-30,2113.96
"Burkman Builders",3447,2012-05-01,3022.5
"Burkman Builders",3455,2012-05-02,6111.26
"Burkman Builders",3459,2012-05-03,7927.43
"Burkman Builders",3463,2012-05-03,1543.1
"Burkman Builders",3467,2012-05-03,3975.09
"Burkman Builders",3474,2012-05-04,1662.54
"Burkman Builders",3480,2012-05-12,4842.87
"Burkman Builders",3485,2012-05-12,9872.96
"Burkman Builders",3488,2012-05-20,2002.62
"Burkman Builders",3494,2012-05-26,5134.67
"Burkman Builders",3499,2012-05-27,1413.86
"Burkman Builders",3502,2012-05-27,1129.89
"Burkman Builders",3508,2012-05-27,3470.49
"Burkman Builders",3514,2012-06-06,1145.07
"Burkman Builders",3518,2012-06-07,4735.61
"Burkman Builders",3522,2012-06-07,4303.35
"Burkman Builders",3529,2012-06-15,1879.27
"Burkman Builders",3531,2012-06-15,4444.97
"Burkman Builders",3538,2012-06-16,7863.46
"Burkman Builders",3544,2012-06-24,4912.79
"Burkman Builders",3550,2012-06-25,2729.7
"Burkman Builders",3553,2012-07-01,8943.89
"Burkman Builders",3559,2012-07-07,1475.17
"Burkman Builders",3562,2012-07-08,3642.16
"Burkman Builders",3568,2012-07-14,2060.58
"Burkman Builders",3574,2012-07-20,5799.92
"Burkman Builders",3579,2012-07-21,6064.13
"Burkman Builders",3582,2012-07-31,4382.2
"Burkman Builders",3587,2012-07-31,1046.97
"Burkman Builders",3591,2012-08-08,1337.16
"Burkman Builders",3599,2012-08-08,3052.75
"Burkman Builders",3605,2012-08-14,2740.49
"Burkman Builders",3610,2012-08-14,5801.24
"Burkman Builders",3615,2012-08-14,3386.53
"Burkman Builders",3620,2012-08-15,1510.77
"Burkman Builders",3625,2012-08-23,1319.54
"Burkman Builders",3627,2012-08-29,6450.09
"Burkman Builders",3631,2012-09-04,3414.74
"Burkman Builders",3639,2012-09-04,2320.41
"Burkman Builders",3641,2012-09-14,1408.88
"Burkman Builders",3647,2012-09-24,8554.33
"Burkman Builders",3654,2012-09-24,1722.83
"Burkman Builders",3658,2012-10-04,2355.06
"Burkman Builders",3664,2012-10-04,5828.2
"Burkman Builders",3670,2012-10-10,1832.77
"Burkman Builders",3672,2012-10-10,2824.89
"Burkman Builders",3677,2012-10-16,1330.89
"Burkman Builders",3684,2012-10-22,9450.33
"Burkman Builders",3686,2012-10-30,1620.4
"Burkman Builders",3694,2012-10-31,1843.91
"Burkman Builders",3700,2012-11-06,1985.12
"Burkman Builders",3704,2012-11-06,3671.03
"Burkman Builders",3709,2012-11-16,6258.78
"Burkman Builders",3714,2012-11-16,1498.05
"Burkman Builders",3718,2012-11-17,1770.07
"Burkman Builders",3723,2012-11-17,3762.7
"Burkman Builders",3729,2012-11-23,1396.16
"Burkman Builders",3734,2012-11-24,1069.85
"Burkman Builders",3738,2012-11-30,4871.89
"Burkman Builders",3745,2012-12-06,1587.45
"Burkman Builders",3750,2012-12-07,1624.57
"Burkman Builders",3752,2012-12-13,5046.03
"Burkman Builders",3760,2012-12-13,1747.18
"Burkman Builders",3763,2012-12-19,1736.49
"Burkman Builders",3769,2012-12-29,2817.7
"Burkman Builders",3772,2012-12-30,8517.36
"Burkman Builders",3780,2012-12-31,1938.34
"Burkman Builders",3783,2012-12-31,1449.03
"Burkman Builders",3787,2013-01-01,6556.89
"Burkman Builders",3795,2013-01-01,2048.12
"Burkman Builders",3799,2013-01-02,5791.27
"Burkman Builders",3804,2013-01-02,8618.73
"Burkman Builders",3807,2013-01-08,9382.85
"Burkman Builders",3812,2013-01-14,4986.0
"Burkman Builders",3816,2013-01-20,6878.74
"Burkman Builders",3823,2013-01-30,2365.81
"Burkman Builders",3830,2013-01-30,2736.18
"Burkman Builders",3832,2013-01-31,2737.35
"Burkman Builders",3839,2013-02-10,1455.65
"Burkman Builders",3843,2013-02-11,5412.57
"Burkman Builders",3848,2013-02-21,8306.27
"Burkman Builders",3853,2013-02-22,8705.18
"Burkman Builders",3858,2013-02-22,1033.67
"Burkman Builders",3861,2013-02-23,2638.51
"Burkman Builders",3868,2013-03-03,8245.96
"Burkman Builders",3872,2013-03-03,6562.13
"Burkman Builders",3880,2013-03-11,2779.34
"Burkman Builders",3881,2013-03-12,4526.06
"Burkman Builders",3889,2013-03-12,4747.37
"Burkman Builders",3892,2013-03-12,1400.11
"Burkman Builders",3900,2013-03-13,2533.57
"Burkman Builders",3902,2013-03-19,2635.9
"Burkman Builders",3907,2013-03-20,2341.18
"Burkman Builders",3915,2013-03-26,4143.16
"Burkman Builders",3919,2013-04-05,6893.39
"Burkman Builders",3921,2013-04-15,1353.52
"Burkman Builders",3929,2013-04-25,2350.78
"Burkman Builders",3933,2013-04-26,3578.01
"Burkman Builders",3938,2013-04-26,2192.51
"Burkman Builders",3942,2013-05-02,1120.31
"Burkman Builders",3948,2013-05-02,8134.43
"Burkman Builders",3955,2013-05-02,8209.99
"Burkman Builders",3960,2013-05-02,3415.27
"Burkman Builders",3962,2013-05-02,2509.6
"Burkman Builders",3968,2013-05-12,3021.47
"Burkman Builders",3971,2013-05-13,2065.38
"Burkman Builders",3976,2013-05-13,1504.79
"Burkman Builders",3981,2013-05-19,8641.45
"Burkman Builders",3990,2013-05-19,5593.12
"Burkman Builders",3995,2013-05-25,2805.72
"Burkman Builders",3998,2013-05-25,1614.55
"Burkman Builders",4003,2013-05-25,1470.48
"Burkman Builders",4010,2013-06-04,1576.17
"Burkman Builders",4014,2013-06-10,5265.62
"Burkman Builders",4018,2013-06-16,6473.28
"Burkman Builders",4024,2013-06-16,5470.54
"Burkman Builders",4026,2013-06-24,5979.36
"Burkman Builders",4031,2013-06-25,7577.39
"Burkman Builders",4036,2013-06-25,4287.2
"Burkman Builders",4036,2013-06-25,2674.12
"Burkman Builders",4036,2013-07-01,4344.67
"Burkman Builders",4051,2013-07-01,8734.57
"Burkman Builders",4059,2013-07-02,2439.71
"Burkman Builders",4062,2013-07-03,1391.73
"Burkman Builders",4070,2013-07-03,8859.02
"Burkman Builders",4075,2013-07-09,7954.0
"Burkman Builders",4080,2013-07-17,1780.53
"Burkman Builders",4083,2013-07-23,3170.96
"Burkman Builders",4090,2013-07-23,1441.39
"Burkman Builders",4092,2013-07-23,3077.64
"Burkman Builders",4096,2013-07-29,1715.87
"Burkman Builders",4104,2013-07-30,1759.47
"Burkman Builders",4109,2013-07-31,5940.21
"Burkman Builders",4112,2013-07-31,6064.02
"Burkman Builders",4116,2013-07-31,1358.43
"Burkman Builders",4121,2013-07-31,1162.49
"Burkman Builders",4129,2013-08-10,1493.85
"Burkman Builders",4132,2013-08-16,1106.03
"Burkman Builders",4139,2013-08-16,1858.92
"Burkman Builders",4142,2013-08-24,1689.76
"Burkman Builders",4149,2013-08-25,2669.94
"Burkman Builders",4153,2013-08-25,2740.88
"Burkman Builders",4160,2013-08-31,1618.7
"Burkman Builders",4160,2013-09-10,1825.0
"Burkman Builders",4160,2013-09-16,6298.58
"Burkman Builders",4171,2013-09-16,6570.31
"Burkman Builders",4177,2013-09-17,7014.26
"Burkman Builders",4181,2013-09-17,1835.81
"Burkman Builders",4186,2013-09-25,1152.41
"Burkman Builders",4191,2013-09-25,2134.35
"Burkman Builders",4197,2013-09-25,2407.67
"Burkman Builders",4201,2013-10-01,2634.26
"Burkman Builders",4208,2013-10-11,3816.63
"Burkman Builders",4215,2013-10-17,2081.64
"Burkman Builders",4220,2013-10-17,1259.95
"Burkman Builders",4224,2013-10-17,7515.4
"Burkman Builders",4227,2013-10-18,1178.39
"Burkman Builders",4234,2013-10-26,1876.87
"Burkman Builders",4236,2013-11-03,1405.12
"Burkman Builders",4242,2013-11-03,9904.41
"Burkman Builders",4248,2013-11-13,9866.15
"Burkman Builders",4253,2013-11-19,4868.68
"Burkman Builders",4257,2013-11-25,1814.73
"Burkman Builders",4265,2013-12-01,3753.98
"Burkman Builders",4270,2013-12-01,3811.7
"Burkman Builders",4274,2013-12-07,9577.75
"Burkman Builders",4277,2013-12-07,2784.23
"Burkman Builders",4285,2013-12-15,8869.9
"Burkman Builders",4289,2013-12-21,3354.63
"Burkman Builders",4293,2013-12-27,1560.81
"Burkman Builders",4299,2013-12-27,1231.5
"Burkman Builders",4302,2014-01-04,8345.1
"Burkman Builders",4308,2014-01-10,6528.57
"Burkman Builders",4312,2014-01-11,1797.61
"Burkman Builders",4319,2014-01-21,3591.46
"Burkman Builders",4322,2014-01-27,1992.02
"Burkman Builders",4328,2014-01-28,1033.1
"Burkman Builders",4334,2014-01-29,1213.45
"Burkman Builders",4338,2014-01-30,1578.56
"Burkman Builders",4344,2014-01-30,4785.26
"Burkman Builders",4347,2014-01-30,3156.63
"Burkman Builders",4354,2014-01-30,1074.39
"Burkman Builders",4358,2014-01-31,1621.44
"Burkman Builders",4365,2014-01-31,4296.63
"Burkman Builders",4367,2014-02-06,9052.15
"Burkman Builders",4373,2014-02-07,5725.12
"Burkman Builders",4380,2014-02-13,1113.07
"Burkman Builders",4383,2014-02-21,1390.95
"Burkman Builders",4387,2014-02-21,1242.19
"Burkman Builders",4392,2014-02-22,6359.63
"Burkman Builders",4397,2014-02-22,1614.89
"Burkman Builders",4401,2014-02-22,7612.67
"Burkman Builders",4410,2014-02-22,1485.14
"Burkman Builders",4413,2014-03-04,2472.24
"Burkman Builders",4419,2014-03-05,6375.38
"Burkman Builders",4421,2014-03-11,8485.31
"Burkman Builders",4426,2014-03-17,2392.95
"Burkman Builders",4431,2014-03-18,3555.53
"Burkman Builders",4440,2014-03-24,8900.08
"Burkman Builders",4444,2014-04-01,1097.2
"Burkman Builders",4447,2014-04-01,1439.0
"Burkman Builders",4454,2014-04-02,1252.5
"Burkman Builders",4457,2014-04-12,3356.04
"Burkman Builders",4462,2014-04-20,8344.33
"Burkman Builders",4469,2014-04-20,4755.56
"Burkman Builders",4474,2014-04-21,4238.5
"Burkman Builders",4478,2014-04-21,3686.44
"Burkman Builders",4485,2014-04-27,4341.55
"Burkman Builders",4489,2014-04-28,3330.47
"Burkman Builders",4493,2014-04-28,7676.79
"Burkman Builders",4499,2014-05-08,3080.92
"Burkman Builders",4501,2014-05-08,4175.09
"Burkman Builders",4507,2014-05-16,2943.09
"Burkman Builders",4513,2014-05-22,3058.84
"Burkman Builders",4517,2014-05-23,2659.15
"Burkman Builders",4522,2014-05-23,1871.52
"Burkman Builders",4528,2014-05-23,1402.09
"Burkman Builders",4531,2014-05-23,4177.28
"Burkman Builders",4539,2014-05-24,3058.92
"Burkman Builders",4544,2014-05-24,3379.83
"Burkman Builders",4547,2014-05-24,2304.22
"Burkman Builders",4553,2014-05-25,2744.47
"Burkman Builders",4556,2014-06-04,1115.07
"Burkman Builders",4562,2014-06-05,1691.32
"Burkman Builders",4567,2014-06-15,3659.76
"Burkman Builders",4571,2014-06-15,7173.06
"Burkman Builders",4576,2014-06-21,4003.71
"Burkman Builders",4581,2014-07-01,3243.27
"Burkman Builders",4589,2014-07-09,8902.69
"Burkman Builders",4592,2014-07-10,1888.65
"Burkman Builders",4596,2014-07-11,1077.64
"Burkman Builders",4603,2014-07-12,7229.15
"Burkman Builders",4606,2014-07-18,5082.97
"Burkman Builders",4611,2014-07-28,1764.6
"Burkman Builders",4619,2014-07-28,1660.82
"Burkman Builders",4624,2014-08-03,9439.62
"Burkman Builders",4627,2014-08-04,5100.04
"Burkman Builders",4634,2014-08-04,8195.86
"Burkman Builders",4638,2014-08-04,3740.32
"Burkman Builders",4642,2014-08-05,1118.98
"Burkman Builders",4649,2014-08-05,1478.11
"Burkman Builders",4653,2014-08-05,1441.59
"Burkman Builders",4656,2014-08-05,4249.16
"Burkman Builders",4662,2014-08-06,2252.36
"Burkman Builders",4669,2014-08-14,1382.05
"Burkman Builders",4673,2014-08-20,5011.04
"Burkman Builders",4678,2014-08-28,4797.86
"Burkman Builders",4684,2014-09-07,4114.44
"Burkman Builders",4688,2014-09-17,2755.55
"Burkman Builders",4693,2014-09-25,3770.89
"Burkman Builders",4700,2014-10-03,4422.47
"Burkman Builders",4702,2014-10-11,9792.22
"Burkman Builders",4708,2014-10-12,1433.72
"Burkman Builders",4715,2014-10-13,1560.79
"Burkman Builders",4716,2014-10-14,3131.24
"Burkman Builders",4721,2014-10-24,2947.15
"Burkman Builders",4726,2014-10-24,5284.78
"Burkman Builders",4735,2014-10-25,1585.3
"Burkman Builders",4740,2014-10-31,7934.93
"Burkman Builders",4743,2014-11-06,4430.0
"Burkman Builders",4750,2014-11-14,2826.43
"Burkman Builders",4754,2014-11-14,1286.55
"Burkman Builders",4760,2014-11-20,5061.44
"Burkman Builders",4764,2014-11-20,1871.06
"Burkman Builders",4768,2014-11-21,1757.42
"Burkman Builders",4771,2014-11-21,2234.15
"Burkman Builders",4779,2014-11-21,4841.83
"Burkman Builders",4781,2014-11-27,2924.05
"Burkman Builders",4786,2014-12-03,2328.41
"Burkman Builders",4795,2014-12-03,5856.85
"Burkman Builders",4797,2014-12-03,3066.86
"Burkman Builders",4804,2014-12-09,6293.76
"Burkman Builders",4809,2014-12-10,2362.25
"Burkman Builders",4812,2014-12-10,1661.07
"Burkman Builders",4819,2014-12-18,3348.81
"Burkman Builders",4824,2014-12-24,1128.33
"Burkman Builders",4828,2014-12-30,1744.94
"Burkman Builders",4833,2014-12-30,1235.45
"Burkman Builders",4839,2014-12-31,9165.25
"Burkman Builders",4843,2015-01-01,1814.0
"Burkman Builders",4849,2015-01-09,8938.96
"Burkman Builders",4851,2015-01-09,3288.76
"Burkman Builders",4851,2015-01-09,5673.73
"Burkman Builders",4864,2015-01-19,6230.24
"Burkman Builders",4867,2015-01-20,4589.23
'''  