####################################################################################
#
# Copyright (c) 2008 Matt Hillary <matt.hillaryATgmailDOTcom               
# 
# This is a detectlet to detect a type of Procurement Fraud where "Cash" is the name 
# or contained in the name of a vendor.
#
# The following is an explanation of this fraud given by the following website: 
# http://www.nysscpa.org/cpajournal/2005/405/essentials/p58.htm.
#
# Fraudsters may establish a fictitious vendor using "cash" in the vendor's name, 
# to make it easier to cash checks. This detectlet searches the table containing 
# vendor information and identifies "cash" in the name. 
#
#                                                                                					
####################################################################################
# UPDATES
#
#
# STATUS: Reviewing
# 
# 
# 
# IDEAS/QUESTIONS/NOTES:
#
#
####################################################################################
DETECTLET_STANDARD = 1.0

from picalo import *
import string

wizard = '''
<wizard>
  <page>
    Select the TABLE containing VENDOR INFORMATION.
    
    <parameter type="Table" variable="table"/>
  </page>
  <page>
    Which COLUMN identifies the NAME of the vendor?
    
    <parameter type="Column" table="table" variable="name_column"/>

	Which COLUMN identifies the ID of the vendor?

    <parameter type="Column" table="table" variable="id_column"/>

   </page>
 </wizard>
'''

RESULTS_TEXT = '''The displayed table lists every vendor that has the name containing the word "Cash."
 Vendor ID shows the ID of the vendor.
 Vendor Name shows the name of the vendor.
'''


def run(table, name_column, id_column):
  '''Procurement Fraud can occur when a vendor's name is "Cash" or contains the word "Cash." Having
     a name "Cash" allows for anyone to cash a check directly. Similarly, a check having a name 
	 containing the word "Cash" is easier for someone to cash.
	      
     This wizard analyzes the selected vendor table for vendor names containing the word Cash.
     
     The wizard goes through the following process:

	 - Searches the Name column of the table and column selected by the user

	 - Creates a new table containing the ID and Name of the vendor contain the string "Cash".
  '''
  
  # run the analysis
  results = Table([
    ( 'VendorID',                 int),
    ( 'VendorName',                unicode),
  ])
  
  counter = 0
  
  for vendor in table:

    if not str(table[counter][name_column]).find("Cash") == -1:
        rec = results.append()
        rec['VendorID'] = table[counter][id_column]
        rec['VendorName'] = table[counter][name_column]

    counter += 1
    show_progress('Analyzing...', float(counter) / len(name_column))
    
  return results, RESULTS_TEXT

  
def example_input():
  import StringIO  # to emulate a file for load_csv
  table = load_csv(StringIO.StringIO(csvdata))
  table.set_type('VendorNumber', int)
  table.set_type('VendorName', unicode)
  table.set_type('VendorAddress', unicode)
  table.set_type('VendorPhone', unicode)
  table.set_type('VendorTIN', unicode)
  return table



# This next part is not required, but it's easier to put the example data
# directly in this file so I don't have to worry about directories.
csvdata = '''\
"VendorNumber","VendorName","VendorAddress","VendorPhone","VendorTIN"
1556,"The Cash Company","465 Eagle Street","8913138181","890-48-5218"
1902,"Pearson Industries","2133 Central Avenue","1245848585","17-1221544"
1904,"Southeast Company","1010 Potter Road","1241848618","18-1544885"
1910,"National Manufacturing, Inc.","4494 University Rd.","1085841811","28-5665488"
1915,"Quality Aluminum Products","1903 Decatur St.","9416542109","31-8633567"
1920,"Central Supply, Inc.","270 Airline Rd.","8941568941","455-45-1648"
1928,"Aurora Information Services","5483 North Bunker Hill Street","4156489156","49-4813862"
1929,"LEP Marketing","1005 Thorton Street",,"49-4186578"
1935,"Long Corp.","2002 Wolfe Ave.","8941561489","70-4589488"
1938,"Keasler Company","5714 North Lion Street","4985697845","20-4188843"
1940,"Lewis Manufacturing, Inc.","P.O. Box 754","9879456984","544-15-4885"
1941,"Knight Corporation","310 Richardson Rd.","1564891563","08-1890214"
1947,"Ridgeland Deli","1808 Washington Avenue","1561894816","09-1948942"
1950,"Connie's Catering Service","1209 Patrick Avenue","1569787916","17-8431231"
1952,"CCS Consulting","377 Alabama Street","8978964894","16554"
1958,"Office SuperCenter, Inc.","Metro Mall Suite #23","8899848948","69-4885118"
1965,"Stark Information Services","4403 17th Avenue","4894988989","08-8909159"
1971,"Polk Publications, Inc.","1661 Main Street","8978163438","AL-5324322"
1977,"Cunnigham Manufacturing, Inc.","1099 East Greentree Lane","1318148131","17-1884315"
1980,"Davis Corporation","3829 Old Yorkville Ave.","8974916159","20-9159888"
1983,"Dailey Supply Company","3430 North Montgomery Road","1946316848","69-9021348"
1989,"Hunt Consulting","4796 Gipson St.","8198423186","20-9841184"
2000,"PPC Communication, Inc.","3951 North Rd.","8918913188","488-51-8489"
2004,"Bryant Catering Service","3959 East Ave.","8191684684","20-3116489"
2015,"Cash","1245 E. Patterson Lane","1951918613","26-2184433"
2018,"Green Corporation","993 Main Street","8948416364","08-3154898"
2024,"Franklin Manufacturing, Inc.","1599 153nd Street","5191318569","17-8901148"
2028,"Grace Consulting","2519 Wood Rd.","8519651561","69-1881566"
2043,"Sylvan, O'Brien, Yeats & Smith","4900 10th St.","5161851351","908-84-2385"
2046,"Strategic Planning Group","2651 Lakeside Road","1561681891","17-8439159"
2049,"PBL Industries","P. O. Box 92","8616984199","17-0621198"
2052,"PDC Consulting","PO Box 2423","1818618618","08-0561895"
2054,"Seville Manufacturing, Inc.","424 Mill St.","8948918591","20-4885448"
2055,"Internet Consulting Group, LLC","P. O. Box 653","1861681681","902-11-1594"
2061,"Leigh Manufacturing, Inc.","2557 Manor Road","6189131866","69-1818648"
2076,"Jefferson & Pearson, CPAs","3856 Oak Lane","5198419509","843-68-7893"
2088,"New Orleans Manufacturing, Inc.","River Plaza Suite B","1981815631","17-5468778"
2103,"Bay Area Aluminum","4561 Shelby Dr.","7276542198","21-6542102"
2191,"Yo Mamas Cash","1729 Coral Dr.","7276549999","26-6542102"
2192,"Cash it Here","999 Cash Dr.","4546542198","61-6542102"
2193,"Paper Supply, Co.","64 El Papel Dr.","5346542198","21-4444102"
2194,"Candy Machine Shop","333 Pingguo St.","3254342198","21-6543202"
2195,"Cash","666 Ghost Lane","7276543243","51-6542302"
'''

