#!/usr/bin/env python

####################################################################################
#                                                                                  #
# Copyright (c) 2005 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This file is part of Picalo.                                                     #
#                                                                                  #
# Picalo is free software; you can redistribute it and/or modify                   #
# it under the terms of the GNU General Public License as published by             #
# the Free Software Foundation; either version 2 of the License, or                # 
# (at your option) any later version.                                              #
#                                                                                  #
# Picalo is distributed in the hope that it will be useful,                        #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                   #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    #
# GNU General Public License for more details.                                     #
#                                                                                  #
# You should have received a copy of the GNU General Public License                #
# along with Foobar; if not, write to the Free Software                            #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA        #
#                                                                                  #
####################################################################################
#
# 12 Sep 2005 First version
#
####################################################################################


import wx, os.path, re, sys, types, inspect, time, imp, traceback
from StringIO import StringIO
from picalo import *
from picalo import global_variables, global_modules, python_modules
import Preferences
from Languages import lang
from picalo.base import Global

RE_POS = re.compile('^(\d+)\|(.*)$')

component_id = 10000

###############################################
###  Helper function to make menus and toolbars

class MenuItem:
  '''A menu option.  Use this class to encode the options on your menu.'''
  def __init__(self, namepath, description='', callback=None, id=None, kind = wx.ITEM_NORMAL, enabled=True, checked=False):
    global component_id
    self.namepath = namepath        # the name path, separated by _, such as "&File_&New_&Table" or "&FILE_SEPARATOR" for a separator
    self.callback = callback        # the callback function
    self.description = description  # the description (shows in status bar)
    self.kind = kind                # the kind of item: wx.ITEM_NORMAL, wx.CHECK, wx.RADIO
    self.enabled = enabled          # inital enabled state of the item
    self.checked = checked          # initial checked state of the item
    self.id = id                    # the id of the element, or None for an automatic id
    if not self.id:
      component_id += 1
      self.id = component_id


class MenuManager:
  '''
  These classes allow you to send in a simple list containing the menu
  items to be created, and it creates them.  The topmenu item can be 
  a wx.MenuBar object (for frames) or a wx.Menu() object (for popup menus).
  
  In addition, the classes support the idea of a menu stack, where
  additional items can be placed on the menu in sets.  For example,
  an application will likely start with a simple, base menu that is visible
  at all times in the application (with items like Help|About).  Additional
  sets of commands can be pushed onto the menu at any time, and the class will
  adjust the menu to reflect new items.  For example, if a user opened an
  editor window, new options for Edit, File, and Search could be pushed
  onto the menu stack.  When the editor is closed, a simple call to pop()
  removes the items and returns to the previous menu.
  
  '''
  def __init__(self, parent, topmenu, menu_list):
    '''Constructor'''
    self.parent = parent
    self.topmenu = topmenu
    self.changestack = []
    self.menuobjects = {}
    
    # create the base menu
    self.push(menu_list)
    
  
  def Destroy(self):
    '''Destroyes the menu associated with this manager'''
    if self.topmenu:
      self.topmenu.Destroy()
    
  
  def getStackSize(self):
    '''Returns the size of the toolbar stack'''
    return len(self.changestack)
    

  def push(self, menu_list):
    '''Pushes a menu change onto the existing menu structure'''
    # create the new menu objects
    global component_id
    changes = []
    for item in menu_list:
      # first go through and add all the parent items for submenus (they are implied in the menu names)
      names = item.namepath.split('/')
      parent = self.topmenu
      for i in range(len(names) - 1):
        nametohere = tuple(names[:i+1])
        if self.menuobjects.has_key(nametohere):  # have we already created this menu?
          parent = self.menuobjects[nametohere]
        else: # need to create the menu
          name = names[i]
          pos = -1
          match = RE_POS.match(name)
          if match:
            pos = int(match.group(1))
            name = match.group(2)
          if i == 0: # topmenu level
            parent = wx.Menu()
            if pos >= 0:
              child = self.topmenu.Insert(pos, parent, name)
            else:
              child = self.topmenu.Append(parent, name)
            changes.append((nametohere, self.topmenu, parent, child))
          else:
            menu = wx.Menu()
            component_id += 1
            if pos >= 0:
              child = parent.InsertMenu(pos, component_id, name, menu, name)
            else:
              child = parent.AppendMenu(component_id, name, menu, name)
            changes.append((nametohere, parent, menu, child))
            parent = menu
          self.menuobjects[nametohere] = parent
          
      # now append the leaf item (the actual menu item)
      name = names[-1]
      if name != '':  # might be a non-leaf item that leaves will be added to later
        pos = -1
        match = RE_POS.match(name)
        if match:
          pos = int(match.group(1))
          name = match.group(2)
        if item.namepath[-9:] == 'SEPARATOR':
          if pos >= 0:
            separator = parent.InsertSeparator(pos)
          else:
            separator = parent.AppendSeparator()
          changes.append((item.namepath, parent, None, separator))
          self.menuobjects[item.namepath] = separator
        else:
          mi = wx.MenuItem(parent, item.id, name, item.description, item.kind)
          mi.Enable(item.enabled)
          if pos >= 0:
            parent.InsertItem(pos, mi)
          else:
            parent.AppendItem(mi)
          changes.append((item.namepath, parent, None, mi))
          self.menuobjects[item.namepath] = mi
          if item.callback:
            wx.EVT_MENU(self.parent, item.id, item.callback)
          
    # add these changes to the change stack
    self.changestack.append(changes)
    
    
  def pop(self):
    '''Pops the most recent pushed elements (those done in a single push call) to return to the menu before the push'''
    # you have to have at least one thing on the stack
    if len(self.changestack) < 1:
      return
      
    # get the last changes
    changes = self.changestack.pop()
    changes.reverse()  # go in reverse order through it
    for fullname, parent, menu, child in changes:  # I have to do this if statement because the methods are different on each level (bad OO wx!)
      if self.menuobjects.has_key(fullname):
        del self.menuobjects[fullname]

      if str(parent.this) == str(self.topmenu.this):  # a topmenu item, the __eq__ of "this" seems broken.  converting to strings fixes it and allows comparison
        for i in range(self.topmenu.GetMenuCount()):
          if str(self.topmenu.GetMenu(i).this) == str(child.this):
            parent.Remove(i)
            break

      else:
        parent.DeleteItem(child)


    
##########################
###   Toolbars
    
class ToolbarItem:
  '''A toolbar item.  Use this class to encode the options for your toolbar'''
  def __init__(self, name, icon=None, pushedicon=None, shorthelp='', longhelp='', callback=None, kind=wx.NORMAL, enabled=True, checked=False):
    self.name = name                # the name of the tool, or "SEPARATOR" for spacing
    self.icon = icon                # the primary icon filename, will be loaded with Utils.getBitmap
    self.pushedicon = pushedicon    # the pushed icon filename for wx.CHECK kinds, will be loaded with Utils.getBitmap
    self.shorthelp = shorthelp      # the help string for the tooltip, or the item name if blank
    self.longhelp = longhelp        # the help string for the status bar, or the item shorthelp or name if blank
    self.callback = callback        # the callback function
    self.kind = kind                # the kind of item: wx.ITEM_NORMAL, wx.CHECK, wx.RADIO
    self.enabled = enabled          # inital enabled state of the item
    self.checked = checked          # initial checked state of the item (pushed or not)



def create_toolbar(mainframe, window_tool_list=[]):
  '''Pushes a toolbar change onto the existing toolbar structure'''
  # create the new menu objects
  global component_id
  toolbar = wx.ToolBar(mainframe.toolbarpanel, style=wx.TB_HORIZONTAL | wx.NO_BORDER | wx.TB_FLAT)  
  for tool_list in ( mainframe.default_tool_list, window_tool_list ):
    for item in tool_list:
      if item.name[-9:] == 'SEPARATOR':
        pos = -1
        match = RE_POS.match(item.name)
        if match:
          pos = int(match.group(1))
        if pos >= 0:
          toolbar.InsertSeparator(pos)
        else:
          id = toolbar.AddSeparator()
      else:
        filename = item.icon
        pos = -1
        match = RE_POS.match(filename)
        if match:
          pos = int(match.group(1))
          filename = match.group(2)
        icon1 = getBitmap(filename)
        icon2 = getBitmap(item.icon)
        component_id += 1
        if pos >= 0:
          tool = toolbar.InsertTool(pos, component_id, icon1, icon2, shortHelpString=item.shorthelp or item.name, longHelpString=item.longhelp or item.shorthelp or item.name)
          toolbar.EnableTool(pos, item.enabled)
        else:  
          tool = toolbar.AddTool(component_id, icon1, icon2, shortHelpString=item.shorthelp or item.name, longHelpString=item.longhelp or item.shorthelp or item.name)
          toolbar.EnableTool(component_id, item.enabled)
        if item.callback:
          toolbar.Bind(wx.EVT_TOOL, item.callback, id=component_id)
  toolbar.Realize()  
  return toolbar
    


class MenuCaller:
  '''Calls a function with given data when a menu item is selected'''
  def __init__(self, func, *data):
    self.func = func
    self.data = data
  
  def __call__(self, event=None):
    self.func(*self.data)


#####################################################################
###   Progress dialog -- this is called from picalo.base.Global
###   when Picalo is running in GUI mode

progressDialog = None

class OperationCancelledError(Exception):
  '''A simple exception to indicate that an operation was cancelled'''
  pass

def _updateProgressDialog(msg='', progress=1.0, title='Progress', parent=None):
  '''Initializes the progress bar.  This should not be called directly.
     Call show_progress() instead, which gets imported when you run
     "from picalo import *".
     '''
  global progressDialog
  # if progress is done, hide the dialog
  if progress >= 1.0 or progress < 0.0:
    if progressDialog:
      progressDialog.SetCursor(wx.NullCursor)
      if parent:
        parent.SetCursor(wx.NullCursor)
      progressDialog.Destroy()
    progressDialog = None
    return

  # create the progress dialog if necessary
  if not progressDialog:
    progressDialog = wx.ProgressDialog(title, msg, maximum=100, style=wx.PD_APP_MODAL | wx.PD_ELAPSED_TIME | wx.PD_REMAINING_TIME | wx.PD_CAN_ABORT)

  # update the progress, returns (continue, skip) where continue is False if cancel was pressed
  cont = progressDialog.Update(int(progress*100.0), msg)
  if not cont[0]:
    progressDialog.Resume()  # must reenable before raising error to stop recursion that occurs because finally clauses are run (which usually call clear_progress)
    if wx.MessageBox('Cancel this operation?', 'Cancel', style=wx.YES_NO) == wx.YES:
      raise OperationCancelledError()




############################
###   Directory loader

# get the resource directory
# it's in the "resources" dir relative to this file location
APPDIR = sys.path[0]
if os.path.isfile(APPDIR):  #py2exe
  APPDIR = os.path.dirname(APPDIR)
resource_dir = os.path.join(APPDIR, 'picalo', 'resources')
wizard_dir = os.path.join(APPDIR, 'picalo', 'gui', 'wizards')
plugin_dir = os.path.join(APPDIR, 'picalo', 'tools')

# a cache of resources
bitmap_cache = {}

def getAppDir():
  '''Returns the application directory'''
  return APPDIR
  

def getResourcePath(name):
  '''Returns the absolute resource path given a relative path
     in the resources directory.
  '''
  return os.path.join(resource_dir, name)
  

def getWizardPath(name):
  '''Returns the absolute resource path given a relative path
     in the wizards directory.'''
  return os.path.join(wizard_dir, name)       
  
  
def getLanguages():
  '''Returns the supported languages for this picalo installation'''
  languages = []
  for file in os.listdir(resource_dir):
    name, ext = os.path.splitext(file)
    if ext == '.language':
      languages.append(name)
  languages.sort()
  return languages
  
  
def getPlugins():
  '''Returns the plugins that are installed to the tools directory'''
  plugins = []
  for modname in os.listdir(plugin_dir):
    if not modname[:1] in ('.', '_'):
      info = imp.find_module(modname, [plugin_dir])
      try:
        mod = imp.load_module(modname, info[0], info[1], info[2])
        plugins.append(mod)
      except Exception, e:
        print >> sys.stderr, 'Error importing ' + modname + ' plugin: ' + str(e)
        traceback.print_exc(file=sys.__stderr__)
  return plugins
  

def getBitmap(name):
  '''Retrieves an image from the resources directory by filename.  
     This is always relative to the application directory.
     Returns a wx.Bitmap object.
  '''
  # if name is empty, simply return
  filename = getResourcePath(name)
  if not name or not os.path.exists(filename):
    raise lang('Could not find bitmap:'), filename
    
  # put in the cache if needed
  if not bitmap_cache.has_key(name):
    img = wx.Image(filename)
    bitmap_cache[name] = wx.BitmapFromImage(img)
    
  # return the resource
  return bitmap_cache[name]
  
  
def getIcon(name):
  '''Retrieves an image from the resources directory by filename.
     This is always relative to the application directory.
     Returns a wx.Icon object.
  '''
  icon = wx.EmptyIcon()
  bmp = getBitmap(name)
  icon.CopyFromBitmap(bmp)
  return icon
  
  
def getHomeDir(): 
    ''' Try to find user's home directory, otherwise return current directory.''' 
    # thanks to Nemesis on usenet for this algorithm #
    try: 
        path1=os.path.expanduser("~") 
    except: 
        path1="" 
    try: 
        path2=os.environ["HOME"] 
    except: 
        path2="" 
    try: 
        path3=os.environ["USERPROFILE"] 
    except: 
        path3="" 

    if not os.path.exists(path1): 
        if not os.path.exists(path2): 
            if not os.path.exists(path3): 
                return os.getcwd() 
            else: return path3 
        else: return path2 
    else: return path1  
  
  
  
#########################################################################
###   Functions and classes for getting documentation from functions

RE_FIRSTSPACES = re.compile('^( *)')
RE_SPACES = re.compile(' +')
RE_PARAM = re.compile('^@param +(.+?): +(.*)$')
RE_TYPE = re.compile('^@type  +(.+?): +(.*)')
RE_RETURN = re.compile('^@return: +(.*)$')
RE_RTYPE = re.compile('^@rtype: +(.*)$') 
RE_EXAMPLE_LINE = re.compile('^( *)Example( \d+){0,1}:', re.IGNORECASE)
RE_PARAM_LINE = re.compile('^( *)[@!]')

class Parameter:
  '''An individual parameter'''
  def __init__(self, type, name, desc, required=True, default=None):
    self.type = type
    self.name = name
    self.desc = desc
    self.required = required
    self.default = default


class Documentation:
  '''Documentation for a function'''
  def __init__(self, module, name, methodname, func):
    '''Parses the documentation for the given function'''
    self.module = module
    self.name = name
    self.func = func
    self.methodname = methodname
    self.desc = []         # each paragraph of description text
    self.examples = []       # zero or more examples
    self.params = []         # the parameters
    self.retval = None       # the return value

    # first remove all the white space at the beginning and ending of each line
    section = 0
    exampletab = '0'
    params = {}
    returntype = None
    paramtypes = {}
    currentdesc = []
    for line in StringIO(func.__doc__):
      line = line.rstrip()
      
      # see if this line puts us in a different section
      match = RE_EXAMPLE_LINE.search(line)
      if match:
        section = 1  # we're in example mode now
        exampletab = str(len(match.group(1)))  # so we can remove tabs that are not needed
        self.examples.append([])  # for this example
        continue  # don't append the example line
      match = RE_PARAM_LINE.search(line)
      if match:
        section = 2

      # parse the line according to the section we're in        
      if section == 0:    # paragraph
        line = line.strip()
        if line == '':  # add a new paragraph
          text = '\n'.join(currentdesc)
          text = unwrap_paragraphs(text)
          self.desc.append(text)  
          currentdesc = []
        else:
          currentdesc.append(line)
          
      elif section == 1:  # example
        self.examples[-1].append(line)
        
      elif section == 2:  # parameters
        line = line.strip()
        if RE_PARAM.match(line):
          match = RE_PARAM.match(line)
          p = Parameter(None, match.group(1), match.group(2))
          params[p.name] = p
          
        elif RE_TYPE.match(line):
          match = RE_TYPE.match(line)
          paramtypes[match.group(1)] = match.group(2)
          
        elif RE_RETURN.match(line):
          match = RE_RETURN.match(line)
          self.retval = Parameter(None, None, match.group(1))
          
        elif RE_RTYPE.match(line):
          match = RE_RTYPE.match(line)
          returntype = match.group(1)

    # see if we have current description left over
    if len(currentdesc) > 0:
      text = '\n'.join(currentdesc)
      text = unwrap_paragraphs(text)
      self.desc.append(text)  
          
    # go through the parameters and set their types
    for param in params.values():
      if paramtypes.has_key(param.name):
        param.type = paramtypes[param.name]
    if self.retval:
      self.retval.type = returntype

    # go through the parameters now that we have their documentation
    args, vargs, vkargs, defaults = inspect.getargspec(func)
    for i, arg in enumerate(args):
      if i == 0 and arg == 'self':  # don't do self
        continue
      try:
        p = params[arg]
      except KeyError:
        p = Parameter('object', arg, '')  # default documentation
      if defaults and i >= len(args) - len(defaults):  # do we have a default value for this one?
        default = defaults[i - (len(args) - len(defaults))]
        p.required = False
        p.default = default
      self.params.append(p)
        
    # go through the example lists and remove extra white space at the beginning of each one (but preserve indendation of if/for/while/etc statements)
    # this also turns the list of lines into a single string
    for i in range(len(self.examples)):
      exlist = self.examples[i]
      tablevel = min([ len(RE_FIRSTSPACES.search(line).group(1)) for line in exlist if len(line) > 0 ])
      self.examples[i] = '\n'.join([ line[tablevel:] for line in exlist])


  def getFullName(self):
    '''Returns module.name if in a module or name if not'''
    if self.methodname:
      return '[' + self.name + '].' + self.methodname
    if self.module:
      return self.module + '.' + self.name
    return self.name


  def getSignature(self):
    '''Returns the method signature with parameter name and types, return value, etc.'''
    s = []
    
    # return value
    if self.retval:
      s.append('<' + self.retval.type + '>')
      s.append(' = ')
        
    # module and method name
    s.append(self.getFullName())
      
    # parameters
    parms = []
    for p in self.params:
      if p.required:
        parms.append('<%s> %s' % (p.type, p.name))
      elif p.type == 'str' or p.type == 'unicode': 
        parms.append('<%s> %s="%s"' % (p.type, p.name, p.default))
      else:
        parms.append('<%s> %s=%s' % (p.type, p.name, p.default))
    s.append('(' + (', '.join(parms)) + ')')

    # join and return
    return ''.join(s)
    

    
def get_documentation():
  '''Retrieves the documentation into a dictionary of categories and 
     dictionary of functions/classes in each category'''
  # gather the documentation for picalo
  func_categories = {}
  func_categories['Core'] = {}
  
  # inner function to add for different modules
  def add_doc(category, modname, funcname, func):
    if isinstance(func, types.FunctionType):
      # just add the function
      if funcname[:1] != '_': # only do public ones
        if isinstance(func, (types.FunctionType, types.MethodType)):   # only do our explicit methods, not inherited ones
          func_categories[category][funcname] = Documentation(modname, funcname, '', func)
      
    elif isinstance(func, (types.ClassType, types.TypeType)):
      # first add the __init__ constructor
      if isinstance(func.__init__, (types.FunctionType, types.MethodType)):   # only do our explicit methods, not inherited ones
        func_categories[category][funcname] = Documentation(modname, funcname, '', func.__init__)
      # next add each public method of the class
      for methodname in dir(func):
        if methodname[:1] != '_':  # only do public ones
          method = getattr(func, methodname)
          if isinstance(method, (types.FunctionType, types.MethodType)):   # only do our explicit methods, not inherited ones
            func_categories[category][funcname + '.' + methodname] = Documentation(modname, funcname, methodname, method)
        

  # built in stuff from __init__.py  
  for name in global_variables:
    add_doc('Core', '', name, globals()[name])

  # module stuff from different modules
  for name in global_modules:
    mod = globals()[name]
    func_categories[name] = {}
    for funcname in mod.__functions__:
      func = getattr(mod, funcname)
      add_doc(name, name, funcname, func)
      
  # typical python modules people like to use
  for name in python_modules:
    mod = sys.modules[name]
    func_categories[name] = {}
    for funcname in dir(mod):
      if not funcname.startswith('_'):
        func = getattr(mod, funcname)
        add_doc(name, name, funcname, func)
        
  # return the dictionary
  return func_categories
   
    
def unwrap_paragraphs_list(text):
  '''Unwraps paragraphs, taking out extra space and extra hard returns.  Returns a list.'''
  paragraphs = []
  current = ''
  inlist = False
  for line in StringIO(text.strip()):
    line = line.strip()
    line = RE_SPACES.sub(' ', line)    # convert double/triple/... spaces to single space, then add the line to the current paragraph. 
    if line == '':  # a paragraph marker
      paragraphs.append(current)
      current = ''
      inlist = False
    
    elif line[0] == '*':  # a list
      inlist = True
      if len(current) > 0:
        current += '\n'
      current += '  ' + line
    
    else: # regular line
      if inlist:
        current += '\n'
        inlist = False
      elif len(current) > 0:
        current += ' '
      current += line
   
  if len(current) > 0:
    paragraphs.append(current)
     
  return paragraphs
  
  
def unwrap_paragraphs(text):
  '''Unwraps paragraphs, taking out extra space and extra hard returns.  Returns the joined text.'''
  return '\n\n'.join(unwrap_paragraphs_list(text))
  
  


  
  
  