#!/usr/bin/python

####################################################################################
#                                                                                  #
# Copyright (c) 2005 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This file is part of Picalo.                                                     #
#                                                                                  #
# Picalo is free software; you can redistribute it and/or modify                   #
# it under the terms of the GNU General Public License as published by             #
# the Free Software Foundation; either version 2 of the License, or                # 
# (at your option) any later version.                                              #
#                                                                                  #
# Picalo is distributed in the hope that it will be useful,                        #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                   #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    #
# GNU General Public License for more details.                                     #
#                                                                                  #
# You should have received a copy of the GNU General Public License                #
# along with Foobar; if not, write to the Free Software                            #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA        #
#                                                                                  #
####################################################################################

import wx, wx.grid, re
from Languages import lang
from picalo import *
import Dialogs
from picalo.base.Global import is_valid_variable, ensure_unique_list_value, make_valid_variable


WHERETYPE_OPTIONS = [ 
  [ 'equal to',                 '='  ],
  [ 'less than',                '<'  ],
  [ 'less than or equal to',    '<=' ],
  [ 'greater than',             '>'  ],
  [ 'greater than or equal to', '>=' ],
  [ 'not equal to',             '<>' ],
]
GROUPBY_OPTIONS = [
  [ 'Group By', ''      ],
  [ 'Sum',      'sum'   ],
  [ 'Count',    'count' ], 
  [ 'Max',      'max'   ],
  [ 'Min',      'min'   ],
  [ 'StdDev',   'stdev' ],
]
RE_NUMBER = re.compile('^[0123456789\.]+$')  # simple way to know if we need quotes or not

class QueryBuilder(Dialogs.Dialog):
  '''Query Builder dialog box'''
  def __init__(self, mainframe):
    '''Constructor'''
    Dialogs.Dialog.__init__(self, mainframe, 'QueryBuilder') 
    

  def init(self, sql=None):
    '''Initializes the dialog each time it is run'''
    # set up controls
    self.getControl('groupby').Clear()
    self.getControl('groupby').AppendItems([ item[0] for item in GROUPBY_OPTIONS ])
    self.getControl('wheretype').Clear()
    self.getControl('wheretype').AppendItems([ item[0] for item in WHERETYPE_OPTIONS ])
    self.getControl('fields').Clear()
    
    # add the TableController to the view (can't put it in with XRC)
    self.tablecontroller = TableController(self.getControl('tablepanel'), self)
    self.getControl('tablepanel').GetSizer().Add(self.tablecontroller, flag=wx.ALL|wx.EXPAND, proportion=1)
    self.getControl('tablepanel').Layout()
    
    # populate the list of database names
    dbnames = self.mainframe.getDatabases()
    assert len(dbnames) > 0, 'You must connect to a database before running queries.'
    dbnames.sort()
    dbchoice = self.getControl('databases')
    dbchoice.Clear()
    dbchoice.AppendItems(dbnames)
    dbchoice.SetSelection(0)
    self.initTables()
    
    # bind events
    self.getControl('databases').Bind(wx.EVT_CHOICE, self.onDatabaseSelect)
    self.getControl('tables').Bind(wx.EVT_LISTBOX_DCLICK, self.onTableSelect)
    self.getControl('addcalculatedfield').Bind(wx.EVT_BUTTON, self.onCalculatedField)
    self.getControl("deletefield").Bind(wx.EVT_BUTTON, self.deleteRow)
    self.getControl("fieldup").Bind(wx.EVT_BUTTON, self.fieldUp)
    self.getControl("fielddown").Bind(wx.EVT_BUTTON, self.fieldDown)
    self.getControl('fields').Bind(wx.EVT_LISTBOX, self.onFieldSelect)
    self.getControl('nameas').Bind(wx.EVT_TEXT, self.onNameChange)
    self.getControl('where').Bind(wx.EVT_TEXT, self.onWhereChange)
    self.getControl('wheretype').Bind(wx.EVT_CHOICE, self.onWhereTypeChange)
    self.getControl('calculation').Bind(wx.EVT_TEXT, self.onCalculationChange)
    self.getControl('groupby').Bind(wx.EVT_CHOICE, self.onGroupByChange)
    self.getControl('groupbybutton').Bind(wx.EVT_TOGGLEBUTTON, self.onGroupByButton)
    
    
  def initTables(self):
    '''Initializes the list of tables when a new database is selected'''
    dbname = self.getControl('databases').GetStringSelection()
    db = self.mainframe.getVariable(dbname)
    tablenames = db.list_tables()
    tablenames.sort()
    tablechoice = self.getControl('tables')
    tablechoice.Clear()
    tablechoice.AppendItems(tablenames)
    
    
  def onDatabaseSelect(self, event=None):
    '''Responds to the selection of a database in the choice box'''
    if self.getControl('sql').GetValue() != '':
      if wx.MessageBox(lang('Selecting a different database will reset this dialog.  Continue?'), lang('Select New Database'), style=wx.YES_NO) != wx.YES:
        return
    self.getControl('fields').Clear()
    self.getControl('sql').SetValue('')
    self.getControl('where').SetValue('')
    self.getControl('calculation').SetValue('')
    self.tablecontroller.clear()
    self.initTables()
    event.Skip()
     
     
  def onFieldSelect(self, event=None):
    '''Responds to the selection of a field in the list box'''
    self.updateFieldInfoControls()
    event.Skip()
    
    
  def onCalculatedField(self, event=None):
    '''Called when a calculated field is added'''
    fieldbox = self.getControl('fields')
    fieldname = ensure_unique_list_value([ fieldbox.GetClientData(i).nameas for i in range(fieldbox.GetCount()) ], 'Unnamed')
    self.addField(None, fieldname)
     
   
  def onTableSelect(self, event=None):
    '''Called when a table is double clicked on'''
    dbname = self.getControl('databases').GetStringSelection()
    db = self.mainframe.getVariable(dbname)
    tablename = self.getControl('tables').GetStringSelection()
    for child in self.tablecontroller.getTableViews():
      if (child.tablename == tablename):
        return
    tableview = TableView(self.tablecontroller, self, db, dbname, tablename)
    self.tablecontroller.layoutNewChild(tableview)
    
    
  def onGroupByButton(self, event=None):
    '''Responds to a change in the group by status of the query'''
    self.updateFieldInfoControls()
    self.createSQL()
    
    
  def onNameChange(self, event=None):
    '''Responds to a change in the name'''
    nameas = self.getControl('nameas').GetValue()
    index = self.getControl('fields').GetSelection()
    if index >= 0:
      self.getControl('fields').GetClientData(index).nameas = nameas
      self.updateListBoxItem()
    self.createSQL()
    

  def onCalculationChange(self, event=None):
    '''Responds to a change in the calculation'''
    calculation = self.getControl('calculation').GetValue()
    index = self.getControl('fields').GetSelection()
    if index >= 0:
      self.getControl('fields').GetClientData(index).calculation = calculation
    self.createSQL()

    
  def onGroupByChange(self, event=None):
    '''Responds to a change in the group by selection'''
    groupby = self.getControl('groupby').GetSelection()
    index = self.getControl('fields').GetSelection()
    if index >= 0:
      self.getControl('fields').GetClientData(index).groupby = groupby
    self.createSQL()

    
  def onWhereTypeChange(self, event=None):
    '''Responds to a change in the where type selection'''
    wheretype = self.getControl('wheretype').GetSelection()
    index = self.getControl('fields').GetSelection()
    if index >= 0:
      self.getControl('fields').GetClientData(index).wheretype = wheretype
    self.createSQL()

    
  def onWhereChange(self, event=None):
    '''Responds to a change in the where clause'''
    where = self.getControl('where').GetValue()
    index = self.getControl('fields').GetSelection()
    if index >= 0:
      self.getControl('fields').GetClientData(index).where = where
    self.createSQL()
    

  def deleteRow(self, event=None):
    '''Called when the user clicks the delete row button'''
    fieldbox = self.getControl('fields')
    if fieldbox.GetCount() > 0:
      index = fieldbox.GetSelection()
      if wx.MessageBox(lang('Remove this field from the query?'), lang('Remove field'), style=wx.YES_NO) == wx.YES:
        fieldbox.Delete(fieldbox.GetSelection())
        fieldbox.SetSelection(min(index, fieldbox.GetCount() - 1))
        self.updateFieldInfoControls()
    self.createSQL()
    
  
  def fieldUp(self, event=None):
    '''Called when the user clicks the Move Up button'''
    fieldbox = self.getControl('fields')
    index = fieldbox.GetSelection()
    if index > 0:
      info = fieldbox.GetClientData(index)
      fieldbox.SetClientData(index, fieldbox.GetClientData(index-1))
      fieldbox.SetClientData(index-1, info)
      self.updateListBoxItem(index)
      self.updateListBoxItem(index-1)
      fieldbox.SetSelection(index-1)
      self.neworder = True
    self.createSQL()
      
    
  def fieldDown(self, event=None):
    '''Called when the user clicks the Move Down button'''
    fieldbox = self.getControl('fields')
    index = fieldbox.GetSelection()
    if index < fieldbox.GetCount() - 1:
      info = fieldbox.GetClientData(index)
      fieldbox.SetClientData(index, fieldbox.GetClientData(index+1))
      fieldbox.SetClientData(index+1, info)
      self.updateListBoxItem(index)
      self.updateListBoxItem(index+1)
      fieldbox.SetSelection(index+1)
      self.neworder = True
    self.createSQL()
  
    
  def addField(self, tablename, fieldname):
    '''Adds a new table to the fields (this is called from TableView below when the user double clicks a field'''
    fields = self.getControl('fields')
    fields.Append(fieldname, FieldInfo(self, tablename, fieldname))
    fields.SetSelection(fields.GetCount() - 1)
    self.updateListBoxItem()
    self.updateFieldInfoControls()
    self.createSQL()
    
    
  def updateListBoxItem(self, index=-1):
    '''Updates the currently-selected item in the list box to correctly reflect its FieldSource information'''
    fields = self.getControl('fields')
    if index < 0:
      index = fields.GetSelection()
    if index > 0:
      fieldinfo = fields.GetClientData(index)
      fields.SetString(index, fieldinfo.nameas)
   
   
  def updateFieldInfoControls(self):
    '''Updates the information controls for the currently-selected item in the fields list box'''
    fields = self.getControl('fields')
    index = fields.GetSelection()
    if index >= 0:
      fieldinfo = fields.GetClientData(index)
      self.getControl('nameas').SetValue(fieldinfo.nameas)
      if self.getControl('groupbybutton').GetValue():
        self.getControl('groupby').Enable(True)
        self.getControl('groupby').Enable(True)
        self.getControl('groupby').SetSelection(fieldinfo.groupby)
      else:
        self.getControl('groupby').SetSelection(-1)
        self.getControl('groupby').Enable(False)
        self.getControl('groupbylabel').Enable(False)
      self.getControl('where').SetValue(fieldinfo.where)
      self.getControl('wheretype').SetSelection(fieldinfo.wheretype)
      if fieldinfo.calculatedfield:
        self.getControl('source').SetLabel('< Calculated Field >')
        self.getControl('calculation').SetValue(fieldinfo.calculation)
        self.getControl('calculation').Enable(True)
        self.getControl('calculationlabel').Enable(True)
      else:  # regular field
        self.getControl('source').SetLabel(fieldinfo.tablename + '.' + fieldinfo.fieldname)
        self.getControl('calculation').SetValue('')
        self.getControl('calculation').Enable(False)
        self.getControl('calculationlabel').Enable(False)


  def createSQL(self):
    '''Creates SQL based on the current settings in the dialog'''
    fields = self.getControl('fields')
    select = []
    fromclause = []
    where = []
    groupby = []
    groupbyquery = self.getControl('groupbybutton').GetValue()
    
    # add the field names to be selected
    for i in range(fields.GetCount()):
      info = fields.GetClientData(i)
      if info.calculatedfield and groupbyquery and info.groupby > 0:
        select.append('%s(%s) AS %s' % (GROUPBY_OPTIONS[info.groupby][1], info.calculation, info.nameas))
      elif info.calculatedfield:
        select.append('%s AS %s' % (info.calculation, info.nameas))
      elif groupbyquery and info.groupby > 0 and info.nameas == info.fieldname:
        select.append('%s(%s.%s)' % (GROUPBY_OPTIONS[info.groupby][1], info.tablename, info.fieldname))
      elif groupbyquery and info.groupby > 0:
        select.append('%s(%s.%s) AS %s' % (GROUPBY_OPTIONS[info.groupby][1], info.tablename, info.fieldname, info.nameas))
      elif info.nameas == info.fieldname:
        select.append('%s.%s' % (info.tablename, info.fieldname))
      else:
        select.append('%s.%s AS %s' % (info.tablename, info.fieldname, info.nameas))
        
    # add tables for the where clause
    for i in range(fields.GetCount()):
      info = fields.GetClientData(i)
      if info.tablename:
        fromclause.append(info.tablename)
    
    # add the from clause and other joins
    for info in self.tablecontroller.joins:
      if info.includerecords == 0:  # regular join
        fromclause.append(info.tableview1.tablename)
        fromclause.append(info.tableview2.tablename)
        where.append('%s.%s %s %s.%s' % (info.tableview1.tablename, info.fieldname1, WHERETYPE_OPTIONS[info.jointype][1], info.tableview2.tablename, info.fieldname2))
      elif info.includerecords == 1:  # left join
        fromclause.append('%s LEFT JOIN %s ON %s.%s %s %s.%s' % (info.tableview1.tablename, info.tableview2.tablename, info.tableview1.tablename, info.fieldname1, WHERETYPE_OPTIONS[info.jointype][1], info.tableview2.tablename, info.fieldname2))
      elif info.includerecords == 2:  # right join
        fromclause.append('%s RIGHT JOIN %s ON %s.%s %s %s.%s' % (info.tableview1.tablename, info.tableview2.tablename, info.tableview1.tablename, info.fieldname1, WHERETYPE_OPTIONS[info.jointype][1], info.tableview2.tablename, info.fieldname2))
    
    # add the where clauses
    for i in range(fields.GetCount()):
      info = fields.GetClientData(i)
      if info.where and info.nameas == info.fieldname and RE_NUMBER.match(info.where):
        where.append("%s.%s %s %s" % (info.tablename, info.fieldname, WHERETYPE_OPTIONS[info.wheretype][1], info.where))
      elif info.where and info.nameas == info.fieldname:
        where.append("%s.%s %s '%s'" % (info.tablename, info.fieldname, WHERETYPE_OPTIONS[info.wheretype][1], info.where))
      elif info.where and RE_NUMBER.match(info.where):
        where.append("%s %s %s" % (info.nameas, WHERETYPE_OPTIONS[info.wheretype][1], info.where))
      elif info.where:
        where.append("%s %s '%s'" % (info.nameas, WHERETYPE_OPTIONS[info.wheretype][1], info.where))
    
    # add the group by
    if groupbyquery:
      for i in range(fields.GetCount()):
        info = fields.GetClientData(i)
        if info.groupby == 0 and info.nameas == info.fieldname and not info.calculatedfield:
          groupby.append('%s.%s' % (info.tablename, info.fieldname))
        elif info.groupby == 0:
          groupby.append('%s' % (info.nameas,))
    
    # set in the dialog
    sql = []
    if len(select) > 0:
      sql.append('SELECT ' + ', '.join(select))
    fromset = set(fromclause)
    if len(fromset) > 0:
      sql.append('FROM ' + ', '.join(fromset))
    if len(where) > 0:
      sql.append('WHERE ' + ' AND '.join(where))
    if len(groupby) > 0:
      sql.append('GROUP BY ' + ', '.join(groupby))
    self.getControl('sql').SetValue(' '.join(sql))
    
    
  def getSQL(self):
    '''Returns the SQL in the text box.  It does not recreate from the GUI controls, but
       just returns what's in the SQL box at the moment.'''
    return self.getControl('sql').GetValue()
    
    
  def ok(self, event=None):
    '''Responds to the ok button'''
    dbname = self.getControl('databases').GetStringSelection()
    queryname = self.getControl('queryname').GetValue()
    assert queryname, 'Please enter a name for this query.'
    sql = self.getSQL()
    assert sql, 'The query is empty.  Please double-click a table name to query.'
    cmds = [
      '%s = Database.Query("%s", conn=%s)' % (queryname, sql, dbname),
      '%s.view()' % (queryname, ),
    ]
    if self.mainframe.execute(cmds):
      self.close()

    
   
   
#############################################################################
###   Small data classes
   
class FieldInfo:
  '''A small class to hold data about a field being selected'''
  def __init__(self, querybuilder, tablename, fieldname):
    self.calculatedfield = False
    if tablename == None or fieldname == None:
      self.calculatedfield = True
    self.tablename = tablename
    self.fieldname = fieldname
    self.nameas = fieldname
    self.where = ''
    self.wheretype = 0
    self.calculation = ''
    self.groupby = 0
    

class JoinInfo:
  '''A small class to hold data about a join between tables'''
  def __init__(self, tableview1, fieldname1, tableview2=None, fieldname2=None):
    self.jointype = 0
    self.includerecords = 0
    self.tableview1 = tableview1
    self.fieldname1 = fieldname1
    self.tableview2 = tableview2
    self.fieldname2 = fieldname2
    

    
   
####################################################################
###   TableController and TableView custom controls
    
class TableController(wx.PyScrolledWindow):
  '''Holds the TableView objects'''
  def __init__(self, parent, querybuilder):
    '''Constructor'''
    wx.PyScrolledWindow.__init__(self, parent)
    self.querybuilder = querybuilder
    self.joinstart = None
    self.joins = []
    self.SetScrollRate(20, 20)
    self.Bind(wx.EVT_PAINT, self.onPaint)
    self.Bind(wx.EVT_LEFT_DOWN, self.onMouseDown)
    
    
  def clear(self):
    '''Clears this table controller'''
    for child in self.getTableViews():
      child.Destroy()
    self.joins = []
    

  def getTableViews(self):
    '''Returns the table views that are children of this view'''
    return [ child for child in self.GetChildren() if isinstance(child, TableView) ]

    
  def layoutNewChild(self, tableview):
    '''Positions a new child that has been added to our window'''
    x = tableview.X_PADDING
    y = tableview.Y_PADDING
    parentsize = self.GetSize()
    done = False
    while not done and y + tableview.DEFAULT_SIZE[1] < parentsize[1]:
      while not done and x + tableview.DEFAULT_SIZE[0] < parentsize[0]:
        for child in self.getTableViews():
          childpos = child.GetPosition()
          childsize = child.GetSize()
          if (x >= childpos[0] and x <= childpos[0] + childsize[0] or x + tableview.DEFAULT_SIZE[0] >= childpos[0] and x + tableview.DEFAULT_SIZE[0] <= childpos[0] + childsize[0]) and \
             (y >= childpos[1] and y <= childpos[1] + childsize[1] or y + tableview.DEFAULT_SIZE[1] >= childpos[1] and y + tableview.DEFAULT_SIZE[1] <= childpos[1] + childsize[1]):
            break  # we're inside of this child
        else:  # if we have a free spot
          done = True
          tableview.SetDimensions(x, y, tableview.DEFAULT_SIZE[0], tableview.DEFAULT_SIZE[1])
        x += tableview.DEFAULT_SIZE[0] + tableview.X_PADDING
      y += tableview.DEFAULT_SIZE[1] + tableview.Y_PADDING
    if not done:
      tableview.SetDimensions(tableview.X_PADDING, tableview.Y_PADDING, tableview.DEFAULT_SIZE[0], tableview.DEFAULT_SIZE[1])
    tableview.Layout()
    self.updateVirtualSize()
    
    
  def doJoin(self, endtableview, endfieldname):
    '''Creates a join between two tables'''
    if self.joinstart == None or self.joinstart.tableview1 == endtableview:
      self.joinstart = None
      return
    self.joinstart.tableview2 = endtableview
    self.joinstart.fieldname2 = endfieldname
    for joininfo in self.joins:  # don't add if already there
      if joininfo.tableview1 == self.joinstart.tableview1 and joininfo.fieldname1 == self.joinstart.fieldname1 and joininfo.tableview2 == self.joinstart.tableview2 and joininfo.fieldname2 == self.joinstart.fieldname2:
        self.joinstart = None
        return
    self.joins.append(self.joinstart)
    self.joinstart = None
    self.querybuilder.createSQL()
    self.Refresh()
    
    
  def getLinePoints(self, joininfo):
    '''Calculates the start point and end point for a line between the two join points'''
    tv1 = joininfo.tableview1.GetPosition()
    tv2 = joininfo.tableview2.GetPosition()
    if tv1[0] < tv2[0]: 
      startpoint = joininfo.tableview1.getRightSide(joininfo.fieldname1)
      endpoint = joininfo.tableview2.getLeftSide(joininfo.fieldname2)
    else:
      startpoint = joininfo.tableview2.getRightSide(joininfo.fieldname2)
      endpoint = joininfo.tableview1.getLeftSide(joininfo.fieldname1)
    return startpoint, endpoint    
    
 
  def onPaint(self, event=None):
    '''Paints the joins on the window'''
    dc = wx.PaintDC(self)
    dc.SetPen(wx.Pen(wx.BLACK, width=2))
    dc.SetTextForeground(wx.BLACK)
    dc.SetFont(wx.NORMAL_FONT)
    for joininfo in self.joins:
      startpoint, endpoint = self.getLinePoints(joininfo)
      dc.DrawLine(startpoint[0], startpoint[1], endpoint[0], endpoint[1])
      dc.DrawText(WHERETYPE_OPTIONS[joininfo.jointype][1], startpoint[0]+((endpoint[0]-startpoint[0])/2.0), startpoint[1]+((endpoint[1]-startpoint[1])/2.0))
    event.Skip()
    
    
  def onMouseDown(self, event=None):
    '''Captures a mouse down event'''
    for joininfo in self.joins:
      startpoint, endpoint = self.getLinePoints(joininfo)
      # slope and y-intercept for this line
      slope = float(endpoint[1] - startpoint[1]) / float(endpoint[0] - endpoint[1])
      yint = float(startpoint[1]) - float(startpoint[0] * slope)
      # slope and y-intercept for the mouse point and one of the points on the line
      slope2 = float(event.m_y - startpoint[1]) / float(event.m_x - endpoint[1])
      yint2 = float(event.m_y) - float(event.m_x * slope)
      # see if they are close
      if abs(slope2 - slope) < 1 and abs(yint2 - yint) < 5 and \
         event.m_x >= startpoint[0] and event.m_x <= endpoint[0] and\
         event.m_y >= startpoint[1] and event.m_y <= endpoint[1]:
        qryjoin = QueryJoin(self.querybuilder.mainframe, self, joininfo)
        qryjoin.runModal()
        qryjoin.destroyDialog()
        self.querybuilder.createSQL()
        break
    event.Skip()
    
    
  def updateVirtualSize(self):
    '''Updates the virtual size of this control based on the children in it'''
    size = [0, 0]
    viewstart = self.GetViewStart()
    scrollrate = self.GetScrollPixelsPerUnit()
    viewpos = [ viewstart[0] * scrollrate[0], viewstart[1] * scrollrate[1] ]
    for child in self.getTableViews():
      childpos = child.GetPosition()
      childsize = child.GetSize()
      size[0] = max(size[0], viewpos[0] + childpos[0] + childsize[0])
      size[1] = max(size[1], viewpos[1] + childpos[1] + childsize[1])
    # only let it get bigger, not smaller (it's wierd to the user when it gets smaller in real time)
    mysize = self.GetVirtualSize()
    size[0] = max(size[0], mysize[0])
    size[1] = max(size[1], mysize[1])
    if mysize != size:
      self.SetVirtualSize(size)
    
    

class TableView(wx.PyWindow):
  '''A view of a table in the dialog'''
  X_PADDING = 15
  Y_PADDING = 0 
  DEFAULT_SIZE = (130, 160)
  
  def __init__(self, parent, querybuilder, database, databasename, tablename):
    '''Constructor'''
    wx.PyWindow.__init__(self, parent)
    self.querybuilder = querybuilder
    self.databasename = databasename
    self.tablename = tablename
    self.SetSizer(wx.BoxSizer(wx.VERTICAL))
    self.lastmousepos = None
    
    # add the title
    self.title = wx.StaticText(self, label=tablename, style=wx.SIMPLE_BORDER|wx.ALIGN_CENTRE)
    self.title.SetBackgroundColour(wx.SystemSettings.GetColour(wx.SYS_COLOUR_HIGHLIGHT))
    self.title.SetForegroundColour(wx.SystemSettings.GetColour(wx.SYS_COLOUR_HIGHLIGHTTEXT))
    self.GetSizer().Add(self.title, flag=wx.ALL|wx.EXPAND|wx.CENTER)
    self.title.Bind(wx.EVT_MOTION, self.onTitleMouseMove)
    self.title.Bind(wx.EVT_LEFT_DOWN, self.onTitleMouseDown)
    self.title.Bind(wx.EVT_LEFT_UP, self.onTitleMouseUp)
    self.title.Bind(wx.EVT_LEAVE_WINDOW, self.onTitleMouseUp)
    
    # add the fields panel
    self.fields = wx.ScrolledWindow(self, style=wx.DOUBLE_BORDER)
    self.fields.SetBackgroundColour(wx.WHITE)
    self.fields.SetScrollRate(20, 20)
    self.GetSizer().Add(self.fields, flag=wx.ALL|wx.EXPAND, proportion=1)
    self.fields.SetSizer(wx.BoxSizer(wx.VERTICAL))
    self.fields.SetToolTipString(lang('Double-click a field name to add it to the query'))
    self.fields.Bind(wx.EVT_SCROLLWIN, self.onScrollBar)
    
    # add the individual fields to the fields panel
    bestsize = [0, 0]
    for item in database._get_columns(tablename):
      text = wx.StaticText(self.fields, label=item)
      self.fields.GetSizer().Add(text, flag=wx.ALL|wx.EXPAND)
      text.Bind(wx.EVT_ENTER_WINDOW, self.onFieldEnter)
      text.Bind(wx.EVT_LEAVE_WINDOW, self.onFieldExit)
      text.Bind(wx.EVT_LEFT_DCLICK, self.onFieldDouble)
      text.Bind(wx.EVT_LEFT_DOWN, self.onFieldMouseDown)
      text.Bind(wx.EVT_LEFT_UP, self.onFieldMouseUp)
      bestsize[1] += text.GetBestSize()[1]
    self.fields.SetVirtualSize(bestsize)
    
  
  def onScrollBar(self, event=None):
    '''Handles a scroll bar event'''
    self.querybuilder.tablecontroller.Refresh()
    event.Skip()
    
    
  def onTitleMouseDown(self, event=None):
    '''Responds to a mouse button down event in the title bar'''
    self.lastmousepos = event.GetPosition()  # start dragging
    self.title.CaptureMouse()
    
    
  def onTitleMouseMove(self, event=None):
    '''Responds to a mouse move event in the title bar'''
    if self.lastmousepos != None:
      pos = event.GetPosition()
      diff = pos[0] - self.lastmousepos[0], pos[1] - self.lastmousepos[1]
      windowpos = self.GetPosition()
      newpos = [ max(0, windowpos[0] + diff[0]), max(0, windowpos[1] + diff[1]) ]  # keep >= 0, 0
      self.SetPosition(newpos)
      self.querybuilder.tablecontroller.Refresh()  # it would be better to simply invalidate it
    
    
  def onTitleMouseUp(self, event=None):
    '''Responds to a mouse button up event in the title bar'''
    self.lastmousepos = None  # cancel dragging
    if self.title.HasCapture():
      self.title.ReleaseMouse()
    self.querybuilder.tablecontroller.updateVirtualSize()
    self.querybuilder.tablecontroller.Refresh()  # it would be better to simply invalidate it
    

  def onFieldEnter(self, event=None): 
    '''Responds to a mouse entering a field'''
    text = event.GetEventObject()
    text.SetBackgroundColour(wx.SystemSettings.GetColour(wx.SYS_COLOUR_HIGHLIGHT))
    text.SetForegroundColour(wx.SystemSettings.GetColour(wx.SYS_COLOUR_HIGHLIGHTTEXT))
    
  
  def onFieldExit(self, event=None):
    '''Responds to a mouse leaving a field'''
    text = event.GetEventObject()
    text.SetBackgroundColour(wx.WHITE)
    text.SetForegroundColour(wx.SystemSettings.GetColour(wx.SYS_COLOUR_WINDOWTEXT))

    
  def onFieldMouseDown(self, event=None):
    '''Responds to a mouse down event in the window'''
    self.querybuilder.tablecontroller.joinstart = JoinInfo(self, event.GetEventObject().GetLabel())
    event.Skip()
    
    
  def onFieldMouseUp(self, event=None):
    '''Responds to a mouse up event in the window'''
    self.querybuilder.tablecontroller.doJoin(self, event.GetEventObject().GetLabel())
    event.Skip()
    
    
  def onFieldDouble(self, event=None):
    '''Responds to a double click on the list'''
    fieldname = event.GetEventObject().GetLabel()
    self.querybuilder.addField(self.tablename, fieldname)
    
    
  def getRightSide(self, fieldname):
    '''Returns the pixel location on the center of the fieldname on its right side, relative to the TableView'''
    tvpos = self.GetPosition()
    fieldpos = self.fields.GetPosition()
    fieldsize = self.fields.GetSize()
    for child in self.fields.GetChildren():
      if child.GetLabel() == fieldname:
        point = child.GetPosition()
        point[0] += tvpos[0] + fieldpos[0] + self.GetSize()[0] 
        point[1] += tvpos[1] + fieldpos[1] + int(child.GetSize()[1] / 2.0)  # center vertically
        point[1] = max(point[1], tvpos[1] + fieldpos[1])
        point[1] = min(point[1], tvpos[1] + fieldpos[1] + fieldsize[1])
        return point
    return [0, 0]    
    
    
  def getLeftSide(self, fieldname):
    '''Returns the pixel location on the center of the fieldname on its left side, relative to the TableView'''
    tvpos = self.GetPosition()
    fieldpos = self.fields.GetPosition()
    fieldsize = self.fields.GetSize()
    for child in self.fields.GetChildren():
      if child.GetLabel() == fieldname:
        point = child.GetPosition()
        point[0] += tvpos[0] + fieldpos[0]
        point[1] += tvpos[1] + fieldpos[1] + int(child.GetSize()[1] / 2.0)  # center vertically
        point[1] = max(point[1], tvpos[1] + fieldpos[1])
        point[1] = min(point[1], tvpos[1] + fieldpos[1] + fieldsize[1])
        return point
    return [0, 0]    
    
    
    
    
############################################
###   Query Join Dialog

class QueryJoin(Dialogs.Dialog):
  def __init__(self, mainframe, tablecontroller, joininfo):
    Dialogs.Dialog.__init__(self, mainframe, 'QueryJoin')
    self.tablecontroller = tablecontroller
    self.joininfo = joininfo
    
  
  def init(self):
    '''Initializes the dialog'''
    self.getControl('jointypetext1').SetLabel('Join where records in %s are' % (self.joininfo.tableview1.tablename))
    self.getControl('jointypetext2').SetLabel('records in %s.' % (self.joininfo.tableview2.tablename,))
    self.getControl('jointype').Clear()
    self.getControl('jointype').AppendItems([ item[0] for item in WHERETYPE_OPTIONS ])
    self.getControl('jointype').SetSelection(self.joininfo.jointype)
    self.getControl('includerecords').SetItemLabel(0, 'Select only matching records from %s and %s.' % (self.joininfo.tableview1.tablename, self.joininfo.tableview2.tablename))
    self.getControl('includerecords').SetItemLabel(1, 'Select all records from %s and only matching records from %s.' % (self.joininfo.tableview1.tablename, self.joininfo.tableview2.tablename))
    self.getControl('includerecords').SetItemLabel(2, 'Select only matching records from %s and all records from %s.' % (self.joininfo.tableview1.tablename, self.joininfo.tableview2.tablename))
    self.getControl('includerecords').SetSelection(self.joininfo.includerecords)
    self.getControl('deletebutton').Bind(wx.EVT_BUTTON, self.onDeleteButton)
    
  
  def onDeleteButton(self, event=None):
    '''Responds to a delete'''
    if wx.MessageBox(lang('Are you sure you want to permanenetly delete this join?'), lang('Delete Join'), style=wx.YES_NO) == wx.YES:
      self.tablecontroller.joins.remove(self.joininfo)
      self.cancel()
    
  
  def ok(self, event=None):
    '''Responds to the ok button'''
    self.joininfo.jointype = self.getControl('jointype').GetSelection()
    self.joininfo.includerecords = self.getControl('includerecords').GetSelection()
    self.close()

    
    
    
    
    
    