#!/usr/bin/env python

####################################################################################
#                                                                                  #
# Copyright (c) 2006 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This file is part of Picalo.                                                     #
#                                                                                  #
# Picalo is free software; you can redistribute it and/or modify                   #
# it under the terms of the GNU General Public License as published by             #
# the Free Software Foundation; either version 2 of the License, or                # 
# (at your option) any later version.                                              #
#                                                                                  #
# Picalo is distributed in the hope that it will be useful,                        #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                   #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    #
# GNU General Public License for more details.                                     #
#                                                                                  #
# You should have received a copy of the GNU General Public License                #
# along with Foobar; if not, write to the Free Software                            #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA        #
#                                                                                  #
####################################################################################

from picalo.gui import Editor
from picalo.gui import Utils
from picalo.gui import Dialogs
from picalo.gui import Spreadsheet
from picalo.gui.Languages import lang
from picalo.lib import GUID
import wx, wx.wizard
import imp, re, traceback, os.path, zipfile, sys
import xml.dom.minidom
from StringIO import StringIO
import types
from picalo import Table, TableArray, TableList
from picalo.base.Global import clear_progress, make_valid_variable
from MiniHelp import MiniHelp


VALID_VARIABLE_CHARS = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789_'

############################################
###   Wizard

def load_wizard(mainframe, wizname):
  '''Loads a wizard from a wizname in the gui/wizards directory.  The
     wizname should be the root of the Python file (without the .py extension).
     This method figures out the name of the wizard from the filename.'''
  return PicaloWizard(mainframe, Utils.getWizardPath(wizname + '.py'), wizname)
  

def _load_module(wizardname, filename):
  '''Loads a wizard module from a filename'''
  # load the code object
  fname, ext = os.path.splitext(filename)
  if ext == '.pyc':
    mod = imp.load_compiled(wizardname, filename)
  else: 
    mod = imp.load_source(wizardname, filename)
    
  # return the module
  return mod
  
  
def check_wizard(wizardname, code_object):
  '''Checks a wizard for the required wizard attributes'''
  # ensure it has the right members
  assert hasattr(code_object, 'run'), 'The wizard module, ' + str(wizardname) + ', does not have a run() function.  The wizard cannot be loaded -- please contact the wizard author.'
  assert hasattr(code_object, 'wizard'), 'The wizard module, ' + str(wizardname) + ', does not have the required "wizard" xml describing the wizard pages.  The wizard cannot be loaded -- please contact the wizard author.'
#  assert hasattr(code_object, 'DETECTLET_STANDARD'), 'The wizard module, ' + str(wizardname) + ', does not have the required "DETECTLET_VERSION" variable.  The wizard cannot be loaded -- please contact the wizard author.'
  
  

class PicaloWizard:
  '''The main Picalo wizard'''
  def __init__(self, mainframe, filename, wizard):
    self.mainframe = mainframe
    self.filename = filename
    self.wizard = wizard
    

  def __call__(self, event=None):
    '''Called when the menu option is selected'''
    # get the module and function to use
    fname, ext = os.path.splitext(self.filename)
    self.mod = _load_module(self.wizard, self.filename)

    # verify it
    check_wizard(self.wizard, self.mod)

    # get the function and documentation
    doc = Utils.Documentation(self.wizard, self.wizard, '', self.mod.run)
    
    # read the xml 
    if not hasattr(self.mod, 'wizard'):
      wx.MessageBox(lang('The wizard module,') + ' ' + str(self.wizard) + ' ' + lang('does not have the required "wizard" xml describing the wizard pages.  The wizard cannot be run.'), lang('Deteclet Error'))
      return
    try:
      self.wizardxml = xml.dom.minidom.parseString(self.mod.wizard)
    except Exception, e:
      wx.MessageBox(lang('Could not load "wizard" xml from the wizard:') + ' ' + str(e), lang('Wizard Error'))
      return

    # create the wizard
    title = lang('Wizard:') + ' ' + re.sub(r'(?<=.)([A-Z])', r' \1', self.wizard)
    self.wiz = wx.wizard.Wizard(self.mainframe, title=title, bitmap=Utils.getBitmap('wizardsplash.png')) 
    
    # set up the variables map
    self.variables = {}
    self.parameters = {}
 
    # set up page 1
    page1 = wx.wizard.WizardPageSimple(self.wiz)
    page1.SetSizer(wx.BoxSizer())
    sizer = wx.FlexGridSizer(cols=1, vgap=10)
    page1.GetSizer().Add(sizer, border=5, flag=wx.ALL|wx.EXPAND)
    sizer.AddGrowableCol(0)
    sizer.AddGrowableRow(1)
    sizer.Add(wx.StaticText(page1, label=lang("This wizard is described as follows:")))
    sizer.Add(wx.TextCtrl(page1, size=(500, -1), value='\n\n'.join(doc.desc), style=wx.TE_MULTILINE|wx.TE_READONLY), flag=wx.ALL|wx.EXPAND)
    if hasattr(self.mod, 'example_input'):
      examplebutton = wx.Button(page1, label=lang("Show Example Input Data"))
      examplebutton.Bind(wx.EVT_BUTTON, self.showExampleInput)
      sizer.Add(examplebutton)
    sizer.Add(wx.StaticText(page1, label=lang('Click "Next" to begin.')), flag=wx.ALIGN_RIGHT)
    self.wiz.FitToPage(page1)
    
    # set up the information gathering pages
    try:
      lastpage = page1
      for pagexml in [ child for child in self.wizardxml.documentElement.childNodes if child.nodeName == 'page' ]:
        # create this wizard page
        wizardpage = WizardPage(self, self.wiz, self.mainframe, pagexml)
        
        # chain together with the last one
        wx.wizard.WizardPageSimple_Chain(lastpage, wizardpage)
        lastpage = wizardpage
        
    except AssertionError, e:
      wx.MessageBox(str(e), lang('Wizard Error'))
      self.wiz.Destroy()
      self.wiz = None
      return
      
    # set up the last page
    self.finalpage = wx.wizard.WizardPageSimple(self.wiz)
    self.finalpage.SetSizer(wx.BoxSizer())
    sizer = wx.FlexGridSizer(cols=1, vgap=10)
    sizer.AddGrowableCol(0)
    self.finalpage.GetSizer().Add(sizer, border=5, flag=wx.ALL|wx.EXPAND)
    sizer.Add(wx.StaticText(self.finalpage, label=lang("Please enter a variable name to store the results in:")), flag=wx.ALL|wx.EXPAND)
    resultsname = ''
    for char in self.wizard:
      if char in VALID_VARIABLE_CHARS:
        resultsname += char
    self.results = wx.TextCtrl(self.finalpage, value=resultsname, size=(400,-1))
    sizer.Add(self.results, flag=wx.ALL|wx.EXPAND)
    wx.wizard.WizardPageSimple_Chain(lastpage, self.finalpage)
      
    # now that the pages are all set up, call init on all the parameters
    for parameter in self.parameters.values():
      parameter.init(self.parameters)
      
    # bind to the page changing event
    self.wiz.Bind(wx.wizard.EVT_WIZARD_PAGE_CHANGING, self.validateCurrentPage)
    self.wiz.Bind(wx.wizard.EVT_WIZARD_PAGE_CHANGED, self.onNewPage)

    # run the wizard
    self.wiz.RunWizard(page1)
        
    # clean up shop
    self.wiz.Destroy()
    self.wiz = None
    
    
  def showExampleInput(self, event=None):
    '''Shows the example input to the user'''
    # open the table in the GUI
    data = self.mod.example_input()
    if isinstance(data,  (Table, TableList, TableArray)):
      flatname = make_valid_variable('Example_' + self.mod.__name__)
      name = flatname
      index = 0
      while self.mainframe.shell.varExists(name):
        index += 1
        name = flatname + str(index)
      self.mainframe.shell.interp.locals[name] = data
      self.mainframe.openTable(name)
    elif isinstance(data, (list, tuple)):
      for t in data:
        flatname = make_valid_variable('Example_' + self.mod.__name__)
        name = flatname
        index = 0
        while self.mainframe.shell.varExists(name):
          index += 1
          name = flatname + str(index)
        self.mainframe.shell.interp.locals[name] = t
        self.mainframe.openTable(name)
    else:
      raise AssertionError, 'Error in wizard: example input is not in the right format'
    
    # update the table list boxes
    for parameter in self.parameters.values():
      if isinstance(parameter, TableParameter):
        parameter.refreshTables()
    
    
  def onNewPage(self, event=None):
    '''Called when the user comes to a new page.  It allows the controls
       on the page to set up.  This allows them to change appropriately
       based upon the existing variable values.'''
    page = event.GetPage()
    if isinstance(page, WizardPage):
      for param in page.params:
        param.show(self.variables, self.parameters)
    

  def validateCurrentPage(self, event=None):
    '''Called when the user starts to change the page.  This function then
       validates the current page to see if the user can switch.'''
    if event.GetDirection():  # only validate when going forward
      page = event.GetPage()
      if isinstance(page, WizardPage):
        try:
          for param in page.params:
            param.update(self.variables, self.parameters)
        except Exception, e:
          wx.MessageBox(str(e), lang('Deteclet Error'))
          event.Veto()
          
      elif page == self.finalpage:
        # ensure we have a results variable
        if not re.match('^[A-Za-z_][A-Za-z0-9_]*$', self.results.GetValue()):
          wx.MessageBox(lang('Please enter a valid variable name to store the results in.'), lang('Wizard Error'))
          event.Veto()
          return

        # run the analysis
        try:     
          # run the actual wizard function
          try:
            info = self.mod.run(**self.variables)
          finally:
            clear_progress(force=True)
            
          # check the results
          assert info == None or len(info) == 2 and isinstance(info[0], (types.NoneType, Table, TableList, TableArray)) and isinstance(info[1], (types.NoneType, types.StringTypes)), lang('Invalid return from a Wizard.  It must return a Table/TableList/TableArray and a String.')
          
          # set the variable in the shell locals
          if len(info) > 0 and info[0] != None:
            self.mainframe.shell.interp.locals[self.results.GetValue()] = info[0]
            self.mainframe.openTable(self.results.GetValue())
  
          # show the "interpreting the results" window
          if len(info) > 1 and info[1] != None:
            mh = MiniHelp(self.mainframe, self.wiz.GetTitle(), info[1])
            mh.Show()
         
        except AssertionError, e:
#          traceback.print_exc()
          wx.MessageBox(str(e), lang('Wizard'))
          event.Veto()
        
        except Exception, e:
          # if error, let the user go back to the wizard and fix it.
          traceback.print_exc(file=self.mainframe.output)
          self.mainframe.output.select()
          wx.MessageBox(str(e) + '\n\n' + lang('(Detailed information has been printed in the Script Output window.)'), lang('Wizard'))
          event.Veto()

          
class WizardPage(wx.wizard.WizardPageSimple):
  '''A custom-built wizard page with one or more parameter panels on it'''
  def __init__(self, detwiz, wiz, mainframe, pagexml):
    wx.wizard.WizardPageSimple.__init__(self, wiz)
    self.parent = detwiz
    self.pagexml = pagexml
    self.params = []
    self.sizer = wx.FlexGridSizer(cols=1, vgap=10)
    self.SetSizer(self.sizer)
    self.Bind(wx.EVT_SIZE, self.onSize)
    
    # create the parameters
    for node in self.pagexml.childNodes:
      if node.nodeType == node.TEXT_NODE or node.nodeType == node.CDATA_SECTION_NODE:
        for paragraph in Utils.unwrap_paragraphs_list(node.nodeValue):
          label = wx.StaticText(self)
          label.paragraph = paragraph
          self.sizer.Add(label, flag=wx.EXPAND|wx.ALL)
      
      elif node.nodeName == 'parameter':
        # create the parameter panel
        assert node.hasAttribute('type') and PARAMETERS.has_key(node.getAttribute('type').lower()), lang('Parameter has invalid type:') + ' ' + parameterxml.getAttribute('variable') + '. ' + lang('Wizard cannot continue.')
        klass = PARAMETERS[node.getAttribute('type').lower()]
        param = klass(self, mainframe, node)
        self.params.append(param)
        detwiz.parameters[param.variable] = param
      
    
  def onSize(self, event):
    '''Sets the size of the labels -- this has to be done after the dialog has 
       shown itself.  StaticText components have to have their
       size set explicitly or they only go to one line (or the entire 
       dialog) -- not their actual desired size.  Since I don't know wxWidgets
       algorithm for splitting lines, I do my own splitting so the size is
       always consistent.  The splitting algorithm itself isn't perfect, but it
       works for almost all text.'''
    width = self.GetClientSize()[0]
    textheight = self.GetTextExtent(' ')[1]
    for label in self.GetChildren():
      if isinstance(label, wx.StaticText):
        height = 0
        paragraphs = []
        words = []
        lines = []
        for word in label.paragraph.split():
          words.append(word)
          if self.GetTextExtent(' '.join(words))[0] > width:
            lines.append(' '.join(words[:-1]))
            words = words[-1:]
        if len(words) > 0:
          lines.append(' '.join(words))
        height += ((len(lines)) * textheight)
        label.SetLabel('\n'.join(lines))
        label.SetMinSize([width, height])
    event.Skip()
    

    
################################################
###   Parameter panels

RE_SPACES = re.compile(' +')
class BaseParameter:
  '''Base class of parameters'''
  attributes = []
  
  def __init__(self, parent, mainframe, parameterxml):
    self.parent = parent
    self.mainframe = mainframe
    self.parameterxml = parameterxml
    self.paramtype = parameterxml.getAttribute('type').lower()
    
    # ensure that we have the required variables
    assert parameterxml.hasAttribute('variable'), lang('Invalid wizard xml: a parameter does not define the required "variable" attribute.  Wizard cannot continue.')
    self.variable = str(parameterxml.getAttribute('variable'))
    for attribute in self.attributes:
      assert parameterxml.hasAttribute('variable'), lang('Invalid wizard xml: the') + ' ' + self.variable + ' ' + lang('parameter does not define the required') + '"' + attribute + '" ' + lang('attribute.  Wizard cannot continue.')
    self.default = parameterxml.getAttribute('default')

   
  def init(self, parameters):
    '''This method is called once all the parameters are created.
       It allows parameters to set themselves up if they
       are dependent upon other parameters.'''
    pass
    
  def show(self, variables, parameters):
    '''This method is called when a parameter is shown on the
       page.  It is called every time the user visits the page'''
    pass
    
  def update(self, variables, parameters):
    '''Updates the variables map with this control's value.
       The control can throw an exception is the user input
       is not valid.  A message will be shown to the user
       containing the exception text.
       Sunclasses need to override this method.
       The method is called when the user wants to move beyond
       the page containing this parameter.'''
    pass


class TableParameter(BaseParameter):
  '''Allows the user to choose a table'''
  attributes = []
  def __init__(self, parent, mainframe, parameterxml):
    BaseParameter.__init__(self, parent, mainframe, parameterxml)
    self.multiple = parameterxml.hasAttribute('multiple') and parameterxml.getAttribute('multiple').lower() == 'true'
    if self.multiple:
      self.tables = wx.CheckListBox(self.parent, size=[400, 100])
    else:
      self.tables = wx.ListBox(self.parent, size=[400, 100])
    self.parent.sizer.Add(self.tables)
    self.refreshTables()
    self.tables.Bind(wx.EVT_LISTBOX, self.refreshColumns)


  def refreshTables(self):
    '''Refreshes the table listing (this is called when the user views the example data)'''
    # populate it with the list of tables
    self.memtables = self.mainframe.getTables()
    self.memtables.sort()
    self.tables.Clear()
    self.tables.AppendItems(self.memtables)
    if len(self.memtables) > 0:
      if self.default:
        for i in range(len(self.memtables)):
          if self.memtables[i].name == self.default:
            self.tables.Select(i)
      elif self.tables.GetSelection() < 0:
        self.tables.SetSelection(0)
      # if something is initially selected      
      self.refreshColumns()


  def update(self, variables, parameters):
    if self.multiple:
      values = []
      for i in range(self.tables.GetCount()):
        if self.tables.IsChecked(i):
          values.append(self.tables.GetString(i))
      variables[self.variable] = values
    else:
      variables[self.variable] = self.mainframe.shell.getVariable(self.tables.GetStringSelection())
      
      
  def init(self, parameters):
    self.refreshColumns()
    
    
  def refreshColumns(self, event=None):
    '''Looks for ColumnParameter parameters based on this table 
       and updates them.  I do this in the table because many
       columns might depend upon a given table, and it seems that wx
       only allows on listener per event type.'''
    tablename = self.tables.GetStringSelection()
    for param in self.parent.parent.parameters.values():
      if isinstance(param, ColumnParameter) and param.tablevar == self.variable:
        selected = param.columns.GetSelections()  # so we can reselect them (if the table hasn't changed)
        param.columns.Clear()
        if tablename:
          table = self.mainframe.shell.getVariable(tablename)
          param.columns.AppendItems(table.get_column_names())
          if len(selected) == 0:
            param.columns.Select(0)
          else:
            for i in selected:
              if i < param.columns.GetCount():
                param.columns.Select(i)
    
    

class DatabaseConnectionParameter(BaseParameter):
  '''Allows the user to choose a database connection'''
  def __init__(self, parent, mainframe, parameterxml):
    BaseParameter.__init__(self, parent, mainframe, parameterxml)
    self.dbconns = wx.ListBox(self.parent, size=[400, 100])
    self.parent.sizer.Add(self.dbconns)
    self.refreshConnections()
    

  def refreshConnections(self):
    '''Refreshes the database connections'''
    # populate it with the list of connections
    self.memconns = self.mainframe.getDatabases()
    self.memconns.sort()
    self.dbconns.Clear()
    self.dbconns.AppendItems(self.memconns)
    if len(self.memconns) > 0:
      if self.default:
        for i in range(len(self.memconns)):
          if self.memconns[i].name == self.default:
            self.dbconns.Select(i)
      elif self.dbconns.GetSelection() < 0:
        self.dbconns.SetSelection(0)


  def update(self, variables, parameters):
    variables[self.variable] = self.mainframe.shell.getVariable(self.dbconns.GetStringSelection())

      
    
class ColumnParameter(BaseParameter):
  '''Allows the user to choose a column'''
  attributes = [ 'table' ]
  def __init__(self, parent, mainframe, parameterxml):
    BaseParameter.__init__(self, parent, mainframe, parameterxml)
    self.multiple = parameterxml.hasAttribute('multiple') and parameterxml.getAttribute('multiple').lower() == 'true'
    self.tablevar = parameterxml.getAttribute('table')
    if self.multiple:
      self.columns = wx.CheckListBox(self.parent, size=[400, 100])
    else:
      self.columns = wx.ListBox(self.parent, size=[400, 100])
    self.parent.sizer.Add(self.columns)
    
  def init(self, parameters):
    assert parameters.has_key(self.tablevar), lang('The') + ' ' + self.variable + ' ' + lang('parameter references a nonexistant table variable:') + ' ' + self.tablevar
      
  def update(self, variables, parameters):
    if self.multiple:
      values = []
      for i in range(self.columns.GetCount()):
        if self.columns.IsChecked(i):
          values.append(self.columns.GetString(i))
      variables[self.variable] = values
    else:
      variables[self.variable] = self.columns.GetStringSelection()

    
class StringParameter(BaseParameter):
  '''Allows the user to enter a String'''
  attributes = [ ]
  def __init__(self, parent, mainframe, parameterxml):
    BaseParameter.__init__(self, parent, mainframe, parameterxml)
    self.textfield = wx.TextCtrl(self.parent, size=[400, -1])
    self.mask = parameterxml.getAttribute('mask')
    if self.default:
      self.textfield.SetValue(self.default)
    self.parent.sizer.Add(self.textfield)

  def update(self, variables, parameters):
    assert re.match(self.mask, self.textfield.GetValue()), lang('Please enter a valid string.')
    variables[self.variable] = self.textfield.GetValue()


class IntParameter(StringParameter):
  '''Allows the user to enter an integer'''
  def __init__(self, parent, mainframe, parameterxml):
    StringParameter.__init__(self, parent, mainframe, parameterxml)
    if not self.mask:
      self.mask = "-{0,1}\d+"
    self.min = parameterxml.hasAttribute('min') and int(parameterxml.getAttribute('min')) or None
    self.max = parameterxml.hasAttribute('max') and int(parameterxml.getAttribute('max')) or None
    
  def update(self, variables, parameters):
    assert re.match(self.mask, self.textfield.GetValue()), lang('Please enter a valid integer.')
    value = int(self.textfield.GetValue())
    assert self.min == None or value >= self.min, lang('Please enter an integer greater than or equal to') + ' ' + str(self.min)
    assert self.max == None or value <= self.max, lang('Please enter an integer less than or equal to') + ' ' + str(self.max)
    variables[self.variable] = value


class FloatParameter(StringParameter):
  '''Allows the user to enter a float'''
  def __init__(self, parent, mainframe, parameterxml):
    StringParameter.__init__(self, parent, mainframe, parameterxml)
    if not self.mask:
      self.mask = "-{0,1}\d+(\.\d*)*"
    self.min = parameterxml.hasAttribute('min') and float(parameterxml.getAttribute('min')) or None
    self.max = parameterxml.hasAttribute('max') and float(parameterxml.getAttribute('max')) or None
    
  def update(self, variables, parameters):
    assert re.match(self.mask, self.textfield.GetValue()), lang('Please enter a valid floating-point number.')
    value = float(self.textfield.GetValue())
    assert self.min == None or value >= self.min, lang('Please enter a number greater than or equal to') + ' ' + str(self.min)
    assert self.max == None or value <= self.max, lang('Please enter a number less than or equal to') + ' ' + str(self.max)
    variables[self.variable] = value
    
    
class ChoiceParameter(BaseParameter):
  '''Allows the user to choose from a list of values'''
  def __init__(self, parent, mainframe, parameterxml):
    BaseParameter.__init__(self, parent, mainframe, parameterxml)
    self.choices = []
    for option in [ child for child in parameterxml.childNodes if child.nodeName == 'option' ]:
      text = ''.join([ child.nodeValue for child in option.childNodes if child.nodeType == child.TEXT_NODE or child.nodeType == child.CDATA_SECTION_NODE ])
      value = option.getAttribute('value') or text
      self.choices.append([value, text])
    self.multiple = parameterxml.hasAttribute('multiple') and parameterxml.getAttribute('multiple').lower() == 'true'
    if self.paramtype == 'choice':
      self.listbox = wx.Choice(self.parent, choices=[ text for value, text in self.choices ])
    elif self.multiple:
      self.listbox = wx.CheckListBox(self.parent, size=[-1, 100], choices=[ text for value, text in self.choices ])
      self.listbox.SetMinSize([400,-1])
    else:
      self.listbox = wx.ListBox(self.parent, size=[-1, 100], choices=[ text for value, text in self.choices ])
      self.listbox.SetMinSize([400,-1])
    self.listbox.Select(0)
    if self.default:
      for i in range(len(self.choices)):
        if self.choices[i][0] == self.default:
          self.listbox.Select(i)
    self.parent.sizer.Add(self.listbox)
    
  def update(self, variables, parameters):
    if self.paramtype == 'choice' or not self.multiple:
      variables[self.variable] = self.choices[self.listbox.GetSelection()][0]
    else:
      variables[self.variable] = [self.choices[i][0] for i in range(len(self.choices)) if self.listbox.IsChecked(i)]



class FilenameParameter(BaseParameter):
  '''Allows the user to select a filename'''
  attributes = [ ]
  def __init__(self, parent, mainframe, parameterxml):
    BaseParameter.__init__(self, parent, mainframe, parameterxml)
    self.sizer = wx.BoxSizer(wx.HORIZONTAL)
    # the text fielld
    self.textfield = wx.TextCtrl(self.parent, size=[350, -1])
    if self.default:
      self.textfield.SetValue(self.default)
    self.sizer.Add(self.textfield)
    # the file button
    self.filebutton = wx.Button(self.parent, label="Choose...")
    self.filebutton.Bind(wx.EVT_BUTTON, self.onChooseFileButton)
    self.sizer.AddSpacer(10)
    self.sizer.Add(self.filebutton)
    self.parent.sizer.Add(self.sizer)

  def update(self, variables, parameters):
    variables[self.variable] = self.textfield.GetValue()
    
  def onChooseFileButton(self, event):
    filename = wx.FileSelector("Choose a File...")
    if filename:
      self.textfield.SetValue(filename)


class DirectorynameParameter(FilenameParameter):
  '''Allows the user to select a directory name'''
  def onChooseFileButton(self, event):
    dirname = wx.DirSelector()
    if dirname:
      self.textfield.SetValue(dirname)
      
    
    
# map used to create parameters from the xml
# keys must be lowercase (so they can be found case insensitively)
PARAMETERS = {
  'database': DatabaseConnectionParameter,
  'table':    TableParameter,
  'column':   ColumnParameter,
  'int':      IntParameter,
  'float':    FloatParameter,
  'string':   StringParameter,
  'list':     ChoiceParameter,
  'choice':   ChoiceParameter,
  'filename': FilenameParameter,
  'dirname':  DirectorynameParameter,
}



   
