#!/usr/bin/python

####################################################################################
#                                                                                  #
# Copyright (c) 2005 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This file is part of Picalo.                                                     #
#                                                                                  #
# Picalo is free software; you can redistribute it and/or modify                   #
# it under the terms of the GNU General Public License as published by             #
# the Free Software Foundation; either version 2 of the License, or                # 
# (at your option) any later version.                                              #
#                                                                                  #
# Picalo is distributed in the hope that it will be useful,                        #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                   #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    #
# GNU General Public License for more details.                                     #
#                                                                                  #
# You should have received a copy of the GNU General Public License                #
# along with Foobar; if not, write to the Free Software                            #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA        #
#                                                                                  #
####################################################################################

import wx, os, os.path, sys, threading, traceback, time
import Utils, Spreadsheet, Dialogs
from picalo import Table, TableArray, TableList, Database
from Languages import lang
from QueryBuilder import QueryBuilder
from picalo.base.Global import make_valid_variable

##############################################
###  PROGRAMMERS NOTE:
###
###  This class is quite a bit more complex than it should be because it scans the disk in
###  a thread and updates the GUI in onIdle.  This is necessary because the disk scanning
###  can take a long time, especially if the target directory is on a slow, networked
###  drive.  Since we can't predict how long it will take, it cannot be run in the
###  onIdle function -- doing this freezes the GUI while it is working.  We also can't
###  modify the GUI from a thread -- it has to be done in the wx GUI main thread.
###
###  Therefore, we have one thread the scan the disk (so it can take as long as it wants)
###  and the GUI updating using onIdle (since we know this take very little time).
###  The outcome is a complex piece of code with a data structure to link the two
###  threads.  Sorry for this but I don't see another way.
###


# the delay between updates in the onIdle method
UPDATE_DELAY = 1

# how many levels deep to let the tree go -- if it goes too deep, it takes all the system resources in idle time and locks the system
MAX_TREE_DEPTH = 2

# used in fileSaveAs below
FILENAME_EXTENSIONS = (
  ( Table,                'Tables (*.pco)|*.pco',               '.pco', ),
  ( TableList,            'Tables (*.pco)|*.pco',               '.pco', ),
  ( TableArray,           'Tables (*.pco)|*.pco',               '.pco', ),
  ( Database._Connection, 'Database Connections (*.pcd)|*.pcd', '.pcd', ),
  ( Database.Query,       'Queries (*.pcq)|*.pcq',              '.pcq', ),
)
        
#######################################################
###   Simple extension to TreeCtrl to allow sorting

class MyTreeCtrl(wx.TreeCtrl):
  def OnCompareItems(self, item1, item2):
    return cmp(self.GetItemPyData(item1), self.GetItemPyData(item2))
        

########################################################
###   ObjectTree class

class ObjectTree(wx.PyPanel):
  '''The left-side object tree in the main GUI.  This panel continually
     polls memory and disk to ensure that the tree is always up to
     date.
  '''
  def __init__(self, parent, mainframe):
    '''Constructor'''
    wx.PyPanel.__init__(self, parent)
    self.parent = parent
    self.mainframe = mainframe
    self.object_cache_thread = threading.Thread(target=self.run)
    self.object_cache_thread.setDaemon(True)
    self.object_cache_thread_running = False
    self.tree_dirty = False
    
    # set up the initialtree
    self.tree = MyTreeCtrl(self)
    imglist = wx.ImageList(16, 16)
    for name, filename in TREEIMAGES:
      imglist.Add(Utils.getBitmap(filename))
    self.tree.AssignImageList(imglist)
    self.tree.Bind(wx.EVT_TREE_ITEM_ACTIVATED, self.onTreeItemDouble)
    self.tree.Bind(wx.EVT_TREE_ITEM_RIGHT_CLICK, self.onTreeRightDown)

    # set up the sizer
    self.SetSizer(wx.BoxSizer(wx.VERTICAL))
    self.GetSizer().Add(self.tree, proportion=1, flag=wx.EXPAND|wx.ALL)

    # tree menu when right clicked
    self.rightclicktreeinfo = None  # the tree item that was right clicked
    self.treemenutoptable = Utils.MenuManager(self, wx.Menu(), [
      Utils.MenuItem(lang('New Table...'), lang('Create a new table'), self.mainframe.menuFileNewTable),
      Utils.MenuItem(lang('Open Table...'), lang('Open a Picalo table from the disk'), self.mainframe.menuFileOpen),
    ])
    self.treemenutable = Utils.MenuManager(self, wx.Menu(), [
      Utils.MenuItem(lang('Open Table'), lang('Open this table into the viewer'), self.onTreeItemOpen),
      Utils.MenuItem(lang('Save'), lang('Save this connection to disk as part of this project'), self.menuSave),
      Utils.MenuItem(lang('Save As...'), lang('Save this connection to disk under a new filename'), self.menuSaveAs),
      Utils.MenuItem('SEPARATOR'),
      Utils.MenuItem(lang('Close Table...'), lang('Close this table if it is open.'), self.menuCloseTab),
      Utils.MenuItem(lang('Delete From Disk...'), lang('Delete the table from disk.'), self.menuDeleteFromDisk),
      Utils.MenuItem('SEPARATOR'),
      Utils.MenuItem(lang('Copy Table...'), lang('Copy this table to another name'), Utils.MenuCaller(self.runRightClickFunction, Spreadsheet, 'menuCopyTable')),
      Utils.MenuItem(lang('Rename Table...'), lang('Rename this table to another name'), Utils.MenuCaller(self.runRightClickFunction, Spreadsheet, 'menuRenameTable')),
      Utils.MenuItem(lang('Export Table...'), lang('Export this table to CSV, TSV, or Excel format'), Utils.MenuCaller(self.runRightClickFunction, Spreadsheet, 'menuExportTable')),
      Utils.MenuItem('SEPARATOR'),
      Utils.MenuItem(lang('Combine Into Regular Table...'), lang('Recombine a table list back into a single table'), Utils.MenuCaller(self.runRightClickFunction, Spreadsheet, 'menuCombineTableList')),
    ])
    self.treemenutopdb = Utils.MenuManager(self, wx.Menu(), [
      Utils.MenuItem(lang('New Connection...'), lang('Connect to a database'), Dialogs.DatabaseConnection(self.mainframe)),
      Utils.MenuItem(lang('Open Connection...'), lang('Open a Picalo database connection from the disk'), self.mainframe.menuFileOpen),
    ])
    self.treemenudb = Utils.MenuManager(self, wx.Menu(), [
      Utils.MenuItem(lang('Open'), lang('Open this database connection'), self.onTreeItemOpen),
      Utils.MenuItem(lang('Save'), lang('Save this connection to disk as part of this project'), self.menuSave),
      Utils.MenuItem(lang('Save As...'), lang('Save this connection to disk under a new filename'), self.menuSaveAs),
      Utils.MenuItem('SEPARATOR'),
      Utils.MenuItem(lang('Refresh'), lang('Refresh the table list for this connection'), self.onTreeItemRefresh),
      Utils.MenuItem(lang('New Query...'), lang('Open the visual query builder to run a new query.'), QueryBuilder(self.mainframe)),
      Utils.MenuItem('SEPARATOR'),
      Utils.MenuItem(lang('Close Connection...'), lang('Close this connection and remove it from memory'), self.menuRemoveFromMemory),
    ])
    self.treemenurelation = Utils.MenuManager(self, wx.Menu(), [
      Utils.MenuItem(lang('Open'), lang('Open the table.'), self.onTreeItemOpen),
      Utils.MenuItem('SEPARATOR'),
      Utils.MenuItem(lang('Count Records'), lang('Count the records in this table'), self.onTreeItemCountRecords),
      Utils.MenuItem(lang('Copy To Picalo Table'), lang('Copy this database relation to a read/write Picalo table.'), self.onTreeItemCopyDBToTable),
    ])
    self.treemenutopqueries = Utils.MenuManager(self, wx.Menu(), [
      Utils.MenuItem(lang('New Query...'), lang('Open the visual query builder to run a new query.'), QueryBuilder(self.mainframe)),
      Utils.MenuItem(lang('Open Query...'), lang('Open a Picalo query from the disk'), self.mainframe.menuFileOpen),
    ])
    self.treemenuquery = Utils.MenuManager(self, wx.Menu(), [
      Utils.MenuItem(lang('Open'), lang('Open the query.'), self.onTreeItemOpen),
      Utils.MenuItem(lang('Save'), lang('Save this query to disk as part of this project'), self.menuSave),
      Utils.MenuItem(lang('Save As...'), lang('Save this query to disk under a new filename'), self.menuSaveAs),
      Utils.MenuItem('SEPARATOR'),
      Utils.MenuItem(lang('Copy To Picalo Table'), lang('Copy this database query to a read/write Picalo table.'), self.onTreeItemCopyDBToTable),
      Utils.MenuItem('SEPARATOR'),
      Utils.MenuItem(lang('Close Query...'), lang('Close this query if it is open.'), self.menuCloseTab),
      Utils.MenuItem(lang('Remove From Memory...'), lang('Remove this query from active memory.'), self.menuRemoveFromMemory),
      Utils.MenuItem(lang('Delete From Disk...'), lang('Delete the query from disk.'), self.menuDeleteFromDisk),
    ])
    self.treemenutopscripts = Utils.MenuManager(self, wx.Menu(), [
      Utils.MenuItem(lang('New Script'), lang('Create a new script'), self.mainframe.menuFileNewScript),
      Utils.MenuItem(lang('Open...'), lang('Open an existing script'), self.mainframe.menuFileOpen),
    ])
    self.treemenuscripts = Utils.MenuManager(self, wx.Menu(), [
      Utils.MenuItem(lang('Open'), lang('Open this script in the editor'), self.onTreeItemOpen),
      Utils.MenuItem('SEPARATOR'),
      Utils.MenuItem(lang('Close Script...'), lang('Close this script if it is open.'), self.menuCloseTab),
      Utils.MenuItem(lang('Delete From Disk...'), lang('Delete the script from disk.'), self.menuDeleteFromDisk),
    ])
    self.treemenumap = { # maps the image name to the correct map
      'table'     : self.treemenutable,
      'dbconn'    : self.treemenudb,
      'topquery'  : self.treemenutopqueries,
      'query'     : self.treemenuquery,
      'topscript' : self.treemenutopscripts,
      'script'    : self.treemenuscripts,
      'dbtable'   : self.treemenurelation,
      'topdb'     : self.treemenutopdb,
      'toptable'  : self.treemenutoptable,
    }

    # update the project tree
    self.loadProject()
    
    
  def loadProject(self):
    '''Updates the project entirely by clearing the tree, essentially starting over'''
    # ensure we have a project directory
    if not os.path.exists(self.mainframe.projectdir):
      return  # the thread below will prompt the user for a project dir
      
    # first clear everything out
    self.tree.DeleteAllItems()
    
    # now repopulate
    projectname = os.path.split(self.mainframe.projectdir)[1]
    self.treeroot = self.tree.AddRoot(projectname, TREEIMAGEMAP['project'])
    self.treetables = self.tree.AppendItem(self.treeroot, lang('Tables'), TREEIMAGEMAP['toptable'])
    self.treedatabases = self.tree.AppendItem(self.treeroot, lang('Databases'), TREEIMAGEMAP['topdb'])
    self.treequeries = self.tree.AppendItem(self.treeroot, lang('Queries'), TREEIMAGEMAP['topquery'])
    self.treescripts = self.tree.AppendItem(self.treeroot, lang('Scripts'), TREEIMAGEMAP['topscript'])
    self.tree.Expand(self.treeroot)
    # on the next onIdle, the tree will automatically regenerate
    
    # update the history
    try:
      historyname = os.path.join(self.mainframe.projectdir, 'Picalo.history')
      if os.path.exists(historyname):
        f = open(historyname)
        self.mainframe.commandlog.clear()
        self.mainframe.commandlog.write(f.read())
        f.close()
    except Exception, e:
      self.mainframe.showError(lang('An error occurred while saving the command history:'), e)
        
    
    
  ####################################################    
  ###   Tree updating code
  
  def run(self):
    '''Thread that checks the project directory for changes'''
    self.object_cache_thread_running = True
    while self.object_cache_thread_running:
      try:
        self.starttime = time.time()
        self.setTreeDirty()

        # ensure we have a project directory
        if not os.path.exists(self.mainframe.projectdir):
          wx.CallAfter(self.promptProjectDirectory)
          
        else:
          # map the project directory
          self.updateTree(0, self.treetables, self.mainframe.projectdir, 'table', '.pco', (Table, TableList, TableArray, ), True)
          self.updateTree(0, self.treedatabases, self.mainframe.projectdir, 'dbconn', '.pcd', (Database._Connection, ), False)
          self.updateTree(0, self.treequeries, self.mainframe.projectdir, 'query', '.pcq', (Database.Query, ), True)
          self.updateTree(0, self.treescripts, self.mainframe.projectdir, 'script', '.py', None, False)
          
          # wait until the gui has run all the wx.CallAfter calls
          wx.CallAfter(self.setTreeDirty, False)  # this will be the final wx.CallAfter, and it removes the dirty flag to let us continue

        # wait until the tree is done
        while self.tree_dirty: 
          time.sleep(0.5)

        # wait a bit if we haven't taken 2 seconds for this (run every 2 second at most)
        timetosleep = 2 - (time.time() - self.starttime)
        if timetosleep > 0:
          time.sleep(timetosleep)
        
      except Exception, e:
        if self.object_cache_thread_running: # sometimes errors throw at shutdown when the thread is still running - ignore these
          traceback.print_exc(file=sys.stdout)
        

  def setTreeDirty(self, dirty=True):
    '''Sets whether the tree is dirty or not.  This is used to stop the running thread from relooping
       until all the gui changes are made'''
    self.tree_dirty = dirty
    
    
  # Programmer note: this function runs in a secondary python thread, so it cannot modify any wx controls directly.
  # It has to call wx.CallAfter which will run the call within the wx thread. 
  def updateTree(self, level, parent, dirpath, typename, extension, objtypes, showinnotebook):
    '''Updates the tree'''
    # if level 0, map all the items in memory to a couple of caches that are used later
    if level == 0 and objtypes != None:
      self.memory_by_id = {}
      self.memory_by_fullpath = {}
      for name, value in self.mainframe.shell.getLocals().items():
        if isinstance(value, objtypes):
          mi = self.memory_by_id.get(id(value))
          if mi != None:
            mi.varnames.append(name)
            mi.varnames.sort(lambda a,b: cmp(a.lower(), b.lower()))
          else:
            mi = MemoryItem(value)
            mi.varnames.append(name)
            self.memory_by_id[id(value)] = mi
            if hasattr(value, 'filename') and value.filename:
              self.memory_by_fullpath[value.filename] = mi
              
    # ensure all tables in memory are showing in the notebook
    if level == 0 and showinnotebook:
      for mi in self.memory_by_id.values():
        for varname in mi.varnames:
          if self.mainframe.getPage(varname):
            break
        else:
          wx.CallAfter(self.mainframe.openTable, mi.varnames[0])
          
    # delete any items in the tree that no longer exist on disk or in memory
    # this also maps the items in the tree so we know what's there later in the function
    tree_by_fullpath = {}
    item = self.tree.GetLastChild(parent)
    while item:
      prev = self.tree.GetPrevSibling(item)
      itemti = self.tree.GetItemPyData(item)
      found = False
      if os.path.exists(itemti.fullpath):  # on disk, so leave in
        found = True
        itemti.treeitem = item
        tree_by_fullpath[itemti.fullpath] = itemti
        mi = self.memory_by_fullpath.get(itemti.fullpath)
        if mi != None:
          mi.treeinfo = itemti
        if itemti.value == None and mi != None:  # see if we need to set the value (just loaded)
          itemti.value = mi.value
        if itemti.value != None and mi == None:  # see if it has been removed from memory
          for varname in itemti.varnames:
            value = self.mainframe.getVariable(varname)
            if value != None and value.__class__ == itemti.value.__class__:
              pass # don't allow closing
            else:
              page = self.mainframe.getPage(varname)
              if page:
                wx.CallAfter(self.mainframe.closePage, page)
          itemti.varnames = []
          itemti.value = None
      if itemti.value != None:                   # if we have a value, see if we're in memory with any varnames that we didn't know about earlier
        mi = self.memory_by_id.get(id(itemti.value))
        if mi != None:
          found = True
          mi.treeinfo = itemti
          itemti.treeitem = item
          itemti.varnames = mi.varnames    # update the varnames in memory
          if itemti.fullpath and not os.path.exists(itemti.fullpath):  # means the file has been deleted
            itemti.fullpath = ''
      if not found:  # see if it still exists by this name as another value of the same type
        for varname in itemti.varnames:
          value = self.mainframe.getVariable(varname)
          if value != None and value.__class__ == itemti.value.__class__:
            found = True
            mi = self.memory_by_id.get(id(value))
            mi.treeinfo = itemti
            break
      if not found:   # not here anymore (either in memory or on disk), so delete
        for varname in itemti.varnames:
          page = self.mainframe.getPage(varname)
          if page:
            wx.CallAfter(self.mainframe.closePage, page)
        itemti.value = None
        wx.CallAfter(self.deepDeleteNode, item)
        self.starttime = 0
      item = prev
    
    # go through the files in the directory and ensure each one is in the tree
    try:
      for filename in os.listdir(dirpath):
        fullpath = os.path.join(dirpath, filename).replace('\\', '/')
        mi = self.memory_by_fullpath.get(fullpath)
        if os.path.isdir(fullpath) and filename[0] != '.':  # don't include hidden directories that start with period
          # add this directory if needed
          if not tree_by_fullpath.has_key(fullpath):
            treeinfo = TreeInfo('directory', fullpath, 'folder', 'folderopen')
            wx.CallAfter(self.addNewChild, parent, treeinfo)
            self.starttime = 0  # this forces a rerun of the refresh algorithm immediately, so the subdirectories are added quickly (subdirs are only parsed on the next run because you have to wait for the CallAfter to run)
          # recurse to subdirectory if it's been added to the tree
          if level < MAX_TREE_DEPTH and tree_by_fullpath.has_key(fullpath):
            treeinfo = tree_by_fullpath[fullpath]
            if treeinfo.treeitem == None or not treeinfo.treeitem.IsOk():
              treeinfo.treeitem = self.getTreeItem(fullpath, parent)
            if treeinfo.treeitem:   # can't recurse to subdirectories until node has been added
              self.updateTree(level+1, treeinfo.treeitem, fullpath, typename, extension, objtypes, showinnotebook)
            
        elif os.path.splitext(filename)[1] == extension:  # item of correct type
          if not tree_by_fullpath.has_key(fullpath):   # item is not in the tree at this level yet
            treeinfo = TreeInfo(typename, fullpath, typename)
            wx.CallAfter(self.addNewChild, parent, treeinfo)
            self.starttime = 0
        
    except OSError:  # thrown when permission denied for directory
      return
      
    # finally, add any in-memory objects that are not on disk at all
    if level == 0 and objtypes != None:
      for mi in self.memory_by_id.values():
        if not mi.treeinfo:  # not in the tree yet
          treeinfo = TreeInfo(typename, '', typename, value=mi.value)
          treeinfo.varnames = mi.varnames
          wx.CallAfter(self.addNewChild, parent, treeinfo)
          self.starttime = 0
          
    # finally, check all titles and sort the list in the tree
    needsSorting = False
    currentTreeInfo = []
    sortedTreeInfo = []
    item, cookie = self.tree.GetFirstChild(parent)
    while item:
      treeinfo = self.tree.GetItemPyData(item)
      if treeinfo != None:
        if self.tree.GetItemText(item) != treeinfo.getTitle():
          wx.CallAfter(self.updateItemTitle, treeinfo)
          needsSorting = True
          # special processing for database items so they refresh
          if treeinfo.typename == 'dbconn' and treeinfo.value != None:
            wx.CallAfter(self.refreshItem, treeinfo)
          # if the item is not open anymore, ensure it doesn't have children
          if treeinfo.value == None:
            wx.CallAfter(self.deepDeleteChildren, treeinfo.treeitem)
        currentTreeInfo.append(treeinfo)
        sortedTreeInfo.append(treeinfo)
        item, cookie = self.tree.GetNextChild(parent, cookie)
    sortedTreeInfo.sort()
    if currentTreeInfo != sortedTreeInfo:
      needsSorting = True
    if needsSorting:
      wx.CallAfter(self.tree.SortChildren, parent)
      
    
  def addNewChild(self, parent, treeinfo):
    '''Adds a new item to the tree.  This is meant to be called in wx.CallAfter.'''
    # insert into tree
    item, cookie = self.tree.GetFirstChild(parent)
    newitem = None
    pos = 0
    while item:
      itemti = self.tree.GetItemPyData(item)
      if cmp(itemti, treeinfo) > 0:
        newitem = self.tree.InsertItemBefore(parent, pos, treeinfo.getTitle())
        self.tree.SetItemPyData(newitem, treeinfo)
        treeinfo.treeitem = newitem
        break
      item, cookie = self.tree.GetNextChild(parent, cookie)
      pos += 1
    # append to the end if not inserted yet
    if newitem == None:
      newitem = self.tree.AppendItem(parent, treeinfo.getTitle())
      self.tree.SetItemPyData(newitem, treeinfo)
      treeinfo.treeitem = newitem
    # set the item image
    if treeinfo.image:
      self.tree.SetItemImage(newitem, TREEIMAGEMAP[treeinfo.image])
    if treeinfo.expandedimage:
      self.tree.SetItemImage(newitem, TREEIMAGEMAP[treeinfo.expandedimage], wx.TreeItemIcon_SelectedExpanded)

      
  def updateItemTitle(self, treeinfo):
    '''Updates the title of an item.  This is a separate method so it can be called with wx.CallAfter.'''
    self.tree.SetItemText(treeinfo.treeitem, treeinfo.getTitle())


  def deepDeleteNode(self, parent):
    '''Deletes a tree node, including any children. This is a separate method so it can be called with wx.CallAfter.'''
    self.deepDeleteChildren(parent)
    self.tree.Delete(parent)
    
    
  def deepDeleteChildren(self, parent):
    '''Deletes all children of a node'''
    item = self.tree.GetLastChild(parent)
    while item:
      prev = self.tree.GetPrevSibling(item)
      self.deepDeleteNode(item)
      item = prev


  def promptProjectDirectory(self, event=None):
    '''Prompt the user for a project directory. This is a separate method so it can be called with wx.CallAfter.'''
    while not os.path.exists(self.mainframe.projectdir):
      self.mainframe.menuFileOpenProject()
      if not os.path.exists(self.mainframe.projectdir):
        if wx.MessageBox(lang('Picalo requires a project folder to work from.  Would you like to quit instead?'), lang('Project Folder'), style=wx.YES_NO) == wx.YES:
          self.mainframe.onClose()
    self.setTreeDirty(False)
    
            
  def getTreeItem(self, fullpath, parent):
    '''Searches the tree (starting at parent) for the node item with the given full path.'''
    item, cookie = self.tree.GetFirstChild(parent)
    while item:
      # check this item
      pydata = self.tree.GetItemPyData(item)
      if pydata != None and pydata.fullpath == fullpath:
        return item
      # recurse to my children
      if self.tree.GetChildrenCount(item) > 0:
        treeinfo = self.getTreeItem(fullpath, item)
        if treeinfo:
          return treeinfo
      # go to my next sibling
      item, cookie = self.tree.GetNextChild(parent, cookie)
    return None


          
  ####################################################################          
  ###   Menu options
          
  def runRightClickFunction(self, mod, funcname):
    '''Runs a menu function (in response to a right click)'''
    if self.rightclicktreeinfo:
      assert len(self.rightclicktreeinfo.varnames) > 0, 'Please open this item first.'
      func = getattr(mod, funcname)
      func(self.mainframe, self.rightclicktreeinfo.varnames[0])


  def onTreeItemDouble(self, event=None, treeinfo=None):
    '''Called when the user double clicks an item in the tree.
       Also called by onTreeItemOpen below, since open is essentially a double-click.
    '''
    if treeinfo == None:
      treeinfo = self.tree.GetItemPyData(event.GetItem())
    if treeinfo:
      if treeinfo.typename == 'table' or treeinfo.typename == 'query':
        if treeinfo.value != None:  # already open
          self.mainframe.openTable(treeinfo.varnames[0])
        else:
          self.mainframe.openFilename(treeinfo.fullpath)
          
      elif treeinfo.typename == 'dbconn':
        if treeinfo.value == None:  # not open yet
          self.mainframe.openFilename(treeinfo.fullpath)
          
      elif treeinfo.typename == 'dbtable':
        parentinfo = self.tree.GetItemPyData(self.tree.GetItemParent(treeinfo.treeitem))
        self.mainframe.openTable(parentinfo.varnames[0] + '.' + treeinfo.getTitle())

      elif treeinfo.typename == 'script':
        self.mainframe.openFilename(treeinfo.fullpath)
        
    if event:    
      event.Skip()          


  def onTreeRightDown(self, event=None):
    '''Called when the user right-clicks an item in the tree'''
    self.rightclicktreeinfo = self.tree.GetItemPyData(event.GetItem())
    if self.rightclicktreeinfo and self.treemenumap.has_key(self.rightclicktreeinfo.typename):
      self.rightclicktreeinfo.treeitem = event.GetItem()
      self.PopupMenu(self.treemenumap[self.rightclicktreeinfo.typename].topmenu)
    event.Skip()


  def onTreeItemOpen(self, event):
    '''Triggered when the user right-clicks a tree item and selects Open'''
    if self.rightclicktreeinfo:
      treeinfo = self.rightclicktreeinfo
      self.onTreeItemDouble(treeinfo=treeinfo)


  def menuCloseTab(self, event=None):
    '''Closes an active tab'''
    if self.rightclicktreeinfo:
      assert len(self.rightclicktreeinfo.varnames) > 0, 'Please open this item first.'
      page = self.mainframe.getPage(self.rightclicktreeinfo.varnames[0]) or self.mainframe.getPageByFilename(self.rightclicktreeinfo.fullpath)
      if page:
        self.mainframe.closePage(page)


  def menuDeleteFromDisk(self, event=None):
    '''Deletes an item from disk'''
    if self.rightclicktreeinfo:
      if wx.MessageBox(lang('This file will be permanently removed from disk.  Continue?'), lang('Delete'), style=wx.YES_NO) == wx.YES:
        # close the object's tab if open
        page = self.mainframe.getPageByFilename(self.rightclicktreeinfo.fullpath)
        if page:
          self.mainframe.closePage(page)
        # delete from active memory if in memory
        if self.rightclicktreeinfo.value != None and self.mainframe.shell.varExists(self.rightclicktreeinfo.varnames[0]):  
          self.mainframe.execute('del %s' % self.rightclicktreeinfo.varnames[0])
        # remove from disk if the path exists
        if self.rightclicktreeinfo.fullpath and os.path.exists(self.rightclicktreeinfo.fullpath):
          self.mainframe.execute('os.remove("%s")' % self.rightclicktreeinfo.fullpath)

      
  def menuRemoveFromMemory(self, event=None):
    '''Removes an item from active memory'''
    if self.rightclicktreeinfo and self.rightclicktreeinfo.value != None and self.mainframe.shell.varExists(self.rightclicktreeinfo.varnames[0]):
      if self.rightclicktreeinfo.typename == 'table':
        if self.rightclicktreeinfo.value.is_changed():
          if wx.MessageBox(lang('This object has changed.  Do you want to save it?'), lang('Remove From Memory'), style=wx.YES_NO) == wx.YES:
            s = Spreadsheet.Spreadsheet(self.mainframe, self.mainframe, self.rightclicktreeinfo.varnames[0], self.rightclicktreeinfo.value)
            if not s.menuFileSave():
              del s
              return
            del s
        self.mainframe.execute('del %s' % (self.rightclicktreeinfo.varnames[0]))
        
      elif self.rightclicktreeinfo.typename == 'dbconn':
        if wx.MessageBox(lang('Close this database connection?'), lang('Close Database'), style=wx.YES_NO) == wx.YES:
          self.mainframe.execute([
            self.rightclicktreeinfo.varnames[0] + '.close()',
            'del ' + self.rightclicktreeinfo.varnames[0],
          ])


  def onTreeItemRefresh(self, event):
    '''Triggered when the user right-clicks a tree item and selects Refresh'''
    if self.rightclicktreeinfo and self.rightclicktreeinfo.value != None:
      self.refreshItem(self.rightclicktreeinfo)
    
    
  def refreshItem(self, treeinfo):
    '''Refreshes an item on the tree'''
    # delete all current children
    self.tree.DeleteChildren(treeinfo.treeitem)
    
    # repopulate
    if isinstance(treeinfo.value, Database._Connection):
      for tablename in treeinfo.value.list_tables():
         childinfo = TreeInfo('dbtable', tablename, 'dbtable')
         childinfo.varnames = [ tablename ]
         self.addNewChild(treeinfo.treeitem, childinfo)
        
    # if a table, ensure it is open in a tab
    if isinstance(treeinfo.value, (Table, TableList, TableArray)) and not self.mainframe.getPage(treeinfo.name):
      self.mainframe.openTable(treeinfo.name)  
        
  
  def onTreeItemCountRecords(self, event):
    '''Displays the record count for a database table'''
    if self.rightclicktreeinfo:
      parentinfo = self.tree.GetItemPyData(self.tree.GetItemParent(self.rightclicktreeinfo.treeitem))
      db = self.mainframe.shell.getVariable(parentinfo.varnames[0])
      cursor = db.cursor()
      results = cursor.query("SELECT COUNT(*) FROM " + self.rightclicktreeinfo.varnames[0])
      row = results.fetchone()
      cursor.close()
      wx.MessageBox(lang('This table has %s records.') % (row[0], ), lang('Count Records'))
        
        
  def onTreeItemCopyDBToTable(self, event):
    '''Copies the given database relation to a Picalo table'''
    if self.rightclicktreeinfo:
      # get the new table name
      assert len(self.rightclicktreeinfo.varnames) > 0, 'Please open this item first.'
      if self.rightclicktreeinfo.typename == 'dbtable':
        parentinfo = self.tree.GetItemPyData(self.tree.GetItemParent(self.rightclicktreeinfo.treeitem))
        objname = parentinfo.varnames[0] + '.' + self.rightclicktreeinfo.varnames[0]
      else:
        objname = self.rightclicktreeinfo.fullpath or self.rightclicktreeinfo.varnames[0]
      newtablename = objname
      if newtablename.find('.') >= 0:
        newtablename = os.path.splitext(newtablename)[1][1:]
      if newtablename == objname:
        newtablename += '_copy'
      while True:
        newtablename = wx.GetTextFromUser("Please enter the table name", "Copy To Picalo Table", newtablename)
        if not newtablename:
          return
        if not self.mainframe.getVariable(newtablename) or wx.MessageBox('A variable with this name already exists.  Overwrite it?', 'Copy To Picalo Table', style=wx.YES_NO) == wx.YES:
          break
    
      # copy the table
      newtablename = make_valid_variable(newtablename)
      self.mainframe.execute([
        "%s = %s[:]" % (newtablename, objname),
      ])
    
   
  def menuSave(self, event=None, value=None):
    '''Saves an object in the tree (database connection or query)'''
    if value != None:  # emulate a right click on the given value
      self.rightclicktreeinfo = self.getTreeInfo(value)
    if not self.rightclicktreeinfo or self.rightclicktreeinfo.value == None:
      return False
    name = self.rightclicktreeinfo.varnames[0]
    obj = self.rightclicktreeinfo.value
    if obj.filename:
      self.mainframe.execute('%s.save("%s")' % (name, obj.filename.replace('"', '\\"')))
      return True
    else:
      return self.menuSaveAs()
    

  def menuSaveAs(self, event=None, value=None):
    '''Saves an object in the tree with a Save As'''
    if value != None:  # emulate a right click on the given value
      self.rightclicktreeinfo = self.getTreeInfo(value)
    if not self.rightclicktreeinfo or self.rightclicktreeinfo.value == None:
      return False
    name = self.rightclicktreeinfo.varnames[0]
    obj = self.rightclicktreeinfo.value
    extension = None
    for ext in FILENAME_EXTENSIONS:
      if isinstance(obj, ext[0]):
        extension = ext
        break
    assert extension != None, 'Unknown file type cannot be saved by ObjectTree.menuSaveAs'
    while True:
      dlg = wx.FileDialog(self, message=lang('Save "%s" As...' % (name, )), defaultDir=self.mainframe.projectdir, defaultFile=name, style=wx.SAVE | wx.CHANGE_DIR, wildcard=extension[1])
      if dlg.ShowModal() == wx.ID_OK:
        filename = dlg.GetPath()
        if os.path.splitext(filename)[1] != extension[2]:
          filename += extension[2]
        if os.path.exists(filename):
          if wx.MessageBox(lang('A file with this name already exists.  Overwrite it?'), lang('Save As'), style=wx.YES_NO) != wx.YES:
            name = os.path.splitext(os.path.split(filename)[1])[0]
            continue
        cmd = name + '.save("' + filename.replace('\\', '/').replace('"', '\\"') + '")'
        self.mainframe.execute(cmd)
        return True
      else:
        return False
    
    
  def getTreeInfo(self, obj):
    '''Searches the tree and returns the TreeInfo object related to the given value'''
    def recurseTree(parent):
      item, cookie = self.tree.GetFirstChild(parent)
      while item:
        treeinfo = self.tree.GetItemPyData(item)
        if treeinfo != None and id(treeinfo.value) == id(obj):
          return treeinfo
        treeinfo = recurseTree(item)
        if treeinfo:
          return treeinfo
        item, cookie = self.tree.GetNextChild(parent, cookie)
      return None
    return recurseTree(self.treeroot)
    
  

########################################################
###   Information about a tree node

class TreeInfo:
  '''Holds information about files/variables.  Used when updating the tree.'''
  def __init__(self, typename, fullpath, image, expandedimage=None, value=None, treeitem=None):
    '''Constructor'''
    self.typename = typename
    self.fullpath = fullpath
    self.image = image
    self.expandedimage = expandedimage
    self.value = value
    self.treeitem = treeitem  # this can go out of scope, so always check treeitem.IsOk()
    self.varnames = []
    
    
  def getTitle(self):
    '''Calculates the title of this item'''
    filename = os.path.split(self.fullpath)[1]
    if self.typename == 'directory':  # a directory
      return filename
    if self.typename == 'dbtable':  # a database table
      return self.fullpath
    if len(self.varnames) > 0 and filename:  # in memory and on disk
      return ', '.join(self.varnames) + ' <' + filename + '>'
    if filename:  # on disk only
      return '<' + filename + '>'
    return ', '.join(self.varnames)
    
  
  def getComparatorTitle(self):
    '''Calculates a name suitable for sorting by'''
    filename = os.path.split(self.fullpath)[1]
    if self.typename == 'directory':  # a directory
      return filename.lower()
    if self.typename == 'dbtable':  # a database table
      return self.fullpath.lower()
    if len(self.varnames) > 0 and filename:  # in memory and on disk
      return (' '.join(self.varnames)).lower()
    if filename:  # on disk only
      return filename.lower()
    return (' '.join(self.varnames)).lower()
    
    
  def __cmp__(self, other):
    '''Compares two TreeInfo items for sorting in the tree. Directories go first, then regular files, then alpha sorting within each group.'''
    if isinstance(other, TreeInfo):
      if self.typename == other.typename:
        return cmp(self.getComparatorTitle(), other.getComparatorTitle())
      elif self.typename == 'directory':
        return -1
      else:
        return 1
    return cmp(id(self), id(other))


class MemoryItem:
  '''A temporary storage item for holding information about an item in memory.  Used in updateTree.'''
  def __init__(self, value):
    self.varnames = []
    self.value = value
    self.treeinfo = None
    
    
    
########################################################
###   Images used in the tree    
    
TREEIMAGES = (
  ( 'project',      'project.png',         ),
  ( 'toptable',     'toptable.png',        ),
  ( 'table',        'table.png',           ),
  ( 'topdb',        'topdb.png',           ),
  ( 'dbconn',       'dbconn.png',          ),
  ( 'dbtable',      'databasetable.png',   ),
  ( 'topquery',     'topquery.png',        ),
  ( 'query',        'query.png',           ),
  ( 'topscript',    'topscript.png',       ),
  ( 'script',       'script.png',          ),
  ( 'folder',       'folder_blue.png',     ),
  ( 'folderopen',   'folder_blue_open.png' ),
)
TREEIMAGEMAP = dict((item[0], i) for i, item in enumerate(TREEIMAGES))
  
  
  
  
  