#!/usr/bin/python

####################################################################################
#                                                                                  #
# Copyright (c) 2005 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This file is part of Picalo.                                                     #
#                                                                                  #
# Picalo is free software; you can redistribute it and/or modify                   #
# it under the terms of the GNU General Public License as published by             #
# the Free Software Foundation; either version 2 of the License, or                # 
# (at your option) any later version.                                              #
#                                                                                  #
# Picalo is distributed in the hope that it will be useful,                        #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                   #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    #
# GNU General Public License for more details.                                     #
#                                                                                  #
# You should have received a copy of the GNU General Public License                #
# along with Foobar; if not, write to the Free Software                            #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA        #
#                                                                                  #
####################################################################################
#
# 12 Sep 2005 First version
#
####################################################################################


import wx
from Languages import lang


class NotebookComponent:
  def __init__(self, notebook, mainframe, title='Untitled'):
    '''This constructor must be called AFTER the wx constructor so we have a window already'''
    self.notebook = notebook
    self.mainframe = mainframe
    self.title = title
    self.menuitems = []
    self.toolbar = None
    self.tabmenuitems = []
    self.tabmenu = None
    self.finddlg = None  # so we can keep state in find dialogs
    self.Bind(wx.EVT_SET_FOCUS, self.onFocus)
    
    
  def onFocus(self, event=None):
    '''Handles an on focus event.  The MainFrame also calls this 
       when the application is focused.
    '''
    self.mainframe.windowFocus(self)
    self.onFocusSub(event)
    event.Skip()
        

  def onFocusSub(self, event=None):
    '''Allows subclasses to add behavior to the onFocus method'''
    pass
    
    
  def onCut(self, event=None):
    '''Processes a clipboard cut operation'''
    pass
    
    
  def onCopy(self, event=None):
    '''Processes a clipboard copy operation'''
    pass


  def onPaste(self, event=None):
    '''Processes a clipboard paste operation'''
    pass
    
    
  def onSelectAll(self, event=None):
    '''Processes a select all operation'''
    pass


  def menuFileSave(self, event=None):
    '''Saves the script'''
    raise NotImplementedError, 'A subclass of NotebookComponent failed to implement menuFileSave'    
    
    
  def menuFileSaveAs(self, event=None):
    '''Saves As'''
    raise NotImplementedError, 'A subclass of NotebookComponent failed to implement menuFileSaveAs'    
    

  def is_changed(self):
    '''Returns whether the window has been changed and needs saving'''
    return False
    
    
  def canClose(self):
    '''Checks to see if this window can close.  If the file needs saving, it prompts the user.
       A False return says that we have unsaved changes and the user doesn't want to close.
    '''
    if self.is_changed():
      choice = wx.MessageBox("'" + self.getTitle() + '" ' + lang('has unsaved changes.  Would you like to save before closing?'), lang('Unsaved changes'), style=wx.YES_NO | wx.CANCEL)
      if choice == wx.YES:
        return self.menuFileSave() # let's them cancel out of the save as box
      elif choice == wx.NO:
        return True
      return False        
    return True
    

  def onClosePage(self):
    '''Called when the page is closed (see MainFrame.closePage)
       Subclasses can perform cleanup behavior in this method.
    '''
    pass
    
    
  def onIdle(self):
    '''Called from the mainframe when the system is idle.
       Subclasses can perform work in this method like updating
       their GUIs.  This method is called *a lot*, so keep it simple.
    '''
    pass
    

  def menuFileClose(self, event=None):
    '''Closes the current page'''
    return self.mainframe.closePage(self)
    

  def getPageIndex(self):
    '''Discovers the current index in the notebook'''
    for i in range(self.notebook.GetPageCount()):
      if self.notebook.GetPage(i) == self:
        return i
    return 0  # we only get here if this is the first page


  def getTitle(self):
    '''Gets the title of this page'''
    return self.title

    
  def setTitle(self, title, permanent=True):
    '''Sets the title of the notebook page this component is in.  If permanent is False,
       the title change is temporary.  This allows temporary setting of the notebook page
       title for things like showing unsaved changed'''
    self.title = title
    self.notebook.SetPageText(self.getPageIndex(), title)
    
    
  def select(self):
    '''Selects this page in its parent notebook'''
    self.notebook.SetSelection(self.getPageIndex())
    
    
  def findDialog(self):
    '''Shows the find dialog for a notebook component'''
    wx.MessageBox(lang('This type of window does not support find and replace.'), lang('Find and Replace'), wx.OK | wx.ICON_ERROR)
    
    
  def updatePreferences(self):
    '''Signals that the preferences have been updated and this window should update itself'''
    pass