#!/usr/bin/python

####################################################################################
#                                                                                  #
# Copyright (c) 2005 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This file is part of Picalo.                                                     #
#                                                                                  #
# Picalo is free software; you can redistribute it and/or modify                   #
# it under the terms of the GNU General Public License as published by             #
# the Free Software Foundation; either version 2 of the License, or                # 
# (at your option) any later version.                                              #
#                                                                                  #
# Picalo is distributed in the hope that it will be useful,                        #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                   #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    #
# GNU General Public License for more details.                                     #
#                                                                                  #
# You should have received a copy of the GNU General Public License                #
# along with Foobar; if not, write to the Free Software                            #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA        #
#                                                                                  #
####################################################################################


from picalo import *
from picalo.lib import GUID
import Utils, Spreadsheet, Shell, Editor, CommandLog, Preferences, ObjectTree
from PicaloWizard import load_wizard
from Languages import lang
from picalo.base.Global import make_valid_variable, ensure_unique_list_value
import picalo.base.Global
import Output, Dialogs, Version, TextImport
from TableProperties import TableProperties
from QueryBuilder import QueryBuilder
import sys, os, os.path, threading, traceback, time, re, zipfile, urllib2, types
import wx, wx.html, time, webbrowser


FILE_TYPES = (
  ( lang("Picalo File (*.pco)"),                 ".pco"),  # Picalo's native format must go first
  ( lang("Comma Separated Values File (*.csv)"), ".csv"),
  ( lang("Tab Separated Values File (*.tsv)")  , ".tsv"),
  ( lang("Microsoft Excel File (*.xls)"),        ".xls"),
)
# number of recent files to show on the menu
NUM_RECENT_FILES = 8  

# the version file on the internet
VERSION_URL = 'http://www.picalo.org/Version.py'
DOWNLOAD_URL = "http://www.picalo.org/download.spy"
VERSION_RE = re.compile('''VERSION *= *['"]([\d\.]+?)['"]''', re.MULTILINE)


########################################################
###   The main frame of the entire program

class MainFrame(wx.Frame):
  '''The Main Frame of the program'''
  
  def __init__(self, picaloapp, parentFrame=None):
    '''Constructor'''
    self.picaloapp = picaloapp
    wx.Frame.__init__(self, parentFrame, wx.ID_ANY, lang('Picalo') + ' ' + str(Version.VERSION), size=(900,600), style=wx.DEFAULT_FRAME_STYLE)
    self.menu = None
    self.SetSizer(wx.BoxSizer(wx.VERTICAL))
    self.projectdir = Preferences.get("projectdir", "")
    if os.path.exists(self.projectdir):
      os.chdir(self.projectdir)
    self.focusedwindow = None
    self.running_onIdle = False
    
    # tell the progress dialogs to show in gui mode
    picalo.base.Global.mainframe = self
    picalo.base.Global.guiUpdateProgressDialog = Utils._updateProgressDialog
    
    # application icon
    iconbundle = wx.IconBundle()
    for iconsize in (16, 32, 48, 64, 72, 128):
      iconbundle.AddIcon(Utils.getIcon('appicon%ix%i.png' % (iconsize, iconsize)))
    self.SetIcons(iconbundle) # sets the icon in the Windows taskbar, alt-tab task switcher, GTK taskbar etc.
    if 'wxMac' in wx.PlatformInfo:   # sets icon in mac os x dock, in windows/linux this adds an icon by the clock -- we don't want that
      self.taskbaricon = wx.TaskBarIcon()
      self.taskbaricon.SetIcon(Utils.getIcon('appicon128x128.png'), 'Picalo')
      
    # add the toolbar
    self.toolbarpanel = wx.Panel(self)
    self.toolbarpanel.SetSizer(wx.BoxSizer())
    self.GetSizer().Add(self.toolbarpanel, flag=wx.EXPAND|wx.BOTTOM, border=2)
    self.SetBackgroundColour(self.toolbarpanel.GetBackgroundColour())  # sets the color of the toolbar border (set the window not toolbar for some reason) 
      
    # main splitter
    self.splitter = wx.SplitterWindow(self)
    self.GetSizer().Add(self.splitter, flag=wx.EXPAND, proportion=1)
    self.splitter.SetMinimumPaneSize(1)
    self.browser = wx.Notebook(self.splitter)
    self.rightPanel = wx.SplitterWindow(self.splitter)
    self.rightPanel.SetSashGravity(1)
    self.rightPanel.SetMinimumPaneSize(1)
    self.splitter.SplitVertically(self.browser, self.rightPanel, 200)    

    # the main right panel    
    self.notebook = wx.Notebook(self.rightPanel)
    self.infoarea = wx.Notebook(self.rightPanel)
    self.rightPanel.SplitHorizontally(self.notebook, self.infoarea, 300)
    
    # icons for the notebook tabs
    imglist = wx.ImageList(16, 16)
    imglist.Add(Utils.getBitmap('close.png'))
    self.notebook.AssignImageList(imglist)
    
    # icons for the infoarea tabs
    imglist = wx.ImageList(16, 16)
    for name in [ 'kdf.png', 'randr.png', 'kdat.png' ]:
      imglist.Add(Utils.getBitmap(name))
    self.infoarea.AssignImageList(imglist)

    # the output area
    self.output = Output.Output(self.infoarea, self)
    self.infoarea.AddPage(self.output, lang('Script Output'), imageId=1)
  
    # the command log area (has to be created before the shell)
    self.commandlog = CommandLog.CommandLog(self.infoarea, self)
    self.infoarea.AddPage(self.commandlog, lang('History'), imageId=2)
  
    # the shell area
    self.shell = Shell.Shell(self.infoarea, mainframe=self)
    self.infoarea.InsertPage(0, self.shell, lang("Shell"), imageId=0)
    self.shell.select()
    
    # icons for the browser tabs
    imglist = wx.ImageList(16, 16)
    for name in [ 'flashkard.png', 'mycomputer.png']:
      imglist.Add(Utils.getBitmap(name))
    self.browser.AssignImageList(imglist)
    
    # left side tree
    self.objecttree = ObjectTree.ObjectTree(self.browser, self)
    self.browser.AddPage(self.objecttree, lang('Project'), imageId=0)
    
    # left side disk browser
    filter_types = ['|'.join([lang('All Files (*.*)'), '*.*'])]
    for ft in FILE_TYPES:
      filter_types.append('|'.join(ft[:2]))
    self.diskbrowser = wx.GenericDirCtrl(self.browser, style=wx.DIRCTRL_SHOW_FILTERS,
                       filter='|'.join(filter_types))
    self.browser.AddPage(self.diskbrowser, lang('Disk'), imageId=1)
    diskbrowserpath = Preferences.get('diskbrowserpath', '') or Utils.getHomeDir()
    if os.path.exists(diskbrowserpath):
      self.diskbrowser.ExpandPath(diskbrowserpath)      
    if Preferences.get('diskbrowserfilter', -1) >= 0:
      self.diskbrowser.GetFilterListCtrl().SetSelection(Preferences.get('diskbrowserfilter'))
    self.diskbrowsermenu = Utils.MenuManager(self, wx.Menu(), [
      Utils.MenuItem(lang('Open As Table...'), lang('Open this file as a table'), self.onDiskBrowserOpenAsTable),
      Utils.MenuItem(lang('Open As Script'), lang('Open this file as in the script editor'), self.onDiskBrowserOpenAsScript),
    ])
    self.browser.SetSelection(Preferences.get('browserindex', 0))
    
    # status bar
    self.statusbar = self.CreateStatusBar()
    self.statusbar.SetFieldsCount(2)
    self.statusbar.SetStatusWidths([-1, 100])
    
    # take over error reporting
    sys.excepthook = self.errorhandler

    # toolbar
    self.default_tool_list = [
      Utils.ToolbarItem(lang('New Table'), 'folder_orange.png', longhelp=lang('Create a new table'), callback=self.menuFileNewTable),
      Utils.ToolbarItem(lang('New Script'), 'folder.png', longhelp=lang('Create a new script'), callback=self.menuFileNewScript),
      Utils.ToolbarItem('SEPARATOR'),
      Utils.ToolbarItem(lang('Open Picalo File'), 'folder_open.png', longhelp=lang('Open an existing Picalo table, database connection, query, or script'), callback=self.menuFileOpen),
      Utils.ToolbarItem('SEPARATOR'),
      Utils.ToolbarItem(lang('Close Tab'), 'tab_remove.png', longhelp=lang('Close the current tab'), callback=self.menuWindowClose),
    ]
    self.toolbar = Utils.create_toolbar(self)
    
    # menu
    menu_list = [
      Utils.MenuItem(lang('&File') + '/' + lang('Set Project Folder...') + '\t' + lang('Ctrl+O'), lang('Sets the root project folder'), self.menuFileOpenProject),
      Utils.MenuItem(lang('&File') + '/SEPARATOR'),
      Utils.MenuItem(lang('&File') + '/' + lang('New Table...'), lang('Create a new table'), self.menuFileNewTable),
      Utils.MenuItem(lang('&File') + '/' + lang('New Database Connection...'), lang('Connect to a database by entering the connection parameters'), Dialogs.DatabaseConnection(self)),
      Utils.MenuItem(lang('&File') + '/' + lang('New Query...'), lang('Create a new query using the visual query builder'), QueryBuilder(self)),
      Utils.MenuItem(lang('&File') + '/' + lang('New Script...'), lang('Create a new script'), self.menuFileNewScript),
      Utils.MenuItem(lang('&File') + '/SEPARATOR'),
      Utils.MenuItem(lang('&File') + '/' + lang('Open...'), lang('Open a Picalo table, database connection, query, or script from the disk'), self.menuFileOpen),
      Utils.MenuItem(lang('&File') + '/' + lang('Import') + '/' + lang('Text File or Excel Spreadsheet...'), lang('Import a fixed, delimited, or Excel file into Picalo'), TextImport.TextImportWizard(self)),
      Utils.MenuItem(lang('&File') + '/' + lang('Import') + '/' + lang('Email...'), lang('Import a set of email files into Picalo'), load_wizard(self, 'ImportEmailToPicaloTable')),
      Utils.MenuItem(lang('&File') + '/SEPARATOR'),
      Utils.MenuItem(lang('&File') + '/' + lang('Preferences...') + '\t' + lang('Ctrl+P'), lang('Set application preferences'), Dialogs.PreferencesDlg(self), id=wx.ID_PREFERENCES),
      Utils.MenuItem(lang('&File') + '/SEPARATOR'),
      Utils.MenuItem(lang('&File') + '/' + lang('Exit') + '\t' + lang('Ctrl+Q'), lang('Exit Picalo'), self.menuFileExit, id=wx.ID_EXIT),
      Utils.MenuItem(lang('&Edit') + '/' + lang('Find and Replace...') + '\t' + lang('Ctrl+F'), lang('Find and replace in the current window'), self.menuEditFind),
      Utils.MenuItem(lang('&Edit') + '/' + lang('Find Next...') + '\t' + lang('Ctrl+G'), lang('Find next in the current window'), self.menuEditFindNext),
      Utils.MenuItem(lang('&Edit') + '/SEPARATOR'),
      Utils.MenuItem(lang('&Edit') + '/' + lang('Cut') + '\t' + lang('Ctrl+X'), lang('Cut the selection to the clipboard'), self.menuEditCut, id=wx.ID_CUT),
      Utils.MenuItem(lang('&Edit') + '/' + lang('Copy') + '\t' + lang('Ctrl+C'), lang('Copy the selection to the clipboard'), self.menuEditCopy, id=wx.ID_COPY),
      Utils.MenuItem(lang('&Edit') + '/' + lang('Paste') + '\t' + lang('Ctrl+V'), lang('Paste the clipboard to the current selection'), self.menuEditPaste, id=wx.ID_PASTE),
      Utils.MenuItem(lang('&Edit') + '/' + lang('Select All'), lang('Select all the text in the current window'), self.menuEditSelectAll, id=wx.ID_SELECTALL),
      Utils.MenuItem(lang('&Edit') + '/SEPARATOR'),
      Utils.MenuItem(lang('&Edit') + '/' + lang('Clear Script Output'), lang('Clear the script output window'), self.menuEditClearOutput),
      Utils.MenuItem(lang('&Edit') + '/' + lang('Clear History...'), lang('Clear the history window'), self.menuEditClearHistory),
      Utils.MenuItem(lang('&Data') + '/' + lang('Select') + '/' + lang('By Record Index...'), lang('Select records from a table within a certain record index range'), Dialogs.SelectByIndex(self)),
      Utils.MenuItem(lang('&Data') + '/' + lang('Select') + '/' + lang('By Exact Match...'), lang('Select records from a table based upon the values in one or more columns'), Dialogs.SelectByKey(self)),
      Utils.MenuItem(lang('&Data') + '/' + lang('Select') + '/' + lang('By Wildcard Pattern...'), lang('Select records from a table based upon a wildcard'), Dialogs.SelectByWildcard(self, 'Wildcard')),
      Utils.MenuItem(lang('&Data') + '/' + lang('Select') + '/' + lang('By Regular Expression Pattern...'), lang('Select records from a table based upon a regular expression'), Dialogs.SelectByWildcard(self, 'RegEx')),
      Utils.MenuItem(lang('&Data') + '/' + lang('Select') + '/' + lang('By Picalo Expression...'), lang('Select records from a table based upon a custom expression'), Dialogs.SelectByExpression(self)),
      Utils.MenuItem(lang('&Data') + '/' + lang('Join') + '/' + lang('By Value...'), lang('Join two tables based upon matching values in one or more columns'), Dialogs.JoinByValue(self)),
      Utils.MenuItem(lang('&Data') + '/' + lang('Join') + '/' + lang('By Soundex...'), lang('Join two tables based upon a soundex calculation'), Dialogs.JoinBySoundex(self)),
      Utils.MenuItem(lang('&Data') + '/' + lang('Join') + '/' + lang('By Fuzzy Match...'), lang('Join two tables based upon a fuzzy match algorithm'), Dialogs.JoinByFuzzyMatch(self)),
      Utils.MenuItem(lang('&Data') + '/' + lang('Join') + '/' + lang('By Expression...'), lang('Join two tables based upon a custom expression'), Dialogs.JoinByExpression(self)),
      Utils.MenuItem(lang('&Data') + '/SEPARATOR'),
      Utils.MenuItem(lang('&Data') + '/' + lang('Stratify') + '/' + lang('By Value...'), lang('Stratify a table into a number of separate tables by column value'), Dialogs.StratifyByValue(self)),
      Utils.MenuItem(lang('&Data') + '/' + lang('Stratify') + '/' + lang('By Date Column...'), lang('Stratify a table into a number of separate tables by date'), Dialogs.StratifyByDate(self)),
      Utils.MenuItem(lang('&Data') + '/' + lang('Stratify') + '/' + lang('Into A Specified Number of Groups...'), lang('Stratify a table into a specified number of separate tables'), Dialogs.StratifyByNumberOfGroups(self)),
      Utils.MenuItem(lang('&Data') + '/' + lang('Stratify') + '/SEPARATOR'),
      Utils.MenuItem(lang('&Data') + '/' + lang('Stratify') + '/' + lang('Combine Table List...'), lang('Combine a table list back into a single table; antistratification'), Dialogs.CombineTableList(self)),
      Utils.MenuItem(lang('&Data') + '/' + lang('Summarize') + '/' + lang('By Value...'), lang('Summarize a table by column value using a set of expressions'), Dialogs.SummarizeByValue(self)),
      Utils.MenuItem(lang('&Data') + '/' + lang('Summarize') + '/' + lang('By Date Column...'), lang('Summarize a table by changes in a date column using a set of expressions'), Dialogs.SummarizeByDate(self)),
      Utils.MenuItem(lang('&Data') + '/' + lang('Summarize') + '/' + lang('Existing Table List...'), lang('Summarize a list of tables using a set of expressions'), Dialogs.Summarize(self)),
      Utils.MenuItem(lang('&Data') + '/' + lang('Pivot Table...'), lang('Calculate summaries on a table using the pivot feature'), Dialogs.PivotTable(self)),
      Utils.MenuItem(lang('&Data') + '/SEPARATOR'),
      Utils.MenuItem(lang('&Data') + '/' + lang('Upload Table To Database...'), lang('Upload a Picalo table to a database connection'), Dialogs.DatabasePostTable(self)),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Descriptives...'), lang('Calculate descriptives for one or more columns in a table'), self.menuAnalyzeDescriptives),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Digital Analysis') + '/' + lang('Calculate Benford\'s Expected...'), lang("Calculate the Benford's expected probabilities for a column"), Dialogs.CalcBenfords(self)),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Digital Analysis') + '/' + lang('Append Benford\'s Expected Col...'), lang("Add a column for Benford's expectation to a table"), Dialogs.AddBenfordsDiffCol(self)),
      Utils.MenuItem(lang('&Analyze') + '/SEPARATOR'),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Find') + '/' + lang('Unordered...'), lang('Find all records in a table that are out of order on a column'), Dialogs.FindUnordered(self)),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Find') + '/' + lang('Duplicates...'), lang('Find all records in a table that have the same value for a column'), Dialogs.FindDuplicates(self)),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Find') + '/' + lang('Gaps...'), lang('Find all records in a table that have gaps in the sequential values of a column'), Dialogs.FindGaps(self)),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Find') + '/SEPARATOR'),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Find') + '/' + lang('Matching by Value'), lang('Selects records from two tables that have matching values in column(s)'), Dialogs.MatchByKey(self, 'Find Matching By Value', 'equals', 'Simple.col_match_same', 'Results Table List:')),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Find') + '/' + lang('Nonmatching by Value'), lang('Selects records from two tables that do not have matching values in column(s)'), Dialogs.MatchByKey(self, 'Find Nonmatching By Value', 'does not equal', 'Simple.col_match_diff', 'Results Table List:')),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Find') + '/' + lang('Matching by Expression'), lang('Selects records from two tables that have matching values in column(s) based on a custom expression'), Dialogs.MatchByExpression(self, 'Find Matching By Expression', 'Simple.custom_match_same', 'Results Table List:')),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Find') + '/' + lang('Nonmatching by Expression'), lang('Selects records from two tables that do not have matching values in column(s) based on a custom expression'), Dialogs.MatchByExpression(self, 'Find Nonmatching By Expression', 'Simple.custom_match_diff', 'Results Table List:')),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Outliers') + '/' + lang('Add ZScore Column...'), lang('Calculate the zscore and add these values as a new column to a table'), Dialogs.AddZScoreCol(self)),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Outliers') + '/' + lang('Select Outliers...'), lang('Select the outliers in a table based on a maximum and a minimum'), Dialogs.SelectOutliers(self, True)),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Outliers') + '/' + lang('Select Outliers by ZScore...'), lang('Select the outliers in a table based on the zscore value in a column'), Dialogs.SelectOutliersByZ(self, True)),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Outliers') + '/' + lang('Exclude Outliers...'), lang('Exclude the outliers in a table based on a maximum and a minimum'), Dialogs.SelectOutliers(self, True)),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Outliers') + '/' + lang('Exclude Outliers by ZScore...'), lang('Exclude the outliers in a table based on the zscore value in a column'), Dialogs.SelectOutliersByZ(self, False)),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Trend') + '/' + lang('Add Cusum Column'), lang('Adds a cusum calculation column to a table'), Dialogs.AddCusumCol(self)),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Trend') + '/' + lang('By Regression Slope'), lang('Analyzes a trend using regression slope'), Dialogs.TrendSingleCol(self, 'Calculate Regression', 'Trending.regression')),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Trend') + '/' + lang('By Average Slope'), lang('Analyzes a trend using average slope between all points'), Dialogs.TrendSingleCol(self, 'Calculate Average Slope', 'Trending.average_slope')),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Trend') + '/' + lang('By High-Low Slope'), lang('Analyzes a trend using the hi') + '/' + lang('low calculation'), Dialogs.TrendSingleCol(self, 'Calculate High Low Slope', 'Trending.highlow_slope')),
      Utils.MenuItem(lang('&Analyze') + '/' + lang('Trend') + '/' + lang('By Handshaking Slope'), lang('Analyzes a trend by handshaking each point to its neighbor'), Dialogs.TrendSingleCol(self, 'Calculate Handshake Slope', 'Trending.handshake_slope')),
      Utils.MenuItem(lang('&Tools') + '/' + lang('Expression Builder...') + '\t' + lang('Ctrl+U'), lang('Open the expression builder to visually create a function call'), self.menuToolsExpressionBuilder),
      Utils.MenuItem(lang('&Help') + '/' + lang('Check for Updates...'), lang('Check the Picalo website to see if any updates to your version are available.'), self.menuHelpCheckForUpdates),
      Utils.MenuItem(lang('&Help') + '/' + lang('Submit a New Bug...'), lang('Submit a Picalo bug as a new ticket in the tracking system.'), self.menuHelpSubmitBug),
      Utils.MenuItem(lang('&Help') + '/SEPARATOR'),
      Utils.MenuItem(lang('&Help') + '/' + lang('Online Python Tutorial'), lang('Open the Python tutorial (online at python.org)'), self.menuHelpPythonTutorial),
      Utils.MenuItem(lang('&Help') + '/' + lang('Introductory Picalo Manual'), lang('Open the Picalo manual'), self.menuHelpIntroManual),
      Utils.MenuItem(lang('&Help') + '/' + lang('Picalo Cookbook'), lang('Open the Picalo Cookbook'), self.menuHelpCookbook),
      Utils.MenuItem(lang('&Help') + '/' + lang('Advanced Picalo Manual'), lang('Open the Picalo manual'), self.menuHelpAdvancedManual),
      Utils.MenuItem(lang('&Help') + '/' + lang('ReadMe.txt'), lang('Open the ReadMe.txt file'), self.menuHelpReadme),
      Utils.MenuItem(lang('&Help') + '/' + lang('License.txt'), lang('Open the License.txt file'), self.menuHelpLicense),
      Utils.MenuItem(lang('&Help') + '/SEPARATOR'),
      Utils.MenuItem(lang('&Help') + '/' + lang('Show Hint...'), lang('Show the hint dialog (i.e. Did You Know?)'), Dialogs.Hints(self)),
      Utils.MenuItem(lang('&Help') + '/' + lang('About...'), lang('About Picalo'), self.menuHelpAbout, id=wx.ID_ABOUT),
    ]
    self.menubar = wx.MenuBar()
    self.menu = Utils.MenuManager(self, self.menubar, menu_list)
    self.SetMenuBar(self.menubar)
    self.lastMenuWindow = None

    # printer object to use for the app
    self.printer = wx.html.HtmlEasyPrinting()

    # events    
    self.Bind(wx.EVT_CLOSE, self.onClose)
    self.Bind(wx.EVT_ACTIVATE, self.onAppActivate)   # window focused
    self.Bind(wx.EVT_IDLE, self.onIdle)
    self.diskbrowser.GetTreeCtrl().Bind(wx.EVT_TREE_ITEM_ACTIVATED, self.onDiskBrowserDoubleClick)
    self.diskbrowser.GetTreeCtrl().Bind(wx.EVT_RIGHT_DOWN, self.onDiskBrowserMouseRight)
    self.notebook.Bind(wx.EVT_RIGHT_DOWN, self.onNotebookMouseRight)
    self.notebook.Bind(wx.EVT_LEFT_DOWN, self.onNotebookMouseLeft)
    self.notebook.Bind(wx.EVT_NOTEBOOK_PAGE_CHANGED, self.onNotebookPageChanged)

    # initialize the plugins and the menu
    self.plugins = Utils.getPlugins()
    for plugin in self.plugins:
      plugin.initialize_plugin(self)
    self.updateMenu(refresh_plugins=True)

    # set the focus on the shell
    self.shell.SetFocus()
    self.Layout()
    
    # start the object cache thread    
    self.objecttree.object_cache_thread.start()
    

  def onAppActivate(self, event=None):
    '''Respond to a focus event'''
    if event.GetActive():
      if self.focusedwindow:
        self.focusedwindow.onFocus(event)
    event.Skip()


  def windowFocus(self, page):
    '''Indicates that a window has been focused on'''
    self.focusedwindow = page
    
    
  def updateMenu(self, page=None, refresh_plugins=False, force_refresh=False):
    '''Updates the menu and toolbar based upon the current page in the notebook'''
    try:
      self.Freeze()
      
      # when the program starts, menu is None
      if self.menu == None:
        return
      
      # get the current window
      if page == None:
        window = self.getCurrentPage()
      else:
        window = page
      
      # this gets called twice sometimes because of multiple events triggering this.
      # if the same window as last time is still active, short circuit
      if window != None and window == self.lastMenuWindow and refresh_plugins == False and force_refresh == False:
        return
      self.lastMenuWindow = window
    
        # clear out the status bar since something serious in the gui changed (like a notebook page change)
      for i in range(self.statusbar.GetFieldsCount()):
        self.setStatus('', i)
    
      # do we need to refresh the plugins list?
      if refresh_plugins:
        while self.menu.getStackSize() > 1 :   # pop back to the bare menu
          self.menu.pop()
        pluginmenu = []
        pluginmenu.append(Utils.MenuItem(lang('&Tools') + '/SEPARATOR'))
        for mod in self.plugins:
          pluginmenu.extend(mod.get_menu_items())
        self.menu.push(pluginmenu)
      
      # redo the recent files lists (tables, databases, and scripts)
      while self.menu.getStackSize() > 2:   # standard menu is regular (above) + wizard list 
        self.menu.pop()
    
      # add the menu for the current window
      if window:
        self.menu.push(window.menuitems)
      
      # add the toolbar for the current window
      for tb in self.toolbarpanel.GetChildren():
        tb.Hide()
      self.toolbarpanel.GetSizer().Clear()
      if window and window.toolbar != None:
        self.toolbarpanel.GetSizer().Add(window.toolbar, proportion=1, flag=wx.EXPAND)
        window.toolbar.Show()
      else:
        self.toolbarpanel.GetSizer().Add(self.toolbar, proportion=1, flag=wx.EXPAND)
        self.toolbar.Show()
        
    finally:
      self.Thaw()    
      
      
  def showError(self, st='', e=None):
    '''Shows an error dialog to the user, with optional message and optional Exception'''
    if st != '' and e != None:
      wx.MessageBox(str(st) + '\n\n' + str(e), lang('Picalo'), wx.OK | wx.ICON_ERROR)
    elif st != '':
      wx.MessageBox(str(st), lang('Picalo'), wx.OK | wx.ICON_ERROR)
    elif e != None:
      wx.MessageBox(str(e), lang('Picalo'), wx.OK | wx.ICON_ERROR)
      
      
  def errorhandler(self, type, value, tb):
    '''Handles all errors for the app'''
    try:
      if type == AssertionError:  # show these with a milder dialog
        wx.MessageBox(str(value), lang('Picalo'), wx.OK | wx.ICON_WARNING)
        
      elif type == Utils.OperationCancelledError: # uesr operation cancelled it
        pass  # ignore this event since the user obviously wanted it cancelled
        
      else:
        traceback.print_exception(type, value, tb, file=self.output) 
        traceback.print_exception(type, value, tb, file=sys.__stderr__)
        title = lang('Error')
        text = str(type) + '\n' + \
               str(value) + '\n\n' + \
               lang('Detailed information has been printed to the Script Output tab.')
        dialog = wx.MessageDialog(self, text, title, wx.OK | wx.ICON_ERROR)
        dialog.ShowModal()
        
    except Exception, e:
      if traceback:  # traceback goes None sometimes when the app finishes
        traceback.print_exc(file=sys.__stderr__) # for this exception
    

  def getVariable(self, varname, default=None):
    '''Retrieves the value for a variable name from the shell interpreter'''
    return self.shell.getVariable(varname, default)
    

  def execute(self, cmds, prompt=False, verbose=True):
    '''Executes a set of Picalo commands.  Do not call Shell.run directly as it 
       circumvents the error behavior in this method.  This method is NOT
       thread safe -- as wx only has one thread, I don't think it needs to
       be thread safe right now.
       
       The method returns True if all commands processed successfully, or
       False if it had exceptions.  If you want to know about the error,
       inspect shell.interp.exc_info. '''
    self.shell.setRedirectErrors(True)
    try:
      # ensure we have a list
      if not isinstance(cmds, (types.ListType, types.TupleType)):
        cmds = [ cmds ]
        
      # run the commands one by one
      for cmd in cmds:
        self.shell.interp.exc_info = None
        self.shell.run(cmd, prompt, verbose)
        # this error handling is wierd, but it has to be because the shell.run catches any exceptions
        # the only way we can catch it is in the interpreter's showtraceback method
        if self.shell.interp.exc_info:  # was there an error in running the command?
          # if a command fails, don't run any more commands
          type, value, tb = self.shell.interp.exc_info
          self.shell.interp.exc_info = None
          while tb.tb_next:  # step back to <input> (what the user knows about) to exclude the detailed Picalo functions
            if tb.tb_frame.f_code.co_filename == '<input>':
              tb = tb.tb_next
              break
            tb = tb.tb_next
          self.errorhandler(type, value, tb)
          return False # don't do any more commands
      return True
    finally:
      self.shell.setRedirectErrors(False)
      
  
  def setStatus(self, msg, col=0):
    '''Sets the status bar text for a given column.
       Column 0 is the left-most column, 
       Column -1 is the right-most column.
    '''
    if col < 0:
      col = self.statusbar.GetFieldsCount() + col
    self.statusbar.SetStatusText(msg, col)
    
    
  def onClose(self, event=None):
    """Event handler for closing."""
    # save program options
    Preferences.put('mainframesize', list(self.GetSize()))
    Preferences.put('mainframeposition', list(self.GetPosition()))
    Preferences.put('mainframerightpanel', self.rightPanel.GetSashPosition())
    Preferences.put('mainframesplitter', self.splitter.GetSashPosition())
    Preferences.put('browserindex', self.browser.GetSelection())
    Preferences.put('diskbrowserpath', self.diskbrowser.GetPath())
    Preferences.put('diskbrowserfilter', self.diskbrowser.GetFilterListCtrl().GetSelection())
    Preferences.save()

    # check for variables (tables, databases, etc.) that need saving
    if not self.checkProjectSave():
      if event:
        event.Veto()
      return
      
    # close up the object cache thread
    self.objecttree.object_cache_thread_running = False

    # kill the program  
    sys.exit(0)
    
    
  def closePage(self, component):    
    '''Closes the current notebook page'''
    # usually the compononent is the current page, so check that first
    pagenum = -1
    if self.notebook.GetPage(self.notebook.GetSelection()) == component:
      pagenum = self.notebook.GetSelection()
    else:
      for i in range(self.notebook.GetPageCount()):
        if self.notebook.GetPage(i) == component:
          pagenum = i
    # if found, close it up
    if pagenum >= 0:
      if component.canClose():
        component.onClosePage()
        self.notebook.DeletePage(pagenum)
        self.updateMenu()


  def addPage(self, component, title):
    '''Adds a component to the notebook at the current selection position, then selects it'''
    if self.notebook.GetPageCount() == 0 or self.notebook.GetSelection() < 0:
      newpos = self.notebook.GetPageCount()
      self.notebook.AddPage(component, title, imageId=0)
      self.notebook.SetSelection(self.notebook.GetPageCount()-1)
    else:
      newpos = self.notebook.GetSelection()+1
      self.notebook.InsertPage(self.notebook.GetSelection()+1, component, title, imageId=0)
      self.notebook.SetSelection(self.notebook.GetSelection()+1)


  def getPage(self, pagename, component_type=None):
    '''Returns the page with the given name in the notebook, or None if not there.
       If component_type is not None, the type of component in the page must match
       the given type.'''
    for i in range(self.notebook.GetPageCount()):
      if self.notebook.GetPageText(i) == pagename:
        if component_type == None:
          return self.notebook.GetPage(i)
        else:
          if isinstance(self.notebook.GetPage(i), component_type):
            return self.notebook.GetPage(i)
    return None
    
    
  def getPageByFilename(self, filename, component_type=None):
    '''Returns the page with the given filename in the notebook, or None if not there.
       If component_type is not None, the type of component in the page must match
       the given type.
    '''
    for i in range(self.notebook.GetPageCount()):
      if self.notebook.GetPage(i).get_filename() == filename:
        if component_type == None:
          return self.notebook.GetPage(i)
        else:
          if isinstance(self.notebook.GetPage(i), component_type):
            return self.notebook.GetPage(i)
    return None
        

  def getCurrentPage(self):
    '''Returns the currently-selected page in the notebook'''
    if self.notebook.GetSelection() >= 0:
      return self.notebook.GetPage(self.notebook.GetSelection())
    return None


  def getCurrentPageName(self):
    '''Returns the currently-selecte page name in the notebook'''
    if self.notebook.GetSelection() >= 0:
      return self.notebook.GetPageText(self.notebook.GetSelection())
    return None
    
  
  def openTable(self, tablename):
    '''Opens a table into the notebook'''
    # get the actual table object
    table = self.shell.getVariable(tablename)
    # first see if it is already open
    for i in range(self.notebook.GetPageCount()):
      if self.notebook.GetPageText(i) == tablename:
        # first tell it to reload the table (just in case changes have been made)
        self.notebook.GetPage(i).refresh()
        self.notebook.SetSelection(i)
        return

    # if a query, ensure it has been loaded
    if isinstance(table, Database.Query):
      connname = self.determineQueryConnection(table)
      if connname:
        self.execute('%s.execute(%s)' % (tablename, connname))
      else:
        return
      
    # open it up!
    spreadsheet = Spreadsheet.Spreadsheet(self.notebook, self, tablename, table)
    self.addPage(spreadsheet, tablename)
    
    
  def determineQueryConnection(self, query):
    '''Determines the appropriate database connection to run a query with.
       Returns the name of the connection (not the actual object, but
       the variable name).
    '''
    dbs = self.getDatabases()
    assert len(dbs) > 0, 'You must first open a database connection before running queries.'
    if len(dbs) == 1:
      return dbs[0]
    else:
      for varname in dbs:
        var = self.getVariable(varname)
        if var._connect_func == query._connect_func and var._connect_args == query._connect_args:
          return varname
      return wx.GetSingleChoice(lang('Please select a database connection to run the query with:'), lang('Open Query'), self.getDatabases())

    
  def getTables(self, tables=True, tablearrays=False, queries=True, relations=True):
    '''Returns the table names currently visible in the tree'''
    gettypes = []
    if tables: gettypes.append(Table)
    if tablearrays: gettypes.append(TableArray)
    if queries: gettypes.append(Database.Query)
    if relations: gettypes.append(Database._Connection)
    gettypes = tuple(gettypes)
    items = []
    for name, value in self.shell.getLocals().items():
      if isinstance(value, gettypes):
        if isinstance(value, Database._Connection): # special treatment of database relations
          for relname in value.list_tables(refresh=False):
            items.append('%s.%s' % (name, relname))
        else:
          items.append(name)
    return items
    
    
  def getDatabases(self):
    '''Returns the databases currently visible in the tree'''
    items = []
    for name, value in self.shell.getLocals().items():
      if isinstance(value, Database._Connection):
        items.append(name)
    return items
    
    

  ##############################
  ###   GUI EVENT HANDLERS   
    
  def onIdle(self, event=None):
    '''Called when the system is idle.  We use it here to refresh the GUI and the project tree.'''
    # on linux, any call to dlg.ShowModal will free up the idle thread to run again, causing an infinite loop.  this code circumvents it.  it doesn't happen in Mac or Windows
    if self.running_onIdle:
      return
    self.running_onIdle = True
    
    try:
      # let the current page update (no need to update any other page since they aren't visible)
      curpage = self.getCurrentPage()
      if curpage:
        curpage.onIdle()
        
    finally:
      self.running_onIdle = False

        
  ################################################
  ###  Table options
        
  def onDiskBrowserDoubleClick(self, event):
    '''Responds to a double click on a disk browser item'''
    filename = self.diskbrowser.GetFilePath()  
    if filename:  
      path, name = os.path.split(filename)
      name2, ext = os.path.splitext(name)
      ext = ext.lower()
      if os.path.isfile(filename) and (ext == '.csv' or ext == '.tsv' or ext == '.pco'):
        self.load_table_dialog.initial_file = filename
        self.load_table_dialog()
        
      elif os.path.isfile(filename) and ext == '.py':
        self.openScript(filename)   
        
      else:  
        if os.name == 'win32' or os.name == 'nt':  # there's no cross platform way to do this that I know of 
          try:
            os.startfile(filename)
          except:
            wx.MessageBox(lang('No application is associated with this type of file.  To open as a table or script, try right-clicking this type of file.'), lang('Picalo'), wx.OK | wx.ICON_ERROR)
    event.Skip()
          

  def onDiskBrowserMouseRight(self, event):
    '''Shows the popup menu for the disk browser'''
    self.PopupMenu(self.diskbrowsermenu.topmenu)
    event.Skip()
    

  def onDiskBrowserOpenAsTable(self, event):
    '''Open the currently-selected item as a table'''
    filename = self.diskbrowser.GetFilePath()  
    if filename:
      self.load_table_dialog.initial_file = filename
      self.load_table_dialog()
    else:
      wx.MessageBox(lang('This item cannot be opened as a table.'), lang('Picalo'), wx.OK | wx.ICON_ERROR)
    event.Skip()

  
  def onDiskBrowserOpenAsScript(self, event):
    '''Open the currently-selected item as a script'''
    filename = self.diskbrowser.GetFilePath()  
    if filename:
      self.openScript(filename)   
    else:
      wx.MessageBox(lang('This item cannot be opened as in the script editor.'), lang('Picalo'), wx.OK | wx.ICON_ERROR)
    event.Skip()
    
    
  def onNotebookMouseRight(self, event):
    '''Called when the user right-clicks on the notebook'''
    index, flags = self.notebook.HitTest(event.GetPosition())
    if index >= 0:
      page = self.notebook.GetPage(index)
      self.notebook.SetSelection(index)
      if page.tabmenu:
        self.PopupMenu(page.tabmenu.topmenu)
    event.Skip()
    
  
  def onNotebookMouseLeft(self, event):
    '''Called when the user left-clicks on a notebook'''
    index, flags = self.notebook.HitTest(event.GetPosition())
    if flags ==  wx.BK_HITTEST_ONICON and index >= 0:  # close icon
      page = self.notebook.GetPage(index)
      page.menuFileClose(self)
    event.Skip()
    

  def onNotebookPageChanged(self, event):
    '''Called when a notebook page has changed -- time to update the menu and toolbar'''
    self.updateMenu(page=self.notebook.GetPage(event.GetSelection()))
    event.Skip()


    
  ########################################
  ###   Menu file handlers 

  def menuFileExit(self, event=None):
    '''Handles the onclose event for this form'''
    self.Close()  # system calls onClose above
    

  def checkProjectSave(self):
    '''Checks all variables to see if any need changing'''
    # check for variables that need saving
    savetext = []
    savenames = []
    savevals = []
    for title, typ in (
      [ 'Table',       Table,                ],
      [ 'TableList',   TableList,            ],
      [ 'TableArray',  TableArray,           ],
      [ 'Database',    Database._Connection, ],
      [ 'Query',       Database.Query,       ],
    ):
      for name, val in self.shell.getLocals().items():
        if isinstance(val, typ):
          if val.is_changed():
            savetext.append('%s: %s' % (title, name))
            savenames.append(name)
            savevals.append(val)
    # scripts need to be checked separately because they are never variables
    for i in range(self.notebook.GetPageCount()):
      page = self.notebook.GetPage(i)
      if isinstance(page, Editor.Editor) and page.is_changed():
        savetext.append('Script: %s' % self.notebook.GetPageText(i))
        savenames.append(self.notebook.GetPageText(i))
        savevals.append(page)
        
    # save the variables that need it
    if len(savetext) > 0:
      dlg = wx.MultiChoiceDialog(self, lang("The following items have changed.  Please check the ones to save:"), lang("Save Items"), savetext)
      if dlg.ShowModal() != wx.ID_OK:
        return False
      for i in dlg.GetSelections():
        val = savevals[i]
        if isinstance(val, (Table, TableList, TableArray, Database.Query)):
          s = Spreadsheet.Spreadsheet(self, self, savenames[i], val)
          if not s.menuFileSave():
            del s
            return False
          del s
        elif isinstance(val, Editor.Editor):
          if not val.menuFileSave():
            return False
        elif isinstance(val, Database._Connection):
          if not self.objecttree.menuSave(value=val):
            return False
            
    # save the history
    try:
      history = self.commandlog.GetText()
      if history:
        if os.path.exists(self.projectdir):
          historyname = os.path.join(self.projectdir, 'Picalo.history')
          f = open(historyname, 'w')
          f.write(self.commandlog.GetText())
          f.close()
    except Exception, e:
      self.showError(lang('An error occurred while saving the command history:'), e)
    
    return True
        

  def menuFileOpenProject(self, event=None):
    '''Starts a new project'''
    # check if we need to save the project
    if not self.checkProjectSave():
      return
    
    # get the new directory name
    dlg = wx.DirDialog(self, lang("Choose an existing project folder, or click 'New Folder' for a new project:"), style=wx.DD_DEFAULT_STYLE | wx.DD_NEW_DIR_BUTTON | wx.DD_CHANGE_DIR)
    if dlg.ShowModal() != wx.ID_OK:
      return
    self.projectdir = dlg.GetPath()
    Preferences.set('projectdir', self.projectdir)
    if os.path.exists(self.projectdir):
      os.chdir(self.projectdir)
    
    # clear out memory of tables, databases, queries, and scripts
    for key, val in self.shell.interp.locals.items():
      if isinstance(val, (Table, TableList, TableArray, Database._Connection)):
        del self.shell.interp.locals[key]
        
    # update the display
    self.commandlog.clear()
    
    # update the tree
    self.objecttree.loadProject()
      
  
  def menuFileNewTable(self, event=None):
    '''Shows the new table dialog box'''
    dlg = TableProperties(self, None)
    dlg.runModal()
    

  def menuFileNewScript(self, event=None):
    '''Creates a new script window'''
    editor = Editor.Editor(self.notebook, self)
    self.addPage(editor, editor.getTitle())
    
    
  def menuFileOpen(self, event=None):
    '''Opens an existing file from disk'''
    # get the new file
    dlg = wx.FileDialog(self, message=lang("Open"), defaultDir=self.projectdir, defaultFile="", style=wx.OPEN | wx.CHANGE_DIR | wx.FILE_MUST_EXIST, wildcard=lang('All Picalo Files') + ' (*.pco, *.pcd, *.pcq, *.py)|*.pco;*.pcd;*.pcq;*.py|Tables (*.pco)|*.pco|Database Connections (*.pcd)|*.pcd|Queries (*.pcq)|*.pcq|Scripts (*.py)|*.py')
    if dlg.ShowModal() != wx.ID_OK:
      return
    self.openFilename(dlg.GetPath().replace('\\', '/'))


  def openFilename(self, filename):
    '''Opens the given filename'''
    # make a unique variable name
    varname = os.path.splitext(os.path.split(filename)[1])[0]
    varname = make_valid_variable(varname)
    varname = ensure_unique_list_value(self.shell.getLocals().keys(), varname)
    
    # execute the appropriate commands to load this type of file
    ext = os.path.splitext(filename)[1].lower()
    if ext == '.pco':
      self.execute([ 
        '%s = load("%s")' % (varname, filename),
      ])
    elif ext == '.pcd':
      self.execute('%s = Database.load("%s")' % (varname, filename))

    elif ext == '.pcq':
      self.execute([
        '%s = Database.load_query("%s")' % (varname, filename),
        '%s.view()' % (varname, ),
      ])
    elif ext == '.py' or ext == '.txt':
      for i in range(self.notebook.GetPageCount()):
        page = self.getPageByFilename(filename)
        if page:
          self.notebook.SetSelection(page)
          return
      editor = Editor.Editor(self.notebook, self, filename)
      path, fname = os.path.split(filename)
      self.addPage(editor, fname)
    

  ###################################
  ###   Edit menu helpers
  
  def menuEditCut(self, event=None):
    '''Calls the cut method of the active window'''
    if self.focusedwindow:
      self.focusedwindow.onCut(event)
    
    
  def menuEditCopy(self, event=None):
    '''Calls the copy method of the active window'''
    if self.focusedwindow:
      self.focusedwindow.onCopy(event)
      
    
  def menuEditPaste(self, event=None):
    '''Calls the paste method of the active window'''
    if self.focusedwindow:
      self.focusedwindow.onPaste(event)
    
    
  def menuEditSelectAll(self, event=None):
    '''Calls the select all method of the active window'''
    if self.focusedwindow:
      self.focusedwindow.onSelectAll(event)
    
  
  def menuEditFind(self, event=None):
    '''Finds in the current window'''
    index = self.notebook.GetSelection()
    if index >= 0:
      component = self.notebook.GetPage(index)
      component.findDialog()
    else:
      wx.MessageBox(lang('Please select a table or script to find in.'), lang('Picalo'), wx.OK | wx.ICON_ERROR)

  
  def menuEditFindNext(self, event=None):
    '''Finds in the current window'''
    index = self.notebook.GetSelection()
    if index >= 0:
      component = self.notebook.GetPage(index)
      component.findNextDialog()
    else:
      wx.MessageBox(lang('Please select a table or script to find in.'), lang('Picalo'), wx.OK | wx.ICON_ERROR)


  def menuEditClearOutput(self, event=None):
    '''Clears the output window'''
    self.output.clear()
  

  def menuEditClearHistory(self, event=None):
    '''Clears the history window, including the file on disk'''
    if wx.MessageBox(lang('This will permanently clear the history for this project, including deleting the history file on disk.  Continue?'), lang("Picalo"), style=wx.YES_NO) == wx.YES:
      os.remove(os.path.join(self.projectdir, 'Picalo.history'))
      self.commandlog.clear()

  
  ######################################
  ###   Help menu handlers

  # Thanks to Paul McNett at http://dabodev.com/ for posting his solution to opening PDFs on Linux
  def previewPDF(self, path):
    """Preview the passed PDF file in the default PDF viewer."""
    try:
      os.startfile(path)
    except AttributeError:
      # startfile only available on Windows
      if sys.platform == "darwin":
        os.system("open %s" % path)
      else:
        # On Linux, try to find an installed viewer and just use the first one
        # found. I just don't know how to reliably get the default viewer from
        # the many distros.
        viewers = ("gpdf", "kpdf", "evince", "acroread", "xpdf", "firefox", "mozilla-firefox")
        viewer = None
        for v in viewers:
          r = os.system("which %s > /dev/null" % v)
          if r == 0:
            viewer = v
            break
        sysfunc = os.popen2
        sysfunc("%s %s" % (viewer, path))


  def menuHelpAbout(self, event=None):
    """Display an About window."""
    splashimg = Utils.getBitmap('splash.png')
    splash = wx.SplashScreen(splashimg, wx.SPLASH_CENTRE_ON_SCREEN | wx.SPLASH_NO_TIMEOUT, 0, self, style=wx.SIMPLE_BORDER|wx.FRAME_NO_TASKBAR)
    splash.Show()
    
    
  def menuHelpSubmitBug(self, event=None):
    '''Display the bug tracker'''
    webbrowser.open("http://www.picalo.org/cgi-bin/trac.cgi/newticket")
    
    
  def menuHelpReadme(self, event=None):
    '''Display the readme.txt file'''
    fname = 'README.TXT'
    editor = Editor.Editor(self.notebook, self, Utils.getResourcePath(fname))
    editor.SetReadOnly(True)
    self.addPage(editor, fname)
    
    
  def menuHelpLicense(self, event=None):
    '''Display the license.txt file'''
    fname = 'LICENSE.TXT'
    editor = Editor.Editor(self.notebook, self, Utils.getResourcePath(fname))
    editor.SetReadOnly(True)
    self.addPage(editor, fname)


  def menuHelpPythonTutorial(self, event=None):
    '''Opens the online python tutorial'''
    webbrowser.open('http://docs.python.org/tut/tut.html')


  def menuHelpIntroManual(self, event=None):
    '''Opens the PDF introductory manual'''
    self.previewPDF(os.path.normpath(Utils.getResourcePath('IntroductoryManual.pdf')))
    
    
  def menuHelpCookbook(self, event=None):
    '''Opens the PDF cookbook'''
    self.previewPDF(os.path.normpath(Utils.getResourcePath('PicaloCookbook.pdf')))

    
  def menuHelpAdvancedManual(self, event=None):
    '''Opens the PDF advanced manual'''
    self.previewPDF(os.path.normpath(Utils.getResourcePath('AdvancedManual.pdf')))

    
  def menuHelpCheckForUpdates(self, event=None, auto=False):
    '''Checks the web site for updates'''
    try:
      # load the file
      f = urllib2.urlopen(VERSION_URL + '?Version=' + str(Version.VERSION), timeout=30)
      versionfile = f.read()
      f.close()
      # extract the version
      match = VERSION_RE.match(versionfile)
      if match:
        version = match.group(1)
        if version == Version.VERSION:
          if not auto:
            wx.MessageBox(lang('Your Picalo installation is up to date!'), lang('Check For Updates'), wx.OK | wx.ICON_INFORMATION)
        elif not auto or version != Preferences.get('skippedversion', '0'):
          if wx.MessageBox(lang('Picalo version %s is now available.  You are using version %s.  Would you like to upgrade your installation?') % (version, Version.VERSION), lang("Picalo"), style=wx.YES_NO) == wx.YES:
            wx.LaunchDefaultBrowser(DOWNLOAD_URL)
          else:
            Preferences.set('skippedversion', version)
    except Exception, e:
      if not auto:
        dialog = wx.MessageDialog(self, lang('An error occurred while checking for updates: ') + str(e) + '\n\n' + lang('Please check your internet connection.\n\nYou can also go to ') + DOWNLOAD_URL + lang(' to download updates.'), lang('Check for Updates'), wx.OK | wx.ICON_ERROR)
        dialog.ShowModal()
  


  #######################################################
  ###   Data menu options
  
  #######################################################
  ###   Analysis menu options
  
  def menuAnalyzeDescriptives(self, event=None):
    '''Analyzes the currently-selected table'''
    # get the table to analyze (first go for the current spreadsheet, if not look at table listing on left)
    tablename = None
    if self.notebook.GetSelection() >= 0 and isinstance(self.notebook.GetPage(self.notebook.GetSelection()), Spreadsheet.Spreadsheet):
      tablename = self.notebook.GetPage(self.notebook.GetSelection()).name
    else:
      raise AssertionError, lang('Please select a tab containing a table to run descriptives.')
    if tablename:
      newtablename = tablename.replace('.', '_') + '_descriptives'
      self.execute([
        newtablename + ' = Simple.describe(' + tablename + ')',
        newtablename + '.view()',
      ])


  #######################################################
  ###   Tools menu
  
  def menuToolsExpressionBuilder(self, event=None):
    '''Opens the expression builder for input into the shell'''
    builder = Dialogs.ExpressionBuilder(self)
    cmds = []
    if builder.runModal():
      self.execute(builder.expression)


  #####################################
  ###   Window menu commands
  
  def menuWindowClose(self, event=None):
    '''Closes the current window'''
    if self.notebook.GetSelection() >= 0:
      self.notebook.GetPage(self.notebook.GetSelection()).menuFileClose(self)
  
  
  def menuWindowCloseAll(self, event=None):
    '''Closes all the open windows'''
    while self.notebook.GetSelection() >= 0:
      if not self.notebook.GetPage(self.notebook.GetSelection()).menuFileClose(self):
        break  # stop if canceled by user
  
  

