#!/usr/bin/python

__doc__ = '''
This module extracts documentation from Picalo and creates a reference manual
section in LaTeX.

Special qualifiers in method doc strings are lines that start with:
@param   parameter description
@returns describes what the method returns

Any other lines are taken as documentation text.

The picalo src directory must be in PYTHONPATH for this to work.
'''

import sys, imp
from picalo import *
from picalo.gui import Utils


# the special characters in latex that must be escaped
latex_special_full = [
  [ '\\', '\\\\' ],
  [ '_', '\\_' ],
  [ '$', '\\$' ],
  [ '&', '\\&' ],
  [ '[', '$[$' ],
  [ ']', '$]$' ],
  [ '{', '\\{' ],
  [ '}', '\\}' ],
  [ '<', '$<$' ],
  [ '>', '$>$' ],
  [ '%', '\\%' ],
  [ '#', '\\#' ],
  [ '^', '\\^' ],
]

# import the commonly-used python modules
from picalo import python_modules
for name in python_modules:
  exec('import ' + name)

##################################
###   parsing function

def format(st):
  '''Formats a string for latex by escaping special characters'''
  if not st:
    return str(st)
  for char in latex_special_full:
    st = st.replace(char[0], char[1])
  return st
  


core_doc = '''
The core functions in Picalo include the basic data types, Table manipulation, and
loading/saving routines.  Users should first read about the Table type, which is the 
foundation of everything done in Picalo.  Almost all Picalo functions take a table
as a parameter and return a Picalo table when done.  This allows you to pass tables
in and out of functions in your analysis routines.
'''

##################################
###   main loop

outfile = open('../../doc_advanced/Reference.tex', 'w')
sys.stdout = outfile
print '% This file is embedded in Manual.tex and should not be compiled separately.'

categories = Utils.get_documentation()
catnames = categories.keys()
catnames.sort(lambda a,b: (a == 'Core' and -1) or (b == 'Core' and 1) or cmp(a.lower(), b.lower()))
for category in catnames:
  print >> sys.__stderr__, 'Chapter:', category
  print '\\chapter{' + format(category) + '}'
  
  # the documentation for the category
  if category == 'Core':
    print format(core_doc)
  elif locals().has_key(category):
    print format(locals()[category].__doc__)
  elif globals().has_key(category):
    print format(globals()[category].__doc__)
    
  # individual function documentation
  functions = categories[category].keys()
  functions.sort(lambda a,b: cmp(a.lower(), b.lower()))
  for function in functions:
    if function[:1] == '_':  # ignore private functions that start with _
      continue
    doc = categories[category][function]
    print >> sys.__stderr__, '\tFunction:', function
    
    # section heading
    print '\\section{' + format(function) + '}'
  
    # function documentation
    for desc in doc.desc:
      inlist = False
      for line in format(desc).splitlines():
        if line.strip()[:1] == '*':
          if not inlist:
            print '\\begin{itemize}'
            inlist = True
          print '\\item ' + line.strip()[1:]
        elif inlist:
          inlist = False
          print '\\end{itemize}'
          print line
        else:
          print line
      if inlist:
        print '\\end{itemize}'
      print
    
    # signature
    print '\\vspace{12pt}'
    print '\\textbf{Signature:}'
    print
    print '\\begin{small}'
    print '\\begin{verbatim}'
    sig = doc.getSignature()
    pos = sig.find('(')
    print sig.replace(',', ',\n' + ' '*pos)
    print '\\end{verbatim}'    
    print '\\end{small}'
    print

    # parameters    
    if len(doc.params) > 0:
      print '\\begin{itemize}'
      for param in doc.params:
        if param.required:
          print '\\item \\textbf{$<$' + format(param.type) + '$>$ ' + format(param.name) + ':} ' + format(param.desc)
        else:
          print '\\item $<$' + format(param.type) + '$>$ ' + format(param.name) + ' (optional): ' + format(param.desc) + ' If omitted, defaults to ' + format(str(param.default)) + '.'
      print '\\end{itemize}'
      print
    
    # return value
    if doc.retval:
      print 'Returns: ' + format(doc.retval.desc)
      print '\\vspace{6pt}'
      print
    print

    # example(s)
    for i, example in enumerate(doc.examples):
      print '\\vspace{6pt}'
      print '\\textbf{Example' + (len(doc.examples) > 1 and (' ' + str(i+1)) or '') + ':}'
      print
#      print '\\begin{small}'
#      print '\\begin{verbatim}'
      print '\\begin{picalo}'
      print example
      print '\\end{picalo}'
#      print '\\end{verbatim}'
#      print '\\end{small}'


    print
outfile.close()


