#!/usr/bin/python

####################################################################################
#                                                                                  #
# Copyright (c) 2003 Dr. Conan C. Albrecht <conan_albrechtATbyuDOTedu>             #
#                                                                                  #
# This file is part of Picalo.                                                     #
#                                                                                  #
# Picalo is free software; you can redistribute it and/or modify                   #
# it under the terms of the GNU General Public License as published by             #
# the Free Software Foundation; either version 2 of the License, or                # 
# (at your option) any later version.                                              #
#                                                                                  #
# Picalo is distributed in the hope that it will be useful,                        #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                   #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    #
# GNU General Public License for more details.                                     #
#                                                                                  #
# You should have received a copy of the GNU General Public License                #
# along with Foobar; if not, write to the Free Software                            #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA        #
#                                                                                  #
####################################################################################
#
# This module tests all the Picalo modules 
#
# Dec 9, 2003   Started this log.  Wrote the initial py test code
#
#
####################################################################################

from picalo import *
import unittest
import time

test_suites = []

######################################
###   py testing

class DatasetTestCase(unittest.TestCase):
  def setUp(self):
    # IMPORTANT: The tests below expect these exact tables and values
    # in the exact positions they are in.  Don't change them.  If you need
    # something else, create other tables.
    self.headers = [ ('id', int), ('name', unicode) ]
    self.rawdata = [
      [ '1', 'Bart' ],
      [ '2', 'Homer' ],
      [ '3', 'Marge' ],
    ]
    
  def tearDown(self):
    pass
    
  def datasetTable(self):
    '''dataset_ Testing the Table object'''
    # test different ways of creating tables
    table1 = Table(self.headers, self.rawdata)
    table2 = Table(self.headers)
    for row in self.rawdata:
      table2.append(row)
    table3 = Table(self.headers)
    table3.extend(self.rawdata)
    self.assertEqual(table1, table2)
    self.assertEqual(table2, table3)
    
    # invalid parameters
    self.assertRaises(AssertionError, Table, 'asdf', self.rawdata)
    self.assertRaises(AssertionError, Table, [2], self.rawdata)
    self.assertRaises(AssertionError, Table, self.headers, 'data')
    self.assertRaises(AssertionError, Table, self.headers, [[1],[2],3,4])
    
    # tests of length methods
    self.assertEqual(table1.column_count(), table2.column_count())
    self.assertEqual(len(table1), len(table2))
    
    # tests of set and get
    table1[2] = [ '4', 'Lisa' ]
    table2[2] = [ '4', 'Lisa' ]
    self.assertEqual(table1, table2)
    self.assertNotEqual(table2, table3)
    table3[2] = table1[2]
    self.assertEqual(table1, table2)
    
    # tests of searching methods
    self.assertEqual(table1.index('id', 'name'), {(2, 'Homer'): [1], (4, 'Lisa'): [2], (1, 'Bart'): [0]})
    self.assertEqual(table1.find(id=2, name='Homer'), [1])
    
    
  def datasetImportExport(self):
    '''dataset_ Testing importing and exporting routines (load and save pco, delimited, csv, tsv)'''
    import StringIO
    table1 = Table(self.headers, self.rawdata)
    buffers = [ StringIO.StringIO() for i in range(5) ]
    table1.save_delimited(buffers[0])
    table1.save_csv(buffers[1])
    table1.save_tsv(buffers[2])
    table1.save_fixed(buffers[3])
    table1.save(buffers[4])
    rbuffers = [ StringIO.StringIO(buffer.getvalue()) for buffer in buffers ]
    table2 = load_delimited(rbuffers[0])
    table3 = load_csv(rbuffers[1])
    table4 = load_tsv(rbuffers[2])
    table5 = load_fixed(rbuffers[3], (0, 2, 9))
    table6 = load(rbuffers[4])
    self.assertEqual(table1, table6)
    table1.set_type('id', str)
    self.assertEqual(table1, table2)
    self.assertEqual(table1, table3)
    self.assertEqual(table1, table4)
    self.assertEqual(table1, table5)
    
  def datasetRecord(self):
    '''dataset_ Testing the Record object'''
    table1 = Table(self.headers, self.rawdata)
    rec1 = table1[0]
    self.assertEqual(rec1['id'], rec1[0])
    rec1['id'] = '4'
    self.assertEqual(rec1[0], 4)
    
  def datasetColumn(self):
    '''dataset_ Testing the Column object'''
    table1 = Table(self.headers, self.rawdata)
    col1 = table1.column('id')
    col2 = table1.column(0)
    self.assertEqual(col1, col2)
    self.assertEqual(len(col1), len(col2))
    col1[0] = '4'
    self.assertEqual(table1[0][0], 4)
    self.assertEqual(table1.get_column_names(), ['id', 'name'])
    self.assertEqual(len(table1.get_columns()), 2)
    self.assertEqual(table1.column_count(), 2)
    table1.append_calculated('id_name', "str(record['id']) + record['name']")
    self.assertEqual(table1[1]['id_name'], '2Homer')
    table1.append_calculated_static('id_name2', str, "str(record.id) + record.name") # also test the table.column method
    self.assertEqual(table1[1]['id_name2'], '2Homer')
    
  def datasetIterator(self):
    '''dataset_ Testing the BasicIterator class'''
    table1 = Table(self.headers, self.rawdata)
    col1 = [ val for val in table1.column('id') ]
    col2 = [ val for val in table1.column(0) ]
    self.assertEqual(col1, col2)    
    
  def datasetAddition(self):
    '''dataset_ Testing addition of two tables'''
    table1 = Table(self.headers, self.rawdata)
    table2 = Table(self.headers, self.rawdata)
    table3 = Table(self.headers, self.rawdata)
    table3.extend(self.rawdata)
    self.assertEqual(table1+table2, table3)
    
  def datasetDateTime(self):
    '''dataset_ Testing the DateTime type'''
    date = DateTime()
    DateTime(str(date))

  def datasetStats(self):
    '''dataset_ Testing the basic statistical function'''
    li = [ 1, 2, 3, 4, 5 ]
    self.assertEqual(variance(li), 2.5)
    self.assertAlmostEqual(stdev(li), 1.58, 2)
    self.assertEqual(sum(li), 15)
    self.assertEqual(min(li), 1)
    self.assertEqual(max(li), 5)
    self.assertEqual(count(li), len(li))
    self.assertEqual(mean(li), 3.0)
  
  def datasetTableList(self):
    '''dataset_ Testing the table list'''
    tl = TableList([Table([('col000', int), ('col001', int)]), Table([('col000', int), ('col001', int)]), Table([('col000', int), ('col001', int)])])

  def datasetSetType(self):
    '''dataset_ Testing the setting of column types'''
    table = Table(self.headers, self.rawdata)
    table.set_type('id', float)
    self.assertEqual(table[0]['id'], 1.0)
    table.set_type('id', unicode)
    self.assertEqual(table[0]['id'], '1.0')
    

  
    
test_suites.append(unittest.makeSuite(DatasetTestCase, 'dataset'))




######################################
###   Benfords.py testing

class BenfordsTestCase(unittest.TestCase):
  def setUp(self):
    pass
        
  def tearDown(self):
    pass
    
  def benfords_calc_benford(self):
    # retrieved these values from Mathpages and Mathworld
    self.assertAlmostEqual(0.30103, Benfords.calc_benford(0, 1), places=5)
    self.assertAlmostEqual(0.04576, Benfords.calc_benford(0, 9), places=5)
    self.assertAlmostEqual(0.11389, Benfords.calc_benford(1, 1), places=5)
    self.assertAlmostEqual(0.10138, Benfords.calc_benford(2, 1), places=5)
  
  def benfords_get_expected(self):
    self.assertAlmostEqual(0.00288, Benfords.get_expected(15782.3, 3), places=5)
    self.assertAlmostEqual(0.02910, Benfords.get_expected(15782.3, 2), places=5)
    self.assertAlmostEqual(0.01208, Benfords.get_expected(352, 2), places=5)
    # should raise an exception because 35 doesn't have 3 digits
    self.assertRaises(AssertionError, Benfords.get_expected, 35, 3)
    
  def benfords_analyze(self):
    table = Table([('val', int)], [[15],[77],[13],[25]])
    results = Benfords.analyze(table, 'val', 2)
    self.assertEqual(results.column_count(), 5)
    self.assertAlmostEqual(results[0]['Difference'], 0.22, 2)
    
test_suites.append(unittest.makeSuite(BenfordsTestCase, 'benfords'))



######################################
###   Simple.py testing

class SimpleTestCase(unittest.TestCase):
  def setUp(self):
    # don't change the values of these tables because the test cases
    # are hard coded to their values
    self.table1 = Table([('col000', int), ('col001', int)], [
      [3,2],
      [1,2],
      [3,4],
      [1,2],
    ])
    # table2 is an already sorted version of table1
    self.table2 = Table([('col000', int), ('col001', int)], [
      [1,2],
      [1,2],
      [3,2],
      [3,4],
    ])
    # table3 is a table that somewhat matches table1
    self.table3 = Table([('col000', int), ('col001', int)], [
      [3,2],
      [4,3],
      [5,4],
      [1,2],
    ])
    self.table4 = Table([('col000', int), ('col001', unicode)], [
      [ '1', 'Bart' ],
      [ '2', 'Homer' ],
      [ '3', 'Marge' ],
    ])
    self.table5 = Table([('col000', int), ('col001', unicode)], [
      [ '1', 'Bart S.' ],
      [ '2', 'Homer Simpson' ],
      [ '3', 'Margarie Simpson' ],
    ])
        
  def tearDown(self):
    pass
    
  def simple_describe(self):
    desc = Simple.describe(self.table1)  # just to test a full table
    desc = Simple.describe(self.table2, 'col001')
    self.assertEqual(desc[0][1], 4)
    self.assertEqual(desc[6][1], 2.0)
    
  def simple_col_join(self):
    joined = Simple.col_join(self.table1, self.table2, ['col001', 'col001'])
    self.assertEqual(len(joined), 10)
    self.assertEqual(joined[0], [3, 2, 1, 2])
    
  def simple_join(self):
    joined = Simple.join(self.table1, self.table2, "record1['col001'] == record2['col001']")
    self.assertEqual(len(joined), 10)
    self.assertEqual(joined[0], [3, 2, 1, 2])
    
  def simple_sort(self):
    Simple.sort(self.table1, True, 0, 1)
    self.assertEqual(self.table1, self.table2)
    
  def simple_get_unordered(self):
    unordered = Simple.get_unordered(self.table3, True, 0, 1)
    self.assertEqual(unordered[0], [2])
    
  def simple_find_duplicates(self):
    dups = Simple.find_duplicates(self.table1, 0, 1)
    expected = [[ 1, 2, (1,3) ]]
    self.assertEqual(dups, expected)
    
  def simple_find_gaps(self):
    gaps = Simple.find_gaps(self.table3, True, 0)
    self.assertEqual(gaps[0][0], 2)
    
  def simple_select(self):
    results = Simple.select(self.table1, "record[1] > 3")
    expected = [[3,4]]
    self.assertEqual(results, expected)
    
  def simple_select_outliers_z(self):
    table = Table([('col001', int)], [[0],[1],[1],[1],[1],[1],[1],[1],[1],[1],[1],[1],[-6]])
    results = Simple.select_outliers_z(table, 0, 3)
    expected = [[-6]]
    self.assertEqual(results, expected)
  
  def simple_select_nonoutliers_z(self):
    table = Table([('col001', int)], [[0],[1],[1],[1],[1],[1],[1],[1],[1],[1],[1],[1],[-6]])
    results = Simple.select_nonoutliers_z(table, 0, 3)
    expected = [[0],[1],[1],[1],[1],[1],[1],[1],[1],[1],[1],[1]]
    self.assertEqual(results, expected)
    
  def simple_select_outliers(self):
    results = Simple.select_outliers(self.table3, 0, 4, 5)
    expected = [[3,2],[1,2]]
    self.assertEqual(results, expected)
    
  def simple_select_nonoutliers(self):
    results = Simple.select_nonoutliers(self.table3, 0, 4, 5)
    expected = [[4,3],[5,4]]
    self.assertEqual(results, expected)
    
  def simple_col_match_same(self):
    t1, t3 = Simple.col_match_same(self.table1, self.table3, [0,0], [1,1])
    expected1 = (
      [ 3, 2 ],
      [ 1, 2 ],
      [ 1, 2 ],
    )
    self.assertEqual(t1, expected1)
    expected3 = (
      [ 3, 2 ],
      [ 1, 2 ],
    )
    self.assertEqual(t3, expected3)
  
  def simple_col_match_diff(self):
    t1, t3 = Simple.col_match_diff(self.table1, self.table3, [0,0], [1,1])
    expected1 = (
      [ 3, 4 ],
    )
    self.assertEqual(t1, expected1)
    expected3 = (
      [ 4, 3 ],
      [ 5, 4 ],
    )
    self.assertEqual(t3, expected3)
    
  def simple_custom_match_same(self):
    t1, t3 = Simple.custom_match_same(self.table1, self.table3, "record1 == record2")
    expected1 = (
      [ 3, 2 ],
      [ 1, 2 ],
      [ 1, 2 ],
    )
    self.assertEqual(t1, expected1)
    expected3 =  (
      [ 3, 2 ],
      [ 1, 2 ],
    )
    self.assertEqual(t3, expected3)
    
  def simple_custom_match_diff(self):
    t1, t3 = Simple.custom_match_diff(self.table1, self.table3, "record1 == record2")
    expected1 =  (
      [ 3, 4 ],
    )
    self.assertEqual(t1, expected1)
    expected3 =  (
      [ 4, 3 ],
      [ 5, 4 ],
    )
    self.assertEqual(t3, expected3)
    
  def simple_soundex(self):
    snd = Simple.soundex('Bart Simpson', len=5)
    self.assertEqual(snd, 'B6325')

  def simple_soundex_col(self):
    table = self.table4[:]
    Simple.soundexcol(table, 'col001')
    self.assertEqual(table[0][2], 'B630')

  def simple_fuzzymatch(self):
    match = Simple.fuzzymatch('Bart Simpson', "Bart A. Simpson")
    self.assertAlmostEqual(match, .692, 3)

  def simple_fuzzycoljoin(self):
    joined = Simple.fuzzycoljoin(self.table4, 1, self.table5, 1, 0.2)
    self.assertEqual(len(joined), 2)

test_suites.append(unittest.makeSuite(SimpleTestCase, 'simple'))




######################################
###   Grouping.py testing

class GroupingTestCase(unittest.TestCase):
  def setUp(self):
    pass
        
  def tearDown(self):
    pass
    
  def grouping_stratify(self):
    table1 = Table([('col001', int), ('col002', int)], [[1,1], [1,2], [1,3], [2,1], [2,2], [3,1]])
    groups = Grouping.stratify(table1, 2)
    self.assertEqual(len(groups), 2)
    groups = Grouping.stratify(table1, 3)
    self.assertEqual(len(groups), 3)
    
  def grouping_stratify_by_value(self):
    table1 = Table([('col001', int), ('col002', int)], [[1,1], [1,2], [1,3], [2,1], [2,2], [3,1]])
    groups = Grouping.stratify_by_value(table1, 0)
    self.assertEqual(len(groups), 3)
    self.assertEqual(groups[0], [[1,1],[1,2],[1,3]])
    self.assertEqual(groups[1], [[2,1],[2,2]])
    self.assertEqual(groups[2], [[3,1]])

  def grouping_stratify_by_expression(self):
    table1 = Table([('col001', int), ('col002', int)], [[1,6], [1,2], [1,4], [2,1], [2,5], [3,1]])
    groups = Grouping.stratify_by_expression(table1, "record[1] % 4.0 == 0.0")
    self.assertEqual(len(groups), 2)
    
  def grouping_stratify_by_step(self):
    table1 = Table([('col000', int), ('col001', int)], [[1,1], [2,2], [4,3], [7,1], [8,2], [16,1]])
    groups = Grouping.stratify_by_step(table1, 0, 5)
    self.assertEqual(len(groups), 4)
    self.assertEqual(groups[0], [[1,1],[2,2],[4,3]])
    self.assertEqual(groups[1], [[7,1],[8,2]])
    self.assertEqual(groups[2], [])
    self.assertEqual(groups[3], [[16,1]])
    
  def grouping_summarize_by_value(self):
    table = Table([('col000', int), ('col001', unicode)], [
      [ '1', 'Bart' ],
      [ '1', 'Homer' ],
      [ '3', 'Lisa' ],
      [ '3', 'Maggie' ],
      [ '3', 'Moe' ],
    ])
    results = Grouping.summarize_by_value(table, 'col000', num='count(group)')
    self.assertEqual(len(results), 2)
    self.assertEqual(results[0][1], 2)
    self.assertEqual(results[1][1], 3)
    
  def grouping_summarize_by_expression(self):
    table1 = Table([('col000', int), ('col001', int)], [[1,6], [1,2], [1,4], [2,1], [2,5], [3,1]])
    result = Grouping.summarize_by_expression(table1, "record[1] % 4.0 == 0.0", total='sum(group.column(0))')
    self.assertEqual(result[0][2], 2)
    self.assertEqual(result[1][2], 8)
    
  def grouping_stratify_by_step(self):
    table1 = Table([('col000', int), ('col001', int)], [[1,1], [2,2], [4,3], [7,1], [8,2], [16,1]])
    result = Grouping.summarize_by_step(table1, 'col000', 5, total='sum(group.column(0))')
    self.assertEqual(result, [
      [ '1',  '6',  7  ],
      [ '6',  '11', 15 ],
      [ '11', '16', 0  ],
      [ '16', '21', 16 ],
    ])
    
    
test_suites.append(unittest.makeSuite(GroupingTestCase, 'grouping'))



######################################
###   Crosstable.py testing

class CrosstableTestCase(unittest.TestCase):
  def setUp(self):
    self.data = Table([
          ('Region', unicode),
          ('Product', unicode),
          ('Salesperson', unicode), 
          ('Customer', unicode),
          ('Amount', int),
        ],[
          [ 'Rural','Computers',  'Mollie', 'Faiz',  500 ],
          [ 'City',  'Monitors',  'Danny',  'Sheng', 700 ],
          [ 'Rural', 'Mice',      'Mollie', 'Brian', 900 ],
          [ 'City',  'Computers', 'Danny',  'Faiz',  300 ],
          [ 'Rural', 'Monitors',  'Mollie', 'Sheng', 500 ],
          [ 'Rural', 'Monitors',  'Mollie', 'Sheng', 500 ],
          [ 'City',  'Mice',      'Danny',  'Brian', 100 ],
          [ 'Rural', 'Monitors',  'Mollie', 'Faiz',  200 ],
          [ 'City',  'Computers', 'Danny',  'Sheng', 400 ],
        ])
    self.data2 = Table([
          ('Region', unicode),
          ('Product', unicode),
          ('Salesperson', unicode), 
          ('Customer', unicode),
          ('Amount', int),
        ],[
          [ 'Rural','Computers',  'Mollie', 'Faiz',  500 ],
          [ 'City',  'Monitors',  'Danny',  'Sheng', 700 ],
          [ 'Rural', 'Mice',      'Mollie', 'Brian', 900 ],
          [ 'City',  'Computers', 'Danny',  'Faiz',  300 ],
          [ 'Rural', 'Monitors',  'Mollie', 'Sheng', 500 ],
          [ 'Rural', 'Monitors',  'Mollie', 'Sheng', 500 ],
          [ 'City',  'Mice',      'Danny',  'Brian', 100 ],
          [ 'Rural', 'Monitors',  'Mollie', 'Faiz',  200 ],
          [ 'City',  'Computers', 'Danny',  'Sheng', 400 ],
        ])
        
  def tearDown(self):
    pass
    
  def crosstable_pivot(self):
    ret = Crosstable.pivot(self.data, 'Salesperson', 'Product', 'sum(group.column("Amount"))')
    self.assertEqual(ret, [
      [ 'Computers', 700,  500,  1200 ],
      [ 'Mice',      100,  900,  1000 ],
      [ 'Monitors',  700,  1200, 1900 ],
      [ 'Totals',    1500, 2600, 4100 ]
    ])
    
  def crosstable_pivot_map(self):
    ret = Crosstable.pivot_map(self.data2, 'Salesperson', 'Product', 'sum(group.column("Amount"))')
    self.assertEqual(ret, {
      ('Monitors', 'Mollie'): (1200,), 
      ('Computers', 'Mollie'): (500,), 
      ('Mice', 'Danny'): (100,), 
      ('Computers', 'Danny'): (700,), 
      ('Mice', 'Mollie'): (900,), 
      ('Monitors', 'Danny'): (700,),
    })
    

test_suites.append(unittest.makeSuite(CrosstableTestCase, 'crosstable'))





######################################
###   Trending.py testing

class TrendingTestCase(unittest.TestCase):
  def setUp(self):
    self.table = Table([('col000', unicode), ('col001', int), ('col002', int)], [
      ['Dan',10,8],
      ['Sally',12,12],
      ['Dan',11,15], 
      ['Sally',12,14], 
      ['Dan',11,16], 
      ['Sally',15,15], 
      ['Dan',16,15], 
      ['Sally',13,14]
    ])
        
  def tearDown(self):
    pass
    
  def trending_cusum(self):
    table = Table([('col000', int), ('col001', int)], ([5,6], [3,2], [4,6]))
    cusum = Trending.cusum(table, 'col000')
    self.assertEqual(cusum, [
      [ 0 ],
      [ -2 ],
      [ -1 ],
    ])
    
  def trending_hilowslope(self):
    results = Trending.highlow_slope(self.table, 'col002', 'col001')
    self.assertEqual(results, [
      [ 10, 8, 11, 16, 8.0 ],
    ])

  def trending_slope(self):
    results = Trending.average_slope(self.table, 2, 1)
    self.assertAlmostEqual(results[0][0], -0.5595, 4)

  def trending_slope(self):
    slope = Trending.slope(5, 4, 9, 7)
    self.assertEqual(slope, 0.75)

  def trending_regression(self):
    results = Trending.regression(self.table, 'col001')
    self.assertAlmostEqual(results[0][0], .619, 3)
    self.assertAlmostEqual(results[0][1], 10.333, 3)
    self.assertAlmostEqual(results[0][2], .043, 3)
    self.assertAlmostEqual(results[0][3], .002, 3)
    
  def trending_handshake_slope(self):
    results = Trending.handshake_slope(self.table, 'col001', 'col002')
    self.assertAlmostEqual(results[0][0], .163, 3)

test_suites.append(unittest.makeSuite(TrendingTestCase, 'trending'))







######################################
###   Main loop

if __name__ == '__main__':
  print
  alltests = unittest.TestSuite(test_suites)
  unittest.TextTestRunner(verbosity=2).run(alltests)
  
